/*
 * Copyright (c) 1990,1,2 Mark Nitzberg
 * and President and Fellows of Harvard College
 * All rights reserved.
 * 
 * read contours and junctions
 */

#include <math.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include "edgemap.h"

char   *PgmName;
char   *CurrentFile = STDIN_STRING;
int     Status = 0;		/* exit status */
int     CurrentLine = 0;

/* file shape */
real    Height, Width;

/* global lists and their sizes */
Contour Contours[MAXCONTOUR];
int     NContours;
Junction Junctions[MAXJUNCTION];
int     NJunctions;

/*
 * returns TRUE for success
 */
int
ReadContours(f)
FILE   *f;
{
    PointList *p;
    Contour *c;
    int     FindOrAddJunction( /* real x, real y, int cno, int end */ );

    if (!ReadHeader(f, &Width, &Height)) {
	Status = 4;
	return FALSE;
    }
    while ((p = ReadContour(f)) != NULL) {

	int     n = p->n;

	/* new contour */
	if (NContours >= MAXCONTOUR) {
	    fprintf(stderr,
		    "%s: file `%s', line %d: too many curves (%d) for me; ignoring the rest.\n",
		    PgmName, CurrentFile, CurrentLine, NContours);
	    Status = 2;
	    return NULL;
	}
	c = &Contours[NContours];
	NContours++;

	c->p = p;

	/* two junctions */
	c->junc[HEAD] = FindOrAddJunction(p->x[0], p->y[0],
					  NContours - 1, HEAD);
	c->junc[TAIL] = FindOrAddJunction(p->x[n - 1], p->y[n - 1],
					  NContours - 1, TAIL);
    }

    return TRUE;
}

void
FreeContours()
{
    int     i;

    for (i = 0; i < NContours; i++)
	FreePointList(Contours[i].p);
    NContours = 0;
    NJunctions = 0;
}

/*
 * return the index of junction that contains point (x,y) or if there is
 * none, add a new junction with contour number cno, end end Returns -1 if
 * the junction existed but was full
 */
int
FindOrAddJunction(x, y, cno, end)
real    x, y;
int     cno, end;
{
    int     i;
    Junction *J;

    for (i = 0; i < NJunctions; i++) {

	Junction *J = &Junctions[i];
	int     cinj;		/* contours at this junction */

	for (cinj = 0; cinj < J->n; cinj++) {

	    Contour *C = &Contours[J->c[cinj].cno];
	    int index = (J->c[cinj].end == HEAD ? 0 : C->p->n - 1);


	    if (C->p->x[index] == x
		&& C->p->y[index] == y) {

		/* J = Junctions[i] is at (x,y) */

		/* check if cno, end is already there */
		int     k;

		for (k = 0; k < J->n; k++) {
		    if (J->c[k].cno == cno
			&& J->c[k].end == end) {
			/* yes, just return the junction # */
			return i;
		    }
		}

		/* add a new cno, end */
		if (J->n >= 4) {
		    static  once;

		    if (!once) {
			fprintf(stderr,
			      "%s: Junctions of > 4 contours not joined.\n",
				PgmName);
			once = TRUE;
		    }
		    return -1;
		}
		J->c[J->n].cno = cno;
		J->c[J->n].end = end;
		J->n++;
		return i;
	    }
	}
    }

    /* not found -- grab a new one */
    if (NJunctions >= MAXJUNCTION) {
	fprintf(stderr,
		"%s: file `%s', line %d: too many junctions (%d) for me; aborting.\n",
		PgmName, CurrentFile, CurrentLine, NJunctions);
	exit(3);
    }
    J = &Junctions[NJunctions];
    NJunctions++;

    J->isT = FALSE;
    J->n = 1;
    J->c[0].cno = cno;
    J->c[0].end = end;

    return NJunctions - 1;
}

void
PrintContours(f, explanation)
FILE   *f;
char   *explanation;
{
    Contour *C;

    /* print header */
    fprintf(f,
	    "# %s from `%s'\n%g %g\n\n# x y theta(deg) curvature\n",
	    explanation, CurrentFile, Width, Height);

    for (C = &Contours[0]; C <= &Contours[NContours - 1]; C++) {

	/* print out the point list */
	PointList *p = C->p;
	int     i;

	fprintf(f, "# CONTOUR %d points", p->n);
	if (p->closed)
	    fprintf(f, " CLOSED");
	if (p->withcorner)
	    fprintf(f, " WITHCORNER");
	if (p->contin)
	    fprintf(f, " CONTINUATION");
	if (p->boundary)
	    fprintf(f, " BOUNDARY");
	fprintf(f, "\n");

	for (i = 0; i < p->n; i++)
	    fprintf(f, "%g %g %g %g\n",
		    p->x[i], p->y[i], DEG(p->theta[i]), p->k[i]);
	fprintf(f, "\n");
    }
}

/*** FileName
 *	"foo.im" --> "foo<extension>"
 *	"foo.bar5" --> "foo<extension>5"
 *	"bar" --> "bar<extension>"
 * (If there was no '.' in the file name, just appends extension.)
 *
 * returns a static buffer
 */
char   *
FileName(name, extension)
char   *name;
char   *extension;
{
    static char buf[512];
    char   *rindex(), *lastdot;
    int     gotnumber = FALSE;
    char    number[16];

    strcpy(buf, name);

    lastdot = rindex(buf, '.');
    if (lastdot) {
	char   *p = buf + strlen(buf) - 1;

	*lastdot = '\0';	/* replace any suffix */

	while ('0' <= *p && *p <= '9') {
	    gotnumber = TRUE;
	    --p;
	}
	strcpy(number, p + 1);
    }
    strcat(buf, extension);
    if (gotnumber)
	strcat(buf, number);	/* copies integer */
    return buf;
}

/*
 * FileNameNoNumber "foo.im5" --> "foo<extension>" "bar" --> "bar<extension>"
 * (If there was no '.' in the file name, just appends extension.)
 * 
 * returns a static buffer
 */
char   *
FileNameNoNumber(name, extension)
char   *name;
char   *extension;
{
    static char buf[512];
    char   *rindex(), *lastdot;

    strcpy(buf, name);

    lastdot = rindex(buf, '.');
    if (lastdot)
	*lastdot = '\0';	/* replace any suffix */

    strcat(buf, extension);
    return buf;
}

/***
 * Routines to read .e files:
 *
 *	    char *PgmName -- name of this program
 *	char *CurrentFile -- name of file being read
 *	  int CurrentLine -- input line number in current file
 *
 *        int ReadHeader(stream, &width, &height) -- read header from .e file
 * PointList *ReadContour(stream) -- read one contour from .e file
 */

real   *
AllocReal(n)
int     n;
{
    char   *calloc();
    real   *r = (real *) calloc(n, sizeof (real));

    if (r == NULL) {
	fprintf(stderr, "%s: Out of memory for real arrays\n",
		PgmName);
	exit(2);
    }
    return r;
}

PointList *
NewPointList(size)
int     size;
{
    char   *calloc();
    PointList *plist = (PointList *) calloc(1, sizeof (PointList));

    if (plist == NULL) {
	fprintf(stderr, "%s: Out of memory for point lists\n",
		PgmName);
	exit(2);
    }
    plist->size = size;
    plist->n = 0;
    plist->closed = FALSE;
    plist->withcorner = FALSE;
    plist->contin = FALSE;
    plist->x = AllocReal(size);
    plist->y = AllocReal(size);
    plist->theta = AllocReal(size);
    plist->k = AllocReal(size);

    return plist;
}

void
FreePointList(plist)
PointList *plist;
{
    if (plist->size > 0) {
	free(plist->x);
	free(plist->y);
	free(plist->theta);
	free(plist->k);
    }
    free(plist);
}


PointList *
ReadContour(stream)
FILE   *stream;
{
    PointList *contour = NULL;
    char   *fgets();
    char    buf[512];
    int     point;

    if (feof(stream))
	return NULL;

    point = 0;

    while (fgets(buf, 512, stream) != NULL) {

	CurrentLine++;

	if (strncmp(buf, "# CONTOUR ", 10) == 0) {
	    char    word[32];
	    int     i, n, index;

	    i = sscanf(buf, "# CONTOUR %d points%n",
		       &n, &index);

	    if (i < 1) {
		fprintf(stderr,
			"%s: file `%s', line %d: CONTOUR line needs # points in contour\n",
			PgmName, CurrentFile, CurrentLine);
		return NULL;
	    }
	    contour = NewPointList(n);

	    while (sscanf(buf + index, " %s%n", word, &i) >0) {

		index +=i;

		if (strcmp(word, "CLOSED") == 0)
		    contour->closed = TRUE;
		else if (strcmp(word, "WITHCORNER") == 0)
		    contour->withcorner = TRUE;
		else if (strcmp(word, "CONTINUATION") == 0)
		    contour->contin = TRUE;
		else if (strcmp(word, "BOUNDARY") == 0)
		    contour->boundary = TRUE;
	    }

	    continue;

	}
	/* skip other comments */
	if (buf[0] == '#')
	    continue;

	/* blank line means done, unless we haven't got started */
	if (buf[0] == '\n' || buf[0] == '\0') {
	    if (point == 0)
		continue;
	    break;
	}
	if (contour == NULL) {
	    fprintf(stderr,
		    "%s: file `%s', line %d: needs line of form # CONTOUR <n> points\n",
		    PgmName, CurrentFile, CurrentLine);
	    return NULL;
	}
	contour->theta[point] = 0;
	contour->k[point] = 0;

	if (sscanf(buf, "%g %g %g %g",
		   &contour->x[point],
		   &contour->y[point],
		   &contour->theta[point],
		   &contour->k[point]) < 2) {
	    fprintf(stderr, "%s: `%s', line %d, needs x y [ theta kappa ]\n",
		    PgmName, CurrentFile, CurrentLine);
	    return NULL;
	}
	/* internally we use radians */
	contour->theta[point] = RAD(contour->theta[point]);
	point++;
    }
    if (contour != NULL)
	contour->n = point;

    return contour;
}

ReadHeader(stream, widthp, heightp)
FILE   *stream;
real   *widthp, *heightp;
{
    char    buf[512];

    while (fgets(buf, 512, stream) != NULL) {

	CurrentLine++;

	if (buf[0] == '#' || buf[0] == '\n' || buf[0] == '\0')
	    continue;
	if (sscanf(buf, "%g %g", widthp, heightp) == 2)
	    break;

	fprintf(stderr, "%s: `%s', line %d needs width & height.\n",
		PgmName, CurrentFile, CurrentLine);
	return FALSE;
    }
    return TRUE;
}
