/***
 * Copyright (c) 1990,1,2 Mark Nitzberg
 * and President and Fellows of Harvard College
 * All rights reserved.
 *
 * For edge map (.e*) data and file handling
 *
 *	char *PgmName -- name of this program
 *	char *CurrentFile -- name of file being read
 *	int CurrentLine -- input line number in current file
 *	int Status -- exit status
 *
 * ReadHeader(stream, &width, &height) -- read header from .e file
 * ReadCurve(stream)  --  read one curve from .e file
 *
 * NewPointList(size) -- allocate a new point list
 * FreePointList(p) -- free point list made by NewPointList()
 * AllocReal(n) -- allocate a vector of n reals
 */

#ifndef _edgemap_h
#define _edgemap_h

#include <stdio.h>
#include <math.h>

/*
 * If you change the "real" type, change calls to scanf(): %lg for double vs
 * %g for float.
 */

#ifndef ONEBYTE			/* hack to test if hvision.h included */
typedef float real;

#define TRUE 1
#define FALSE 0
#define MIN(a,b) ((a)<(b)?(a):(b))
#define MAX(a,b) ((a)>(b)?(a):(b))
#endif				/* ONEBYTE */

#ifndef DEG
#define DEG(rad) ((rad)*180./M_PI)
#define RAD(deg) ((deg)*M_PI/180.)
#endif				/* DEG */

#ifndef IROUND
#define IROUND(x) ((int) ((x)+0.5))
#endif				/* IROUND */

/*------------ info for error handling ----------*/

extern char *PgmName;
extern char *CurrentFile;

#define STDIN_STRING "<standard-input>"
extern int CurrentLine;
extern int Status;


/*------------- point lists ----------------*/
/* a list of points making up a curve */
typedef struct PointList {
    /* arrays */
    real   *x, *y, *theta, *k;

    int     n;			/* number of points (x[0..n-1],y[0..n-1]) */
    int     size;		/* actual number size of arrays */
    char    closed;		/* TRUE for closed contour */
    char    withcorner;		/* TRUE if closed and 1st point is corner
				 * point */
    char    contin;		/* TRUE if contour is a continuation */
    char    boundary;		/* TRUE if contour lies along image boundary */
} PointList;


typedef struct Contour {

    PointList *p;		/* point list w/contour info */
    short   junc[2];		/* index in Junctions[] of the 2 ends */

    real    length;		/* length (not # points!) */
    real    ksq;		/* ~integral of curvature^2 (k^2) */
    real    endtan[2];		/* outward-pointing tangent at the 2 ends */
    short   reg[2];		/* region number of region on each side; LEFT
				 * is on one's left going from HEAD to TAIL */
    real    endavg[2][2];

    /*
     * pixel averages on L&R of each end ie. use endavg[HEAD][LEFT]. LEFT is
     * on one's left as one leaves the contour's end headed outward
     */

    /* ends */
#define HEAD 0
#define TAIL 1
#define OtherEnd(e) (1-(e))

    /* sides */
#define LEFT 0
#define RIGHT 1

} Contour;

#define MAXCONTOUR 2000
extern int NContours;
extern Contour Contours[MAXCONTOUR];

/* curves are joined at junctions */
typedef struct Junction {

    short   n;			/* number of contours meeting here */

    short   isT;		/* TRUE if this is a T */
    short   c1, c2;		/* 2 curves to treat as one if isT NOTE c1,c2
				 * index into c[] */
    struct {
	short   cno;		/* index in Contours[] */
	short   end;		/* which end, HEAD or TAIL */
    }       c[4];
} Junction;

#define MAXJUNCTION 2000
int     NJunctions;
Junction Junctions[MAXJUNCTION];

/* main functions */
int     ReadContours( /* FILE *f */ );
void    FreeContours();
void    PrintContours( /* FILE *f, char *explanation */ );

/* auxiliaries for reading */
PointList *ReadContour( /* FILE *stream */ );
int     ReadHeader( /* FILE *stream, real *widthp, real *heightp */ );

/* utilities */
PointList *NewPointList( /* int size */ );
void    FreePointList();
real   *AllocReal( /* int n */ );
char   *FileName( /* char *name, char *extension */ );
char   *FileNameNoNumber( /* char *name, char *extension */ );

/*-------------- shorthand for tracing contours ------------------*/
#define FORWARD 1		/* for moving through lists */
#define BACKWARD -1
#define	EndIndex(c, end)	(end == HEAD ? 0 : Contours[c].p->n-1)
#define	Direction(end)		(end == HEAD ? FORWARD : BACKWARD)

extern real Height, Width;

#endif				/* _edgemap_h */
