/*
 * Copyright (c) 1990,1,2 Mark Nitzberg
 * and President and Fellows of Harvard College
 * All rights reserved.
 */
#include <hvision.h>
#include "edgemap.h"
#include "region.h"

IMAGE  *RegionImage;
Region  Regions[MAXREGION + 1];

int
SetRegionImageEdgePoint(x, y, cno)
int     x, y;
int     cno;
{
    u_short was;

    if (x < 0 || x >= Width || y < 0 || y >= Height)
	return 0;

    was = US_PIX(RegionImage, y, x);
    US_PIX(RegionImage, y, x) = EDGE_PIXEL + cno;
    return was;
}

TraceRegions()
{
    int     i, j, x, y;
    int     left, bottom, right, top;

    /* make 16-bit RegionImage with zeros everywhere */
    if (RegionImage == NULL)
	RegionImage = hvMakeImage(IROUND(Height), IROUND(Width), 1, 0, UTWOBYTE);

    /* start with 0 at each pixel */
    bzero((char *) RegionImage->im, RegionImage->bytesize);
    NRegions = 0;
    left = right = IROUND(Width/2.0);
    bottom = top = IROUND(Height/2.0);

    /* draw contours into the region image */
    for (i = 0; i < NContours; i++) {
	PointList *p = Contours[i].p;

	if (p->n <= 1)
	    continue;		/* single-point contour? */

	for (j = 0; j < p->n - 1; j++) {
	    int x1 = IROUND(p->x[j]);
	    int y1 = IROUND(p->y[j]);
	    int x2 = IROUND(p->x[j + 1]);
	    int y2 = IROUND(p->y[j + 1]);

	    (void) ApplyToLineClip(x1, y1, x2, y2,
				   SetRegionImageEdgePoint, i);

	    if (! p->boundary)
		continue;

	    /* find boundary bounds */
	    if (x1 < left)
		left = x1;
	    if (x1 > right)
		right = x1;
	    if (y1 < bottom)
		bottom = y1;
	    if (y1 > top)
		top = y1;
	    /* NOTE:  no need to look at x2,y2; think it over */
	}
    }

    if (left == right) {

	/* no boundary */
	left = bottom = 0;
	right = IROUND(Width)-1;
	top = IROUND(Height)-1;
    }

    /* fill regions */
    for (y = bottom; y <= top; y++)
	for (x = left; x <= right; x++)
	    if (US_PIX(RegionImage, y, x) == EMPTY_PIXEL) {
		FillRegion(x, y, ++NRegions);
		/* must have substance */
		if (Regions[NRegions].npix <= 5)
		    NRegions--;
	    }

    fprintf(stderr, "%s: %d regions\n", PgmName, NRegions);
}

SetContourRegions()
{
    int     i;

    for (i = 0; i < NContours; i++) {
	Contours[i].reg[LEFT] = RegionOnOneSide(i, LEFT);
	Contours[i].reg[RIGHT] = RegionOnOneSide(i, RIGHT);
    }
}

/* four-connected flood fill--recursive */
FillRegion(x, y, regno)
int     x, y, regno;
{
    if (x < 0 || x >= Width || y < 0 || y >= Height
	|| US_PIX(RegionImage, y, x) != EMPTY_PIXEL)
	return;

    US_PIX(RegionImage, y, x) = regno;
    Regions[regno].sum += R_PIX(InputImage, y, x);
    Regions[regno].npix++;

    FillRegion(x - 1, y, regno);
    FillRegion(x + 1, y, regno);
    FillRegion(x, y - 1, regno);
    FillRegion(x, y + 1, regno);
}

/*
 * Region Printing
 * 
 * NOTE: No closed contours are allowed, so split any closed contours into two
 * contours at the halfway mark by a Junction.
 */

PrintRegions(stream)
FILE   *stream;
{
    int     i;

    fprintf(stream, "# Region graph from `%s'\n", CurrentFile);
    fprintf(stream, "# edges: from to left right length curvature theta1 theta2\n");

    /* regions separated by continuations average with one another */
    MergeBrightnesses();

    for (i = 0; i < NContours; i++) {
	Contour *c = &Contours[i];

	if (c->p->closed == TRUE)
	    SplitContour(i, (int) c->p->n / 2);
	/* increases NContours & NJunctions; that's OK */

	fprintf(stream, "%d %d %d %d %g %g %g %g\n",
		1 + c->junc[HEAD],
		1 + c->junc[TAIL],
		c->reg[LEFT],
		c->reg[RIGHT],
		c->length,
		c->ksq,
		DEG(c->endtan[HEAD]),
		DEG(c->endtan[TAIL]));
    }

    fprintf(stream, "\n# regions: area mean\n");
    for (i = 1; i <= NRegions; i++) {

	fprintf(stream, "%d %g\n",
		Regions[i].npix, Regions[i].sum / (real) Regions[i].npix);
    }

    fprintf(stream, "\n# Drawing info: left bottom right top black white\n");
    fprintf(stream, "%g %g %g %g %g %g\n\n", 0.0, 0.0, Width, Height, MinPixel, MaxPixel);

    /* edges */
    for (i = 0; i < NContours; i++) {
	int     nchars, totalchars, point;
	PointList *p = Contours[i].p;

	fprintf(stream, "# Contour %d\n", i + 1);

	totalchars = 0;
	for (point = 0; point < p->n; point++) {
	    fprintf(stream, "%g %g%n", p->x[point], Height - 1 - p->y[point], &nchars);
	    totalchars += nchars + 1;

	    if (point < p->n - 1) {
		if (totalchars > 70) {
		    fprintf(stream, "\n ");
		    totalchars = 1;
		} else
		    fprintf(stream, " ");
	    }
	}
	fprintf(stream, "\n");
    }
}

MergeBrightnesses()
{
    int     i, r;
    real    avg, totalpix;
    char    visited[MAXREGION];

    bzero(visited, NRegions + 1);	/* not visited */

    /* for each unvisited region */
    for (r = 1; r <= NRegions; r++) {

	int     ncollected;
	int     gotone;
	short   collected[MAXREGION];

	if (visited[r])
	    continue;

	collected[0] = r;
	ncollected = 1;

	/*
	 * repeatedly add to collected regions all those connected to them
	 * via continuations
	 */
	do {
	    gotone = FALSE;

	    for (i = 0; i < NContours; i++) {
		int     reg;

		/*
		 * if contour is a continuation with exactly one of the
		 * regions it separates among collected, add to collection
		 */
		if (Contours[i].p->contin
		&& (reg = ExactlyOneAmong(collected, ncollected, i)) != 0) {

		    collected[ncollected++] = reg;
		    visited[reg] = TRUE;
		    gotone = TRUE;
		}
	    }
	} while (gotone);

	/* merge the collected regions */
	if (ncollected == 1)
	    continue;

	avg = 0;
	totalpix = 0;
	for (i = 0; i < ncollected; i++) {
	    avg += Regions[collected[i]].sum;
	    totalpix += Regions[collected[i]].npix;
	}
	avg /= totalpix;
	for (i = 0; i < ncollected; i++)
	    Regions[collected[i]].sum = avg * Regions[collected[i]].npix;
    }
}

/*
 * if exactly one of contour's regions is among rlist, return the other, else
 * 0
 */
int
ExactlyOneAmong(rlist, n, cno)
short  *rlist;
int     n, cno;
{
    Contour *C = &Contours[cno];
    int     i, left = C->reg[LEFT], right = C->reg[RIGHT];
    int     gotleft = FALSE, gotright = FALSE;

    for (i = 0; i < n; i++) {
	if (left == rlist[i])
	    gotleft = TRUE;
	if (right == rlist[i])
	    gotright = TRUE;
    }
    if (gotleft && !gotright)
	return right;
    if (gotright && !gotleft)
	return left;
    return 0;
}

/*
 * ApplyToLineClip(fromx, fromy, tox, toy, f, arg) -- call f(x,y, arg) for
 * grid points x,y on ~line from from to to, clipping x and y to Width and
 * Height.
 * 
 * Returns 0 or the last non-zero value returned by (*f)()
 */

#ifndef SGN
#define	SGN(x)		((x)<0 ? -1 : 1)
#endif

int
ApplyToLineClip(fromx, fromy, tox, toy, f, arg)
int     fromx, fromy, tox, toy;
int     (*f) ();
int     arg;
{
    int     x, y, dx, dy, flipxy;
    int     xinc, yinc, incr1, incr2, d, xmin, xmax;
    int     npoints;
    int     val, retval = 0;

    dx = ABS(fromx - tox);
    dy = ABS(fromy - toy);

    if (dy <= dx) {
	/* iterate x, and then sometimes y */
	flipxy = FALSE;

	/* initialize */
	d = (2 * dy) - dx;	/* "divergence" from line */
	incr1 = 2 * dy;		/* increment if d < 0 */
	incr2 = 2 * (dy - dx);	/* increment if d >= 0 */

	/* x moves from from to to, y moves according to slope */
	xinc = SGN(tox - fromx);
	yinc = xinc * SGN((toy - fromy) * (tox - fromx));
	x = fromx;
	y = fromy;
	xmin = MIN(fromx, tox);
	xmax = MAX(fromx, tox);

    } else {
	/* above code with fromx/y, tox/y, and dx/y flipped */
	flipxy = TRUE;

	/* initialize */
	d = (2 * dx) - dy;	/* "divergence" from line */
	incr1 = 2 * dx;		/* increment if d < 0 */
	incr2 = 2 * (dx - dy);	/* increment if d >= 0 */

	/* x moves from from to to, y moves according to slope */
	xinc = SGN(toy - fromy);
	yinc = xinc * SGN((tox - fromx) * (toy - fromy));
	x = fromy;
	y = fromx;
	xmin = MIN(fromy, toy);
	xmax = MAX(fromy, toy);
    }

    /* draw first point */
    if (flipxy)
	val = (*f) (y, x, arg);
    else
	val = (*f) (x, y, arg);

    if (val != 0)
	retval = val;

    for (npoints = 0;; npoints++) {
	x += xinc;
	if (d < 0)
	    d += incr1;
	else {
	    y += yinc;
	    d += incr2;
	}

	if (x < xmin || x > xmax)
	    break;

	if (flipxy)
	    val = (*f) (y, x, arg);
	else
	    val = (*f) (x, y, arg);

	if (val != 0)
	    retval = val;
    }
    return retval;
}

/*
 * Add a junction and a contour by splitting contour cno at its pno'th point
 */
SplitContour(cno, pno)
int     cno, pno;
{
    int     newc, newj, i;
    int     closed, size, oldn;
    Contour *C, *newC;
    Junction *J, *newJ;

    if (NJunctions == MAXJUNCTION || NContours == MAXCONTOUR) {
	fprintf(stderr, "%s: Out of contour space at SplitContour\n",
		PgmName);
	exit(3);
    }
    closed = Contours[cno].p->closed;

    fprintf(stderr, "%s: Splitting %scontour %d at point %d\n",
	    PgmName, closed ? "closed " : "", cno, pno);

    newj = NJunctions++;
    newc = NContours++;

    /*
     * 1st pno+1 points, 0..pno, of contour stay with C; then pno..(C->p->n -
     * 1) go to newC. If C was closed, then newC gets one last point = point
     * 0 of C
     */
    C = &Contours[cno];
    newC = &Contours[newc];
    oldn = C->p->n;
    size = oldn - pno + (closed ? 1 : 0);
    newC->p = NewPointList(size);

    newC->p->n = size;
    C->p->n = pno + 1;

    /* copy C's last "size" points */
    CopyPointList(C->p, pno, newC->p, 0, size);

    /* The new junction C - newJ - newC */
    newJ = &Junctions[newj];
    newJ->n = 2;
    newJ->isT = FALSE;
    newJ->c[0].cno = cno;
    newJ->c[0].end = TAIL;
    newJ->c[1].cno = newc;
    newJ->c[1].end = HEAD;

    newC->junc[HEAD] = newj;
    newC->junc[TAIL] = C->junc[TAIL];
    C->junc[TAIL] = newj;

    /* tell original TAIL's junction the new contour number */
    J = &Junctions[newC->junc[TAIL]];
    for (i = 0; i < J->n; i++)
	if (J->c[i].cno == cno && J->c[i].end == TAIL)
	    break;

    if (i < J->n) {
	for (; i < J->n - 1; i++)
	    J->c[i] = J->c[i + 1];	/* struct copy */
	J->n--;
    }
    if (closed) {
	/* repeat first point at end of newC */
	newC->p->x[pno] = C->p->x[0];
	newC->p->y[pno] = C->p->y[0];
	newC->p->theta[pno] = C->p->theta[0];
	newC->p->k[pno] = C->p->k[0];

	/* link up */
	newC->junc[TAIL] = C->junc[HEAD];
	J = &Junctions[C->junc[HEAD]];
    } else
	J = &Junctions[newC->junc[TAIL]];

    /* add newc to the right junction */
    if (J->n >= 4) {
	fprintf(stderr, "%s: junction of > 4 contours in SplitContour\n",
		PgmName);
	fprintf(stderr, "%s: May give unpredictable results\n",
		PgmName);
	return;
    } else {
	J->c[J->n].cno = newc;
	J->c[J->n].end = TAIL;
	J->n++;
    }

    /* not closed any more */
    C->p->closed = FALSE;
    newC->p->closed = FALSE;
    newC->p->boundary = C->p->boundary;
    newC->p->contin = C->p->contin;

    /* recompute lengths etc */
    FillFields(cno);
    FillFields(newc);
}

FillFields(cno, sigma)
int     cno;
real    sigma;
{
    real    length = 0;
    real    ksq = 0;
    Contour *c = &Contours[cno];
    PointList *p = c->p;
    int     n = p->n;
    int     i;
    int     side;
    real    prevd;

    /* length & curv sq don't matter for boundary */
    if (p->boundary) {
	c->length = 0.0;
	c->ksq = 0.0;
    } else {
	prevd = 0;
	for (i = 0; i < n - 1; i++) {
	    real    d = Distance(p->x[i], p->y[i],
				 p->x[i + 1], p->y[i + 1]);

	    length += d;
	    ksq += (p->k[i] * p->k[i]) * (prevd + d) / 2.0;
	    prevd = d;
	}
	ksq += p->k[n - 1] * p->k[n - 1] * prevd / 2.0;

	c->length = length;
	c->ksq = ksq;
    }

    if (n > 3) {
	/* point end tangents outward */
	if (c->p->boundary) {
	    /* boundary contours don't have tangents */
	    c->endtan[HEAD] = atan2(p->y[0] - p->y[1],
				    p->x[0] - p->x[1]);
	    c->endtan[TAIL] = atan2(p->y[n - 1] - p->y[n - 2],
				    p->x[n - 1] - p->x[n - 2]);
	} else {
	    c->endtan[HEAD] = Towards(p->theta[0],
				      p->x[3], p->y[3],
				      p->x[0], p->y[0]);
	    c->endtan[TAIL] = Towards(p->theta[n - 1],
				      p->x[n - 4], p->y[n - 4],
				      p->x[n - 1], p->y[n - 1]);

	    /* average region luminances near endpoints */
	    for (side = LEFT; side <= RIGHT; side++) {
		c->endavg[HEAD][side] =
		    EndAverage(p->x[0], p->y[0],
			       c->endtan[HEAD], sigma, side);
		c->endavg[TAIL][side] =
		    EndAverage(p->x[n - 1], p->y[n - 1],
			       c->endtan[TAIL], sigma, side);
	    }
	}
    }
    /* region numbers reg[RIGHT and LEFT] set by SetContourRegions() */
}

CopyPointList(p, pfrom, p2, p2from, n)
PointList *p, *p2;
int     pfrom, p2from;
int     n;
{
    n *= sizeof (real);
    bcopy((char *) &p->x[pfrom], (char *) &p2->x[p2from], n);
    bcopy((char *) &p->y[pfrom], (char *) &p2->y[p2from], n);
    bcopy((char *) &p->theta[pfrom], (char *) &p2->theta[p2from], n);
    bcopy((char *) &p->k[pfrom], (char *) &p2->k[p2from], n);
}

/*
 * get an average pixel value near (x,y) on the "side" side (left or right)
 * of the ray leaving (x,y) at angle theta.
 * 
 * ._. /! / / theta /		averages near words "left" or "right" o (x,y)
 * eft = (x,y) + r{cos,sin}(theta+150) left  .		right = (x,y) +
 * r{cos,sin}(theta-150) . right		average is that point
 * Averaged over a .                  nbhd w/gaussian ellipse eccentricity
 * AVG_ECCENTRICITY and std dev Sigma
 */
#define AVG_ECCENTRICITY 3.0

real
EndAverage(x, y, theta, sigma, side)
real    x, y, theta, sigma;
int     side;
{
    real    ctheta;
    real    x0, y0;		/* point of interest */
    static IMAGE *kernel;
    real    r = 2 * sigma;

    /* center */
    ctheta = (side == LEFT ? theta + RAD(150.0) : theta - RAD(150.0));
    x0 = x + r * cos(ctheta);
    y0 = y + r * sin(ctheta);

    if (kernel == NULL)
	kernel = hvMakeImage(IROUND(4 * sigma), IROUND(4 * sigma), 1, 0, REALPIX);

    MakeMask(kernel, sigma, theta, AVG_ECCENTRICITY);
    return WeightedSum(InputImage, kernel, IROUND(x0), IROUND(y0));
}

real
WeightedSum(I, k, x, y)
IMAGE  *I, *k;
int     x, y;
{
    real    w;
    real    sum = 0;
    real    weightedsum = 0;
    int     xoff = x - (k->width / 2);
    int     yoff = y - (k->height / 2);
    int     i, j;

    /* put center of k at (x,y) and do weighted avg where applic. */
    for (i = 0; i < k->height; i++)
	for (j = 0; j < k->width; j++) {
	    int     ix = j + xoff;
	    int     iy = i + yoff;

	    if (0 <= ix && ix < I->width && 0 <= iy && iy < I->height) {
		w = R_PIX(k, i, j);
		sum += w;
		weightedsum += R_PIX(I, iy, ix) * w;
	    }
	}

    if (sum == 0.0)
	return 0.0;
    return weightedsum / sum;
}

/*
 * Same principle as EndAverage, but get region number on side side of the
 * middle of contour # cno by looking in RegionImage
 */
int
RegionOnOneSide(cno, side)
int     cno, side;
{
    /* use the middle of the contour */
    PointList *p = Contours[cno].p;
    int     n = p->n;
    real    x = p->x[n / 2], y = p->y[n / 2];
    real    x2 = p->x[n / 2 + 1], y2 = p->y[n / 2 + 1];
    real    theta, ctheta;
    
    if (p->boundary)
	theta = atan2(y2 - y, x2 - x);
    else
	theta = Towards(p->theta[n / 2], x, y, x2, y2);

    /* Now look right or left (reverse since X increases downward) */
    ctheta = (side == LEFT ? theta - RAD(90.0) : theta + RAD(90.0));

    return RegionFromRay(x, y, ctheta);
}

int
RegionFromRay(x, y, ctheta)
real    x, y, ctheta;
{
    real    r;
    int     regno;

    for (r = 2.0; r <= 20.0; r = r + .5) {
	real    x0 = x + r * cos(ctheta);
	real    y0 = y + r * sin(ctheta);
	int     ix, iy;

	if (x0 < 0 || x0 > Width - 1 || y0 < 0 || y0 > Height - 1)
	    return 0;		/* region number 0! */

	/* in integer coords */
	ix = IROUND(x0);
	iy = IROUND(y0);

	regno = US_PIX(RegionImage, iy, ix);
	if (regno == EMPTY_PIXEL)
	    return 0;
	if (regno < EDGE_PIXEL)
	    return regno;
    }
    fprintf(stderr, "%s: note: (%g,%g) headed %g degrees gives outside region\n",
	    PgmName, x, y, DEG(ctheta));
    return 0;
}


/*
 * pixel x,y -- just round x and y, clip, and return pixel value;
 */
real
ImagePixel(I, x, y)
IMAGE  *I;
real    x, y;
{
    int     ix = IROUND(x);
    int     iy = IROUND(y);

    if (ix < 0)
	ix = 0;
    else if (ix >= Width)
	ix = Width - 1;

    if (iy < 0)
	iy = 0;
    else if (iy >= Height)
	iy = Height - 1;

    return R_PIX(I, iy, ix);
}

/*
 * ___ draw oblong gaussian (___) rotated by theta, ratio = width/height
 * sigma is std dev in x direction, sigma/ratio is std dev in y direction
 * before rotating.  I must be realpix with height=width.
 */
MakeMask(I, sigma, theta, ratio)
IMAGE  *I;
real    sigma;
real    theta, ratio;
{
    int     i, j;
    double  x, y, b, s;
    double  sint, cost;
    int     size = I->height;

    sincos(-theta, &sint, &cost);
    b = sigma;
    s = b / ratio;

    for (i = 0; i < size; ++i) {
	for (j = 0; j < size; ++j) {

	    /*
	     * Translate x and y so origin is at center of image; rotate by
	     * theta
	     */
	    x = cost * (j - (size - 1) / 2) - sint * (i - (size - 1) / 2);
	    y = sint * (j - (size - 1) / 2) + cost * (i - (size - 1) / 2);
	    R_PIX(I, i, j) = exp(-((x * x) / (b * b) + (y * y) / (s * s)));
	}
    }
}

real
Distance(x, y, x2, y2)
real    x, y, x2, y2;
{
    real    dx = x2 - x;
    real    dy = y2 - y;

    return sqrt(dx * dx + dy * dy);
}

/* point angle theta in the direction towards (x,y) to (x2,y2) */
real
Towards(theta, x, y, x2, y2)
real    theta, x, y, x2, y2;
{
    real    dx = x2 - x;
    real    dy = y2 - y;

    if (cos(theta) * dx + sin(theta) * dy < 0)
	theta += M_PI;		/* about-face! */

    if (theta > M_PI)
	theta -= 2 * M_PI;

    return theta;
}
