/*
 * Copyright (c) 1990,1,2 Mark Nitzberg
 * and President and Fellows of Harvard College
 * All rights reserved.
 */
char    usage[] = "\
usage: sp [-m mu] [-n nu] [-a alpha] [-b beta] [-e epsilon]\n\
	  [-s segfile] [-d debuglevel]\n\
\n\
Finds optimal 2d segmentation with overlaps via exhaustive search.\n\
	-d	 debug levels: 1=list partial seg's; 2=on-the-fly energies\n\
	-m mu	 coeff. of variance term\n\
	-n nu	 coeff. of contour lengths term\n\
	-a alpha coeff. of curvature term\n\
	-b beta  coeff. of angle-penalty term\n\
	-e epsilon  coefficient of area term\n\
	-s segfile  read segmentation from segfile instead of computing it\n\
\n\
The program uses the adjacency graph of regions to search\n\
exhaustively over all connected subgraphs for the optimal\n\
configuration of overlapping regions to minimize a nice functional.\n\
\n\
Output: The segmentation with overlaps that minimizes the functional.\n\
Region numbers are given, with < signifying behindness; and | to\n\
separate connected components of a subgraph; plus a postscript program.\n";

char    FullBlown[] = "\n\
	Input:  edges, regions, and optional drawing info.\n\
	`nodes' are endpoints, implicitly numbered 1 .. N.\n\
\n\
E D G E S :\n\
		i j a b length curvature o1 o2\n\
		...\n\
	one line per edge; implicitly numbered 1 ... E.\n\
	This edge goes from node i to j, region a on the left and b on\n\
	the right (1 ... R), with given arc length and total square of\n\
	curvature.  o1 and o2 are angles, in degrees, of the tangents\n\
	of the contour at nodes i and j.  These tangents must point\n\
	outward at the contour endpoints.\n\
	*** See NOTES below.\n\
	A blank line separates edges from regions.\n\
\n\
R E G I O N S :\n\
		area mean\n\
		...\n\
	the area of the region, and its mean intensity.\n\
	Regions are implicitly numbered 1 ... R.\n\
	A blank line separates regions from optional drawing info.\n\
\n\
D R A W I N G   I N F O .  (optional) :\n\
	First line:\n\
		left bottom right top black white\n\
	giving the user pixel area, and the black and white pixel values.\n\
	Then for each edge, a list of points separated by blanks:\n\
		x1 y1 x2 y2 x3 y3 x4 y4\n\
	  	...\n\
	Continuation lines for a single edge start with white space.\n\
	The points must go from the first node to the second node as specified\n\
	in the initial edge information above.\n\
\n\
NOTES:	* Region number 0 means `outside the image'.\n\
	  Use region 0 to identify edges that comprise the image border.\n\
	* If region 0 never occurs, then region 1 is taken to be the surround.\n\
	  That is, region 1 is the one that touches the picture border everywhere.\n\
	* Loops (edges with the same node at both ends) are forbidden.\n\
	    If your graph has a loop, add a node.\n\
	* Regions _may_ have holes.\n\
	* Several edges _may_ share the same pair of nodes.\n\
	* Nodes _may_ have as few as two edges (corners).\n\
	* An edge shared by two regions which are the same region\n\
	    will be deleted during pre-processing.\n";


#include <stdio.h>
#include <math.h>
#include "graphparts.h"

#define	LINE	512		/* You know; crops up when you least expect */

#define	TRUE	1
#define	FALSE	0

#define	MIN(x,y)	((x)<(y)?(x):(y))
#define	MAX(x,y)	((x)>(y)?(x):(y))

#define	DRAW_AREA_WIDTH 500.0	/* fixed width of draw area */

int     DrawInfoAvail;		/* TRUE if we have info to draw PostScript */

int     RegionOutside;		/* the region number of the region outside
				 * all objects (0 or 1) */

/*
 * SUBGRAPHS are stored as bit-maps, one bit per region.
 * 
 * REGION_BIT(1 ... R) gives the corresponding bit.
 */
typedef unsigned long subgraph_bitmap;

#define	REGION_BIT(region)	(subgraph_bitmap) (1 << ((region)-1))
#define WHOLE_GRAPH		(subgraph_bitmap) ((1<<R) - 1)

/*
 * Adj[r] is non-zero at every REGION_BIT of neighbors of region r.
 */
subgraph_bitmap Adj[MAXR + 1];

/*
 * Edj[r1][r2] gives the index number of the edge shared by regions r1 & r2.
 */
index   Edj[MAXR + 1][MAXR + 1];

int     CurrentPartialSize;	/* # regions (1..R) of current partial result */

int     Debug = 0;		/* 1 for sub-seg lists, 2 for on-the-fly
				 * energies */
real    mu = 1.0;		/* coeff of variance term */
real    nu = .1;		/* coeff of contour length term of energy */
real    alpha = .1;		/* coeff of curvature term */
real    beta = .1;		/* coeff of angle-penalty term */
real    epsilon = 0.001;	/* coeff of area term */
int     auto_alpha = TRUE, auto_nu = TRUE, auto_epsilon = TRUE, auto_beta = TRUE;

/*
 * "Subgraphs" table: best segmentation of each connected subgraph
 */
#define	MAXS	50000		/* max # subgraphs */

struct partial {
    int     size;		/* # regions in this subgraph */
    subgraph_bitmap subgraph;	/* regions in this subgraph */
    index  *o[MAXHOLES];	/* NULL-terminated list of boundaries */

    /* best segmentation:  rear (connected) and front sub-sub-graphs */
    subgraph_bitmap front, rear;
    real    eoptimal;		/* energy of best segmentation */
    real    e_thisregion;	/* length & area term of energy of this
				 * subgraph when used as a single region */
    real    e_area;		/* area term alone of energy of this subgraph
				 * when used as a single region */
}       Subgraphs[MAXS];
int     S;			/* maximum number of subgraph partial results */


char *SegFile;


/*---------- Hi. -----------*/
main(argc, argv)
int     argc;
char  **argv;
{
    double  atof();
    char    c;
    extern char *optarg;
    extern int optind;

    while ((c = getopt(argc, argv, "e:m:n:a:b:d:hs:")) != -1) {

	switch (c) {

	case '?':		/* getopt() arg not in list */
	    fprintf(stderr, usage);
	    exit(1);

	case 'e':
	    epsilon = atof(optarg);
	    auto_epsilon = FALSE;
	    break;
	case 'm':
	    mu = atof(optarg);
	    break;
	case 'n':
	    nu = atof(optarg);
	    auto_nu = FALSE;
	    break;
	case 'a':
	    alpha = atof(optarg);
	    auto_alpha = FALSE;
	    break;
	case 'b':
	    beta = atof(optarg);
	    auto_beta = FALSE;
	    break;
	case 'd':
	    Debug = atoi(optarg);
	    break;
	case 'h':
	    fprintf(stderr, "%s%s", usage, FullBlown);
	    exit(1);
	case 's':
	    SegFile = optarg;
	    break;

	}
    }

    if (optind < argc) {
	if (freopen(argv[optind], "r", stdin) == NULL) {
	    fprintf(stderr, "sp: couldn't read %s\n", argv[optind]);
	    exit(2);
	}
    }
    read_input();

    set_parameters();

    partial_results();

    if (DrawInfoAvail)
	postscript_header();

    printf("%% INPUT:  %d Nodes, %d Edges, %d Regions\n", N, E, R);
    printf("%% PARAMETERS:  alpha=%g nu=%g beta=%g epsilon=%g\n\n",
	   alpha, nu, beta, epsilon);

    /* output in one-liner form */
    if (SegFile == NULL)
	print(WHOLE_GRAPH, 0);

    /* recursive layered postscript output */
    if (DrawInfoAvail)
	postscript();
}

/*
 * input processing
 * 
 * 1. no negative areas, lengths, or curvature-squared-terms. 2. No edges
 * connecting a node to itself. 3. No edges dividing a region from itself.
 */
#include <ctype.h>

read_input()
{
    int     i, j, ei;

    ReadEdges();

    ReadRegions();

    /* make sure there's some of N, E and R */
    if (Debug > 1)
	printf("Read %d nodes, %d edges, %d regions.\n", N, E, R);
    if (N < 1 || E < 1 || R < 1) {
	fprintf(stderr, "Something's not quite right.\n");
	exit(2);
    }

    /*
     * Build adjacency matrix of regions (Adj[r] = neighbors) and edge map
     * which takes 2 regions to the edge they share (Edj[r1][r2]); 0 if not
     * shared.
     */
    for (ei = 1; ei <= E; ei++) {
	i = Edges[ei].region[0];
	j = Edges[ei].region[1];
	if (i == 0 || j == 0)
	    continue;

	Adj[j] |= REGION_BIT(i);
	Adj[i] |= REGION_BIT(j);
	Edj[i][j] = Edj[j][i] = ei;
    }

    /* build boundary contour lists of regions */
    for (i = 1; i <= R; i++)
	trace_boundary(i);

    /* Read optional drawing info */
    ReadDrawInfo();

    /* Dump it all out--for debugging */
    if (Debug > 0)
	PrintInput();
}

PrintInput()
{
    int     i, j;

    printf("\nEdges:\n");
    for (i = 1; i <= E; i++) {
	struct edge *e;

	e = &Edges[i];
	printf("E%d. length %g curvature %g   ",
	       i, e->arclength, e->curvature);
	printf("%3g N%d <-> N%d %3g; ",
	       e->angle[0], (int) e->node[0],
	       (int) e->node[1], e->angle[1]);
	if (e->region[0] == 0)
	    printf("oo");
	else
	    printf("R%d", e->region[0]);
	printf("<->");
	if (e->region[1] == 0)
	    printf("oo\n");
	else
	    printf("R%d\n", e->region[1]);
    }

    printf("\nRegions:\n");
    for (i = 1; i <= R; i++) {
	struct region *r;
	int     b1;
	index  *p;

	r = &Regions[i];
	printf("R%d. area %g mean %g <-> ",
	       i, r->area, r->mean);

	/* neighbors */
	if (Adj[i] == 0)
	    printf("(no neighbors)");

	else
	    for (j = 1; j <= R; j++)
		if (Adj[i] & REGION_BIT(j))
		    printf("R%d ", j);
	printf("\n\t");

	/* boundaries */
	for (b1 = 0; Regions[i].o[b1] != NULL; b1++) {
	    for (p = Regions[i].o[b1]; *p != 0; p++)
		printf("%c%d ", (p - Regions[i].o[b1]) & 1 ? 'E' : 'N', *p);

	    if (Regions[i].o[b1 + 1] != NULL)
		printf("/ ");
	}
	printf("\n");
    }
    printf("\n");
    fflush(stdout);
}

ReadEdges()
{
    int     i, n0, n1, left, right;
    real    o1, o2;
    struct edge *e;
    char    buf[LINE];

    /*
     * if region # 0 appears, then it is the "infinity" region. otherwise,
     * region # 1 is assumed to be "outside" all objects
     */
    RegionOutside = 1;

    N = 0;
    i = 1;
    for (;;) {
	if (gets(buf) == NULL) {
	    printf("Input data incomplete.  Everything OK?\n");
	    exit(2);
	}
	/* comment lines */
	if (buf[0] == '#')
	    continue;

	e = &Edges[i];
	if (sscanf(buf, "%d%d%d%d%lg%lg%lg%lg",
		   &n0,
		   &n1,
		   &left,
		   &right,
		   &e->arclength,
		   &e->curvature,
		   &o1, &o2) != 8)
	    break;

	/* checks */
	if (e->arclength < 0.0) {
	    fprintf(stderr,
		    "Arclength %g < 0 at edge %d.  Sorry.\n",
		    e->arclength, i);
	    exit(2);
	}
	if (e->curvature < 0.0) {
	    fprintf(stderr,
		    "Curvature %g < 0 at edge %d.  Sorry.\n",
		    e->curvature, i);
	    exit(2);
	}
	if (n0 == n1) {
	    fprintf(stderr,
		    "Edge %d can't link node N%d to itself.  Sorry.\n",
		    i, n0);
	    exit(2);
	}
	e->node[0] = n0;
	e->node[1] = n1;
	e->region[0] = left;
	e->region[1] = right;
	if (left == 0 || right == 0)
	    RegionOutside = 0;

	while (o1 < 0.0)
	    o1 += 360.0;
	while (o1 >= 360.0)
	    o1 -= 360.0;
	e->angle[0] = o1;

	while (o2 < 0)
	    o2 += 360.0;
	while (o2 >= 360.0)
	    o2 -= 360.0;
	e->angle[1] = o2;

	/* assume largest node # = # nodes */
	N = MAX(N, MAX(n0, n1));
	i++;
    }
    E = i - 1;
}

ReadRegions()
{
    /* read Regions until end-of-file or line w/o 2 numbers */
    int     i;
    char    buf[LINE];
    struct region *r;

    i = 1;
    for (;;) {
	if (gets(buf) == NULL)
	    break;

	/* comment lines */
	if (buf[0] == '#')
	    continue;

	r = &Regions[i];

	if (sscanf(buf, "%lg %lg",
		   &r->area, &r->mean) != 2)
	    break;

	if (r->area < 0.0) {
	    fprintf(stderr,
		    "Area %g < 0 at region %d.  Sorry.\n",
		    r->area, i);
	    exit(2);
	}
	i++;
    }
    R = i - 1;
}

ReadDrawInfo()
{
    int     i, nvec;
    struct edge *e;
    char    buf[LINE], *p;
    vector  veclist[MAXCHAIN];
    real    width, height;

    do {
	if (gets(buf) == NULL) {
	    printf("No drawing info for edges; no PostScript output for you.\n");
	    return;
	}
    } while (buf[0] == '#');

    if (sscanf(buf, "%lg%lg%lg%lg%lg%lg",
	       &DrawInfo.left,
	       &DrawInfo.bottom,
	       &DrawInfo.right,
	       &DrawInfo.top,
	       &DrawInfo.black,
	       &DrawInfo.white) != 6) {
	printf("Drawing info looks wrong; no postscript output.\n");
	return;
    }
    /* get oriented:  which way is up, and how high? */
    width = DrawInfo.right - DrawInfo.left;
    height = DrawInfo.top - DrawInfo.bottom;

    DrawInfo.xscale = DRAW_AREA_WIDTH / width;	/* negative if necessary */
    DrawInfo.yscale = DrawInfo.xscale;
    if (width * height < 0)
	DrawInfo.yscale = -DrawInfo.yscale;

    /* draw_coordinate_y = (user_y - yoff) * yscale */
    DrawInfo.xoff = DrawInfo.left;
    DrawInfo.yoff = DrawInfo.bottom;

    /* draw area height */
    DrawInfo.H = fabs(height * DrawInfo.yscale);

    /* page height of one picture */
    DrawInfo.Hp = 0.126 * DrawInfo.H - 0.000036 * DrawInfo.H * DrawInfo.H;

    /* page separation height between pictures */
    DrawInfo.Hs = 8;		/* about .111 inches */

    nvec = 0;
    e = &Edges[1];

    for (i = 0; gets(buf) != NULL;) {

	/* comment lines */
	if (buf[0] == '#' || buf[0] == '\0')
	    continue;

	/* start a new edge at 1 or when non-continuation */
	if (i == 0 || !isspace(buf[0])) {
	    if (i > 0) {
		/* store prev. chain code with the edge */
		e->chain = (vector *) malloc(sizeof (vector) * nvec);
		bcopy((char *) veclist, e->chain, sizeof (vector) * nvec);
		e->nchain = nvec;
	    }
	    e = &Edges[++i];
	    nvec = 0;
	}
	/* read x y */
	p = buf;

	while (sscanf(p, "%lg%lg",
		      &veclist[nvec].x,
		      &veclist[nvec].y) == 2) {

	    /* skip over two numbers */
	    while (isspace(*p))
		p++;
	    while (isdigit(*p) || *p == '.' || *p == '+' || *p == '-' || *p == 'e')
		p++;
	    while (isspace(*p))
		p++;
	    while (isdigit(*p) || *p == '.' || *p == '+' || *p == '-' || *p == 'e')
		p++;
	    nvec++;
	}

	if (nvec == 0) {
	    printf("Trivial chain code for E%d; no postscript output.\n", i);
	    return;
	}
    }

    /* we need to draw each edge */
    if (i < E) {
	printf("Incomplete drawing info; no postscript output.\n");
	return;
    }
    /* store prev. chain code with the edge */
    e->chain = (vector *) malloc(sizeof (vector) * nvec);
    bcopy((char *) veclist, e->chain, sizeof (vector) * nvec);
    e->nchain = nvec;
    DrawInfoAvail = TRUE;
}

/*
 * find the node,edge,node,edge lists of region number r
 * 
 * Strategy:  make a pile of edges that share this region with another; mark the
 * edges unvisited.
 * 
 * loop while there's an unvisited edge: find the node touched by unvisited
 * edges with the most edges start with that node and any unvisited edge it
 * touches, and trace until you hit it again, marking the edges as visited.
 * Once traced, this is a contour; add it to the list of boundary lists.
 * 
 * NULL-terminate the list of boundary lists for r.
 */
trace_boundary(r)
int     r;
{
    int     i, visited[MAXE + 1], ne, e0, direction;
    index   oneresult[MAXE * 2], resulti, hole, node;
    real    angle, closest_angle, v;

    hole = 0;
    ne = 0;
    for (i = 1; i <= E; i++) {
	visited[i] = TRUE;
	if (Edges[i].region[0] == r || Edges[i].region[1] == r) {
	    visited[i] = FALSE;
	    ne++;
	}
    }

    while (ne > 0) {

	/* just start at an edge; keep your right hand on r */
	for (i = 1; i <= E; i++)
	    if (!visited[i])
		break;

	e0 = i;			/* starting edge */
	resulti = 0;

NextEdgeInContour:

	if (Edges[e0].region[0] == r)
	    direction = 1;	/* go from 1 to 0 to keep r on right */
	else
	    direction = 0;	/* go from 0 to 1 to keep r on right */

	/* node, edge */
	oneresult[resulti++] = Edges[e0].node[direction];
	oneresult[resulti++] = e0;

	visited[e0] = TRUE;
	ne--;
	if (ne == 0) {
	    oneresult[resulti++] = 0;
	    Regions[r].o[hole++] = indexlist_dup(oneresult);
	    break;
	}
	/* trace from other node */
	node = Edges[e0].node[1 - direction];	/* other node */
	angle = Edges[e0].angle[1 - direction];	/* its tangent angle */

	/*
	 * find edge touching node node, such that region r is on the right
	 * leaving from node node, and with smallest counterclockwise angle
	 * difference
	 */
	closest_angle = 359.9999999;	/* anything is better */
	e0 = 0;			/* closest edge */

	for (i = 1; i <= E; i++) {
	    if (!visited[i]) {
		if (Edges[i].node[0] == node)
		    direction = 0;
		else if (Edges[i].node[1] == node)
		    direction = 1;
		else
		    continue;

		/*
		 * does edge i, starting from node direction, have region r
		 * on the right?
		 */
		if (Edges[i].region[1 - direction] != r)
		    continue;

		/* angle */
		v = Edges[i].angle[direction] - angle;
		if (v < 0)
		    v += 360;
		if (v >= 360)
		    v -= 360;
		if (v < closest_angle) {
		    closest_angle = v;
		    e0 = i;
		}
	    }
	}

	/* if there was one, continue with this e0 */
	if (e0 != 0)
	    goto NextEdgeInContour;

	/* otherwise this contour's done */
	oneresult[resulti++] = 0;
	Regions[r].o[hole++] = indexlist_dup(oneresult);
    }

    Regions[r].o[hole] = NULL;
    return;
}

/*
 * angle penalty given edge, node, edge
 */
real
penalty(ei1, n, ei2)
index   ei1, n, ei2;
{
    int     end1, end2;
    struct edge *e1, *e2;
    index   temp;
    real    angle;

    if (ei1 > ei2) {
	temp = ei1;
	ei1 = ei2;
	ei2 = temp;
    }
    e1 = &Edges[ei1];
    e2 = &Edges[ei2];

    /* double-check */
    if (e1->node[0] == n)
	end1 = 0;
    else if (e1->node[1] == n)
	end1 = 1;
    else {
	printf("Angle penalty E%d-/-N%d-E%d.  Sorry\n",
	       ei1, n, ei2);
	abort();
    }

    /* double-check */
    if (e2->node[0] == n)
	end2 = 0;
    else if (e2->node[1] == n)
	end2 = 1;
    else {
	printf("Angle penalty E%d-N%d-/-E%d.  Sorry\n",
	       ei1, n, ei2);
	abort();
    }

    /* no penalty if e1 or e2 is image border */
    if (image_boundary(e1) || image_boundary(e2))
	return 0.0;

    /*
     * penalty is 1 + cos of angle v1.v2 = |v1||v2|cos of angle
     */

    /*
     * return	e1->end_tangent[end1].x * e2->end_tangent[end2].x +
     * e1->end_tangent[end1].y * e2->end_tangent[end2].y + 1.0;
     */
    angle = e1->angle[end1] - e2->angle[end2] - 180.0;
    if (angle < -180.0)
	return fabs(M_PI * (angle + 360.0) / 180.0);
    else
	return fabs(M_PI * angle / 180.0);
}

set_parameters()
{
    real    a, ma, msqa, v;
    double  sqrt();
    int     i;

    if (auto_alpha || auto_nu || auto_epsilon) {

	a = ma = msqa = 0;

	for (i = 1; i <= R; i++) {
	    v = Regions[i].area;
	    a += v;
	    v *= Regions[i].mean;
	    ma += v;
	    msqa += v * Regions[i].mean;
	}
	/* variance */
	v = (msqa - ma * ma / a) / a;

	/* alpha scale of variance * length / (# regions^2) */
	if (auto_alpha)
	    alpha = v * sqrt(a) / (R * R) * .01;

	/* nu scale of variance * length / (# regions^2) */
	if (auto_nu)
	    nu = v * sqrt(a) / (R * R) * .01;

	/* beta scale of variance * length / (# regions^2) */
	if (auto_beta)
	    beta = 5 * v * sqrt(a) / (R * R) * .01;

	/* epsilon on scale of variance / (# regions^3) */
	if (auto_epsilon)
	    epsilon = v / (R * R * R) * .001;
    }
}

partial_results()
{
    int     i, pos;
    index   region;
    subgraph_bitmap s, t;

    /* start with 1-region subgraphs */
    for (region = 1; region <= R; region++) {
	s = REGION_BIT(region);
	pos = new_partial(s, region, NULL);
    }

    if (SegFile != NULL)
	return;

    if (Debug > 0)
	print_number_partials(1);

    for (CurrentPartialSize = 2;
	 CurrentPartialSize <= R; CurrentPartialSize++) {

	/* form all connected subgraphs of size "CurrentPartialSize" */
	for (i = 0; i < S; i++) {

	    /* for each subgraph of size "CurrentPartialSize-1" */
	    if (Subgraphs[i].size != CurrentPartialSize - 1)
		continue;

	    s = Subgraphs[i].subgraph;

	    /* build a list of all neighbors of s not in s */
	    t = 0;
	    for (region = 1; region <= R; region++)
		if (REGION_BIT(region) & s)
		    t |= Adj[region];
	    t &= ~s;

	    /* now add each of those neighbors to s */
	    for (region = 1; region <= R; region++) {
		if (REGION_BIT(region) & t) {
		    pos =
			new_partial(s | REGION_BIT(region),
				    region, &Subgraphs[i]);

		    /*
		     * pos = place inserted in Subgraphs if inserted before
		     * i, move over.
		     */
		    if (pos <= i)
			i++;
		}
	    }
	}
	if (Debug > 0)
	    print_number_partials(CurrentPartialSize);
    }
}

/*
 * print number of partials of a given size
 */
print_number_partials(size)
int     size;
{
    int     i, total;

    total = 0;
    for (i = 0; i < S; i++)
	if (Subgraphs[i].size == size)
	    total++;

    printf("----------- %d subgraph%c with %d region%c -----------\n",
	   total, total != 1 ? 's' : '\0', size, size != 1 ? 's' : '\0');

    return;
}

/*
 * print one line for a given partial
 */
print_one_partial(p)
struct partial *p;
{
    index   region;

    for (region = R; region >= 1; region--)
	putchar((p->subgraph & REGION_BIT(region)) ? '1' : '0');

    printf(" as ");

    for (region = R; region >= 1; region--)
	putchar((p->rear & REGION_BIT(region)) ? '1' : '0');

    printf(" < ");
    for (region = R; region >= 1; region--)
	putchar((p->front & REGION_BIT(region)) ? '1' : '0');

    printf(" Ereg=%g E=%g\n", p->e_thisregion, p->eoptimal);

    /* make sure it goes out even if logging to a file */
    fflush(stdout);
}

/* return pointer to entry in "Partial" with subgraph r, or NULL if not found */
struct partial *
lookup(r)
subgraph_bitmap r;
{
    /* binary search in partial results for subgraph r */
    int     low, high, mid;

    if (S == 0)
	return NULL;

    if (r == 0) {
	printf("Lookup called with subgraph 0.\n");
	abort();
	return NULL;
    }
    low = 0;
    high = S - 1;
    while (low <= high) {
	mid = (low + high) / 2;

	/* compare */
	if (r < Subgraphs[mid].subgraph)
	    high = mid - 1;
	else if (r > Subgraphs[mid].subgraph)
	    low = mid + 1;
	else
	    return &Subgraphs[mid];
    }
    return NULL;
}

/* insert new partial result into Subgraphs and return index */
int
insert_partial(p)
struct partial *p;
{
    /* binary search to find insertion point */
    int     low, high, mid, i;
    subgraph_bitmap r;
    struct partial save;

    r = p->subgraph;

    if (Debug > 0)
	print_one_partial(p);

    if (r == 0) {
	printf("Insert called with subgraph 0.\n");
	abort();
	return NULL;
    }
    low = 0;
    high = S - 1;
    while (low <= high) {
	mid = (low + high) / 2;

	/* compare */
	if (r < Subgraphs[mid].subgraph)
	    high = mid - 1;
	else if (r > Subgraphs[mid].subgraph)
	    low = mid + 1;
	else {
	    printf("Insert: subgraph 0x%x already in table.\n", r);
	    abort();
	}
    }

    /* insert at low */
    save = *p;			/* struct assign */
    for (i = S; i > low; i--)
	Subgraphs[i] = Subgraphs[i - 1];
    Subgraphs[low] = save;
    S++;
    return low;
}

common_edge(b1, b2)
index  *b1, *b2;
{
    index  *b, c, d;

    for (; (c = *++b1) != 0; b1++)
	for (b = b2; (d = *++b) != 0; b++)
	    if (c == d)
		return TRUE;
    return FALSE;
}

/*
 * splice(new, old, result) -- splice boundaries old and new into result. Old
 * may be NULL.
 */
splice(new, old, result)
index **new, **old, **result;
{
    int     n, none;
    index **oldj, **newj, **r;

    if (old == NULL) {
	while ((*result++ = *new++) != 0)
	    /* coast */;
	return;
    }
    /* find one boundary in each of old and new that will join */
    n = 0;			/* index into result */
    none = TRUE;

    for (oldj = old; *oldj; oldj++) {
	for (newj = new; *newj; newj++) {
	    if (common_edge(*oldj, *newj)) {
		join(*oldj, *newj, result, &n);
		none = FALSE;
		goto break2;
	    }
	}
    }

break2:
    if (none) {
	printf("Splice() called with disjoint regions.\n");
	abort();
    }
    /* add all but *oldj and *newj to the result */
    for (r = old; *r; r++)
	if (r != oldj)
	    result[n++] = *r;

    for (r = new; *r; r++)
	if (r != newj)
	    result[n++] = *r;
    result[n] = NULL;
}

/*
 * reverse b[1]...b[length(b)-1], leaving b[0] in place.
 * 
 * This is because of the node-edge-node-edge business.
 */
reverse_boundary(b)
index  *b;
{
    int     low, hi;
    index   temp;

    if (Debug > 3) {
	printf("Reverse: ");
	print_boundary(b);
    }
    low = 1;
    hi = indexlist_size(b) - 1;
    while (low < hi) {
	/* swap low, hi */
	temp = b[low];
	b[low] = b[hi];
	b[hi] = temp;
	low++;
	hi--;
    }

    if (Debug > 3) {
	printf("gives: ");
	print_boundary(b);
	printf("\n");
    }
}


/*
 * join(b1, b2, result, np) -- join 2 boundaries Tries to join boundaries b1
 * and b2. If joined, places resulting boundaries at result[*np, *np+1, ...]
 * and increments *np by the number of resulting boundaries. All new
 * boundaries are in allocated memory, thank you.
 */
join(b1, b2, result, np)
index  *b1, *b2, **result;
int    *np;
{
    int     i, j;

    index  *b[2];
    int     length[2];
    index   map[2][MAXE * 2];
    index   visited[2][MAXE * 2];	/* nonzero if visited */

    index   oneresult[MAXE * 2];/* one boundary */
    int     resulti, matched, reversed, boundary;
    int     npb4;

    if (Debug > 5) {
	printf("Joining ... ");
	print_boundary(b1);
	printf("to ");
	print_boundary(b2);
	printf("\n");
	npb4 = *np;
    }
    matched = 0;
    reversed = 0;
    b[0] = b1;
    b[1] = b2;
    length[0] = indexlist_size(b[0]);
    length[1] = indexlist_size(b[1]);
again:

    /*
     * Edge at b[n][i] matches b[n xor 1][map[n][i]] Since i indexes an edge
     * in a "node,edge,node,..." list, i is odd, and hence non-zero. We
     * initialize the m arrays to zero to mean no match.
     */
    for (i = 0; i < MAXE * 2; i++) {
	map[0][i] = map[1][i] = 0;
	visited[0][i] = visited[1][i] = FALSE;
    }

    /* lists are node-edge-node-edge... */
    for (i = 1; i < length[0]; i += 2) {
	for (j = 1; j < length[1]; j += 2) {
	    if (b[0][i] == b[1][j]) {
		matched++;

		/* common edge: reverse if necessary */
		if (b[0][i - 1] != b[1][j - 1]) {
		    reversed++;
		    if (matched + reversed > 2) {
			printf("\
There's a twist joining 2 subgraphs.  N%d-E%d-N%d\n",
			       b[0][i - 1], b[0][i], b[0][i + 1]);
			abort();
		    }
		    reverse_boundary(b[1]);
		    goto again;
		}
		/* cross-index */
		map[0][i] = j;
		map[1][j] = i;

		/* only one matching j per i */
		break;
	    }
	}
    }

    if (matched == 0) {
	printf("Join() called with disjoint boundaries.\n");
	abort();
    }
    for (;;) {
	/* find 1st non-matching unvisited edge */
	for (i = 1; i < length[0]; i += 2)
	    if (map[0][i] == 0 && visited[0][i] == FALSE)
		break;

	for (j = 1; j < length[1]; j += 2)
	    if (map[1][j] == 0 && visited[1][j] == FALSE)
		break;

	if (i >= length[0] && j >= length[1]) {
	    if (Debug > 5) {
		printf("Result: ");
		for (i = npb4; i < *np; i++) {
		    print_boundary(result[i]);
		    if (i < *np - 1)
			printf("; ");
		}
		printf("\n");
	    }
	    return;
	}
	if (i < length[0])
	    boundary = 0;
	else {
	    boundary = 1;
	    i = j;
	}

	resulti = 0;		/* index in oneresult */

	/* this macro actually makes debugging easier */

#define	BUMP_i	/* move i in the right direction */\
		if (boundary == 0) {	/* forward */\
			if (i == length[boundary]-1)\
				i = 1;\
			else\
				i += 2;\
		} else {\
			if (i == 1)\
				i = length[boundary]-1;\
			else\
				i -= 2;\
		}		/* end BUMP_i */


	while (!visited[boundary][i]) {

	    /* previous node */
	    if (boundary == 0) {
		/* moving forward--subtract 1 */
		oneresult[resulti++] = b[boundary][i - 1];
	    } else {
		/* moving backward--add 1 & possibly wrap */
		if (i == length[boundary] - 1)
		    oneresult[resulti++] = b[boundary][0];
		else
		    oneresult[resulti++] = b[boundary][i + 1];
	    }
	    /* now the edge */
	    oneresult[resulti++] = b[boundary][i];	/* edge */
	    visited[boundary][i] = TRUE;

	    BUMP_i;

	    /* now do the hop if we've hit a shared edge */
	    if (map[boundary][i] != 0) {

		i = map[boundary][i];	/* matching edge */
		boundary ^= 1;	/* on other boundary */

		BUMP_i;

		/* if we've hit another shared edge, hop back */
		if (map[boundary][i] != 0) {
		    i = map[boundary][i];	/* matching edge */
		    boundary ^= 1;	/* on other boundary */

		    BUMP_i;
		}
	    }
	}

	/* bingo.  done with that boundary */
	oneresult[resulti] = 0;
	result[(*np)++] = indexlist_dup(oneresult);
    }
}

print_boundary(b)
index  *b;
{
    int     i;

    for (i = 0; *b; b++, i++)
	printf("%c%d ", i & 1 ? 'E' : 'N', *b);
}


/*
 * Compute and insert a new partial result for a given connected subgraph.
 * Returns index into Subgraphs where inserted.
 */
int
new_partial(r, added_region, prev)
subgraph_bitmap r;
index   added_region;
struct partial *prev;		/* NULL, or subgraph before added_region
				 * added */
{
    struct partial *result;
    int     i, size;
    unsigned long counter;
    subgraph_bitmap p, q, t;
    index   indexlist[MAXR + 1];
    real    contour_energy();

    /* first things first:  is r already there? */
    if (lookup(r) != NULL)
	return S;		/* "inserted beyond end" (hack alert) */

    if (S == MAXS) {
	printf("Bad news:  too many (%d) subgraphs.\n", MAXS);
	exit(2);
    }
    result = &Subgraphs[S];	/* use the one past the end */

    result->subgraph = r;

    /* area energy */
    result->e_area = 0.0;
    if (prev != NULL)
	result->e_area = prev->e_area;

    result->e_area += Regions[added_region].area * epsilon;

    /* whole region energy */
    result->e_thisregion = result->e_area;

    /*
     * Splice the boundaries of the previous connected subgraph to the
     * boundaries of the single added region.
     */
    if (prev == NULL) {
	result->size = 1;
	result->front = 0;
	result->rear = REGION_BIT(added_region);
	splice(Regions[added_region].o, NULL, result->o);

	/* 1 region -- this is the optimal segmentation */
	result->e_thisregion += contour_energy(result->o);
	result->eoptimal = result->e_thisregion;
	return insert_partial(result);
    }
    size = result->size = prev->size + 1;
    build_indexlist(r, indexlist);

    splice(Regions[added_region].o, prev->o, result->o);
    result->e_thisregion += contour_energy(result->o);

    /* find optimal segmentation energy */
    result->eoptimal = 1.0e20;	/* start huge so anything's better */

    /* for every proper sub-subgraph q of r "foreground" NOTE q empty ok */
    for (counter = 0; counter <= (1 << size) - 2; counter++) {

	/*
	 * map low n bits of counter to n region bits of r.
	 */
	q = 0;
	for (i = 0; i < size; i++)
	    if (counter & (1 << i))
		q |= REGION_BIT(indexlist[i]);

	/*
	 * for every connected p < r such that r\p < q
	 */

	/* compute best "background" p */
	for (i = 0; i < S; i++) {
	    p = Subgraphs[i].subgraph;

	    /* This loop handles p != r; is p < r? */
	    if ((p & r) != p)	/* no, not a sub-subgraph */
		continue;

	    /* check that r\p < q */
	    t = (r & ~p);
	    if ((t & q) != t)
		continue;

	    /* split off another function */
	    compute_energy(q, p, result, Subgraphs[i].e_thisregion);
	}
	compute_energy(q, r, result, result->e_thisregion);
    }
    return insert_partial(result);
}

/*
 * compute_energy(foreground, background, result, background_energy)
 * 
 * NOTE: foreground may be empty.
 */
compute_energy(q, p, result, background_energy)
subgraph_bitmap q, p;
struct partial *result;
real    background_energy;
{
    struct partial *smaller;
    int     j, size, com, ncom;
    subgraph_bitmap t;
    char    labels[MAXR + 1];	/* for connected components */
    index   indexlist[MAXR + 1];
    real    energy, mma, ma, a, v;

    energy = 0;

    if (q != 0) {
	/* q is in front--divide into connected components */
	size = build_indexlist(q, indexlist);
	ncom = concom(indexlist, labels);

	/* get energy from each component of q */
	for (com = 1; com <= ncom; com++) {
	    t = 0;
	    for (j = 0; j < size; j++)
		if (labels[j] == com)
		    t |= REGION_BIT(indexlist[j]);

	    /* t is one component */
	    smaller = lookup(t);
	    if (smaller == NULL) {
		printf("Uh-oh. Subgraph 0x%x missing.\n",
		       t);
		abort();
	    }
	    energy += smaller->eoptimal;
	}
    }
    if (Debug > 1 && CurrentPartialSize == R) {
	/* print it even though it's not optimal */

	for (j = 1; j <= R; j++)
	    if (p & REGION_BIT(j))
		printf("%d ", j);
	if (q != 0) {
	    printf("< ");
	    if (ncom == 1)
		print(q, 1);
	    else {
		printf("( ");
		for (com = 1; com <= ncom; com++) {
		    t = 0;
		    for (j = 0; j < size; j++)
			if (labels[j] == com)
			    t |= REGION_BIT(indexlist[j]);
		    print(t, 1);
		    if (com < ncom)
			printf("| ");
		}
		printf(")");
	    }
	}
    }
    energy += background_energy;/* area & perim p */

    /* add energy from variance of visible parts of p */
    t = (p & ~q);
    size = build_indexlist(t, indexlist);
    mma = ma = a = 0;
    for (j = 0; j < size; j++) {
	a += (v = Regions[indexlist[j]].area);
	ma += (v *= Regions[indexlist[j]].mean);
	mma += v * Regions[indexlist[j]].mean;
    }
    energy += mu * (mma - ma * ma / a);

    /*
     * if we've bettered the result, remember it
     */
    if (energy < result->eoptimal) {
	result->eoptimal = energy;
	result->front = q;
	result->rear = p;
    }
    if (Debug > 1 && CurrentPartialSize == R)
	printf(" E=%g\n", energy);
}

real
contour_energy(o)
index **o;
{
    index **op, *b;
    real    e;
    int     i, size;

    e = 0.0;

    for (op = o; *op != NULL; op++) {

	/* start with edge */
	size = indexlist_size(b = *op);
	for (i = 1; i < size - 1; i += 2)
	    if (!image_boundary(&Edges[b[i]]))
		/* this edge, and edge-node-edge angle */
		e += nu * Edges[b[i]].arclength
		    + alpha * Edges[b[i]].curvature
		    + beta * penalty(b[i], b[i + 1], b[i + 2]);

	/* last one wraps to beginning */
	if (!image_boundary(&Edges[b[i]]))
	    e += nu * Edges[b[i]].arclength
		+ alpha * Edges[b[i]].curvature
		+ beta * penalty(b[i], b[0], b[1]);
    }
    return e;
}

/* TRUE if this edge is on the image boundary */
int
image_boundary(e)
struct edge *e;
{
    return (e->region[0] == 0 || e->region[1] == 0);
}

/*
 * print a subgraph.  Call it with level 0.
 */
print(r, level)
subgraph_bitmap r;
int     level;
{
    struct partial *result;
    int     i, j, size, com, ncom;
    subgraph_bitmap t;
    char    labels[MAXR + 1];	/* for connected components */
    index   indexlist[MAXR + 1];

    result = lookup(r);
    if (result == NULL) {
	printf("Print couldn't find subgraph 0x%x (level %d).\n",
	       r, level);
	return;
    }
    if (level == 0)
	printf("%% Optimal Segmentation Energy %g: ", result->eoptimal);

    /* no, do the rearmost, & then recurse */
    for (i = 1; i <= R; i++)
	if (result->rear & REGION_BIT(i))
	    printf("%d ", i);

    if (result->front == 0) {
	if (level == 0)
	    printf("\n");
	return;
    }
    /* recurse w/connected components of front */
    size = build_indexlist(result->front, indexlist);
    ncom = concom(indexlist, labels);

    printf("< ");
    if (ncom == 1) {
	print(result->front, level + 1);
    } else {
	printf("( ");

	/* print each component of result->front */
	for (com = 1; com <= ncom; com++) {
	    t = 0;
	    for (j = 0; j < size; j++)
		if (labels[j] == com)
		    t |= REGION_BIT(indexlist[j]);

	    /* t is one component */
	    print(t, level + 1);
	    if (com < ncom)
		printf("| ");
	}
	printf(") ");
    }

    if (level == 0)
	printf("\n");
}

/*
 * print a postscript version of the output using the layered look.
 */
int     StackHeight = 0;	/* height of picture stack */

int     UncollapsedHeight = 0;
subgraph_bitmap PrintStack[MAXR + 1];
short   PrintLevel[MAXR + 1];	/* PrintLevel[lev] is the actual level on
				 * which to print */

char    PSHead[] = "\
%%!PS-Adobe-1.0\n\
%%%%Pages: 1\n\
%%%%BoundingBox: %g %g %g %g\n\
%%%%EndComments\n\
\n\
/Vx -100 def /Vy -1750 def /Vz 950 def /Yp -1000 def\n\
/Zimbed 325 def\n\
/xmap { /y0 exch def /x0 exch def\n\
Yp Vy sub y0 Vy sub div x0 Vx sub mul Vx add } def\n\
/ymap { /y0 exch def /x0 exch def\n\
Yp Vy sub y0 Vy sub div Zimbed Vz sub mul Vz add } def\n\
/ProjectiveMap {1 index 1 index xmap 3 1 roll ymap} def\n\
/M { ProjectiveMap moveto } def\n\
/L { ProjectiveMap lineto } def\n\
/C {3 {ProjectiveMap 6 2 roll} repeat curveto} def\n\
/InitPictureStack { /H exch def /Hs 8 def\n\
H H mul -0.000036 mul 0.126 H mul add /Hp exch def\n\
144 -620 translate gsave } def\n\
/PictureMoveto { grestore gsave Hs Hp add mul 0 exch translate } def\n\
/EndPictureStack { grestore } def\n\
/FramePicture { newpath 0 0 M 500 0 L 1 setlinewidth stroke\n\
newpath 500 0 M 500 H L 0 H L 0 0 L .2 setlinewidth stroke } def\n\
/FramePath { newpath 0 0 M 500 0 L 500 H L 0 H L closepath } def\n\
\n\
%%%%EndProlog\n\
%%%%Page: 1 1\n\n";

postscript_header()
{
    if (SegFile)
	ReadSegFile();
    else
	ComputePrintLevels();

    /* bounding box */
    printf(PSHead, 70.0, 85.0, 306.0,
	 85. + StackHeight * DrawInfo.Hp + (StackHeight - 1) * DrawInfo.Hs);
}

ComputePrintLevels()
{
    int     level, printlevel;
    subgraph_bitmap map;
    int     bit;

    bzero((char *) PrintStack, sizeof (PrintStack));

    FillPrintStack(WHOLE_GRAPH, 0);

    printlevel = 0;
    map = 0;			/* bitmap accumulating at printlevel */
    for (level = 0; level < UncollapsedHeight; level++) {
	if ((map & PrintStack[level]) != 0) {
	    printlevel++;
	    map = 0;
	}
	map |= PrintStack[level];
	PrintLevel[level] = printlevel;
    }
    StackHeight = printlevel + 2;

    /* DEBUG */
    fprintf(stderr, "Stack:\n");
    for (level = UncollapsedHeight; level-- > 0;) {
	fprintf(stderr, "level %d:", PrintLevel[level]);
	for (bit = 1; bit <= R; bit++)
	    if (REGION_BIT(bit) & PrintStack[level])
		fprintf(stderr, " %d", bit);
	fprintf(stderr, "\n");
    }
}

ReadSegFile()
{
    int     printlevel;
    int     n, sofar, region;
    FILE   *f;
    char    buf[512];

    if ((f = fopen(SegFile, "r")) == NULL) {
	fprintf(stderr, "sp: can't read segmentation file %s\n",
		SegFile);
	exit(2);
    }
    printlevel = 0;
    bzero((char *) PrintStack, sizeof (PrintStack));
    while (fgets(buf, 512, f) != NULL) {
	if (buf[0] == '#' || buf[0] == '\n')
	    continue;
	if (printlevel == MAXR) {
	    fprintf(stderr,
		    "sp: file %s has > %d levels--aborting\n",
		    SegFile, MAXR);
	    exit(2);
	}
	sofar = 0;
	while (sscanf(buf+sofar, "%d%n", &region, &n) == 1) {
	    sofar += n;
	    PrintStack[printlevel] |= REGION_BIT(region);
	}
	PrintLevel[printlevel] = printlevel;
	if (PrintStack[printlevel] != 0)
	    printlevel++;
    }
    fclose(f);

    StackHeight = printlevel + 1;	/* for the full picture on bottom */
}

FillPrintStack(r, level)
subgraph_bitmap r;
int     level;
{
    struct partial *result;
    int     j, size, com, ncom;
    subgraph_bitmap t;
    char    labels[MAXR + 1];	/* for connected components */
    index   indexlist[MAXR + 1];

    result = lookup(r);
    if (result == NULL) {
	fprintf(stderr, "FillPrintStack no partial subgraph 0x%x (level %d).\n",
		r, level);
	return;
    }
    /* track number of overlays in stack */
    if (UncollapsedHeight < level + 1)
	UncollapsedHeight = level + 1;

    PrintStack[level] |= result->rear;

    if (result->front != 0) {
	/* recurse w/connected components of front */
	size = build_indexlist(result->front, indexlist);
	ncom = concom(indexlist, labels);

	if (ncom == 1) {
	    FillPrintStack(result->front, level + 1);
	} else {
	    /* print each component of result->front */
	    for (com = 1; com <= ncom; com++) {
		t = 0;
		for (j = 0; j < size; j++)
		    if (labels[j] == com)
			t |= REGION_BIT(indexlist[j]);

		/* t is one component */
		FillPrintStack(t, level + 1);
	    }
	}
    }
}

postscript()
{
    subgraph_bitmap rear, front;
    int     i, level;
    int     j, size, touch, com, ncom;
    subgraph_bitmap t;
    char    labels[MAXR + 1];	/* for connected components */
    index   indexlist[MAXR + 1];

    /* level 0, the whole picture */
    printf("%g InitPictureStack\n", DrawInfo.H);
    printf("0 PictureMoveto FramePicture\n");

    /* see if any edge touches the border */
    touch = FALSE;
    for (j = 1; j <= E; j++)
        if (image_boundary(&Edges[j]))
    	    touch = TRUE;

    /* draw the original picture down at level 0,no outlines */
    if (!touch) {
        /* region 1 touches the frame everywhere */
        printf("FramePath\n");
        postscript_fill(REGION_BIT(1));
    }
    for (j = R; j >= (touch ? 1 : 2); j--) {
        postscript_contour(REGION_BIT(j), FALSE);
        postscript_fill(REGION_BIT(j));
    }

    /*
     * From rear to front, print a postscript path, fill with the mean color
     * of the visible part.
     */
    for (level = 0; level < StackHeight-1; level++) {

	/* the real picture stack starts at 1 */
	printf("%d PictureMoveto FramePicture\n", PrintLevel[level] + 1);

	rear = PrintStack[PrintLevel[level]];
	front = 0;
	for (i = level+1; i < StackHeight-1; i++)
	    front |= PrintStack[PrintLevel[i]];

	if (rear + 1 == (1 << R)) {
	    /* if rear is entire domain, whole Frame path is the path */
	    printf("FramePath\n");
	    postscript_fill(rear & ~front);
	} else {
	    /* split & draw */
	    size = build_indexlist(rear, indexlist);
	    ncom = concom(indexlist, labels);

	    if (ncom == 1) {
		postscript_contour(rear, TRUE);
		postscript_fill(rear & ~front);
	    } else {
		/* print each component of rear */
		for (com = 1; com <= ncom; com++) {
		    t = 0;
		    for (j = 0; j < size; j++)
		        if (labels[j] == com)
			    t |= REGION_BIT(indexlist[j]);

		    /* t is one component */
		    postscript_contour(t, TRUE);
		    postscript_fill(t & ~front);
	        }
	    }
        }
    }
    /* end of story */
    printf("EndPictureStack\nshowpage\n\n%%%%Trailer\n");
}

index **
ConnectedOutline(t)
subgraph_bitmap t;
{
    index **o;
    subgraph_bitmap bit, done;
    int     gotone;

    o = (index **) calloc(MAXHOLES, sizeof(index *));
    o[0] = NULL;

    for (bit = 1; bit <= R; bit++)
	if ((REGION_BIT(bit) & t) != 0)
	    break;

    bcopy((char *)Regions[bit].o,
	  (char *)o, MAXHOLES*sizeof(index *));
    done = REGION_BIT(bit);
    t &= ~REGION_BIT(bit);

    while (t != 0) {
	gotone = FALSE;
	for (bit = 1; bit <= R; bit++)
	    if ((REGION_BIT(bit) & t) != 0 && (Adj[bit] & done) != 0) {
		gotone = TRUE;
		break;
	    }
	if (! gotone) {
	    fprintf(stderr, "Looks like no neighbors between %x and %x\n",
		    done, t);
	    abort();
	}
	splice(Regions[bit].o, o, o);
	done |= REGION_BIT(bit);
	t &= ~REGION_BIT(bit);
    }
    return o;
}

/*
 * Print postscript version of contour using partial's outline edge chains
 */
postscript_contour(t, strokep)
subgraph_bitmap t;
int     strokep;
{
    int     j, k, length, n;
    index **o;
    index  *in;
    struct edge *e;

    o = ConnectedOutline(t);

    printf("newpath\n");

#define	XS(x)	(((x) - DrawInfo.xoff)*DrawInfo.xscale)
#define	YS(y)	(((y) - DrawInfo.yoff)*DrawInfo.yscale)

    while ((in = *o++) != NULL) {

	length = indexlist_size(in);

	/* lists are node-edge-node-edge */
	for (j = 0; j < length; j += 2) {
	    n = *in++;
	    printf("%% edge %d\n", *in);
	    e = &Edges[*in++];

	    /*
	     * trace the chain forward or backward according as n is the
	     * first or second node connected by e
	     */
	    if (e->node[0] == n) {
		/* forward */
		for (k = 0; k < e->nchain; k++)
		    printf("%g %g %c\n",
			   XS(e->chain[k].x), YS(e->chain[k].y),
			   (j + k == 0) ? 'M' : 'L');
	    } else {
		/* backward */
		for (k = e->nchain - 1; k >= 0; k--)
		    printf("%g %g %c\n",
			   XS(e->chain[k].x), YS(e->chain[k].y),
			   (k == e->nchain - 1 && j == 0) ? 'M' : 'L');
	    }
	}
	/* chain codes automatically "close" the path */
    }
    if (strokep)
	printf("gsave 0 setgray stroke grestore\n");
}

/* find average gray value of region(s) print postscript fill command */
postscript_fill(visible)
subgraph_bitmap visible;
{
    index  *in;
    index   indexlist[MAXR + 1];
    real    gray, area;

    /* find average visible gray value (postscript 0 = black, 1 = white) */
    build_indexlist(visible, indexlist);
    gray = 0;
    area = 0;
    for (in = indexlist; *in != 0; in++) {
	gray += Regions[*in].mean * Regions[*in].area;
	area += Regions[*in].area;
    }
    gray /= area;

    /* convert gray from user scale to 0-1 */
    gray = (gray - DrawInfo.black) / (DrawInfo.white - DrawInfo.black);

    printf("%g setgray eofill\n\n", gray);
}

/*
 * concom -- find connected components
 * 
 * concom(indexlist, labels) returns # connected components in subgraph given by
 * indexlist. Puts component number of region indexlist[i] at labels[i]
 * (i=0...)
 */
concom(indexlist, labels)
index  *indexlist;
char   *labels;
{
    /* the ol' labelling trick */
    int     i, j, k, size, label, from, to;
    char    map[MAXR + 1];

    label = 1;			/* components numbered from 1 */
    size = indexlist_size(indexlist);

    for (i = 0; i < size; i++) {

	/*
	 * find the lowest label of all so-far-labelled neighbors.
	 */
	to = size + 1;		/* unreasonably large */
	for (j = 0; j < i; j++)
	    if ((Adj[indexlist[i]] & REGION_BIT(indexlist[j])) != 0
		&& labels[j] < to)
		to = labels[j];

	/* if not, give this region a new label and continue */
	if (to == size + 1) {
	    labels[i] = label++;
	    continue;
	}
	labels[i] = to;

	/* renumber for each now-wrongly-labelled neighbor */
	for (j = 0; j < i; j++) {
	    if ((Adj[indexlist[i]] & REGION_BIT(indexlist[j])) != 0
		&& labels[j] != to) {

		/* renumber everything from labels[j] to to */
		from = labels[j];	/* copy before we renumber! */

		for (k = 0; k < i; k++)
		    if (labels[k] == from)
			labels[k] = to;
	    }
	}
    }

    /* relabel without gaps */
    k = 1;			/* k is our counter */
    for (i = 1; i <= size; i++)
	map[i] = 0;

    for (i = 0; i < size; i++) {
	if ((j = map[labels[i]]) == 0)
	    map[labels[i]] = j = k++;
	labels[i] = j;
    }

    return k - 1;
}

/*
 * build_indexlist(bitmap, indexlist) builds a region list from a bitmap and
 * returns # regions. indexlist should have room for MAXR+1 indeces.
 */
build_indexlist(bitmap, indexlist)
subgraph_bitmap bitmap;
index  *indexlist;
{
    index  *n;
    int     i;

    n = indexlist;
    for (i = 1; i <= R; i++)
	if (bitmap & REGION_BIT(i))
	    *n++ = i;
    *n++ = '\0';

    return n - indexlist - 1;
}
