/* $Id: xmesa1.c,v 1.14 1996/01/22 21:05:04 brianp Exp $ */

/*
 * Mesa 3-D graphics library
 * Version:  1.2
 * Copyright (C) 1995  Brian Paul  (brianp@ssec.wisc.edu)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


/*
$Log: xmesa1.c,v $
 * Revision 1.14  1996/01/22  21:05:04  brianp
 * changed HPCR undithered pixelformat to PF_HPCR, per Alex De Bruyn
 *
 * Revision 1.13  1996/01/19  18:09:27  brianp
 * fixed potential X error handler bug in alloc_shm_back_buffer()
 *
 * Revision 1.12  1996/01/07  22:51:16  brianp
 * re-enable XSync() in glXSwapBuffers()
 *
 * Revision 1.11  1996/01/05  01:19:01  brianp
 * added profiling to XMesaSwapBuffers()
 *
 * Revision 1.10  1995/12/21  17:58:13  brianp
 * fixed the call to get_drawable_size in case of 8-byte integers
 *
 * Revision 1.9  1995/12/12  21:44:04  brianp
 * only print color warning message if MESA_DEBUG is defined
 *
 * Revision 1.8  1995/11/30  00:52:30  brianp
 * fixed a few type warnings for Sun compiler
 *
 * Revision 1.7  1995/11/30  00:20:56  brianp
 * added new PF_GRAYSCALE mode
 *
 * Revision 1.6  1995/11/08  22:08:50  brianp
 * added 8-bit TrueColor dithering
 *
 * Revision 1.5  1995/11/03  17:41:48  brianp
 * removed unused vars, fixed code for C++ compilation
 *
 * Revision 1.4  1995/11/01  15:14:03  brianp
 * renamed all class variables per Steven Spitz
 *
 * Revision 1.3  1995/10/30  15:13:01  brianp
 * replaced Current variable with XMesa
 *
 * Revision 1.2  1995/10/19  15:53:20  brianp
 * new arguments to gl_new_context()
 *
 * Revision 1.1  1995/10/17  21:36:55  brianp
 * Initial revision
 *
 */



/*
 * Mesa/X11 interface, part 1.
 *
 * This file contains the implementations of all the XMesa* functions.
 *
 *
 * NOTES:
 *
 * The window coordinate system origin (0,0) is in the lower-left corner
 * of the window.  X11's window coordinate origin is in the upper-left
 * corner of the window.  Therefore, most drawing functions in this
 * file have to flip Y coordinates.
 *
 * Define SHM in the Makefile with -DSHM if you want to compile in support
 * for the MIT Shared Memory extension.  If enabled, when you use an Ximage
 * for the back buffer in double buffered mode, the "swap" operation will
 * be faster.  You must also link with -lXext.
 *
 */


#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#ifdef SHM
#  include <sys/ipc.h>
#  include <sys/shm.h>
#  include <X11/extensions/XShm.h>
#endif
#include "GL/xmesa.h"
#include "xmesaP.h"
#include "context.h"
/*#include "dd.h"*/
#include "macros.h"
#include "xform.h"



XMesaContext XMesa = NULL;




/**********************************************************************/
/*****                      Private Functions                     *****/
/**********************************************************************/


/*
 * X/Mesa Error reporting function:
 */
static void error( const char *msg )
{
   fprintf( stderr, "X/Mesa error: %s\n", msg );
}


/*
 * Return the host's byte order as LSBFirst or MSBFirst ala X.
 */
static int host_byte_order( void )
{
   int i = 1;
   char *cptr = (char *) &i;
   return (*cptr==1) ? LSBFirst : MSBFirst;
}



/*
 * Error handling.
 */
static int mesaXErrorFlag = 0;

static int mesaHandleXError( Display *dpy, XErrorEvent *event )
{
    mesaXErrorFlag = 1;
    return 0;
}


/*
 * Check if the X Shared Memory extension is available.
 * Return:  0 = not available
 *          1 = shared XImage support available
 *          2 = shared Pixmap support available also
 */
static int check_for_xshm( Display *display )
{
#ifdef SHM
   int major, minor, ignore;
   Bool pixmaps;

   if (XQueryExtension( display, "MIT-SHM", &ignore, &ignore, &ignore )) {
      if (XShmQueryVersion( display, &major, &minor, &pixmaps )==True) {
	 return (pixmaps==True) ? 2 : 1;
      }
      else {
	 return 0;
      }
   }
   else {
      return 0;
   }
#else
   /* Can't compile XSHM support */
   return 0;
#endif
}


/*
 * Allocate a shared memory XImage back buffer for the given context.
 * Return:  GL_TRUE if success, GL_FALSE if error
 */
static GLboolean alloc_shm_back_buffer( XMesaContext c )
{
#ifdef SHM
   /*
    * We have to do a _lot_ of error checking here to be sure we can
    * really use the XSHM extension.  It seems different servers trigger
    * errors at different points if the extension won't work.  Therefore
    * we have to be very careful...
    */
   GC gc;
   int (*old_handler)( Display *, XErrorEvent * );

   c->backimage = XShmCreateImage( c->display, c->visual, c->depth,
				   ZPixmap, NULL, &c->shminfo,
				   c->width, c->height );
   if (c->backimage == NULL) {
      error( "alloc_back_buffer: Shared memory error (XShmCreateImage), disabling." );
      c->shm = 0;
      return GL_FALSE;
   }

   c->shminfo.shmid = shmget( IPC_PRIVATE, c->backimage->bytes_per_line
			     * c->backimage->height, IPC_CREAT|0777 );
   if (c->shminfo.shmid < 0) {
      perror("alloc_back_buffer");
      XDestroyImage( c->backimage );
      c->backimage = NULL;
      error( "alloc_back_buffer: Shared memory error (shmget), disabling." );
      c->shm = 0;
      return GL_FALSE;
   }

   c->shminfo.shmaddr = c->backimage->data
                      = (char*)shmat( c->shminfo.shmid, 0, 0 );
   if (c->shminfo.shmaddr == (char *) -1) {
      perror("alloc_back_buffer");
      XDestroyImage( c->backimage );
      c->backimage = NULL;
      error("alloc_back_buffer: Shared memory error (shmat), disabling.");
      c->shm = 0;
      return GL_FALSE;
   }

   c->shminfo.readOnly = False;
   mesaXErrorFlag = 0;
   old_handler = XSetErrorHandler( mesaHandleXError );
   /* This may trigger the X protocol error we're ready to catch: */
   XShmAttach( c->display, &c->shminfo );
   XSync( c->display, False );

   if (mesaXErrorFlag) {
      /* we are on a remote display, this error is normal, don't print it */
      XFlush( c->display );
      mesaXErrorFlag = 0;
      XDestroyImage( c->backimage );
      shmdt( c->shminfo.shmaddr );
      shmctl( c->shminfo.shmid, IPC_RMID, 0 );
      c->backimage = NULL;
      c->shm = 0;
      (void) XSetErrorHandler( old_handler );
      return GL_FALSE;
   }

   shmctl( c->shminfo.shmid, IPC_RMID, 0 ); /* nobody else needs it */

   /* Finally, try an XShmPutImage to be really sure the extension works */
   gc = XCreateGC( c->display, c->frontbuffer, 0, NULL );
   XShmPutImage( c->display, c->frontbuffer, gc,
		 c->backimage, 0, 0, 0, 0, 1, 1 /*one pixel*/, False );
   XSync( c->display, False );
   XFreeGC( c->display, gc );
   (void) XSetErrorHandler( old_handler );
   if (mesaXErrorFlag) {
      XFlush( c->display );
      mesaXErrorFlag = 0;
      XDestroyImage( c->backimage );
      shmdt( c->shminfo.shmaddr );
      shmctl( c->shminfo.shmid, IPC_RMID, 0 );
      c->backimage = NULL;
      c->shm = 0;
      return GL_FALSE;
   }

   return GL_TRUE;
#else
   /* Can't compile XSHM support */
   return GL_FALSE;
#endif
}



/*
 * Setup an off-screen pixmap or Ximage to use as the back buffer.
 * Input:  c - the X/Mesa context.
 */
void xmesa_alloc_back_buffer( XMesaContext c )
{
   if (c->db_state==BACK_XIMAGE) {
      /* Deallocate the old backimage, if any */
      if (c->backimage) {
#ifdef SHM
	 if (c->shm) {
	    XShmDetach( c->display, &c->shminfo );
	    XDestroyImage( c->backimage );
	    shmdt( c->shminfo.shmaddr );
	 }
	 else
#endif
	   XDestroyImage( c->backimage );
	 c->backimage = NULL;
      }

      /* Allocate new back buffer */
      if (c->shm==0 || alloc_shm_back_buffer(c)==GL_FALSE) {
	 /* Allocate a regular XImage for the back buffer. */
	 /* TODO: this number of bytes computation is a hack! */
         char *img = (char *) malloc( c->width * c->height * 4 );
	 c->backimage = XCreateImage( c->display, c->visual, c->depth,
				      ZPixmap, 0,   /* format, offset */
				      img, c->width, c->height,
				      8, 0 );  /* pad, bytes_per_line */
	 if (!c->backimage) {
	    /* TODO: more related error checks are needed throughout xmesa.c */
	    error("alloc_back_buffer: XCreateImage failed.");
	 }
      }
      c->backpixmap = None;
   }
   else if (c->db_state==BACK_PIXMAP) {
      Pixmap old_pixmap = c->backpixmap;
      /* Free the old back pixmap */
      if (c->backpixmap) {
	 XFreePixmap( c->display, c->backpixmap );
      }
      /* Allocate new back pixmap */
      c->backpixmap = XCreatePixmap( c->display, c->frontbuffer,
				     c->width, c->height, c->depth );
      c->backimage = NULL;
      /* update other references to backpixmap */
      if (c->buffer==old_pixmap) {
	 c->buffer = c->backpixmap;
      }
   }
}



/*
 * Return the width and height of the given drawable.
 */
static void get_drawable_size( Display *dpy, Drawable d,
			       unsigned int *width, unsigned int *height )
{
   Window root;
   int x, y;
   unsigned int bw, depth;

   XGetGeometry( dpy, d, &root, &x, &y, width, height, &bw, &depth );
}




static int setup_grayscale( XMesaContext c, Window window )
{
   int r, g, b, i;
   int colorsfailed = 0;
   XColor xcol;
   XColor *ctable = NULL;
   Colormap cmap;
   XWindowAttributes attr;

   assert( c->depth>=4 && c->depth<=16 );

   /* Need to know the colormap in this case */
   XGetWindowAttributes( c->display, window, &attr );
   cmap = attr.colormap;

   /* This is a hack for GLUT: pre-allocate the gray needed for pop-up menus */
   xcol.red = xcol.green = xcol.blue = 0xaa00;
   XAllocColor( c->display, cmap, &xcol );

#define distance2(r1,g1,b1,r2,g2,b2)  ( ((r2)-(r1)) * ((r2)-(r1)) + \
    ((g2)-(g1)) * ((g2)-(g1)) + ((b2)-(b1)) * ((b2)-(b1)) )

   /* Allocate 256 shades of gray */
   for (g=0;g<256;g++) {
      xcol.red = xcol.green = xcol.blue  = (g << 8) | g;
      if (!XAllocColor(c->display, cmap, &xcol)) {
         /* Search for best match (contributed by Michael Pichler) */
         int p, bestmatch;
         double dist, mindist;  /* 3*2^16^2 exceeds long int precision */
         int cmap_size = c->visual->map_entries;

         /* query whole colormap if not yet done */
         if (!ctable) {
            ctable = (XColor *) malloc( cmap_size * sizeof(XColor) );
            for (p = 0;  p < cmap_size;  p++)
              ctable[p].pixel = p;
            XQueryColors (c->display, cmap, ctable, cmap_size);
         }
 
         /* find best match */
         mindist = 0.0;
         bestmatch = -1;
         p = cmap_size;
         while (p--) {
            int mapgray;
            mapgray = (ctable[p].red + ctable[p].green + ctable[p].blue) / 3U;
            dist = ABS( mapgray - ((g << 8) | g) );
            if (bestmatch < 0 || dist < mindist)
              mindist = dist, bestmatch = p;
         }
         xcol.pixel = bestmatch;
         colorsfailed++;
      }
      c->color_table[g] = xcol.pixel;
      c->red_table[xcol.pixel]   = g * 30 / 100;
      c->green_table[xcol.pixel] = g * 59 / 100;
      c->blue_table[xcol.pixel]  = g * 11 / 100;
   }

#undef distance2

   if (ctable) {
      free(ctable);
   }

   if (colorsfailed && getenv("MESA_DEBUG")) {
      fprintf( stderr,
               "Note: %d out of 256 needed colors do not match exactly.\n",
	       colorsfailed, 256 );
   }

#define WEIGHT
#ifdef WEIGHT
   c->rmult = 30 * 255 / 100;
   c->gmult = 59 * 255 / 100;
   c->bmult = 11 * 255 / 100;
#else
   c->rmult = 255/3;
   c->gmult = 255/3;
   c->bmult = 255/3;
#endif
   c->amult = 255;
   c->dithered_pf = PF_GRAYSCALE;
   c->undithered_pf = PF_GRAYSCALE;
   c->pixel = c->color_table[255];      /* white */
   c->clearpixel = c->color_table[0];   /* black */

   return 1;
}



/*
 * Setup RGB rending for a window with a PseudoColor, StaticColor,
 * GrayScale or StaticGray visual.  We try to allocate a palette of 225
 * colors (5 red, 9 green, 5 blue) and dither to approximate a 24-bit RGB
 * color.  While this function was originally designed just for 8-bit
 * visuals, it has also proven to work from 4-bit up to 16-bit visuals.
 * The fact that this method works for gray scale displays depends on
 * XAllocColor allocating a good gray to approximate an RGB color.
 * Dithering code contributed by Bob Mercier.
 */
static int setup_dithered_color( XMesaContext c, Window window )
{
   int r, g, b, i;
   int colorsfailed = 0;
   XColor xcol;
   XColor *allcolors = NULL, *acptr;
   Colormap cmap;
   XWindowAttributes attr;

   assert( c->depth>=4 && c->depth<=16 );

   /* Need to know the colormap in this case */
   XGetWindowAttributes( c->display, window, &attr );
   cmap = attr.colormap;

   /* This is a hack for GLUT: pre-allocate the gray needed for pop-up menus */
   xcol.red = xcol.green = xcol.blue = 0xaa00;
   XAllocColor( c->display, cmap, &xcol );

#define distance2(r1,g1,b1,r2,g2,b2)  ( ((r2)-(r1)) * ((r2)-(r1)) + \
    ((g2)-(g1)) * ((g2)-(g1)) + ((b2)-(b1)) * ((b2)-(b1)) )

   /* Allocate X colors and initialize color_table[], red_table[], etc */
   for (r = 0; r < _R; r++) {
      for (g = 0; g < _G; g++) {
	 for (b = 0; b < _B; b++) {
	    xcol.red   = r * 65535 / (_R-1);
	    xcol.green = g * 65535 / (_G-1);
	    xcol.blue  = b * 65535 / (_B-1);
	    if (!XAllocColor(c->display, cmap, &xcol)) {
	       /* Search for best match (contributed by Michael Pichler) */
	       int p, bestmatch;
	       double dist, mindist;  /* 3*2^16^2 exceeds long int precision */
	       int cmap_size = c->visual->map_entries;

	       /* query whole colormap if not yet done */
	       if (!allcolors) {
		  allcolors = (XColor *) malloc( cmap_size * sizeof(XColor) );
		  for (p = 0;  p < cmap_size;  p++)
		    allcolors[p].pixel = p;
		  XQueryColors (c->display, cmap, allcolors, cmap_size);
	       }
 
	       /* find best match */
	       mindist = 0.0;
	       bestmatch = -1;
	       p = cmap_size;
	       while (p--) {
		  acptr = allcolors + p;
		  dist = distance2( (double)xcol.red, (double)xcol.green,
				    (double)xcol.blue, (double)acptr->red,
				    (double)acptr->green, (double)acptr->blue);
		  if (bestmatch < 0 || dist < mindist)
		     mindist = dist, bestmatch = p;
	       }
	       xcol.pixel = bestmatch;
	       colorsfailed++;
	    }
	    i = _MIX( r, g, b );
	    c->color_table[i] = xcol.pixel;
	    c->red_table[xcol.pixel]   = r * 255 / (_R-1);
	    c->green_table[xcol.pixel] = g * 255 / (_G-1);
	    c->blue_table[xcol.pixel]  = b * 255 / (_B-1);
	 }
      }
   }

#undef distance2

   if (allcolors) {
      free(allcolors);
   }

   if (colorsfailed && getenv("MESA_DEBUG")) {
      fprintf( stderr,
               "Note: %d out of %d needed colors do not match exactly.\n",
	       colorsfailed, _R*_G*_B );
   }

   c->rmult = 255;
   c->gmult = 255;
   c->bmult = 255;
   c->amult = 255;
   c->dithered_pf = PF_DITHER;
   c->undithered_pf = PF_LOOKUP;
   c->pixel = c->color_table[_MIX(_R-1,_G-1,_B-1)];  /* white */
   c->clearpixel = c->color_table[_MIX(0,0,0)];      /* black */

   return 1;
}



/*
 * Setup RGB rending for a window with a True/DirectColor visual.
 */
static int setup_truecolor( XMesaContext c, Window window )
{
   unsigned long rmask, gmask, bmask;

   /*
    * Use the red, green, and blue mask values in the visual structure
    * to compute the multiplier and shift values for converting RGB
    * triplets from floats in [0,1] to packed pixel values.
    */

   /* Red */
   c->rshift = 0;
   rmask = c->visual->red_mask;
   while ((rmask & 1)==0) {
      c->rshift++;
      rmask = rmask >> 1;
   }
   c->rmult = (GLint) rmask;

   /* Green */
   c->gshift = 0;
   gmask = c->visual->green_mask;
   while ((gmask & 1)==0) {
      c->gshift++;
      gmask = gmask >> 1;
   }
   c->gmult = (GLint) gmask;

   /* Blue */
   c->bshift = 0;
   bmask = c->visual->blue_mask;
   while ((bmask & 1)==0) {
      c->bshift++;
      bmask = bmask >> 1;
   }
   c->bmult = (GLint) bmask;

   /* Alpha */
   c->ashift = 24;
   c->amult = 255;

   if (host_byte_order() != ImageByteOrder(c->display)) {
      /* Must reverse order of bytes in back buffer's XImage before sending */
      /* to server. */
      c->swapbytes = GL_TRUE;
   }

   if (c->rshift==0 && c->gshift==8 && c->bshift==16
       && ImageByteOrder(c->display)==MSBFirst && sizeof(GLuint)==4) {
      /* a common case for 24-bit displays */
      c->undithered_pf = c->dithered_pf = PF_8A8B8G8R;
      c->pixel = 0xffffffff;  /* white */
      c->clearpixel = 0x0;    /* black */
   }
   else if (  c->rshift==5 && c->gshift==2 && c->bshift==0
           && XInternAtom(c->display, "_HP_RGB_SMOOTH_MAP_LIST", True)) {
      /* HP Color Recovery
       * To work properly, the atom _HP_RGB_SMOOTH_MAP_LIST must be defined
       * on the root window AND the colormap obtainable by XGetRGBColormaps
       * for that atom must be set on the window.  (see also tkInitWindow)
       * If that colormap is not set, the output will look stripy.
       */
      c->undithered_pf = PF_HPCR;  /* can't really disable dithering for now */
      c->dithered_pf = PF_HPCR;
      c->pixel = c->visual->red_mask | c->visual->green_mask
               | c->visual->blue_mask;  /* white */
      c->clearpixel = 0x0;    /* black */
      c->rmult = 255;
      c->gmult = 255;
      c->bmult = 255;
      c->amult = 255;
   }
   else if (c->depth==8) {
      /* dither if 8-bit */
      setup_dithered_color( c, window );
   }
   else {
      /* general case */
      c->undithered_pf = c->dithered_pf = PF_TRUECOLOR;
      c->pixel = c->visual->red_mask | c->visual->green_mask
	         | c->visual->blue_mask;  /* white */
      c->clearpixel = 0x0;    /* black */
   }

   return 1;
}



/*
 * Setup RGB rending for a window with a monochrome visual.
 */
static int setup_monochrome( XMesaContext c )
{
   /* Simulate RGB in monochrome */
   c->rmult = 255;
   c->gmult = 255;
   c->bmult = 255;
   c->amult = 255;
   c->dithered_pf = c->undithered_pf = PF_1BIT;

   /* is it OK to hardcode these values? */
   c->pixel = 1;       /* white */
   c->clearpixel = 0;  /* black */

   return 1;
}


extern void init_gamma_tables( struct gl_context *c );


/*
 * When a context is "made current" for the first time, we can finally
 * finish initializing the context.
 * Input:  c - the XMesaContext to initialize
 *         window - the window we're rendering into if RGB mode & PseudoColor
 *
 */
static GLboolean initialize_context( XMesaContext c, Window window )
{
   XGCValues gcvalues;

   assert( c->initialized==GL_FALSE );

   if (c->rgb_flag==GL_FALSE) {
      /* COLOR-INDEXED WINDOW:
       * Even if the visual is TrueColor or DirectColor we treat it as
       * being color indexed.  This is weird but might be useful to someone.
       */
      c->dithered_pf = c->undithered_pf = PF_INDEX;
      c->pixel = 1;
      c->clearpixel = 0;
      c->rmult = c->gmult = c->bmult = c->amult = 0;
   }
   else {
      /* RGB WINDOW:
       * If the visual is TrueColor or DirectColor we're all set.  Other-
       * wise, we simulate RGB mode using a color-mapped window.
       */
      int xclass;
#if defined(__cplusplus) || defined(c_plusplus)
      xclass = c->visual->c_class;
#else
      xclass = c->visual->class;
#endif
      if (xclass==TrueColor || xclass==DirectColor) {
	 (void) setup_truecolor( c, window );
      }
      else if (xclass==StaticGray && c->depth==1) {
	 (void) setup_monochrome( c );
      }
      else if (xclass==GrayScale || xclass==StaticGray) {
         (void) setup_grayscale( c, window );
      }
      else if ((xclass==PseudoColor || xclass==StaticColor)
               && c->depth>=4 && c->depth<=16) {
	 (void) setup_dithered_color( c, window );
      }
      else {
	 error("XMesaCreateContext: can't simulate RGB mode with given visual.");
	 return GL_FALSE;
      }
   }

   /* Now allocate the GL context */
   c->gl_ctx = gl_new_context( c->rgb_flag,
                               (GLfloat) c->rmult,
                               (GLfloat) c->gmult,
                               (GLfloat) c->bmult,
                               (GLfloat) c->amult,
                               c->db_state ? GL_TRUE : GL_FALSE,
                               c->share_list ? c->share_list->gl_ctx : NULL );

   /* Dithering is enabled by default */
   c->pixelformat = c->dithered_pf;

   {
      unsigned int w, h;
      get_drawable_size( c->display, window, &w, &h );
      c->width = w;
      c->height = h;
   }

   if (c->db_state) {
      /* Double buffered */
      xmesa_alloc_back_buffer( c );
      if (c->db_state==BACK_PIXMAP) {
         c->buffer = c->backpixmap;
      }
      else {
         c->buffer = XIMAGE;
      }
   }
   else {
      /* Single Buffered */
      c->buffer = c->frontbuffer;
   }

   /* X11 graphics context */
   c->gc1 = XCreateGC( c->display, window, 0, NULL );
   XSetForeground( c->display, c->gc1, c->pixel );
   XSetBackground( c->display, c->gc1, c->pixel );
   XSetFunction( c->display, c->gc1, GXcopy );
   c->gc2 = XCreateGC( c->display, window, 0, NULL );
   XSetForeground( c->display, c->gc2, c->pixel );
   XSetBackground( c->display, c->gc2, c->pixel );
   XSetFunction( c->display, c->gc2, GXcopy );
   /*
    * Don't generate Graphics Expose/NoExpose events in swapbuffers().
    * Patch contributed by Michael Pichler May 15, 1995.
    */
   gcvalues.graphics_exposures = False;
   c->cleargc = XCreateGC( c->display, window, GCGraphicsExposures, &gcvalues);
   XSetForeground( c->display, c->cleargc, c->clearpixel );
   XSetBackground( c->display, c->cleargc, c->clearpixel );
   XSetFunction( c->display, c->cleargc, GXcopy );

   /* Initialize the row buffer XImage for use in write_color_span() */
   c->rowimage = XCreateImage( c->display, c->visual, c->depth,
                               ZPixmap, 0,                  /*format, offset*/
                               (char*) malloc(MAX_WIDTH*4), /*data*/
                               MAX_WIDTH, 1,                /*width, height*/
                               32,                          /*bitmap_pad*/
                               0                         /*bytes_per_line*/ );


   c->initialized = GL_TRUE;

   return GL_TRUE;
}




/**********************************************************************/
/*****                       Public Functions                     *****/
/**********************************************************************/


/*
 * Create a new XMesaContext for rendering into an X11 window.
 *
 * Input:  display - X11 display
 *         visinfo - X11 VisualInfo describing the colorbuffer we'll use
 *         rgb_flag - GL_TRUE = RGB mode,
 *                    GL_FALSE = color index mode
 *         db_flag - GL_TRUE = double-buffered,
 *                   GL_FALSE = single buffered
 *         ximage_flag - GL_TRUE = use an XImage for back buffer,
 *                       GL_FALSE = use an off-screen pixmap for back buffer
 *         share_list - another XMesaContext with which to share display
 *                      lists or NULL if no sharing is wanted.
 * Return:  an XMesaContext or NULL if error.
 */
XMesaContext XMesaCreateContext( Display *display,
				 XVisualInfo *visinfo,
				 GLboolean rgb_flag,
				 GLboolean db_flag,
				 GLboolean ximage_flag,
				 XMesaContext share_list )
{
   XMesaContext c;

   /* allocate xmesa_context struct initialized to zeros */
   c = (struct xmesa_context *) calloc( 1, sizeof(struct xmesa_context) );
   if (!c) {
      return NULL;
   }

   c->rgb_flag = rgb_flag;
   c->share_list = share_list;

   /* X stuff */
   c->display = display;
   c->visual = visinfo->visual;
   c->depth = visinfo->depth;
   c->shm = check_for_xshm( display );
   c->swapbytes = GL_FALSE;

   /* Double buffering configuration */
   if (db_flag) {
      if (ximage_flag) {
	 c->db_state = BACK_XIMAGE;
      }
      else {
	 c->db_state = BACK_PIXMAP;
      }
   }
   else {
      c->db_state = 0;
   }

   /*XSynchronize( display, 1 );*/    /* This makes debugging X easier */

   return c;
}




void XMesaDestroyContext( XMesaContext c )
{
   if (c->gl_ctx)  gl_destroy_context( c->gl_ctx );

   if (c->gc1)  XFreeGC( c->display, c->gc1 );
   if (c->gc2)  XFreeGC( c->display, c->gc2 );
   if (c->cleargc)  XFreeGC( c->display, c->cleargc );

   if (c->backimage) {
#ifdef SHM
       if (c->shm) {
	   XShmDetach( c->display, &c->shminfo );
	   XDestroyImage( c->backimage );
	   shmdt( c->shminfo.shmaddr );
       }
       else
#endif
	   XDestroyImage( c->backimage );
   }
   if (c->backpixmap) {
      XFreePixmap( c->display, c->backpixmap );
   }
   if (c->rowimage) {
      free( c->rowimage->data );
      c->rowimage->data = NULL;
      XDestroyImage( c->rowimage );
   }
   free( c );
}



/*
 * Bind an X/Mesa context to an X window.
 */
GLboolean XMesaBindWindow( XMesaContext c, Window w )
{
   if (c->buffer==c->frontbuffer) {
      c->buffer = w;
   }
   c->frontbuffer = w;
   if (!c->initialized) {
      if (!initialize_context( c, w )) {
	 return GL_FALSE;
      }
   }
   return GL_TRUE;
}



/*
 * Bind an X/Mesa context to an X pixmap.
 */
GLboolean XMesaBindPixmap( XMesaContext c, Pixmap p )
{
   if (c->buffer==c->frontbuffer) {
      c->buffer = p;
   }
   c->frontbuffer = p;

   if (!c->initialized) {
      int xclass;
#if defined(__cplusplus) || defined(c_plusplus)
      xclass = c->visual->c_class;
#else
      xclass = c->visual->class;
#endif
      if (c->rgb_flag && xclass==PseudoColor) {
	 /* Error, there isn't enough information carried in a pixmap */
	 /* to initialize the context.  Specifically, we need to know */
	 /* the colormap at this point for this configuration. */
	 return GL_FALSE;
      }
      if (!initialize_context( c, p )) {
	 return GL_FALSE;
      }
   }
   return GL_TRUE;
}



GLboolean XMesaMakeCurrent( XMesaContext c )
{
   if (c) {
      if (!c->initialized) {
         return GL_FALSE;
      }
      gl_set_context( c->gl_ctx );
      XMesa = c;

      xmesa_setup_DD_pointers();

      if (XMesa->gl_ctx->Viewport.Width==0) {
	 /* initialize viewport to window size */
	 gl_viewport( 0, 0, XMesa->width, XMesa->height );
	 CC.Scissor.Width = XMesa->width;
	 CC.Scissor.Height = XMesa->height;
      }
   }
   else {
      /* Detach */
      XMesa = NULL;
   }
   return GL_TRUE;
}



XMesaContext XMesaGetCurrentContext( void )
{
   return XMesa;
}



/*
 * Perform byte swapping on an XImage if the host and server use a different
 * byte ordering.
 */
static void swap_image_data( XImage *img )
{
   register GLuint i, j, *ptr4, p;

   if (img->bits_per_pixel==32) {
      for (i=0;i<img->height;i++) {
         ptr4 = (GLuint *) (img->data + i*img->bytes_per_line);
         for (j=0;j<img->width;j++) {
            p = ptr4[j];
            ptr4[j] = ((p & 0xff000000) >> 24)
                    | ((p & 0x00ff0000) >> 8)
                    | ((p & 0x0000ff00) << 8)
                    | ((p & 0x000000ff) << 24);
         }
      }
   }
   /* 12-bit??? */
}


/*
 * Copy the back buffer to the front buffer.  If there's no back buffer
 * this is a no-op.
 */
void XMesaSwapBuffers( void )
{
#ifdef PROFILE
   GLdouble t0 = gl_time();
#endif
   if (XMesa->db_state) {
      if (XMesa->backimage) {
	 /* Copy Ximage from host's memory to server's window */
#ifdef SHM
	 if (XMesa->shm) {
	    XShmPutImage( XMesa->display, XMesa->frontbuffer,
			  XMesa->cleargc,
			  XMesa->backimage, 0, 0,
			  0, 0, XMesa->width, XMesa->height, False );
	    /* wait for finished event??? */
	 }
	 else
#endif
         {
            if (XMesa->swapbytes) {
               swap_image_data( XMesa->backimage );
            }
            XPutImage( XMesa->display, XMesa->frontbuffer,
                       XMesa->cleargc,
                       XMesa->backimage, 0, 0,
                       0, 0, XMesa->width, XMesa->height );
         }
      }
      else {
	 /* Copy pixmap to window on server */
	 XCopyArea( XMesa->display,
		    XMesa->backpixmap,   /* source drawable */
		    XMesa->frontbuffer,  /* dest. drawable */
		    XMesa->cleargc,
		    0, 0, XMesa->width, XMesa->height,  /* source region */
		    0, 0                 /* dest region */
		   );
      }
   }
   XSync( XMesa->display, False );
#ifdef PROFILE
   CC.SwapCount++;
   CC.SwapTime += gl_time() - t0;
#endif
}



/*
 * Return a pointer to the XMesa backbuffer Pixmap or XImage.  This function
 * is a way to get "under the hood" of X/Mesa so one can manipulate the
 * back buffer directly.
 * Output:  pixmap - pointer to back buffer's Pixmap, or 0
 *          ximage - pointer to back buffer's XImage, or NULL
 * Return:  GL_TRUE = context is double buffered
 *          GL_FALSE = context is single buffered
 */
GLboolean XMesaGetBackBuffer( Pixmap *pixmap, XImage **ximage )
{
   if (XMesa->db_state) {
      if (pixmap)  *pixmap = XMesa->backpixmap;
      if (ximage)  *ximage = XMesa->backimage;
      return GL_TRUE;
   }
   else {
      *pixmap = 0;
      *ximage = NULL;
      return GL_FALSE;
   }
}
