/*
 *		Copyright IBM Corporation 1989
 *
 *                      All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of IBM not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * IBM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * IBM BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 *
 *
 * University of Illinois at Urbana-Champaign
 * Department of Computer Science
 * 1304 W. Springfield Ave.
 * Urbana, IL	61801
 *
 * (C) Copyright 1987, 1988 by The University of Illinois Board of Trustees.
 * All rights reserved.
 *
 * Tool: X 11 Graphical Kernel System
 * Author: Gregory Scott Rogers
 * Author: Sung Hsien Ching Kelvin
 * Author: Yu Pan
 *
 *  XGKS -- Xevent interrupt handeling and process routines
 */

/*LINTLIBRARY*/

#include "udposix.h"
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
#include "gks_implem.h"

#ifndef lint
    static char afsid[]	= "$__Header$";
    static char rcsid[]	= "$Id: xevent.c,v 2.16 1991/10/29 16:22:46 steve Exp $";
#endif


    static int
redraw(xev, ws)
    XEvent         *xev;
    WS_STATE_PTR    ws;
{
    Display		*dpy	= ws->dpy;
    Window		win	= ws->win;
    XWindowAttributes	win_att;

    /*
     * Disable all input devices.
     */
    XgksIDevDisable(ws);

    /*
     * Get current window width and height values and update the 
     * transformation.
     */
    XGetWindowAttributes(dpy, win, &win_att);
    ws->wbound.x = win_att.width;
    ws->wbound.y = win_att.height;
    xXgksUpdateTrans(ws);

    /*
     * Redraw the window contents.
     */
    XClearArea(dpy, win, 0, 0, 0, 0, False);
#ifdef DEBUG
	(void) fprintf(stderr, "redraw: calling XgksXReDrawWs()\n");
#endif
    XgksXReDrawWs(ws);

    /* if necessary call user defined redraw notifying function */
    if (ws->redrawfuncp != NULL)
	(*(ws->redrawfuncp)) (ws->ws_id, GRD_X);

    XSync(dpy, 0);

    /*
     * Enable the input devices.
     */
    XgksIDevEnable(ws);

    return 0;
}


/*
 * Handle X Expose events.
 *
 * This routine will redraw the XGKS window if and only if the given Expose
 * event is the last such event in the queue.
 */
    static int
XgksExposeEvent(xev, ws)
    XEvent         *xev;
    WS_STATE_PTR    ws;
{
    XEvent		tmpxev;
    Window		win	= ws->win;
    Display		*dpy	= ws->dpy;

    /*
     * If this is not the last of a set of contiguous Expose events, then
     * ignore it.
     */
    if (xev->xexpose.count != 0)
	return 0;

    /*
     * If there is not another expose event further on in the queue, then we
     * have to process this expose; otherwise, we can ignore the current
     * expose.  If the expose count on the one we found is greater than 0,
     * then we can ignore that one also because we know another one has to
     * follow it.  If the expose count on the one we found is 0 then it has
     * to be re-sent to put it back at the tail of the queue because the
     * XCheckTypedWindowEvent() removed it.
     */
    if (XCheckTypedWindowEvent(dpy, xev->xexpose.window, Expose, &tmpxev)) {
	/*
	 * Found another expose.  If its the last of a contiguous set, then
	 * resend it to put it back in the queue so that we will eventually 
	 * respond to it.
	 */
#ifdef DEBUG
	(void) fprintf(stderr, "XgksExposeEvent: calling XSendEvent()\n");
#endif
	if (tmpxev.xexpose.count == 0)
	    XSendEvent(dpy, win, True, ExposureMask, &tmpxev);

	/*
	 * We know there's an expose after this one so don't process this one.
	 */
	return 0;
    }

    /*
     * Call the redraw function.
     */
    return redraw(xev, ws);
}


/*
 * The XEvent interrupt processing routine.
 *
 * This routine can be called both synchronously (while awaiting an event) 
 * and asynchronously (as a signal-handler).  It takes every opportunity
 * to obtain an X-event to which it can respond, but doesn't wait indefinitely
 * for such an event.
 */
    void
xProcessEvents()
{
    Gint            i;

    for (i = 0; i < MAX_OPEN_WS; i++) {
	if (xgks_state.openedws[i].ws_id != INVALID && 
	    xgks_state.openedws[i].ws != NULL &&
	    xgks_state.openedws[i].ws->dpy != NULL &&
	    !xgks_state.openedws[i].ws->ws_is_closing) {

	    XEvent		xev;
	    Window		win	= xgks_state.openedws[i].win;
	    WS_STATE_ENTRY	*ws	= xgks_state.openedws[i].ws;

#	    ifdef EVENTDEBUG
		(void) fputs("xProcessEvents(): calling XCheckWindowEvent()\n",
			     stderr);
#	    endif
	    
	    while (XCheckWindowEvent(ws->dpy, win, ~NoEventMask, &xev) 
		   == True) {

#		ifdef EVENTDEBUG
		    print_event(&xev);
#		endif

		switch (xev.type) {

		case Expose:
		    (void) XgksExposeEvent(&xev, ws);
		    break;

		case KeyPress:
		case MotionNotify:
		case ButtonPress:
		case ButtonRelease:
		    XgksIProcessXEvent(&xev, ws);
		    break;

		default:
		    break;
		}
	    }					/* event loop */
	}					/* workstation is X11 window */
    }						/* GKS workstation loop */
}


/*
 * Start the SIGIO interrupt system.
 */
    int
xXgksSIGIOStart(ws)
    WS_STATE_PTR    ws;
{
    Display        *dpy	= ws->dpy;
    pid_t           pid = getpid();

    if (dpy == NULL)				/* not opened yet */
	return INVALID;

    /*
     * Ignore SIGIO signals.
     */
    (void) sio_off();

    /*
     * Set the process-group ID of processes that will receive SIGIO and 
     * SIGURG signals associated with the display file-descriptor to the 
     * PID of this process.
     */
    (void) sockspgrp(dpy->fd, pid);

    /*
     * Make I/O on the display file-descriptor asynchronous.
     */
    (void) sockasync(dpy->fd, 1);

    return 0;
}


static int      SigCount = 0;


    int
XgksSIGIO_OFF(dpy)
    Display        *dpy;
{

    SigCount++;

#ifdef SIGDEBUG
    (void) fprintf(stderr, "XgksSIGIO_OFF SigCount == %d\n", SigCount);
#endif

    if (SigCount > 1)				/* already off */
	return 0;

    /* If socket does not exist, I/O is not possible. */
    if (dpy == NULL)
	return 0;

    /*
     * Ignore SIGIO signals.
     */
    (void) sio_off();

    /*
     * Make I/O on the display file-descriptor synchronous.
     */
    (void) sockasync(dpy->fd, 0);

    return 0;
}


    int
XgksSIGIO_ON(dpy)
    Display        *dpy;
{
    pid_t           pid = getpid();

    SigCount--;

#ifdef SIGDEBUG
    (void) fprintf(stderr, "XgksSIGIO_ON SigCount == %d\n", SigCount);
#endif

    if (SigCount > 0)				/* only on last request */
	return 0;

    /* If socket does not exist, I/O is not possible. */
    if (dpy == NULL)
	return 0;

    /* I don't know why this is here.  -- SRE 5/2/91 */
    xProcessEvents();

    /*
     * Register the SIGIO signal-handler.
     */
    (void) sio_on(xProcessEvents);

    /*
     * Set the process-group ID of processes that will receive SIGIO and 
     * SIGURG signals associated with  the display file-descriptor to the 
     * PID of this process.
     */
    (void) sockspgrp(dpy->fd, pid);

    /*
     * Make I/O on the display file-descriptor asynchronous.
     */
    (void) sockasync(dpy->fd, 1);

    return 0;
}


#ifdef EVENTDEBUG
    static int
print_event(evnt)
    XEvent         *evnt;
{
    switch (evnt->type) {
    case 0:
	(void) fprintf(stderr, "some kind of error");
	break;
    case 1:
	(void) fprintf(stderr, "some kind of reply");
	break;
    case KeyPress:
	(void) fprintf(stderr, "KeyPress");
	break;
    case KeyRelease:
	(void) fprintf(stderr, "KeyRelease");
	break;
    case ButtonPress:
	(void) fprintf(stderr, "ButtonPress");
	break;
    case ButtonRelease:
	(void) fprintf(stderr, "ButtonRelease");
	break;
    case MotionNotify:
	(void) fprintf(stderr, "MotionNotify");
	break;
    case EnterNotify:
	(void) fprintf(stderr, "EnterNotify");
	break;
    case LeaveNotify:
	(void) fprintf(stderr, "LeaveNotify");
	break;
    case FocusIn:
	(void) fprintf(stderr, "FocusIn");
	break;
    case FocusOut:
	(void) fprintf(stderr, "FocusOut");
	break;
    case KeymapNotify:
	(void) fprintf(stderr, "KeymapNotify");
	break;
    case Expose:
	(void) fprintf(stderr, "Expose");
	break;
    case GraphicsExpose:
	(void) fprintf(stderr, "GraphicsExpose");
	break;
    case NoExpose:
	(void) fprintf(stderr, "NoExpose");
	break;
    case VisibilityNotify:
	(void) fprintf(stderr, "VisibilityNotify");
	break;
    case CreateNotify:
	(void) fprintf(stderr, "CreateNotify");
	break;
    case DestroyNotify:
	(void) fprintf(stderr, "DestroyNotify");
	break;
    case UnmapNotify:
	(void) fprintf(stderr, "UnmapNotify");
	break;
    case MapNotify:
	(void) fprintf(stderr, "MapNotify");
	break;
    case MapRequest:
	(void) fprintf(stderr, "MapRequest");
	break;
    case ReparentNotify:
	(void) fprintf(stderr, "ReparentNotify");
	break;
    case ConfigureNotify:
	(void) fprintf(stderr, "ConfigureNotify");
	break;
    case ConfigureRequest:
	(void) fprintf(stderr, "ConfigureRequest");
	break;
    case GravityNotify:
	(void) fprintf(stderr, "GravityNotify");
	break;
    case ResizeRequest:
	(void) fprintf(stderr, "ResizeRequest");
	break;
    case CirculateNotify:
	(void) fprintf(stderr, "CirculateNotify");
	break;
    case CirculateRequest:
	(void) fprintf(stderr, "CirculateRequest");
	break;
    case PropertyNotify:
	(void) fprintf(stderr, "PropertyNotify");
	break;
    case SelectionClear:
	(void) fprintf(stderr, "SelectionClear");
	break;
    case SelectionRequest:
	(void) fprintf(stderr, "SelectionRequest");
	break;
    case SelectionNotify:
	(void) fprintf(stderr, "SelectionNotify");
	break;
    case ColormapNotify:
	(void) fprintf(stderr, "ColormapNotify");
	break;
    case ClientMessage:
	(void) fprintf(stderr, "ClientMessage");
	break;
    case MappingNotify:
	(void) fprintf(stderr, "MappingNotify");
	break;
    default:
	if (evnt->type >= LASTEvent) {
	    (void) fprintf(stderr, "extension event #%d", evnt->type);
	} else {
	    (void) fprintf(stderr, "\nInternal Error in XUnhandledWireEvent!");
	}
	break;
    }
    (void) fputc('\n', stderr);
    return 0;
}
#endif	/* EVENTDEBUG defined */
