-- (C) Copyright International Business Machines Corporation 23 January 
-- 1990.  All Rights Reserved. 
--  
-- See the file USERAGREEMENT distributed with this software for full 
-- terms and conditions of use. 
-- File: userrm.p
-- Author: Andy Lowry
-- SCCS Info: @(#)userrm.p	1.2 3/13/90

-- This process filters requests to the system resource manager for
-- processes running on behalf of a particular user.  The filter
-- accepts requests according to the rManager interface, which does
-- not include user names, and adds user names to convert the requests
-- into the sysRManager interface.  This behavior cannot be
-- compromised by the user process, so once the user's identity has
-- been verified, this filter provides a secure method for
-- implementing any access control policy based on user ID's.

userRM: using (rManager, sysRManager, userRM)

process (Q: userRMQ)
  
declare
  initArgs: userRM;
  
  postQ: postResourceQ;
  postArgs: postResource;
  insertQ: insertResourceQ;
  insertArgs: insertResource;
  getQ: getResourceQ;
  getArgs: getResource;
  removeQ: getResourceQ;
  removeArgs: getResource;
  deleteQ: deleteResourceQ;
  deleteArgs: deleteResource;
  
  sysRM: sysRManager;
  userName: charString;

begin
  receive initArgs from Q;
  
  -- create ports for calls from user processes
  new initArgs.rm;
  new postQ; connect initArgs.rm.post to postQ;
  new insertQ; connect initArgs.rm.insert to insertQ;
  new getQ; connect initArgs.rm.get to getQ;
  new removeQ; connect initArgs.rm.remove to removeQ;
  new deleteQ; connect initArgs.rm.delete to deleteQ;
  
  -- save the system resource manager and the user name
  sysRM := initArgs.sysRM;
  userName := initArgs.userName;
  
  -- return filtered capabilities and start filtering
  return initArgs;
  discard Q;
  
  while 'true' repeat
    select
      -- in each case we pass the augmented call on to the system
      -- resource manager and propogate any exception returns
    event postQ
      receive postArgs from postQ;
      block begin
	call sysRM.post(userName, postArgs.resourceName, 
	  postArgs.resource, postArgs.accessFunction);
	return postArgs;
      on (sysPostResource.duplicateName)
	return postArgs exception duplicateName;
      on (others)
	discard postArgs;
      end block;
      
    event insertQ
      receive insertArgs from insertQ;
      block begin
	call sysRM.insert(userName, insertArgs.resourceName,
	  insertArgs.resource, insertArgs.accessFunction);
	return insertArgs;
      on (sysInsertResource.duplicateName)
	return insertArgs exception duplicateName;
      on (others)
	discard insertArgs;
      end block;
      
    event getQ
      receive getArgs from getQ;
      block begin
	getArgs.resource <- 
	    sysRM.get(userName, getArgs.resourceName, getArgs.parameter);
	return getArgs;
      on (sysGetResource.notFound)
	return getArgs exception notFound;
      on (sysGetResource.accessDenied)
	return getArgs exception accessDenied;
      on (others)
	discard getArgs;
      end block;
      
    event removeQ
      receive removeArgs from removeQ;
      block begin
	removeArgs.resource <- sysRM.remove(userName,
	  removeArgs.resourceName, removeArgs.parameter);
	return removeArgs;
      on (sysGetResource.notFound)
	return removeArgs exception notFound;
      on (sysGetResource.accessDenied)
	return removeArgs exception accessDenied;
      on (others)
	discard removeArgs;
      end block;
      
    event deleteQ
      receive deleteArgs from deleteQ;
      block begin
	call sysRM.delete(userName, deleteArgs.resourceName);
	return deleteArgs;
      on (sysDeleteResource.notOwner)
	return deleteArgs exception notOwner;
      on (others)
	discard deleteArgs;
      end block;
      
    otherwise
      -- can't happen
    end select;
  end while;
end process
