-- (C) Copyright International Business Machines Corporation 23 January 
-- 1990.  All Rights Reserved. 
--  
-- See the file USERAGREEMENT distributed with this software for full 
-- terms and conditions of use. 
-- File: appt_scheduler.p
-- Author: Arthur P. Goldberg

appt_scheduler: USING( 
    stdenv,
    terminalio,
    safewindow,
    appt_scheduler_init, 
    appt_scheduler_member, 
    appt_scheduler_operator, 
    appt_scheduler_registrant, 
    appt_scheduler_structs, 
    events, 
    times,
    timesaux) 

LINKING( comp_times, intersect_time_ranges, insert_tr_in_ts, union_time_sets, not_time_set, intersect_time_sets, difference_time_sets ) 

  PROCESS
   
   ( Init_Port : Appt_Scheduler_Init_In ) 
  DECLARE
    InitCM : Appt_Scheduler_Init; 
    
    -- call messages
    Initiate_PlanCM : Initiate_Plan;
    Initiate_ScheduleCM : Initiate_Schedule;
    Initiate_CancelCM : Initiate_Cancel;
    List_MembersCM : List_Members;
    List_EventsCM : List_Events;
    
    DepartCM : Depart;
    Add_MemberCM : Add_Member;
    Shutdown_Appt_SchedulerCM : Shutdown_Appt_Scheduler;
    Reset_Appt_SchedulerCM : Reset_Appt_Scheduler;
    
    -- output ports
    std: stdenv; -- standard environment
    
    -- input ports
    -- from operator
    Shutdown_Appt_Scheduler: Shutdown_Appt_Scheduler_In;
    Reset_Appt_Scheduler: Reset_Appt_Scheduler_In;
    -- from registrant
    Add_Member: Add_Member_In;
    -- from members
    Initiate_Plan : Initiate_Plan_In ;
    Initiate_Schedule : Initiate_Schedule_In ;
    Initiate_Cancel : Initiate_Cancel_In ;
    List_Members : List_Members_In ;
    List_Events : List_Events_In ;
    Depart : Depart_In ;
    
    -- tables
    All_Members : Members;
    Group_Events : Table_Of_Events;
    Member_To_Port_Map : Member_To_Ports_Table ; 
    
    -- procedures
    union_time_sets : Time_Set_In2_Out1_Procedure_Caller;
    not_time_set : Time_Set_In1_Out1_Procedure_Caller;
    intersect_time_sets : Time_Set_In2_Out1_Procedure_Caller;
    intersect_time_ranges: Time_Range_In2_Out1_Procedure_Caller;
    difference_time_sets : Time_Set_In2_Out1_Procedure_Caller;
    comp_times : Time_Value_Call2_Return_Boolean_INTFACE_Caller;
    insert_tr_in_ts : Ins_TR_In_TS_Procedure_Caller;
    
    -- trace information
    opentracewindow: safewindowFn;
    tracewindow: terminalCM;
    
    -- variables and constants
    Running : Boolean;
    
    -- Always = [ -OO, +OO )
    Always : Time_Range;
    -- Never = [ +OO, +OO )
    Never : Time_Range;
    
  BEGIN
    
    new Always;
    unite Always.Begin_Time.The_Infinity from 'Neg_Inf';
    unite Always.End_Time.The_Infinity from 'Pos_Inf';
    
    new Never;
    unite Never.Begin_Time.The_Infinity from 'Pos_Inf';
    unite Never.End_Time.The_Infinity from 'Pos_Inf';
    
    -- create procedures 
    Comp_Times <- procedure of process comp_times; 
    insert_tr_in_ts <- procedure of process insert_tr_in_ts ; 
    intersect_time_sets <- procedure of process intersect_time_sets ; 
    intersect_time_ranges <- procedure of process intersect_time_ranges;
    union_time_sets <- procedure of process union_time_sets ; 
    not_time_set <- procedure of process not_time_set ; 
    difference_time_sets <- procedure of process difference_time_sets ; 
    
    -- create input ports
    -- from registrant
    new Add_Member ;
    -- from operator
    new Shutdown_Appt_Scheduler ;
    new Reset_Appt_Scheduler ;
    -- from members
    new  Initiate_Plan ;
    new  Initiate_Schedule ;
    new  Initiate_Cancel ;
    new  List_Members ;
    new  List_Events ;
    new  Depart ;
    
    -- get initialized
    receive InitCM from Init_Port;
    unwrap opentracewindow from InitCM.rm.get("master", "OpenTraceWindow", "appt_scheduler") {init};
    tracewindow <- opentracewindow("trace appt_scheduler ", "-Wh 12 -Ww 40 -inverse");

    
    std := InitCM.std;
    
    -- return output ports connected to Appt_Scheduler services
    connect InitCM.Add_Member to Add_Member ;
    connect InitCM.Shutdown_Appt_Scheduler to Shutdown_Appt_Scheduler ;
    connect InitCM.Reset_Appt_Scheduler to Reset_Appt_Scheduler ;
    call tracewindow.terminal.PutLine("Initialized");
    return InitCM;
    
    -- create tables
    new All_Members;
    new Group_Events;
    new Member_To_Port_Map;
    
    -- handle calls from the Operator and the Members until shutdown
    Running := 'true';
    while (Running) repeat
        -- handle calls
        select
            -- call from a registrant
          event Add_Member
            receive Add_MemberCM from Add_Member;
            call tracewindow.terminal.PutLine("Received Add_Member request from "|Add_MemberCM.WhoAmI.Name);
            block 
              begin
                insert evaluate Member: Appt_Schd2Member_Ports from
                    new Member;
                    Member.Member_ID := Add_MemberCM.WhoAmI.Name;
                    Member.Request_Plan_Info := Add_MemberCM.Request_Plan_Info;
                    Member.Inform_Schedule := Add_MemberCM.Inform_Schedule;
                    Member.Inform_Cancel := Add_MemberCM.Inform_Cancel;
                    Member.Shutdown := Add_MemberCM.Shutdown;
                    Member.Reset_Member := Add_MemberCM.Reset_Member;
                  end into Member_To_Port_Map;
                insert copy of Add_MemberCM.WhoAmI into All_Members;
                connect Add_MemberCM.Initiate_Plan to Initiate_Plan;
                connect Add_MemberCM.Initiate_Schedule to Initiate_Schedule;
                connect Add_MemberCM.Initiate_Cancel to Initiate_Cancel;
                connect Add_MemberCM.List_Members to List_Members;
                connect Add_MemberCM.List_Events to List_Events;
                connect Add_MemberCM.Depart to Depart;
                call tracewindow.terminal.PutLine("Returning");
                return Add_MemberCM;
              on (DuplicateKey)
                call tracewindow.terminal.PutLine("Returning with Add_Failed");
                return Add_MemberCM exception Add_Failed;
              end block;
            
            -- calls from the operator
          event Shutdown_Appt_Scheduler 
            receive Shutdown_Appt_SchedulerCM from Shutdown_Appt_Scheduler;
            call tracewindow.terminal.PutLine("Received Shutdown");
            Running <- 'false';
            call tracewindow.terminal.PutLine("Returning");
            return Shutdown_Appt_SchedulerCM;
            
          event Reset_Appt_Scheduler 
            receive Reset_Appt_SchedulerCM from Reset_Appt_Scheduler;
            call tracewindow.terminal.putLine("Received Reset");
            call tracewindow.terminal.PutLine("Returning");
            return Reset_Appt_SchedulerCM;
            
            
            -- calls from a member
            
          event Initiate_Plan 
            receive Initiate_PlanCM from Initiate_Plan;
            call tracewindow.terminal.PutLine("Received Initiate_Plan from "|Initiate_PlanCM.Organizer);
            -- do intersection(Desired_Times, 
            -- for each participant Available_Times in Personal_Appts,
            -- for each participant Available_Times in Group_Appts)
            block 
              DECLARE
                Busy_Times : Time_Set;  -- times when participants are busy
                Available_Times : Time_Set;
              begin
                
                new Initiate_PlanCM.Uncontacted_Participants;
                new Initiate_PlanCM.Available_Times;
                
                -- get union of all busy times of participants in group_appts 
                new Busy_Times;
                -- for each event 
                -- for each participant
                -- if the participant is involved in the event add it to the union
                
                -- Optimization: take advantage of Desired_Times to limit search
                for The_Event in Group_Events [] inspect
                    for Participant in Initiate_PlanCM.Desired_Participants [] inspect
                        if (exists of The_Event.Participants[Participant]) then
                            call insert_tr_in_ts( The_Event.Scheduled_Time, Busy_Times );
                          end if;
                      end for;
                  end for;
                
                -- get intersection of available times of participants in private_appts
                Available_Times := not_time_set( Busy_Times );
                
                new Initiate_PlanCM.Uncontacted_Participants;
                for Participant in Initiate_PlanCM.Desired_Participants where(Participant <> Initiate_PlanCM.Organizer) inspect
                    block 
                      declare
                        Uncontacted_Participant : charstring;
                        New_Available_Times : Time_Set;
                      begin
                        inspect Appt_Schd2Member_Ports in Member_To_Port_Map[Participant] 
                          begin
                            call tracewindow.terminal.PutLine("Calling Request Plan Info for member "|Participant);
                            New_Available_Times := intersect_time_sets( Available_Times,
                                
                                Appt_Schd2Member_Ports.Request_Plan_Info(Initiate_PlanCM.Desired_Times));
                            call tracewindow.terminal.PutLine("Call returned");
                            Available_Times := New_Available_Times;
                          end inspect;
                      on (NotFound) -- Problem: This is an inconsistency
                        call std.terminal.putline("Request_Plan_Info: given name of Participant not in Member_To_Port_Map");
                        -- participant not contacted, so add to the list
                      on (Disconnected)
                        Uncontacted_Participant := Participant;
                        insert Uncontacted_Participant into Initiate_PlanCM.Uncontacted_Participants;
                      on (Request_Plan_Info.Discarded)
                        Uncontacted_Participant := Participant;
                        insert Uncontacted_Participant into Initiate_PlanCM.Uncontacted_Participants;
                      end block;
                    
                  end for;
                
                Initiate_PlanCM.Available_Times := Available_Times;
              end block;
            call tracewindow.terminal.PutLine("Returning");
            return Initiate_PlanCM;
            
          event Initiate_Schedule 
            receive Initiate_ScheduleCM from Initiate_Schedule;
            call tracewindow.terminal.PutLine("Received Initiate Schedule from "|Initiate_ScheduleCM.Organizer);
            -- create Event_ID
            -- try to schedule all participants other than organizer
            -- check against Group_Events
            -- check against Personal_Events
            
            -- on (Not_Scheduled) exception
            -- on (Disconnected)
            -- on (Request_Plan_Info.Discarded)
            -- add participant to list of Unscheduled participants
            -- if empty list of Unscheduled participants
            --  store in Group_Events 
            --  return with Event_ID
            -- else
            --  cancel at each scheduled participant
            --  return with Not_Scheduled exception, list of Unscheduled participants
            block 
              declare
                Unscheduled_Participant : charstring;
                Unscheduled_Participants : Member_IDs;
                Time_Range:Time_Range;
                The_Event:An_Event;
              begin
                
                new The_Event;
                
                The_Event.Scheduled_Time := Initiate_ScheduleCM.Scheduled_Time ;
                The_Event.Organizer := Initiate_ScheduleCM.Organizer ;
                The_Event.Participants := Initiate_ScheduleCM.Participants ;
                -- Problem: should be real current time
                The_Event.Planned_At := Never;
                The_Event.Annotation := Initiate_ScheduleCM.Annotation ;
                The_Event.Location := Initiate_ScheduleCM.Location ;
                The_Event.Event_ID := unique;
                
                new Unscheduled_Participants;
                
                for Participant in Initiate_ScheduleCM.Participants [] inspect
                    block begin
                        -- check against Group_Events
                        for A_Group_Event in Group_Events where(exists of A_Group_Event.Participants[Participant]) inspect
                            
			    call intersect_time_ranges(A_Group_Event.Scheduled_Time, 
				Initiate_ScheduleCM.Scheduled_Time,Time_Range);
			    if Time_Range <> Never then 
				Unscheduled_Participant := Participant;
				insert Unscheduled_Participant into Unscheduled_Participants;
				exit participant_cant_schedule;
				
			      end if;
                          end for;
                        
                        
                        -- check against individual members except organizer
                        if Participant <> Initiate_ScheduleCM.Organizer then
			    block begin
				inspect Appt_Schd2Member_Ports in Member_To_Port_Map[Participant] 
				  begin
				    call tracewindow.terminal.PutLine("Calling Inform Schedule for member "|Participant);
				    call Appt_Schd2Member_Ports.Inform_Schedule(The_Event);
				    call tracewindow.terminal.PutLine("Call returned");
				  end inspect;
			      on (NotFound) -- Problem: This is an inconsistency
				call std.terminal.putline("Inform_Schedule: given name of Participant not in Member_To_Port_Map");
				insert copy of Participant into Unscheduled_Participants;
				exit participant_cant_schedule;
				-- participant not contacted, so add to the list
			      on (Disconnected, Inform_Schedule.Discarded, Inform_Schedule.Not_Scheduled)
			        call tracewindow.terminal.PutLine("Call returned unsuccessfully");
				Unscheduled_Participant := Participant;
				insert Unscheduled_Participant into Unscheduled_Participants;
				exit participant_cant_schedule;
			      end block;
			  end if;
                        
                        -- prevents duplicate key exception
                        -- for insert Unscheduled_Participant into Unscheduled_Participants;
                      on exit (participant_cant_schedule)
                      end block;
                  end for;
                
                if (size of Unscheduled_Participants = 0) then
                    -- great! it was scheduled!
                    --  return with Event_ID
                    Initiate_ScheduleCM.Event_ID := The_Event.Event_ID;
                    --  store in Group_Events 
                    insert The_Event into Group_Events;
                    call tracewindow.terminal.PutLine("Returning normally"); 
                    return Initiate_ScheduleCM;
                  else
                    --  cancel each scheduled participant except organizer
                    for A_Scheduled_Participant in Initiate_ScheduleCM.Participants where(not exists of Unscheduled_Participants[A_Scheduled_Participant] and A_Scheduled_Participant <> Initiate_ScheduleCM.Organizer) inspect
                        block 
                          begin
                            inspect Appt_Schd2Member_Ports in Member_To_Port_Map[A_Scheduled_Participant] 
                              begin
                                call tracewindow.terminal.PutLine("Calling Inform Cancel for member "|A_Scheduled_Participant);
                                call Appt_Schd2Member_Ports.Inform_Cancel(The_Event.Event_ID);
                                call tracewindow.terminal.PutLine("Call returned");
                              end inspect;
                          on (NotFound)
                            call std.terminal.putline("Should not occur-- member no longer exists:" | A_Scheduled_Participant);
                          on (Inform_Cancel.Not_Cancelled) -- Problem: This is an inconsistency
                            call tracewindow.terminal.PutLine("Call returned with Not_Cancelled exception");
                            call std.terminal.putline("Scheduled but could not cancel:" | A_Scheduled_Participant);
                            -- Problem: participant not contacted, need to do somethin 
                          on (Disconnected)
                            call tracewindow.terminal.PutLine("Call returned with Disconnected exception");
                           call std.terminal.putline("Scheduled but disconnected trying to cancel:" | A_Scheduled_Participant);
                          end block;
                        
                      end for;
                    call tracewindow.terminal.PutLine("Returning with Not Scheduled exception");
                    --  return with Not_Scheduled exception, list of Unscheduled participants
                    Initiate_ScheduleCM.Unscheduled_Participants := Unscheduled_Participants;
                    return Initiate_ScheduleCM exception Not_Scheduled;
                  end if;
                
              end block;
            
            
          event Initiate_Cancel 
            receive Initiate_CancelCM from Initiate_Cancel;
            call tracewindow.terminal.PutLine("Received Initiate Cancel from member "|Initiate_CancelCM.Organizer);
            block
              declare
                
                Cancelled_Event: An_Event;
              begin
                remove Cancelled_Event from An_Event in Group_Events where 
                   ((An_Event.Event_ID = Initiate_CancelCM.Event_ID) and
                       (An_Event.Organizer = Initiate_CancelCM.Organizer)); 
                
                block begin
                    for Participant in Cancelled_Event.Participants where(Participant <> Initiate_CancelCM.Organizer) inspect
                        block 
                          begin
                            inspect Appt_Schd2Member_Ports in Member_To_Port_Map[Participant] 
                              begin
                                call tracewindow.terminal.PutLine("Calling Inform Cancel for member "|Participant);
                                call Appt_Schd2Member_Ports.Inform_Cancel(Cancelled_Event.Event_ID);
                                call tracewindow.terminal.PutLine("Call returned");
                              end inspect;
                          on (Inform_Cancel.Not_Cancelled)
                            call tracewindow.terminal.PutLine("Call returned with Not Cancelled exception");
                            call std.terminal.putline("Inform_Cancel: participant would not cancel");
                          on (NotFound) -- Problem: This is an inconsistency
                            call std.terminal.putline("Inform_Cancel: given name of Participant not in Member_To_Port_Map");
                          on (Disconnected, Inform_Cancel.Discarded)
                            call tracewindow.terminal.PutLine("Call returned with Disconnected or Discarded exception");
                            call std.terminal.putline("Inform_Cancel: participant is gone");                    
                          end block;
                      end for;
                  end block;
                call tracewindow.terminal.PutLine("Returning");
                return Initiate_CancelCM;
                
                --  Problem:  more semantics needed
              on (NotFound)
                call tracewindow.terminal.PutLine("Returning with Cancel Not Permitted exception");
                return Initiate_CancelCM exception Cancel_Not_Permitted;
              end block;
            
          event List_Members 
            receive List_MembersCM from List_Members;
            call tracewindow.terminal.PutLine("Received List Members Rq");
            List_MembersCM.The_Members := All_Members; 
            call tracewindow.terminal.PutLine("Returning");
            return List_MembersCM;
            
          event List_Events 
            receive List_EventsCM from List_Events;
            call tracewindow.terminal.PutLine("Received List Events Rq from "|List_EventsCM.Organizer);
            List_EventsCM.The_Events := every of An_Event in Group_Events where 
               ( exists of An_Event.Participants[List_EventsCM.Organizer]);
            call tracewindow.terminal.PutLine("Returning");
            return List_EventsCM;
            
          event Depart 
            receive DepartCM from Depart;
            call tracewindow.terminal.PutLine("Received Depart from member "|DepartCM.Organizer);
            block
              declare
                member: member;
                memberports: appt_schd2member_ports;
              begin
                remove member from m in all_members where(m.name = departcm.organizer);
                discard member;
                remove memberports from member_to_port_map[departcm.organizer];
                discard memberports;
              on (NotFound)
              end block;
            call tracewindow.terminal.PutLine("Returning");
            return DepartCM;
            
            
          otherwise
          end select;
      end while;
    call tracewindow.terminal.PutLine("Shutting Down");
    -- cleanup prior to shutdown
  end process
