-- (C) Copyright International Business Machines Corporation 23 January 
-- 1990.  All Rights Reserved. 
--  
-- See the file USERAGREEMENT distributed with this software for full 
-- terms and conditions of use. 
-- SCCS Info: @(#)unix.d	1.5 1/25/92

-- Interfaces for using the I/O support built into the interpreter
unix: using(terminalIO) definitions
  
  -- type specifiers for the various flavors open calls
  openType: enumeration (
    'read',			-- open for reading
    'write',			-- truncate or create for writing
    'append',			-- open for writing at end of file, or
				-- create for writing
    'updateRead',		-- open for reading and writing
    'updateWrite',		-- truncate or create for reading and writing
    'updateAppend'		-- create open at EOF for reading and writing
  );
  
  popenType: enumeration (	-- only two types allowed for popen
    'read',			-- read from child process' stdout
    'write'			-- write to child process' stdin
  );


  -- file status info record returned by fstat
  fileStatus: record (
    device: integer,		-- device number on which file resides
    inodeNumber: integer,	-- inode in filesystem
    mode: integer,		-- file protection bits
    nlink: integer,		-- number of hard links
    uid: integer,		-- user id of file's owner
    gid: integer,		-- id of group to which file belongs
    deviceID: integer,		-- device id for special file
    size: integer,		-- number of bytes in the file
    lastAccess: integer,	-- encoded time of last access
    lastModification: integer,	-- encoded time of last modification
    lastStatusChange: integer,	-- encoded time of last change of status
    blockSize: integer,		-- preferred block size for file system I/O
    blocks: integer		-- number of blocks allocated
  );
  
  -- file access types
  fileAccessType: enumeration ('read', 'write', 'execute', 'exists');

  -- capabilities for creating streams, plus a few other miscellaneous
  -- routines
  stdio: record (
    -- streams built from stdin and friends
    stdin: stream,
    stdout: stream,
    stderr: stream,
    -- capabilities for creating new streams
    fopen: fopenFn,
    fdopen: fdopenFn,
    popen: popenFn,
    tmpfile: tmpfileFn,
    -- miscellaneous capabilities
    clearerrno: clearerrnoFn,
    ctermid: ctermidFn,
    cuserid: cuseridFn,
    tmpnam: tmpnamFn,
    access: fileAccessFn,
    readErrno: readErrnoFn
  );

  -- a stream is represented as a collection of capabilities for
  -- performing various operations on the stream
  stream: record (		
    -- input capabilities
    getc: getCharFunc,		-- read one char from the stream
    gets: getStringFunc,	-- read to newline, strip the newline
    fgets: getBoundedStringFunc,-- read n characters or up to newline
    fread: getBoundedStringFunc,-- read n characters
    getw: getWordFunc,		-- read a word as an integer value
    -- output capabilities
    putc: putCharFunc,		-- write one char to the stream
    puts: putStringFunc,	-- write string plus newline to stream
    fputs: putStringFunc,	-- write string to stream
    fwrite: putSubStringFunc,	-- write n chars to stream
    putw: putWordFunc,		-- write integer value as a word to stream
    -- miscellaneous capabilities
    fclose: fcloseFn,		-- close the stream
    pclose: pcloseFn,		-- close stream created with popen
    freopen: freopenFn,		-- reopen a stream with another file
    clearerr: clearerrFn,	-- clear error condition on the stream
    ferror: ferrorFn,		-- test error condition on the stream
    ferrno: readErrnoFn,	-- get last error code on the stream
    feof: feofFn,		-- test for EOF condition on the stream
    fflush: fflushFn,		-- flush output pending on the stream
    fseek: fseekFn,		-- set position of stream for I/O
    rewind: rewindFn,		-- reset stream to beginning
    ftell: ftellFn,		-- get position of stream for I/O
    setlinebuf: setlinebufFn,	-- set line buffered mode for stream output
    fstat: fstatFn,		-- get information about file on stream
    fileno: filenoFn		-- get file descriptor underlying stream
  );
  
  clearerrno: callmessage ()
  exit {};
  clearerrnoQ: inport of clearerrno {};
  clearerrnoFn: outport of clearerrnoQ;

  ctermid: callmessage (
    id: charstring		-- terminal id
  )
  exit {full};
  ctermidQ: inport of ctermid {};
  ctermidFn: outport of ctermidQ;
  
  cuserid: callmessage (
    id: charstring		-- user id
  )
  exit {full};
  cuseridQ: inport of cuserid {};
  cuseridFn: outport of cuseridQ;
  
  tmpnam: callmessage (
    name: charstring		-- generated name
  )
  exit {full};
  tmpnamQ: inport of tmpnam {};
  tmpnamFn: outport of tmpnamQ;
  
  fileAccess: callmessage (
    filename: charstring,	-- filename to check access on
    mode: fileAccessType,	-- type of access to check
    ok: boolean			-- is the access allowed?
  )
  constant (filename, mode)
  exit {full};
  fileAccessQ: inport of fileAccess {init(filename),init(mode)};
  fileAccessFn: outport of fileAccessQ;

  readErrno: callmessage (
    errno: integer		-- most recent error code
  )
  exit {full};
  readErrnoQ: inport of readErrno {};
  readErrnoFn: outport of readErrnoQ;

  fopen: callmessage (
    filename: charstring,	-- name of file to open
    type: openType,		-- how to open the file
    stream: stream		-- the stream that was opened
  )
  constant (filename, type)
  exit {full}
  exception couldNotOpen {init(filename), init(type)};
  fopenQ: inport of fopen {init(filename), init(type)};
  fopenFn: outport of fopenQ;
  
  fdopen: callmessage (
    filedes: integer,		-- file descriptor from which to
				-- create stream
    type: openType,		-- how to open the stream
    stream: stream		-- the stream that was opened
  )
  constant (filedes, type)
  exit {full}
  exception couldNotOpen {init(filedes), init(type)};
  fdopenQ: inport of fdopen {init(filedes), init(type)};
  fdopenFn: outport of fdopenQ;
  
  popen: callmessage (
    command: charstring,	-- command to run in child process
    type: popenType,		-- read from stdout or write to stdin
				-- of child
    stream: stream		-- the stream that was opened
  )
  constant (command, type)
  exit {full}
  exception couldNotOpen {init(command), init(type)};
  popenQ: inport of popen {init(command), init(type)};
  popenFn: outport of popenQ;
  
  freopen: callmessage (
    filename: charstring,	-- new filename to be opened on the stream
    type: openType		-- how to open the new file
  )
  constant (filename, type)
  exit {full}
  exception CouldNotOpen {full};
  freopenQ: inport of freopen {full};
  freopenFn: outport of freopenQ;
  
  tmpfile: callmessage (
    stream: stream		-- stream that was opened
  )
  exit {full}
  exception couldNotOpen {};
  tmpfileQ: inport of tmpfile {};
  tmpfileFn: outport of tmpfileQ;
  
  -- operations on streams

  clearerr: callmessage ()
  exit {};
  clearerrQ: inport of clearerr {};
  clearerrFn: outport of clearerrQ;
  
  ferror: callmessage (
    result: boolean		-- did an error occur on the stream?
  )
  exit {full};
  ferrorQ: inport of ferror {};
  ferrorFn: outport of ferrorQ;
  
  feof: callmessage (
    result: boolean		-- did EOF occur on the stream?
  )
  exit {full};
  feofQ: inport of feof {};
  feofFn: outport of feofQ;
  
  fileno: callmessage (
    fileno: integer		-- file descriptor underlying this stream
  )
  exit {full};
  filenoQ: inport of fileno {};
  filenoFn: outport of filenoQ;
  
  fflush: callmessage ()
  exit {};
  fflushQ: inport of fflush {};
  fflushFn: outport of fflushQ;
  
  seekType: enumeration ('BeginningOfFile', 'CurrentPosition', 'EndOfFile');
  fseek: callmessage (
    offset: integer,		-- position to seek to
    seekType: seekType		-- position relative to which offset
				-- is interpreted
  )
  constant (offset, seekType)
  exit {full}
  exception couldNotSeek {full};
  fseekQ: inport of fseek {full};
  fseekFn: outport of fseekQ;
  
  rewind: callmessage ()
  exit {};
  rewindQ: inport of rewind {};
  rewindFn: outport of rewindQ;
  
  ftell: callmessage (
    offset: integer		-- current position in stream
  )
  exit {full}
  exception Failure {};
  ftellQ: inport of ftell {};
  ftellFn: outport of ftellQ;

  setlinebuf: callmessage ()
  exit {};
  setlinebufQ: inport of setlinebuf {};
  setlinebufFn: outport of setlinebufQ;
  
  fstat: callmessage (
    fileStatus: fileStatus
  )
  exit {full};
  fstatQ: inport of fstat {};
  fstatFn: outport of fstatQ;

  fclose: callmessage ()
  exit {}
  exception Failure {};
  fcloseQ: inport of fclose {};
  fcloseFn: outport of fcloseQ;
  
  pclose: callmessage (
  )
  exit {};
  pcloseQ: inport of pclose {};
  pcloseFn: outport of pcloseQ;
  
end definitions

