-- (C) Copyright International Business Machines Corporation 23 January 
-- 1990.  All Rights Reserved. 
--  
-- See the file USERAGREEMENT distributed with this software for full 
-- terms and conditions of use. 
-- File: type.bs
-- Author: Dan Yellin and Andy Lowry
-- SCCS Info: @(#)type.bs	1.6 1/25/92

-- This process performs type checking and type inferencing on a
-- Hermes absprog process (an object of type predefined!proc).  

-- Overview: We keep a work list of clauses remaining to be checked.
-- Initially this worklist contains only the program main clause.
-- Each time a clause is removed from the work list, all its
-- statements are checked before another clause is processed (this is
-- crucial in certain cases for the correctness of the algorithm).

-- The following steps are involved in checking a single statement:
--   1. Insert any embedded clauses into the work list so they will be
--   processed later.

--   2. If an embedded clause computes a value used in this statement,
--   look in the statement qualifier for the variable that will hold
--   the value, and enter its type (as determined from the current
--   statement) into the type resolution table.

--   3. Check each operand in the current statement, using rules
--   retrieved from the checking table according to this statement's
--   operation code.  Build and collect error objects for all
--   violations.

--   4. If any operand's type is unknown and cannot yet be inferred,
--   add an entry to the backpatch table so that is type will be
--   inferred as soon as all required information is available.

--   5. If any operand's type is unknown and can be inferred, add its
--   type to the resolution table.  Remove all entries from the
--   backpatch table that were waiting for this type, and reapply the
--   indicated inference or class rules.  Build and collect error
--   objects for all violations.

-- When all the clauses have been processed, the backpatch table
-- should be empty.  If it is not, the remaining entries represent
-- types that could not be inferred.  Build and collect error objects
-- for all such entries.

-- Finally, if there were any string_literal or named_literal
-- statements in the program, the characters in the strings and the
-- named literals need to be checked to make sure they appear in the
-- enumeration to which they belong.

-- Note that becuase of the manner in which the backpatch table
-- entries are processed (each entry is processed as soon as the
-- required information is available), a single pass over the program
-- suffices to perform all necessary checking and inferencing.

-- Following are some examples of type checking and inferencing:


-- Example 1: "r <- convertof(i)" becomes, in absprog:
--	convert, t, i	   #1
--	<-, r, t           #2

-- In this case, when type checking stmt #1, the type of t is unknown.
-- When checking stmt #2, the type of t is inferred, the backpatch
-- table is consulted, and stmt #1 is now checked.  (If type checking
-- stmts in the reverse order, no backpatching would be performed).

-- Example 2: "bool <- (a+b) < (c+d)" (bool can be any bool type) becomes:
--	add, t1, a, b    #1
--	add, t2, c, d    #2
--	<, t3, t1, t2    #3
--	<-, bool, t3      #4

-- Checking #1 and #2 infers types of t1 and t2.  However, at #3,
-- <boolean(t3)> is added to the backpatch table, since the type
-- returned by < is a boolean family type, not necessarily
-- predefined!boolean.  When checking #4, the type of t3 is inferred
-- (it is the same as the type of bool) so <boolean(t3)> is removed
-- from the backpatch table and checked.  (If type checking is
-- performed in the reverse order, then: When checking stmt 3, t3 is
-- known but t1 and t2 are unknown.  Hence <orderedscalar(t1)>,
-- <orderedscalar(t2)>, <sameas(t1,t2)> and <sameas(t2,t1)> are all
-- added to the backpatch table.  When checking stmt 2, t2 becomes
-- known, so <orderedscalar(t2)> is removed from the backpatch table
-- and tested.  <sameas(t1,t2)> is also removed and used to infer t1.
-- This causes the remaining backpatch entries to be removed and
-- tested.)

-- Example 3: b <- (x or y) = ((u < v) = f(w)) becomes:
--	or, t1, x, y          #1
--	funcCall, t2, f, w    #2
--	<, t3, u, v           #3
--	=, t4, t2, t3         #4
--      =, t5, t1, t4         #5
--      <-, b, t5             #6
-- Checking #1 & 2 infers types of t1 & t2.  At #3, <boolean(t3)> is
-- added to the backpatch table.  At #4, the type of t3 is inferred,
-- so <boolean(t3)> is removed from the backpatch table and tested.
-- However, t4 is unknown, so <boolean(t4)> is added to backpatch.  At
-- #5, t4 is inferred but t5 is unknown, so <boolean(t4)> is removed
-- and checked, and <boolean(t5)> is added.  At #6, t5 is inferred and
-- <boolean(t5)> is removed from backpatch and tested.  (In reverse:
-- t5 is inferred at #6.  At #5, <sameas(t1,t4)> and <(sameas(t4,t1)>
-- are added to backpatch.  At #4, <boolean(t4)>, <sameas(t2,t3)>,
-- and <sameas(t3,t2)> are all added.  At #3, <boolean(t3)> is added.
-- At #2, t2 is inferred from the funcCall interface.  This triggers
-- removal of <sameas(t3,t2)>, inferring t3.  This in turn triggers
-- removal and testing of <boolean(t3)> and <sameas(t2,t3)>.  At #1,
-- t1 is inferred, and this causes <sameas(t4,t1)> to be removed and
-- used to infer t4.  This in turn causes removal and checking of
-- <boolean(t4)> and <sameas(t1,t4)>.  This completes checking of the
-- clause.)

-- Note that the above scheme works regardless of the order of
-- processing of the rules that apply to a particular statement.

-- make typemarks easier and less obtrusive... we need typemarks
-- because who's going to infer our types?!
#include "typemark.h"

type: using (checking_table, positions, inferredtype, errors, typecheck, type)
linking (
  qualchecker,stmtchecker,recheckinfer,findtype,finddef,interfacecheck,
  componentsoverlap,infer,assign,
#if FullChecking
  class, recheckclass,literalcheck
#else
  dummy_class
#endif
)

process (Q: typeCheckQueue)
  
declare
  args: typeCheckCall;
  inferred: inferredDefinitions;
  workList: setOfClauseId;
  curClause: clauseid;
  mainClause: clauseid;
  bpRec: backPatchRecord;
  backpatch: backPatchTable;
  newlyInferred: rootnames;
  errors: errors;
  qualCheck: qualCheckCapa;
  stmtCheck: stmtCheckCapa;
  recheckInfer: recheckInferCapa;
  findType: findTypeCapa;
  findDef: findDefCapa;
  componentsOverlap: componentsOverlapCapa;
  intfCheck: interfaceCheckCapa;
  assign: assignCapa;
  infer: inferCapa;
  position: aposition;
  class: classCapa;
  literalStmts: setOfAposition;
#ifdef FullChecking
  recheckClass: recheckClassCapa;
  literalCheck: literalCheckCapa;
#endif
begin
  receive args from Q;
  
  -- collect capabilities to type checking service processes
  qualCheck <- qualCheckCapa#(procedure of program#(process qualchecker));
  stmtCheck <- stmtCheckCapa#(procedure of program#(process stmtchecker));
  recheckInfer <- 
      recheckInferCapa#(procedure of program#(process recheckinfer));
  findType <- findTypeCapa#(procedure of program#(process findtype));
  findDef <- findDefCapa#(procedure of program#(process finddef));
  intfCheck <- 
      interfaceCheckCapa#(procedure of program#(process interfacecheck));
  componentsOverlap <- 
      componentsOverlapCapa#(procedure of program#(process componentsoverlap));
  -- infer, assign, and class capabilities are all returned by an
  -- initial call on process which thereafter fields and distributes
  -- the calls to appropriate specialized processes
  infer <- inferCapa#((setupInferCapa#(create of program#(process infer)))
	(findType, args.program, args.definitions));
  assign <- assignCapa#((setupAssignCapa#(create of program#(process assign)))
	(findType, args.program, args.definitions));
#if FullChecking
  class <- classCapa#((setupClassCapa#(create of program#(process class)))
	(findType, args.program, args.definitions));

  -- other capabilities not used in the bootstrapping type checker
  recheckClass <- 
      recheckClassCapa#(procedure of program#(process recheckclass));
  literalCheck <- 
      literalCheckCapa#(procedure of program#(process literalcheck));
#else
  -- for bootstrapping type checker we load a phony, do-nothing class
  -- checker 
  class <- classCapa#(procedure of program#(process dummy_class));
#endif

  -- initialize our "global" objects
  new inferred;
  new workList;
  new errors;
  new position;
  new backpatch;
  new literalStmts;

  -- put the main clause of the program into the worklist, and fire up
  -- the type checking loop!
  inspect scope in args.program.executable_part.scopes
	[args.program.executable_part.main_scope] begin
    insert clauseid#(copy of scope.clause) into workList;
    mainClause := scope.clause;	-- remember which clause is the main one
  end inspect;

  while B(I(size of workList) <> ZERO) repeat
    remove curClause from workList[];
    position.clause := curClause;

    inspect clause in args.program.executable_part.clauses[curClause] begin

#if FullChecking
      -- If we're checking the main clause, check the init port to make
      -- sure it's really an inport object.
      if curClause = mainClause then
	block declare
	  initObj: objectname;
	begin
	  -- set our "position" to be the first statement in the main
	  -- clause, for error reporting purposes
	  block begin
	    inspect firstStmt in clause.statements[] begin
	      position.statement := firstStmt.id;
	    end inspect;
	  on (NotFound)
	    -- main clause is empty??? Make up a bogus statement id
	    position.statement <- unique;
	  end block;
	  -- create an objectname for the initport
	  new initObj;
	  new initObj.root;
	  initObj.root.scope := args.program.executable_part.main_scope;
	  initObj.root.root := args.program.initport;
	  new initObj.components;
	  -- now check the initport type, maybe adding it to the
	  -- backpatch table (though this should never happen with a
	  -- program created by the front-end, since the concrete syntax
	  -- requires a declaration for the initport)
	  call class('inport', initObj, inferred, backpatch, errors, position);
	end block;
      end if;
#endif    
      
      -- loop over all the statements in the clause
      for stmt in clause.statements[] inspect
	position.statement := stmt.id;

	-- we'll keep a list of all newly inferred rootnames here...
	new newlyInferred;
	
	-- do qualifier-specific checking and inferencing, adding
	-- embedded clauses to the worklist
	call qualCheck(stmt, args.program.executable_part.scopes,
	  workList, inferred, newlyInferred, backpatch, literalStmts,
	  assign, infer, class, errors, position);

	-- perform normal table-driven checking and inferencing
	inspect chkRec in args.checking_table[stmt.operator] begin
	  call stmtCheck(stmt, curClause, args.program.executable_part.scopes,
	    args.definitions, inferred, backpatch, chkRec, errors,
	    findType, findDef, assign, infer, class, intfCheck, 
	    componentsOverlap, newlyInferred);
	end inspect;
	
	-- trigger any backpatch entries waiting for newly inferred
	-- variables
	while B(I(size of newlyInferred) <> ZERO) repeat
	  block declare
	    var: rootname;
	  begin
	    remove var from newlyInferred[];
	    block begin
	      while B('true') repeat
		remove bpRec from bp in backpatch
		    where (B(bp.triggerObj = var));
		select (ruleType#(case of bpRec.info))
		where (ruleType#'inference')
		  -- refire an inference rule... this time it should
		  -- not fail
		  reveal bpRec.info.infer;
		  call recheckInfer(bpRec.info.infer, inferred, 
		    newlyInferred, backpatch, errors, infer);
		
#if FullChecking
		where ('class')
		  reveal bpRec.info.class;
		  call recheckClass(bpRec.info.class, inferred,
		    errors, class);
#endif
		
		where (ruleType#'call')
		  reveal bpRec.info.call;
		  inspect bpClause in args.program.executable_part.clauses
			[bpRec.info.call.clause] begin
		    inspect bpStmt in clause.statements where
			  (B(bpStmt.id = bpRec.info.call.statement)) begin
		      call intfCheck(bpStmt, bpRec.info.call.clause,
			args.program.executable_part.scopes,
			args.definitions, inferred, backpatch,
			errors, findType, findDef, componentsOverlap,
                        newlyInferred);
		    end inspect;
		  end inspect;
		  
		otherwise
		  -- this should not happen
		end select;
	      end while;
	    on (notFound)
	      -- no more backpatch records for this variable
	    end block;
	  end block;
	end while;

      end for;

    end inspect;
  end while;

#if FullChecking
  -- all clauses have been processed, and we won't be making any more
  -- inferences.... now we need to check to make sure that all the
  -- string and named literal statements had proper qualifiers
  for litPos in literalStmts[] inspect
    inspect clause in args.program.executable_part.clauses[litPos.clause]
    begin
      inspect stmt in clause.statements where (B(stmt.id = litPos.statement))
      begin
	call literalCheck(stmt, litPos.clause,
	  args.program.executable_part.scopes, args.definitions,
	  inferred, errors, findType, findDef);
      end inspect;
    end inspect;
  end for;
#endif      

  -- If the backpatch table is nonempty, then generate an error
  -- message for each entry
  while B(I(size of backpatch) <> ZERO) repeat
    block declare
      error: error;
      errObj: errorObject;
      duplicates: backpatchTable;
      objname: objectname;
    begin
      remove bpRec from backpatch[];
      extract duplicates from bp in backpatch 
	  where (B(bp.triggerObj = bpRec.triggerObj));
      new error;
      error.code <- errorcode#'uninferred type';
      select ruleType#(case of bpRec.info)
      where (ruleType#'inference')
	reveal bpRec.info.infer;
	unite error.position.apos 
	    from aposition#(copy of bpRec.info.infer.position);
      where (ruleType#'class')
	reveal bpRec.info.class;
	unite error.position.apos 
	    from aposition#(copy of bpRec.info.class.position);
      where (ruleType#'call')
	reveal bpRec.info.call;
	unite error.position.apos from aposition#(copy of bpRec.info.call);
      otherwise
	exit cantHappen;
      end select;
      new error.objects;
      new objname;
      objname.root := bpRec.triggerObj;
      new objname.components;
      unite errObj.objectname from objname;
      insert errObj into error.objects;
      insert error into errors;
    end block;
  end while;

  -- done checking clauses... give back the information we gathered,
  -- and return with an exception if there were any errors
  args.errors <- errors;
  args.inferred <- inferred;

  if B(I(size of args.errors) > ZERO) then
    return args exception typeErrors;
  else
    return args;
  end if;
  
on exit(cantHappen)
  print charString#"CantHappen happened in type";
end process
