(* Gene Rollins (rollins@cs.cmu.edu)
   School of Computer Science, Carnegie-Mellon Univ., Pittsburgh, PA 15213 *)

functor SourceGroupFun
  (structure Data :DATA
   structure DirFile :DIRFILE
   structure Pathname :PATHNAME
   structure ListSort :LISTSORT
   structure Hash :HASH
   structure Hasher :HASHER
   structure MLdepends :MLDEPENDS
   structure NameRefTable :NAMEREFTABLE
   structure NamespaceTable :NAMESPACETABLE
   structure Connections :CONNECTIONS
   structure Names :NAMES
   structure Group :GROUP
   structure Util :UTIL
   structure Args :ARGS
   structure ToolInternals :TOOL_INTERNALS
   sharing Data = Group.Data = Args.Data = ToolInternals.Data
   sharing Names = MLdepends.Names
   sharing NameRefTable = Connections.NameRefTable = 
           Data.NameRefTable = Group.NameRefTable
   sharing Hash = Data.Hash
   sharing NamespaceTable = Data.NamespaceTable
  ) = struct

val version = 1.0

structure ToolInternals = ToolInternals
open Data
type time = System.Timer.time

exception SourceGroupInternalError
exception Skip

val printModifiedSources = ref true
val continueAfterError = ref true
val makeLibraries = ref true
val debugMake = ref false

val hasher = Hasher.hasher
val smlH = hasher "sml"
val signatureH =  hasher "signature"
val functorH = hasher "functor"

fun string'equal ((x, y):string*string) :bool = (x = y)
val normalize = ListSort.unique'sort string'equal String.<
val findFileThruGroups =Group.findFileThruGroups false SourceGroupInternalError

type fileref = bool * (string * int)
fun filerefName ((_,(name,_)):fileref) :string = name
fun filerefNameEqual (((_,(a,_)):fileref),((_,(b,_)):fileref)) = (a=b)
fun filerefNameLess (((_,(a,_)):fileref),((_,(b,_)):fileref)) = (a<b)
val filerefNormalize = ListSort.unique'sort filerefNameEqual filerefNameLess

fun analyzeNameUsage (redefine:bool) (filename :string) =
  let val (sigexports1, sigimports1, funexports1) =
            (Names.break (MLdepends.parse filename))
              handle _ =>
                (if redefine
                   then Util.wn ["Parsing error in file ", filename,
                            "; assuming no interface changes"]
                   else Util.wn ["Parsing error in file ", filename,
                            "; not included in group"];
                 raise Skip)
      val (sigexports2, sigimports2, funexports) =
            (normalize sigexports1,normalize sigimports1,normalize funexports1)
      val (sigexports, sigimports) = Util.remDuplicates sigexports2 sigimports2
      val importTable = NameRefTable.create namespaceSize
      val exportTable = NameRefTable.create namespaceSize
  in
    NameRefTable.enter importTable signatureH sigimports;
    NameRefTable.enter exportTable signatureH sigexports;
    NameRefTable.enter exportTable functorH funexports;
    (importTable, exportTable)
  end

fun defineAuto (directory:string) (filelist :string list) =
  let val group :groupInfo = Group.newSubGroup NONE
      fun do'file filename =
       case Group.findFile group (hasher filename) of
          (SOME _) =>
            (Util.wn ["File ", filename, " already in group\n"]; NONE)
         | NONE =>
            (print filename; print "\n";
             let val (imports, exports) = analyzeNameUsage false filename in
               Group.defineSource (group, (Util.modtime filename), directory)
                                  ("", filename, imports, exports)
             end) handle Skip => NONE
      val infoList = map do'file filelist
  in
    Group.createNamespaces true group infoList;
    group
  end

fun defineManual (cwd:string)
      ((connFile :string), (acc :groupInfo list)) :groupInfo list =
  let val fullConnFilename = Util.absoluteName cwd connFile
      val _ = (print ("[Connections " ^ (Util.trim fullConnFilename) ^ "]\n"))
      val group :groupInfo = Group.newSubGroup (SOME fullConnFilename)
      val connections = Connections.get fullConnFilename
      val defTime = Util.modtime fullConnFilename
      val dir = Pathname.directoryPart fullConnFilename
      val infoList = map (Group.defineSource (group, defTime, dir)) connections
  in
    Group.createNamespaces true group infoList;
    group::acc
  end

fun setFileDependencies (group as G{main'groups,lib'groups,...} :groupInfo)
                        (filename,_) (F {imports,depends,...}:fileInfo) =
  let fun defining'file (spaceH as (space,_)) (name, acc) =
       (case Group.lookupThruGroups group spaceH (hasher name) of
           NONE => (Util.wn[space," ",name," undefined in ",filename]; acc)
         | (SOME fref) =>
             if (filerefName fref) = filename then acc else fref::acc)
      val result = filerefNormalize
                     (NameRefTable.nestedFold (!imports) defining'file [])
  in
    depends := result
  end

fun computeDependencies (mainGroup:groupInfo) =
  let fun doDependencies (group:groupInfo)
                         (isLibrary:bool) (_:unit) :unit =
        if isLibrary then ()
          else (Group.scanFiles group (setFileDependencies mainGroup); ())
  in Group.groupsFold mainGroup false doDependencies (); ()
  end

fun create (description :groupDescription list) :group =
  let val cwd = Util.getwd()
      val (autoFiles, connFiles, linkFiles, libGroups, includeGroups) = 
            Args.rearrange description
      val autoPart = defineAuto cwd autoFiles
      val newPart = fold (defineManual cwd) connFiles [autoPart]
      val mainGroup :groupInfo =
            Group.newMainGroup (includeGroups @ newPart) libGroups
  in
    if linkFiles = [] then ()
      else Util.wn ["Link files not yet implemented; ignored"];
    computeDependencies mainGroup;
    Group.groupId mainGroup
  end

fun dependsOn (group:group) (sourceName:string) :string list =
  let val (F{depends,...}) =
        findFileThruGroups (Group.findGroup group) (hasher sourceName)
  in map filerefName (!depends) end

fun connections (g:group) (out:outstream) =
  let val group = Group.findGroup g
      val pr = (outputc out)
      fun prs x = (pr x; pr " ")
      fun prq x = (pr "\""; pr x; pr "\"")
      fun printNameRefList (direction:string) (space,_) nameList =
        case nameList of
           [] => ()
         | (head::tail) =>
             (pr "\n  "; pr direction; prs space; map prs nameList; ()) 
      fun conn (filename,_)(F{toolH as (tool,_),imports,exports,...}:fileInfo)=
        (pr "source "; prs tool; prq filename;
         NameRefTable.scan (!imports) (printNameRefList "import ");
         NameRefTable.scan (!exports) (printNameRefList "export ");
         pr ";\n\n")
      fun do'group (group:groupInfo)(isLibrary)(acc:unit):unit =
        if isLibrary then () else Group.scanFiles group conn
  in
    Group.groupsFold group false do'group ()
  end

type currency =
  {sourceName:string, sourceTime:time, envUpdateTime:time,
   targetTime:time, sourceModified:bool, envCurrent:bool, targetCurrent:bool}

fun newAs' (a:currency, b:currency) = Util.newAs (#sourceTime a, #sourceTime b)

val indent'size :int ref = ref 0
fun inci () = inc indent'size
fun deci () = dec indent'size
fun ind x = if x = 0 then () else (print "  "; ind (x-1))

fun bs true = "T" | bs false = "F"
fun ts (y:time) = 
  let val x = Util.seconds y in 
    if x=0 then "000000000" else Integer.makestring x end
fun show (position:int) (sourceName:string)
         ({envUpdateTime,targetTime,sourceModified,
           envCurrent,sourceTime,targetCurrent,...}:currency) =
  if !debugMake
    then
     (Util.printSep " "
        ["&", Integer.makestring position,
         ts sourceTime, ts envUpdateTime, ts targetTime,
         bs sourceModified, bs envCurrent, bs targetCurrent];
      ind (!indent'size);
      Util.printSep " " ["", sourceName,"\n"])
    else ()

datatype 'a traversal = CIRCLE | OK of 'a
fun circle name = Util.err ["Circular dependency with file ", name]

fun sourceTraversal
      (group:groupInfo) (first :string list)
      (operate :string -> fileInfo -> bool) (accum:bool) :bool =
  let val trav = Hash.createDefault ([]:(bool traversal) list)
      fun do'dependee ((isLibrary,sourceNameH as (sourceName,_)),accum) :bool =
            let val dependee = isLibrary orelse
                      case Hash.lookup trav sourceNameH of
                         (SOME CIRCLE) => circle sourceName
                       | (SOME (OK success)) => success
                       | NONE => do'source sourceNameH
                                  (findFileThruGroups group sourceNameH)
            in dependee andalso accum end
      and do'source (sourceNameH as (sourceName,_))
                    (fileInfo as F {depends, toolH, env'current,
                                    target'current,...}:fileInfo) :bool =
          case Hash.lookup trav sourceNameH of
             (SOME CIRCLE) => circle sourceName
           | (SOME (OK success)) => success
           | NONE => 
              let val _ = Hash.enter trav sourceNameH CIRCLE
                  val dependees = fold do'dependee (!depends) true
                  val success = dependees andalso (operate sourceName fileInfo)
              in
                Hash.enter trav sourceNameH (OK success);
                success
              end
      fun try'source (nameH as (name,_)) fileInfo accum =
        (do'source nameH fileInfo) andalso accum
      fun do'group (grp:groupInfo) (isLibrary:bool) accum :bool =
        if isLibrary then accum else Group.foldFiles grp try'source accum
      fun do'file (filename:string, accum:bool) :bool =
        let val filenameH = hasher filename
            val fileInfo = findFileThruGroups group filenameH
        in (do'source filenameH fileInfo) andalso accum
        end
      val first'result = fold do'file (rev first) true
  in
    Group.groupsFold group false do'group first'result
  end

fun currencyCheck
      (group as G{groupId,...}:groupInfo) :bool * (string list) =
  let val trav = Hash.createDefault ([]:(currency traversal) list)
      val _ = (indent'size := 0)
      fun do'dependee ((isLibrary,sourceNameH as (sourceName,_)), accum) :currency =
            let val envUpdateTime'depender = (#envUpdateTime accum)
                val targetTime'depender = (#targetTime accum)
                val dependee =
                  case Hash.lookup trav sourceNameH of
                     (SOME CIRCLE) => circle sourceName
                   | (SOME (OK currency)) => currency 
                   | NONE => 
                       do'source isLibrary sourceNameH
                                 (findFileThruGroups group sourceNameH)
                val _ = (show 2 sourceName dependee)
                val result =
                  {envUpdateTime = envUpdateTime'depender,
                   targetTime = targetTime'depender,
                   sourceTime = (#sourceTime accum),
                   sourceName = (#sourceName accum),
                   sourceModified = (#sourceModified accum),
                   envCurrent = ((#envCurrent accum) andalso
                     (#envCurrent dependee) andalso
                     (Util.newAs(envUpdateTime'depender,#envUpdateTime dependee))),
                   targetCurrent = ((#targetCurrent accum) andalso
                     (#targetCurrent dependee) andalso
                     (Util.newAs(targetTime'depender,#targetTime dependee)))}
            in
              show 1 sourceName result;
              result
            end
      and do'source isLibrary (sourceNameH as (sourceName,_))
                    (fileInfo as F {depends, toolH, envUpdateTime,
                                    envObjectTime, env'current,
                                    target'current,...}:fileInfo) :currency =
          case Hash.lookup trav sourceNameH of
             (SOME CIRCLE) => circle sourceName
           | (SOME (OK currency)) => currency
           | NONE => 
              let val _ = Hash.enter trav sourceNameH CIRCLE
                  val ToolInternals.Tool {targetNameOf,validTarget,...} =
                        ToolInternals.getToolInfo toolH
                  val targetName = targetNameOf sourceName
                  val sourceTime = Util.modtime sourceName
                  val targetTime = Util.modtime targetName
                  val sourceUnchanged = 
                        isLibrary orelse (Util.newAs (!envObjectTime, sourceTime))
                  val targetOK = 
                        if isLibrary then true
                          else (Util.newAs (targetTime, sourceTime)) andalso
                               (validTarget (!groupId) sourceName targetName)
                  val start = {envUpdateTime=(!envUpdateTime),
                        targetTime=targetTime,
                        sourceTime=sourceTime, sourceName=sourceName,
                        sourceModified=not sourceUnchanged,
                        envCurrent=sourceUnchanged, targetCurrent=targetOK}
                  val _ = (show 3 sourceName start; inci())
                  val currency =
                        if isLibrary then start
                          else fold do'dependee (!depends) start
                  val _ = deci()
              in
                Hash.enter trav sourceNameH (OK currency);
                env'current := (#envCurrent currency);
                target'current := (#targetCurrent currency);
                currency
              end
      fun process'source (nameH as (name,_)) fileInfo accum =
        let val {envCurrent,...} = do'source false nameH fileInfo
        in accum andalso envCurrent end
      fun do'group (grp:groupInfo) (isLibrary:bool) accum =
        if isLibrary then accum else Group.foldFiles grp process'source accum
      fun appendModifiedSource (sourceName,_) currencyTraversal accum =
        case (currencyTraversal :currency traversal) of
           CIRCLE => accum
         | (OK (currency as {sourceModified,...})) =>
             if sourceModified then currency::accum else accum
      fun convert (currency as {sourceName,...}:currency) = sourceName
      val environmentCurrent = Group.groupsFold group false do'group true
      val modsources = Hash.fold trav appendModifiedSource []
      val modifiedSources = map convert (ListSort.sort newAs' modsources)
  in (environmentCurrent, modifiedSources) end

fun updateFileInfo
      (group:groupInfo)
      (filenameH as (filename,_))
      (info as F {infoTime,...}:fileInfo) acc =
  let val sourceTime = Util.modtime filename in
    if Util.isZeroTime sourceTime
      then Util.err ["source file ", filename, " not found"]
      else
        if Util.newer (sourceTime, !infoTime)
          then
            let val (imports, exports) = analyzeNameUsage true filename in
              (Group.defineSource (group, sourceTime, "")
                                  ("", filename, imports, exports)) :: acc
            end handle Skip => acc
          else acc
  end

fun groupInfoChanges
      (group as G{infoTime=groupInfoTime,...}:groupInfo)
      (filenameH as (filename,_))
      (info as F {infoTime,...}:fileInfo) changed =
  if Util.newer (!infoTime, !groupInfoTime)
    then (SOME info)::changed else changed

fun updateDefinitions (group:groupInfo) (isLibrary:bool) (acc:unit) :unit =
  if isLibrary then () else
    (case Group.connFile group of
        NONE =>
          (Group.foldFiles group (updateFileInfo group) [];
           Group.updateNamespaces group
             (Group.foldFiles group (groupInfoChanges group) []))
      | (SOME connFile) =>
          let val defTime = Util.modtime connFile
              val dir = Pathname.directoryPart connFile in
            if Util.newer (defTime, Group.infoTime group)
              then
                let val connections = Connections.get connFile
                    val _ = Group.clearFiles group
                    val infoList =
                          map (Group.defineSource (group, defTime, dir))
                              connections
                in
                  Group.createNamespaces false group infoList
                end
              else ()
          end)

fun processSource
      (fileAction :sourceInfo -> unit) (group as G {groupId,...} :groupInfo)
      (sourceName :string)
      (F {depends, env'current, target'current,
          toolH as (tool,_), infoTime,...}:fileInfo) :bool =
 (if tool = "" then true else
    let val ToolInternals.Tool
              {targetNameOf,loadSource,genTarget,loadTarget,checkLoad,
               compileSource,...} = ToolInternals.getToolInfo toolH
        val targetName = targetNameOf sourceName
        val importedFiles = map filerefName (!depends)
        fun doAction () =
          fileAction (Source
           {sourceName = sourceName, targetName = targetName, group = !groupId,
            envCurrent = (!env'current), targetCurrent = (!target'current),
            toolName = tool, dependsOn = importedFiles,
            loadSource = fn()=>loadSource (!groupId) sourceName targetName,
            genTarget = fn()=>genTarget (!groupId) sourceName targetName,
            loadTarget = fn()=>loadTarget (!groupId) sourceName targetName,
            compileSource=fn()=>compileSource (!groupId) sourceName targetName,
            checkLoad = fn()=>checkLoad (!groupId) sourceName targetName})
    in
      Util.handleInterrupt doAction;
      true
    end handle
           Interrupt => raise Interrupt
         | anyOtherException =>
             if (!continueAfterError)
               then false
               else (print "\n? errors were found during make.\n";
                     raise CompilingError))

fun checkSources (group as G{groupId,...}:groupInfo) :string list =
  let val (_, changes) = currencyCheck group in 
    if (!printModifiedSources)
      then (print "Modified sources:\n ";
            Util.printSep "\n  " (map Util.trim changes); print "\n")
      else ();
    changes
  end

fun makeGroup (libraryLevel:int) (fileAction :sourceInfo -> unit)
              (group as G{initialized,...} :groupInfo) :int =
  let val maxLibraryTime = 
            Group.groupsFold group false (doLibraries libraryLevel) 0
  in
    if (libraryLevel = 0) orelse
       ((!initialized <= maxLibraryTime) andalso (!makeLibraries))
      then
        let val _ = Group.groupsFold group false updateDefinitions ()
            val _ = computeDependencies group
            val changes = checkSources group
            val result = sourceTraversal group changes
                                         (processSource fileAction group) true
        in
          if result
            then let val (environCurrent,_) = currencyCheck group
                 in initialized :=
                      (if environCurrent
                            then Util.seconds(Util.currentTime ()) else 0)
                 end
            else 
              (print "\n? Errors were found during make.\n";
               initialized := 0;
               raise CompilingError);
          (!initialized)
        end
      else
        (!initialized)
  end

and doLibraries (libraryLevel:int) (group:groupInfo)
                (isLibrary:bool) (accum:int) :int =
  if isLibrary then accum
    else fold (makeLibrary libraryLevel) (rev(Group.libraryGroups group)) accum

and makeLibrary (libraryLevel:int)
      ((group as G{libraryAction,...}:groupInfo),accum:int) :int =
  let val thisLibraryTime = 
            makeGroup (libraryLevel+1) (Group.libraryAction group) group
  in
    if thisLibraryTime > accum then thisLibraryTime else accum
  end

fun make (fileAction :sourceInfo -> unit) (groupId:group) :unit =
  (makeGroup 0 fileAction (Group.findGroup groupId); ())

fun updateLibraryAction g = Group.updateLibraryAction (Group.findGroup g)
end
