structure Namespace :NAMESPACE = struct
open Array List
infix 9 sub

exception NamespaceInternalError

datatype space = StructureSpace | FunctorSpace | SignatureSpace
datatype scopeKind = LocalScope | LetScope | PlainScope
type spaceTable = (string,bool) Hash.table
datatype scope =
  Scope of {defTables:spaceTable array, parent:scope option, kind:scopeKind}

fun err (x:int) =
  (print "Namespace Error: ";print x;print "\n"; raise NamespaceInternalError)
fun ordOf (space:space) =
  case space of 
     StructureSpace => 0 | FunctorSpace => 1 | SignatureSpace => 2

fun emptySpaceTable () = Hash.createDefault ([]:bool list)
fun emptySpaces () :spaceTable array = 
  let val t = array (3, emptySpaceTable()) in
    update (t, 1, emptySpaceTable());
    update (t, 2, emptySpaceTable());
    t
  end

fun defTable (Scope {defTables,...}:scope) (space:space) =
  defTables sub (ordOf space)

val theImportTable = ref (emptySpaces())
fun importTable (space:space) = (!theImportTable) sub (ordOf space)

val theTopScope =
      ref (Scope {defTables = emptySpaces(), parent = NONE, kind=PlainScope})
fun topScope () = (!theTopScope)

val debug1 = ref false
fun newScope parentScope =
  (if (!debug1) then (print "newScope\n") else ();
   Scope {defTables=emptySpaces(), parent=(SOME parentScope), kind=PlainScope})
fun popScope (Scope {parent,...}) =
 (if (!debug1) then (print "popScope\n") else ();
  case parent of
     (SOME sc) => sc
   | NONE => err 0)
fun letScope parentScope = 
  (if (!debug1) then (print "letScope\n") else ();
   Scope {defTables=emptySpaces(), parent=(SOME parentScope), kind=LetScope})
fun localScope parentScope = 
  (if (!debug1) then (print "localScope\n") else ();
   Scope {defTables=emptySpaces(), parent=(SOME parentScope), kind=LocalScope})

fun lookup (scope as Scope{parent,...}:scope)(space:space)(nameH:string*int) =
  case Hash.lookup (defTable scope space) nameH of
     (SOME value) => (SOME (value, scope))
   | NONE =>
       (case parent of
           (SOME sc) => lookup sc space nameH
         | NONE => NONE)

fun enterImport (scope:scope) (space:space) (name:string) =
  let val nameH = Hasher.hasher name in
    case lookup scope space nameH of
       NONE => Hash.enter (importTable space) nameH true
     | (SOME (value, defining'scope)) => ()
  end

fun enterDefinition
      (scope as Scope{kind,parent,...}:scope) (space:space) (nameH:string*int) =
 (case kind of
     PlainScope => Hash.enter (defTable scope space) nameH true
   | LetScope =>   Hash.enter (defTable scope space) nameH true
   | LocalScope =>
       (case parent of
           NONE => err 1
         | (SOME (Scope{kind,parent,...})) =>
             (case kind of
                 PlainScope => err 2 | LocalScope => err 3
               | LetScope =>
                   (case parent of
                       NONE => err 4
                     | (SOME sc) => enterDefinition sc space nameH))))
                     
val debug0 = ref false
fun strRef (sc:scope) (name:string) =
  (enterImport sc StructureSpace name;
   if (!debug0) then (print "strRef "; print name; print "\n") else ();
   sc)
fun funRef (sc:scope) (name:string) =
  (enterImport sc FunctorSpace name;
   if (!debug0) then (print "funRef "; print name; print "\n") else ();
   sc)
fun sigRef (sc:scope) (name:string) =
  (enterImport sc SignatureSpace name;
   if (!debug0) then (print "sigRef "; print name; print "\n") else ();
   sc)
fun strDef (sc:scope) (name:string) =
  (enterDefinition sc StructureSpace (Hasher.hasher name);
   if (!debug0) then (print "strDef "; print name; print "\n") else ();
   sc)
fun funDef (sc:scope) (name:string) =
  (enterDefinition sc FunctorSpace (Hasher.hasher name);
   if (!debug0) then (print "funDef "; print name; print "\n") else ();
   sc)
fun sigDef (sc:scope) (name:string) =
  (enterDefinition sc SignatureSpace (Hasher.hasher name);
   if (!debug0) then (print "sigDef "; print name; print "\n") else ();
   sc)

exception Exists
fun anyOne (_) (_) :'a = raise Exists
fun isEmpty t :bool = (Hash.scan t anyOne; true) handle Exists => false

val pervStructures = 
 ["System","List","ByteArray","SourceGroup","Compile","GnuTags","Vector","IO",
  "String","Ref","Array","Real","Integer","General","Bits","Inside","Bool"]
val pervSignatures =
  ["REF","LIST","ARRAY","BYTEARRAY","IO","BOOL", "SOURCEGROUP","VECTOR",
   "STRING","INTEGER","REAL","GENERAL"]

fun addPervasives (sc:scope) =
  let val pervScope = letScope sc in
    map (strDef pervScope) pervStructures;
    map (sigDef pervScope) pervSignatures;
    localScope pervScope
  end

fun init (noPervasives:bool) = 
  let val emptyScope = 
            (Scope{defTables=emptySpaces(),parent=NONE,kind=PlainScope})
  in
    if noPervasives
      then theTopScope := emptyScope
      else theTopScope := addPervasives emptyScope;
   theImportTable := emptySpaces()
  end

fun attach (name:string,_) (_) accum = name::accum

fun nameList (space:space, imports:bool) =
  let val table = (if imports then importTable else defTable(topScope())) space
  in Hash.fold table attach [] end

fun connectionLists () =
  let val _ = (theTopScope := popScope(popScope (!theTopScope)))
      val sigImports = nameList (SignatureSpace, true)
      val strImports = nameList (StructureSpace, true)
      val funImports = nameList (FunctorSpace, true)
      val sigExports = nameList (SignatureSpace, false)
      val strExports = nameList (StructureSpace, false)
      val funExports = nameList (FunctorSpace, false)
  in ((sigImports,strImports,funImports),
      (sigExports,strExports,funExports))
  end

fun printAll (out:outstream) =
  let val pr = outputc out
      val width = ref 0
      fun namePrinter (name:string,key) (_:bool) =
        let val len = size name in
          if (!width) + len < 70 then ()
            else (pr "\n    "; width := 4);
          pr name; pr " "; width := (!width) + len + 1
        end
      fun dumpTable (label:string) (table:spaceTable) =
          if isEmpty table then ()
            else (pr label; width := 19; Hash.scan table namePrinter)
      val (Scope{parent,...}) = !theTopScope
  in
    case parent of
       NONE => ()
     | (SOME _) => (theTopScope := popScope(popScope (!theTopScope)));
    dumpTable "\n  import structure " (importTable StructureSpace);
    dumpTable "\n  import functor   " (importTable FunctorSpace);
    dumpTable "\n  import signature " (importTable SignatureSpace);
    dumpTable "\n  export structure " (defTable (topScope()) StructureSpace);
    dumpTable "\n  export functor   " (defTable (topScope()) FunctorSpace);
    dumpTable "\n  export signature " (defTable (topScope()) SignatureSpace);
    pr ";\n\n"
  end
end
