signature SEPCOMP = sig
 val makeObjectFile :string * string * staticEnv -> staticEnv
     (* makeObjectFile(src,target,senv) ==> senv'
      *  compiles the source file src to produce a compUnit that
      *  is written to file target.  The result senv' is the incremental
      *  static environment produced by compilation *)

 val compileFile :string * string * environment -> environment
     (* compileFile(src,target,env) ==> env'
      *  similar to makeObjectFile but also executes compiled compUnit
      *  and produces full incremental environment env' *)

 val loadSourceFile :string * environment -> environment
     (* loadSourceFile(src,env) ==> env'
      *  compiles and executes source file src to produce incremental
      *  full environment env', but does not write out compUnit *)

 val loadObjectFile :string * environment -> environment
     (* loadObjectFile(target,env) ==> env'
      *  reads compUnit from file target and executes it to produce
      *  incremental environment env' *)

 val validObjectFile :string -> bool
     (* checks to see if its argument designates a valid target file
      *  for the current system version and architecture. *)

 val openEnv :environment -> unit
     (* adds the environment to the toplevel environment. *)
end


structure SepComp :SEPCOMP = struct

open System.Env System.Compile

fun withInStream (stream :instream)
      (action :instream -> 'a -> 'b) (argument:'a) :'b =
  let val result = action stream argument
                     handle exn => (close_in stream; raise exn)
  in close_in stream; result end

fun withSource (sourceName:string)
      (action :source -> 'a -> 'b) (argument:'a) :'b =
  let val sourceStream = open_in sourceName
      val source = makeSource (sourceName, 1, sourceStream, false, std_out)
      val result = action source argument
                     handle exn => (closeSource source; raise exn)
  in closeSource source; result end

fun withOutStream (stream :outstream)
      (action :outstream -> 'a -> 'b) (argument:'a) :'b =
  let val result = action stream argument
                     handle exn => (close_out stream; raise exn)
  in close_out stream; result end

exception Interrupt

fun handleInterrupt (operation : unit -> unit) =
  let exception Done
      val old'handler = System.Signals.inqHandler(System.Signals.SIGINT)
      fun reset'handler () =
        System.Signals.setHandler(System.Signals.SIGINT, old'handler)
  in ( callcc (fn k =>
         ( System.Signals.setHandler(System.Signals.SIGINT,SOME(fn _ => k)) ;
           operation ();
           raise Done )) ;
        (* print ("\n--- interrupt ---\n"); *)
        raise Interrupt )
      handle Done => (reset'handler ())
           | exn  => (reset'handler (); raise exn)
  end

fun reading file = print ("[reading " ^ file ^ "]\n")
fun writing file = print ("[writing " ^ file ^ "]\n")

fun fail s = raise (Compile s)

(* Version stamp included in object files. Specifies compiler
 * version and machine type (= !System.architecture). Perhaps should
 * include OS version as well.
 *)
val targetVersion = System.version ^ !System.architecture^ "\n"

val targetRead :instream * int -> compUnit = System.Unsafe.blast_read
val targetWrite' :(outstream * compUnit) -> int = System.Unsafe.blast_write

fun targetWrite (stream, obj) =    (* Silent version. *)
  let val gcmessages = System.Control.Runtime.gcmessages 
      val oldmsgs = !gcmessages
  in gcmessages := 0;
     (targetWrite'(stream, obj); gcmessages := oldmsgs)
       handle e => (gcmessages := oldmsgs; raise e)
  end

fun readCompUnit (targetName:string) :compUnit =
  let fun reader target () =
        if (input_line target) <> targetVersion
          then
            (print ("? target file " ^ targetName ^
                    " is the wrong format; quitting\n");
             fail "targetVersion")
          else targetRead (target,can_input target)
  in
    withInStream (open_in targetName) reader ()
  end

fun deleteFile (filename:string) =
  (System.Unsafe.SysIO.unlink filename) handle _ => ()

fun ignoreWriteError targetName =
  (deleteFile targetName;  (* remove half-baked target *)
   print "% writing target file failed, ignored; compilation continued.\n")

fun writeCompUnit (compiledUnit:compUnit, targetName:string) =
  let fun writer target () =
        (outputc target targetVersion;
         targetWrite (target,compiledUnit))
  in
    handleInterrupt (withOutStream (open_out targetName) writer)
  end
    handle
       Io msg =>
        (print ("\n" ^ msg ^ "\n");
         ignoreWriteError targetName)
     | Interrupt =>
        (print "\n% Interrupt encountered.\n";
         ignoreWriteError targetName)
     | any =>
        (print ("% Exception " ^ (System.exn_name any) ^ " raised.\n");
         ignoreWriteError targetName)

fun compileSource (sourceName:string, senv:staticEnv) :compUnit =
  let fun comp source () = compile (source, senv) in
    withSource sourceName comp ()
  end

fun compileFile (sourceName:string, targetName:string, env:environment) =
  let val _ = reading sourceName
      val compUnit = compileSource(sourceName, staticPart env)
  in
    writing targetName;
    writeCompUnit(compUnit,targetName);
    execute(compUnit, env)
  end

fun makeObjectFile (sourceName:string, targetName:string, senv:staticEnv) =
  let val _ = reading sourceName
      val compUnit as ({staticEnv,...},_) = compileSource(sourceName,senv)
  in
    writing targetName;
    writeCompUnit(compUnit,targetName);
    staticEnv
  end

fun loadSourceFile (sourceName:string, env:environment) :environment =
  (reading sourceName;
   execute(compileSource(sourceName,staticPart env),env))

fun loadObjectFile (targetName:string, env:environment) :environment =
  let val _ = reading targetName
      val (static,code) = readCompUnit targetName in
    execute((changeLvars static,code),env)
  end

fun validObjectFile (filename:string) :bool =
  let fun checker stream () = ((input_line stream) = targetVersion)
  in
    withInStream (open_in filename) checker ()
  end handle _ => false

fun openEnv (env:environment) :unit =
  topLevelEnvRef := concatEnv(env, !topLevelEnvRef)

end
