#include <mips/cachectl.h>
#include "spim.h"
#include "inst.h"
#include "mem.h"

#define DCACHESZ  1024   /* 2 ^ DCACHEBT */
#define DCACHEBT    12
#define DLINESZ      1
#define ICACHESZ  1024   /* 2 ^ ICACHEBT */
#define ICACHEBT    12
#define ILINESZ      1
#define CFILL       16   /* Fill CFILL words on cache miss */

typedef struct cent {  /* cache entry */
    unsigned tag;
    unsigned valid:1;
    unsigned dirty:1;
} Cent;
static Cent icache[ILINESZ][ICACHESZ], dcache[DLINESZ][DCACHESZ];
static unsigned tihit = 0, timiss = 0;
static unsigned tdhit = 0, tdmiss = 0, tdinvals = 0, tdirty = 0;
static unsigned tread = 0, twrite = 0;

typedef struct iindextag {  /* index & tag */
    unsigned junk:2;
    unsigned index:ICACHEBT-2;
    unsigned tag:32-ICACHEBT;
} IIT;
typedef struct dindextag {  /* index & tag */
    unsigned junk:2;
    unsigned index:DCACHEBT-2;
    unsigned tag:32-DCACHEBT;
} DIT;

void initcache(void) {
    int line, index;
    tihit = timiss = tdhit = tdmiss = tdinvals = tdirty = tread = twrite = 0;
    for(line = 0; line < ILINESZ; ++line)
	for(index = 0; index < ICACHESZ; ++index) {
	    icache[line][index].valid = 0;
	    icache[line][index].dirty = 0;
	}
    for(line = 0; line < DLINESZ; ++line)
	for(index = 0; index < DCACHESZ; ++index) {
	    dcache[line][index].valid = 0;
	    dcache[line][index].dirty = 0;
	}
}

void doiaccess(unsigned addr, int rw) {
    IIT *a;
    int i, j, line, flag = 0;

    if(rw == C_READ) tread++;
    else             twrite++;
    a = (IIT *) &addr;
    /*     printf("ICACHE: %x %x\n", a->index, a->tag); */
    for(line = 0; line < ILINESZ; ++line)
	if((icache[line][a->index].tag == a->tag) &&
	   (icache[line][a->index].valid == 1)) {
	    tihit++;
	    flag = 1; break;
	}
    if(flag == 0) { /* miss */
	timiss++;
	for(i = 0; i < CFILL * 4; i += 4) {
	    j = addr + i;
	    a = (IIT *) &j;
	    for(line = 0; ((icache[line][a->index].valid == 1) && 
			   (line < (ILINESZ-1))); ++line)
		;
	    icache[line][a->index].tag = a->tag;
	    icache[line][a->index].valid = 1;
	}
    }
}

void dodaccess(unsigned addr, int rw, int size) {
    DIT *a;
    int i,j, line, flag = 0;

    a = (DIT *) &addr;
    /* printf("DCACHE: %x %x\n", a->index, a->tag); */
    if(rw == C_READ) tread++;
    else             twrite++;
    for(line = 0; line < DLINESZ; ++line)
	if((dcache[line][a->index].tag == a->tag) &&
	   (dcache[line][a->index].valid == 1)) {
	    if((rw == C_WRITE) && (size != C_WORD)) {
		/*
		 * the cache behaves as write-through when a byte or
		 *  half-word store is done, unconditionally invalidating
		 * that cache word.
		 */
		dcache[line][a->index].valid = 0;
		tdinvals++;
	    } else {
		tdhit++;
		if(rw == C_WRITE)
		    dcache[line][a->index].dirty = 1;
		/* printf("valid %x\n", addr); */
	    }
	    flag = 1; break;
	}
    if(flag == 0) {
	tdmiss++;
	for(i = 0; i < CFILL * 4; i += 4) {
	    j = addr + i;
	    a = (DIT *) &j;
	    /*
	     * first find an empty slot in the cache if possible
	     */
	    for(line = 0;((dcache[line][a->index].valid == 1) &&
			  (line < (DLINESZ - 1))); ++line)
		;
	    /*
	     * the replaced items may have had to be written back to memory
	     * keep track of how many dirty items we replace
	     */
	    if((dcache[line][a->index].valid == 1) && 
	       (dcache[line][a->index].dirty == 1))
		tdirty++;
	    dcache[line][a->index].tag = a->tag;
	    dcache[line][a->index].valid = 1;
	}
    }
}

int cache_flush(char *addr, int nbytes, int cache) {
    int flag = -1;
    if(cache & ICACHE) {
	flag = 0;
    }
    if(cache & DCACHE) {
	flag = 0;
    }
    return flag;
}

int cache_ctl(char *addr, int nbytes, int op) {
    int flag = -1;
    if(op & CACHEABLE) {
	flag = 0;
    }
    if(op & UNCACHEABLE) {
	flag = 0;
    }
    return flag;
}


void print_cstats(void) {
    printf("Cache Statistics\n");
    printf("Reads: %d   Writes: %d\n",tread,twrite);
    printf("ICACHE: hits:%d misses:%d\n",tihit,timiss);
    printf("DCACHE: hits:%d misses:%d invalidates:%d\n",tdhit,tdmiss,tdinvals);
    printf("DCACHE: dirty items replaced:%d\n",tdirty);
}

#ifdef DEBUG
main() {
    dodaccess(0xffff0002,C_WRITE,C_WORD);
    dodaccess(0xffff0004,C_WRITE,C_WORD);
    dodaccess(0xffff8002,C_WRITE,C_WORD);
    dodaccess(0xffff8008,C_WRITE,C_WORD);
    dodaccess(0xffff1002,C_WRITE,C_WORD);
    print_cstats();
}
#endif
