#include <stdio.h>
#include <syscall.h>
#include <sys/types.h>
#include "/usr/include/signal.h"
#include "mips-syscall.h"
#define ERR_REG 7

#include "spim.h"
#include "inst.h"
#include "mem.h"
#include "reg.h"
#include "spim-syscall.h"

/* Imported variables: */
extern int errno;
extern int program_break;
extern int systrace; /* from spim.c */

int prog_sigmask = 0;
mem_addr exception_address = 0;
struct sigvec sighandler[NSIG];
extern int syscall(int, ...);

/* does the unix system call: returns negative on error */
static int unixsyscall(void) {
    int arg0, arg1, arg2, arg3;
		  
    SYSTRACE(REG_V0, REG_A0, REG_A1, REG_A2, REG_A3);

    arg0 = SYSCALL_ARG(REG_V0,arg0, REG_A0);
    arg1 = SYSCALL_ARG(REG_V0,arg1, REG_A1);
    arg2 = SYSCALL_ARG(REG_V0,arg2, REG_A2);
    arg3 = SYSCALL_ARG(REG_V0,arg3, REG_A3);
    R[REG_RES] = syscall(R[REG_V0], arg0, arg1, arg2, arg3);
    if(systrace)      fprintf(stderr, " %d\n", R[REG_RES]);
    /* We have to see whether an error has occurred during
       the system call. If so, the libc wrap around must be
       notifified by setting register 7 to be less than zero
       and the return value should be set to errno. If not
       register 7 should be zero. r7 acts like the carry flag
       in the old days.
       */
    if(R[REG_RES] < 0) {
	R[ERR_REG] = -1;
	R[REG_RES] = errno;
	return -1;
    } else {
	R[ERR_REG] = 0;
	return R[REG_RES];
    }
}

static int reverse_fds(int fd) {
    int i;
    for(i=0; i < MAXFDS; ++i)
	if(prog_fds[i] == fd)
	    return i;
    run_error("Couldn't reverse translate fds\n");
    return -1;
}

/*
 * decides which syscall it should do or simulate
 * returns zero to signal exit, non-zero for continue
 */
int dosyscall(void) {
#ifdef mips
    if (!source_file) {
	/* Use actual MIPS system calls after translating arguments
	   from simulated memory to actual memory and to different
	   file descriptors */
	if (R[REG_V0] > MAX_SYSCALL) {
	    run_error("Illegal system call: %d\n", R[REG_V0]);
	    return 0;
	}
	switch(syscall_table[R[REG_V0]].syscall_type) {
	case BAD_SYSCALL:
	    run_error ("Unknown system call: %d\n", R[REG_V0]);
	    break;
	case UNIX_SYSCALL:
	    unixsyscall();
	    break;
	case SPC_SYSCALL:
	    /* These special syscalls need to be simulated by SPIM */
	    switch(R[REG_V0]) {
	    case SYS_syscall:
		if(systrace) fprintf(stderr, "SYSCALL syscall - ");
		R[REG_V0] = R[REG_A0];
		R[REG_A0] = R[REG_A1];
		R[REG_A1] = R[REG_A2];
		R[REG_A2] = R[REG_A3];
		READ_MEM_WORD_NORECORD(R[REG_A3], R[REG_SP]+16);
		dosyscall();
		break;
	    case SYS_sysmips: {
		/* 
		 * the table smipst goes from the sysmips arguments to syscall
		 * numbers
		 */
		int callno, smipst[] = { 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
					    7, 7, SYS_getrusage, SYS_wait3, 
					     SYS_cacheflush, SYS_cachectl};
		if(systrace) fprintf(stderr, "SYSCALL sysmips - ");
		callno= R[REG_A0];
		callno=((callno >0x100) ? smipst[callno - 0x100 + 10] : smipst[callno]);
		R[REG_V0] = callno;
		R[REG_A0] = R[REG_A1];
		R[REG_A1] = R[REG_A2];
		R[REG_A2] = R[REG_A3];
		READ_MEM_WORD_NORECORD(R[REG_A3], R[REG_SP]+16);
		dosyscall();
		break;
	    }
	    case SYS_exit: {
		/* close all file descriptors */
		int i;
		for(i=0; i < MAXFDS; ++i)
		    close(prog_fds[i]);
		return 0;
	    }
	    case SYS_close: {
		/* mark the file descriptor closed */
		int ret = unixsyscall();
		if(ret >= 0)
		    prog_fds[R[REG_A0]] = -1;
		break;
	    }		      
	    case SYS_open: case SYS_creat: case SYS_dup: {
		/* update the fd translation table */
		int ret = unixsyscall();
		if(ret >= 0)
		    prog_fds[ret] = ret;
		break;
	    }
	    case SYS_pipe: {
		/* This isn't too useful unless we implement fork() or other
		   fd passing mechanisms */
		int fd1, fd2;
		if(unixsyscall() >= 0) {
		    READ_MEM_WORD_NORECORD(fd1, MEM_ADDRESS(R[REG_A0]));
		    READ_MEM_WORD_NORECORD(fd2, MEM_ADDRESS(R[REG_A0]));
		    prog_fds[fd1] = fd1;
		    prog_fds[fd2] = fd2;
		}
		break;
	    }
	    case SYS_select: {
		int fd;
		/*
		 * we have to use this kludge to circumvent typechecking
		 * because the memory read macros take the lefthand side
		 * as an argument instead of simply returnign the value
		 * at the address
		 */
		long int kludge;
		fd_set a, b, c;
		fd_set *readfds=&a, *writefds=&b, *exceptfds=&c;
		struct timeval *timeout;

		FD_ZERO(readfds);
		FD_ZERO(writefds);
		FD_ZERO(exceptfds);
		READ_MEM_WORD_NORECORD(kludge, R[REG_SP]+16);
		if(kludge == NULL)
		    timeout = NULL;
		else
		    timeout = (struct timeval *) MEM_ADDRESS(kludge);
		if(R[REG_A1] == NULL)
		    readfds = NULL;
		else 
		    for(fd=0; fd < R[REG_A0]; ++fd)
			if(FD_ISSET(fd, (fd_set *) MEM_ADDRESS(R[REG_A1])))
			    FD_SET(prog_fds[fd], readfds);
		if(R[REG_A2] == NULL)
		    writefds = NULL;
		else 
		    for(fd=0; fd < R[REG_A0]; ++fd)
			if(FD_ISSET(fd, (fd_set *) MEM_ADDRESS(R[REG_A2])))
			    FD_SET(prog_fds[fd], writefds);
		if(R[REG_A3] == NULL)
		    exceptfds = NULL;
		else 
		    for(fd=0; fd < R[REG_A0]; ++fd)
			if(FD_ISSET(fd, (fd_set *) MEM_ADDRESS(R[REG_A3])))
			    FD_SET(prog_fds[fd], exceptfds);
		if(systrace)
		    fprintf(stderr, "SYSCALL select(%d, 0x%x, 0x%x, 0x%x, 0x%x) = ", R[REG_A0], R[REG_A1], R[REG_A2], R[REG_A3], timeout);
		R[REG_RES] = select(R[REG_A0], readfds, writefds, exceptfds, timeout);
		if(readfds == NULL)
		    R[REG_A1] = NULL;
		else 
		    for(fd=0; fd < R[REG_A0]; ++fd)
			if(FD_ISSET(fd, readfds))
			    FD_SET(reverse_fds(fd), 
				   (fd_set *) MEM_ADDRESS(R[REG_A1]));
		if(writefds == NULL)
		    R[REG_A2] = NULL;
		else 
		    for(fd=0; fd < R[REG_A0]; ++fd)
			if(FD_ISSET(fd, writefds))
			    FD_SET(reverse_fds(fd),
				   (fd_set *) MEM_ADDRESS(R[REG_A2]));
		if(exceptfds == NULL)
		    R[REG_A3] = NULL;
		else 
		    for(fd=0; fd < R[REG_A0]; ++fd)
			if(FD_ISSET(fd, exceptfds))
			    FD_SET(reverse_fds(fd),
				   (fd_set *) MEM_ADDRESS(R[REG_A3]));
		if(systrace)
		    fprintf(stderr, "0x%x\n", R[REG_RES]);
		if(R[REG_RES] < 0) {
		    R[ERR_REG] = -1;
		    R[REG_RES] = errno;
		    return -1;
		} else {
		    R[ERR_REG] = 0;
		    return R[REG_RES];
		}
	    }
	    case SYS_sbrk: {
		if(systrace)
		    fprintf(stderr, "SYSCALL sbrk(%d) = ", R[REG_A0]);
		expand_data(R[REG_A0]);
		R[REG_RES] = program_break;
		program_break += R[REG_A0];
		R[ERR_REG] = 0;
		if(systrace)
		    fprintf(stderr, "0x%x\n", R[REG_RES]);
		break;
	    }
	    case SYS_brk:
		if(systrace)
		    fprintf(stderr, "SYSCALL brk(0x%x) = ", R[REG_A0]);
		/* Round up to 4096 byte (page) boundary */
		if (((int) R[REG_A0] - (int) data_top) > 0)
		    expand_data (ROUND(R[REG_A0], 4096)-(int)data_top);
		R[REG_RES] = program_break;
		program_break = ROUND(R[REG_A0], 4096);
		R[ERR_REG] = 0;
		if(systrace)
		    fprintf(stderr, "0x%x\n", R[REG_RES]);
		break;
	    case SYS_sigvec: { 
		int x;
		
		if(systrace)
		    fprintf(stderr, "SYSCALL sigvec(%d, 0x%x, 0x%x) = ", R[REG_A0], R[REG_A1], R[REG_A2]);
		if(R[REG_A2] != 0)
		    *(struct sigvec *)R[REG_A2]=sighandler[R[REG_A0]];
		READ_MEM_WORD(x, R[REG_A1]);
		sighandler[R[REG_A0]].sv_handler = (void(*)()) x;
		READ_MEM_WORD(x,R[REG_A1] + sizeof (int *));
		sighandler[R[REG_A0]].sv_mask = x;
		READ_MEM_WORD(x,R[REG_A1] + sizeof(int *) + sizeof(sigset_t));
		sighandler[R[REG_A0]].sv_flags = x;
		exception_address = R[31] + 0x2a4; /* Non-Portable */
		if(systrace)
		    fprintf(stderr, "%d\n", R[REG_RES]);
		R[ERR_REG] = 0;
		break;
	    }
	    case SYS_sigreturn:
		if(systrace)	    fprintf(stderr, "SYSCALL sigreturn() = ");
		dosigreturn();
		if(systrace)	    fprintf(stderr, "%d\n", R[REG_RES]);
		R[ERR_REG] = 0;
		break;
	    case SYS_sigsetmask:
		if(systrace)	   
		    fprintf(stderr, "SYSCALL sigsetmask(0x%x) = ", R[REG_A0]);
		R[REG_RES] = prog_sigmask;
		prog_sigmask = R[REG_A0];
		if(systrace)	    fprintf(stderr, "%d\n", R[REG_RES]);
		R[ERR_REG] = 0;
		break;
	    case SYS_sigblock:
		if(systrace)	   
		    fprintf(stderr, "SYSCALL sigsetmask(0x%x) = ", R[REG_A0]);
		R[REG_RES] = prog_sigmask;
		prog_sigmask |= R[REG_A0];
		if(systrace)	    fprintf(stderr, "%d\n", R[REG_RES]);
		R[ERR_REG] = 0;
		break;
	    case SYS_cacheflush:
		if(systrace)	   
		    fprintf(stderr, "SYSCALL cacheflush(0x%x, 0x%x, 0x%x) = ",
			    R[REG_A0],R[REG_A1],R[REG_A2]);
		R[REG_RES] = 
		    cache_flush((void*)MEM_ADDRESS(R[REG_A0]),R[REG_A1],R[REG_A2]);
		R[ERR_REG] = 0;
		if(systrace)	    fprintf(stderr, "%d\n", R[REG_RES]);
		break;
	    case SYS_cachectl:
		if(systrace)	   
		    fprintf(stderr, "SYSCALL cachectl(0x%x, 0x%x, 0x%x) = ", 
			    R[REG_A0],R[REG_A1],R[REG_A2]);
		R[REG_RES] = 
		    cache_ctl((void*)MEM_ADDRESS(R[REG_A0]),R[REG_A1],R[REG_A2]);
		R[ERR_REG] = 0;
		if(systrace)	    fprintf(stderr, "%d\n", R[REG_RES]);
		break;
	    default:
		run_error("Unknown special system call: %d\n",R[REG_V0]);
		break;
	    }
	    break;

	default:
	    run_error("Unknown type for syscall: %d\n", R[REG_V0]);
	    break;
	}
    }
    else
#endif
    {
	/* Dummy system calls for SPIMs assembly option */
/*	switch (R[REG_V0]) {
	case PRINT_INT_SYSCALL:
	    write_output (console_out, "%d", R[REG_A0]);
	    break;
	case PRINT_FLOAT_SYSCALL: {
	    float val = FPR_S (REG_FA0);
	    write_output (console_out, "%f", val);
	    break;
	}
	case PRINT_DOUBLE_SYSCALL:
	    write_output (console_out, "%f", FPR[REG_FA0/2]);
	    break;
	case PRINT_STRING_SYSCALL:
	    write_output (console_out, "%s", MEM_ADDRESS (R[REG_A0]));
	    break;
	case READ_INT_SYSCALL: {
	    static char str [256];

	    read_input (str, 256);
	    R[REG_RES] = atol (str);
	    break;
	}
	case READ_FLOAT_SYSCALL: {
	    static char str [256];
	    
	    read_input (str, 256);
	    FGR [REG_FRES] = (float) atof (str);
	    break;
	}
	case READ_DOUBLE_SYSCALL: {
	    static char str [256];

	    read_input (str, 256);
	    FPR [REG_FRES] = atof (str);
	    break;
	}
	case READ_STRING_SYSCALL: {
	    read_input((char *)MEM_ADDRESS (R[REG_A0]), R[REG_A1]);
	    break;
	}
	case SBRK_SYSCALL: {
	    mem_addr x = data_top;
	    expand_data (R[REG_A0]);
	    R[REG_RES] = x;
	    break;
	}
	case EXIT_SYSCALL:
	    return 0;
	default:
	    run_error ("Unknown system call: %d\n", R[REG_V0]);
	    break;
	}
*/
    }
    return 1;
}

