#include <stdio.h>
#include <ctype.h>
#include <setjmp.h>
#include <signal.h>

#include "spim.h"
#include "inst.h"
#include "mem.h"
#include "reg.h"
#include "sym_tbl.h"
#include "y.tab.h"

#define ARGMAX 256

/* Internal functions: */
static void top_level(void);
static void control_c_seen(int);
static int parse_spim_command(FILE *, int);
static int read_assembly_command(FILE *);
static int str_prefix(char *, char *, int);
static long int get_opt_int(void);
static void flush_to_newline(void);
static int print_reg(int, int);

/* Imported variables: */
extern int yylval;			  /* Value of token from YYLEX */
extern char *input_file_name;             /* File for yyparse() to read from */

static int load_trap_handler = 0;
static char *argvec[ARGMAX];
static int argcount = 0;

/* Exported variables */
jmp_buf spim_top_level_env; /* For ^C */
char **environ;
int systrace = 0;               /* Non-zero => report system calls */
int bare_machine = 0;		/* Non-zero => ignore assembler
				   embellishments to bare hardware */
int quiet = 0;			/* Non-zero => no message on traps. */

void main(int argc, char **argv, char **env) {
    int i, j = 0;
    char *file_name = NULL;

    environ = env;
    for (i = 1; i < argc; i++)
	if(streq(argv [i], "-bare"))
	    bare_machine = 1, quiet = 1;
	else if(streq(argv [i], "-asm"))
	    bare_machine = 0;
	else if(streq(argv [i], "-trap"))
	    load_trap_handler = 1;
	else if(streq(argv [i], "-notrap"))
	    load_trap_handler = 0;
	else if(streq(argv [i], "-quiet"))
	    quiet = 1;
	else if(streq(argv [i], "-noquiet"))
	    quiet = 0;
	else if(streq(argv [i], "-systrace"))
	    systrace = 1;
    /* 
     * This stuff about "initial limits" MUST disappear
	else if (streq(argv [i], "-stext"))
	    initial_text_size = atoi (argv[++i]);
	else if (streq(argv [i], "-sdata"))
	    initial_data_size = atoi (argv[++i]);
	else if (streq(argv [i], "-ldata"))
	    initial_data_limit = atoi (argv[++i]);
	else if (streq(argv [i], "-sstack"))
	    initial_stack_size = atoi (argv[++i]);
	else if (streq(argv [i], "-lstack"))
	    initial_stack_limit = atoi (argv[++i]);
	else if (streq(argv [i], "-sktext"))
	    initial_k_text_size = atoi (argv[++i]);
	else if (streq(argv [i], "-skdata"))
	    initial_k_data_size = atoi (argv[++i]);
	else if (streq(argv [i], "-lkdata"))
	    initial_k_data_limit = atoi (argv[++i]);
    */

	else {
	    if(file_name == NULL)
		file_name = argv[i];
	    if(j > ARGMAX) {
		run_error("Too many arguments or argument too long\n");
	    }
	    if(argv[i] != NULL) {
		argvec[j] = (char *)malloc(strlen((char *)argv[i]) + 1);
		strcpy(argvec[j++], (char *)argv[i]);
	    }
	}
    argvec[j] = NULL;
    argcount  = j;

    initialize_world (load_trap_handler);
    if((file_name != NULL) && (read_assembly_file(file_name) == 0))
	if (setjmp(spim_top_level_env) == 0)
	    init_and_run(0, DEFAULT_RUN_STEPS, 0,0, argcount, argvec, environ);
	else
	    top_level();
    else
	top_level();
    print_cstats();
    exit(0);
}

/* Top-level read-eval-print loop for SPIM. */
static void top_level(void)
{
    int redo = 0;		/* Non-zero means reexecute last command */

    signal (SIGINT, control_c_seen);
    while (1) {
	if (redo == 0)
	    write_output(message_out, "(spim) ");
	if (setjmp(spim_top_level_env) == 0)
	    redo = parse_spim_command(stdin, redo);
	else
	    redo = 0;
	fflush(stdout);
	fflush(stderr);
    }
}

static void control_c_seen(int dummy)
{
    write_output(message_out, "Execution interrupted\n");
    longjmp(spim_top_level_env, 1);
}

void getargs(void) {
    int i = 0, token;
    char argv0[] = "spim-debug";
    char *str, buf[20]; /* buffer helps in converting integers to strs */
    
    argvec[i] = (char *) malloc(strlen(argv0) + 1);
    strcpy(argvec[i++], argv0);
    /*
     * This is a kludge. The lexer should NOT read user input.
     * We have to work around the lexer returning Y_INTs now.
     * Also, registers, fpregisters etc. still cannot be arguments.
     */
    while(((token = yylex()) == Y_ID) || (token == Y_INT)) {
	if(token == Y_ID)
	    str = (char *) yylval;
	else {
	    sprintf(buf,"%d",(int) yylval);
	    str = buf;
	}
	if(i > ARGMAX) {
	    run_error("Too many arguments or argument too long\n");
	}
	argvec[i] = (char *)malloc(strlen(str) + 1);
	strcpy(argvec[i++], str);
	}
    argvec[i] = NULL;
    argcount  = i;
}

/* SPIM commands */
enum {UNKNOWN_CMD, EXIT_CMD, READ_CMD, RUN_CMD, STEP_CMD, PRINT_CMD, 
PRINT_SYM_CMD, REINITIALIZE_CMD, ASM_CMD, REDO_CMD, NOP_CMD, HELP_CMD,
CONTINUE_CMD, SET_BKPT_CMD, DELETE_BKPT_CMD, LIST_BKPT_CMD, EXECUTE_CMD,
REGS_CMD, ARGS_CMD};

/* Parse a SPIM command from the FILE and execute it.  If REDO is non-zero,
   don't read a new command; just rexecute the previous one.
   Return non-zero if the command was to redo the previous command. */
static int parse_spim_command (FILE *file, int redo) {
    static int prev_cmd = NOP_CMD; /* Default redo */
    static int prev_token;
    int cmd;
    
    initialize_scanner (file);
    switch (cmd = (redo ? prev_cmd : read_assembly_command (file))) {
    case EXIT_CMD:
	exit (0);
    case READ_CMD:
    {
	int token = (redo ? prev_token : yylex ());

	if (!redo) flush_to_newline ();
	if (token == Y_STR) {
	    read_assembly_file((char *) yylval);
	    initialize_scanner(file); /* Reinitialize! */
	}
	else
	    error("Must supply a filename to read\n");
	prev_cmd = READ_CMD;
	return(0);
    }
    case RUN_CMD:
	getargs();
	prev_cmd = RUN_CMD;
	if(init_and_run(0,DEFAULT_RUN_STEPS,0,0, argcount, argvec, environ))
	    write_output (message_out,"Breakpoint encountered at 0x%08x\n",PC);
	else 
	    write_output(message_out, "Program finished.\n");
	return (0);
    case CONTINUE_CMD:
	prev_cmd = CONTINUE_CMD;
	if(init_and_run(PC, DEFAULT_RUN_STEPS, 0, 1, 
			argcount, argvec, environ) != 0)
	    write_output(message_out,"Breakpoint encountered at 0x%08x\n", PC);
	flush_to_newline();
	return (0);
    case STEP_CMD: {
	static int steps = 0;
	mem_addr addr;
	prev_cmd = STEP_CMD;

	if(!redo) {
	    if((steps = get_opt_int()) != 0)
		getargs();
	}
	addr = starting_address();
	if(steps <= 0)	    steps = 1;
	init_and_run(addr, steps, 1, 1, argcount, argvec, environ);
	return (0);
    }
    case REGS_CMD: { 
	int i;
	for(i=0; i < 32; ++i)
	    print_reg(i, 0);
	flush_to_newline();
	return(0);
    }
    case ARGS_CMD: { 
	if(redo) return(0);
	getargs();
	prev_cmd = ARGS_CMD;
	return(0);
    }
    case PRINT_CMD: {
	int token = (redo ? prev_token : yylex ());
	static int loc;
	
	if (token == Y_REG) {
	    if (redo) loc += 1;
	    else loc = yylval;
	    print_reg (loc, 0);
	}
	else if (token == Y_FP_REG) {
	    if (redo) loc += 2;
	    else loc = yylval;
	    print_reg (loc, 1);
	}
	else if (token == Y_INT) {
	    if (redo) loc += 4;
	    else loc = yylval;
	    print_mem (loc);
	}
	else if (token == Y_ID) {
	    if (!print_reg (yylval, 2)) {
		if (redo) loc += 4;
		else loc = find_symbol_address ((char *) yylval);
		if (loc != 0)
		    print_mem (loc);
		else
		    error ("Unknown label: %s\n", yylval);
	    }
	}
	else
	    error ("Print what?\n");
	if (!redo) flush_to_newline ();
	prev_cmd = PRINT_CMD;
	prev_token = token;
	return (0);
    }
    case PRINT_SYM_CMD:
	print_symbols ();
	if (!redo) flush_to_newline ();
	prev_cmd = NOP_CMD;
	return (0);
    case REINITIALIZE_CMD:
	initialize_world(load_trap_handler);
	prev_cmd = NOP_CMD;
	return (0);
    case ASM_CMD:
	input_file_name = "<standard input>";
	yyparse();
	prev_cmd = ASM_CMD;
	return (0);
    case REDO_CMD:
	return (1);
    case NOP_CMD:
	prev_cmd = NOP_CMD;
	return (0);
    case HELP_CMD:
	if (!redo) flush_to_newline();
	write_output(message_out, "\nSPIM is a MIPS R2000 simulator.\n");
	write_output(message_out, "Its top-level commands are:\n");
	write_output(message_out, "exit  -- Exit from the simulator\n");
	write_output(message_out,
       "read \"FILE\" -- Read FILE (assembly or a.out format) into memory\n");
	write_output (message_out,"load \"FILE\" -- Same as read\n");
      write_output(message_out,
		 "execute \"A.OUT\" -- Same as read\n");
      write_output(message_out,
		   "run <ADDR> -- Start the program at optional ADDRESS\n");
      write_output(message_out,
		   "step <N> -- Step the program for N instructions\n");
      write_output(message_out,
		 "continue -- Continue program execution without stepping\n");
      write_output (message_out, "print $N -- Print register N\n");
      write_output (message_out,
		 "print $fN -- Print floating point register N\n");
      write_output(message_out,
		 "print ADDR -- Print contents of memory at ADDRESS\n");
      write_output(message_out, "regs -- Print all CPU registers\n");
      write_output(message_out,
		 "reinitialize -- Clear the memory and registers\n");
      write_output(message_out,
		 "breakpoint <ADDR> -- Set a breakpoint at address\n");
      write_output(message_out,
		 "delete <ADDR> -- Delete all breakpoints at address\n");
      write_output(message_out, "list -- List all breakpoints\n");
      write_output(message_out,
	     ". -- Rest of line is assembly instruction to put in memory\n");
      write_output(message_out,
		   "<cr> -- Newline reexecutes previous command\n");
      write_output(message_out, "? -- Print this message\n");
      write_output (message_out,
		"\nMost commands can be abbreviated to their unique prefix\n");
      write_output (message_out, "e.g., ex, re, l, ru, s, p\n\n");
      prev_cmd = HELP_CMD;
      return (0);
    case SET_BKPT_CMD:
    case DELETE_BKPT_CMD:
    {
	int token = (redo ? prev_token : yylex ());
	static mem_addr addr;

	if (!redo) flush_to_newline ();
	if (token == Y_INT)
	  addr = redo ? addr + 4 : yylval;
	else if (token == Y_ID)
	  addr = redo ? addr + 4 : find_symbol_address((char *)yylval);
	else
	  error ("Must supply an address for breakpoint\n");
	if (cmd == SET_BKPT_CMD)
	  add_breakpoint (addr);
	else
	  delete_breakpoint (addr);
	prev_cmd = cmd;
	return (0);
    }

    case LIST_BKPT_CMD:
      if (!redo) flush_to_newline ();
      list_breakpoints ();
      prev_cmd = LIST_BKPT_CMD;
      return (0);

#ifdef mips
    case EXECUTE_CMD:
    {
	int token = (redo ? prev_token : yylex ());
	
	if (!redo) flush_to_newline ();
	if (token == Y_STR)
	    read_aout_file ((char *) yylval);
	else
	    error ("Must supply a filename to read\n");
	prev_cmd = EXECUTE_CMD;
	return (0);
      }
#endif
    default:
	while (yylex () != Y_NL) ;
	error ("Unknown spim command\n");
	return (0);
    }
}

/* Read a SPIM command from the FILE and return its ennuemerated value. */
static int read_assembly_command (FILE *file) {
  int token = yylex ();

  if (token == Y_NL)		/* Blank line means redo */
      return (REDO_CMD);
  else if (token != Y_ID)	/* Better be a string */
      return (UNKNOWN_CMD);
  else if (str_prefix ((char *) yylval, "exit", 2))
      return (EXIT_CMD);
  else if (str_prefix ((char *) yylval, "quit", 2))
      return (EXIT_CMD);
  else if (str_prefix ((char *) yylval, "print", 1))
      return (PRINT_CMD);
  else if (str_prefix ((char *) yylval, "regs", 1))
      return (REGS_CMD);
  else if (str_prefix ((char *) yylval, "print_symbol", 6))
      return (PRINT_SYM_CMD);
  else if (str_prefix ((char *) yylval, "run", 2))
      return (RUN_CMD);
  else if (str_prefix ((char *) yylval, "args", 2))
      return (ARGS_CMD);
  else if (str_prefix ((char *) yylval, "read", 2))
      return (READ_CMD);
  else if (str_prefix ((char *) yylval, "load", 2))
      return (READ_CMD);
  else if (str_prefix ((char *) yylval, "reinitialize", 6))
      return (REINITIALIZE_CMD);
  else if (str_prefix ((char *) yylval, "step", 1))
      return (STEP_CMD);
  else if (str_prefix ((char *) yylval, "help", 1))
      return (HELP_CMD);
  else if (str_prefix ((char *) yylval, "continue", 1))
      return (CONTINUE_CMD);
  else if (str_prefix ((char *) yylval, "breakpoint", 2))
      return (SET_BKPT_CMD);
  else if (str_prefix ((char *) yylval, "delete", 1))
      return (DELETE_BKPT_CMD);
  else if (str_prefix ((char *) yylval, "list", 2))
      return (LIST_BKPT_CMD);
#ifdef mips
  else if (str_prefix ((char *) yylval, "execute", 3))
      return (EXECUTE_CMD);
#endif
  else if (*(char *) yylval == '?')
      return (HELP_CMD);
  else if (*(char *) yylval == '.')
      return (ASM_CMD);
  else
      return (UNKNOWN_CMD);
}


/* Return non-nil if STRING1 is a (proper) prefix of STRING2. */
static int str_prefix (char *s1, char *s2, int min_match) {
    for ( ; *s1 == *s2 && *s1 != '\0'; s1 ++, s2 ++) min_match --;
    return (*s1 == '\0' && min_match <= 0);
}

/* Read and return an integer from the current line of input.  If the
   line doesn't contain an integer, return 0. */
static long get_opt_int (void) {
    int token;

    if ((token = yylex()) == Y_INT) {
	return(yylval);
    }
    return(0);
}

/* Flush the rest of the input line up to and including the next newline. */
static void flush_to_newline(void) {
  while (yylex () != Y_NL) ;
}

/* Print register number N.  TYPE code indicate which register set to use.
   Return non-zero if register N was valid register string. */
static int print_reg(int reg_no, int type_code) {
  switch (type_code) {
  case 0:
      write_output (message_out, "Reg %d = 0x%08x (%d)\n",
		    reg_no, R[reg_no], R[reg_no]);
      break;
  case 1:
      write_output (message_out, "FP reg %d = %f (double)\n",
		    reg_no, FPR_D (reg_no));
      write_output (message_out, "FP reg %d = %f (single)\n",
		 reg_no, FPR_S (reg_no));
      break;
  case 2:
  {
      char *s = (char *) reg_no, *s1 = (char *) reg_no;

      /* Convert to lower case */
      for ( ; *s != '\0'; s ++) *s = tolower (*s);
      /* Drop leading $ */
      if (*s1 == '$') s1 += 1;
      
      if (streq(s1, "pc"))
	  write_output(message_out, "PC = 0x%08x (%d)\n", PC, PC);
      else if (streq(s1, "hi"))
	  write_output(message_out, "HI = 0x%08x (%d)\n", HI, HI);
      else if (streq(s1, "lo"))
	  write_output(message_out, "LO = 0x%08x (%d)\n", LO, LO);
      else if (streq(s1, "fpcond"))
	  write_output(message_out, "FpCond = 0x%08x (%d)\n", FpCond, FpCond);
      else if (streq(s1, "cause"))
	  write_output(message_out, "Cause = 0x%08x (%d)\n", Cause, Cause);
      else if (streq(s1, "epc"))
	  write_output(message_out, "EPC = 0x%08x (%d)\n", EPC, EPC);
      else if (streq(s1, "status"))
	  write_output(message_out, "Status = 0x%08x (%d)\n",
			Status_Reg, Status_Reg);
      else if (streq(s1, "badvaddr"))
	  write_output(message_out, "BadVAddr = 0x%08x (%d)\n",
			BadVAddr, BadVAddr);
      else if (streq(s1, "context"))
	  write_output(message_out, "Context = 0x%08x (%d)\n",
			Context, Context);
      else if (streq(s1, "prid"))
	  write_output(message_out, "PRId = 0x%08x (%d)\n", PRId, PRId);
      else
	  return (0);
      break;
  }
  default:
      write_output(message_out, "Strange type code to print_reg\n");
      break;
  }
  return (1);
}
