#include <iostream>
#include <random>
#include <string>
#include <cmath>
#include <chrono>
#include "dlib/matrix.h"
#include "classes.h"
#include "KLP.h"

//#include <fstream>   // for writing to .txt files

using namespace dlib;
using namespace std;

// define design & design optimization dimensions (globally)
const int n_betas = 20;
const int n_attr  = 5;
const int n_draws = 100;
const int n_tasks = 21; 
const int n_cols = n_attr*2;


// define design draws (globablly)  
struct draws {
  // for current design	
  matrix <double> beta;
  matrix <double> expV;
  matrix <double> P;
  matrix <double> P1minP;
  matrix <double> trX_P1minP;
  matrix <double> Fisher;
  double D_eff;
  
  // for updating the DCE design
  double newExpV,newP, newP1minP;
  matrix <double> newTrX_P1minP;
  matrix <double> newFisher;
  double newD_eff;
} draw[n_draws];


void str2mat(const string input, int n_rows, int n_cols, matrix<double> &output, string delimiter)
{
	if (delimiter.length() > 0)
	{
		int start = 0;
		for (size_t i = 0; i < n_rows; ++i)
		{
			for (size_t j = 0; j < n_cols; j++)
			{
				int ind = input.find(delimiter, start);
				output(i, j) = atof(input.substr(start, ind - start).c_str());
				start = ind + 1;
			}
		}
	}
	else
	{
		int start = 0;
		for (size_t i = 0; i < n_rows; ++i)
		{
			for (size_t j = 0; j < n_cols; j++)
			{
				char  str[2] = { '\0','\0' };
				str[0]=input[start];
				output(i, j) = atof(str);
				start += 1;
			}
		}
	}
}
void mat2str(const matrix<double> &input, int n_rows, int n_cols, string &output, string delimiter)
{
	string output2;
	if (delimiter.length() > 0)
	{
		for (size_t i = 0; i < n_rows; i++)
		{
			for (size_t j = 0; j < n_cols; j++)
			{
				std::ostringstream strs;
				strs << input(i, j);
				output2 += strs.str();
				if (i != n_rows - 1 || j != n_cols - 1)
					output2 += delimiter;
			}
		}
	}
	else
	{
		for (size_t i = 0; i < n_rows; i++)
		{
			for (size_t j = 0; j < n_cols; j++)
			{
				std::ostringstream strs;
				strs << input(i, j);
				output2 += strs.str();
			}
		}
	}
	output = output2;
}


void design2X(const matrix<double> &design, matrix<double> &X, matrix<double> &X_sign, matrix<double> &X_likelihood, int n_observed, const matrix<int> &max_levels, matrix<int> &beg_col , matrix<int> &end_col)       
{
    // step1. X_matrix column indices per attribute
    beg_col(0)=0; // first attribute
    end_col(0)=max_levels(0)-2; // first attribute
    for (int i = 1; i < n_attr; i++){
      beg_col(i)=beg_col(i-1) + max_levels(i)-1;
      end_col(i)=end_col(i-1) + max_levels(i)-1;
    }
    
    //cout << "beg_col: \n" << beg_col << endl;
    //cout << "end_col: \n" << end_col << endl;
    //cout << "max_levels: \n" << max_levels << endl;
    
    // step2. create dummy-coded design (omitted base-case = level 1)
    matrix <double, n_tasks,2*n_betas> X_dummy;
    X_dummy = 0.0;
    
    int attr = 0;
    int level = 2;
     
    for (int j = 0; j < 2*n_betas; j++){
     for (int i = 0; i < n_tasks; i++){
       if (design(i,attr)==level) X_dummy(i,j)=1;
     }
     
     level++; 
     if (level>max_levels(attr)){
     	attr++;
     	level=2;
    }}
    
    //cout << "X_dummy: \n" << X_dummy << endl;
    
    
    // step3. create X matrix (differenced)
    // matrix <double, n_tasks, n_betas> X; --> input
    for (int j = 0; j < n_betas; j++){
     for (int i = 0; i < n_tasks; i++){
       X(i,j) = X_dummy(i,j) - X_dummy(i,j+n_betas);
    }}  
	 
    // step4. Create X_sign matrix 
	// -1==lower; 0==equal; 1==higher     
   	// matrix <int, n_tasks, n_attr> X_sign; --> input  
	    	
    X_sign=0;
	for (int i = n_observed; i < n_tasks; i++ ){
   	 for (int j = 0; j < n_attr; j++){
   	  for (int t = end_col(j); t>=beg_col(j);t--){
	   if (X(i,t)==-1){
	  	 X_sign(i,j)=-1;
		 break;   	
	   } 
	   else if (X(i,t)==1){
	     X_sign(i,j)=1;	
	     break;
    }}}} 

    // step 5. Create X-likelihood
    X_likelihood = rowm(X,range(0,n_observed-1));
    
    
    //cout << "X (before optimization) = \n" << X << endl; 
    //cout << "X_sign: \n" << X_sign << endl;	  	
    //cout << "X_Likelihood: \n" << X_likelihood << endl;       
}


void definePriors(matrix <double> &prior_mu, matrix <double> &prior_tau, double &ldet)
{
    // step1. initialise the prior mean
    prior_mu = -0.14,
               -0.36,
               -0.89,
               -1.29,
               -0.19,
               -0.34,
               -0.83,
               -1.20,
               -0.13,
               -0.24,
               -0.66,
               -0.89,
               -0.22,
               -0.39,
               -1.02,
               -1.34,
               -0.28,
               -0.55,
               -1.26,
               -1.48;	
   
    // step2. initialize the prior precision matrix
    prior_tau = 18.49, -6.29, -4.40, -0.88, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                -6.29, 17.45, -4.01, -2.08, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                -4.40, -4.01, 17.35, -8.03, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                -0.88, -2.08, -8.03, 11.44, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                 0, 0, 0, 0, 17.50, -4.98, -1.30, -2.00, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                 0, 0, 0, 0, -4.98, 20.32, -7.35, -3.54, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                 0, 0, 0, 0, -1.30, -7.35, 19.19, -6.44, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                 0, 0, 0, 0, -2.00, -3.54, -6.44, 11.37, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
                 0, 0, 0, 0, 0, 0, 0, 0, 18.53, -5.91, -5.01, -3.06, 0, 0, 0, 0, 0, 0, 0, 0,
                 0, 0, 0, 0, 0, 0, 0, 0, -5.91, 19.38, -3.69, -4.64, 0, 0, 0, 0, 0, 0, 0, 0,
                 0, 0, 0, 0, 0, 0, 0, 0, -5.01, -3.69, 15.21, -5.27, 0, 0, 0, 0, 0, 0, 0, 0,
                 0, 0, 0, 0, 0, 0, 0, 0, -3.06, -4.64, -5.27, 13.80, 0, 0, 0, 0, 0, 0, 0, 0,
                 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 15.95, -5.59, -3.84, 0.19, 0, 0, 0, 0,
                 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -5.59, 18.65, -4.48, -3.53, 0, 0, 0, 0,
                 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -3.84, -4.48, 17.33, -8.40, 0, 0, 0, 0,
                 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.19, -3.53, -8.40, 11.55, 0, 0, 0, 0,
                 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20.68, -11.56, -1.66, 1.66,
                 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -11.56, 19.35, -4.40, -2.91,
                 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -1.66, -4.40, 10.96, -6.06,
                 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1.66, -2.91, -6.06, 7.35;
                 
    // step 3. store the log-determinant of the prior precision matrix
    ldet = log(det(prior_tau));                    
                 
}



void calculatePosterior(const matrix<double> &X, const matrix<double> X_likelihood, matrix<double> &b_optim, int n_observed, const matrix<double> &prior_mu, const matrix<double> &prior_tau, const double &ldet, double &postLa, matrix <double> &chol_covar) 	
{
	                   
    // step1. define the posterior objects that will be passed to the optimizer 
    // (see below) in order to locate the minimum of the neg. log-posterior
    nlpostClass nlpost(X_likelihood, prior_mu, prior_tau, ldet, n_observed, n_betas);
    nlpost_gradClass nlpost_grad(X_likelihood, prior_mu, prior_tau, n_observed, n_betas);
       
    // start the minimizer (here we use the BFGS algorithm). 
	// Note that the minimum will be saved in "b_optim" (i.e. overwrites the
    // starting value)
    
	// step2. minimize the negative log-likelihood  
    
	//cout << "b_optim (before): \n" << b_optim << endl;
	
	// the (neg) posterior minimum found is saved in "b_optim"
    find_min(bfgs_search_strategy(),
             objective_delta_stop_strategy(1e-13),
             nlpost, nlpost_grad,
             b_optim, -100); // if optimiser goes under 100 it stops.
    
    //cout << "b_optim (after): \n" << b_optim << endl; 
    
    
    // step 3. compute the posterior Hessian and (cholesky of the) covariance matrix
    matrix <double, n_betas, n_betas> hess = nlogPost_hess(b_optim, X_likelihood, prior_tau, n_observed, n_betas);
    chol_covar = chol(inv(hess));
    
   	// step 4. compute the Laplace approx. of the posterior normalizing constant
    postLa = 0.5*n_betas*log(2*pi) - nlpost(b_optim) - 0.5*log(det(hess));    

    // write hessian to file (and compare results in Matlab)
    //std::ofstream output("output_matrix.txt"); 
    //for (int i=0;i<n_betas;i++)
    //{
    //	for (int j=0;j<n_betas;j++)
    //	{
    //		output << hess(i,j) << " "; 
    //	}
    //	output << "\n";
    //}  
}


void defineLHS(matrix <double> &LHS)
{    
    if (n_draws == 25)
    LHS =   0.40037081, 0.96561607, 0.397398527, 0.3140954, 0.54583782, 0.85650485, 0.46119228, 0.22869428, 0.96056765, 0.19217748, 0.04640668, 0.19092014, 0.29053384, 0.48884678, 0.22052596, 0.26192832, 0.66996403, 0.003253766, 0.566851037, 0.79640614,
            0.21161751, 0.01411606, 0.715280874, 0.24579739, 0.60255177, 0.81883901, 0.40523647, 0.33326591, 0.43191674, 0.30688677, 0.00501921, 0.68918198, 0.4773185, 0.10917435, 0.631677578, 0.4376876, 0.08196812, 0.144216822, 0.104096674, 0.102006,
            0.52940979, 0.23701106, 0.058603196, 0.44258732, 0.92223244, 0.56689643, 0.12275848, 0.06283051, 0.55169104, 0.099986, 0.51837496, 0.26789955, 0.86996088, 0.96571039, 0.520049987, 0.39804143, 0.56332088, 0.520701769, 0.916770187, 0.75734453,
            0.28746133, 0.74862020, 0.246966743, 0.7029953, 0.01901675, 0.31988824, 0.06753281, 0.10434417, 0.13151649, 0.5202369, 0.71891902, 0.04267614, 0.25907733, 0.38128252, 0.483521408, 0.09218651, 0.19656075, 0.708906888, 0.313883157, 0.29684108,
            0.91851491, 0.52214236, 0.11864926, 0.21078964, 0.05765878, 0.92340689, 0.88871429, 0.58504852, 0.73321594, 0.39782881, 0.55305988, 0.30589503, 0.80625369, 0.1280513, 0.687228513, 0.0214583, 0.35694212, 0.753672629, 0.662830929, 0.37910475,
            0.34834181, 0.40937139, 0.519660118, 0.38161685, 0.18357556, 0.9974188, 0.61554817, 0.43571105, 0.27440105, 0.64213157, 0.2556852, 0.94301273, 0.65998857, 0.86329549, 0.846150968, 0.30052335, 0.81706843, 0.904990563, 0.157880496, 0.92101126,
            0.56622520, 0.29543760, 0.802258076, 0.74592933, 0.78835466, 0.27913104, 0.99355691, 0.37169022, 0.31162203, 0.01843438, 0.80179022, 0.43807192, 0.89446323, 0.44380324, 0.097615634, 0.05119387, 0.50501557, 0.421984507, 0.060768806, 0.32895101,
            0.80522818, 0.36370000, 0.009385709, 0.90342976, 0.98838237, 0.78115219, 0.58064027, 0.47688527, 0.38744034, 0.59915173, 0.97207016, 0.76578611, 0.36536859, 0.29281766, 0.744239675, 0.9433093, 0.47995289, 0.671911452, 0.259098872, 0.86180631,
            0.13189537, 0.78365221, 0.34455425, 0.93697381, 0.29975447, 0.69137984, 0.64619008, 0.92001895, 0.08838923, 0.32142881, 0.41756498, 0.81351603, 0.95137241, 0.35654071, 0.327343119, 0.44711521, 0.97719532, 0.277137436, 0.879542272, 0.20732265,
            0.93532389, 0.50981089, 0.413413197, 0.41077939, 0.11446612, 0.15067156, 0.33810707, 0.81441602, 0.70162726, 0.05622037, 0.64203259, 0.60248161, 0.96542015, 0.27306783, 0.658451928, 0.99312401, 0.38805749, 0.064053479, 0.371852194, 0.41437254,
            0.85031926, 0.95643327, 0.856586374, 0.07631156, 0.45973904, 0.73859199, 0.48126653, 0.760033, 0.16846261, 0.48309492, 0.73572826, 0.46622552, 0.13908532, 0.81154342, 0.007507623, 0.74065734, 0.53434509, 0.33876468, 0.00053087, 0.52335693,
            0.19346034, 0.24903686, 0.123107339, 0.16896248, 0.37086202, 0.03155454, 0.7527082, 0.62797093, 0.22037197, 0.60498905, 0.28286677, 0.52143225, 0.03284455, 0.06521731, 0.077555977, 0.60046944, 0.69538197, 0.773338344, 0.440706505, 0.69581998,
            0.26175927, 0.15367466, 0.977006695, 0.15791782, 0.12877851, 0.67593803, 0.26932785, 0.15139578, 0.9418588, 0.42741159, 0.91512567, 0.37012728, 0.63447731, 0.42574527, 0.364536874, 0.81409451, 0.75799912, 0.981526975, 0.497823879, 0.50770097,
            0.10714175, 0.34486943, 0.46982957, 0.59064514, 0.7081285, 0.09765406, 0.19028372, 0.91453991, 0.83883535, 0.69593585, 0.39322427, 0.90533091, 0.70992612, 0.79658379, 0.132951419, 0.2123713, 0.1268052, 0.840169387, 0.217717377, 0.65559251,
            0.69130184, 0.06808387, 0.527952809, 0.86637187, 0.21703082, 0.34377029, 0.93395802, 0.24699976, 0.47379406, 0.73391849, 0.22056631, 0.50964294, 0.40605471, 0.91596126, 0.454020962, 0.58811462, 0.26237464, 0.206256457, 0.991244669, 0.99521179,
            0.66047248, 0.60449562, 0.887754375, 0.52867016, 0.51342661, 0.61564736, 0.01955101, 0.2909338, 0.01311796, 0.44097563, 0.1032636, 0.560489, 0.48433367, 0.1649433, 0.434564713, 0.89072522, 0.07915065, 0.592536046, 0.925320388, 0.83701114,
            0.44126523, 0.66668797, 0.940478906, 0.49940675, 0.43834918, 0.19400326, 0.81765087, 0.50942855, 0.63999056, 0.14900521, 0.8796793, 0.99686736, 0.07136766, 0.65248495, 0.943977651, 0.13764515, 0.41608162, 0.288305568, 0.739904672, 0.60397315,
            0.72524391, 0.17678222, 0.30813881, 0.3226449, 0.24395618, 0.05656062, 0.11674622, 0.8595887, 0.04182978, 0.83347943, 0.77647371, 0.23876964, 0.59367487, 0.70429604, 0.800551507, 0.34872368, 0.89360569, 0.099499332, 0.55275022, 0.45970197,
            0.76832009, 0.09685706, 0.660594454, 0.7981068, 0.85593854, 0.90254578, 0.30587567, 0.98062514, 0.64719977, 0.85681685, 0.4720389, 0.35104107, 0.099856, 0.5435483, 0.170924556, 0.16643248, 0.79165488, 0.47939593, 0.699892586, 0.13291012,
            0.51191964, 0.46465920, 0.197173419, 0.02380267, 0.88485576, 0.45731226, 0.7879695, 0.52795299, 0.51675255, 0.26375033, 0.59715943, 0.67076967, 0.17215574, 0.93880948, 0.568806727, 0.70860213, 0.02694228, 0.81049337, 0.820467448, 0.03817665,
            0.97384832, 0.88996749, 0.217057605, 0.83299992, 0.33898046, 0.41387146, 0.20593012, 0.01737531, 0.84732103, 0.77288873, 0.35230895, 0.87167513, 0.32520643, 0.62133537, 0.294168271, 0.66749612, 0.6310369, 0.519573797, 0.191373242, 0.19797826,
            0.05926757, 0.58304997, 0.602664521, 0.6102703, 0.7566087, 0.51318491, 0.69064969, 0.74167156, 0.88928994, 0.88503666, 0.94369472, 0.02635477, 0.55266124, 0.01092231, 0.269919508, 0.53203555, 0.28933766, 0.383308575, 0.62306381, 0.88877768,
            0.01891994, 0.69453705, 0.591864873, 0.99772734, 0.57793375, 0.20709352, 0.39024726, 0.70963139, 0.76347709, 0.23312059, 0.15359342, 0.14135096, 0.20365408, 0.7296104, 0.918631873, 0.86838393, 0.86215133, 0.616620705, 0.32508726, 0.59434126,
            0.63778608, 0.85253339, 0.775672877, 0.65965579, 0.6528504, 0.39837262, 0.85530281, 0.64175464, 0.34479438, 0.98898225, 0.18022766, 0.10829074, 0.78966378, 0.57701972, 0.782735887, 0.79051734, 0.20666533, 0.930179405, 0.408079276, 0.07780036,
            0.38276677, 0.82373447, 0.728337983, 0.11560487, 0.8095107, 0.53842471, 0.54191445, 0.18294129, 0.56466449, 0.92939848, 0.60301047, 0.730483, 0.73210762, 0.23772155, 0.993732805, 0.4923594, 0.940501, 0.189253437, 0.781073495, 0.27422342;
    else if (n_draws == 100)
    LHS = 0.265309394, 0.384492078, 0.99770682, 0.056047793, 0.890042637, 0.16733157, 0.319756973, 0.839916211, 0.05302584, 0.022721483, 0.446059949, 0.771225313, 0.5067177, 0.5608419, 0.380760356, 0.235040143, 0.359588434, 0.264258122, 0.177849651, 0.662128913,
          0.568182184, 0.472766435, 0.84532285, 0.075169893, 0.733605437, 0.675847149, 0.366780253, 0.89748872, 0.79223787, 0.967231818, 0.55138743, 0.098766737, 0.108416466, 0.4407151, 0.608019035, 0.544225622, 0.890183448, 0.110524157, 0.248235999, 0.854077097,
          0.421371667, 0.628230362, 0.64158268, 0.417098113, 0.252775494, 0.035078886, 0.553703588, 0.800667589, 0.72611923, 0.886800813, 0.893246169, 0.678506293, 0.580806957, 0.973528, 0.18065313, 0.060326607, 0.17928306, 0.882488663, 0.303590242, 0.80126882,
          0.571473959, 0.119991841, 0.52546065, 0.171026428, 0.372508134, 0.993546463, 0.267404285, 0.123984489, 0.53507141, 0.311405438, 0.987936232, 0.981647918, 0.488070364, 0.740565, 0.810930523, 0.315624869, 0.37776275, 0.154505463, 0.812421082, 0.625240456,
          0.026977768, 0.754417723, 0.91700688, 0.848958587, 0.870576511, 0.290611482, 0.207520987, 0.750322651, 0.18904839, 0.362369479, 0.466907308, 0.457990442, 0.390659463, 0.997803, 0.247197886, 0.665433455, 0.755174603, 0.593020621, 0.948195197, 0.44846661,
          0.147057111, 0.438080245, 0.08886653, 0.485347496, 0.8503565, 0.641712569, 0.864648948, 0.44834482, 0.63371589, 0.247343196, 0.94998588, 0.157279889, 0.473298728, 0.779805, 0.721875528, 0.899760347, 0.607268144, 0.147514707, 0.040458074, 0.233211443,
          0.597536262, 0.597373575, 0.02772701, 0.109041638, 0.760677948, 0.367675594, 0.096448101, 0.403047909, 0.65441924, 0.546447728, 0.287012435, 0.249767152, 0.619788061, 0.1592703, 0.926126447, 0.027526631, 0.13804031, 0.296318148, 0.63434273, 0.464464282,
          0.288358367, 0.254598478, 0.11436296, 0.373499842, 0.325295937, 0.106847973, 0.136397101, 0.79876113, 0.70281117, 0.109261892, 0.351789768, 0.948531035, 0.194510817, 0.1325739, 0.803566043, 0.691218239, 0.585968778, 0.642994013, 0.803134213, 0.740199823,
          0.751886513, 0.524848647, 0.60889113, 0.715536441, 0.602920967, 0.382818065, 0.666058747, 0.541826857, 0.58664653, 0.356396591, 0.975735955, 0.163223204, 0.959491821, 0.0615486, 0.023527998, 0.297062652, 0.443395411, 0.990608456, 0.872693682, 0.298638262,
          0.932042072, 0.123796765, 0.55867525, 0.503614126, 0.694775214, 0.158204179, 0.423282689, 0.682763533, 0.93623338, 0.234489326, 0.033122546, 0.530037879, 0.770185271, 0.7587762, 0.080031342, 0.465161804, 0.77614653, 0.413271941, 0.92878349, 0.885748379,
          0.207373674, 0.939290118, 0.17363044, 0.602705674, 0.630812048, 0.335558034, 0.450339005, 0.956662736, 0.66010696, 0.05639951, 0.828225766, 0.016895463, 0.655489001, 0.5166326, 0.150037429, 0.157088275, 0.285848307, 0.451555973, 0.738353182, 0.702694098,
          0.368930484, 0.18083764, 0.44433957, 0.821998834, 0.464667643, 0.136305369, 0.678970503, 0.39714215, 0.04723959, 0.849079979, 0.81139249, 0.743589062, 0.49692539, 0.4259936, 0.716699383, 0.967980831, 0.270886136, 0.912861325, 0.747099354, 0.051146497,
          0.698946006, 0.847471946, 0.19241417, 0.860631672, 0.542556193, 0.469352643, 0.595501033, 0.474277501, 0.27901718, 0.032166219, 0.839505794, 0.957018319, 0.381864311, 0.8983338, 0.938489776, 0.070063385, 0.66983568, 0.707002587, 0.150138469, 0.485102683,
          0.951659592, 0.940574357, 0.01260861, 0.181430151, 0.356843311, 0.606236024, 0.37436633, 0.992805259, 0.38351153, 0.739270369, 0.374169818, 0.596274448, 0.52326823, 0.7908503, 0.94764158, 0.560270456, 0.362692059, 0.308160173, 0.275652588, 0.103019627,
          0.716741865, 0.742362982, 0.41709914, 0.231233919, 0.079833713, 0.802274159, 0.049844128, 0.413224625, 0.08035535, 0.783000961, 0.507192615, 0.711678619, 0.096690133, 0.1424277, 0.43766856, 0.377603751, 0.901717111, 0.003431281, 0.561391947, 0.58678171,
          0.586935607, 0.950041188, 0.2638934, 0.062594747, 0.481887615, 0.393289267, 0.525824506, 0.96568856, 0.03539445, 0.556548808, 0.11553663, 0.201221408, 0.605456332, 0.4788596, 0.079843113, 0.807182394, 0.429542904, 0.980656933, 0.531815477, 0.425359778,
          0.547821541, 0.69284615, 0.16641204, 0.933532039, 0.849992044, 0.552224473, 0.121312524, 0.347121361, 0.09627265, 0.805895809, 0.771889517, 0.964548681, 0.131019503, 0.6730321, 0.303266289, 0.997341297, 0.846135164, 0.545436327, 0.617275634, 0.685740349,
          0.439156376, 0.50454584, 0.06003574, 0.193912257, 0.651726994, 0.003452515, 0.40274747, 0.677826793, 0.22746444, 0.063229242, 0.908199878, 0.843594985, 0.631957341, 0.5593198, 0.496684451, 0.865574675, 0.053165711, 0.715340089, 0.686161845, 0.19274247,
          0.249085742, 0.77014174, 0.76443838, 0.091774543, 0.019026146, 0.28621555, 0.78909113, 0.520359933, 0.13481185, 0.510964639, 0.34705325, 0.03138955, 0.552188353, 0.3076143, 0.874105308, 0.099480795, 0.471738849, 0.221987578, 0.753932411, 0.213731776,
          0.886940376, 0.677348755, 0.7830517, 0.81455435, 0.723708652, 0.251172664, 0.980152223, 0.530132041, 0.82446053, 0.89975422, 0.208074861, 0.216032869, 0.331267733, 0.8601916, 0.792085761, 0.950482212, 0.227843718, 0.551969976, 0.593343889, 0.384042708,
          0.82114137, 0.737454393, 0.87516166, 0.831824548, 0.097152654, 0.41546569, 0.083406621, 0.577660598, 0.28524254, 0.327007971, 0.844955392, 0.806686898, 0.238706927, 0.9295102, 0.143348234, 0.901474221, 0.409823414, 0.388485813, 0.381203734, 0.76513644,
          0.183211971, 0.854549308, 0.13769337, 0.401747351, 0.004287846, 0.812809218, 0.827933646, 0.224481624, 0.75681444, 0.43663091, 0.291084143, 0.631332991, 0.790880267, 0.8536814, 0.885118145, 0.408748113, 0.246562066, 0.331646749, 0.772649851, 0.115313157,
          0.551877169, 0.309860222, 0.94987387, 0.35032761, 0.569331505, 0.910903659, 0.767837013, 0.284439218, 0.06501354, 0.455785254, 0.698442978, 0.412216425, 0.929293132, 0.9117154, 0.007534925, 0.246479097, 0.579867112, 0.477000992, 0.201754978, 0.027402252,
          0.709409976, 0.164786191, 0.24040039, 0.241648228, 0.028484243, 0.421202662, 0.245492056, 0.362204258, 0.36739246, 0.506141758, 0.523165434, 0.068007744, 0.25181605, 0.5803436, 0.032278262, 0.16695997, 0.765198597, 0.517560461, 0.990132605, 0.086568411,
          0.102344524, 0.921918097, 0.88482984, 0.525798443, 0.459471665, 0.661727333, 0.002310726, 0.978992294, 0.88986469, 0.426644298, 0.487485444, 0.379962138, 0.90893818, 0.2298634, 0.509057473, 0.390696361, 0.325400809, 0.202189141, 0.479426257, 0.832728749,
          0.669164759, 0.823253407, 0.93151881, 0.855913308, 0.536487865, 0.304094951, 0.581632806, 0.275889109, 0.64508514, 0.923527283, 0.154396278, 0.702075539, 0.281654804, 0.1953169, 0.838232957, 0.042972966, 0.73880306, 0.36581598, 0.671702366, 0.957434024,
          0.388440804, 0.836108975, 0.92670456, 0.210239114, 0.954836935, 0.357398952, 0.930015269, 0.389168695, 0.34489106, 0.370567871, 0.88063994, 0.641348642, 0.537486518, 0.7095206, 0.989421175, 0.557719178, 0.874568479, 0.743918825, 0.423073509, 0.979772632,
          0.849362906, 0.89735119, 0.73993319, 0.895357553, 0.249735979, 0.476553422, 0.279485208, 0.485509106, 0.52257014, 0.287712872, 0.954184541, 0.905102422, 0.566053846, 0.111544, 0.784938987, 0.433211001, 0.062911974, 0.66687829, 0.57632805, 0.879020498,
          0.291429256, 0.965331144, 0.28033523, 0.655768716, 0.668568905, 0.221706658, 0.354365807, 0.030974942, 0.0268835, 0.228987441, 0.221995632, 0.025088602, 0.347305433, 0.8019486, 0.655232888, 0.711526371, 0.491528202, 0.180410368, 0.293741381, 0.8920263,
          0.516357808, 0.314233444, 0.50642236, 0.977543678, 0.948123537, 0.741058535, 0.060225893, 0.652498664, 0.35993692, 0.210268153, 0.543528886, 0.298027735, 0.265297623, 0.05377428, 0.426239237, 0.827468515, 0.566690268, 0.530836294, 0.141675139, 0.969377322,
          0.055775101, 0.656407419, 0.5883531, 0.949411449, 0.575461413, 0.490500447, 0.843551255, 0.172530918, 0.95683197, 0.335307742, 0.578652698, 0.912058851, 0.439568112, 0.9630858, 0.67417853, 0.602215799, 0.259242709, 0.678919827, 0.839610639, 0.825547107,
          0.906818923, 0.906090145, 0.66284779, 0.887767218, 0.197350529, 0.177799546, 0.755236927, 0.308924246, 0.12604098, 0.255806727, 0.628660333, 0.441835458, 0.821584008, 0.3309478, 0.313548613, 0.588418031, 0.198706046, 0.230168873, 0.098519744, 0.122653886,
          0.324149501, 0.453260921, 0.10318021, 0.361537227, 0.389522162, 0.053270659, 0.68938611, 0.003559247, 0.97133846, 0.168259818, 0.415764817, 0.177259351, 0.021300993, 0.1769307, 0.138762588, 0.489537658, 0.548757308, 0.492208761, 0.581549641, 0.322282608,
          0.644631647, 0.985056287, 0.71246266, 0.555247438, 0.5186828, 0.713896811, 0.607808129, 0.141014825, 0.24592315, 0.931959695, 0.851729926, 0.057182134, 0.364379566, 0.5728791, 0.418365585, 0.182594511, 0.949029971, 0.651729881, 0.015117146, 0.632841372,
          0.830045638, 0.007414637, 0.43840762, 0.624951435, 0.525211415, 0.909937889, 0.467084838, 0.269827897, 0.60762916, 0.818577861, 0.916154071, 0.109464097, 0.858388566, 0.769336, 0.538129008, 0.819658858, 0.021083939, 0.867841676, 0.411091601, 0.53172267,
          0.774026999, 0.14951051, 0.62235519, 0.548745458, 0.987710527, 0.14392543, 0.198742574, 0.07081778, 0.37486093, 0.970318574, 0.925565195, 0.58651334, 0.705718348, 0.2041437, 0.743708537, 0.791805061, 0.832262068, 0.316970518, 0.339768461, 0.496444412,
          0.522490041, 0.789045052, 0.86478594, 0.479339467, 0.907219228, 0.083175097, 0.335875506, 0.09828244, 0.33620452, 0.096919044, 0.049038749, 0.483462062, 0.242640552, 0.2397361, 0.738356392, 0.505573654, 0.695574835, 0.286034619, 0.860790238, 0.169277377,
          0.00084884, 0.887162567, 0.20191543, 0.591274868, 0.676529795, 0.956451628, 0.903875451, 0.743110836, 0.17981272, 0.204048725, 0.407461164, 0.727831485, 0.317827307, 0.07966608, 0.476289617, 0.642887357, 0.672000937, 0.758805831, 0.061934295, 0.438823184,
          0.602066534, 0.97965319, 0.39195561, 0.618541426, 0.047539421, 0.010944105, 0.805347448, 0.422852912, 0.61413508, 0.678134171, 0.096417188, 0.28752888, 0.006840947, 0.4991316, 0.104488698, 0.323313039, 0.438588816, 0.164639136, 0.261377859, 0.529928296,
          0.335440589, 0.346815135, 0.69200159, 0.02849519, 0.064338429, 0.263812999, 0.698343283, 0.297787177, 0.39016751, 0.703307993, 0.6379581, 0.125977063, 0.468041949, 0.3726446, 0.011642685, 0.681929421, 0.381067599, 0.954622543, 0.341174065, 0.997409106,
          0.969132637, 0.663388552, 0.89415563, 0.207040303, 0.71593246, 0.064359971, 0.886579759, 0.567469882, 0.91479333, 0.48198633, 0.603116252, 0.430158367, 0.080495686, 0.6184634, 0.165089939, 0.510177678, 0.918287108, 0.693296296, 0.362517146, 0.2744896,
          0.737618629, 0.60346658, 0.14150808, 0.804073441, 0.137997107, 0.680347707, 0.816650208, 0.028976673, 0.49856824, 0.041463237, 0.788586806, 0.27126492, 0.802393289, 0.3660452, 0.572710446, 0.369114058, 0.781829142, 0.091973077, 0.826720924, 0.774854438,
          0.473435697, 0.631287344, 0.48580198, 0.086050516, 0.629373866, 0.584946341, 0.910811151, 0.77392976, 0.5619102, 0.007940882, 0.395905911, 0.653455992, 0.970079524, 0.594281, 0.232324491, 0.926976605, 0.791319819, 0.500278009, 0.965720983, 0.373076192,
          0.45300919, 0.179684168, 0.79437417, 0.769066289, 0.497569637, 0.764648054, 0.073017378, 0.324586082, 0.48991726, 0.953819993, 0.424893354, 0.555964827, 0.058165654, 0.105408, 0.294104748, 0.281345432, 0.521776447, 0.609804457, 0.160259287, 0.044404635,
          0.806180218, 0.580646696, 0.6134843, 0.532531502, 0.995541677, 0.889714555, 0.621655977, 0.908993365, 0.2038391, 0.381270494, 0.49488423, 0.135510064, 0.307406306, 0.7398423, 0.68056997, 0.216084573, 0.005784203, 0.802693555, 0.628149884, 0.918273656,   
          0.924179594, 0.355859591, 0.51701735, 0.437780856, 0.588680614, 0.327304947, 0.870683673, 0.462332292, 0.41689687, 0.980932525, 0.93449882, 0.979017111, 0.206476617, 0.6385929, 0.250286354, 0.271385195, 0.013435727, 0.121136996, 0.643127994, 0.319446403, 
          0.793723658, 0.090732302, 0.27182251, 0.111334765, 0.179909069, 0.536497769, 0.442825369, 0.722596091, 0.01430944, 0.761164578, 0.702651073, 0.609377766, 0.784542823, 0.2862586, 0.547994602, 0.100595104, 0.340176304, 0.94541696, 0.897119997, 0.710231002, 
          0.650706895, 0.150711833, 0.36323784, 0.705636816, 0.278678109, 0.690771458, 0.320190712, 0.109741528, 0.46983532, 0.184355772, 0.177145077, 0.04730933, 0.987997776, 0.5233583, 0.041407871, 0.633068053, 0.291998133, 0.037173449, 0.194206523, 0.699195888, 
          0.767434324, 0.40690773, 0.07308651, 0.686168513, 0.740944472, 0.896493464, 0.773628766, 0.737493292, 0.9282898, 0.418634367, 0.435562678, 0.894481101, 0.747375895, 0.3169158, 0.094138249, 0.205024214, 0.045666386, 0.061007463, 0.495500983, 0.471336571,  
          0.501687655, 0.017622953, 0.42889417, 0.15562041, 0.433244019, 0.799578879, 0.29177755, 0.11063725, 0.16881845, 0.19367983, 0.273042725, 0.929304525, 0.412955672, 0.08197993, 0.44502402, 0.383411105, 0.084073602, 0.891366412, 0.460599853, 0.30886764,   
          0.083917438, 0.390938559, 0.68486987, 0.660589913, 0.267310232, 0.095363517, 0.853713051, 0.858109076, 0.25835819, 0.685616566, 0.335579803, 0.766698763, 0.890402854, 0.3811793, 0.829245718, 0.756578626, 0.983287385, 0.446494121, 0.075905927, 0.513170805, 
          0.89410024, 0.320205147, 0.745779, 0.388410103, 0.416744429, 0.755852401, 0.995804539, 0.783830201, 0.29176923, 0.137925931, 0.617057236, 0.883869148, 0.811560982, 0.04850932, 0.62564597, 0.331335892, 0.685318847, 0.018398416, 0.662968971, 0.356773469,  
          0.091231411, 0.516074447, 0.22075439, 0.280133588, 0.472749106, 0.842029272, 0.433915863, 0.18399038, 0.90570243, 0.839773765, 0.242730566, 0.837963998, 0.683344071, 0.0259635, 0.228396105, 0.220303332, 0.708476106, 0.686830514, 0.608400985, 0.928759639, 
          0.279813235, 0.447652236, 0.54827509, 0.962740226, 0.759418005, 0.208119045, 0.974971164, 0.597300266, 0.8422689, 0.466767878, 0.585353184, 0.194688876, 0.729273711, 6.95172E-05, 0.328182153, 0.725571576, 0.121015042, 0.105073416, 0.255016286, 0.785018054,
          0.192805288, 0.489749539, 0.95466138, 0.30578094, 0.681292013, 0.876542056, 0.234785477, 0.455908805, 0.7700427, 0.565675355, 0.995030617, 0.26469079, 0.017276243, 0.3243842, 0.064814334, 0.308301796, 0.452501762, 0.576978611, 0.912964273, 0.179608514,  
          0.396911002, 0.46328725, 0.70992298, 0.879254883, 0.424012162, 0.655326047, 0.636129829, 0.35361535, 0.96308421, 0.276175369, 0.063462935, 0.50214456, 0.939941352, 0.09677504, 0.351746848, 0.084826693, 0.655598642, 0.932144915, 0.129161921, 0.20359555,  
          0.499977214, 0.378317154, 0.21131149, 0.955567987, 0.796231809, 0.379257041, 0.573811118, 0.252399229, 0.89369901, 0.152274377, 0.189206243, 0.853665479, 0.737338693, 0.290174, 0.763082858, 0.613512006, 0.391882176, 0.785710167, 0.009608346, 0.931084676 , 
          0.626022584, 0.139879428, 0.40231177, 0.399360247, 0.300201128, 0.922657743, 0.744463278, 0.697526156, 0.15622069, 0.074559694, 0.681757995, 0.075367479, 0.119489075, 0.4652978, 0.378760272, 0.913679227, 0.216221622, 0.177517477, 0.767793295, 0.817556812, 
          0.979464164, 0.103722674, 0.1237478, 0.273887484, 0.805120751, 0.575553449, 0.893140532, 0.508762822, 0.30584303, 0.522868474, 0.122915456, 0.254637911, 0.596331387, 0.8882097, 0.284794916, 0.497181864, 0.854654053, 0.832045668, 0.054698839, 0.606468238 , 
          0.377643835, 0.336520347, 0.32591394, 0.985472501, 0.036639145, 0.516074098, 0.473816988, 0.666035422, 0.98737787, 0.610695595, 0.750866013, 0.518996612, 0.141145347, 0.6929128, 0.216976368, 0.878634882, 0.747686379, 0.582171248, 0.210271678, 0.011142102, 
          0.729646196, 0.565249385, 0.83096126, 0.744797, 0.361189442, 0.56266828, 0.79674882, 0.876552696, 0.3285309, 0.396096727, 0.018236469, 0.61359793, 0.033115795, 0.8350422, 0.191076469, 0.138540754, 0.819403881, 0.431191506, 0.431414357, 0.096891098,    
          0.672635301, 0.227935302, 0.53617335, 0.750871192, 0.500833048, 0.987242886, 0.530910148, 0.219071814, 0.7192267, 0.441392349, 0.006593451, 0.142695541, 0.294143067, 0.6847211, 0.059830893, 0.983121793, 0.630835304, 0.733814724, 0.850485529, 0.757684867, 
          0.869476114, 0.360400535, 0.77089059, 0.925285608, 0.976682817, 0.735388999, 0.287325869, 0.515233022, 0.42996548, 0.40879483, 0.25430564, 0.002190862, 0.699850679, 0.6041308, 0.964296857, 0.35493003, 0.864517263, 0.567127536, 0.729462372, 0.143787811,  
          0.038613567, 0.571234096, 0.15965023, 0.495347007, 0.142505603, 0.851598376, 0.103042765, 0.6346572, 0.50663523, 0.306190185, 0.471098948, 0.864133326, 0.452335732, 0.9533342, 0.203604765, 0.429513721, 0.938336162, 0.872809996, 0.48205735, 0.546615684,  
          0.167280111, 0.617057411, 0.65311456, 0.724831596, 0.2197869, 0.725233945, 0.731120792, 0.317634357, 0.31309578, 0.740972306, 0.051760872, 0.547527961, 0.875845228, 0.0153018, 0.33101059, 0.946016232, 0.414975275, 0.138796625, 0.79025549, 0.225460641,   
          0.239825273, 0.023605523, 0.97252225, 0.789427553, 0.235579267, 0.195188131, 0.30315218, 0.940796541, 0.26630671, 0.578757627, 0.6403098, 0.49596831, 0.8830771, 0.5017301, 0.662376196, 0.110334747, 0.649246359, 0.073816212, 0.550548038, 0.263633442,    
          0.417749357, 0.424052234, 0.46812992, 0.918320714, 0.827452158, 0.315884232, 0.165071556, 0.819383415, 0.67090563, 0.869442121, 0.313169439, 0.46120065, 0.517713874, 0.03108324, 0.367905603, 0.938973016, 0.189103734, 0.921478912, 0.983652166, 0.575766459, 
          0.352855894, 0.051412518, 0.56831725, 0.017032895, 0.92361366, 0.437039676, 0.962567764, 0.559154009, 0.5190208, 0.602219536, 0.070533845, 0.380589541, 0.447380027, 0.6447169, 0.582799611, 0.143391312, 0.162703099, 0.909701195, 0.785337024, 0.247746871,  
          0.300457584, 0.071646552, 0.18191385, 0.690235098, 0.395897426, 0.823884684, 0.544544874, 0.198161167, 0.85567389, 0.917486118, 0.263802531, 0.755573576, 0.408143649, 0.8184727, 0.992052025, 0.008769844, 0.62263694, 0.429843816, 0.402073195, 0.646889201, 
          0.178166457, 0.270918741, 0.05009207, 0.322664712, 0.087225663, 0.073368597, 0.942159562, 0.614182706, 0.59240731, 0.720505692, 0.747788497, 0.568803605, 0.175701463, 0.9360013, 0.637765095, 0.535379868, 0.232773274, 0.242091838, 0.236422299, 0.849145507, 
          0.013075732, 0.201913013, 0.72908624, 0.588920937, 0.288388954, 0.406528169, 0.706447716, 0.927911468, 0.40293945, 0.019754709, 0.671096198, 0.392457322, 0.157638578, 0.167256, 0.516699022, 0.418066815, 0.517975679, 0.969288609, 0.103461011, 0.678479921, 
          0.631119654, 0.218722909, 0.90136889, 0.295551537, 0.157023364, 0.453943902, 0.115565275, 0.60981264, 0.11160814, 0.718381114, 0.023886131, 0.471719013, 0.865426542, 0.7198756, 0.972348964, 0.594816227, 0.261199052, 0.321399203, 0.452813973, 0.902834616, 
          0.404500312, 0.499804232, 0.29970436, 0.900899903, 0.292685219, 0.119065217, 0.489407537, 0.764099889, 0.81228452, 0.58169358, 0.861633086, 0.303172767, 0.065857334, 0.53004, 0.905560965, 0.053106233, 0.889765157, 0.82149289, 0.841378451, 0.555234853,   
          0.34379284, 0.089454344, 0.03861519, 0.167944893, 0.882181207, 0.237941031, 0.159793541, 0.71880869, 0.45772182, 0.348281663, 0.167173827, 0.365846137, 0.271668767, 0.4182504, 0.396506917, 0.521698681, 0.504124644, 0.053551715, 0.021464488, 0.281712136,  
          0.531840063, 0.413652799, 0.30896794, 2.87477E-05, 0.963773172, 0.448627184, 0.175382252, 0.438356981, 0.23165804, 0.591944795, 0.766719722, 0.79134161, 0.54599571, 0.2529389, 0.701294952, 0.251632316, 0.963668041, 0.978440675, 0.315753437, 0.075440224,  
          0.46010478, 0.068573461, 0.85717048, 0.349856948, 0.64315246, 0.040351197, 0.419432665, 0.014451478, 0.73608026, 0.126609046, 0.717594456, 0.575542624, 0.040489245, 0.8274301, 0.851585729, 0.454083621, 0.309522952, 0.619231853, 0.225927735, 0.721331296,  
          0.117280597, 0.534359095, 0.38041948, 0.9915858, 0.816147215, 0.94323172, 0.641181926, 0.889866883, 0.19349382, 0.656334046, 0.219240219, 0.321235823, 0.648516329, 0.948795, 0.613343036, 0.573310349, 0.337326411, 0.042940962, 0.704812648, 0.15031799,   
          0.045368862, 0.238209914, 0.98807798, 0.130848764, 0.444092958, 0.489321181, 0.492271249, 0.241762545, 0.74882015, 0.264112105, 0.561967179, 0.188327655, 0.849056007, 0.6599883, 0.957981088, 0.775695508, 0.617447173, 0.794805612, 0.905049567, 0.401789106, 
          0.984112559, 0.546279584, 0.75988126, 0.046103885, 0.056995242, 0.839843684, 0.951338274, 0.165234237, 0.57275754, 0.627986437, 0.362209717, 0.669510497, 0.228651691, 0.3442412, 0.912074512, 0.832150378, 0.806599198, 0.812940875, 0.51618144, 0.453954593, 
          0.446971407, 0.683644954, 0.9665482, 0.125480017, 0.707393013, 0.597986847, 0.560216378, 0.64385243, 0.99761792, 0.696027155, 0.450206173, 0.930234083, 0.375963212, 0.2658674, 0.866743203, 0.780060855, 0.071905901, 0.375627764, 0.131000884, 0.062363256,  
          0.814825418, 0.247863864, 0.80026583, 0.318518489, 0.31261093, 0.936404954, 0.517024477, 0.625042404, 0.94677706, 0.113386307, 0.735516547, 0.522399649, 0.356161295, 0.8417361, 0.75215025, 0.038653776, 0.205401993, 0.465432419, 0.037645123, 0.39555423,  
          0.312781878, 0.710681113, 0.37409215, 0.63409482, 0.229113063, 0.782796324, 0.023832868, 0.98050871, 0.76546667, 0.666610861, 0.104017433, 0.222762412, 0.323706092, 0.8705448, 0.895238207, 0.127309934, 0.538602274, 0.849780275, 0.376440707, 0.333430332,  
          0.215398515, 0.64163777, 0.23750293, 0.775462389, 0.166695964, 0.544490262, 0.396899442, 0.494574571, 0.00100436, 0.635576469, 0.08211568, 0.732987119, 0.215236489, 0.3911797, 0.113475054, 0.198584043, 0.155786364, 0.358630692, 0.954229435, 0.798726141,  
          0.487534837, 0.792710348, 0.33330747, 0.330216189, 0.124417108, 0.248825879, 0.383611838, 0.040589239, 0.80172538, 0.999862842, 0.720389376, 0.116004523, 0.665396796, 0.4584082, 0.696908069, 0.476092022, 0.146708556, 0.218529293, 0.975445853, 0.730738259, 
          0.787614947, 0.870750589, 0.25928217, 0.567423242, 0.778112729, 0.613396649, 0.030826449, 0.37388583, 0.69939125, 0.084352135, 0.307475463, 0.626114858, 0.078286689, 0.4394725, 0.278955284, 0.845568469, 0.102079413, 0.391732955, 0.283461145, 0.130949799, 
          0.134445909, 0.704605641, 0.59742374, 0.737149324, 0.205527937, 0.968733993, 0.146888379, 0.063501857, 0.21492397, 0.49443221, 0.963360704, 0.233628915, 0.189147823, 0.2458491, 0.648531683, 0.883170016, 0.480844794, 0.523257181, 0.398940537, 0.255603584, 
          0.747148381, 0.802060438, 0.00036337, 0.229579509, 0.781478579, 0.520979564, 0.830995341, 0.82298004, 0.43971171, 0.753620459, 0.799701088, 0.406442382, 0.75988444, 0.2134272, 0.458761978, 0.340681817, 0.991988338, 0.025139458, 0.442938284, 0.657349564,  
          0.685742665, 0.81642377, 0.49108753, 0.146572371, 0.408672191, 0.023210525, 0.252298621, 0.08001953, 0.55586507, 0.777783191, 0.323900235, 0.699019509, 0.943469856, 0.7260682, 0.402499879, 0.7320517, 0.929322875, 0.27280012, 0.521232869, 0.031143388,   
          0.259769258, 0.555429797, 0.31075513, 0.791569332, 0.187695934, 0.180718998, 0.180313923, 0.205325359, 0.10227803, 0.872967107, 0.13845417, 0.346302499, 0.6747101, 0.4847026, 0.52121637, 0.266175006, 0.033945926, 0.779270448, 0.119686454, 0.18789571,   
          0.870913742, 0.280711874, 0.34135072, 0.455996405, 0.932989014, 0.860815164, 0.725676737, 0.866280012, 0.83140842, 0.822900039, 0.805889394, 0.334337034, 0.428088527, 0.1229773, 0.466597201, 0.655173023, 0.464970049, 0.340405653, 0.880714971, 0.008984229, 
          0.993311928, 0.912047581, 0.67661149, 0.510431175, 0.333668187, 0.70482354, 0.01946611, 0.587856817, 0.47469067, 0.172854273, 0.237727677, 0.083444205, 0.832726206, 0.1822898, 0.77704169, 0.761227554, 0.823071589, 0.763757696, 0.544825896, 0.569664803,  
          0.613046646, 0.728668182, 0.81952414, 0.031743782, 0.108753114, 0.500735901, 0.223287942, 0.932565761, 0.87259251, 0.142398918, 0.597919964, 0.420095142, 0.12893407, 0.4086258, 0.568370104, 0.709338392, 0.092354152, 0.481717967, 0.933572907, 0.502205689, 
          0.946331705, 0.04660316, 0.0989672, 0.674649345, 0.836051826, 0.34232607, 0.652438984, 0.233141543, 0.14958144, 0.791115693, 0.535462465, 0.351431153, 0.578302849, 0.2716944, 0.843614314, 0.176161066, 0.319779376, 0.724973297, 0.355138333, 0.982287291,  
          0.917851336, 0.037948161, 0.47821352, 0.429278236, 0.34108307, 0.215350133, 0.057526681, 0.914184374, 0.86918077, 0.646039989, 0.661445198, 0.813790551, 0.968912138, 0.6232961, 0.265462135, 0.745387386, 0.5512134, 0.408687499, 0.08529247, 0.616001331,   
          0.069640055, 0.263785915, 0.35857365, 0.575047419, 0.552263708, 0.777962928, 0.920025564, 0.842035108, 0.07914967, 0.904982753, 0.389949366, 0.997973127, 0.166593567, 0.3544205, 0.550597818, 0.856640066, 0.721086643, 0.195599069, 0.710148103, 0.59674962, 
          0.22499699, 0.996466203, 0.6363462, 0.445094347, 0.915980399, 0.634497309, 0.509572574, 0.708828636, 0.62678245, 0.940741639, 0.144412725, 0.872142579, 0.765397315, 0.9056536, 0.346483246, 0.442362315, 0.112846858, 0.632643768, 0.327902161, 0.861928756,  
          0.153195869, 0.869287987, 0.45015953, 0.264891901, 0.599031455, 0.972869654, 0.212924165, 0.132955897, 0.78620302, 0.856320944, 0.656796543, 0.31435313, 0.919002115, 0.7893179, 0.481068194, 0.976950194, 0.593978452, 0.629848543, 0.18947098, 0.413438554,  
          0.853358056, 0.298180081, 0.04741116, 0.64244418, 0.117621811, 0.275499727, 0.341606991, 0.157444252, 0.44481845, 0.479928181, 0.519068386, 0.821327689, 0.998263171, 0.9858324, 0.596091105, 0.629531344, 0.714927068, 0.852883399, 0.690255882, 0.34171147,  
          0.120397517, 0.195941676, 0.8206055, 0.255632642, 0.61182592, 0.625949234, 0.710556832, 0.054137574, 0.54395605, 0.29202933, 0.198613715, 0.788748573, 0.622520438, 0.6617718, 0.172675052, 0.670611564, 0.97126655, 0.252193299, 0.505342369, 0.941964818, 
          0.072394156, 0.765757277, 0.578632, 0.461423585, 0.869153177, 0.122064346, 0.619657831, 0.337523763, 0.68910294, 0.534561079, 0.871323165, 0.685584658, 0.712137064, 0.5496472, 0.125467672, 0.010400807, 0.9597477, 0.085330659, 0.655644777, 0.362841459;	
}




void insertDraws(const matrix<double> &LHS, const matrix<double> &chol_covar, const matrix<double> b_optim) 
{ 
    // take either 25 or 100 pre-optimized draws from MVN: betas = b_optim + chol_covariance * LHS_stdnorm    
    matrix <double> betas = LHS * chol_covar;
    for (int i = 0; i < n_tasks; i++ ){
     for (int j = 0; j < n_betas; j++){
   	 betas(i,j) = b_optim(j) + betas(i,j);
    }}
   
    //cout << "betas: \n" << betas << endl;
    
    // write betas to file (to compare D-efficiency calculations)
    //std::ofstream output("betas_3x3.txt"); 
    //for (int i=0;i<n_draws;i++)
    //{
    //	for (int j=0;j<n_betas;j++)
    //	{
    //		output << betas(i,j) << " "; 
    //	}
    //	output << "\n";
    //} 
    //output.close();
    
	// insert betas in data structure
	for (int i = 0; i < n_draws; i++ ){
     draw[i].beta = rowm(betas,i);
    }
    
    //cout << "draw[0].beta = \n" << draw[0].beta << endl;
}


void FullEval(matrix<double> &X, double &efficiency)
{
    // loop over all draws
    #pragma omp parallel for num_threads(8)
    for (int d=0; d<n_draws; d++){
    	
     // initialize matrices
     draw[d].expV       = zeros_matrix<double>(n_tasks,1);
     draw[d].P          = zeros_matrix<double>(n_tasks,1);
     draw[d].P1minP     = zeros_matrix<double>(n_tasks,1);
     draw[d].trX_P1minP = zeros_matrix<double>(n_betas,n_tasks);
     draw[d].Fisher     = zeros_matrix<double>(n_betas,n_betas);

     draw[d].newTrX_P1minP = zeros_matrix<double>(n_betas,1);
     draw[d].newFisher     = zeros_matrix<double>(n_betas,n_betas); 
		 
     // initialize matrices
     draw[d].expV       = 0;
     draw[d].P          = 0;
     draw[d].P1minP     = 0;
     draw[d].trX_P1minP = 0;
     draw[d].Fisher     = 0;

     draw[d].newTrX_P1minP = 0;
     draw[d].newFisher     = 0; 

     // loop over all choice tasks
     for (int t=0; t<n_tasks; t++){	
      draw[d].expV(t)   = exp( dot(draw[d].beta, rowm(X,t)) );      
      draw[d].P(t)      = draw[d].expV(t) / (1 + draw[d].expV(t));
      draw[d].P1minP(t) = draw[d].P(t) * (1 - draw[d].P(t) );
	  
	  // required for the element-wise calculation of the Fisher matrix
	  for (int i=0; i<n_betas; i++){ 
	   draw[d].trX_P1minP(i,t) = X(t,i) * draw[d].P1minP(t); 
     }}
     
     // Fisher matrix
	 //draw[d].Fisher = (trans(X)*diagm(draw[d].P1minP)) * X;
	 
	 // Fisher matrix (element-wise)
	 for (int j=0; j<n_betas; j++){
      for (int i=0; i<n_betas; i++){
      	
	   // for computational stability, add very small number to diagonal
	   if (i==j) draw[d].Fisher(i,j)=draw[d].Fisher(i,j)+0.0000001;
	   
       for (int t=0; t<n_tasks; t++){	
        draw[d].Fisher(i,j)=draw[d].Fisher(i,j) + draw[d].trX_P1minP(i,t) * X(t,j);
	 }}}
	 
	 // D-efficiency of draw d
	 draw[d].D_eff= pow( det(draw[d].Fisher) , (-1.0/n_betas ) );

	}
	
	// average efficiency (over all draws)
    efficiency = draw[0].D_eff;
	for (int d=1; d<n_draws; d++) {
	  efficiency = efficiency + draw[d].D_eff;
    } 
	efficiency = efficiency / n_draws;
	
	//cout << "D-error (before optimization) = " << efficiency << endl; 
}



void PartialEval(matrix<double> &X, double &efficiency, int taskNr, const matrix<double> &newTask)
{
    				        	  
    #pragma omp parallel for num_threads(2)
    for (int d=0; d<n_draws; d++){
      draw[d].newExpV   = exp( dot(draw[d].beta, newTask) );      
      draw[d].newP      = draw[d].newExpV / (1 + draw[d].newExpV);
      draw[d].newP1minP = draw[d].newP * (1 - draw[d].newP );
   
      // required for the element-wise calculation of the Fisher matrix
      for (int i=0; i<n_betas; i++){ 
        draw[d].newTrX_P1minP(i) = newTask(i) * draw[d].newP1minP; 
      }
     	  
      // subtract old TrX_P1minP from Fisher & add new TrX_P1minP
      for (int j=0; j<n_betas; j++){
       for (int i=0; i<n_betas; i++){
         draw[d].newFisher(j,i) = draw[d].Fisher(j,i) 
                                - draw[d].trX_P1minP(i,taskNr) * X(taskNr,j) 
                                + draw[d].newTrX_P1minP(i) * newTask(j);      
      }}
     
      // new D-efficiency of draw d
      draw[d].newD_eff= pow( det(draw[d].newFisher) , (-1.0/n_betas ) );
    }
   
   // calculate average of new D-efficiency
   double new_efficiency = draw[0].newD_eff;
   for (int d=0; d<n_draws; d++){
     new_efficiency += draw[d].newD_eff;
   }
   new_efficiency = new_efficiency / n_draws;
   
   
   // F. evaluate efficiency of replacement task
   if (new_efficiency < efficiency) {
            
      efficiency = new_efficiency; 
        
	  // insert improved choice task
      set_rowm(X,taskNr) = newTask;
      
      // update values cf. replacement task
      for (int d=0; d<n_draws; d++){
      
 	    // next three lines are not strictly needed
 	    // comment for a small speed increase - should be removed from draw[] struct (!!)	 
        //  draw[d].expV(taskNr)             = draw[d].newExpV;
        //  draw[d].P(taskNr)                = draw[d].newP;
        //  draw[d].P1minP(taskNr)           = draw[d].newP1minP;
      
        set_colm( draw[d].trX_P1minP,taskNr) = draw[d].newTrX_P1minP;        
        draw[d].Fisher                       = draw[d].newFisher;
      }
   } 
}




void optimizeDesign(matrix<double> &X, matrix<double> &X_sign, int n_observed, matrix<int> &beg_col , matrix<int> &end_col)
{   
    // step1. setup RNG 
    std::random_device rnd;
    std::mt19937 generator(rnd()); // initialize with truly random number
    
    std::uniform_int_distribution<int> first_rand_int(0,n_attr-1);
    std::uniform_int_distribution<int> second_rand_int(0,n_attr-2);

   
    // step 2. determine initial efficiency
    double efficiency;
    FullEval(X, efficiency);
	
	
	// step3. setup candidates 
    int n_candidates, attr1, attr2, randAttr1, randAttr2;
    bool has_lower, has_higher;

    matrix <double> diff(21,4);
    diff= 0,  0,  0,  0, 
    
         -1,  0,  0,  0, 
          0, -1,  0,  0, 
          1, -1,  0,  0, 
          0,  0, -1,  0, 
          1,  0, -1,  0, 
          0,  1, -1,  0, 
          0,  0,  0, -1, 
          1,  0,  0, -1, 
          0,  1,  0, -1, 
          0,  0,  1, -1, 
          
          1,  0,  0,  0, 
          0,  1,  0,  0, 
         -1,  1,  0,  0, 
          0,  0,  1,  0, 
         -1,  0,  1,  0, 
          0, -1,  1,  0, 
          0,  0,  0,  1, 
         -1,  0,  0,  1, 
          0, -1,  0,  1, 
          0,  0, -1,  1; 
		
			                 
   // step4. update all possible FUTURE choice tasks 
   
   // initialize timer 
   std::chrono::time_point<std::chrono::system_clock> startTime;
   typedef std::chrono::system_clock Time;
   typedef std::chrono::duration<double> duration;
   duration diffTime;
   startTime = Time::now();

   
   do{
    for(int t=n_observed; t<n_tasks; t++){
     
     // A. Determine attributes to update 
     // random 1st attribute (0-1-2)
     randAttr1 = first_rand_int(generator);
          
     // random 2nd attribute (0-1)
     randAttr2 = second_rand_int(generator);
     if (randAttr1==randAttr2) randAttr2++; // avoid randAttr1 == randAttr2
 	
 	
 	 // B. Determine the candidate set 
   	 has_lower = false;
	 has_higher = false;
		  
	 for (int attr = 0; attr<n_attr; attr++){	 
		    if ((attr != randAttr1) && (attr != randAttr2)) {
			  if (X_sign(t,attr) == -1) 
			     has_lower = true; 
	          else if (X_sign(t,attr) == 1) 
			     has_higher = true;
            }
     }
   
 	 // C. copy current task
     matrix <double> newTask(1,n_betas);
     newTask = rowm(X,t);
     	
     // D. Insert and evaluate all possible replacements	     
     
     
		 // 1. Always possible (lower-higher)
		 for (attr1 = 1; attr1<=10; attr1++){
           // replace attr1 in newTask
		   set_colm(newTask, range( beg_col(randAttr1),end_col(randAttr1) )) = rowm(diff,attr1);
		   for (attr2 = 11; attr2<=20; attr2++){		
 	         // repl attr2 in newTask
 	         set_colm(newTask, range( beg_col(randAttr2),end_col(randAttr2) )) = rowm(diff,attr2);
	         //evaluate efficiency   
			 PartialEval(X, efficiency, t, newTask); 
		    }
	     }
		 
		 // 2. Always possible (higher-lower)
		 for (attr1 = 11; attr1<=20; attr1++){
           // replace attr1 in newTask
		   set_colm(newTask, range( beg_col(randAttr1),end_col(randAttr1) )) = rowm(diff,attr1);
		   for (attr2 = 1; attr2<=10; attr2++){		
 	         // repl attr2 in newTask
 	         set_colm(newTask, range( beg_col(randAttr2),end_col(randAttr2) )) = rowm(diff,attr2);  
			 PartialEval(X, efficiency, t, newTask); 
		    }
	     }


	     if (has_lower) {
	      
	        // 3. if has_lower (equal-higher) 
	        attr1 = 0;
            // replace attr1 in newTask
		    set_colm(newTask, range( beg_col(randAttr1),end_col(randAttr1) )) = rowm(diff,attr1);
		    for (attr2 = 11; attr2<=20; attr2++){		
 	         // repl attr2 in newTask
 	         set_colm(newTask, range( beg_col(randAttr2),end_col(randAttr2) )) = rowm(diff,attr2); 
			 PartialEval(X, efficiency, t, newTask); 
		    }
		    
	        // 4. if has_lower (higher-equal) 
	        attr2 = 0;
            // replace attr1 in newTask
		    set_colm(newTask, range( beg_col(randAttr2),end_col(randAttr2) )) = rowm(diff,attr2);
		    for (attr1 = 11; attr1<=20; attr1++){		
 	         // repl attr1 in newTask
 	         set_colm(newTask, range( beg_col(randAttr1),end_col(randAttr1) )) = rowm(diff,attr1);
			 PartialEval(X, efficiency, t, newTask); 
		    }    
            
            // 5. if has_lower (higher-higher) 
		    for (attr1 = 11; attr1<=20; attr1++){
              // replace attr1 in newTask
		      set_colm(newTask, range( beg_col(randAttr1),end_col(randAttr1) )) = rowm(diff,attr1);
		      for (attr2 = 11; attr2<=20; attr2++){		
 	            // repl attr2 in newTask
 	            set_colm(newTask, range( beg_col(randAttr2),end_col(randAttr2) )) = rowm(diff,attr2); 
			    PartialEval(X, efficiency, t, newTask); 
		       }
	        }
         }
           
           
	     if (has_higher) {
	      
	        // 6. if has_lower (equal-lower) 
	        attr1 = 0;
            // replace attr1 in newTask
		    set_colm(newTask, range( beg_col(randAttr1),end_col(randAttr1) )) = rowm(diff,attr1);
		    for (attr2 = 1; attr2<=10; attr2++){		
 	         // repl attr2 in newTask
 	         set_colm(newTask, range( beg_col(randAttr2),end_col(randAttr2) )) = rowm(diff,attr2); 
			 PartialEval(X, efficiency, t, newTask); 
		    }
		    
	        // 7. if has_lower (lower-equal) 
	        attr2 = 0;
            // replace attr1 in newTask
		    set_colm(newTask, range( beg_col(randAttr2),end_col(randAttr2) )) = rowm(diff,attr2);
		    for (attr1 = 1; attr1<=10; attr1++){		
 	         // repl attr1 in newTask
 	         set_colm(newTask, range( beg_col(randAttr1),end_col(randAttr1) )) = rowm(diff,attr1);
			 PartialEval(X, efficiency, t, newTask); 
		    }    
            
            // 8. if has_lower (lower-lower) 
		    for (attr1 = 1; attr1<=10; attr1++){
              // replace attr1 in newTask
		      set_colm(newTask, range( beg_col(randAttr1),end_col(randAttr1) )) = rowm(diff,attr1);
		      for (attr2 = 1; attr2<=10; attr2++){		
 	            // repl attr2 in newTask
 	            set_colm(newTask, range( beg_col(randAttr2),end_col(randAttr2) )) = rowm(diff,attr2); 
			    PartialEval(X, efficiency, t, newTask); 
		       }
	        }
         }
 	 
 	 
 	     if (has_lower && has_higher) {
 	         attr1 = 0;
            // replace attr1 in newTask
		    set_colm(newTask, range( beg_col(randAttr1),end_col(randAttr1) )) = rowm(diff,attr1);
		    attr2 = 0;		
 	        // repl attr2 in newTask
 	        set_colm(newTask, range( beg_col(randAttr2),end_col(randAttr2) )) = rowm(diff,attr2);
			PartialEval(X, efficiency, t, newTask); 
		 }  
 	 
     // E. Update X_sign
     // update X_sign for attribute #1
     X_sign(t,randAttr1)=0;
     for (int i = end_col(randAttr1); i>=beg_col(randAttr1);i--){
       if (X(t,i)==-1){
         X_sign(t,randAttr1)=-1;
         break;   	
       } 
       else if (X(t,i)==1){
         X_sign(t,randAttr1)=1;	
         break;
       } 
     }
     
     // update X_sign for attribute #2
     X_sign(t,randAttr2)=0;
     for (int i = end_col(randAttr2); i>=beg_col(randAttr2);i--){
       if (X(t,i)==-1){
        X_sign(t,randAttr2)=-1;
        break;   	
       } 
       else if (X(t,i)==1){
         X_sign(t,randAttr2)=1;	
         break;
       }      
     }
     

    } // end loop over choice tasks 
    	
    // calculate elapsed time since last succesful update  	
    diffTime = Time::now() - startTime;
    //cout << "Optimized for: " << diffTime.count() << "sec\n";
    
   } while (diffTime.count()<1); // loop until no improvement is found in 1 second
     
   //cout << "Overall optimization took: " << diffTime.count() << "sec\n";
   
   //cout << "D-error (after optimization) = " << efficiency << "\n" <<endl;
   //cout << "X-matrix (after optimization, before KLP sort) = \n" << X << endl; 	
   	
   // write X to file (to compare D-efficiency calculations)
   //std::ofstream output2("X_design.txt"); 
   //for (int i=0;i<n_tasks; i++)
   //{
   //	for (int j=0;j<n_betas;j++)
   //	{
   //		output2 << X(i,j) << " "; 
   //	}
   //	output2 << "\n";
   //}    	
   //output2.close();	
   	
}
   	
   	
   	
void sortKLP(matrix<double> &X, const matrix<double> &X_likelihood, const matrix<double> &prior_mu, const matrix<double> &prior_tau, double ldet, double postLa, int n_observed)   	
{
   	// if there are 2 or more potential future choice tasks:
   	if ((n_tasks-n_observed)>1) {
	      	
   	  // compute KLP for all potential future choice tasks
   	  matrix <double, n_tasks, 1> allKLP;
	  for(int n=0; n<n_observed; n++) allKLP(n)=-9999.0;  // already observed tasks set to -9999.0;
   		  
   	  #pragma omp parallel for num_threads(8) 
   	  for(int n=n_observed; n<n_tasks; n++){
        // the computation of KLP by Laplace approx. is done by the
        // function KLP. See its definition in the header file "KLP.h"
    
        allKLP(n) = KLP(X_likelihood,     // already observed choice tasks 
                        trans(rowm(X,n)), // next choice task (candidate task) 
                        prior_tau,        // prior (precision matrix)
                        ldet,             // log determinant of prior precision matrix
                        prior_mu,         // prior (mean)
                        n_observed,       // number of observed choice tasks
                        n_betas,          // number of parameters
                        postLa);          // posterior normalizing constant, calculated using LA
      }
   
      //cout << "allKLP = \n" << allKLP << endl;
      
      // set task with highest KLP as next choice task
      int optimalKLP = index_of_max(allKLP);
      
      //cout << "index of next choiceTask (with highest KLP) = " << optimalKLP << "\n" << endl;
      
      if (optimalKLP!=n_observed){
       	
       	// make copy
		matrix <double, 1, n_betas> copyTask;
        copyTask = rowm(X,optimalKLP);

        // swap tasks
        //cout << "X (before swap) \n" << X << endl; 
        set_rowm(X,optimalKLP) = rowm(X,n_observed); 
        set_rowm(X,n_observed) = copyTask;
		//cout << "X (after swap) \n" << X << endl;
		
    }}
      	
   //cout << "X-matrix (after KLP sort) = \n" << X << endl; 	
}
	

void X2design(const matrix<double> &X, matrix<double> &design, const matrix<int> &max_levels, matrix<int> &beg_col , matrix<int> &end_col, int n_observed)
{
   for (int t=n_observed; t<n_tasks; t++){ 
    for (int i = 0; i < n_attr; i++){
    		
     int  xValue = max_levels(i); // start with highest attribute value
     bool has_negative = false;
     bool has_positive = false;
     
	 for (int j=end_col(i); j>= beg_col(i); j--){
	  if (X(t,j)==1) {
	   has_positive=true; 	   
	   design(t,i)=xValue;
	   
	   // set X_dummy
	   //X_dummy(t,j)=1.0;
	   //X_dummy(t,j+n_betas)=0.0;
	   
      }
	  else if (X(t,j)==-1){
	   has_negative=true; 	
	   design(t,i+n_attr)=xValue;
	   	      
	   // set X_dummy	   
	   //X_dummy(t,j)=0.0;
	   //X_dummy(t,j+n_betas)=1.0;	
	  } 
	  //else{ 
	  // // set X_dummy
	  // X_dummy(t,j)=0.0;
	  // X_dummy(t,j+n_betas)=0.0;	
	  //}
	 
	  xValue--; 
	 }
	 
	 if (has_positive==false) design(t,i)=1;
     if (has_negative==false) design(t,i+n_attr)=1;	
	 // if both are equal, set A and B to a random integer between 1 and max_levels(i) -> not (yet) implemented
     //if (has_negative==false && has_positive==false){} 
     }
    }

    //cout << "FINAL design = \n" << design << endl; 
}


void cppAdaptive(const string &obsVector, string &futVector, string &betaVector, int n_observed)
{
  // initialize matrices
  matrix <double> b_optim(n_betas, 1);
  matrix <double> design(n_tasks, n_cols);
  
  matrix <int> max_levels(2*n_attr,1);  // maximum # levels per attribute 
  matrix <int> beg_col(n_attr,1); // begin column of attribute in X
  matrix <int> end_col(n_attr,1); // last column of attribute in X   
      
  matrix <double> X(n_tasks, n_betas);
  matrix <double> X_sign(n_tasks, n_attr);
  matrix <double> X_likelihood(n_observed, n_betas);
  
  matrix <double> prior_mu(n_betas, 1);
  matrix <double> prior_tau(n_betas, n_betas);
  matrix <double> chol_covar(n_betas, n_betas);
  matrix <double> LHS(n_draws, n_betas);
  double ldet, postLa;

  // insert inputs
  max_levels = 5;
  str2mat(betaVector, n_betas, 1, b_optim, ",");
  str2mat( (obsVector + futVector), n_tasks, n_cols, design, ""); 
  //cout << "INITIAL design = \n" << design << endl; 
  
  design2X(design, X, X_sign, X_likelihood, n_observed, max_levels, beg_col, end_col);
  definePriors(prior_mu, prior_tau, ldet);
  defineLHS(LHS);
  
  // update design
  calculatePosterior(X, X_likelihood, b_optim, n_observed, prior_mu, prior_tau, ldet, postLa, chol_covar);
  insertDraws(LHS, chol_covar, b_optim); 
  optimizeDesign(X, X_sign, n_observed, beg_col, end_col);              
  sortKLP(X, X_likelihood, prior_mu, prior_tau, ldet, postLa, n_observed);
  
  // return strings
  X2design(X, design, max_levels, beg_col, end_col, n_observed);
  mat2str(b_optim,n_betas, 1, betaVector, ",");
  mat2str(rowm(design,range(n_observed,n_tasks-1)),n_tasks-n_observed, n_cols, futVector, "");
  
}
