package Net::Async::Redis::Commands;

use strict;
use warnings;

our $VERSION = '6.000'; # VERSION

=head1 NAME

Net::Async::Redis::Commands - mixin that defines the Redis commands available

=head1 DESCRIPTION

This is autogenerated from the list of commands available in L<https://redis.io/commands>.

It is intended to be loaded by L<Net::Async::Redis> to provide methods
for each available Redis command.

=cut

=head1 PACKAGE VARIABLES

=head2 KEY_FINDER

This maps the argument index for the C<key> information in each command.

=cut

our %KEY_FINDER = (
    'APPEND' => 1,
    'BITCOUNT' => 1,
    'BITFIELD' => 1,
    'BITFIELD RO' => 2,
    'BITFIELD_RO' => 1,
    'BITOP' => 3,
    'BITPOS' => 1,
    'BLMPOP' => 3,
    'BLPOP' => 1,
    'BRPOP' => 1,
    'BZMPOP' => 3,
    'BZPOPMAX' => 1,
    'BZPOPMIN' => 1,
    'CLUSTER' => 3,
    'CLUSTER KEYSLOT' => 2,
    'DECR' => 1,
    'DECRBY' => 1,
    'DEL' => 1,
    'DUMP' => 1,
    'EVAL' => 3,
    'EVAL RO' => 4,
    'EVAL_RO' => 3,
    'EVALSHA' => 3,
    'EVALSHA RO' => 4,
    'EVALSHA_RO' => 3,
    'EXISTS' => 1,
    'EXPIRE' => 1,
    'EXPIREAT' => 1,
    'EXPIRETIME' => 1,
    'FCALL' => 3,
    'FCALL RO' => 4,
    'FCALL_RO' => 3,
    'GEOADD' => 1,
    'GEODIST' => 1,
    'GEOHASH' => 1,
    'GEOPOS' => 1,
    'GEORADIUS' => 1,
    'GEORADIUS RO' => 2,
    'GEORADIUS_RO' => 1,
    'GEORADIUSBYMEMBER' => 1,
    'GEORADIUSBYMEMBER RO' => 2,
    'GEORADIUSBYMEMBER_RO' => 1,
    'GEOSEARCH' => 1,
    'GET' => 1,
    'GETBIT' => 1,
    'GETDEL' => 1,
    'GETEX' => 1,
    'GETRANGE' => 1,
    'GETSET' => 1,
    'HDEL' => 1,
    'HEXISTS' => 1,
    'HGET' => 1,
    'HGETALL' => 1,
    'HINCRBY' => 1,
    'HINCRBYFLOAT' => 1,
    'HKEYS' => 1,
    'HLEN' => 1,
    'HMGET' => 1,
    'HMSET' => 1,
    'HRANDFIELD' => 1,
    'HSCAN' => 1,
    'HSET' => 1,
    'HSETNX' => 1,
    'HSTRLEN' => 1,
    'HVALS' => 1,
    'INCR' => 1,
    'INCRBY' => 1,
    'INCRBYFLOAT' => 1,
    'LINDEX' => 1,
    'LINSERT' => 1,
    'LLEN' => 1,
    'LMPOP' => 2,
    'LPOP' => 1,
    'LPOS' => 1,
    'LPUSH' => 1,
    'LPUSHX' => 1,
    'LRANGE' => 1,
    'LREM' => 1,
    'LSET' => 1,
    'LTRIM' => 1,
    'MEMORY' => 3,
    'MEMORY USAGE' => 2,
    'MGET' => 1,
    'MIGRATE' => 3,
    'MOVE' => 1,
    'MSET' => 1,
    'MSETNX' => 1,
    'OBJECT' => 3,
    'OBJECT ENCODING' => 2,
    'OBJECT FREQ' => 2,
    'OBJECT IDLETIME' => 2,
    'OBJECT REFCOUNT' => 2,
    'PERSIST' => 1,
    'PEXPIRE' => 1,
    'PEXPIREAT' => 1,
    'PEXPIRETIME' => 1,
    'PFADD' => 1,
    'PFCOUNT' => 1,
    'PSETEX' => 1,
    'PSUBSCRIBE' => 1,
    'PTTL' => 1,
    'PUBLISH' => 1,
    'RENAME' => 1,
    'RENAMENX' => 1,
    'RESTORE' => 1,
    'RPOP' => 1,
    'RPUSH' => 1,
    'RPUSHX' => 1,
    'SADD' => 1,
    'SCARD' => 1,
    'SDIFF' => 1,
    'SDIFFSTORE' => 2,
    'SET' => 1,
    'SETBIT' => 1,
    'SETEX' => 1,
    'SETNX' => 1,
    'SETRANGE' => 1,
    'SINTER' => 1,
    'SINTERCARD' => 2,
    'SINTERSTORE' => 2,
    'SISMEMBER' => 1,
    'SMEMBERS' => 1,
    'SMISMEMBER' => 1,
    'SORT' => 1,
    'SORT RO' => 2,
    'SORT_RO' => 1,
    'SPOP' => 1,
    'SRANDMEMBER' => 1,
    'SREM' => 1,
    'SSCAN' => 1,
    'STRLEN' => 1,
    'SUBSCRIBE' => 1,
    'SUBSTR' => 1,
    'SUNION' => 1,
    'SUNIONSTORE' => 2,
    'TOUCH' => 1,
    'TTL' => 1,
    'TYPE' => 1,
    'UNLINK' => 1,
    'WATCH' => 1,
    'XACK' => 1,
    'XADD' => 1,
    'XAUTOCLAIM' => 1,
    'XCLAIM' => 1,
    'XDEL' => 1,
    'XGROUP' => 2,
    'XGROUP CREATE' => 2,
    'XGROUP CREATECONSUMER' => 2,
    'XGROUP DELCONSUMER' => 2,
    'XGROUP DESTROY' => 2,
    'XGROUP SETID' => 2,
    'XINFO' => 2,
    'XINFO CONSUMERS' => 2,
    'XINFO GROUPS' => 2,
    'XINFO STREAM' => 2,
    'XLEN' => 1,
    'XPENDING' => 1,
    'XRANGE' => 1,
    'XREAD' => 3,
    'XREADGROUP' => 5,
    'XREVRANGE' => 1,
    'XSETID' => 1,
    'XTRIM' => 1,
    'ZADD' => 1,
    'ZCARD' => 1,
    'ZCOUNT' => 1,
    'ZDIFF' => 2,
    'ZDIFFSTORE' => 3,
    'ZINCRBY' => 1,
    'ZINTER' => 2,
    'ZINTERCARD' => 2,
    'ZINTERSTORE' => 3,
    'ZLEXCOUNT' => 1,
    'ZMPOP' => 2,
    'ZMSCORE' => 1,
    'ZPOPMAX' => 1,
    'ZPOPMIN' => 1,
    'ZRANDMEMBER' => 1,
    'ZRANGE' => 1,
    'ZRANGEBYLEX' => 1,
    'ZRANGEBYSCORE' => 1,
    'ZRANK' => 1,
    'ZREM' => 1,
    'ZREMRANGEBYLEX' => 1,
    'ZREMRANGEBYRANK' => 1,
    'ZREMRANGEBYSCORE' => 1,
    'ZREVRANGE' => 1,
    'ZREVRANGEBYLEX' => 1,
    'ZREVRANGEBYSCORE' => 1,
    'ZREVRANK' => 1,
    'ZSCAN' => 1,
    'ZSCORE' => 1,
    'ZUNION' => 2,
    'ZUNIONSTORE' => 3,
);

=head1 METHODS - Bitmap

=head2 bitcount

Count set bits in a string.

=over 4

=item * key

=item * [start end [BYTE|BIT]]

=back

L<https://redis.io/commands/bitcount>

=cut

sub bitcount : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(BITCOUNT) => @args)
}

=head2 bitfield

Perform arbitrary bitfield integer operations on strings.

=over 4

=item * key

=item * [GET encoding offset]

=item * [SET encoding offset value]

=item * [INCRBY encoding offset increment]

=item * [OVERFLOW WRAP|SAT|FAIL]

=back

L<https://redis.io/commands/bitfield>

=cut

sub bitfield : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(BITFIELD) => @args)
}

=head2 bitfield_ro

Perform arbitrary bitfield integer operations on strings. Read-only variant of BITFIELD.

=over 4

=item * key

=item * GET encoding offset

=back

L<https://redis.io/commands/bitfield-ro>

=cut

sub bitfield_ro : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(BITFIELD_RO) => @args)
}

=head2 bitop

Perform bitwise operations between strings.

=over 4

=item * operation

=item * destkey

=item * key [key ...]

=back

L<https://redis.io/commands/bitop>

=cut

sub bitop : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(BITOP) => @args)
}

=head2 bitpos

Find first bit set or clear in a string.

=over 4

=item * key

=item * bit

=item * [start [end [BYTE|BIT]]]

=back

L<https://redis.io/commands/bitpos>

=cut

sub bitpos : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(BITPOS) => @args)
}

=head2 getbit

Returns the bit value at offset in the string value stored at key.

=over 4

=item * key

=item * offset

=back

L<https://redis.io/commands/getbit>

=cut

sub getbit : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(GETBIT) => @args)
}

=head2 setbit

Sets or clears the bit at offset in the string value stored at key.

=over 4

=item * key

=item * offset

=item * value

=back

L<https://redis.io/commands/setbit>

=cut

sub setbit : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SETBIT) => @args)
}

=head1 METHODS - Cluster

=head2 asking

Sent by cluster clients after an -ASK redirect.

L<https://redis.io/commands/asking>

=cut

sub asking : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ASKING) => @args)
}

=head2 cluster_addslots

Assign new hash slots to receiving node.

=over 4

=item * slot [slot ...]

=back

L<https://redis.io/commands/cluster-addslots>

=cut

sub cluster_addslots : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLUSTER ADDSLOTS) => @args)
}

=head2 cluster_addslotsrange

Assign new hash slots to receiving node.

=over 4

=item * start-slot end-slot [start-slot end-slot ...]

=back

L<https://redis.io/commands/cluster-addslotsrange>

=cut

sub cluster_addslotsrange : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLUSTER ADDSLOTSRANGE) => @args)
}

=head2 cluster_bumpepoch

Advance the cluster config epoch.

L<https://redis.io/commands/cluster-bumpepoch>

=cut

sub cluster_bumpepoch : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLUSTER BUMPEPOCH) => @args)
}

=head2 cluster_count_failure_reports

Return the number of failure reports active for a given node.

=over 4

=item * node-id

=back

L<https://redis.io/commands/cluster-count-failure-reports>

=cut

sub cluster_count_failure_reports : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLUSTER COUNT-FAILURE-REPORTS) => @args)
}

=head2 cluster_countkeysinslot

Return the number of local keys in the specified hash slot.

=over 4

=item * slot

=back

L<https://redis.io/commands/cluster-countkeysinslot>

=cut

sub cluster_countkeysinslot : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLUSTER COUNTKEYSINSLOT) => @args)
}

=head2 cluster_delslots

Set hash slots as unbound in receiving node.

=over 4

=item * slot [slot ...]

=back

L<https://redis.io/commands/cluster-delslots>

=cut

sub cluster_delslots : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLUSTER DELSLOTS) => @args)
}

=head2 cluster_delslotsrange

Set hash slots as unbound in receiving node.

=over 4

=item * start-slot end-slot [start-slot end-slot ...]

=back

L<https://redis.io/commands/cluster-delslotsrange>

=cut

sub cluster_delslotsrange : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLUSTER DELSLOTSRANGE) => @args)
}

=head2 cluster_failover

Forces a replica to perform a manual failover of its master.

=over 4

=item * [FORCE|TAKEOVER]

=back

L<https://redis.io/commands/cluster-failover>

=cut

sub cluster_failover : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLUSTER FAILOVER) => @args)
}

=head2 cluster_flushslots

Delete a node's own slots information.

L<https://redis.io/commands/cluster-flushslots>

=cut

sub cluster_flushslots : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLUSTER FLUSHSLOTS) => @args)
}

=head2 cluster_forget

Remove a node from the nodes table.

=over 4

=item * node-id

=back

L<https://redis.io/commands/cluster-forget>

=cut

sub cluster_forget : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLUSTER FORGET) => @args)
}

=head2 cluster_getkeysinslot

Return local key names in the specified hash slot.

=over 4

=item * slot

=item * count

=back

L<https://redis.io/commands/cluster-getkeysinslot>

=cut

sub cluster_getkeysinslot : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLUSTER GETKEYSINSLOT) => @args)
}

=head2 cluster_info

Provides info about Redis Cluster node state.

L<https://redis.io/commands/cluster-info>

=cut

sub cluster_info : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLUSTER INFO) => @args)
}

=head2 cluster_keyslot

Returns the hash slot of the specified key.

=over 4

=item * key

=back

L<https://redis.io/commands/cluster-keyslot>

=cut

sub cluster_keyslot : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLUSTER KEYSLOT) => @args)
}

=head2 cluster_links

Returns a list of all TCP links to and from peer nodes in cluster.

L<https://redis.io/commands/cluster-links>

=cut

sub cluster_links : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLUSTER LINKS) => @args)
}

=head2 cluster_meet

Force a node cluster to handshake with another node.

=over 4

=item * ip

=item * port

=back

L<https://redis.io/commands/cluster-meet>

=cut

sub cluster_meet : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLUSTER MEET) => @args)
}

=head2 cluster_myid

Return the node id.

L<https://redis.io/commands/cluster-myid>

=cut

sub cluster_myid : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLUSTER MYID) => @args)
}

=head2 cluster_nodes

Get Cluster config for the node.

L<https://redis.io/commands/cluster-nodes>

=cut

sub cluster_nodes : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLUSTER NODES) => @args)
}

=head2 cluster_replicas

List replica nodes of the specified master node.

=over 4

=item * node-id

=back

L<https://redis.io/commands/cluster-replicas>

=cut

sub cluster_replicas : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLUSTER REPLICAS) => @args)
}

=head2 cluster_replicate

Reconfigure a node as a replica of the specified master node.

=over 4

=item * node-id

=back

L<https://redis.io/commands/cluster-replicate>

=cut

sub cluster_replicate : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLUSTER REPLICATE) => @args)
}

=head2 cluster_reset

Reset a Redis Cluster node.

=over 4

=item * [HARD|SOFT]

=back

L<https://redis.io/commands/cluster-reset>

=cut

sub cluster_reset : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLUSTER RESET) => @args)
}

=head2 cluster_saveconfig

Forces the node to save cluster state on disk.

L<https://redis.io/commands/cluster-saveconfig>

=cut

sub cluster_saveconfig : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLUSTER SAVECONFIG) => @args)
}

=head2 cluster_set_config_epoch

Set the configuration epoch in a new node.

=over 4

=item * config-epoch

=back

L<https://redis.io/commands/cluster-set-config-epoch>

=cut

sub cluster_set_config_epoch : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLUSTER SET-CONFIG-EPOCH) => @args)
}

=head2 cluster_setslot

Bind a hash slot to a specific node.

=over 4

=item * slot

=item * IMPORTING node-id|MIGRATING node-id|NODE node-id|STABLE

=back

L<https://redis.io/commands/cluster-setslot>

=cut

sub cluster_setslot : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLUSTER SETSLOT) => @args)
}

=head2 cluster_slaves

List replica nodes of the specified master node.

=over 4

=item * node-id

=back

L<https://redis.io/commands/cluster-slaves>

=cut

sub cluster_slaves : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLUSTER SLAVES) => @args)
}

=head2 cluster_slots

Get array of Cluster slot to node mappings.

L<https://redis.io/commands/cluster-slots>

=cut

sub cluster_slots : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLUSTER SLOTS) => @args)
}

=head2 readonly

Enables read queries for a connection to a cluster replica node.

L<https://redis.io/commands/readonly>

=cut

sub readonly : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(READONLY) => @args)
}

=head2 readwrite

Disables read queries for a connection to a cluster replica node.

L<https://redis.io/commands/readwrite>

=cut

sub readwrite : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(READWRITE) => @args)
}

=head1 METHODS - Connection

=head2 auth

Authenticate to the server.

=over 4

=item * [username]

=item * password

=back

L<https://redis.io/commands/auth>

=cut

sub auth : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(AUTH) => @args)
}

=head2 client_caching

Instruct the server about tracking or not keys in the next request.

=over 4

=item * YES|NO

=back

L<https://redis.io/commands/client-caching>

=cut

sub client_caching : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLIENT CACHING) => @args)
}

=head2 client_getname

Get the current connection name.

L<https://redis.io/commands/client-getname>

=cut

sub client_getname : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLIENT GETNAME) => @args)
}

=head2 client_getredir

Get tracking notifications redirection client ID if any.

L<https://redis.io/commands/client-getredir>

=cut

sub client_getredir : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLIENT GETREDIR) => @args)
}

=head2 client_id

Returns the client ID for the current connection.

L<https://redis.io/commands/client-id>

=cut

sub client_id : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLIENT ID) => @args)
}

=head2 client_info

Returns information about the current client connection.

L<https://redis.io/commands/client-info>

=cut

sub client_info : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLIENT INFO) => @args)
}

=head2 client_kill

Kill the connection of a client.

=over 4

=item * [ip:port]

=item * [ID client-id]

=item * [TYPE NORMAL|MASTER|SLAVE|REPLICA|PUBSUB]

=item * [USER username]

=item * [ADDR ip:port]

=item * [LADDR ip:port]

=item * [SKIPME yes/no]

=back

L<https://redis.io/commands/client-kill>

=cut

sub client_kill : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLIENT KILL) => @args)
}

=head2 client_list

Get the list of client connections.

=over 4

=item * [TYPE NORMAL|MASTER|REPLICA|PUBSUB]

=item * [ID client-id [client-id ...]]

=back

L<https://redis.io/commands/client-list>

=cut

sub client_list : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLIENT LIST) => @args)
}

=head2 client_no_evict

Set client eviction mode for the current connection.

=over 4

=item * ON|OFF

=back

L<https://redis.io/commands/client-no-evict>

=cut

sub client_no_evict : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLIENT NO-EVICT) => @args)
}

=head2 client_pause

Stop processing commands from clients for some time.

=over 4

=item * timeout

=item * [WRITE|ALL]

=back

L<https://redis.io/commands/client-pause>

=cut

sub client_pause : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLIENT PAUSE) => @args)
}

=head2 client_reply

Instruct the server whether to reply to commands.

=over 4

=item * ON|OFF|SKIP

=back

L<https://redis.io/commands/client-reply>

=cut

sub client_reply : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLIENT REPLY) => @args)
}

=head2 client_setname

Set the current connection name.

=over 4

=item * connection-name

=back

L<https://redis.io/commands/client-setname>

=cut

sub client_setname : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLIENT SETNAME) => @args)
}

=head2 client_tracking

Enable or disable server assisted client side caching support.

=over 4

=item * ON|OFF

=item * [REDIRECT client-id]

=item * [PREFIX prefix [PREFIX prefix ...]]

=item * [BCAST]

=item * [OPTIN]

=item * [OPTOUT]

=item * [NOLOOP]

=back

L<https://redis.io/commands/client-tracking>

=cut

sub client_tracking : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLIENT TRACKING) => @args)
}

=head2 client_trackinginfo

Return information about server assisted client side caching for the current connection.

L<https://redis.io/commands/client-trackinginfo>

=cut

sub client_trackinginfo : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLIENT TRACKINGINFO) => @args)
}

=head2 client_unblock

Unblock a client blocked in a blocking command from a different connection.

=over 4

=item * client-id

=item * [TIMEOUT|ERROR]

=back

L<https://redis.io/commands/client-unblock>

=cut

sub client_unblock : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLIENT UNBLOCK) => @args)
}

=head2 client_unpause

Resume processing of clients that were paused.

L<https://redis.io/commands/client-unpause>

=cut

sub client_unpause : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CLIENT UNPAUSE) => @args)
}

=head2 echo

Echo the given string.

=over 4

=item * message

=back

L<https://redis.io/commands/echo>

=cut

sub echo : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ECHO) => @args)
}

=head2 hello

Handshake with Redis.

=over 4

=item * [protover [AUTH username password] [SETNAME clientname]]

=back

L<https://redis.io/commands/hello>

=cut

sub hello : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(HELLO) => @args)
}

=head2 ping

Ping the server.

=over 4

=item * [message]

=back

L<https://redis.io/commands/ping>

=cut

sub ping : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(PING) => @args)
}

=head2 quit

Close the connection.

L<https://redis.io/commands/quit>

=cut

sub quit : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(QUIT) => @args)
}

=head2 reset

Reset the connection.

L<https://redis.io/commands/reset>

=cut

sub reset : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(RESET) => @args)
}

=head2 select

Change the selected database for the current connection.

=over 4

=item * index

=back

L<https://redis.io/commands/select>

=cut

sub select : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SELECT) => @args)
}

=head1 METHODS - Generic

=head2 copy

Copy a key.

=over 4

=item * source

=item * destination

=item * [DB destination-db]

=item * [REPLACE]

=back

L<https://redis.io/commands/copy>

=cut

sub copy : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(COPY) => @args)
}

=head2 del

Delete a key.

=over 4

=item * key [key ...]

=back

L<https://redis.io/commands/del>

=cut

sub del : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(DEL) => @args)
}

=head2 dump

Return a serialized version of the value stored at the specified key.

=over 4

=item * key

=back

L<https://redis.io/commands/dump>

=cut

sub dump : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(DUMP) => @args)
}

=head2 exists

Determine if a key exists.

=over 4

=item * key [key ...]

=back

L<https://redis.io/commands/exists>

=cut

sub exists : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(EXISTS) => @args)
}

=head2 expire

Set a key's time to live in seconds.

=over 4

=item * key

=item * seconds

=item * [NX|XX|GT|LT]

=back

L<https://redis.io/commands/expire>

=cut

sub expire : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(EXPIRE) => @args)
}

=head2 expireat

Set the expiration for a key as a UNIX timestamp.

=over 4

=item * key

=item * unix-time-seconds

=item * [NX|XX|GT|LT]

=back

L<https://redis.io/commands/expireat>

=cut

sub expireat : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(EXPIREAT) => @args)
}

=head2 expiretime

Get the expiration Unix timestamp for a key.

=over 4

=item * key

=back

L<https://redis.io/commands/expiretime>

=cut

sub expiretime : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(EXPIRETIME) => @args)
}

=head2 keys

Find all keys matching the given pattern.

=over 4

=item * pattern

=back

L<https://redis.io/commands/keys>

=cut

sub keys : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(KEYS) => @args)
}

=head2 migrate

Atomically transfer a key from a Redis instance to another one.

=over 4

=item * host

=item * port

=item * key|""

=item * destination-db

=item * timeout

=item * [COPY]

=item * [REPLACE]

=item * [AUTH password]

=item * [AUTH2 username password]

=item * [KEYS key [key ...]]

=back

L<https://redis.io/commands/migrate>

=cut

sub migrate : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(MIGRATE) => @args)
}

=head2 move

Move a key to another database.

=over 4

=item * key

=item * db

=back

L<https://redis.io/commands/move>

=cut

sub move : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(MOVE) => @args)
}

=head2 object_encoding

Inspect the internal encoding of a Redis object.

=over 4

=item * key

=back

L<https://redis.io/commands/object-encoding>

=cut

sub object_encoding : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(OBJECT ENCODING) => @args)
}

=head2 object_freq

Get the logarithmic access frequency counter of a Redis object.

=over 4

=item * key

=back

L<https://redis.io/commands/object-freq>

=cut

sub object_freq : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(OBJECT FREQ) => @args)
}

=head2 object_idletime

Get the time since a Redis object was last accessed.

=over 4

=item * key

=back

L<https://redis.io/commands/object-idletime>

=cut

sub object_idletime : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(OBJECT IDLETIME) => @args)
}

=head2 object_refcount

Get the number of references to the value of the key.

=over 4

=item * key

=back

L<https://redis.io/commands/object-refcount>

=cut

sub object_refcount : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(OBJECT REFCOUNT) => @args)
}

=head2 persist

Remove the expiration from a key.

=over 4

=item * key

=back

L<https://redis.io/commands/persist>

=cut

sub persist : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(PERSIST) => @args)
}

=head2 pexpire

Set a key's time to live in milliseconds.

=over 4

=item * key

=item * milliseconds

=item * [NX|XX|GT|LT]

=back

L<https://redis.io/commands/pexpire>

=cut

sub pexpire : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(PEXPIRE) => @args)
}

=head2 pexpireat

Set the expiration for a key as a UNIX timestamp specified in milliseconds.

=over 4

=item * key

=item * unix-time-milliseconds

=item * [NX|XX|GT|LT]

=back

L<https://redis.io/commands/pexpireat>

=cut

sub pexpireat : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(PEXPIREAT) => @args)
}

=head2 pexpiretime

Get the expiration Unix timestamp for a key in milliseconds.

=over 4

=item * key

=back

L<https://redis.io/commands/pexpiretime>

=cut

sub pexpiretime : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(PEXPIRETIME) => @args)
}

=head2 pttl

Get the time to live for a key in milliseconds.

=over 4

=item * key

=back

L<https://redis.io/commands/pttl>

=cut

sub pttl : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(PTTL) => @args)
}

=head2 randomkey

Return a random key from the keyspace.

L<https://redis.io/commands/randomkey>

=cut

sub randomkey : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(RANDOMKEY) => @args)
}

=head2 rename

Rename a key.

=over 4

=item * key

=item * newkey

=back

L<https://redis.io/commands/rename>

=cut

sub rename : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(RENAME) => @args)
}

=head2 renamenx

Rename a key, only if the new key does not exist.

=over 4

=item * key

=item * newkey

=back

L<https://redis.io/commands/renamenx>

=cut

sub renamenx : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(RENAMENX) => @args)
}

=head2 restore

Create a key using the provided serialized value, previously obtained using DUMP.

=over 4

=item * key

=item * ttl

=item * serialized-value

=item * [REPLACE]

=item * [ABSTTL]

=item * [IDLETIME seconds]

=item * [FREQ frequency]

=back

L<https://redis.io/commands/restore>

=cut

sub restore : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(RESTORE) => @args)
}

=head2 scan

Incrementally iterate the keys space.

=over 4

=item * cursor

=item * [MATCH pattern]

=item * [COUNT count]

=item * [TYPE type]

=back

L<https://redis.io/commands/scan>

=cut

sub scan : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SCAN) => @args)
}

=head2 sort

Sort the elements in a list, set or sorted set.

=over 4

=item * key

=item * [BY pattern]

=item * [LIMIT offset count]

=item * [GET pattern [GET pattern ...]]

=item * [ASC|DESC]

=item * [ALPHA]

=item * [STORE destination]

=back

L<https://redis.io/commands/sort>

=cut

sub sort : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SORT) => @args)
}

=head2 sort_ro

Sort the elements in a list, set or sorted set. Read-only variant of SORT.

=over 4

=item * key

=item * [BY pattern]

=item * [LIMIT offset count]

=item * [GET pattern [GET pattern ...]]

=item * [ASC|DESC]

=item * [ALPHA]

=back

L<https://redis.io/commands/sort-ro>

=cut

sub sort_ro : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SORT_RO) => @args)
}

=head2 touch

Alters the last access time of a key(s). Returns the number of existing keys specified.

=over 4

=item * key [key ...]

=back

L<https://redis.io/commands/touch>

=cut

sub touch : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(TOUCH) => @args)
}

=head2 ttl

Get the time to live for a key in seconds.

=over 4

=item * key

=back

L<https://redis.io/commands/ttl>

=cut

sub ttl : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(TTL) => @args)
}

=head2 type

Determine the type stored at key.

=over 4

=item * key

=back

L<https://redis.io/commands/type>

=cut

sub type : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(TYPE) => @args)
}

=head2 unlink

Delete a key asynchronously in another thread. Otherwise it is just as DEL, but non blocking.

=over 4

=item * key [key ...]

=back

L<https://redis.io/commands/unlink>

=cut

sub unlink : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(UNLINK) => @args)
}

=head2 wait

Wait for the synchronous replication of all the write commands sent in the context of the current connection.

=over 4

=item * numreplicas

=item * timeout

=back

L<https://redis.io/commands/wait>

=cut

sub wait : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(WAIT) => @args)
}

=head1 METHODS - Geo

=head2 geoadd

Add one or more geospatial items in the geospatial index represented using a sorted set.

=over 4

=item * key

=item * [NX|XX]

=item * [CH]

=item * longitude latitude member [longitude latitude member ...]

=back

L<https://redis.io/commands/geoadd>

=cut

sub geoadd : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(GEOADD) => @args)
}

=head2 geodist

Returns the distance between two members of a geospatial index.

=over 4

=item * key

=item * member1

=item * member2

=item * [M|KM|FT|MI]

=back

L<https://redis.io/commands/geodist>

=cut

sub geodist : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(GEODIST) => @args)
}

=head2 geohash

Returns members of a geospatial index as standard geohash strings.

=over 4

=item * key

=item * member [member ...]

=back

L<https://redis.io/commands/geohash>

=cut

sub geohash : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(GEOHASH) => @args)
}

=head2 geopos

Returns longitude and latitude of members of a geospatial index.

=over 4

=item * key

=item * member [member ...]

=back

L<https://redis.io/commands/geopos>

=cut

sub geopos : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(GEOPOS) => @args)
}

=head2 georadius

Query a sorted set representing a geospatial index to fetch members matching a given maximum distance from a point.

=over 4

=item * key

=item * longitude

=item * latitude

=item * radius

=item * M|KM|FT|MI

=item * [WITHCOORD]

=item * [WITHDIST]

=item * [WITHHASH]

=item * [COUNT count [ANY]]

=item * [ASC|DESC]

=item * [STORE key]

=item * [STOREDIST key]

=back

L<https://redis.io/commands/georadius>

=cut

sub georadius : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(GEORADIUS) => @args)
}

=head2 georadius_ro

A read-only variant for GEORADIUS.

=over 4

=item * key

=item * longitude

=item * latitude

=item * radius

=item * M|KM|FT|MI

=item * [WITHCOORD]

=item * [WITHDIST]

=item * [WITHHASH]

=item * [COUNT count [ANY]]

=item * [ASC|DESC]

=back

L<https://redis.io/commands/georadius-ro>

=cut

sub georadius_ro : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(GEORADIUS_RO) => @args)
}

=head2 georadiusbymember

Query a sorted set representing a geospatial index to fetch members matching a given maximum distance from a member.

=over 4

=item * key

=item * member

=item * radius

=item * M|KM|FT|MI

=item * [WITHCOORD]

=item * [WITHDIST]

=item * [WITHHASH]

=item * [COUNT count [ANY]]

=item * [ASC|DESC]

=item * [STORE key]

=item * [STOREDIST key]

=back

L<https://redis.io/commands/georadiusbymember>

=cut

sub georadiusbymember : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(GEORADIUSBYMEMBER) => @args)
}

=head2 georadiusbymember_ro

A read-only variant for GEORADIUSBYMEMBER.

=over 4

=item * key

=item * member

=item * radius

=item * M|KM|FT|MI

=item * [WITHCOORD]

=item * [WITHDIST]

=item * [WITHHASH]

=item * [COUNT count [ANY]]

=item * [ASC|DESC]

=back

L<https://redis.io/commands/georadiusbymember-ro>

=cut

sub georadiusbymember_ro : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(GEORADIUSBYMEMBER_RO) => @args)
}

=head2 geosearch

Query a sorted set representing a geospatial index to fetch members inside an area of a box or a circle.

=over 4

=item * key

=item * [FROMMEMBER member]

=item * [FROMLONLAT longitude latitude]

=item * [BYRADIUS radius M|KM|FT|MI]

=item * [BYBOX width height M|KM|FT|MI]

=item * [ASC|DESC]

=item * [COUNT count [ANY]]

=item * [WITHCOORD]

=item * [WITHDIST]

=item * [WITHHASH]

=back

L<https://redis.io/commands/geosearch>

=cut

sub geosearch : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(GEOSEARCH) => @args)
}

=head2 geosearchstore

Query a sorted set representing a geospatial index to fetch members inside an area of a box or a circle, and store the result in another key.

=over 4

=item * destination

=item * source

=item * [FROMMEMBER member]

=item * [FROMLONLAT longitude latitude]

=item * [BYRADIUS radius M|KM|FT|MI]

=item * [BYBOX width height M|KM|FT|MI]

=item * [ASC|DESC]

=item * [COUNT count [ANY]]

=item * [STOREDIST]

=back

L<https://redis.io/commands/geosearchstore>

=cut

sub geosearchstore : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(GEOSEARCHSTORE) => @args)
}

=head1 METHODS - Hash

=head2 hdel

Delete one or more hash fields.

=over 4

=item * key

=item * field [field ...]

=back

L<https://redis.io/commands/hdel>

=cut

sub hdel : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(HDEL) => @args)
}

=head2 hexists

Determine if a hash field exists.

=over 4

=item * key

=item * field

=back

L<https://redis.io/commands/hexists>

=cut

sub hexists : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(HEXISTS) => @args)
}

=head2 hget

Get the value of a hash field.

=over 4

=item * key

=item * field

=back

L<https://redis.io/commands/hget>

=cut

sub hget : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(HGET) => @args)
}

=head2 hgetall

Get all the fields and values in a hash.

=over 4

=item * key

=back

L<https://redis.io/commands/hgetall>

=cut

sub hgetall : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(HGETALL) => @args)
}

=head2 hincrby

Increment the integer value of a hash field by the given number.

=over 4

=item * key

=item * field

=item * increment

=back

L<https://redis.io/commands/hincrby>

=cut

sub hincrby : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(HINCRBY) => @args)
}

=head2 hincrbyfloat

Increment the float value of a hash field by the given amount.

=over 4

=item * key

=item * field

=item * increment

=back

L<https://redis.io/commands/hincrbyfloat>

=cut

sub hincrbyfloat : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(HINCRBYFLOAT) => @args)
}

=head2 hkeys

Get all the fields in a hash.

=over 4

=item * key

=back

L<https://redis.io/commands/hkeys>

=cut

sub hkeys : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(HKEYS) => @args)
}

=head2 hlen

Get the number of fields in a hash.

=over 4

=item * key

=back

L<https://redis.io/commands/hlen>

=cut

sub hlen : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(HLEN) => @args)
}

=head2 hmget

Get the values of all the given hash fields.

=over 4

=item * key

=item * field [field ...]

=back

L<https://redis.io/commands/hmget>

=cut

sub hmget : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(HMGET) => @args)
}

=head2 hmset

Set multiple hash fields to multiple values.

=over 4

=item * key

=item * field value [field value ...]

=back

L<https://redis.io/commands/hmset>

=cut

sub hmset : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(HMSET) => @args)
}

=head2 hrandfield

Get one or multiple random fields from a hash.

=over 4

=item * key

=item * [count [WITHVALUES]]

=back

L<https://redis.io/commands/hrandfield>

=cut

sub hrandfield : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(HRANDFIELD) => @args)
}

=head2 hscan

Incrementally iterate hash fields and associated values.

=over 4

=item * key

=item * cursor

=item * [MATCH pattern]

=item * [COUNT count]

=back

L<https://redis.io/commands/hscan>

=cut

sub hscan : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(HSCAN) => @args)
}

=head2 hset

Set the string value of a hash field.

=over 4

=item * key

=item * field value [field value ...]

=back

L<https://redis.io/commands/hset>

=cut

sub hset : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(HSET) => @args)
}

=head2 hsetnx

Set the value of a hash field, only if the field does not exist.

=over 4

=item * key

=item * field

=item * value

=back

L<https://redis.io/commands/hsetnx>

=cut

sub hsetnx : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(HSETNX) => @args)
}

=head2 hstrlen

Get the length of the value of a hash field.

=over 4

=item * key

=item * field

=back

L<https://redis.io/commands/hstrlen>

=cut

sub hstrlen : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(HSTRLEN) => @args)
}

=head2 hvals

Get all the values in a hash.

=over 4

=item * key

=back

L<https://redis.io/commands/hvals>

=cut

sub hvals : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(HVALS) => @args)
}

=head1 METHODS - Hyperloglog

=head2 pfadd

Adds the specified elements to the specified HyperLogLog.

=over 4

=item * key

=item * [element [element ...]]

=back

L<https://redis.io/commands/pfadd>

=cut

sub pfadd : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(PFADD) => @args)
}

=head2 pfcount

Return the approximated cardinality of the set(s) observed by the HyperLogLog at key(s).

=over 4

=item * key [key ...]

=back

L<https://redis.io/commands/pfcount>

=cut

sub pfcount : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(PFCOUNT) => @args)
}

=head2 pfdebug

Internal commands for debugging HyperLogLog values.

L<https://redis.io/commands/pfdebug>

=cut

sub pfdebug : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(PFDEBUG) => @args)
}

=head2 pfmerge

Merge N different HyperLogLogs into a single one.

=over 4

=item * destkey

=item * sourcekey [sourcekey ...]

=back

L<https://redis.io/commands/pfmerge>

=cut

sub pfmerge : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(PFMERGE) => @args)
}

=head2 pfselftest

An internal command for testing HyperLogLog values.

L<https://redis.io/commands/pfselftest>

=cut

sub pfselftest : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(PFSELFTEST) => @args)
}

=head1 METHODS - List

=head2 blmove

Pop an element from a list, push it to another list and return it; or block until one is available.

=over 4

=item * source

=item * destination

=item * LEFT|RIGHT

=item * LEFT|RIGHT

=item * timeout

=back

L<https://redis.io/commands/blmove>

=cut

sub blmove : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(BLMOVE) => @args)
}

=head2 blmpop

Pop elements from a list, or block until one is available.

=over 4

=item * timeout

=item * numkeys

=item * key [key ...]

=item * LEFT|RIGHT

=item * [COUNT count]

=back

L<https://redis.io/commands/blmpop>

=cut

sub blmpop : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(BLMPOP) => @args)
}

=head2 blpop

Remove and get the first element in a list, or block until one is available.

=over 4

=item * key [key ...]

=item * timeout

=back

L<https://redis.io/commands/blpop>

=cut

sub blpop : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(BLPOP) => @args)
}

=head2 brpop

Remove and get the last element in a list, or block until one is available.

=over 4

=item * key [key ...]

=item * timeout

=back

L<https://redis.io/commands/brpop>

=cut

sub brpop : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(BRPOP) => @args)
}

=head2 brpoplpush

Pop an element from a list, push it to another list and return it; or block until one is available.

=over 4

=item * source

=item * destination

=item * timeout

=back

L<https://redis.io/commands/brpoplpush>

=cut

sub brpoplpush : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(BRPOPLPUSH) => @args)
}

=head2 lindex

Get an element from a list by its index.

=over 4

=item * key

=item * index

=back

L<https://redis.io/commands/lindex>

=cut

sub lindex : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(LINDEX) => @args)
}

=head2 linsert

Insert an element before or after another element in a list.

=over 4

=item * key

=item * BEFORE|AFTER

=item * pivot

=item * element

=back

L<https://redis.io/commands/linsert>

=cut

sub linsert : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(LINSERT) => @args)
}

=head2 llen

Get the length of a list.

=over 4

=item * key

=back

L<https://redis.io/commands/llen>

=cut

sub llen : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(LLEN) => @args)
}

=head2 lmove

Pop an element from a list, push it to another list and return it.

=over 4

=item * source

=item * destination

=item * LEFT|RIGHT

=item * LEFT|RIGHT

=back

L<https://redis.io/commands/lmove>

=cut

sub lmove : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(LMOVE) => @args)
}

=head2 lmpop

Pop elements from a list.

=over 4

=item * numkeys

=item * key [key ...]

=item * LEFT|RIGHT

=item * [COUNT count]

=back

L<https://redis.io/commands/lmpop>

=cut

sub lmpop : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(LMPOP) => @args)
}

=head2 lpop

Remove and get the first elements in a list.

=over 4

=item * key

=item * [count]

=back

L<https://redis.io/commands/lpop>

=cut

sub lpop : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(LPOP) => @args)
}

=head2 lpos

Return the index of matching elements on a list.

=over 4

=item * key

=item * element

=item * [RANK rank]

=item * [COUNT num-matches]

=item * [MAXLEN len]

=back

L<https://redis.io/commands/lpos>

=cut

sub lpos : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(LPOS) => @args)
}

=head2 lpush

Prepend one or multiple elements to a list.

=over 4

=item * key

=item * element [element ...]

=back

L<https://redis.io/commands/lpush>

=cut

sub lpush : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(LPUSH) => @args)
}

=head2 lpushx

Prepend an element to a list, only if the list exists.

=over 4

=item * key

=item * element [element ...]

=back

L<https://redis.io/commands/lpushx>

=cut

sub lpushx : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(LPUSHX) => @args)
}

=head2 lrange

Get a range of elements from a list.

=over 4

=item * key

=item * start

=item * stop

=back

L<https://redis.io/commands/lrange>

=cut

sub lrange : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(LRANGE) => @args)
}

=head2 lrem

Remove elements from a list.

=over 4

=item * key

=item * count

=item * element

=back

L<https://redis.io/commands/lrem>

=cut

sub lrem : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(LREM) => @args)
}

=head2 lset

Set the value of an element in a list by its index.

=over 4

=item * key

=item * index

=item * element

=back

L<https://redis.io/commands/lset>

=cut

sub lset : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(LSET) => @args)
}

=head2 ltrim

Trim a list to the specified range.

=over 4

=item * key

=item * start

=item * stop

=back

L<https://redis.io/commands/ltrim>

=cut

sub ltrim : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(LTRIM) => @args)
}

=head2 rpop

Remove and get the last elements in a list.

=over 4

=item * key

=item * [count]

=back

L<https://redis.io/commands/rpop>

=cut

sub rpop : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(RPOP) => @args)
}

=head2 rpoplpush

Remove the last element in a list, prepend it to another list and return it.

=over 4

=item * source

=item * destination

=back

L<https://redis.io/commands/rpoplpush>

=cut

sub rpoplpush : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(RPOPLPUSH) => @args)
}

=head2 rpush

Append one or multiple elements to a list.

=over 4

=item * key

=item * element [element ...]

=back

L<https://redis.io/commands/rpush>

=cut

sub rpush : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(RPUSH) => @args)
}

=head2 rpushx

Append an element to a list, only if the list exists.

=over 4

=item * key

=item * element [element ...]

=back

L<https://redis.io/commands/rpushx>

=cut

sub rpushx : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(RPUSHX) => @args)
}

=head1 METHODS - Pubsub

=head2 psubscribe

Listen for messages published to channels matching the given patterns.

=over 4

=item * pattern [pattern ...]

=back

L<https://redis.io/commands/psubscribe>

=cut

sub psubscribe : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(PSUBSCRIBE) => @args)
}

=head2 publish

Post a message to a channel.

=over 4

=item * channel

=item * message

=back

L<https://redis.io/commands/publish>

=cut

sub publish : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(PUBLISH) => @args)
}

=head2 pubsub_channels

List active channels.

=over 4

=item * [pattern]

=back

L<https://redis.io/commands/pubsub-channels>

=cut

sub pubsub_channels : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(PUBSUB CHANNELS) => @args)
}

=head2 pubsub_numpat

Get the count of unique patterns pattern subscriptions.

L<https://redis.io/commands/pubsub-numpat>

=cut

sub pubsub_numpat : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(PUBSUB NUMPAT) => @args)
}

=head2 pubsub_numsub

Get the count of subscribers for channels.

=over 4

=item * [channel [channel ...]]

=back

L<https://redis.io/commands/pubsub-numsub>

=cut

sub pubsub_numsub : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(PUBSUB NUMSUB) => @args)
}

=head2 pubsub_shardchannels

List active shard channels.

=over 4

=item * [pattern]

=back

L<https://redis.io/commands/pubsub-shardchannels>

=cut

sub pubsub_shardchannels : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(PUBSUB SHARDCHANNELS) => @args)
}

=head2 pubsub_shardnumsub

Get the count of subscribers for shard channels.

=over 4

=item * [channel [channel ...]]

=back

L<https://redis.io/commands/pubsub-shardnumsub>

=cut

sub pubsub_shardnumsub : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(PUBSUB SHARDNUMSUB) => @args)
}

=head2 punsubscribe

Stop listening for messages posted to channels matching the given patterns.

=over 4

=item * [pattern [pattern ...]]

=back

L<https://redis.io/commands/punsubscribe>

=cut

sub punsubscribe : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(PUNSUBSCRIBE) => @args)
}

=head2 spublish

Post a message to a shard channel.

=over 4

=item * channel

=item * message

=back

L<https://redis.io/commands/spublish>

=cut

sub spublish : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SPUBLISH) => @args)
}

=head2 ssubscribe

Listen for messages published to the given shard channels.

=over 4

=item * channel [channel ...]

=back

L<https://redis.io/commands/ssubscribe>

=cut

sub ssubscribe : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SSUBSCRIBE) => @args)
}

=head2 subscribe

Listen for messages published to the given channels.

=over 4

=item * channel [channel ...]

=back

L<https://redis.io/commands/subscribe>

=cut

sub subscribe : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SUBSCRIBE) => @args)
}

=head2 sunsubscribe

Stop listening for messages posted to the given shard channels.

=over 4

=item * [channel [channel ...]]

=back

L<https://redis.io/commands/sunsubscribe>

=cut

sub sunsubscribe : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SUNSUBSCRIBE) => @args)
}

=head2 unsubscribe

Stop listening for messages posted to the given channels.

=over 4

=item * [channel [channel ...]]

=back

L<https://redis.io/commands/unsubscribe>

=cut

sub unsubscribe : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(UNSUBSCRIBE) => @args)
}

=head1 METHODS - Scripting

=head2 eval

Execute a Lua script server side.

=over 4

=item * script

=item * numkeys

=item * [key [key ...]]

=item * [arg [arg ...]]

=back

L<https://redis.io/commands/eval>

=cut

sub eval : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(EVAL) => @args)
}

=head2 eval_ro

Execute a read-only Lua script server side.

=over 4

=item * script

=item * numkeys

=item * key [key ...]

=item * arg [arg ...]

=back

L<https://redis.io/commands/eval-ro>

=cut

sub eval_ro : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(EVAL_RO) => @args)
}

=head2 evalsha

Execute a Lua script server side.

=over 4

=item * sha1

=item * numkeys

=item * [key [key ...]]

=item * [arg [arg ...]]

=back

L<https://redis.io/commands/evalsha>

=cut

sub evalsha : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(EVALSHA) => @args)
}

=head2 evalsha_ro

Execute a read-only Lua script server side.

=over 4

=item * sha1

=item * numkeys

=item * key [key ...]

=item * arg [arg ...]

=back

L<https://redis.io/commands/evalsha-ro>

=cut

sub evalsha_ro : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(EVALSHA_RO) => @args)
}

=head2 fcall

Invoke a function.

=over 4

=item * function

=item * numkeys

=item * key [key ...]

=item * arg [arg ...]

=back

L<https://redis.io/commands/fcall>

=cut

sub fcall : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(FCALL) => @args)
}

=head2 fcall_ro

Invoke a read-only function.

=over 4

=item * function

=item * numkeys

=item * key [key ...]

=item * arg [arg ...]

=back

L<https://redis.io/commands/fcall-ro>

=cut

sub fcall_ro : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(FCALL_RO) => @args)
}

=head2 function_delete

Delete a function by name.

=over 4

=item * library-name

=back

L<https://redis.io/commands/function-delete>

=cut

sub function_delete : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(FUNCTION DELETE) => @args)
}

=head2 function_dump

Dump all functions into a serialized binary payload.

L<https://redis.io/commands/function-dump>

=cut

sub function_dump : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(FUNCTION DUMP) => @args)
}

=head2 function_flush

Deleting all functions.

=over 4

=item * [ASYNC|SYNC]

=back

L<https://redis.io/commands/function-flush>

=cut

sub function_flush : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(FUNCTION FLUSH) => @args)
}

=head2 function_kill

Kill the function currently in execution.

L<https://redis.io/commands/function-kill>

=cut

sub function_kill : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(FUNCTION KILL) => @args)
}

=head2 function_list

List information about all the functions.

=over 4

=item * [LIBRARYNAME library-name-pattern]

=item * [WITHCODE]

=back

L<https://redis.io/commands/function-list>

=cut

sub function_list : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(FUNCTION LIST) => @args)
}

=head2 function_load

Create a function with the given arguments (name, code, description).

=over 4

=item * engine-name

=item * library-name

=item * [REPLACE]

=item * [DESCRIPTION library-description]

=item * function-code

=back

L<https://redis.io/commands/function-load>

=cut

sub function_load : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(FUNCTION LOAD) => @args)
}

=head2 function_restore

Restore all the functions on the given payload.

=over 4

=item * serialized-value

=item * [FLUSH|APPEND|REPLACE]

=back

L<https://redis.io/commands/function-restore>

=cut

sub function_restore : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(FUNCTION RESTORE) => @args)
}

=head2 function_stats

Return information about the function currently running (name, description, duration).

L<https://redis.io/commands/function-stats>

=cut

sub function_stats : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(FUNCTION STATS) => @args)
}

=head2 script_debug

Set the debug mode for executed scripts.

=over 4

=item * YES|SYNC|NO

=back

L<https://redis.io/commands/script-debug>

=cut

sub script_debug : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SCRIPT DEBUG) => @args)
}

=head2 script_exists

Check existence of scripts in the script cache.

=over 4

=item * sha1 [sha1 ...]

=back

L<https://redis.io/commands/script-exists>

=cut

sub script_exists : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SCRIPT EXISTS) => @args)
}

=head2 script_flush

Remove all the scripts from the script cache.

=over 4

=item * [ASYNC|SYNC]

=back

L<https://redis.io/commands/script-flush>

=cut

sub script_flush : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SCRIPT FLUSH) => @args)
}

=head2 script_kill

Kill the script currently in execution.

L<https://redis.io/commands/script-kill>

=cut

sub script_kill : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SCRIPT KILL) => @args)
}

=head2 script_load

Load the specified Lua script into the script cache.

=over 4

=item * script

=back

L<https://redis.io/commands/script-load>

=cut

sub script_load : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SCRIPT LOAD) => @args)
}

=head1 METHODS - Server

=head2 acl_cat

List the ACL categories or the commands inside a category.

=over 4

=item * [categoryname]

=back

L<https://redis.io/commands/acl-cat>

=cut

sub acl_cat : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ACL CAT) => @args)
}

=head2 acl_deluser

Remove the specified ACL users and the associated rules.

=over 4

=item * username [username ...]

=back

L<https://redis.io/commands/acl-deluser>

=cut

sub acl_deluser : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ACL DELUSER) => @args)
}

=head2 acl_dryrun

Returns whether the user can execute the given command without executing the command.

=over 4

=item * username

=item * command

=item * [arg [arg ...]]

=back

L<https://redis.io/commands/acl-dryrun>

=cut

sub acl_dryrun : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ACL DRYRUN) => @args)
}

=head2 acl_genpass

Generate a pseudorandom secure password to use for ACL users.

=over 4

=item * [bits]

=back

L<https://redis.io/commands/acl-genpass>

=cut

sub acl_genpass : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ACL GENPASS) => @args)
}

=head2 acl_getuser

Get the rules for a specific ACL user.

=over 4

=item * username

=back

L<https://redis.io/commands/acl-getuser>

=cut

sub acl_getuser : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ACL GETUSER) => @args)
}

=head2 acl_list

List the current ACL rules in ACL config file format.

L<https://redis.io/commands/acl-list>

=cut

sub acl_list : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ACL LIST) => @args)
}

=head2 acl_load

Reload the ACLs from the configured ACL file.

L<https://redis.io/commands/acl-load>

=cut

sub acl_load : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ACL LOAD) => @args)
}

=head2 acl_log

List latest events denied because of ACLs in place.

=over 4

=item * [count|RESET]

=back

L<https://redis.io/commands/acl-log>

=cut

sub acl_log : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ACL LOG) => @args)
}

=head2 acl_save

Save the current ACL rules in the configured ACL file.

L<https://redis.io/commands/acl-save>

=cut

sub acl_save : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ACL SAVE) => @args)
}

=head2 acl_setuser

Modify or create the rules for a specific ACL user.

=over 4

=item * username

=item * [rule [rule ...]]

=back

L<https://redis.io/commands/acl-setuser>

=cut

sub acl_setuser : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ACL SETUSER) => @args)
}

=head2 acl_users

List the username of all the configured ACL rules.

L<https://redis.io/commands/acl-users>

=cut

sub acl_users : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ACL USERS) => @args)
}

=head2 acl_whoami

Return the name of the user associated to the current connection.

L<https://redis.io/commands/acl-whoami>

=cut

sub acl_whoami : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ACL WHOAMI) => @args)
}

=head2 bgrewriteaof

Asynchronously rewrite the append-only file.

L<https://redis.io/commands/bgrewriteaof>

=cut

sub bgrewriteaof : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(BGREWRITEAOF) => @args)
}

=head2 bgsave

Asynchronously save the dataset to disk.

=over 4

=item * [SCHEDULE]

=back

L<https://redis.io/commands/bgsave>

=cut

sub bgsave : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(BGSAVE) => @args)
}

=head2 command

Get array of Redis command details.

L<https://redis.io/commands/command>

=cut

sub command : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(COMMAND) => @args)
}

=head2 command_count

Get total number of Redis commands.

L<https://redis.io/commands/command-count>

=cut

sub command_count : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(COMMAND COUNT) => @args)
}

=head2 command_docs

Get array of specific Redis command documentation.

=over 4

=item * [command-name [command-name ...]]

=back

L<https://redis.io/commands/command-docs>

=cut

sub command_docs : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(COMMAND DOCS) => @args)
}

=head2 command_getkeys

Extract keys given a full Redis command.

L<https://redis.io/commands/command-getkeys>

=cut

sub command_getkeys : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(COMMAND GETKEYS) => @args)
}

=head2 command_getkeysandflags

Extract keys given a full Redis command.

L<https://redis.io/commands/command-getkeysandflags>

=cut

sub command_getkeysandflags : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(COMMAND GETKEYSANDFLAGS) => @args)
}

=head2 command_info

Get array of specific Redis command details, or all when no argument is given.

=over 4

=item * [command-name [command-name ...]]

=back

L<https://redis.io/commands/command-info>

=cut

sub command_info : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(COMMAND INFO) => @args)
}

=head2 command_list

Get an array of Redis command names.

=over 4

=item * [FILTERBY MODULE module-name|ACLCAT category|PATTERN pattern]

=back

L<https://redis.io/commands/command-list>

=cut

sub command_list : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(COMMAND LIST) => @args)
}

=head2 config_get

Get the values of configuration parameters.

=over 4

=item * parameter [parameter ...]

=back

L<https://redis.io/commands/config-get>

=cut

sub config_get : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CONFIG GET) => @args)
}

=head2 config_resetstat

Reset the stats returned by INFO.

L<https://redis.io/commands/config-resetstat>

=cut

sub config_resetstat : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CONFIG RESETSTAT) => @args)
}

=head2 config_rewrite

Rewrite the configuration file with the in memory configuration.

L<https://redis.io/commands/config-rewrite>

=cut

sub config_rewrite : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CONFIG REWRITE) => @args)
}

=head2 config_set

Set configuration parameters to the given values.

=over 4

=item * parameter value [parameter value ...]

=back

L<https://redis.io/commands/config-set>

=cut

sub config_set : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(CONFIG SET) => @args)
}

=head2 dbsize

Return the number of keys in the selected database.

L<https://redis.io/commands/dbsize>

=cut

sub dbsize : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(DBSIZE) => @args)
}

=head2 failover

Start a coordinated failover between this server and one of its replicas.

=over 4

=item * [TO host port [FORCE]]

=item * [ABORT]

=item * [TIMEOUT milliseconds]

=back

L<https://redis.io/commands/failover>

=cut

sub failover : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(FAILOVER) => @args)
}

=head2 flushall

Remove all keys from all databases.

=over 4

=item * [ASYNC|SYNC]

=back

L<https://redis.io/commands/flushall>

=cut

sub flushall : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(FLUSHALL) => @args)
}

=head2 flushdb

Remove all keys from the current database.

=over 4

=item * [ASYNC|SYNC]

=back

L<https://redis.io/commands/flushdb>

=cut

sub flushdb : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(FLUSHDB) => @args)
}

=head2 info

Get information and statistics about the server.

=over 4

=item * [section [section ...]]

=back

L<https://redis.io/commands/info>

=cut

sub info : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(INFO) => @args)
}

=head2 lastsave

Get the UNIX time stamp of the last successful save to disk.

L<https://redis.io/commands/lastsave>

=cut

sub lastsave : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(LASTSAVE) => @args)
}

=head2 latency_doctor

Return a human readable latency analysis report.

L<https://redis.io/commands/latency-doctor>

=cut

sub latency_doctor : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(LATENCY DOCTOR) => @args)
}

=head2 latency_graph

Return a latency graph for the event.

=over 4

=item * event

=back

L<https://redis.io/commands/latency-graph>

=cut

sub latency_graph : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(LATENCY GRAPH) => @args)
}

=head2 latency_histogram

Return the cumulative distribution of latencies of a subset of commands or all.

=over 4

=item * [command [command ...]]

=back

L<https://redis.io/commands/latency-histogram>

=cut

sub latency_histogram : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(LATENCY HISTOGRAM) => @args)
}

=head2 latency_history

Return timestamp-latency samples for the event.

=over 4

=item * event

=back

L<https://redis.io/commands/latency-history>

=cut

sub latency_history : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(LATENCY HISTORY) => @args)
}

=head2 latency_latest

Return the latest latency samples for all events.

L<https://redis.io/commands/latency-latest>

=cut

sub latency_latest : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(LATENCY LATEST) => @args)
}

=head2 latency_reset

Reset latency data for one or more events.

=over 4

=item * [event [event ...]]

=back

L<https://redis.io/commands/latency-reset>

=cut

sub latency_reset : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(LATENCY RESET) => @args)
}

=head2 lolwut

Display some computer art and the Redis version.

=over 4

=item * [VERSION version]

=back

L<https://redis.io/commands/lolwut>

=cut

sub lolwut : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(LOLWUT) => @args)
}

=head2 memory_doctor

Outputs memory problems report.

L<https://redis.io/commands/memory-doctor>

=cut

sub memory_doctor : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(MEMORY DOCTOR) => @args)
}

=head2 memory_malloc_stats

Show allocator internal stats.

L<https://redis.io/commands/memory-malloc-stats>

=cut

sub memory_malloc_stats : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(MEMORY MALLOC-STATS) => @args)
}

=head2 memory_purge

Ask the allocator to release memory.

L<https://redis.io/commands/memory-purge>

=cut

sub memory_purge : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(MEMORY PURGE) => @args)
}

=head2 memory_stats

Show memory usage details.

L<https://redis.io/commands/memory-stats>

=cut

sub memory_stats : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(MEMORY STATS) => @args)
}

=head2 memory_usage

Estimate the memory usage of a key.

=over 4

=item * key

=item * [SAMPLES count]

=back

L<https://redis.io/commands/memory-usage>

=cut

sub memory_usage : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(MEMORY USAGE) => @args)
}

=head2 module_list

List all modules loaded by the server.

L<https://redis.io/commands/module-list>

=cut

sub module_list : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(MODULE LIST) => @args)
}

=head2 module_load

Load a module.

=over 4

=item * path

=item * [arg [arg ...]]

=back

L<https://redis.io/commands/module-load>

=cut

sub module_load : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(MODULE LOAD) => @args)
}

=head2 module_unload

Unload a module.

=over 4

=item * name

=back

L<https://redis.io/commands/module-unload>

=cut

sub module_unload : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(MODULE UNLOAD) => @args)
}

=head2 monitor

Listen for all requests received by the server in real time.

L<https://redis.io/commands/monitor>

=cut

sub monitor : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(MONITOR) => @args)
}

=head2 psync

Internal command used for replication.

=over 4

=item * replicationid

=item * offset

=back

L<https://redis.io/commands/psync>

=cut

sub psync : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(PSYNC) => @args)
}

=head2 replconf

An internal command for configuring the replication stream.

L<https://redis.io/commands/replconf>

=cut

sub replconf : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(REPLCONF) => @args)
}

=head2 replicaof

Make the server a replica of another instance, or promote it as master.

=over 4

=item * host

=item * port

=back

L<https://redis.io/commands/replicaof>

=cut

sub replicaof : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(REPLICAOF) => @args)
}

=head2 restore_asking

An internal command for migrating keys in a cluster.

L<https://redis.io/commands/restore-asking>

=cut

sub restore_asking : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(RESTORE-ASKING) => @args)
}

=head2 role

Return the role of the instance in the context of replication.

L<https://redis.io/commands/role>

=cut

sub role : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ROLE) => @args)
}

=head2 save

Synchronously save the dataset to disk.

L<https://redis.io/commands/save>

=cut

sub save : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SAVE) => @args)
}

=head2 shutdown

Synchronously save the dataset to disk and then shut down the server.

=over 4

=item * [NOSAVE|SAVE]

=item * [NOW]

=item * [FORCE]

=item * [ABORT]

=back

L<https://redis.io/commands/shutdown>

=cut

sub shutdown : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SHUTDOWN) => @args)
}

=head2 slaveof

Make the server a replica of another instance, or promote it as master.

=over 4

=item * host

=item * port

=back

L<https://redis.io/commands/slaveof>

=cut

sub slaveof : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SLAVEOF) => @args)
}

=head2 slowlog_get

Get the slow log's entries.

=over 4

=item * [count]

=back

L<https://redis.io/commands/slowlog-get>

=cut

sub slowlog_get : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SLOWLOG GET) => @args)
}

=head2 slowlog_len

Get the slow log's length.

L<https://redis.io/commands/slowlog-len>

=cut

sub slowlog_len : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SLOWLOG LEN) => @args)
}

=head2 slowlog_reset

Clear all entries from the slow log.

L<https://redis.io/commands/slowlog-reset>

=cut

sub slowlog_reset : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SLOWLOG RESET) => @args)
}

=head2 swapdb

Swaps two Redis databases.

=over 4

=item * index1

=item * index2

=back

L<https://redis.io/commands/swapdb>

=cut

sub swapdb : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SWAPDB) => @args)
}

=head2 sync

Internal command used for replication.

L<https://redis.io/commands/sync>

=cut

sub sync : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SYNC) => @args)
}

=head2 time

Return the current server time.

L<https://redis.io/commands/time>

=cut

sub time : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(TIME) => @args)
}

=head1 METHODS - Set

=head2 sadd

Add one or more members to a set.

=over 4

=item * key

=item * member [member ...]

=back

L<https://redis.io/commands/sadd>

=cut

sub sadd : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SADD) => @args)
}

=head2 scard

Get the number of members in a set.

=over 4

=item * key

=back

L<https://redis.io/commands/scard>

=cut

sub scard : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SCARD) => @args)
}

=head2 sdiff

Subtract multiple sets.

=over 4

=item * key [key ...]

=back

L<https://redis.io/commands/sdiff>

=cut

sub sdiff : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SDIFF) => @args)
}

=head2 sdiffstore

Subtract multiple sets and store the resulting set in a key.

=over 4

=item * destination

=item * key [key ...]

=back

L<https://redis.io/commands/sdiffstore>

=cut

sub sdiffstore : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SDIFFSTORE) => @args)
}

=head2 sinter

Intersect multiple sets.

=over 4

=item * key [key ...]

=back

L<https://redis.io/commands/sinter>

=cut

sub sinter : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SINTER) => @args)
}

=head2 sintercard

Intersect multiple sets and return the cardinality of the result.

=over 4

=item * numkeys

=item * key [key ...]

=item * [LIMIT limit]

=back

L<https://redis.io/commands/sintercard>

=cut

sub sintercard : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SINTERCARD) => @args)
}

=head2 sinterstore

Intersect multiple sets and store the resulting set in a key.

=over 4

=item * destination

=item * key [key ...]

=back

L<https://redis.io/commands/sinterstore>

=cut

sub sinterstore : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SINTERSTORE) => @args)
}

=head2 sismember

Determine if a given value is a member of a set.

=over 4

=item * key

=item * member

=back

L<https://redis.io/commands/sismember>

=cut

sub sismember : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SISMEMBER) => @args)
}

=head2 smembers

Get all the members in a set.

=over 4

=item * key

=back

L<https://redis.io/commands/smembers>

=cut

sub smembers : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SMEMBERS) => @args)
}

=head2 smismember

Returns the membership associated with the given elements for a set.

=over 4

=item * key

=item * member [member ...]

=back

L<https://redis.io/commands/smismember>

=cut

sub smismember : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SMISMEMBER) => @args)
}

=head2 smove

Move a member from one set to another.

=over 4

=item * source

=item * destination

=item * member

=back

L<https://redis.io/commands/smove>

=cut

sub smove : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SMOVE) => @args)
}

=head2 spop

Remove and return one or multiple random members from a set.

=over 4

=item * key

=item * [count]

=back

L<https://redis.io/commands/spop>

=cut

sub spop : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SPOP) => @args)
}

=head2 srandmember

Get one or multiple random members from a set.

=over 4

=item * key

=item * [count]

=back

L<https://redis.io/commands/srandmember>

=cut

sub srandmember : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SRANDMEMBER) => @args)
}

=head2 srem

Remove one or more members from a set.

=over 4

=item * key

=item * member [member ...]

=back

L<https://redis.io/commands/srem>

=cut

sub srem : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SREM) => @args)
}

=head2 sscan

Incrementally iterate Set elements.

=over 4

=item * key

=item * cursor

=item * [MATCH pattern]

=item * [COUNT count]

=back

L<https://redis.io/commands/sscan>

=cut

sub sscan : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SSCAN) => @args)
}

=head2 sunion

Add multiple sets.

=over 4

=item * key [key ...]

=back

L<https://redis.io/commands/sunion>

=cut

sub sunion : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SUNION) => @args)
}

=head2 sunionstore

Add multiple sets and store the resulting set in a key.

=over 4

=item * destination

=item * key [key ...]

=back

L<https://redis.io/commands/sunionstore>

=cut

sub sunionstore : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SUNIONSTORE) => @args)
}

=head1 METHODS - Sorted-set

=head2 bzmpop

Remove and return members with scores in a sorted set or block until one is available.

=over 4

=item * timeout

=item * numkeys

=item * key [key ...]

=item * MIN|MAX

=item * [COUNT count]

=back

L<https://redis.io/commands/bzmpop>

=cut

sub bzmpop : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(BZMPOP) => @args)
}

=head2 bzpopmax

Remove and return the member with the highest score from one or more sorted sets, or block until one is available.

=over 4

=item * key [key ...]

=item * timeout

=back

L<https://redis.io/commands/bzpopmax>

=cut

sub bzpopmax : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(BZPOPMAX) => @args)
}

=head2 bzpopmin

Remove and return the member with the lowest score from one or more sorted sets, or block until one is available.

=over 4

=item * key [key ...]

=item * timeout

=back

L<https://redis.io/commands/bzpopmin>

=cut

sub bzpopmin : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(BZPOPMIN) => @args)
}

=head2 zadd

Add one or more members to a sorted set, or update its score if it already exists.

=over 4

=item * key

=item * [NX|XX]

=item * [GT|LT]

=item * [CH]

=item * [INCR]

=item * score member [score member ...]

=back

L<https://redis.io/commands/zadd>

=cut

sub zadd : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZADD) => @args)
}

=head2 zcard

Get the number of members in a sorted set.

=over 4

=item * key

=back

L<https://redis.io/commands/zcard>

=cut

sub zcard : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZCARD) => @args)
}

=head2 zcount

Count the members in a sorted set with scores within the given values.

=over 4

=item * key

=item * min

=item * max

=back

L<https://redis.io/commands/zcount>

=cut

sub zcount : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZCOUNT) => @args)
}

=head2 zdiff

Subtract multiple sorted sets.

=over 4

=item * numkeys

=item * key [key ...]

=item * [WITHSCORES]

=back

L<https://redis.io/commands/zdiff>

=cut

sub zdiff : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZDIFF) => @args)
}

=head2 zdiffstore

Subtract multiple sorted sets and store the resulting sorted set in a new key.

=over 4

=item * destination

=item * numkeys

=item * key [key ...]

=back

L<https://redis.io/commands/zdiffstore>

=cut

sub zdiffstore : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZDIFFSTORE) => @args)
}

=head2 zincrby

Increment the score of a member in a sorted set.

=over 4

=item * key

=item * increment

=item * member

=back

L<https://redis.io/commands/zincrby>

=cut

sub zincrby : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZINCRBY) => @args)
}

=head2 zinter

Intersect multiple sorted sets.

=over 4

=item * numkeys

=item * key [key ...]

=item * [WEIGHTS weight [weight ...]]

=item * [AGGREGATE SUM|MIN|MAX]

=item * [WITHSCORES]

=back

L<https://redis.io/commands/zinter>

=cut

sub zinter : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZINTER) => @args)
}

=head2 zintercard

Intersect multiple sorted sets and return the cardinality of the result.

=over 4

=item * numkeys

=item * key [key ...]

=item * [LIMIT limit]

=back

L<https://redis.io/commands/zintercard>

=cut

sub zintercard : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZINTERCARD) => @args)
}

=head2 zinterstore

Intersect multiple sorted sets and store the resulting sorted set in a new key.

=over 4

=item * destination

=item * numkeys

=item * key [key ...]

=item * [WEIGHTS weight [weight ...]]

=item * [AGGREGATE SUM|MIN|MAX]

=back

L<https://redis.io/commands/zinterstore>

=cut

sub zinterstore : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZINTERSTORE) => @args)
}

=head2 zlexcount

Count the number of members in a sorted set between a given lexicographical range.

=over 4

=item * key

=item * min

=item * max

=back

L<https://redis.io/commands/zlexcount>

=cut

sub zlexcount : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZLEXCOUNT) => @args)
}

=head2 zmpop

Remove and return members with scores in a sorted set.

=over 4

=item * numkeys

=item * key [key ...]

=item * MIN|MAX

=item * [COUNT count]

=back

L<https://redis.io/commands/zmpop>

=cut

sub zmpop : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZMPOP) => @args)
}

=head2 zmscore

Get the score associated with the given members in a sorted set.

=over 4

=item * key

=item * member [member ...]

=back

L<https://redis.io/commands/zmscore>

=cut

sub zmscore : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZMSCORE) => @args)
}

=head2 zpopmax

Remove and return members with the highest scores in a sorted set.

=over 4

=item * key

=item * [count]

=back

L<https://redis.io/commands/zpopmax>

=cut

sub zpopmax : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZPOPMAX) => @args)
}

=head2 zpopmin

Remove and return members with the lowest scores in a sorted set.

=over 4

=item * key

=item * [count]

=back

L<https://redis.io/commands/zpopmin>

=cut

sub zpopmin : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZPOPMIN) => @args)
}

=head2 zrandmember

Get one or multiple random elements from a sorted set.

=over 4

=item * key

=item * [count [WITHSCORES]]

=back

L<https://redis.io/commands/zrandmember>

=cut

sub zrandmember : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZRANDMEMBER) => @args)
}

=head2 zrange

Return a range of members in a sorted set.

=over 4

=item * key

=item * min

=item * max

=item * [BYSCORE|BYLEX]

=item * [REV]

=item * [LIMIT offset count]

=item * [WITHSCORES]

=back

L<https://redis.io/commands/zrange>

=cut

sub zrange : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZRANGE) => @args)
}

=head2 zrangebylex

Return a range of members in a sorted set, by lexicographical range.

=over 4

=item * key

=item * min

=item * max

=item * [LIMIT offset count]

=back

L<https://redis.io/commands/zrangebylex>

=cut

sub zrangebylex : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZRANGEBYLEX) => @args)
}

=head2 zrangebyscore

Return a range of members in a sorted set, by score.

=over 4

=item * key

=item * min

=item * max

=item * [WITHSCORES]

=item * [LIMIT offset count]

=back

L<https://redis.io/commands/zrangebyscore>

=cut

sub zrangebyscore : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZRANGEBYSCORE) => @args)
}

=head2 zrangestore

Store a range of members from sorted set into another key.

=over 4

=item * dst

=item * src

=item * min

=item * max

=item * [BYSCORE|BYLEX]

=item * [REV]

=item * [LIMIT offset count]

=back

L<https://redis.io/commands/zrangestore>

=cut

sub zrangestore : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZRANGESTORE) => @args)
}

=head2 zrank

Determine the index of a member in a sorted set.

=over 4

=item * key

=item * member

=back

L<https://redis.io/commands/zrank>

=cut

sub zrank : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZRANK) => @args)
}

=head2 zrem

Remove one or more members from a sorted set.

=over 4

=item * key

=item * member [member ...]

=back

L<https://redis.io/commands/zrem>

=cut

sub zrem : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZREM) => @args)
}

=head2 zremrangebylex

Remove all members in a sorted set between the given lexicographical range.

=over 4

=item * key

=item * min

=item * max

=back

L<https://redis.io/commands/zremrangebylex>

=cut

sub zremrangebylex : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZREMRANGEBYLEX) => @args)
}

=head2 zremrangebyrank

Remove all members in a sorted set within the given indexes.

=over 4

=item * key

=item * start

=item * stop

=back

L<https://redis.io/commands/zremrangebyrank>

=cut

sub zremrangebyrank : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZREMRANGEBYRANK) => @args)
}

=head2 zremrangebyscore

Remove all members in a sorted set within the given scores.

=over 4

=item * key

=item * min

=item * max

=back

L<https://redis.io/commands/zremrangebyscore>

=cut

sub zremrangebyscore : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZREMRANGEBYSCORE) => @args)
}

=head2 zrevrange

Return a range of members in a sorted set, by index, with scores ordered from high to low.

=over 4

=item * key

=item * start

=item * stop

=item * [WITHSCORES]

=back

L<https://redis.io/commands/zrevrange>

=cut

sub zrevrange : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZREVRANGE) => @args)
}

=head2 zrevrangebylex

Return a range of members in a sorted set, by lexicographical range, ordered from higher to lower strings.

=over 4

=item * key

=item * max

=item * min

=item * [LIMIT offset count]

=back

L<https://redis.io/commands/zrevrangebylex>

=cut

sub zrevrangebylex : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZREVRANGEBYLEX) => @args)
}

=head2 zrevrangebyscore

Return a range of members in a sorted set, by score, with scores ordered from high to low.

=over 4

=item * key

=item * max

=item * min

=item * [WITHSCORES]

=item * [LIMIT offset count]

=back

L<https://redis.io/commands/zrevrangebyscore>

=cut

sub zrevrangebyscore : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZREVRANGEBYSCORE) => @args)
}

=head2 zrevrank

Determine the index of a member in a sorted set, with scores ordered from high to low.

=over 4

=item * key

=item * member

=back

L<https://redis.io/commands/zrevrank>

=cut

sub zrevrank : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZREVRANK) => @args)
}

=head2 zscan

Incrementally iterate sorted sets elements and associated scores.

=over 4

=item * key

=item * cursor

=item * [MATCH pattern]

=item * [COUNT count]

=back

L<https://redis.io/commands/zscan>

=cut

sub zscan : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZSCAN) => @args)
}

=head2 zscore

Get the score associated with the given member in a sorted set.

=over 4

=item * key

=item * member

=back

L<https://redis.io/commands/zscore>

=cut

sub zscore : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZSCORE) => @args)
}

=head2 zunion

Add multiple sorted sets.

=over 4

=item * numkeys

=item * key [key ...]

=item * [WEIGHTS weight [weight ...]]

=item * [AGGREGATE SUM|MIN|MAX]

=item * [WITHSCORES]

=back

L<https://redis.io/commands/zunion>

=cut

sub zunion : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZUNION) => @args)
}

=head2 zunionstore

Add multiple sorted sets and store the resulting sorted set in a new key.

=over 4

=item * destination

=item * numkeys

=item * key [key ...]

=item * [WEIGHTS weight [weight ...]]

=item * [AGGREGATE SUM|MIN|MAX]

=back

L<https://redis.io/commands/zunionstore>

=cut

sub zunionstore : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(ZUNIONSTORE) => @args)
}

=head1 METHODS - Stream

=head2 xack

Marks a pending message as correctly processed, effectively removing it from the pending entries list of the consumer group. Return value of the command is the number of messages successfully acknowledged, that is, the IDs we were actually able to resolve in the PEL.

=over 4

=item * key

=item * group

=item * id [id ...]

=back

L<https://redis.io/commands/xack>

=cut

sub xack : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(XACK) => @args)
}

=head2 xadd

Appends a new entry to a stream.

=over 4

=item * key

=item * [NOMKSTREAM]

=item * [MAXLEN|MINID [=|~] threshold [LIMIT count]]

=item * *|id

=item * field value [field value ...]

=back

L<https://redis.io/commands/xadd>

=cut

sub xadd : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(XADD) => @args)
}

=head2 xautoclaim

Changes (or acquires) ownership of messages in a consumer group, as if the messages were delivered to the specified consumer.

=over 4

=item * key

=item * group

=item * consumer

=item * min-idle-time

=item * start

=item * [COUNT count]

=item * [JUSTID]

=back

L<https://redis.io/commands/xautoclaim>

=cut

sub xautoclaim : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(XAUTOCLAIM) => @args)
}

=head2 xclaim

Changes (or acquires) ownership of a message in a consumer group, as if the message was delivered to the specified consumer.

=over 4

=item * key

=item * group

=item * consumer

=item * min-idle-time

=item * id [id ...]

=item * [IDLE ms]

=item * [TIME unix-time-milliseconds]

=item * [RETRYCOUNT count]

=item * [FORCE]

=item * [JUSTID]

=back

L<https://redis.io/commands/xclaim>

=cut

sub xclaim : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(XCLAIM) => @args)
}

=head2 xdel

Removes the specified entries from the stream. Returns the number of items actually deleted, that may be different from the number of IDs passed in case certain IDs do not exist.

=over 4

=item * key

=item * id [id ...]

=back

L<https://redis.io/commands/xdel>

=cut

sub xdel : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(XDEL) => @args)
}

=head2 xgroup_create

Create a consumer group.

=over 4

=item * key

=item * groupname

=item * id|$

=item * [MKSTREAM]

=item * [ENTRIESREAD entries_read]

=back

L<https://redis.io/commands/xgroup-create>

=cut

sub xgroup_create : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(XGROUP CREATE) => @args)
}

=head2 xgroup_createconsumer

Create a consumer in a consumer group.

=over 4

=item * key

=item * groupname

=item * consumername

=back

L<https://redis.io/commands/xgroup-createconsumer>

=cut

sub xgroup_createconsumer : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(XGROUP CREATECONSUMER) => @args)
}

=head2 xgroup_delconsumer

Delete a consumer from a consumer group.

=over 4

=item * key

=item * groupname

=item * consumername

=back

L<https://redis.io/commands/xgroup-delconsumer>

=cut

sub xgroup_delconsumer : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(XGROUP DELCONSUMER) => @args)
}

=head2 xgroup_destroy

Destroy a consumer group.

=over 4

=item * key

=item * groupname

=back

L<https://redis.io/commands/xgroup-destroy>

=cut

sub xgroup_destroy : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(XGROUP DESTROY) => @args)
}

=head2 xgroup_setid

Set a consumer group to an arbitrary last delivered ID value.

=over 4

=item * key

=item * groupname

=item * id|$

=item * [ENTRIESREAD entries_read]

=back

L<https://redis.io/commands/xgroup-setid>

=cut

sub xgroup_setid : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(XGROUP SETID) => @args)
}

=head2 xinfo_consumers

List the consumers in a consumer group.

=over 4

=item * key

=item * groupname

=back

L<https://redis.io/commands/xinfo-consumers>

=cut

sub xinfo_consumers : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(XINFO CONSUMERS) => @args)
}

=head2 xinfo_groups

List the consumer groups of a stream.

=over 4

=item * key

=back

L<https://redis.io/commands/xinfo-groups>

=cut

sub xinfo_groups : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(XINFO GROUPS) => @args)
}

=head2 xinfo_stream

Get information about a stream.

=over 4

=item * key

=item * [FULL [COUNT count]]

=back

L<https://redis.io/commands/xinfo-stream>

=cut

sub xinfo_stream : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(XINFO STREAM) => @args)
}

=head2 xlen

Return the number of entries in a stream.

=over 4

=item * key

=back

L<https://redis.io/commands/xlen>

=cut

sub xlen : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(XLEN) => @args)
}

=head2 xpending

Return information and entries from a stream consumer group pending entries list, that are messages fetched but never acknowledged.

=over 4

=item * key

=item * group

=item * [[IDLE min-idle-time] start end count [consumer]]

=back

L<https://redis.io/commands/xpending>

=cut

sub xpending : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(XPENDING) => @args)
}

=head2 xrange

Return a range of elements in a stream, with IDs matching the specified IDs interval.

=over 4

=item * key

=item * start

=item * end

=item * [COUNT count]

=back

L<https://redis.io/commands/xrange>

=cut

sub xrange : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(XRANGE) => @args)
}

=head2 xread

Return never seen elements in multiple streams, with IDs greater than the ones reported by the caller for each stream. Can block.

=over 4

=item * [COUNT count]

=item * [BLOCK milliseconds]

=item * STREAMS key [key ...] id [id ...]

=back

L<https://redis.io/commands/xread>

=cut

sub xread : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(XREAD) => @args)
}

=head2 xreadgroup

Return new entries from a stream using a consumer group, or access the history of the pending entries for a given consumer. Can block.

=over 4

=item * GROUP group consumer

=item * [COUNT count]

=item * [BLOCK milliseconds]

=item * [NOACK]

=item * STREAMS key [key ...] id [id ...]

=back

L<https://redis.io/commands/xreadgroup>

=cut

sub xreadgroup : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(XREADGROUP) => @args)
}

=head2 xrevrange

Return a range of elements in a stream, with IDs matching the specified IDs interval, in reverse order (from greater to smaller IDs) compared to XRANGE.

=over 4

=item * key

=item * end

=item * start

=item * [COUNT count]

=back

L<https://redis.io/commands/xrevrange>

=cut

sub xrevrange : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(XREVRANGE) => @args)
}

=head2 xsetid

An internal command for replicating stream values.

=over 4

=item * key

=item * last-id

=item * [ENTRIESADDED entries_added]

=item * [MAXDELETEDID max_deleted_entry_id]

=back

L<https://redis.io/commands/xsetid>

=cut

sub xsetid : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(XSETID) => @args)
}

=head2 xtrim

Trims the stream to (approximately if '~' is passed) a certain size.

=over 4

=item * key

=item * MAXLEN|MINID [=|~] threshold [LIMIT count]

=back

L<https://redis.io/commands/xtrim>

=cut

sub xtrim : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(XTRIM) => @args)
}

=head1 METHODS - String

=head2 append

Append a value to a key.

=over 4

=item * key

=item * value

=back

L<https://redis.io/commands/append>

=cut

sub append : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(APPEND) => @args)
}

=head2 decr

Decrement the integer value of a key by one.

=over 4

=item * key

=back

L<https://redis.io/commands/decr>

=cut

sub decr : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(DECR) => @args)
}

=head2 decrby

Decrement the integer value of a key by the given number.

=over 4

=item * key

=item * decrement

=back

L<https://redis.io/commands/decrby>

=cut

sub decrby : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(DECRBY) => @args)
}

=head2 get

Get the value of a key.

=over 4

=item * key

=back

L<https://redis.io/commands/get>

=cut

sub get : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(GET) => @args)
}

=head2 getdel

Get the value of a key and delete the key.

=over 4

=item * key

=back

L<https://redis.io/commands/getdel>

=cut

sub getdel : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(GETDEL) => @args)
}

=head2 getex

Get the value of a key and optionally set its expiration.

=over 4

=item * key

=item * [EX seconds|PX milliseconds|EXAT unix-time-seconds|PXAT unix-time-milliseconds|PERSIST]

=back

L<https://redis.io/commands/getex>

=cut

sub getex : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(GETEX) => @args)
}

=head2 getrange

Get a substring of the string stored at a key.

=over 4

=item * key

=item * start

=item * end

=back

L<https://redis.io/commands/getrange>

=cut

sub getrange : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(GETRANGE) => @args)
}

=head2 getset

Set the string value of a key and return its old value.

=over 4

=item * key

=item * value

=back

L<https://redis.io/commands/getset>

=cut

sub getset : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(GETSET) => @args)
}

=head2 incr

Increment the integer value of a key by one.

=over 4

=item * key

=back

L<https://redis.io/commands/incr>

=cut

sub incr : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(INCR) => @args)
}

=head2 incrby

Increment the integer value of a key by the given amount.

=over 4

=item * key

=item * increment

=back

L<https://redis.io/commands/incrby>

=cut

sub incrby : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(INCRBY) => @args)
}

=head2 incrbyfloat

Increment the float value of a key by the given amount.

=over 4

=item * key

=item * increment

=back

L<https://redis.io/commands/incrbyfloat>

=cut

sub incrbyfloat : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(INCRBYFLOAT) => @args)
}

=head2 lcs

Find longest common substring.

=over 4

=item * key1

=item * key2

=item * [LEN]

=item * [IDX]

=item * [MINMATCHLEN len]

=item * [WITHMATCHLEN]

=back

L<https://redis.io/commands/lcs>

=cut

sub lcs : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(LCS) => @args)
}

=head2 mget

Get the values of all the given keys.

=over 4

=item * key [key ...]

=back

L<https://redis.io/commands/mget>

=cut

sub mget : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(MGET) => @args)
}

=head2 mset

Set multiple keys to multiple values.

=over 4

=item * key value [key value ...]

=back

L<https://redis.io/commands/mset>

=cut

sub mset : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(MSET) => @args)
}

=head2 msetnx

Set multiple keys to multiple values, only if none of the keys exist.

=over 4

=item * key value [key value ...]

=back

L<https://redis.io/commands/msetnx>

=cut

sub msetnx : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(MSETNX) => @args)
}

=head2 psetex

Set the value and expiration in milliseconds of a key.

=over 4

=item * key

=item * milliseconds

=item * value

=back

L<https://redis.io/commands/psetex>

=cut

sub psetex : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(PSETEX) => @args)
}

=head2 set

Set the string value of a key.

=over 4

=item * key

=item * value

=item * [EX seconds|PX milliseconds|EXAT unix-time-seconds|PXAT unix-time-milliseconds|KEEPTTL]

=item * [NX|XX]

=item * [GET]

=back

L<https://redis.io/commands/set>

=cut

sub set : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SET) => @args)
}

=head2 setex

Set the value and expiration of a key.

=over 4

=item * key

=item * seconds

=item * value

=back

L<https://redis.io/commands/setex>

=cut

sub setex : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SETEX) => @args)
}

=head2 setnx

Set the value of a key, only if the key does not exist.

=over 4

=item * key

=item * value

=back

L<https://redis.io/commands/setnx>

=cut

sub setnx : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SETNX) => @args)
}

=head2 setrange

Overwrite part of a string at key starting at the specified offset.

=over 4

=item * key

=item * offset

=item * value

=back

L<https://redis.io/commands/setrange>

=cut

sub setrange : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SETRANGE) => @args)
}

=head2 strlen

Get the length of the value stored in a key.

=over 4

=item * key

=back

L<https://redis.io/commands/strlen>

=cut

sub strlen : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(STRLEN) => @args)
}

=head2 substr

Get a substring of the string stored at a key.

=over 4

=item * key

=item * start

=item * end

=back

L<https://redis.io/commands/substr>

=cut

sub substr : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(SUBSTR) => @args)
}

=head1 METHODS - Transactions

=head2 discard

Discard all commands issued after MULTI.

L<https://redis.io/commands/discard>

=cut

sub discard : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(DISCARD) => @args)
}

=head2 exec

Execute all commands issued after MULTI.

L<https://redis.io/commands/exec>

=cut

sub exec : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(EXEC) => @args)
}

=head2 multi

Mark the start of a transaction block.

L<https://redis.io/commands/multi>

=cut

sub multi : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(MULTI) => @args)
}

=head2 unwatch

Forget about all watched keys.

L<https://redis.io/commands/unwatch>

=cut

sub unwatch : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(UNWATCH) => @args)
}

=head2 watch

Watch the given keys to determine execution of the MULTI/EXEC block.

=over 4

=item * key [key ...]

=back

L<https://redis.io/commands/watch>

=cut

sub watch : method {
    my ($self, @args) = @_;
    $self->execute_command(qw(WATCH) => @args)
}


=head1 METHODS - Legacy

These take a subcommand as a parameter and construct the method name by
combining the main command with subcommand - for example, C<< ->xgroup(CREATE => ...) >>
would call C<< ->xgroup_create >>.

=cut

=head2 acl

=cut

sub acl : method {
    my ($self, $cmd, @args) = @_;
    $cmd =~ tr/ /_/;
    my $method = "acl_" . lc($cmd);
    return $self->$method(@args);
}

=head2 client

=cut

sub client : method {
    my ($self, $cmd, @args) = @_;
    $cmd =~ tr/ /_/;
    my $method = "client_" . lc($cmd);
    return $self->$method(@args);
}

=head2 client_no

=cut

sub client_no : method {
    my ($self, $cmd, @args) = @_;
    $cmd =~ tr/ /_/;
    my $method = "client_no_" . lc($cmd);
    return $self->$method(@args);
}

=head2 cluster

=cut

sub cluster : method {
    my ($self, $cmd, @args) = @_;
    $cmd =~ tr/ /_/;
    my $method = "cluster_" . lc($cmd);
    return $self->$method(@args);
}

=head2 cluster_count

=cut

sub cluster_count : method {
    my ($self, $cmd, @args) = @_;
    $cmd =~ tr/ /_/;
    my $method = "cluster_count_" . lc($cmd);
    return $self->$method(@args);
}

=head2 cluster_count_failure

=cut

sub cluster_count_failure : method {
    my ($self, $cmd, @args) = @_;
    $cmd =~ tr/ /_/;
    my $method = "cluster_count_failure_" . lc($cmd);
    return $self->$method(@args);
}

=head2 cluster_set

=cut

sub cluster_set : method {
    my ($self, $cmd, @args) = @_;
    $cmd =~ tr/ /_/;
    my $method = "cluster_set_" . lc($cmd);
    return $self->$method(@args);
}

=head2 cluster_set_config

=cut

sub cluster_set_config : method {
    my ($self, $cmd, @args) = @_;
    $cmd =~ tr/ /_/;
    my $method = "cluster_set_config_" . lc($cmd);
    return $self->$method(@args);
}

=head2 config

=cut

sub config : method {
    my ($self, $cmd, @args) = @_;
    $cmd =~ tr/ /_/;
    my $method = "config_" . lc($cmd);
    return $self->$method(@args);
}

=head2 function

=cut

sub function : method {
    my ($self, $cmd, @args) = @_;
    $cmd =~ tr/ /_/;
    my $method = "function_" . lc($cmd);
    return $self->$method(@args);
}

=head2 latency

=cut

sub latency : method {
    my ($self, $cmd, @args) = @_;
    $cmd =~ tr/ /_/;
    my $method = "latency_" . lc($cmd);
    return $self->$method(@args);
}

=head2 memory

=cut

sub memory : method {
    my ($self, $cmd, @args) = @_;
    $cmd =~ tr/ /_/;
    my $method = "memory_" . lc($cmd);
    return $self->$method(@args);
}

=head2 memory_malloc

=cut

sub memory_malloc : method {
    my ($self, $cmd, @args) = @_;
    $cmd =~ tr/ /_/;
    my $method = "memory_malloc_" . lc($cmd);
    return $self->$method(@args);
}

=head2 module

=cut

sub module : method {
    my ($self, $cmd, @args) = @_;
    $cmd =~ tr/ /_/;
    my $method = "module_" . lc($cmd);
    return $self->$method(@args);
}

=head2 object

=cut

sub object : method {
    my ($self, $cmd, @args) = @_;
    $cmd =~ tr/ /_/;
    my $method = "object_" . lc($cmd);
    return $self->$method(@args);
}

=head2 pubsub

=cut

sub pubsub : method {
    my ($self, $cmd, @args) = @_;
    $cmd =~ tr/ /_/;
    my $method = "pubsub_" . lc($cmd);
    return $self->$method(@args);
}

=head2 script

=cut

sub script : method {
    my ($self, $cmd, @args) = @_;
    $cmd =~ tr/ /_/;
    my $method = "script_" . lc($cmd);
    return $self->$method(@args);
}

=head2 slowlog

=cut

sub slowlog : method {
    my ($self, $cmd, @args) = @_;
    $cmd =~ tr/ /_/;
    my $method = "slowlog_" . lc($cmd);
    return $self->$method(@args);
}

=head2 xgroup

=cut

sub xgroup : method {
    my ($self, $cmd, @args) = @_;
    $cmd =~ tr/ /_/;
    my $method = "xgroup_" . lc($cmd);
    return $self->$method(@args);
}

=head2 xinfo

=cut

sub xinfo : method {
    my ($self, $cmd, @args) = @_;
    $cmd =~ tr/ /_/;
    my $method = "xinfo_" . lc($cmd);
    return $self->$method(@args);
}

1;

__END__

=head1 AUTHOR

Tom Molesworth <TEAM@cpan.org>

=head1 LICENSE

This was autogenerated from the official Redis documentation, which is published
under the L<Creative Commons Attribution-ShareAlike license|https://github.com/redis/redis-doc/blob/master/LICENSE>.

The Perl code is copyright Tom Molesworth 2015-2021, and licensed under the same
terms as Perl itself.

