package Promises6::Promise;
use Evo::Base -base;

has [qw(builder)];
has deferred => sub { Carp::croak 'build me using deferred object' };

sub then($self, @subs) {
  my $d = $self->builder->deferred;
  my $listener = $self->builder->listener(cbs => \@subs, deferred => $d);

  $self->deferred->subscribe($listener);
  $d->promise;
}

1;

# ABSTRACT: A promise.

__END__

=pod

=encoding UTF-8

=head1 NAME

Promises6::Promise - A promise.

=head1 VERSION

version 0.001

=head1 SYNOPSIS

  use Evo::Base -strict;
  use Promises6::Builder;

  my $bldr     = Promises6::Builder->new();
  my $deferred = $bldr->deferred;
  my $promise  = $deferred->promise;
  my $promise2 = $promise->then(sub($v) { say "resolved $v" });

  $deferred->resolve('foo');

=head1 DESCRIPTION

This class represents a promise. Each promise has one L</"deferred"> attribute,
while C<deferred> can have one or many promises.

=head1 ATTRIBUTES

=head2 deferred

A sheperd for our promise

=head2 builder

An instance of L<Promises6::Builder>

=head1 METHODS

=head2 then

  my $on_fulfill  = sub($v) { say "fulfilled with $v" };
  my $on_reject   = sub($r) { say "rejected with $r" };
  my $on_progress = sub($p) { say "progress notification $p" };

  $promise->then($on_fulfill, $on_reject, $on_progress);

  $deferred->notify('50%');
  $deferred->notify('100%');
  $deferred->resolve('foo');

This method is the main method in promises. The first argument will be called
as a function with the result of the promise when promise have changed a state
to fulfilled, the second - to rejected. The third will be called when promise
send notify messages.

C<then> returns a new C<promise>, the result of that invocation will be resolved
by that promise.

If arguments are not functions, they will be ignored.

If invocation of the arguments causes an error, the promise will be rejected
with the exception (usually with the C<$@>), but I can bade a separete
exception class in the future.

Each argument won't be called more than once.

=head1 AUTHOR

alexbyk <alexbyk.com>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by alexbyk.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
