#
#
#  Copyright (c) 2003 Andrew W. Speer <andrew.speer@isolutions.com.au>. All rights 
#  reserved.
#
#  This file is part of WebDyne.
#
#  WebDyne is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
#  $Id: WebDyne.pm,v 1.75 2006/05/28 05:37:37 aspeer Exp $
#
package WebDyne;


#  Compiler Pragma
#
sub BEGIN	{ $^W=0 };
use strict	qw(vars);
use vars	qw($VERSION $REVISION %CGI_TAG_WEBDYNE @ISA $AUTOLOAD);
use warnings;
no  warnings	qw(uninitialized redefine);
use integer;


#  WebMod Modules, Constants
#
use WebMod::Base qw(:all);
use WebDyne::Constant;


#  External Modules
#
use Storable;
use HTTP::Status qw(is_success is_error RC_OK);
use Fcntl;
use Tie::IxHash;
use Time::HiRes qw(time);
use Digest::MD5 qw(md5_hex);
use File::Spec::Unix;
use overload;


#  Inherit from the Compile module, not loaded until needed though.
#
@ISA=qw(WebDyne::Compile);


#  Version information. Must be all on one line
#
$VERSION = eval { require WebDyne::VERSION; do $INC{'WebDyne/VERSION.pm'}};


#  Revision info, by CVS
#
#
$REVISION= (qw$Revision: 1.75 $)[1];


#  Debug load
#
debug("%s loaded, version $VERSION, revision $REVISION", __PACKAGE__);


#  Shortcut error handler, save using ISA;
#
require WebDyne::Err;
*err_html=\&WebDyne::Err::err_html || *err_html;


#  Init log file for tracking render times
#
our $Log_or=WebMod::Log->new({

    filename	=>	$WEBDYNE_RENDER_LOG_FN,
    handler	=>	$WEBDYNE_RENDER_LOG_HNDLR,
    mode	=>	O_WRONLY|O_CREAT|O_APPEND,
    autoflush	=>	1,
    timestamp	=>	1,

});


#  Our webdyne "special" tags
#
%CGI_TAG_WEBDYNE=map { $_=>1 } (

    'block',
    'perl',
    'subst',
    'dump',

   );


#  Var to hold package wide hash, for data shared across package
#
my %Package;


#  Do some class wide initialisation 
#
&init_class();


#  Handle internal die's gracefully if possible
#
$SIG{__DIE__} =sub { return err(@_) };
$SIG{__WARN__}=sub { return err(@_) } if $WEBDYNE_WARNINGS_FATAL;


#  All done. Positive return
#
1;


#==================================================================================================


sub handler : method {


    #  Get self ref/class, request ref
    #
    my ($self, $r, $param_hr)=@_;


    #  Start timer
    #
    my $time=Time::HiRes::time();


    #  Work out class and correct self ref
    #
    my $class=ref($self) || do {


	#  Need new self ref, as self is actually class
	#
	my %self=(

	    _time	    =>  $time,
	    _r		    =>	$r,
	    %{delete $self->{'_self'}},

	   );
	$self=bless \%self, $self;
	ref($self);


    };


    #  Debug
    #
    debug("in WebDyne::handler. class $class, self $self, r $r, param_hr %s",
	  Dumper($param_hr));


    #  Init package if not yet done.
    #
    $Package{_init_handler} ||= (do {
	$r=$self->init_handler($r, $param_hr) || return $self->err_html()} && 1);


    #  Skip all processing if header request only and Apache2
    #
    if ($r->header_only()) { return &head_request($r) };


    #  Debug
    #
    debug("enter handler, r $r, is_main %s, location %s file %s, param %s",
	  $r->is_main(), $r->location(), $r->filename(), Dumper($param_hr));


    #  Get full path, mtime, inode of source file
    #
    my $srce_pn=$r->filename() ||
	return $self->err_html('unable to get request filename');
    my $srce_mtime=(stat($srce_pn))[9] || do {

        #  File not found, we don't want to handle this anymore ..
        #
        return &Apache::DECLINED;

    };
    debug("srce_pn $srce_pn, srce_mtime (real) $srce_mtime");


    #  Used to use inode as unique identifier for file in cache, but that
    #  did not take into account the fact that the same file may have diff
    #  Apache locations (and thus WebDyne::Chain) handlers for the same
    #  physical file. So we now use an md5 hash of location and file name,
    #  but the var name is still "inode";
    #
    RENDER_BEGIN:
    my $srce_inode=($self->{'_inode'} ||= md5_hex($r->location, $srce_pn) ||
	return $self->err_html("could not get md5 for file $srce_pn, $!"));
    debug("srce_inode $srce_inode");


    #  Get "effective" source mtime, as may be a combination of things including
    #  template (eg menu) mtime. Here so can be subclassed by other handler like
    #  menu systems
    #
    debug("about to call source_mtime, self $self");
    $srce_mtime=${
        $self->source_mtime($srce_mtime) || return $self->err_html() } || $srce_mtime;
    debug("srce_pn $srce_pn, srce_mtime (computed) $srce_mtime");


    #  Var to hold pointer to cached metadata area, so we are not constantly
    #  dereferencing $Package{'_cache'}{$srce_inode};
    #
    my $cache_inode_hr=(
	$Package{'_cache'}{$srce_inode} ||= {

	    data	 =>	undef, # holds compiled representation of html/psp file
	    mtime	 =>	undef, # last modified time of the Storable disk cache file
	    nrun	 =>	undef, # number of times this page run by this mod_perl child
	    lrun	 =>	undef, # last run time of this page by this mod_perl child

	    # Created if needed
	    #
	    # meta	 =>  undef,  # page meta data, held in meta section or supplied by add-on modules
	    # eval_cr	 =>  undef,  # where anonymous sub's representing eval'd perl code within this page are held
	    # perl_init	 =>  undef,  # flags that perl code in __PERL__ block has been init'd (run once at page load) 

	}) || return $self->err_html('unable to initialize cache_inode_hr ref');



    #  Need to stat cache file mtime in case another process has updated it (ie via self->cache_compile(1)) call,
    #  which will make our memory cache stale. Would like to not have to do this stat one day, perhaps via shmem
    #  or similar check
    #
    my $cache_pn=File::Spec->catfile($WEBDYNE_CACHE_DN, $srce_inode);
    my $cache_mtime=((-f $cache_pn) && (stat(_))[9]);


    #  Test if compile/reload needed
    #
    if ($self->{'_compile'} || ($cache_inode_hr->{'mtime'} < $srce_mtime) || ($cache_mtime > $cache_inode_hr->{'mtime'})) {


	#  Debug
	#
	debug("compile/reload needed _compile %s, cache_inode_hr mtime %s, srce_mtime $srce_mtime",
	      $self->{'_compile'}, $cache_inode_hr->{'mtime'});


	#  Null out cache_inode to clear any flags
	#
	foreach my $key (keys %{$cache_inode_hr}) {
            $cache_inode_hr->{$key}=undef;
        }


	#  Try to clear/reset package name space if possible
	#
	eval {
	    require Symbol;
	    &Symbol::delete_package("WebDyne::${srce_inode}");
	} || do {
	    eval 1; #clear $@ after error above
	    my $stash_hr=*{"WebDyne::${srce_inode}::"}{HASH};
	    foreach (keys %{$stash_hr}) {
		undef *{"WebDyne::${srce_inode}::${_}"};
	    }
	    %{$stash_hr}=();
	    delete *WebDyne::{'HASH'}->{$srce_inode};
	};


	#  Debug
	#
	debug("srce_pn $srce_pn, cache_pn $cache_pn, mtime $cache_mtime");


	my $container_ar;
	if ($self->{'_compile'} || ($cache_mtime < $srce_mtime)) {


	    #  Debug
	    #
	    debug("compiling srce: $srce_pn, dest $cache_pn");


	    #  Recompile from source
	    #
	    eval { require WebDyne::Compile }
		|| return $self->err_html(
		    errsubst('unable to load WebDyne:Compile, %s', $@ || 'undefined error' ));


	    #  Source newer than compiled version, must recompile file
	    #
	    $container_ar=$self->compile({

		srce    =>	$srce_pn,
		dest    =>	$cache_pn,

	    }) || return $self->err_html();


	    #  Check for any unhandled errors during compile
	    #
	    errstr() && return $self->err_html();


	    #  Update mtime flag
	    #
	    $cache_mtime=(stat($cache_pn))[9] ||
		return $self->err_html("could not stat cache file '$cache_pn'");
	    $cache_inode_hr->{'mtime'}=$cache_mtime;


	}
	else {

	    #  Debug
	    #
	    debug("loading from disk cache");


	    #  Load from storeable file
	    #
	    $container_ar=Storable::lock_retrieve($cache_pn) ||
		retuern $self->err_html("Storable error when retreiveing cached file '$cache_pn', $!");


	    #  Update mtime flag
	    #
	    $cache_inode_hr->{'mtime'}=$cache_mtime;


	    #  Re-run perl-init for this node. Not done above because handled in compile if needed
	    #
	    if (my $meta_hr=$container_ar->[0]) {
		if (my $perl_ar=$meta_hr->{'perl'}) {
		    $self->perl_init($perl_ar) || return $self->err_html();
		}
	    }
	}


	#  Done, install into memory cache
	#
	if (my $meta_hr=$container_ar->[0] and $cache_inode_hr->{'meta'}) {

	    #  Need to merge meta info
	    #
	    map { $cache_inode_hr->{'meta'}{$_} ||= $meta_hr->{$_} } keys %{$meta_hr}

	}
	elsif ($meta_hr) {

	    #  No merge - just use from container
	    #
	    $cache_inode_hr->{'meta'}=$meta_hr;

        }
	$cache_inode_hr->{'data'}=$container_ar->[1];


    }


    #  Separate meta and actual data
    #
    my ($meta_hr, $data_ar)=@{$cache_inode_hr}{qw(meta data)};
    debug('meta_hr %s, ', Dumper($meta_hr));


    #  Custom handler ?
    #
    if (my $handler_ar=$meta_hr->{'handler'}) {
	my ($handler, $handler_param_hr)=ref($handler_ar) ? @{$handler_ar} : $handler_ar;
	if (ref($self) ne $handler) {
	    debug("passing to custom handler '$handler', param %s", Dumper($handler_param_hr));
	    unless ($Package{'_handler_load'}{$handler}) {
		debug("need to load handler '$handler' -  trying");
		(my $handler_fn=$handler)=~s/::/\//g;
		$handler_fn.='.pm';
		eval { require $handler_fn } ||
		    return $self->err_html("unable to load custom handler '$handler', $@");
		UNIVERSAL::can($handler, 'handler') ||
		    return $self->err_html("custom handler '$handler' does not seem to have a 'handler' method to call");
		debug('loaded OK');
		$Package{'_handler_load'}{$handler}++;
	    }
	    my %handler_param_hr=(%{$param_hr}, %{$handler_param_hr}, meta=>$meta_hr);
	    bless $self, $handler;
	    return &{"${handler}::handler"}($self, $r, \%handler_param_hr);
	}
    }


    #  Contain cache code ?
    #
    if ((my $cache=($self->{'_cache'} || $meta_hr->{'cache'})) && !$self->{'_cache_run_fg'}++) {
        debug("found cache routine $cache, adding to inode $srce_inode");
	my $cache_inode;
	my $eval_cr=$Package{'_eval_cr'}{'!'};
	if (ref($cache) eq 'CODE') {
	    my %param=(
		cache_cr    => $cache,
		srce_inode  => $srce_inode
	       );
	    $cache_inode=${
		$eval_cr->($self, undef, \%param, q[$_[1]->{'cache_cr'}->($_[0], $_[1]->{'srce_inode'})],  0) ||
		    return $self->err_html(errsubst(
			'error in cache code: %s', errstr() || $@ || 'no inode returned'));
	    }
	}
	else {
	    $cache_inode=${
		$eval_cr->($self, undef, $srce_inode, $cache,  0) ||
		    return $self->err_html(errsubst(
			'error in cache code: %s', errstr() || $@ || 'no inode returned'));
	    }
	}
	$cache_inode=$cache_inode ? md5_hex($srce_inode, $cache_inode) : $self->{'_inode'};

	#  Will probably make inodes with algorithm below some day so we can implement a "maxfiles type limit on
	#  the number of cache files generated. Not today though ..
	#
	#$cache_inode=$cache_inode ? $srce_inode .'_'. md5_hex($cache_inode) : $self->{'_inode'};
	debug("cache inode $cache_inode, compile %s", $self->{'_compile'});

 	if (($cache_inode ne $srce_inode) || $self->{'_compile'}) {
	    #  Using a cache file, different inode. 
	    #
	    debug("goto RENDER_BEGIN, inode node was $srce_inode, now $cache_inode");
	    $self->{'_inode'}=$cache_inode;
	    goto RENDER_BEGIN;
	    #return &handler($self,$r,$param_hr); #should work instead of goto for pendants
	}

   }


    #  Is it plain HTML which can be/is pre-rendered and stored on disk ? Note to self, leave here - should
    #  run after any cache code is run, as that may change inode.
    #
    my $html_sr;
    if ($self->{'_static'} || ($meta_hr && ($meta_hr->{'html'} || $meta_hr->{'static'}))) {
	my $cache_pn=File::Spec->catfile($WEBDYNE_CACHE_DN, $srce_inode);
	if ((-f "${cache_pn}.html") && ((stat(_))[9] >= $srce_mtime) && !$self->{'_compile'}) {
	    debug("returning pre-rendered file ${cache_pn}.html");
	    my $r_child=$r->lookup_file("${cache_pn}.html");
	    $r_child->content_type('text/html');
	    return $r_child->run();
	}
	else {
	    debug("storing html %s",  \$data_ar->[0]);
	    $r->register_cleanup(
		sub { &cache_html("${cache_pn}.html", ($meta_hr->{'static'} || $self->{'_static'}) ? $html_sr : \$data_ar->[0]) }
	       );
	    }
    }


    #  Send no-cache headers ?
    #
    if ($meta_hr->{'no_cache'} || $WEBDYNE_NO_CACHE) {
        $self->no_cache() || return $self->err_html()
    };


    #  Set default content type, can be overridden by render handler if
    #  needed
    #
    $r->content_type('text/html');


    #  Debug
    #
    #debug('about to render');


    #  Redirect print function to our own routine
    #
    my $select=select();
    tie (*WEBDYNE, 'WebDyne::TieHandle', $self) ||
	return $self->err_html("unable to tie output to 'WebDyne::TieHandle', $!");
    select WEBDYNE;


    #  Get the actual html
    #
    $html_sr=$self->render({ data=>$data_ar, param=>$param_hr }) || do {


	#  Our render routine returned an error. Debug
	#
	RENDER_ERROR:
	debug("render error $r, is_main %s", $r->is_main);


	#  Return error
	#
	select $select;
	return $self->err_html();


    };


    #  Done with redirect
    #
    select $select;


    #  Check for any unhandled errors during render.
    #
    debug('errstr after render %s', errstr());
    errstr() && return $self->err_html();


    #  Check for any blocks that user wanted rendered but were
    #  not present anywhere
    #
    if ($WEBDYNE_DELAYED_BLOCK_RENDER && (my $block_param_hr=delete $self->{'_block_param'})) {
 	my @block_error;
 	foreach my $block_name (keys %{$block_param_hr}) {
 	    unless (exists $self->{'_block_render'}{$block_name}) {
 		push @block_error, $block_name;
 	    }
 	}
 	if (@block_error) {
 	    return $self->err_html(
 		err('unable to locate block(s) %s for render', join(', ', map {"'$_'"} @block_error)))
 	}
    }



    #  Debug
    #
    #debug('render complete');


    #  If no error, status must be ok unless otherwise set
    #
    $r->status(RC_OK) unless $r->status();
    debug('r status set, %s', $r->status());


    #  Formulate header, calc length of return.
    #
    #  Modify to remove error checking - WebDyne::FakeRequest does not supply
    #  hash ref, so error generated. No real need to check
    #
    my $header_out_hr=$r->headers_out(); # || return err();
    my %header_out=(

	'Content-Length'    =>  length ${$html_sr},

    );
    map { $header_out_hr->{$_}=$header_out{$_} } keys %header_out;


    #  Debug
    #
    debug('sending header');


    #  Send header
    #
    $r->send_http_header();


    #  Print. Commented out version only seems to work in Apache 1/mod_perl1
    #
    #$r->print($html_sr);
    $MP2 ? $r->print(${$html_sr}) : $r->print($html_sr);
    RENDER_COMPLETE:


    #  Work out the form render time, log
    #
    my $time_render=sprintf('%0.4f', Time::HiRes::time()-$time);
    $Log_or->write("page $srce_pn render time , $time_render sec");
    debug("form $srce_pn render time $time_render");


    #  Update cache script frequency used, time used indicators, nrun=number
    #  of runs, lrun=last run time
    #
    $cache_inode_hr->{'nrun'}++;
    $cache_inode_hr->{'lrun'}=time();


    #  Do we need to do house cleaning on cache after this run ? If so
    #  add a perl handler to do it after we finish
    #
    if ($WEBDYNE_CACHE_CHECK_FREQ && $r->is_main() && !((my $nrun=++$Package{'_nrun'}) % $WEBDYNE_CACHE_CHECK_FREQ)) {


	#  Debug
	#
	debug("run $nrun times, scheduling cache clean") if $r->is_main();


	#  Yes, we need to clean cache after finished
	#
        $r->register_cleanup(
	    sub { &cache_clean($Package{'_cache'}) }
	   );


	#  Used to be sub { $self->cache_clean() }, but for some reason this
	#  made httpd peg at 100% CPU usage after cleanup. Removing $self ref
	#  fixed.
	#


    }
    else {


	#  Debug
	#
	debug("run $nrun times, no cache check needed");

    }



    #  Debug exit
    #
    debug("handler $r exit"); #, Dumper($self));


    #  Complete
    #
    HANDLER_COMPLETE:
    debug("handler $r exit, uri %s status %s, Apache::OK %s", $r->uri, $r->status, &Apache::OK);
    return &Apache::OK;


}


sub init_class {


    #  Try to load correct modules depending on Apache ver, taking special care
    #  with constants. This mess will disappear when this module is updated to
    #  only support MP2
    #
    if ($MP2) {

	eval {
	    require Apache2;
	    require Apache::compat;
	    require Apache::Const; Apache::Const->import(-compile => qw(OK DECLINED));
	} || eval {
	    require Apache2::compat;
	    require Apache2::Const; Apache2::Const->import(-compile => qw(OK DECLINED));
	};
	unless (UNIVERSAL::can('Apache','OK')) {
	    if (UNIVERSAL::can('Apache2::Const','OK')) {
		*Apache::OK=\&Apache2::Const::OK;
		*Apache::DECLINED=\&Apache2::Const::DECLINED;
	    }
	    elsif (UNIVERSAL::can('Apache::Const','OK')) {
		*Apache::OK=\&Apache::Const::OK;
		*Apache::DECLINED=\&Apache::Const::DECLINED;
	    }
	    else {
		*Apache::OK=sub { 0 } unless defined &Apache::OK;
		*Apache::DECLINED=sub { -1 } unless defined &Apache::DECLINED;
	    }
	}
    }
    else {
	eval {
	    require Apache::Constants; Apache::Constants->import(qw(OK DECLINED));
	    *Apache::OK=\&Apache::Constants::OK;
	    *Apache::DECLINED=\&Apache::Constants::DECLINED;
	}
    }


    #  If set, delete all old cache files at startup
    #
    if ($WEBDYNE_STARTUP_CACHE_FLUSH) {
	my @file_cn=glob(File::Spec->catfile($WEBDYNE_CACHE_DN, '*'));
	foreach my $fn (grep {/\w{32}(\.html)?$/} @file_cn) {
	    unlink $fn; #don't error here if problems, user will never see it
	}
    }


    #  Pre-compile some of the CGI functions we will need. Do here rather than in init
    #  so that can be executed at module load, and thus shared in memory between Apache
    #  children. Force run of start_ and end_ functions because CGI seems to lose them
    #  if not called at least once after compilation
    #
    require CGI;
    # CGI::->method is needed because perl 5.6.0 will use WebDyne::CGI->method instead of
    # CGI->method. CGI::->method makes it happy
    CGI::->import('-no_xhtml', '-no_sticky');
    my @cgi_compile=qw(:all area map unescapeHTML form col colgroup spacer nobr);
    CGI::->compile(@cgi_compile);
    foreach (grep { !/^:/ } @cgi_compile) { map { CGI::->$_ } ("start_${_}", "end_${_}") }


    #  Make errors non-fatal in WebMod::Err package
    #
    &WebMod::Err::errnofatal(1);


    #  Turn off XHTML in CGI. -no_xhtml should do it above, but this makes sure
    #
    $CGI::XHTML=0;
    $CGI::NOSTICKY=1;


    #  Alias request method
    #
    *WebDyne::r=\&WebDyne::request || *WebDyne::r;


    #  Add comment function to CGI, only called if user has commented out some
    #  HTML that includes a susbst type section, eg <!-- ${foo} -->
    #
    *{'CGI::~comment'}=sub {"<!--$_[1]-->"};


    #  Eval routine for eval'ing perl code in a non-safe way (ie hostile
    #  code could probably easily subvert us, as all operations are
    #  allowed, including redefining our subroutines etc).
    #
    my $eval_cr=sub {


	#  Get self ref
	#
	my ($self, $data_ar, $eval_param_hr, $eval_text, $index)=@_;


	#  Debug
	#
	my $inode=$self->{'_inode'} || 'ANON'; # Anon used when no inode present, eg wdcompile


	#  Get CGI vars
	#
	my $param_hr=($self->{'_eval_cgi_hr'} ||= do {

	    my $cgi_or=$self->{'_CGI'} || $self->CGI();
	    $cgi_or->Vars();

        });


	#  Only eval subroutine if we have not done already, if need to eval store in
	#  cache so only done once. Note how self is undefined before eval, stops me
	#  accidently using it inline - must do $self=shift() in inline code.
	#
	my $eval_cr=$Package{'_cache'}{$inode}{'eval_cr'}{$data_ar}{$index} ||= do {
	    #$Package{'_cache'}{$inode}{'perl_init'} ||= $self->perl_init();
	    no strict; my $self;
	    no integer;
	    eval("package WebDyne::${inode}; $WebDyne::WEBDYNE_EVAL_USE_STRICT; sub{$eval_text}") || return
		err("error in eval text '%s', error: $@", $eval_text);
	};
	#debug("eval done, eval_cr $eval_cr");


	#  Run eval
	#
	my $html_sr=eval {

	    #  The following line puts all CGI params in %_ during the eval so they are easy to
	    #  get to ..
	    local *_=$param_hr;
	    $eval_cr->($self, $eval_param_hr) 
	};
	if (!defined($html_sr) || $@) {


	    #  An error occurred - handle it and return.
	    #
	    return errstr() ? err() : err($@ || 'undefined return from inline code, or did not return non-zero/non-null value');

	}


	#  Array returned ? Convert if so
	#
	(ref($html_sr) eq 'ARRAY') && do {
	    $html_sr=\ join(undef, map { ref($_) ? ${$_} : $_ } @{$html_sr})
	};


        #  Any printed data ?
        #
        $self->{'_print_ar'} && do {
	    $html_sr=\ join(undef, grep {$_} map { ref($_) ? ${$_} : $_ } @{delete $self->{'_print_ar'}}) };


	#  Always return a scalar ref
	#
	return ref($html_sr) ? $html_sr : \$html_sr;


    };


    #  The code ref for the eval statement if using Safe module
    #
    my $eval_safe_cr=sub {


	#  Get self ref
	#
	my ($self, $data_ar, $eval_param_hr, $eval_text, $index)=@_;


	#  Inode
	#
	my $inode=$self->{'_inode'} || 'ANON'; # Anon used when no inode present, eg wdcompile


	#  Get CGI vars
	#
	my $param_hr=($self->{'_eval_cgi_hr'} ||= do {

	    my $cgi_or=$self->{'_CGI'} || $self->CGI();
	    $cgi_or->Vars();

        });

	#  Init Safe mode environment space
	#
	my $safe_or=$self->{'_eval_safe'} || do {
	    debug('safe init (eval_init)');
	    require Safe;
	    require Opcode;
	    Safe->new($inode);
	};
	$self->{'_eval_safe'} ||= do {
	    $safe_or->permit_only(@{$WEBDYNE_EVAL_SAFE_OPCODE_AR});
	    $safe_or;
	};


	#  Only eval subroutine if we have not done already, if need to eval store in
	#  cache so only done once
	#
	local *_=$param_hr;
	${ $safe_or->varglob('_self') } = $self;
	${ $safe_or->varglob('_eval_param_hr') } = $eval_param_hr;
	my $html_sr=$safe_or->reval("sub{$eval_text}->(\$::_self, \$::_eval_param_hr)", $WebDyne::WEBDYNE_EVAL_USE_STRICT) ||
	    return errstr() ? err() : err($@ || 'undefined return from Safe->reval()');


	#  Run through the same sequence as non-safe routine
	#
	if (!defined($html_sr) || $@) {


	    #  Error
	    #
	    return errstr() ? err() : err($@ || 'undefined return from inline code, or did not return true (1) value');

	}


	#  Array returned ? Convert if so
	#
	(ref($html_sr) eq 'ARRAY') && do {
	    $html_sr=\ join(undef, map { ref($_) ? ${$_} : $_ } @{$html_sr})
	};


        #  Any printed data ?
        #
        $self->{'_print_ar'} && do {
	    $html_sr=\ join(undef, grep {$_} map { ref($_) ? ${$_} : $_ } @{delete $self->{'_print_ar'}}) };


	#  Make sure we return a ref
	#
	return ref($html_sr) ? $html_sr : \$html_sr;


    };


    #  Hash eval routine, works similar to the above, but returns a hash ref
    #
    my $eval_hash_cr=sub {


	#  Get self ref, data_ar etc
	#
	my ($self, $data_ar, $eval_param_hr, $eval_text, $index)=@_;


	#  Get code ref from cache of possible, otherwise create
	#
	my $eval_cr=$Package{'_cache'}{$self->{'_inode'}}{'eval_hash_cr'}{$data_ar}{$index} ||= do {
	    eval("sub{$eval_text}") || return(err("$@"));
	};


	#  Create an indexed, tied hash ref and return it
	#
	tie (my %value, 'Tie::IxHash', $eval_cr->($self, $eval_param_hr));
	\%value;

    };


    #  Array eval routine, works similar to the above, but returns an array ref
    #
    my $eval_array_cr=sub {


	#  Get self ref, data_ar etc
	#
	my ($self, $data_ar, $eval_param_hr, $eval_text, $index)=@_;


	#  Get code ref from cache of possible, otherwise create
	#
	my $eval_cr=$Package{'_cache'}{$self->{'_inode'}}{'eval_array_cr'}{$data_ar}{$index} ||= do {
	    eval("sub{$eval_text}") || return(err("$@"));
	};


	#  Run the code and return an anon array ref
	#
	[$eval_cr->($self, $eval_param_hr)];

    };


    #  Init anon text and attr evaluation subroutines, store in class space
    #  for quick retrieval when needed, save redefining all the time
    #
    my %eval_cr=(

	'$' => sub {
	    (my $value=$_[2]->{$_[3]}) || do {
		if (!exists($_[2]->{$_[3]}) && $WEBDYNE_STRICT_VARS) {
		    return err("no '$_[3]' parameter value supplied, parameters are: %s", join(',', map {"'$_'"} keys %{$_[2]}))
		} };
	    #  Get rid of any overloading 
	    if (ref($value) && overload::Overloaded($value)) { $value="$value" }
	    return ref($value) ? $value : \$value },
	'@' => $eval_array_cr,
	'%' => $eval_hash_cr,
	'!' => $WEBDYNE_EVAL_SAFE ? $eval_safe_cr : $eval_cr,
	'+' => sub { return \ ($_[0]->{'_CGI'}->param($_[3])) },
	'*' => sub { return \ $ENV{$_[3]} },
	'^' => sub { my $m=$_[3]; my $r=$_[0]->{'_r'}; 
	    UNIVERSAL::can($r, $m) ? \$r->$m : err("unknown request method '$m'") }

       );


    #  Store in class name space
    #
    $Package{'_eval_cr'}=\%eval_cr;


    #  Override various Apache functions. Again is mess until modules supports mod_perl2
    #  only.
    #
    my $apache_request_rec=do {
	if ($MP2) {
	    $INC{'Apache2/RequestRec.pm'} ? 'Apache2::RequestRec' : 'Apache::RequestRec';
	}
	else {
	    'Apache';
	}
    };


    #  Fix r->notes function to work under mod_perl 2
    #
    my $apache_notes_cr=UNIVERSAL::can($apache_request_rec, 'notes');
    *{"${apache_request_rec}::notes"}=\&notes;
    $Package{'_apache_notes_cr'}=$apache_notes_cr;


    #  Fix r->headers_out function to always return main header object
    #
    my $apache_headers_out_cr=UNIVERSAL::can($apache_request_rec, 'headers_out');
    *{"${apache_request_rec}::headers_out"}=\&headers_out;
    $Package{'_apache_headers_out_cr'}=$apache_headers_out_cr;


    #  Fix r->content_type function to always work on main object
    #
    my $apache_content_type_cr=UNIVERSAL::can($apache_request_rec, 'content_type');
    *{"${apache_request_rec}::content_type"}=\&content_type;
    $Package{'_apache_content_type_cr'}=$apache_content_type_cr;





}


sub init_handler {


    #  Used to init package, move init code out of handler
    #
    my ($self, $r, $param_hr)=@_;
    debug("in init. self $self, r $r");


    #  Get fake request object if no request object supplied to
    #  us, eg running from command line, rather than within
    #  Apache
    #
    $r ||= do {


	#  Debug
	#
	debug("creating Apache::FakeRequest object");


	#  Yes, need fake request object
	#
	require WebDyne::FakeRequest;


	#  And create new object
	#
	my $r_fake=WebDyne::FakeRequest->new(

	    is_main	    => 1,
	    %{$param_hr},

	   ) || return err('unable to get WebDyne::FakeRequest object');


	#  Store away
	#
	$self->request($r_fake);


	#  Done
	#
	$r_fake;

    };



    #  Set init flag, so we are not run again
    #
    $Package{'_init_handler'}++;


    #  All done, return the request object
    #
    return $r;

}


sub cache_clean {


    #  Get cache_hr, only param supplied
    #
    my $cache_hr=shift();
    debug('in cache_clean');


    #  Values we want, either last run time (lrun) or number of times run
    #  (nrun)
    #
    my $clean_method=$WEBDYNE_CACHE_CLEAN_METHOD ? 'nrun' : 'lrun';


    #  Sort into array of inode values, sorted descending by clean attr
    #
    my @cache=sort { $cache_hr->{$b}{$clean_method} <=> $cache_hr->{$a}{$clean_method} }
	keys %{$cache_hr};
    debug('cache clean array %s', Dumper(\@cache));


    #  If > high watermark entries, we need to clean
    #
    if (@cache > $WEBDYNE_CACHE_HIGH_WATER) {


	#  Yes, clean
	#
	debug('cleaning cache');


	#  Delete excess entries
	#
	my @clean=map { delete $cache_hr->{$_} }  @cache[$WEBDYNE_CACHE_LOW_WATER..$#cache];


	#  Debug
	#
	debug('removed %s entries from cache', scalar @clean);

    }
    else {

	#  Nothing to do
	#
	debug('no cleanup needed, cache size %s less than high watermark %s',
	      scalar @cache, $WEBDYNE_CACHE_HIGH_WATER);

    }


    #  Done
    #
    return \undef;

}


sub head_request {


    #  Head request only
    #
    my $r=shift();


    #  Clear any handlers
    #
    $r->set_handlers( PerlHandler=>undef );


    #  Send the request
    #
    $r->send_http_header();


    #  Done
    #
    return &Apache::OK;

}


sub render {


    #  Convert data array structure into HTML
    #
    my ($self, $param_hr)=@_;


    #  If not supplied param as hash ref assume all vars are params to be subs't when
    #  rendering this data block
    #
    ref($param_hr) || ($param_hr={ param=>{ @_[1..$#_] } });


    #  Debug
    #
    debug('in render');
    #debug('render %s', Dumper($param_hr));


    #  Get node array ref
    #
    my $data_ar=$param_hr->{'data'} ||
	$self->{'_perl'}[0][$WEBDYNE_NODE_CHLD_IX] ||
	    return err('unable to get HTML data array');


    #  Debug
    #
    debug("render data_ar $data_ar %s", Dumper($data_ar));


    #  If block name spec'd register it now
    #
    $param_hr->{'block'} && (
	$self->render_block($param_hr) || return err());


    #  Get CGI object
    #
    my $cgi_or=$self->{'_CGI'} || $self->CGI() || 
	return err("unable to get CGI object from self ref");


    #  Any data params for this render
    #
    my $param_data_hr=$param_hr->{'param'};


    #  Recursive anon sub to do the render, init and store in class space
    #  if not already done, saves a small amount of time if doing many
    #  iterations
    #
    my $render_cr=$Package{'_render_cr'} ||= sub {


	#  Get self ref, node array etc
	#
	my ($render_cr, $self, $cgi_or, $data_ar, $param_data_hr)=@_;


	#  Get tag
	#
	my ($html_tag, $html_line_no)=
	    @{$data_ar}[$WEBDYNE_NODE_NAME_IX, $WEBDYNE_NODE_LINE_IX];
	my $html_chld;


	#  Debug
	#
	debug("render tag $html_tag, line $html_line_no");


	#  Get attr hash ref
	#
	my $attr_hr=$data_ar->[$WEBDYNE_NODE_ATTR_IX];


	#  If subst flag present, means we need to process attr values
	#
	if ($data_ar->[$WEBDYNE_NODE_SBST_IX]) {
	    $attr_hr=$self->subst_attr($data_ar, $attr_hr, $param_data_hr) ||
		return errsubst("error at line $html_line_no - %s", errstr());
	}


	#  If param present, use for sub-render
	#
	$attr_hr->{'param'} && ($param_data_hr=$attr_hr->{'param'});


	#  Process sub nodes to get child html data, only if not a perl tag or block tag
	#  though - they will choose when to render sub data. Subst is OK
	#
	if (!$CGI_TAG_WEBDYNE{$html_tag} || ($html_tag eq 'subst')) {


	    #  Not a perl tag, recurse through children and render them, building
	    #  up HTML from inside out
	    #
	    my @data_child_ar=$data_ar->[$WEBDYNE_NODE_CHLD_IX] ? @{$data_ar->[$WEBDYNE_NODE_CHLD_IX]} : undef;
	    foreach my $data_chld_ar (@data_child_ar) {


		#  Debug
		#
		debug('data_chld_ar %s', Dumper($data_chld_ar));


		#  Only recurse on children which are are refs, as these are sub nodes. A
		#  child that is not a ref is merely HTML text
		#
		if (ref($data_chld_ar)) {


		    #  It is a sub node, render recursively
		    #
		    $html_chld.=${
			($render_cr->($render_cr, $self, $cgi_or, $data_chld_ar, $param_data_hr) ||
			     return err())};
		    #$html_chld.="\n";

		}
		else {


		    #  Text node only, add text to child html string
		    #
		    $html_chld.=$data_chld_ar;

		}

	    }

	}
	else {

	    debug("skip child render, under $html_tag tag");

	}


	#  Debug
	#
	debug("html_chld $html_chld");


	#  Render *our* node now, trying to use most efficient/appropriated method depending on a number
	#  of factors
	#
	if ($CGI_TAG_WEBDYNE{$html_tag}) {


	    #  Debug
	    #
	    #debug("rendering webdyne tag $html_tag");


	    #  Special WebDyne tag, render using our self ref, not CGI object
	    #
	    my $html_sr=($self->$html_tag($data_ar, $attr_hr, $param_data_hr, $html_chld) ||
		return errsubst("error at line $html_line_no - %s", errstr()));


	    #  Debug
	    #
	    debug("CGI tag $html_tag render return $html_sr (%s)", Dumper($html_sr));


	    #  Return
	    #
	    return $html_sr;


	}
	elsif ($attr_hr) {


	    #  Normal CGI tag, with attributes and perhaps child text
	    #
	    return \ ($cgi_or->$html_tag(grep {$_} $attr_hr, $html_chld) ||
		return err("error at line $html_line_no - CGI tag '<$html_tag>' ".
			       'did not return any text'));

	}
	elsif ($html_chld) {


	    #  Normal CGI tag, no attributes but with child text
	    #
	    return \ ($cgi_or->$html_tag($html_chld) ||
		return err("error at line $html_line_no - CGI tag '<$html_tag>' ".
			       'did not return any text'));

	}
	else {


	    #  Empty CGI object, eg <hr>
	    #
	    return \ ($cgi_or->$html_tag() ||
	       return err("error at line $html_line_no - CGI tag '<$html_tag>' ".
		       'did not return any text'));

	}


    };


    #  At the top level the array may have completly text nodes, and no children, so
    #  need to take care to only render children if present.
    #
    my @html;
    foreach my $data_ar (@{$data_ar}) {


	#  Is this a sub node, or only text (ref means sub-node)
	#
	if (ref($data_ar)) {


	    #  Sub node, we call call render routine
	    #
	    push @html,
		${ $render_cr->($render_cr, $self, $cgi_or, $data_ar, $param_data_hr) || return err() };


	}
	else {


	    #  Text only, do not render just push onto return array
	    #
	    push @html, $data_ar;

	}
    }



    #  Return scalar ref of completed HTML string
    #
    debug('render exit, html %s', Dumper(\@html));
    return \ join(undef, @html);


};


sub redirect {


    #  Redirect render to different location
    #
    my ($self, $param_hr)=@_;


    #  Debug
    #
    debug('in redirect, param %s', Dumper($param_hr));


    #  If redirecting to a different uri, run its handler
    #
    if ($param_hr->{'uri'} || $param_hr->{'file'}) {


	#  Get HTML from subrequest
	#
	my $status=${ $self->subrequest($param_hr) ||
	    return err() };
	debug("redirect status $status");
	if ($status && !is_success($status)) {
	    return err();
	}


	#  GOTOs considered harmful - except here. Speed things up significantly, removes uneeded checks
	#  for redirects in render code etc.
	#
	goto HANDLER_COMPLETE;



    }
    else {


	#  HTML must be a param
	#
	my $html_sr=$param_hr->{'html'} || $param_hr->{'text'} ||
	    return err('no data supplied to redirect method');


	#  Print directly and shorcut return from render routine with non-harmful GOTO. Should
	#  always be SR, but be generous.
	#
	my $r=$self->r() || return err();
	$r->print(ref($html_sr) ? ${$html_sr} : $html_sr);
	goto RENDER_COMPLETE;


    }


}


sub subrequest {


    #  Redirect render to different location
    #
    my ($self, $param_hr)=@_;


    #  Debug
    #
    debug('in subrequest %s', Dumper($param_hr));


    #  Get request object, var for subrequest object
    #
    my ($r, $cgi_or)=map { $self->$_() || return err("unable to run '$_' method") } qw(request CGI);
    my $r_child;


    #  If redirecting to a different uri, run its handler
    #
    if (my $uri=$param_hr->{'uri'}) {


	#  Get a new request object
	#
	$r_child=$r->lookup_uri($uri) ||
	    return err('undefined lookup_uri error');


    }
    elsif (my $file=$param_hr->{'file'}) {

	#  Get cwd, make request absolute rel to cwd if no dir given.
	#
	my $dn=(File::Spec->splitpath($r->filename()))[1];
	my $file_pn=File::Spec->rel2abs($file, $dn);


	#  Get a new request object
	#
	$r_child=$r->lookup_file($file_pn) ||
	    return err('undefined lookup_file error');

    }
    else {


	#  Must be one or other
	#
	return err('must specify file or uri for subrequest');

    }


    #  Save child object, else cleanup handlers will be run when
    #  we exit and r_child is destroyed, but before r (main) is
    #  complete.
    #
    #  UPDATE no longer needed, leave here as reminder though ..
    #
    #push @{$self->{'_r_child'}},$r_child;


    #  Unless OK, return err
    #
    my $status=$r_child->status();
    debug("r_child status return: $status");
    if (($status && !is_success($status)) || (my $errstr=errstr())) {
	if ($errstr) {
	    return errsubst(
		"error in status phase of subrequest to '%s': $errstr",
		$r_child->uri() || $param_hr->{'file'}
	       )
	}
	else {
	    return err(
		"error in status phase of subrequest to '%s', return status was $status",
		$r_child->uri() || $param_hr->{'file'}
	       )
	}
    };


    #  Debug
    #
    debug('cgi param %s', Dumper($param_hr->{'param'}));


    #  Set up CGI with any new params
    #
    while (my($param, $value) = each %{$param_hr->{'param'}}) {


	#  Add to CGI
	#
	$cgi_or->param($param, $value);
	debug("set cgi param $param, value $value");


    }


    #  Debug
    #
    debug('about to call child handler with params %s', Dumper($param_hr->{'param'}));


    #  Update self with new request object, run content generation phase. Check for success here
    #  also.
    #
    $status=$r_child->run();
    debug("r_child run return: $status");
    if (($status && !is_success($status)) || (my $errstr=errstr())) {
	if ($errstr) {
	    return errsubst(
		"error in run phase of subrequest to '%s': $errstr",
		$r_child->uri() || $param_hr->{'file'}
	       )
	}
	else {
	    return err(
		"error in run phase of subrequest to '%s', return status was $status",
		$r_child->uri() || $param_hr->{'file'}
	       )
	}
    };


    #  Done
    #
    return \$status;


}


sub render_block {


    #  Render a <block> section of HTML
    #
    my ($self, $param_hr)=@_;


    #  Has user only given name as param
    #
    ref($param_hr) || ($param_hr={ name=>$param_hr, param=>{@_[2..$#_]} });


    #  Get block name
    #
    my $name=$param_hr->{'name'} || $param_hr->{'block'} ||
	return err('no block name specified');


    #  Get current data block
    #
    my $data_ar=$self->{'_perl'}[0] ||
	return err("unable to get current data node");


    #  Find block name
    #
    #my $data_block_ar;
    my @data_block_ar;


    #  Debug
    #
    debug("render_block self $self, name $name, data_ar $data_ar");


    #  Have we seen this search befor ?
    #
    unless (exists($self->{'_block_cache'}{$name})) {


	#  No, search for block
	#
	debug("searching for node $name in data_ar");


	#  Do it
	#
	my $data_block_all_ar=$self->find_node({

	    data_ar	    =>  $data_ar,
	    tag		    =>	'block',
	    all_fg	    =>	1,

	}) || return err();


	#  Debug
	#
	debug('find_node returned %s', join('*', @{$data_block_all_ar}));


	#  Go through each block found and svae in block_cache
	#
	foreach my $data_block_ar (@{$data_block_all_ar}) {


	    #  Get block name
	    #
	    my $name=$data_block_ar->[$WEBDYNE_NODE_ATTR_IX]->{'name'};
	    debug("looking at block $data_block_ar, name $name");


	    #  Save
	    #
	    #$self->{'_block_cache'}{$name}=$data_block_ar;
	    push @{$self->{'_block_cache'}{$name} ||= []}, $data_block_ar;


	}


	#  Done, store
	#
	@data_block_ar=@{$self->{'_block_cache'}{$name}};


    }
    else {


	#  Yes, set data_block_ar to whatever we saw before, even if it is
	#  undef
	#
	@data_block_ar=@{$self->{'_block_cache'}{$name}};


	#  Debug
	#
	debug("retrieved data_block_ar @data_block_ar for node $name from cache");


    }


    #  Debug
    #
    #debug("set block node to $data_block_ar %s", Dumper($data_block_ar));


    #  No data_block_ar ? Could not find block - remove this line if global block
    #  rendering is desired (ie blocks may lay outside perl code calling render_bloc())
    #
    unless (@data_block_ar) {
	return err("could not find block '$name' to render") unless $WEBDYNE_DELAYED_BLOCK_RENDER;
    }


    #  Now, was it set to something ?
    #
    my @html_sr;
    foreach my $data_block_ar (@data_block_ar) {


	#  Debug
	#
	debug("rendering block name $name, data $data_ar with param %s", Dumper($param_hr->{'param'}));


	#  Yes, Get HTML for block immedialtly
	#
	my $html_sr=$self->render({

	    data	=>  $data_block_ar->[$WEBDYNE_NODE_CHLD_IX],
	    param	=>  $param_hr->{'param'},

	}) || return err();


	#  Debug
	#
	debug("block $name rendered HTML $html_sr %s, pushing onto name $name, data_ar $data_block_ar", ${$html_sr});


	#  Store away for this block
	#
	push @{$self->{'_block_render'}{$name}{$data_block_ar} ||= []}, $html_sr;


	#  Store 
	#
	push @html_sr, $html_sr;


    }
    if (@html_sr) {


	#  Return scalar or array ref, depending on number of elements
	#
	#debug('returning %s', Dumper(\@html_sr));
	return $#html_sr ? $html_sr[0]: \@html_sr;

    }
    else {


	#  No, could not find block below us, store param away for later
	#  render
	#
	push @{$self->{'_block_param'}{$name} ||=[]},$param_hr->{'param'};


	#  Debug
	#
	debug("block $name not found in tree, storing params for later render");


	#  Done, return undef at this stage
	#
	return \undef;

    }


}


sub block {


    #  Called when we encounter a <block> tag
    #
    my ($self, $data_ar, $attr_hr, $param_data_hr, $text)=@_;
    debug("in block code, data_ar $data_ar");


    #  Get block name
    #
    my $name=$attr_hr->{'name'} ||
	return err('no block name specified');
    debug("in block, looking for name $name, attr given %s", Dumper($attr_hr));


    #  Only render if registered, do once for every time spec'd
    #
    if (exists ($self->{'_block_render'}{$name}{$data_ar})) {


	#  The block name has been pre-rendered - return it
	#
	debug("found pre-rendered block $name");


	#  Var to hold render result
	#
	my $html_ar=delete $self->{'_block_render'}{$name}{$data_ar};


	#  Return result as a single scalar ref
	#
	return \ join(undef, map {${$_}} @{$html_ar});


    }
    elsif (exists ($self->{'_block_param'}{$name})) {


	#  The block params have been registered, but the block itself was
	#  not yet rendered. Do it now
	#
	debug("found block param for $name in register");


	#  Var to hold render result
	#
	my @html_sr;


	#  Render the block for as many times as it has parameters associated
	#  with it, eg user may have called ->render_block several times in
	#  their code
	#
	foreach my $param_data_block_hr (@{$self->{'_block_param'}{$name}}) {


	    #  If no explicit data hash, use parent hash - not sure how useful
	    #  this really is
	    #
	    $param_data_block_hr ||= $param_data_hr;


	    #  Debug
	    #
	    debug("about to render block $name, param %s", Dumper($param_data_block_hr));


	    #  Render it
	    #
  	    push @html_sr, $self->render({

  		data	=> $data_ar->[$WEBDYNE_NODE_CHLD_IX],
  		param	=> $param_data_block_hr

  	       }) || return err();

	}


	#  Return result as a single scalar ref
	#
	return \ join(undef, map {${$_}} @html_sr);

    }
    elsif ($attr_hr->{'display'}) {


	#  User wants block displayed normally
	#
	return $self->render({

	    data	=> $data_ar->[$WEBDYNE_NODE_CHLD_IX],
  	    param	=> $param_data_hr

	   }) || err();

    }
    else {


	#  Block name not registered, therefore do not render - return
	#  blank
	#
	return \undef;

    }


}


sub perl {


    #  Called when we encounter a <perl> tag
    #
    my ($self, $data_ar, $attr_hr, $param_data_hr, $text)=@_;
    #debug("rendering perl tag in block $data_ar, attr %s");


    #  If inline, run now
    #
    if (my $perl_code=$attr_hr->{'perl'}) {


	#  May be inline code params to supply to this block
	#
	my $perl_param_hr=$attr_hr->{'param'};


	#  Run the same code as the inline eval (!{! ... !}) would run,
	#  for consistancy
	#
	return $Package{'_eval_cr'}{'!'}->($self, $data_ar, $perl_param_hr, $perl_code) ||
	    err();


    }
    else {


	#  Not inline, must want to call a handler, get method and caller
	#
	#my $function=join('::', grep {$_} @{$attr_hr}{qw(package class method)}) ||
	my $function=join('::', grep {$_} map { exists($attr_hr->{$_}) && $attr_hr->{$_}} qw(package class method)) ||
	    return err('could not determine perl routine to run');


	#  Try to get the package name as an array, pop the method off
	#
	my @package=split(/\:+/, $function);
	my $method=pop @package;


	#  And return package
	#
	my $package=join('::', grep {$_} @package);


	#  Debug
	#
	debug("perl package $package, method $method");


	#  If no method by now, dud caller
	#
	$method ||
	    return err("no package/method in perl block");


        #  If the require fails, we want to catch it in an eval
        #  and return a meaningful error message. BTW this is an
	#  order of magnitued faster than doing eval("require $package");
        #
        debug("about to require $package") if $package;
        my $package_fn=join('/', @package).'.pm';
        $package && !$INC{$package_fn} && (
	    eval { require $package_fn } ||
		return errsubst(
		    "error loading package '$package', %s", errstr() || $@ || 'undefined error')
	       );
	debug("package $package loaded OK");


	#  Push data_ar so we can use it if the perl routine calls self->render(). render()
	#  then has to "know" where it is in the data_ar structure, and can get that info
	#  here.
	#
	#unshift @{$self->{'_perl'}}, $data_ar->[$WEBDYNE_NODE_CHLD_IX];
	unshift @{$self->{'_perl'}}, $data_ar;


	#  Run the eval code to get HTML
	#
	my $html_sr=$Package{'_eval_cr'}{'!'}->($self, $data_ar, $attr_hr->{'param'}, "&${function}") ||
	    return err();


	#  Debug
	#
	#debug('perl eval return %s', Dumper($html_sr));


	#  Modify return value if we were returned an array. COMMENTED OUT - is done in eval
	#
	#(ref($html_sr) eq 'ARRAY') && do {
	#    $html_sr=\ join(undef, map { ref($_) ? ${$_} : $_ } @{$html_sr})
	#};


	#  Unless we have a scalar ref by now, the eval returned the
	#  wrong type of value.
	#
	(ref($html_sr) eq 'SCALAR') ||
	    return err("error in perl method '$method': code did not return ".
			   'a SCALAR ref value.');


        #  Any printed data ?  COMMENTED OUT - is done in eval
        #
	#$self->{'_print_ar'} && do {
	#    $html_sr=\ join(undef, grep {$_} map { ref($_) ? ${$_} : $_ } @{delete $self->{'_print_ar'}}) };


	#  Pop perl data_ar ref from stack
	#
	pop @{$self->{'_perl'}};


	#  And return scalar val
	#
	return $html_sr

    }

}


sub perl_init {


    #  Init the perl package space for this inode
    #
    {
	my ($self, $perl_ar, $inode)=@_;
	$inode ||= $self->{'_inode'} || 'ANON';	#ANON used when run from command line


	#  Only run once
	#
	debug("perl_init inode $inode");
	#$Package{'_cache'}{$inode}{'perl_init'}++ && return \undef;
	debug("init perl code $perl_ar, %s", Dumper($perl_ar));
	*{"WebDyne::${inode}::err"}=\&WebMod::Err::err;
	*{"WebDyne::${inode}::self"}=sub {$self};
	*{"WebDyne::${inode}::AUTOLOAD"}=sub { die("unknown function $AUTOLOAD") };

	@_=($self, $perl_ar, $inode);

    }


    #  Try not to use named vars, so not present in eval package
    #
    for (0 .. $#{$_[1]}) {


	#  Do not execute twice
	#
	$_=$_[1]->[$_]; # Get scalar ref of perl code to execute.
	debug("looking at perl code $_");
	$Package{'_cache'}{$_[2]}{'perl_init'}{$_}++ && next;
	debug("executing perl code $_");

	#  Wrap in anon CR, eval for syntax
	#
	if ($WEBDYNE_EVAL_SAFE) {

	    #  Safe mode, vars don't matter so much
	    #
	    my $self=$_[0];
	    my $safe_or=$self->{'_eval_safe'} || do {
		debug('safe init (perl_init)');
		require Safe;
		require Opcode;
		Safe->new($self->{'_inode'});
	    };
	    $self->{'_eval_safe'} ||= do {
		$safe_or->permit_only(@{$WEBDYNE_EVAL_SAFE_OPCODE_AR});
		$safe_or;
	    };
	    $safe_or->reval(${$_}, $WebDyne::WEBDYNE_EVAL_USE_STRICT) || do {
		undef *{"WebDyne::$_[2]::self"};
		if (errstr()) {
		    return errsubst("error in __PERL__ block: %s", errstr());
		}
		elsif ($@) {
		    return errsubst("error in __PERL__ block: $@");
		}
	    };

	    #  Make sure not changed
	    #
	    $_[0]=$self;

	}
	else {
	    my $eval_cr=eval("package WebDyne::$_[2]; $WebDyne::WEBDYNE_EVAL_USE_STRICT; ${$_}") || do {
		undef *{"WebDyne::$_[2]::self"};
		if (errstr()) {
		    return errsubst("error in __PERL__ block: %s", errstr());
		}
		elsif ($@) {
		    return errsubst("error in __PERL__ block: $@");
		}
	    };
	}


    }


    #  Done
    #
    undef *{"WebDyne::$_[2]::self"};
    debug('perl_init complete');
    \undef;

}


sub subst {


    #  Called to eval text block, replace params
    #
    my ($self, $data_ar, $attr_hr, $param_data_hr, $text)=@_;


    #  Debug
    #
    #debug("eval $text %s", Dumper($param_data_hr));


    #  Get eval code refs for subst
    #
    my $eval_cr=$Package{'_eval_cr'} ||
	return err('unable to get eval code ref table');


    #  Var to hold result
    #
    my $text_subst=$text;


    #  Do we have to replace something in the text, look for pattern. We
    #  should always find something, as subst tag is only inserted at
    #  compile time in front of text with one of theses patterns
    #
    my $index;
    while ($text=~/([$|!|+|*|^]{1})\{([$|!|+|*|^]?)(.*?)\2\}/gs) {


	#  Yes, Save
	#
	my ($oper, $excl, $eval_text)=($1,$2,$3);
	debug("subst hit on text $text, oper $oper excl $excl text $eval_text");


	#  Run the appropriate eval
	#
	my $eval_sr=(

	    $eval_cr->{$oper} || return err("unknown eval operator, '$oper'")

	   )->($self, $data_ar, $param_data_hr, $eval_text, $index++) || do {

	       my $fragment=(length($text) > 80) ? substr($text,0,80) . '...' : $text;
	       $fragment=~s/^\n*//;

	       return errsubst(

		   "eval error in fragment '$fragment', error was: %s",

		   errstr())

	   };


	# Should be a scalar ref
	#
	unless ((my $ref=ref($eval_sr)) eq 'SCALAR') {
	    return err("eval of '$eval_text' returned $ref ref, should return SCALAR ref");
	}



	#  Probably should have something now
	#
	if (!defined(${$eval_sr}) && $WEBDYNE_STRICT_DEFINED_VARS) {
	    return err("eval of '$eval_text' returned no value")
	}



	#  Work out what we are replacing, do it
	#
	my $eval_expr="$oper\{${excl}${eval_text}${excl}\}";
	$text_subst=~s/\Q$eval_expr\E/${$eval_sr}/g;


    }


    #  Debug
    #
    #debug("return $text_subst");


    #  Done
    #
    return \$text_subst;


}


sub subst_attr {


    #  Called to eval tag attributes
    #
    my ($self, $data_ar, $attr_hr, $param_hr)=@_;


    #  Debug
    #
    #debug('subst_attr %s', Dumper({%{$attr_hr}, perl=>undef}));


    #  Get eval code refs for subst
    #
    my $eval_cr=$Package{'_eval_cr'} ||
	return err('unable to get eval code ref table');


    #  Hash to hold results
    #
    my %attr=%{$attr_hr};


    #  Go through each attribute and value
    #
    my $attr_ix=0;
    while ( my($attr, $value)=each %attr ) {


	#  Skip perl attr, as that is perl code, do not do any
	#  regexp on perl code, as we will probably botch it
	#
	next if ($attr eq 'perl');


	#  Do we have to replace something in the attr value
	#
	while ($value=~/([$|@|%|!|+]{1})\{([$|@|%|!|+]?)(.*?)\2\}/gs) {


	    #  Yes, Save
	    #
	    my ($oper, $excl, $eval_text)=($1,$2,$3);
	    #debug("value $value, oper $oper, excl $excl, eval_text $eval_text");


	    #  If we had a hit on the ` chars, get rid of them
	    #
	    $2 && do { $value=~s/\`//g };


	    #  Do the appropriate eval
	    #
	    my $eval_return=(

		$eval_cr->{$oper} || return err("unknown eval operator, '$oper'")

	       )->($self, $data_ar, $param_hr, $eval_text, $attr_ix++) || do {

		   return errsubst (

		       "eval error in code fragment '$value', error was: %s", errstr() );

	       };


	    #  Debug
	    #
	    #debug("eval_return $eval_text=>$eval_return, %s", Dumper($eval_return));


	    #  If value_eval is a ref, get the ref text. No good showing a
	    #  scalar ref in a text field
	    #
	    if (ref($eval_return) eq 'SCALAR') { 

	    	$eval_return=${$eval_return};
		my $eval_expr="$oper\{${excl}${eval_text}${excl}\}";
		if ($value ne $eval_expr) {
			#debug("need to subst $eval_expr in $value");
			($_=$value)=~s/\Q$eval_expr\E/$eval_return/g;
			$eval_return=$_;
		}
		#else {
			#debug("value $value = eval_expr $eval_expr, no work needed");

		#}
		#debug("scalar adjust return to $eval_return");

	    }


	    #  Replace the attr value
	    #
	    $attr{$attr}=$eval_return;
	    $value=$eval_return;

	}
    }


    #  Debug
    #
    #debug('returning attr hash %s', Dumper({%attr, perl=>undef }));


    #  Return new attribute hash
    #
    \%attr;

}


sub find_node {


    #  Find a particular node in the tree
    #
    my ($self, $param_hr)=@_;


    #  Get max depth we can descend to, zero out in params
    #
    my ($data_ar, $tag, $attr_hr, $depth_max, $prnt_fg, $all_fg)=@{$param_hr}{
	qw(data_ar tag attr_hr depth prnt_fg all_fg) };
    debug("find_node looking for tag $tag in data_ar %s", Dumper($data_ar));


    #  Array to hold results, depth
    #
    my ($depth, @node);


    #  Create recursive anon sub
    #
    my $find_cr=sub {


	#  Get params
	#
	my ($find_cr, $data_ar, $data_prnt_ar)=@_;


	#  Do we match at this level ?
	#
	if ((my $data_ar_tag=$data_ar->[$WEBDYNE_NODE_NAME_IX]) eq $tag) {


	    #  Match for tag name, now check any attrs
	    #
	    my $tag_attr_hr=$data_ar->[$WEBDYNE_NODE_ATTR_IX];


	    #  Debug
	    #
	    debug("tag '$tag' match, $data_ar_tag, checking attr %s", Dumper($tag_attr_hr));


	    #  Check for match
	    #
	    if ((grep { $tag_attr_hr->{$_} eq $attr_hr->{$_} } keys %{$tag_attr_hr}) ==
		    (keys %{$attr_hr})) {


		#  Match, debug
		#
		debug("$data_ar_tag attr match, saving");


		#  Tag name and attribs match, push onto node
		#
		push @node, $prnt_fg ? $data_prnt_ar : $data_ar;
		return $node[0] unless $all_fg;


	    }

	}
	else {

	    debug("mismatch on tag $data_ar_tag for tag '$tag'");

	}


	#  Return if out of depth
	#
	return if ($depth_max && (++$depth > $depth_max));


	#  Start looking through current node
	#
	my @data_child_ar=$data_ar->[$WEBDYNE_NODE_CHLD_IX] ? @{$data_ar->[$WEBDYNE_NODE_CHLD_IX]} : undef;
	foreach my $data_child_ar (@data_child_ar) {


	    #  Only check and/or recurse through children that are child nodes, (ie
	    #  are refs), ignor non-ref (text) nodes
	    #
	    ref($data_child_ar) && do {


		#  We have a ref, recurse look for match
		#
		if (my $data_match_ar=$find_cr->($find_cr, $data_child_ar, $data_ar)){


		    #  Found match during recursion, return
		    #
		    return $data_match_ar unless $all_fg;

		}

	    }

	}

    };


    #  Start it running with our top node
    #
    $find_cr->($find_cr, $data_ar);


    #  Debug
    #
    debug('find complete, return node %s', \@node);


    #  Return results
    #
    return \@node;

}


sub delete_node {


    #  Delete a particular node from the tree
    #
    my ($self, $param_hr)=@_;


    #  Get max depth we can descend to, zero out in params
    #
    my ($data_ar, $node_ar)=@{$param_hr}{qw(data_ar node_ar) };
    debug("delete node $node_ar starting from data_ar $data_ar");


    #  Create recursive anon sub
    #
    my $find_cr=sub {


	#  Get params
	#
	my ($find_cr, $data_ar)=@_;


	#  Iterate through child nodes
	#
	foreach my $data_chld_ix (0 .. $#{$data_ar->[$WEBDYNE_NODE_CHLD_IX]}) {

	    my $data_chld_ar=$data_ar->[$WEBDYNE_NODE_CHLD_IX][$data_chld_ix] ||
	        return err("unable to get chld node from $data_ar");
            debug("looking at chld node $data_chld_ar");

	    if ($data_chld_ar eq $node_ar) {

	        #  Found node we want to delete. Get rid of it, all done
	        #
	        debug("match - splicing at chld $data_chld_ix from array %s", Dumper($data_ar));
	        splice(@{$data_ar->[$WEBDYNE_NODE_CHLD_IX]}, $data_chld_ix, 1);
	        return \1;

            }
            else {


                #  Not target node - recurse
                #
                debug("no match - recursing to chld $data_chld_ar");
                ${$find_cr->($find_cr, $data_chld_ar) || return err()} &&
                    return \1;

            }
        }


        #  All done, but no cigar
        #
        return \undef;

    };


    #  Start
    #
    return $find_cr->($find_cr, $data_ar) || err()

}


sub CGI {


    #  Accessor method for CGI object
    #
    return shift()->{'_CGI'} ||= do {

	#  Debug
	#
	debug('CGI init');


	#  Need to turn off XHTML generation - CGI wants to turn it on every time for
	#  some reason
	#
	$CGI::XHTML=0;
	$CGI::NOSTICKY=1;


        #  And create it
        #
        CGI::->new();

   };

}


sub request {


    #  Accessor method for Apache request object
    #
    my $self=shift();
    return @_ ? $self->{'_r'}=shift() : $self->{'_r'};

}


sub dump {


    #  Run the dump CGI dump routine. Is here because it produces different output each
    #  time it is run, and if not a WebDyne tag it would be optimised to static text by
    #  the compiler
    #
    my ($self, $data_ar, $attr_hr)=@_;
    return ($WEBDYNE_DUMP_FLAG || $attr_hr->{'force'} || $attr_hr->{'display'}) ? \$self->{'_CGI'}->Dump() : \undef;

}


sub notes {


    #  Replaces r->notes to ensure compat bewteen mod_perl 1 and 2 ways of doing it. *APACHE*
    #  func again
    #
    my $r=shift();
    while ($r->main()) { $r=$r->main() }
    debug("in Apache::notes, r $r, args @_");
    my $apache_notes_cr=$Package{'_apache_notes_cr'} ||
	return err('could not get apache_notes code ref from class');


    #  Do differently depending on mod_perl version
    #
    if ($MP2) {

	return (@_==2) ? $apache_notes_cr->($r)->set(@_) : $apache_notes_cr->($r)->get(@_);

    }
    else {

	$apache_notes_cr->($r, @_);

    }
}


sub headers_out {


    #  Replaces r->headers_out to always return main header object, because that is the only one we
    #  send
    #
    my $r=shift();


    #  Get original code ref
    #
    my $apache_headers_out_cr=$Package{'_apache_headers_out_cr'} ||
	return err('could not get apache_headers_out code ref from class');


    #  Make sure we have main request
    #
    if ((my $r_main=$r->main() || $r) ne $r) {


	#  OK, did not have main request before. Got it now
	#
	$r=$r_main;

    }

    return $apache_headers_out_cr->($r, @_)

}


sub content_type {


    #  Replaces r->content_type to always reflect main header object, because that is the only one we
    #  send
    #
    my $r=shift();
    debug("setting content type of $r to @_");


    #  Get original code ref
    #
    my $apache_content_type_cr=$Package{'_apache_content_type_cr'} ||
	return err('could not get apache_content_type code ref from class');


    #  Make sure we have main request
    #
    if ((my $r_main=$r->main() || $r) ne $r) {


	#  OK, did not have main request before. Got it now
	#
	$r=$r_main;

    }

    return $apache_content_type_cr->($r, @_)

}


sub source_mtime {

    #  Get mtime of source file. Is a no-op here so can be subclassed by other handlers. We
    #  return undef, means engine will use original source mtime
    #
    \undef;

}


sub cache_mtime {

    #  Mtime accessor - will return mtime of srce inode (default), or mtime of supplied
    #  inode if given
    #
    my $self=shift();
    my $inode_pn=${
	$self->cache_filename(@_) || return err() };
    \ (stat($inode_pn))[9];

}


sub cache_filename {

    #  Get cache fq filename given inode or using srce inode if not supplied
    #
    my $self=shift();
    my $inode=@_ ? shift() : $self->{'_inode'};
    my $inode_pn=File::Spec->catfile($WEBDYNE_CACHE_DN, $inode);
    \$inode_pn;

}


sub cache_inode {

    #  Get cache inode string, or generate new unique inode
    #
    my $self=shift();
    @_&& ($self->{'_inode'}=md5_hex($self->{'_inode'}, $_[0]));

    #  See comment in handler section about future inode gen
    #
    #@_ && ($self->{'_inode'}.=('_'. md5_hex($_[0])));
    \$self->{'_inode'};

}


sub cache_html {

    #  Write an inode that is fully HTML out to disk to we dispatch it as a subrequest
    #  next time. This is a &register_cleanup callback
    #
    my ($cache_pn, $html_sr)=@_;
    debug("cache_html @_");


    #  No point || return err(), just die so is written to logs, otherwise go for it
    #
    my $cache_fh=IO::File->new($cache_pn, O_WRONLY|O_CREAT|O_TRUNC) ||
	die("unable to open cache file $cache_pn for write, $!");
    CORE::print $cache_fh ${$html_sr};
    $cache_fh->close();
    \undef;

}


sub cache_compile {

    #  Compile flag accessor - if set will force inode recompile, regardless of mtime
    #
    my $self=shift();
    @_ && ($self->{'_compile'}=shift());
    debug("cache_compile set to %s", $self->{'_compile'});
    \$self->{'_compile'};

}


sub no_cache {


    #  Nothing to do with page cache - sends 'no-cache' headers to client
    #
    my $self=shift();
    my $r=$self->r() || return err();
    my $header_out_hr=$r->headers_out(); # || return err();
    my %header_out=(

	'Cache-Control'     =>  'no-cache',
    	'Pragma'            =>  'no-cache',
    	'Expires'           =>  '-5'

    );
    map { $header_out_hr->{$_}=$header_out{$_} } keys %header_out;
    \undef;

}


sub filter {


    #  No op
    #
    my ($self, $data_ar)=@_;
    $data_ar;

}


sub meta {

    #  Return/read/update meta info hash
    #
    my ($self, @param)=@_;
    my $inode=$self->{'_inode'};
    debug("get meta data for inode $inode");
    my $meta_hr=$Package{'_cache'}{$inode}{'meta'} ||= (delete $self->{'_meta_hr'} || {});
    debug("exitsing meta $meta_hr %s", Dumper($meta_hr));
    if (@param==2) {
        return $meta_hr->{$param[0]}=$param[1];
    }
    elsif (@param) {
        return $meta_hr->{$param[0]};
    }
    else {
        return $meta_hr;
    }

}


sub static {


    #  Set static flag for this instance only. If all instances wanted
    #  set in meta data. This method used by WebDyne::Static module 
    #
    my $self=shift();
    $self->{'_static'}=1;


}


sub cache {

    #  Set cache handler for this instance only. If all instances wanted
    #  set in meta data. This method used by WebDyne::Cache module
    #
    my $self=shift();
    $self->{'_cache'}=shift() || 
        return err('cache code ref or method name must be supplied');

}


sub set_filter {

    #  Set cache handler for this instance only. If all instances wanted
    #  set in meta data. This method used by WebDyne::Cache module
    #
    my $self=shift();
    $self->{'_filter'}=shift() || 
        return err('filter name must be supplied');

}


sub set_handler {


    #  Set/return internal handler. Only good in __PERL__ block, after
    #  that is too late !
    #
    my $self=shift();
    my $meta_hr=$self->meta() || return err();
    @_ && ($meta_hr->{'handler'}=shift());
    \$meta_hr->{'handler'};


}


sub inode {


    #  Return inode name
    #
    my $self=shift();
    @_ ? $self->{'_inode'}=shift() : $self->{'_inode'};

}


sub DESTROY {


    #  Stops AUTOLOAD chucking wobbly at end of request because no DESTROY method
    #  found, logs total page cycle time
    #
    my $self=shift();


    #  Work out complete request cylcle time
    #
    debug("in destroy self $self, param %s", Dumper(\@_));
    my $time_request=sprintf('%0.4f', Time::HiRes::time()-$self->{'_time'});
    $Log_or->write("page request cycle time , $time_request sec");
    debug("page request cycle time , $time_request sec");


    #  Destroy object
    #
    %{$self}=();
    undef $self;

}


sub AUTOLOAD {


    #  Get self ref
    #
    my $self=$_[0];


    #  Get method user was looking for
    #
    my $method=(reverse split(/\:+/, $AUTOLOAD))[0];


    #  Vars for iterator, call stack
    #
    my $i; my @caller;


    #  Start going backwards through call stack, looking for package that can
    #  run method, pass control to it if found
    #
    my %caller;
    while (my $caller=(caller($i++))[0]) {
	next if ($caller{$caller}++);
	push @caller, $caller;
	if (my $cr=UNIVERSAL::can($caller, $method)) {
	    # POLLUTE is virtually useless - no speedup in real life ..
	    if ($WEBDYNE_AUTOLOAD_POLLUTE) {
		my $class=ref($self);
		*{"${class}::${method}"}=$cr;
	    }
	    #return $cr->($self, @_);
	    goto &{$cr}
	}
    }


    #  If we get here, we could not find the method in any caller. Error
    #
    err("unable to find method '$method' in call stack: %s", join(', ', @caller));
    goto RENDER_ERROR;

}


#  Package to tie select()ed output handle to so we can override print() command
#
package WebDyne::TieHandle;


sub TIEHANDLE {

    my ($class, $self)=@_;
    bless \$self, $class;
}


sub PRINT {

    my $self=shift();
    push @{${$self}->{'_print_ar'} ||= []}, @_;
    \undef;

}


sub PRINTF {

    my $self=shift();
    push @{${$self}->{'_print_ar'} ||= []}, sprintf(@_);
    \undef;

}



sub DESTROY {
}


sub UNTIE {
}


sub AUTOLOAD {
}



