#
#
#  Copyright (c) 2003 Andrew W. Speer <andrew.speer@isolutions.com.au>. All rights 
#  reserved.
#
#  This file is part of WebDyne::State::BerkeleyDB.
#
#  WebDyne::Session is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
#  $Id: State_BerkeleyDB.pm,v 1.3 2006/05/23 05:36:32 aspeer Exp $
#
package WebDyne::State::BerkeleyDB;


#  Compiler Pragma
#
use strict qw(vars);
use vars   qw($VERSION $REVISION @ISA);


#  Webmod Modules.
#
use WebMod::Base qw(:all);
use WebDyne::Constant;
use WebDyne::State::BerkeleyDB::Constant;


#  External modules
#
use File::Spec;
use BerkeleyDB;
use WebDyne::State;
@ISA=qw(WebDyne::State);


#  Version information in a format suitable for CPAN etc. Must be
#  all on one line
#
$VERSION = eval { require WebDyne::State::VERSION; do $INC{'WebDyne/State/VERSION.pm'}};


#  Release information
#
$REVISION= (qw$Revision: 1.3 $)[1];


#  Shortcut error handler.
#
require WebDyne::Err;
*err_html=\&WebDyne::Err::err_html || *err_html;


#  Debug
#
debug("%s loaded, version $VERSION, revision $REVISION", __PACKAGE__);


#  Class hash ref for package wide storage
#
my %Package;


#  And done
#
1;


#------------------------------------------------------------------------------


#  Common methods (same for all State handlers
#
sub import {


    #  Will only work if called from within a __PERL__ block in WebDyne
    #
    my $self_cr=UNIVERSAL::can(scalar caller, 'self') || return;
    my $self=$self_cr->() || return;
    $self->set_handler('WebDyne::Chain');
    my $meta_hr=$self->meta();
    push @{$meta_hr->{'webdynechain'}}, __PACKAGE__;


}


sub handler : method {

    my $self=shift();
    $Package{'_init'} ||=$self->state_init();
    $self->SUPER::handler(@_);


}


sub new {

    #  Only used by util programs, eg dump
    #
    my $class=shift();
    my $self=bless(\my %self, ref($class) || $class);
    $self->state_init();
    $self;

}


sub state_init {


    #  Get self ref
    #
    my $self=shift();
    debug('state init');


    #  Start by setting up BerkelyDB env
    #
    my $db_env=$self->_state_db_env() ||
	return err();


    #  Get canon name for statedb file
    #
    my $cn=$self->_state_cn() || return err();


    #  Options for db open
    #
    my %options=(

	-Filename       =>      $cn,
	-Flags          =>      DB_CREATE,
	-Mode           =>      0640,
	-Env            =>      $db_env,

       );


    #  Open db
    #
    my $state_tr=tie(my %state, 'BerkeleyDB::Hash', %options) ||
	return err($BerkeleyDB::Error);


    #  Store in class ref
    #
    $Package{'_session'}{'_state_tr'}=$state_tr;
    $Package{'_session'}{'_state_hr'}=\%state;


    #  Done
    #
    return 1;


}


sub state_fetch {


    #  Get a reference to the state database
    #
    my ($self, $param_hr)=@_;


    #  Debug
    #
    debug('in state_fetch');


    #  Either return the database handle held in our self var, or create
    #  a new one
    #
    my $state_hr=exists ($self->{'_state_hr'})
    	? $self->{'_state_hr'} || {}
	    : ($self->{'_state_hr'} = do {


		#  Debug
		#
		debug('going to disk for state_fetch');


		#  Get the session id
		#
		my $session_id=$param_hr->{'session_id'} ||
		    $self->session_id() || return err('unable to determine session_id');


		#  Get state hash ref
		#
		my $state_hr=$self->_state_hr() ||
		    return err('unable to get state hash ref');


		#  Get the state database for this session
		#
		my $state_freeze=($state_hr->{$session_id});


		#  Thaw
		#
		$state_freeze ? Storable::thaw($state_freeze) :
		    { _mtime    =>	time() };


	    });


    #  Debug
    #
    debug('state_fetch self %s returning %s', $self->{'_state_hr'}, Dumper($state_hr));


    #  If raw, return immediately
    #
    $param_hr->{'raw'} && (return $state_hr);


    #  Return single key if wished
    #
    if (my $key=ref($param_hr) ? $param_hr->{'key'} : $param_hr) {
	return exists ($state_hr->{'_data'}{$key})
	    ? \ ($state_hr->{'_data'}{$key})
		: undef

    }



    #  If user param, return user, otherwise data section
    #
    if ($param_hr->{'user'}) { return \ ($state_hr->{'_user'})   }
    else { return $state_hr=($state_hr->{'_data'} ||= {})     }


}


sub state_store {


    #  Get a reference to params
    #
    my ($self, $param_hr)=(shift, shift);


    #  Get the session id
    #
    my $session_id=$self->session_id() ||
	return err();
    debug("self $self, session_id $session_id");


    #  Get state db
    #
    my $state_hr=$self->state_fetch({ raw=>1, session_id=>$session_id }) ||
	return err();


    #  Save
    #
    if (!ref($param_hr)) {

	my %data=($param_hr, @_);
	$state_hr->{'_data'}=\%data;

    }
    elsif (my $data_hr=$param_hr->{'data'}) {

	$state_hr->{'_data'}=$data_hr;


    }
    elsif (my $user=$param_hr->{'user'}) {

	$state_hr->{'_user'}=$user;


    }
    elsif (my $key=$param_hr->{'key'}) {

	$state_hr->{'_data'}{$key}=$param_hr->{'value'};

    }
    elsif (ref($param_hr) eq 'HASH') {

	$state_hr->{'_data'}=$param_hr;

    }
    else {

	return err('nothing to store');

    }


    #  Debug
    #
    debug('state_hr now %s', Dumper($state_hr));


    #  If not already flagged to save to disk, do now
    #
    unless ($self->{'_state_save_fg'}++ || $self->{'_state_delete_fg'}) {


	#  Get request object
	#
	my $r=$self->request();
	debug("pushing state_save cleanup to request object $r");


	#  Push disk save callback, unless we have already done it
	#
	$r->register_cleanup(


	    sub {


		#  Debug
		#
		debug("in state_save cleanup $self data %s", Dumper($state_hr));


		#  Run the real save routine
		#
		$self->state_save({

		    data	    =>  $state_hr,
		    session_id	    =>	$session_id,


		});

	    });

    }


    #  Debug
    #
    debug('storing state %s', Dumper($state_hr));


    #  Done
    #
    \undef;

}


sub state_delete {


    #  Delete state db
    #
    my $self=shift();


    #  Get the session id
    #
    my $session_id=$self->session_id() ||
	return err();
    debug("self $self, session_id $session_id");


    #  Set to undef in self ref, so fetch from now on returns nothing
    #
    $self->{'_state_hr'}=undef;


    #  If not already flagged to delete off disk, do now
    #
    unless ($self->{'_state_delete_fg'}++) {


	#  Get request object
	#
	my $r=$self->request();
	debug("pushing state_save cleanup to request object $r");


	#  Push disk save callback, unless we have already done it
	#
	$r->register_cleanup(


	    sub {


		#  Debug
		#
		debug("in state_delete cleanup $self");


		#  Run the real save routine
		#
		$self->state_save({

		    session_id	    =>	$session_id,
		    delete	    =>  1,


		});

	    });

    }


    #  And return
    #
    \undef;


}


sub state_save {


    #  Get a reference to the state database
    #
    my ($self, $param_hr)=@_;


    #  Get the session id
    #
    my ($state_hr, $session_id, $delete_fg)=@{$param_hr}{qw(data session_id delete)};
    debug("in state_save self $self stat_hr %s, sess $session_id, del $delete_fg", Dumper($state_hr));


    #  Skip if marked for deletion
    #
    if ($self->{'_state_delete_fg'} && !$delete_fg) {

	debug("save aborted due to pending deletion, $delete_fg %s", !$delete_fg);
	return \undef

    }
    else {

	debug('proceeding to save/delete')

    }


    #  Get berkeley env
    #
    my $db_env=$self->_state_db_env() ||
	return err();
    debug('got dbenv');


    #  Get file name
    #
    my $cn=$self->_state_cn() || return err();


    #  Options for db open
    #
    my %options=(

	-Filename       =>      $cn,
	-Flags          =>      DB_CREATE,
	-Mode           =>      0640,
	-Env            =>      $db_env,

       );


    #  Open db
    #
    my $state_tr=tie(my %state, 'BerkeleyDB::Hash', %options) ||
	return err($BerkeleyDB::Error);


    #  And store/delete
    #
    if ($delete_fg) {


	#  Delete
	#
	delete $state{$session_id};
	debug('state_hr deleted from disk');


    }
    else {


	#  Update mtime, freeze
	#
	$state_hr->{'_mtime'}=time();
	$state{$session_id}=Storable::freeze($state_hr);
	debug('state_hr saved to disk');


    }


    #  Close
    #
    undef $state_tr;
    untie %state;


    #  All done
    #
    return \undef;

}


#  Specific methods unique to BerkeleyDB stat handler
#
sub filename {

    my $self=shift();
    if (@_) {
	ref($self) ? $self->{'_state'}{+__PACKAGE__}{'filename'}=$_[1] : $Package{'_filename'}=$_[1];
    }
    return  $self->{'_state'}{+__PACKAGE__}{'filename'} || $Package{'_filename'} || $WEBDYNE_BERKELEYDB_STATE_FN;

}


#  Private methods
#
sub _state_dn {

    my $self=shift();
    my $filename=$self->filename() || return err('unable to get state filename');
    (File::Spec->splitpath($filename))[1] || $WEBDYNE_BERKELEYDB_STATE_DN;

}


sub _state_fn {

    my $self=shift();
    my $filename=$self->filename() || return err('unable to get state filename');
    (File::Spec->splitpath($filename))[2];

}


sub _state_cn {

    my $self=shift();
    my $dn=$self->_state_dn() || return err();
    my $fn=$self->_state_fn() || return err();
    File::Spec->catfile($dn, $fn) || return err();

}


sub state_tr {

    debug('get state_tr');
    return $Package{'_session'}{'_state_tr'};

}


sub _state_hr {

    debug('get state_hr');
    return $Package{'_session'}{'_state_hr'};

}


sub _state_db_env {


    #  Setup the statedb environment
    #
    my $self=shift();
    debug('get state_db_env');


    #  Get dir name
    #
    my $dn=$self->_state_dn() || return err();


    #  Init Berkeley DB Env
    #
    $Package{'_db_env'} ||= (BerkeleyDB::Env->new(

	-Home	=>  $dn,
	-Flags	=>  DB_CREATE | DB_INIT_CDB | DB_INIT_MPOOL

       ) || return err("unable to init Berkeley::DB::Env, $BerkeleyDB::Error"));


}


