#
#
#  Copyright (c) 2003 Andrew W. Speer <andrew.speer@isolutions.com.au>. All rights 
#  reserved.
#
#  This file is part of WebMod::Err.
#
#  WebMod::Err is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
#  $Id: Err.pm,v 1.11 2005/09/28 15:13:27 aspeer Exp $

#
#  Error logging routines
#
package WebMod::Err;


#  Compiler pragma
#
use strict 	qw(vars);
use vars 	qw($VERSION $REVISION $PACKAGE @ISA %EXPORT_TAGS);
no warnings	qw(uninitialized);


#  WebMod modules
#
use WebMod::Constant qw(:var);
require WebMod::Proto;
require WebMod::Log;


#  External Modules
#
use IO::File;
use Data::Dumper;


#  Use Exporter
#
require Exporter;


#  Inheritance
#
@ISA=qw(Exporter);


#  Version Info, must be all one line for MakeMaker, CPAN.
#
$VERSION = eval { require WebMod::Err::VERSION; do $INC{'WebMod/Err/VERSION.pm'}};


#  Release info
#
$REVISION = (qw $Revision: 1.11 $)[1];


#  Package
#
$PACKAGE=__PACKAGE__;


#  Push warn function onto EXPORT_OK stack
#
%EXPORT_TAGS=(
    all	    =>	[qw(err errstr errclr errdump errsubst mutter errnofatal)]
);


#  Use the export_ok_tags function to push the all tags
#  onto the EXPORT_OK array
#
&Exporter::export_ok_tags('all');


#  Our error stack
#
our @Err;


#  Var to hold class, init'd below
#
our $Class_hr;


#  Init log handlers and class var
#
$PACKAGE->init($PACKAGE) ||
    die("unable to init $PACKAGE");



#  All done, return OK
#
return \undef;


#============================================================================


sub errnofatal {


    #  
    #
    @_ ? $Class_hr->{'nofatal'}=@_ : $Class_hr->{'nofatal'};
    
}


sub err {


    #  Read in the paramaters for this call, default to a message string
    #
    my $param=proto([

	qw(message:@* mutter:! nofatal:! callstack:$),

       ], \@_) || die("error in '${PACKAGE}::err' parmaters");


    #  Get an shortcut for the message array ref
    #
    my $message_ar=$param->{'message'} ||
	die('unable to get message array ref');


    #  If no params are supplied, return the last warning, otherwise
    #  sprintf the paramaters, using the first as a template.
    #
    my $message=sprintf( shift(@{$message_ar}), @{$message_ar}, @_ ) || do {


	#  No message supplied, use last message, or assume undefined. Try
	#  to get last message
	#
	@Err ? $Err[$#Err]->[0] && return undef :  'undefined error';


    };


    #  Chomp the message
    #
    chomp($message);


    #  Make sure the callstack param is set to a number
    #
    $param->{'callstack'} ||= 0;


    #  Init the caller var and array
    #
    my @caller;
    my $caller=(caller($param->{'callstack'}))[0];


    #  Populate the caller array
    #
    for (my $i=$param->{'callstack'}; my @info=(caller($i))[0..3]; $i++) {


	#  Push onto the caller array
	#
	push @caller, \@info;


    }

    #  Vars to hold short and long log messages
    #
    my ($log_shrt, $log_long);


    #  If this message is *not* the same as the last one we saw,
    #  we will log it
    #
    unless ($message eq (@Err && $Err[0]->[0])) {


        #  Add to stack
        #
        unshift @Err, [$message, @caller];


        #  Are we muttering ?
        #
        if ($param->{'mutter'}) {


            #  Yes, make brief message, no stack addition
            #
            $log_shrt=sprintf("%s $message", $caller[0][0]);


            #  Get the long message
            #
            #$log_long=errdump() || $log_shrt;


            #  Log it
            #
            &log({message_shrt=>$log_shrt, message_long=>$log_long}) ||
        	die("error in ${PACKAGE}::log");


            #  Since we are only muttering, shift message off the stack
            #
            shift @Err;

        }
        else {


            #  Get longer (short) log message !
            #
            $log_shrt=sprintf(

        	"$message from %s, line %s in %s",

        	$caller[1][3] || $caller[0][0], @{$caller[0]}[2,1]

        	);


            #  And the long message
            #
            my $log_long=errdump() || $log_shrt;


            #  Log it
            #
            &log({message_shrt=>$log_shrt, message_long=>$log_long}) ||
        	die("error in ${PACKAGE}::log");

        };


	#  If caller has a debug function enabled, call this with the warning
	#
	if (UNIVERSAL::can($caller, 'debug')) {


	    #  Yes, they are using the debug module, so can we call it
	    #
	    &{"${caller}::debug"}($log_shrt);


	}

    }


    #  Return undef
    #
    return ($Class_hr->{'nofatal'} || $param->{'nofatal'}) ? undef : die(&errdump);

}


sub mutter {


    #  Get params formatted
    #
    my $param=proto([], \@_) ||
	die('error in paramaters');
    $param->{'nofatal'}++;


    #  Add the mutter param, indicate to start one callstack back
    #
    map { $param->{$_}++ } qw(mutter callstack);


    #  And call the err method
    #
    &err($param, @_);


}


sub errstr {


    #  Check that there are messages in the stack before trying to get
    #  the last one
    #
    if (my $count=@Err) {


	#  There are objects in the array, so it is safe to do a fetch
	#  on the last (-1) array slot
	#
	my $errstr=$Err[--$count]->[0];


	#  Now clear it, unless peek param given
	#
	#$param->{'peek'} || undef @Err;


	#  And return the errstr
	#
	return $errstr;

    }
    else {


	#  Nothing in the array stack, return undef
	#
	return undef;

    }

}


sub errclr {


    #  Clear the warning stack
    #
    undef @Err;


    #  Replace errors if args
    #
    @_ && (return &err(@_));


    #  Return OK always
    #
    return OK;

}


sub errsubst {


    #  Replace the current error message with a new one, keeping callback
    #  stack
    #
    my $param=proto([

	qw(message:@*),

       ], \@_) || die("error in '${PACKAGE}::err' parmaters");


    #  Get an shortcut for the message array ref
    #
    my $message_ar=$param->{'message'} ||
	die('unable to get message array ref');


    #  If no params are supplied, return the last warning, otherwise
    #  sprintf the paramaters, using the first as a template.
    #
    my $message=sprintf( shift(@{$message_ar}), @{$message_ar}, @_ ) || do {


	#  No message supplied, use last message, or assume undefined. Try
	#  to get last message
	#
	@Err ? $Err[$#Err]->[0] && return undef :  'undefined error';


    };


    #  Chomp the message
    #
    chomp($message);


    #  Replace if present, define if not
    #
    @Err ? ($Err[$#Err]->[0] = $message) : err({ callstack=>1 }, $message);


    #  Return
    #
    return undef;


}


sub errdump {


    #  Use can send additional info to dump as key/value pairs in hash ref
    #  supplied as arg
    #
    my $info_hr=shift();


    #  Return a dump of error in a nice format, no params. Do this with
    #  format strings, so define the ones we will use
    #
    my @format=(

	'+' . ('-' x 78) .  "+\n",
	"| @<<<<< | ^<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< |\n",
	"|        | ^<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<~~ |\n"

       );


    #  Go through the message stack on error at a time in reverse order
    #
    foreach my $err_ar (reverse @Err) {


	#  Get message
	#
	my $message=ucfirst($err_ar->[0]);
	$message=~s/\.?$/\./;


	#  Print out date, time, error message
	#
	formline $format[0];
	formline $format[1], 'Date', scalar(localtime());
	formline $format[0];
	formline $format[1], 'Error', $message;
	(formline $format[2], $message) if $message;
	formline $format[0];


	#  Flag so we know we have printed the caller field
	#
	my $caller_fg;


	#  Go through callback stack
	#
	for (my $i=1; defined($err_ar->[$i]); $i++) {


	    #  Get method, line no and file
	    #
	    my $method=$err_ar->[$i+1][3] || $err_ar->[$i][0] ||  last;
	    my $lineno=$err_ar->[$i][2] || next;
	    my $filenm=$err_ar->[$i][1];


	    #  Print them out, print out caller label unless we
	    #  have already done so
	    #
	    formline $format[1],
	    $caller_fg ++ ? '' : 'Caller' , "$method, line $lineno";

	}


	#  Include any user supplied info
	#
	while (my ($key, $value)=each %{$info_hr}) {


	    #  Print separator, info
	    #
	    formline $format[0];
	    formline $format[1], $key, $value;
	    (formline $format[2], $value) if $value;

	}


	#  Finish off formatting, print PID. Dont ask me why $$ has to be "$$",
	#  it does not show up any other way
	#
	formline $format[0];
	formline $format[1], 'PID', "$$";
	formline $format[0];
	formline "\n";


    }


    #  Empty the format accumulator and return it
    #
    my $return=$^A; undef $^A;
    return $return;

}


sub errstack {

    #  Return the raw error stack
    #
    return \@Err;

}


sub init {


    #  Initialize the module, mainly setup log handlers
    #
    my $param=proto([{self=>\$Class_hr}], \@_) ||
	die('error in paramaters');


    #  Initialise the master log file and logging facility, errors
    #  are considered non fatal.
    #
    $Class_hr->{'log_shrt_or'}=
	WebMod::Log->new(

	    $LOG_SHORT_PARAM_HR,

	   ) || errclr();


    #  Try to open the main log file, may as well *try* to log an error
    #  if this fails
    #
    $Class_hr->{'log_long_or'}=
	WebMod::Log->new(

	    $LOG_LONG_PARAM_HR,

	   ) || errclr();


    #  All done
    #
    return \undef;


}


sub log {


    #  Get the message to log
    #
    my $param=proto(['message_shrt=$, message_long:$'], \@_) ||
	die('error in paramaters');


    #  Store message into var
    #
    my ($message_shrt, $message_long)=@{$param}
        {qw(message_shrt message_long)};


    #  Write to the log file. Non fatal if no file handle
    #
    my $log_shrt_or=$Class_hr->{'log_shrt_or'} ||
	return \undef;
    my $log_long_or=$Class_hr->{'log_long_or'} ||
	return \undef;


    #  Print the message to the log files
    #
    $log_shrt_or->write($message_shrt);
    $log_long_or->write($message_long) if $message_long;


    #  All done
    #
    return \undef;


}


sub proto {


    #  Use the WebMod::Proto::proto function, preserving calling
    #  info
    #
    goto(&WebMod::Proto::proto);

}


