#
#
#  Copyright (c) 2003 Andrew W. Speer <andrew.speer@isolutions.com.au>. All rights 
#  reserved.
#
#  This file is part of WebMod::Log.
#
#  WebMod::Log is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
#  $Id: Log.pm,v 1.6 2005/03/20 14:49:27 aspeer Exp $

#
#  Package to manage log file connections
#
package WebMod::Log;


#  Compiler pragma
#
use strict 	qw(vars);
use vars 	qw($VERSION $REVISION $PACKAGE);


#  External modules
#
use Carp;


#  Version Info, must be all one line for MakeMaker, CPAN.
#
$VERSION = eval { require WebMod::Log::VERSION; do $INC{'WebMod/Log/VERSION.pm'}};


#  Release info
#
$REVISION = (qw $Revision: 1.6 $)[1];


#  Package
#
$PACKAGE=__PACKAGE__;


#  All done, return OK
#
return 1;


#============================================================================


sub new {


    #  Start a new log object
    #
    my ($class, $param_hr)=@_;
    if ($param_hr && !(ref($param_hr) eq 'HASH')) {
	return err('params must be supplied as a HASH reference')}


    #  Array to hold class object handlers
    #
    my @self;
    my $self_or=bless(\@self, $class);


    #  If params, add handler
    #
    ($param_hr && $self_or->add($param_hr));


    #  Return
    #
    return $self_or;

}


sub add {


    #  Add a handler
    #
    my ($self, $param_hr)=@_;
    ($param_hr && (ref($param_hr) eq 'HASH')) ||
	return err('params must be supplied as a HASH reference');


    #  Get handler name
    #
    my $handler=$param_hr->{'handler'} ||
	return err('no handler supplied');


    #  Get handler perl module, eval if not loaded
    #
    (my $handler_fn=$handler . '.pm')=~s/::/\//g;
    $INC{$handler_fn} || do {
	eval("require $handler") || return err(
	    $@ || "handler $handler did not return a true value on load");
    };


    #  Invoke
    #
    my $handler_or=$handler->new($param_hr) ||
	return err("handler $handler did not return a true value on new");


    #  Add to self ref
    #
    push @{$self}, $handler_or;


    #  Return
    #
    return $handler;

}


sub write {


    #  Call each handler's write function
    #
    my $self=shift;


    #  Message can be given as params, array ref or hash
    #
    my ($message_ar, $param_hr);


    #  Switch on param ref type
    #
    switch: {


	#  Hash
	#
	if (ref($_[0]) eq 'HASH') {
	    $param_hr=$_[0];
	    (ref($_[0]->{'message'}) eq 'ARRAY')
		? ($message_ar=$_[0]->{'message'})
		    : ($message_ar=[$_[0]->{'message'}]);
	    last;
	};


	#  Array or plain
	#
	(ref($_[0]) eq 'ARRAY')
	    ? ($message_ar=$_[0])
		: ($message_ar=\@_);

    };


    #  Get actual message
    #
    my $message=sprintf(shift(@{$message_ar}), @{$message_ar}) ||
	sprintf('null message from caller %s', join(',', @{[caller(0)]}[0..3]));


    #  If supplied as param hash, replace message key
    #
    $param_hr->{'message'}=$message;


    #  Go through handlers and write
    #
    foreach my $handler_or (grep {$_} @{$self}) {


	#  Write
	#
	$handler_or->write($param_hr)
	    || return err("%s unable to write message", ref($handler_or));

    }


    #  Done
    #
    return 1;

}


sub close {


    #  Close all handlers
    #
    my $self=shift;


    #  Go through and close
    #
    foreach my $handler_or (@{$self}) {


	#  Write
	#
	$handler_or->close()
	    || return err("%s unable to write message", ref($handler_or));

    }


    #  Done
    #
    return 1;

}


sub err {


    #  Handle errors
    #
    my $errstr=&method(2).': '.sprintf(shift(), @_);
    croak($errstr);

}


sub method {


    #  Quick and dirty fetch of user method
    #
    return (split(/:/, (caller(defined($_[0]) ? $_[0] : 1))[3]))[-1];

}


sub DESTROY {

    1;
    
}
