#
#
#  Copyright (c) 2003 Andrew W. Speer <andrew.speer@isolutions.com.au>. All rights 
#  reserved.
#
#  This file is part of WebDyne::Chain.
#
#  WebDyne::Chain is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
#  $Id: Chain.pm,v 1.26 2006/05/17 08:41:29 aspeer Exp $
#
package WebDyne::Chain;


#  Compiler Pragma
#
sub BEGIN	{ $^W=0 };
use strict	qw(vars);
use vars	qw($VERSION $REVISION);
use warnings;
no  warnings	qw(uninitialized);


#  Webmod, WebDyne Modules.
#
use WebMod::Base qw(:all);
use WebDyne::Constant;


#  Version information in a formate suitable for CPAN etc. Must be
#  all on one line
#
$VERSION = eval { require WebDyne::Chain::VERSION; do $INC{'WebDyne/Chain/VERSION.pm'}};


#  Release information
#
$REVISION= (qw$Revision: 1.26 $)[1];


#  Debug using WebDyne debug handler
#
debug("%s loaded, version $VERSION, revision $REVISION", __PACKAGE__);


#  Shortcut error handler, save using ISA;
#
require WebDyne::Err;
*err_html=\&WebDyne::Err::err_html || *err_html;


#  Package wide hash ref for data storage
#
my %Package;


#  And done
#
1;


#------------------------------------------------------------------------------


sub handler : method {


    #  Get class, request object
    #
    my ($self, $r, $param_hr)=@_;
    my $class=ref($self) || do {


	#  Need new self ref
	#
	my %self=(

	    _time	    =>  time(),
	    _r		    =>	$r,
	    %{delete $self->{'_self'}},

	   );
	$self=bless \%self, $self;
	ref($self);


    };


    #  Debug
    #
    debug("in WebDyne::Chain::handler, class $class, r $r, is_main %s, self $self, param_hr %s",
	  $r->is_main(), Dumper($param_hr));


    #  Log URI
    #
    debug("URI %s", $r->uri());


    #  Get string of modules to chain
    #
    debug('getting meta');
    my $module_ar=$param_hr->{'meta'}{'webdynechain'} || do {
      my @module=split(/\s+/, $param_hr->{'WebDyneChain'} || do {
	  ref($r)=~/FakeRequest/ ?
	      $r->{'WebDyneChain'} || $ENV{'WebDyneChain'} : $r->dir_config('WebDyneChain');
	      });
      \@module;
    };
    my @module=(@{$module_ar}, 'WebDyne');


    #  Cannot chain if no modules. If OK add our own module to chain
    #
    @module ||
	return $self->err_html('unable to determine module chain - have you set WebDyneChain var ?');
    unshift @module, __PACKAGE__;


    #  If get here chain is OK, continue
    #
    $Package{'_chain_ar'}=\@module;


    #  Get location. Used to use r->location, now use module array to generate pseudo
    #  location data;
    #
    my $location=join(undef, @module);
    debug("location $location");
    unless ($Package{'_chain_loaded_hr'}{$location}++) {
	debug("modules not loaded, doing now");
	foreach my $package (@module) {
	    eval("require $package") || do {
		err("unable to load package $package, $@");
		goto CHAIN_ERROR;
	    };
	    debug("loaded $package");
	}
    }


    #  If location not same as last time we were run, then unload chain
    #
    if ((my $location_current=$Package{'_location_current'}) ne $location) {


	#  Need to unload cached code refs
	#
	debug("location_current '$location_current' is ne this location ('$location'). restoring cr's");
	&ISA_restore();


	#  Update location
	#
	$Package{'_location_current'}=$location;


	#  If code ref's cached, load up now
	#
	if (my $chain_hr=$Package{'_chain_hr'}{$location}) {


	    #  Debug
	    #
	    debug("found cached code ref's for location $location loading");


	    #  Yes found, load up
	    #
	    while (my($method,$cr)=each %{$chain_hr}) {


		#  Debug
		#
		debug("loading cr $cr for method $method");


		#  Install code ref
		#
		*{$method}=$cr;

	    }


	    #  Update current pointer
	    #
	    $Package{'_chain_current_hr'}=$chain_hr;


	}
    }
    else {

	debug('location chain same as last request, caching');

    }


    #  Debug
    #
    debug('module array %s', Dumper(\@module));


    #  All done, pass onto next handler in chain. NOTE no error handler (eg || $self->err_html). It is
    #  not our job to check for errors here, we should just pass back whatever the next handler does.
    #
    return $self->SUPER::handler($r, @_[2..$#_]);


    #  Only get here if error handler invoked
    #
    CHAIN_ERROR:
    return $self->err_html();


}


sub ISA_restore {


    #  Get cuurent chain hash
    #
    my $chain_hr=delete $Package{'_chain_current_hr'};
    debug('in ISA_restore, chain %s', Dumper($chain_hr));


    #  Go through each module, restoring
    #
    foreach my $method (keys %{$chain_hr}) {


	#  Free up
	#
	debug("free $method");
	undef *{$method};


    }


}


sub DESTROY {


    #  Get chain array ref
    #
    my $self=shift();
    my $chain_ar=$Package{'_chain_ar'};
    debug("self $self, going through DESTROY chain %s", Dumper($chain_ar));


    #  Handle destroys specially, mini version of AUTOLOAD code below
    #
    foreach my $i (1 .. $#{$chain_ar}) {
	my $package_chain=$chain_ar->[$i];
	debug("looking for DESTROY $package_chain");
	if (my $cr=UNIVERSAL::can($package_chain, 'DESTROY')) { 
		debug("DESTROY hit on $package_chain");
		$cr->($self);
	}
    }


    #  Destroy object
    #
    %{$self}=();
    undef $self;


}



sub UNIVERSAL::AUTOLOAD {


    #  Get self ref, calling class, autoloaded method
    #
    my $self=$_[0];
    my $autoload=$UNIVERSAL::AUTOLOAD || return;


    #  Do not handle DESTROY's
    #
    return if $autoload=~/::DESTROY$/;


    #  Debug
    #
    debug("in UNIVERSAL::AUTOLOAD, self $self, autoload $autoload, caller %s",
	  Dumper([caller(1)]));


    #  Get apache request ref, location. If not present means called by non-WebDyne class, not supported
    #
    my $r;
    unless (eval {$r=$self->{'_r'}}) {
	err("call to run WebDyne::Chain::AUTOLOAD for non WebDyne compatible method $autoload, self $self.");
	goto CHAIN_ERROR;
    }



    #  Get method user was looking for, keep full package name.
    #
    my ($package_autoload, $method_autoload)=($autoload=~/(.*)::(.*?)$/);
    debug("package_autoload $package_autoload, method_autoload $method_autoload");


    #  And chain for this location
    #
    my $chain_ar=$Package{'_chain_ar'};
    my $location=join(undef, @{$chain_ar});
    debug('going through chain %s', Dumper($chain_ar));


    #  Caller information
    #
    my $subroutine_caller=(caller(1))[3];
    my $subroutine_caller_cr=\&{"$subroutine_caller"};
    my ($package_caller, $method_caller)=($subroutine_caller=~/(.*)::(.*?)$/);
    debug("package_caller $package_caller, method_caller $method_caller");


    #  If SUPER method trawl through chain to find the package it was called from, make sure we start
    #  from there in iteration code below
    #
    my $i=0;
    if ($autoload=~/\QSUPER::$method_autoload\E$/) {
	debug("SUPER method");
	for (1; $i < @{$chain_ar}; $i++) {
	    if (UNIVERSAL::can($chain_ar->[$i], $method_caller) eq $subroutine_caller_cr) {
		$i++;
		last;
	    }
	    else {
		debug("miss on package $chain_ar->[$i], $_ ne $subroutine_caller_cr");
	    }
	}
	debug("loop finished, i $i, chain_ar %s", $#{$chain_ar});
    }


    #  Iterate through the chain (in order) looking for the method
    #
    foreach $i ($i .. $#{$chain_ar}) {


	#  Can this package in the chain support the calling method ?
	#
	debug("look for $method_autoload in package $chain_ar->[$i]");
	if (my $cr=UNIVERSAL::can($chain_ar->[$i], $method_autoload)) {


	    #  Yes. Check for loops
	    #
	    if ($cr eq $subroutine_caller_cr) {
		err("detected AUTOLOAD loop for method '$method_autoload' ".
			"package $package_caller. Current chain: %s", join(', ', @{$chain_ar}));
		goto CHAIN_ERROR;
	    }


	    #  Update
	    #
	    debug('hit');
	    *{$autoload}=$cr;


	    #  And keep a record
	    #
	    $Package{'_chain_hr'}{$location}{$autoload}=$cr;
	    $Package{'_chain_current_hr'} ||= $Package{'_chain_hr'}{$location};


	    #  And dispatch. The commented out code is good for debugging internal
	    #  server errors, esp if comment out *{$autoload} above and turn on
	    #  debugging
	    #
	    goto &{$cr};

	}
	else {


	    #  Debug
	    #
	    debug("unable to find method $method_autoload in package $chain_ar->[$i]");

	}

    }


    #  Last resort - look back through call chain
    #
    debug("checking back through callstack for method $method_autoload");
    my %chain=map { $_=> 1} @{$chain_ar};
    my @caller;
    for ($i=0; my $caller=(caller($i))[0]; $i++) {
	next if $chain{$caller}++; #already looked there 
	push @caller, $caller;
	if (my $cr=UNIVERSAL::can($caller, $method_autoload)) {
 	    if ($cr eq $subroutine_caller_cr) {
 		err("detected AUTOLOAD loop for method '$method_autoload' ".
 			"package $package_caller. Current chain: %s", join(', ', @{$chain_ar}));
 		goto CHAIN_ERROR;
 	    }
	    if ($WEBDYNE_AUTOLOAD_POLLUTE) {
		*{$autoload}=$cr;
		$Package{'_chain_hr'}{$location}{$autoload}=$cr;
	    }
	    goto &{$cr}
	}
    }


    #  Return err
    #
    err("method '$method_autoload' not found in call chain: %s", join(',', @caller));
    goto CHAIN_ERROR;

}


__END__


#  NOTES
#
Used to do this a whole different way. See code archived in version 1.8 for different take on
    chaining handlers

