package POE::Component::IRC::Cookbook::Translator;

use strict;
use warnings;

1;
__END__

=head1 NAME

POE::Component::IRC::Cookbook::Translator - A bot that can translate text

=head1 DESCRIPTION

This bot uses L<POE::Component::Lingua::Translate|POE::Component::Lingua::Translate>
to translate text for channel members. It makes use of the C<BotCommand> plugin
to handle the translate command.

=head1 SYNOPSIS

 #!/usr/bin/perl

 use strict;
 use warnings;
 use Encode qw(decode);
 use Encode::Guess;
 use POE;
 use POE::Component::IRC::Common qw(parse_user);
 use POE::Component::IRC::State;
 use POE::Component::IRC::Plugin::AutoJoin;
 use POE::Component::IRC::Plugin::BotCommand;
 use POE::Component::Lingua::Translate;

 POE::Session->create(
     package_states => [
         main => [ qw(_start irc_botcmd_trans translated) ]
     ],
     heap => {
        translators => { },
     }
 );

 $poe_kernel->run();

 sub _start {
     my $heap = $_[HEAP];
     my $irc = POE::Component::IRC::State->spawn(
         Nick   => 'translator_bot',
         Server => 'irc.freenode.net',
     );
     $heap->{irc} = $irc;

     $irc->plugin_add('AutoJoin', POE::Component::IRC::Plugin::AutoJoin->new(
         Channels => [ '#test_channel1', '#test_channel2' ]
     ));
     
     $irc->plugin_add('BotCommand', POE::Component::IRC::Plugin::BotCommand->new(
         Commands => {
            trans => 'Usage: trans <from>,<to> <text>'
         }
     ));

     $irc->yield(register => 'botcmd_trans');
     $irc->yield(connect => );
     return;
 }

 sub irc_botcmd_trans {
     my $heap = $_[HEAP];
     my $irc = $heap->{irc};
     my $nick = parse_user( $_[ARG0] );
     my $channel = $_[ARG1];
     my ($from, $to, $text) = split /,|\s+/, $_[ARG2], 3;
     
     if (!exists $heap->{translators}->{$from . $to}) {
         eval {
             $heap->{translators}->{$from . $to} = POE::Component::Lingua::Translate->new(
                 alias     => $from . $to,
                 back_end  => 'Babelfish',
                 src       => $from,
                 dest      => $to,
             );
         };
         
         if ($@) {
             $irc->yield(privmsg => $channel, "$nick: There was an error: $@");
             return;
         }
     }

     $poe_kernel->post($from . $to => translate =>
         to_utf8($text),
         {
             channel => $channel,
             nick    => $nick,
         }
     );
     return;
 }

 # Some IRC users use CP1252, some use UTF-8. Let's decode it properly.
 sub to_utf8 {
     my ($line) = @_;
     my $utf8 = guess_encoding($line, 'utf8');
     return ref $utf8 ? decode('utf8', $line) : decode('cp1252', $line);
 }
 
 sub translated {
     my $irc = $_[HEAP]->{irc};
     my ($text, $context, $error) = @_[ARG0, ARG1, ARG2]; 

     if ($error) {
         $irc->yield(
             'privmsg',
             $context->{channel},
             $context->{nick} . ": There was an error: $error",
         );
         return;
     }
     
     $irc->yield(
         'privmsg',
         $context->{channel},
         $context->{nick} . ': ' . $text,
     );
     return;
 }

=head1 AUTHOR

Hinrik E<Ouml>rn SigurE<eth>sson, hinrik.sig@gmail.com

=cut
