=head1 NAME

Sys::Statistics::Linux::NetStats - Collect linux net statistics.

=head1 SYNOPSIS

   use Sys::Statistics::Linux::NetStats;

   my $lxs = new Sys::Statistics::Linux::NetStats;
   $lxs->init;
   sleep 1;
   my $stats = $lxs->get;

=head1 DESCRIPTION

This module collects statistics by the virtual F</proc> filesystem (procfs) and is developed on default vanilla
kernels. It is tested on x86 hardware with the distributions SuSE (SuSE on s390 and s390x architecture as well),
Red Hat, Debian, Asianux, Slackware and Mandrake on kernel versions 2.4 and 2.6 and should run on all linux
kernels with a default vanilla kernel as well. It is possible that this module doesn't run on all distributions
if the procfs is too much changed.

Further it is necessary to run it as a user with the authorization to read the F</proc> filesystem.

=head1 DELTAS

It's necessary to initialize the statistics by calling C<init()>, because the statistics are deltas between
the call of C<init()> and C<get()>. By calling C<get()> the deltas be generated and the initial values
be updated automatically. This way making it possible that the call of C<init()> is only necessary
after the call of C<new()>. Further it's recommended to sleep for a while - at least one second - between
the call of C<init()> and/or C<get()> if you want to get useful statistics.

=head1 NET STATISTICS

Generated by F</proc/net/dev>.

   rxbyt    -  Number of bytes received.
   rxpcks   -  Number of packets received.
   rxerrs   -  Number of errors that happend while received packets.
   rxdrop   -  Number of packets that were dropped.
   rxfifo   -  Number of FIFO overruns that happend on received packets.
   rxframe  -  Number of carrier errors that happend on received packets.
   rxcompr  -  Number of compressed packets received.
   rxmulti  -  Number of multicast packets received.
   txbyt    -  Number of bytes transmitted.
   txpcks   -  Number of packets transmitted.
   txerrs   -  Number of errors that happend while transmitting packets.
   txdrop   -  Number of packets that were dropped.
   txfifo   -  Number of FIFO overruns that happend on transmitted packets.
   txcolls  -  Number of collisions that were detected.
   txcarr   -  Number of carrier errors that happend on transmitted packets.
   txcompr  -  Number of compressed packets transmitted.

=head1 METHODS

=head2 new()

Call C<new()> to create a new object.

   my $lxs = new Sys::Statistics::Linux::NetStats;

=head2 init()

Call C<init()> to initialize the statistics.

   $lxs->init;

=head2 get()

Call C<get()> to get the statistics. C<get()> returns the statistics as a hash reference.

   my $stats = $lxs->get;

=head1 EXPORTS

No exports.

=head1 SEE ALSO

B<proc(5)>

=head1 REPORTING BUGS

Please report all bugs to <jschulz.cpan(at)bloonix.de>.

=head1 AUTHOR

Jonny Schulz <jschulz.cpan(at)bloonix.de>.

=head1 COPYRIGHT

Copyright (c) 2006, 2007 by Jonny Schulz. All rights reserved.

This program is free software; you can redistribute it and/or modify it under the same terms as Perl itself.

=cut

package Sys::Statistics::Linux::NetStats;
our $VERSION = '0.02';

use strict;
use warnings;
use IO::File;
use Carp qw(croak);

sub new {
   return bless {
      files => {
         netstats => '/proc/net/dev',
      },
      init  => {},
      stats => {},
   }, shift;
}

sub init {
   my $self = shift;
   $self->{init} = $self->_load;
}

sub get {
   my $self  = shift;
   my $class = ref $self;

   croak "$class: there are no initial statistics defined"
      unless %{$self->{init}};

   $self->{stats} = $self->_load;
   $self->_deltas;
   return $self->{stats};
}

#
# private stuff
#

sub _load {
   my $self  = shift;
   my $class = ref $self;
   my $file  = $self->{files};
   my $fh    = new IO::File;
   my %stats;

   $fh->open($file->{netstats}, 'r') or croak "$class: unable to open $file->{netstats} ($!)";

   while (my $line = <$fh>) {
      next unless $line =~ /^\s*(\w+):\s*(.*)/;
      @{$stats{$1}}{qw(
         rxbyt rxpcks rxerrs rxdrop rxfifo rxframe rxcompr rxmulti
         txbyt txpcks txerrs txdrop txfifo txcolls txcarr txcompr
      )} = split /\s+/, $2;
   }

   $fh->close;
   return \%stats;
}

sub _deltas {
   my $self  = shift;
   my $class = ref $self;
   my $istat = $self->{init};
   my $lstat = $self->{stats};

   foreach my $dev (keys %{$lstat}) {
      unless (exists $istat->{$dev}) {
         delete $lstat->{$dev};
         next;
      }

      my $idev = $istat->{$dev};
      my $ldev = $lstat->{$dev};

      while (my ($k, $v) = each %{$ldev}) {
         croak "$class: different keys in statistics"
            unless defined $idev->{$k};
         croak "$class: value of '$k' is not a number"
            unless $v =~ /^\d+$/ && $ldev->{$k} =~ /^\d+$/;
         $ldev->{$k} -= $idev->{$k};
         $idev->{$k}  = $v;
      }
   }
}

1;
