# WebService::E4SE [![Build Status](https://travis-ci.org/genio/webservice-e4se.svg?branch=master)](https://travis-ci.org/genio/webservice-e4se)

Communicate with the various Epicor E4SE web services.

## SYNOPSIS

```perl
use WebService::E4SE;

# create a new object
my $ws = WebService::E4SE->new(
	username => 'AD\username',                  # NTLM authentication
	password => 'A password',                   # NTLM authentication
	realm => '',                                # LWP::UserAgent and Authen::NTLM
	site => 'epicor:80',                        # LWP::UserAgent and Authen::NTLM
	base_url => URL->new('http://epicor/e4se'), # LWP::UserAgent and Authen::NTLM
	timeout => 30,                              # LWP::UserAgent
);

# get an array ref of web service APIs to communicate with
my $res = $ws->files();
say Dumper $res;

# returns a list of method names for the file you wanted to know about.
my @operations = $ws->operations('Resource');
say Dumper @operations;

# call a method and pass some named parameters to it
my ($res,$trace) = $ws->call('Resource','GetResourceForUserID', userID=>'someuser');

# give me the XML::Compile::WSDL11 object
my $wsdl = $ws->get_object('Resource'); #returns the usable XML::Compile::WSDL11 object
```

## DESCRIPTION

[WebService::E4SE](https://metacpan.org/pod/WebService::E4SE) allows us to connect to
[Epicor's E4SE](http://www.epicor.com/products/e4se.aspx) SOAP-based APIs
service to access our data or put in our timesheet.

Each action on the software calls a SOAP-based web service API method. Each API
call is authenticated via NTLM.

There are more than 100 web service files you could work with (.asmx
extensions) each having their own set of methods. On your installation of E4SE,
you can get a listing of method calls available by visiting one of those files
directly (```http://your_epicor_server/e4se/Resource.asmx``` for example).

The module will grab the WSDL from the file you're trying to deal with.  It will
make use of that WSDL with
[XML::Compile::WSDL11](https://metacpan.org/pod/XML::Compile::WSDL11).  You can
force a reload of the WSDL at any time.  So, we build the
[XML::Compile::WSDL11](https://metacpan.org/pod/XML::Compile::WSDL11) object
and hold onto it for any further calls to that file.  These are generated by
the calls you make, so hopefully we don't kill you with too many objects.  You
can work directly with the new
[XML::Compile::WSDL11](https://metacpan.org/pod/XML::Compile::WSDL11) object if
you like, or use the abstracted out methods listed below.

For transportation, we're using
[XML::Compile::Transport::SOAPHTTP](https://metacpan.org/pod/XML::Compile::Transport::SOAPHTTP)
using [LWP::UserAgent](https://metacpan.org/pod/LWP::UserAgent) with
[Authen::NTLM](https://metacpan.org/pod/Authen::NTLM).

## ATTRIBUTES

[WebService::E4SE](https://metacpan.org/pod/WebService::E4SE) makes the following attributes available:

### base_url

```perl
my $url = $ws->base_url;
$url = $ws->base_url(URI->new('http://epicor/e4se'));
```

This should be the base [URL](https://metacpan.org/pod/URI) for your E4SE installation.

### files

```perl
my $files = $ws->files;
$files = $ws->files(['file1', 'file2']);
say join ', ', @$files;
```

This is reference to an array of file names that this web service has
knowledge of for an E4SE installation. If your installation has some services
that we're missing, you can inject them here. This will clobber, not
merge/append.

### force_wsdl_reload

```perl
my $force = $ws->force_wsdl_reload;
$force = $ws->force_wsdl_reload(1);
```

This attribute is defaulted to false (0).  If set to true, the next call to a
method that would require a [XML::Compile::WSDL11](https://metacpan.org/pod/XML::Compile::WSDL11) object will go out to the
server and re-grab the WSDL and re-setup that WSDL object no matter if we have
already generated it or not. The attribute will be reset to false (0) directly
after the next WSDL object setup.

### password

```perl
my $pass = $ws->password;
$pass = $ws->password('foobarbaz');
```

This will be your domain password.  No attempt to hide this is made.

### realm

```perl
my $realm = $ws->realm;
$realm = $ws->realm('MyADRealm');
```

Default is an empty string.  This is for the [Authen::NTLM](https://metacpan.org/pod/Authen::NTLM) module and can generally be left blank.

### site

```perl
my $site = $ws->site;
$site = $ws->site('epicor:80');
```

This is for the [Authen::NTLM](https://metacpan.org/pod/Authen::NTLM) module.  Set this accordingly.

### username

```perl
my $user = $ws->username;
$user = $ws->username('AD\myusername');
```

Usually, you need to prefix this with the domain your E4SE installation is using.

## METHODS

[WebService::E4SE](https://metacpan.org/pod/WebService::E4SE) makes the following methods available:

### call

```perl
use Try::Tiny;
try {
	my ( $res, $trace) = $ws->call('Resource', 'GetResourceForUserID', %parameters );
	say Dumper $res;
}
catch {
	warn "An error happened: $_";
	exit(1);
}
```

This method will call an API method for the file you want.  It will die on
errors outside of [XML::Compile::WSDL11](https://metacpan.org/pod/XML::Compile::WSDL11)'s knowledge, otherwise
it's just a little wrapper around [XML::Compile::WSDL11](https://metacpan.org/pod/XML::Compile::WSDL11)->call();

Another way to do this would be

```perl
$ws->get_object('Reource')->call( 'GetResourceForUserID', %params );
```

### get_object

```perl
my $wsdl = $ws->get_object('Resource');
```

This method will return an [XML::Compile::WSDL11](https://metacpan.org/pod/XML::Compile::WSDL11) object for the file name
you supply.  This handles going to the file's WSDL URL, grabbing that URL
with [LWP::UserAgent](https://metacpan.org/pod/LWP::UserAgent) and [Authen::NTLM](https://metacpan.org/pod/Authen::NTLM), and using that WSDL response to
setup a new [XML::Compile::WSDL11](https://metacpan.org/pod/XML::Compile::WSDL11) object.

Note that if you have previously setup a [XML::Compile::WSDL](https://metacpan.org/pod/XML::Compile::WSDL) object for that
file name, it will just return that object rather than going to the server and
requesting a new WSDL.

### operations

```perl
my $available_operations = $ws->operations( $file );
```

This method will return a list of  [XML::Compile::SOAP::Operation](https://metacpan.org/pod/XML::Compile::SOAP::Operation) objects
that are available for the given file.

## AUTHOR

Chase Whitener -- capoeirab@cpan.org

## BUGS

Please report any bugs or feature requests on [GitHub](https://github.com/genio/webservice-e4se/issues).
We appreciate any and all criticism, bug reports, enhancements, or fixes.
