#!/usr/bin/perl

# $Id: create_object_security.pl,v 1.4 2001/07/16 21:00:47 lachoy Exp $

use strict;
use Getopt::Long qw( GetOptions );
use OpenInteract::Startup;
use SPOPS::Secure qw( :level :scope );

# create_object_security.pl

#   Take a class name and scope information and create security
#   settings for all objects in the class

my %VALID_SCOPE = (
   world => SEC_SCOPE_WORLD,
   user  => SEC_SCOPE_USER,
   group => SEC_SCOPE_GROUP );

my %VALID_LEVEL = (
    none  => SEC_LEVEL_NONE,
    read  => SEC_LEVEL_READ,
    write => SEC_LEVEL_WRITE );

{
    $| = 1;
    my ( $OPT_scope, $OPT_scope_id, $OPT_level, $OPT_class,
         $OPT_website_dir, $OPT_where, $OPT_verbose, $OPT_help );
    my %options = ( 'scope=s'    => \$OPT_scope,
                    'scope_id=s' => \$OPT_scope_id,
                    'level=s'    => \$OPT_level,
                    'class=s'    => \$OPT_class,
                    'where=s'    => \$OPT_where,
                    'verbose'    => \$OPT_verbose,
                    'help'       => \$OPT_help );

    my $R = OpenInteract::Startup->setup_static_environment_options( \&usage, \%options );

    if ( $OPT_help ) { print usage(); exit(0); }

    my @errors = ();
    unless ( $OPT_scope =~ /^(world|group|user)$/ ) {
        push @errors, "* --scope must be set to 'world', 'group', or 'user'\n";
    }
    if ( $OPT_scope ne 'world' and ! $OPT_scope_id ) {
        push @errors, "* --scope_id must be set if the --scope is set to 'group' or 'user'\n";   
    }
    unless ( $OPT_level =~ /^(none|read|write)$/ ) {
        push @errors, "* --level must be set to 'none', 'read' or 'write'\n";
    }
    unless ( $OPT_class ) {
        push @errors, "* --class must be set to the class of the object you want to\n",
                      "set security for (e.g., 'MySite::News')\n";
    }

    unless ( $OPT_class->can( 'fetch' ) ) {
        push @errors, "* It doesn't appear that ($OPT_class) is a valid class for an\n",
                      "SPOPS class defined in your OpenInteract setup.\n";
    }

    unless ( $OPT_class->isa( 'SPOPS::Security' ) ) {
        warn "It doesn't appear that ($OPT_class) is using security. I can create\n",
             "security settings for it but they won't get used until the class\n",
             "uses security. To do so, ensure that 'SPOPS::Secure' or a child of\n",
             "it is in the 'isa' of the class.\n";
    }
    if ( scalar @errors ) {
        die "Incomplete parameters.\n\n", join( "\n", @errors ), "\n";
    }

    $R->CONFIG->{DEBUG} = 0;

    my $scope = $VALID_SCOPE{ lc $OPT_scope } || die "Invalid scope given! (Scope: $OPT_scope)\n";
    my $level = $VALID_LEVEL{ lc $OPT_level } || die "Invalid level given! (Level: $OPT_level)\n";

    my $begin_time = time;
    $OPT_verbose && print "Objects for which security was successfully set:\n";
  
    my $object_iter = $OPT_class->fetch_iterator({ skip_security => 1,
                                                   column_group  => '_id_field',
                                                   where         => $OPT_where });
    my ( $count, $failure );
    my $sql = qq/
        INSERT INTO sys_security
        ( class, object_id, scope, scope_id, security_level )
        VALUES 
        ( ?,     ?,         ?,     ?,        ? ) /;

    my $sth = eval { $R->db->prepare( $sql ) };
    if ( $@ ) {
        my $ei = SPOPS::Error->get;
        die "Could not prepare statement handle! Error: $ei->{system_msg}\n";
    }

    while ( my $object = $object_iter->get_next ) {
        eval { $sth->execute( $OPT_class, $object->id, $scope, $OPT_scope_id, $level ) };
        if ( $@ ) {
            $OPT_verbose && print "  FAIL: ", $object->id, " ($@)\n";
            $failure++;
        }
        else {
            $OPT_verbose && print "  OK: ", $object->id, "\n";
        }
        $count++;
        unless ( $OPT_verbose ) { 
            printf( '%8d', $count ) if ( $count % 250 == 0 ); 
            print "\n"              if ( $count % 2000 == 0 )
        }
    }
    $OPT_verbose && print "\nComplete!\n",
                          "Records:  $count\n",
                          "Success:  ", $count - $failure, "\n",
                          "Failed:   $failure\n",
                          "Started:  ", scalar localtime( $begin_time ), "\n",
                          "Finished: ", scalar localtime, "\n";
}


sub usage {
    return <<USAGE;
create_object_security.pl - Create security settings for all objects
of a particular class

Usage:

 create_object_security.pl [options]

Options:

 --class        class of object (e.g., 'MySite::News')

 --level        Security level to set (none|read|write)

 --scope        Scope to set security for (world|group|user)

 --scope_id     ID of user/group -- ignored for 'world'

 --website_dir  Path to OI website (or set \$ENV{OIWEBSITE})

 --where        SQL statement to restrict results (optional)

 --help         Display this message

Example:

Create security setting so that all members of group with ID 3 get
write permission for all 'MySite::News' objects.

 perl create_object_security.pl --class=MySite::News \
                                --level=write
                                --scope=group \
                                --scope_id=3 \
                                --website_dir=/home/httpd/mysite.com
USAGE
}

__END__

=pod

=head1 NAME

create_object_security.pl - Batch-create security settings for objects

=head1 SYNOPSIS

 # Create security setting so that all members of group with ID 3 get
 # write permission for all 'MySite::News' objects.

 perl create_object_security.pl --class=MySite::News \
                                --level=write
                                --scope=group \
                                --scope_id=3 \
                                --website_dir=/home/httpd/mysite.com

=head1 DESCRIPTION

Creates security settings for all objects in a class. Run with
'--help' to get syntax and more info.

=head1 COPYRIGHT

Copyright (c) 2001 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
