#!/usr/bin/perl

# $Id: add_objects_to_index.pl,v 1.4 2001/07/12 18:00:38 lachoy Exp $

# add_objects_to_index.pl -- Re-index a particular class.

use strict;
use OpenInteract::Startup;

{
    my $usage = "Usage: $0 --website_dir=/path/to/my_site object-tag";
 
    my $object_tag = shift;
    die "$usage\n" unless ( $object_tag );

    my ( $OPT_where );
    my %options = ( 'where=s' => \$OPT_where );

    my $R = OpenInteract::Startup->setup_static_environment_options( $usage, \%options );

    # Try to get the class corresponding to the object tag passed in

    my $obj_class = eval { $R->$object_tag() };
    if ( $@ or ! $obj_class ) {
        my $error_msg = $@ || 'none returned';
        die "Cannot retrieve objects without a class -- no match for ",
            "$object_tag. (Error: $error_msg)\n";
    }

    # Ensure the object class is currently being indexed

    unless ( $obj_class->isa( 'OpenInteract::FullText' ) ) {
        die "Failed! The class ($obj_class) corresponding to tag \n",
            "($object_tag) does not currently use the full-text indexing\n",
            "engine. Change the 'isa' tag for the object.\n";
    }

    my $CONFIG = $obj_class->CONFIG;
    my $ft_fields = $CONFIG->{fulltext_field};
    unless ( ref $ft_fields eq 'ARRAY' and scalar @{ $ft_fields } ) {
        die "Failed! You must define a list of fields to index in the\n",
            "'fulltext_field' key in your object configuration.\n";
    }

    # Retrieve all the objects -- but define a column group on the fly
    # for just the fields we need (sneaky)

    $CONFIG->{column_group}->{fulltext} ||= $ft_fields;

    my $iter = eval { $obj_class->fetch_iterator({ 
                                     skip_security => 1,
                                     where         => $OPT_where,
                                     column_group  => 'fulltext' }) };
    if ( $@ ) {
        my $ei = SPOPS::Error->get;
        die "Fetch of objects failed!\n",
            "Simple error: $@\n",
            "Enlightening error: $ei->{system_msg}\n";
    }

    my $start_time = scalar localtime;
    print "Starting to index each object. This might take a while...\n";

    # Index each object

    my ( $count, $ok );
    $count = $ok = 0;
    while ( my $obj = $iter->get_next ) {
        $obj->reindex_object;
        $count++;
        print "($count) ", $obj->id;
        if ( $@ ) { 
            print " FAIL ($@)\n";
        }
        else {
            print " OK\n";
            $ok++;
        }
    }
    print "Done.\nObjects attempted/indexed: $count/$ok\n";
    print "Start: $start_time\n",
          "End:   ", scalar localtime, "\n";

    # Some DBD drivers complain about not explicitly disconnecting, which
    # worries people...

    $R->db->disconnect;
}
 
__END__

=pod

=head1 NAME

add_objects_to_index.pl - Reindex objects in a particular class

=head1 SYNOPSIS

 # 'news' is the label you use in your 'spops.perl' file for the
 # object -- e.g., 'user' for 'OpenInteract::User' objects or
 # 'sitetemplate' for 'OpenInteract::SiteTemplate' objects.

 $ perl add_objects_to_index.pl --website_dir=/home/httpd/www.myoisite.com news

OR (using a bash shell):

 $ export OIWEBSITE=/home/httpd/www.myoisite.com
 $ perl add_objects_to_index.pl news

 # Find objects matching only a particular criteria

 $ perl add_objects_to_index.pl --where="title like '%all your base%'"

=head1 DESCRIPTION

Cycles through every available object in the class given (or each
object matching a particular criteria) and calls 'reindex_object' on
it. Pretty simple.

Note that the '--where' option requires you to do value quoting
manually. If your clause fails, you will definitely hear about it.

=head1 BUGS 

This can take an amazingly long time to run on some databases. MySQL,
in particular, seems to have some performance issues with relatively
large index tables.

=head1 TO DO

B<Web interface>

This same function should be implemented via a web interface.

=head1 SEE ALSO

L<OpenInteract::FullText>

=head1 COPYRIGHT

Copyright (c) 2001 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
