package Object::InsideOut; {

require 5.006;

use strict;
use warnings;

our $VERSION = '0.02.00';

my $phase = 'COMPILE';   # Phase of the Perl interpreter

### Exception Processing ###

# Exceptions generated by this module
use Exception::Class (
    'OIO' => {
        'description' => 'Generic Object::InsideOut exception',
        # First 3 fields must be:  'Package', 'File', 'Line'
        'fields' => ['Package', 'File', 'Line', 'Error'],
    },

    'OIO::Code' => {
        'isa' => 'OIO',
        'description' =>
            'Object::InsideOut exception that indicates a coding error',
        'fields' => ['Info'],
    },

    'OIO::Internal' => {
        'isa' => 'OIO::Code',
        'description' =>
            'Object::InsideOut exception that indicates a internal problem',
        'fields' => ['Code', 'Declaration'],
    },

    'OIO::Attribute' => {
        'isa' => 'OIO::Code',
        'description' =>
            'Object::InsideOut exception that indicates a coding error',
        'fields' => ['Attribute'],
    },

    'OIO::Method' => {
        'isa' => 'OIO',
        'description' =>
            'Object::InsideOut exception that indicates an method calling error',
    },

    'OIO::Args' => {
        'isa' => 'OIO::Method',
        'description' =>
            'Object::InsideOut exception that indicates an argument error',
        'fields' => ['Usage'],
    },
);


# A 'throw' method that adds location information to the exception object
sub OIO::die
{
    my $class = shift;
    my %args  = @_;

    # Get location information
    my ($pkg, $file, $line);

    if (exists($args{'location'})) {
        # Location specified in an array ref
        ($pkg, $file, $line) = @{delete($args{'location'})};

    } elsif (exists($args{'caller_level'})) {
        # Location specified as a caller() level
        ($pkg, $file, $line) = caller(1 + delete($args{'caller_level'}));

    } else {
        # Default location
        ($pkg, $file, $line) = caller(1);
    }

    $class->throw(%args,
                  'Package' => $pkg,
                  'File'    => $file,
                  'Line'    => $line);
}


# Provides a fully formated error message for the exception object
sub OIO::full_message
{
    my $self = shift;

    # Start with error class and message
    my $msg = ref($self) . ' error: ' . $self->message();
    chomp($msg);

    # Add fields, if any
    my @fields = $self->Fields();
    shift(@fields) for (1..3);   # Drop location fields
    for my $field (@fields) {
        if (exists($self->{$field})) {
            $msg .= "\n$field: " . $self->{$field};
            chomp($msg);
        }
    }

    # Add location
    if (defined($self->{'Package'})) {
        $msg .= "\nPackage: " . $self->{'Package'}
              . "\nFile: "    . $self->{'File'}
              . "\nLine: "    . $self->{'Line'};
    }

    return ($msg . "\n");
}


# Catch untrapped errors
# Usage:  local $SIG{__DIE__} = 'OIO::trap';
sub OIO::trap
{
    # Just rethrow if already an exception object
    if (Object::InsideOut::Util::is_it($_[0], 'Exception::Class::Base')) {
        $_[0]->rethrow();
    }

    # Turn on stack trace
    OIO->Trace(1);

    # Package the error into an object
    OIO->die(
        'message' => 'Trapped uncaught error',
        'Error'   => join('', @_));
}


### Additional Supporting Code ###

require Object::InsideOut::Util;


# Obtain certain functionality either directly from Scalar::Util (best),
# or from code that we install ourselves
BEGIN {
    # Try to use an available version of Scalar::Util
    eval { require Scalar::Util; };

    # Check for success by looking for 'weaken'
    if (! Scalar::Util->can('weaken')) {
        # Regardless of the reason for the above failure, install our own
        # versions of necessary functions

        # Create a 'no-op' version of 'weaken'
        *Scalar::Util::weaken = sub ($) { };

        no warnings 'redefine';

        # A simplified version of 'blessed'
        *Scalar::Util::blessed = sub ($) {
            UNIVERSAL::can($_[0], 'can');
        };

        # A adequate version of 'refaddr'
        *Scalar::Util::refaddr = sub ($) {
            0+bless($_[0], 'UNIVERSAL');
        };
    }

    # This is copied from Scalar::Util
    if (! Scalar::Util->can('looks_like_number')) {
        *Scalar::Util::looks_like_number = sub {
            local $_ = shift;
            return $] < 5.008005 unless defined;
            return 1 if (/^[+-]?\d+$/); # is a +/- integer
            return 1 if (/^([+-]?)(?=\d|\.\d)\d*(\.\d*)?([Ee]([+-]?\d+))?$/); # a C float
            return 1 if ($] >= 5.008 and /^(Inf(inity)?|NaN)$/i) or ($] >= 5.006001 and /^Inf$/i);
            0;
        };
    }
}


### Class Tree Building (via 'import()') ###

# Cache of class trees
my (%TREE_TOP_DOWN, %TREE_BOTTOM_UP);

# Doesn't export anything - just builds class trees and stores sharing flags
sub import
{
    my $self  = shift;      # Ourself (i.e., 'Object::InsideOut')
    my $class = caller();   # The class that is using us

    no strict 'refs';

    # Check for class's global sharing flag
    # (normally set in the app's main code)
    if (defined(${$class.'::shared'})) {
        set_sharing($class, ${$class.'::shared'}, (caller())[1..2]);
    }

    # Import packages
    my @packages;
    while (@_) {
        my $pkg = shift;
        if (! $pkg) {
            next;
        }

        # Handle thread object sharing flag
        if ($pkg =~ /^:(NOT?_?|!)?SHAR/i) {
            my $sharing = (defined($1)) ? 0 : 1;
            set_sharing($class, $sharing, (caller())[1..2]);
            next;
        }

        if (! $class->isa($pkg)) {
            # Load the package, if needed
            if (! @{$pkg.'::ISA'}) {
                eval "require $pkg";
                if ($@) {
                    OIO::Code->die(
                        'message' => "Failure loading package '$pkg'",
                        'Error'   => $@);
                }
            }

            # Add to package list
            push(@packages, $pkg);
        }

        # Import the package, if needed
        if (ref($_[0])) {
            my $imports = shift;
            if (ref($imports) ne 'ARRAY') {
                OIO::Code->die(
                    'message' => "Arguments to '$pkg' must be contained within an array reference");
            }
            eval { import $pkg @{$imports}; };
            if ($@) {
                OIO::Code->die(
                    'message' => "Failure importing package '$pkg'",
                    'Error'   => $@);
            }
        }
    }

    # Create calling class's @ISA array
    push(@{$class.'::ISA'}, $self, @packages);

    # Create class tree
    my @tree;
    my %seen;   # Used to prevent duplicate entries in @tree
    for my $parent (@packages) {
        # Parent class must exist
        if (! exists($TREE_TOP_DOWN{$parent})) {
            OIO::Code->die(
                'message' => "Cannot declare '$parent' as a parent class for '$class'",
                'Error'   => "Class '$parent' is not an Object::InsideOut class");
        }

        for my $ancestor (@{$TREE_TOP_DOWN{$parent}}) {
            if (! exists($seen{$ancestor})) {
                push(@tree, $ancestor);
                $seen{$ancestor} = undef;
            }
        }
    }

    # Add calling class to tree
    if (! exists($seen{$class})) {
        push(@tree, $class);
    }

    # Save the trees
    $TREE_TOP_DOWN{$class} = \@tree;
    @{$TREE_BOTTOM_UP{$class}} = reverse(@tree);
}


### Attribute Support ###

# Maintain references to all object attribute hashes by package for easy
# manipulation of attribute data during global object actions (e.g., cloning,
# destruction).  Object attribute hashes are marked with an attribute called
# 'Field'.
my %FIELDS;

my %DUMP_FIELDS;

# Allow a single object ID specifier subroutine per class tree.  The
# subroutine ref provided will return the object ID to be used for the object
# that is created by this package.  The ID subroutine is marked with an
# attribute called 'ID', and is :HIDDEN during the CHECK phase by default.
my %ID_SUBS;

# Contains the ID sub, if any, to be used for each class
my %ID_SUB_CACHE;

# Allow a single object initialization hash per class.  The data in these
# hashes is used to initialize newly create objects. The initialization hash
# is marked with an attribute called 'InitArgs'.
my %INIT_ARGS;

# Allow a single initialization subroutine per class that is called as part of
# initializing newly created objects.  The initialization subroutine is marked
# with an attributed called 'Init', and is :HIDDEN during the CHECK phase by
# default.
my %INITORS;

# Allow a single data replication subroutine per class that is called when
# objects are cloned.  The data replication subroutine is marked with an
# attributed called 'Replicate', and is :HIDDEN during the CHECK phase by
# default.
my %REPLICATORS;

# Allow a single data destruction subroutine per class that is called when
# objects are destroyed.  The data destruction subroutine is marked with an
# attributed called 'Destroy', and is :HIDDEN during the CHECK phase by
# default.
my %DESTROYERS;

# Allow a single 'autoload' subroutine per class that is called when an object
# method is not found.  The automethods subroutine is marked with an
# attributed called 'Automethod', and is :HIDDEN during the CHECK phase by
# default.
my %AUTOMETHODS;

# Methods that support 'cumulativity' from the top of the class tree
# downwards.  These cumulative methods are marked with an attributed called
# 'Cumulative'.
my %CUMULATIVE;

# Methods that support 'cumulativity' from the bottom of the class tree
# upwards.  These cumulative methods are marked with an attributed
# 'Cumulative(bottom up)'.
my %ANTICUMULATIVE;

# Restricted methods are only callable from within the class hierarchy, and
# are marked with an attributed called 'Restricted'.
my %RESTRICTED;

# Restricted methods are only callable from within the class itself, and
# are marked with an attributed called 'Private'.
my %PRIVATE;

# Methods that are made uncallable after the CHECK phase.  They are marked
# with an attributed called 'HIDDEN'.
my %HIDDEN;

# Methods that are support overloading capabilities for objects.
my %OVERLOAD;

# These are the attributes for designating 'overload' methods.
my %OVERLOAD_TYPES = (
    'STRINGIFY' => q/""/,
    'NUMERIFY'  => q/0+/,
    'BOOLIFY'   => q/bool/,
    'ARRAYIFY'  => q/@{}/,
    'HASHIFY'   => q/%{}/,
    'GLOBIFY'   => q/*{}/,
    'CODIFY'    => q/&{}/,
);


# This subroutine handles attributes on hashes as part of this package.
# See 'perldoc attributes' for details.
sub MODIFY_HASH_ATTRIBUTES
{
    my ($pkg, $hash, @attrs) = @_;

    my @unused_attrs;   # List of any unhandled attributes

    # Process attributes
    for my $attr (@attrs) {
        # Declaration for object attribute hash
        if ($attr =~ /^Field/i) {
            if ($phase eq 'COMPILE') {
                # Save save hash ref and accessor declarations
                # Accessors will be build during CHECK phase
                my ($decl) = $attr =~ /^Fields?\s*(?:[(]\s*(.*)\s*[)])/i;
                push(@{$FIELDS{$pkg}}, [ $hash, $decl ]);

            } else {   # $phase eq 'RUNNING'
                # Save the hash ref
                push(@{$FIELDS{$pkg}}, $hash);

                # Share the hash, if applicable
                if (is_sharing($pkg)) {
                    threads::shared::share($hash)
                }

                # Process any accessor declarations
                if (my ($decl) = $attr =~ /^Fields?\s*(?:[(]\s*(.*)\s*[)])/i) {
                    create_accessors($pkg, $hash, $decl);
                }
            }
        }

        # Declaration for object initializer hash
        elsif ($attr =~ /^InitArgs?$/i) {
            $INIT_ARGS{$pkg} = $hash;

            # Extract field info for '_DUMP'
            while (my ($name, $val) = each(%{$hash})) {
                if (ref($val) eq 'HASH') {
                    if (my $field = hash_re($val, qr/^FIELD$/i)) {
                        while (my ($name2, $field2) = each(%{$DUMP_FIELDS{$pkg}})) {
                            if ($field == $field2) {
                                delete($DUMP_FIELDS{$pkg}{$name2});
                                last;
                            }
                        }
                        $DUMP_FIELDS{$pkg}{$name} = $field;
                    }
                }
            }
        }

        # Handle ':shared' attribute associated with threads::shared
        elsif ($attr eq 'shared') {
            if ($threads::shared::threads_shared) {
                threads::shared::share($hash);
            }
        }

        # Unhandled
        else {
            push(@unused_attrs, $attr);
        }
    }

    # Return any unused attributes
    return (@unused_attrs);
}


# Handles subroutine attributes supported by this package.
# See 'perldoc attributes' for details.
sub MODIFY_CODE_ATTRIBUTES
{
    my ($pkg, $code, @attrs) = @_;
    my $info = [ $code, [ $pkg, (caller(2))[1,2] ] ];

    my @unused_attrs;   # List of any unhandled attributes

    # Save the code refs in the appropriate hashes
    ATTR:
    while (my $attribute = shift(@attrs)) {
        my ($attr, $arg) = $attribute =~ /(\w+)(?:[(]\s*(.*)\s*[)])?/;
        $attr = uc($attr);
        $arg = ($arg) ? uc($arg) : 'HIDDEN';
        if ($attr eq 'ID') {
            $ID_SUBS{$pkg} = $info;
            push(@attrs, $arg);

        } elsif ($attr eq 'INIT') {
            $INITORS{$pkg} = $code;
            push(@attrs, $arg);

        } elsif ($attr =~ /^REPL(ICATE)?$/) {
            $REPLICATORS{$pkg} = $code;
            push(@attrs, $arg);

        } elsif ($attr =~ /^DEST(ROY)?$/) {
            $DESTROYERS{$pkg} = $code;
            push(@attrs, $arg);

        } elsif ($attr =~ /^AUTO(METHOD)?$/) {
            $AUTOMETHODS{$pkg} = $code;
            push(@attrs, $arg);

        } elsif ($attr =~ /^CUM(ULATIVE)?$/) {
            if (($arg =~ /BOTTOM\s+UP/) || ($arg =~ /BASE\s+FIRST/)) {
                push(@{$ANTICUMULATIVE{$pkg}}, $info);
            } else {
                push(@{$CUMULATIVE{$pkg}}, $info);
            }

        } elsif ($attr =~ /^RESTRICT(ED)?$/) {
            push(@{$RESTRICTED{$pkg}}, $info);

        } elsif ($attr =~ /^PRIV(ATE)?$/) {
            push(@{$PRIVATE{$pkg}}, $info);

        } elsif ($attr eq 'HIDDEN') {
            push(@{$HIDDEN{$pkg}}, $info);

        } elsif ($attr eq 'SCALARIFY') {
            OIO::Attribute->die(
                'location' => $info->[1],
                'message'  => q/:SCALARIFY not allowed/,
                'Info'     => q/The scalar of an object is its object ID, and can't be redefined/);

        } elsif ($attr =~ /IFY$/) {
            # Overload (-ify) attributes
            for my $ify_attr (keys(%OVERLOAD_TYPES)) {
                if ($attr eq $ify_attr) {
                    push(@{$OVERLOAD{$pkg}}, [$ify_attr, @{$info} ]);
                    next ATTR;
                }
            }

        } elsif ($attr !~ /^PUB(LIC)?$/) {   # PUBLIC is ignored
            # Not handled
            push(@unused_attrs, $attribute);
        }
    }

    # Return any unused attributes
    return (@unused_attrs);
}


### 'CHECK' Phase Attribute Handling ###

# Forward declaration of thread object sharing flag hash
# (Used in CHECK block below)
my %IS_SHARING;


# Finds a subroutine's name in a package from its code ref
sub sub_name # :HIDDEN attribute set 'by hand' at end of CHECK block
{
    my ($pkg, $ref, $attr, $location) = @_;
    no strict 'refs';
    for my $name (keys(%{$pkg.'::'})) {
        my $candidate = *{$pkg.'::'.$name}{'CODE'};
        if ($candidate && $candidate == $ref) {
            return $name;
        }
    }

    # Not found
    OIO::Attribute->die(
        'location' => $location,
        'message'  => q/Subroutine name not found/,
        'Info'     => "Can't use anonymous subroutine for $attr attribute");
}


CHECK {
    no warnings 'redefine';
    no strict 'refs';

    # Verify that there is only one :ID sub in each class tree
    if (%ID_SUBS) {
        for my $class (keys(%TREE_TOP_DOWN)) {
            my $id_sub_pkg;
            for my $pkg (@{$TREE_TOP_DOWN{$class}}) {
                if ($ID_SUBS{$pkg}) {
                    if ($id_sub_pkg) {
                        my ($p,    $file,  $line)  = @{$ID_SUBS{$pkg}->[1]};
                        my ($pkg2, $file2, $line2) = @{$ID_SUBS{$id_sub_pkg}->[1]};
                        OIO::Attribute->die(
                            'caller_level' => 1,
                            'message'      => "Multiple :ID subs defined within hierarchy for '$class'",
                            'Info'         => ":ID subs in class '$pkg' (file '$file', line $line), and class '$pkg2' (file '$file2' line $line2)");
                    }
                    $id_sub_pkg = $pkg;
                }
            }

            # Add classes to ID sub cache, if applicable
            if ($id_sub_pkg) {
                for my $pkg (@{$TREE_TOP_DOWN{$class}}) {
                    $ID_SUB_CACHE{$pkg} = $ID_SUBS{$id_sub_pkg}->[0];
                }
            }
        }
        undef(%ID_SUBS);   # No longer needed
    }


    # If needed, process any thread object sharing flags
    if (%IS_SHARING && $threads::shared::threads_shared) {
        for my $flag_class (keys(%IS_SHARING)) {
            # Find the class in any class tree
            for my $tree (values(%TREE_TOP_DOWN)) {
                if (grep /^$flag_class$/, @$tree) {
                    # Check each class in the tree
                    for my $class (@$tree) {
                        if (exists($IS_SHARING{$class})) {
                            # Check for sharing conflicts
                            if ($IS_SHARING{$class}->[0] != $IS_SHARING{$flag_class}->[0]) {
                                my ($pkg1, @loc, $pkg2, $file, $line);
                                if ($IS_SHARING{$flag_class}->[0]) {
                                    $pkg1 = $flag_class;
                                    @loc  = ($flag_class, (@{$IS_SHARING{$flag_class}})[1..2]);
                                    $pkg2 = $class;
                                    ($file, $line) = (@{$IS_SHARING{$class}})[1..2];
                                } else {
                                    $pkg1 = $class;
                                    @loc  = ($class, (@{$IS_SHARING{$class}})[1..2]);
                                    $pkg2 = $flag_class;
                                    ($file, $line) = (@{$IS_SHARING{$flag_class}})[1..2];
                                }
                                OIO::Code->die(
                                    'location' => \@loc,
                                    'message'  => "Can't combine thread-sharing classes ($pkg1) with non-sharing classes ($pkg2) in the same class tree",
                                    'Info'     => "Class '$pkg2' was declared as non-sharing in '$file' line $line");
                            }
                        } else {
                            # Add the sharing flag to this class
                            $IS_SHARING{$class} = $IS_SHARING{$flag_class};
                        }
                    }
                }
            }
        }
    }


    # Process :FIELD declarations for shared hashes and accessors
    for my $pkg (keys(%FIELDS)) {
        my @hashes;
        for my $item (@{$FIELDS{$pkg}}) {
            my ($hash, $decl) = @{$item};

            # Share the hash, if applicable
            if (is_sharing($pkg)) {
                threads::shared::share($hash)
            }

            # Process any accessor declarations
            if ($decl) {
                create_accessors($pkg, $hash, $decl);
            }

            # Save hash refs
            push(@hashes, $hash);
        }

        # :FIELD declarations have been removed
        $FIELDS{$pkg} = \@hashes;
    }


    # Only install AUTOLOAD if we have Automethods
    if (%AUTOMETHODS) {
        # Create AUTOLOAD under Object::InsideOut
        *Object::InsideOut::AUTOLOAD = create_AUTOLOAD(\%AUTOMETHODS,
                                                       \%TREE_BOTTOM_UP);

        # Install our version of UNIVERSAL::can that understands :Automethod
        *UNIVERSAL::can = create_UNIVERSAL_can(\&UNIVERSAL::can,
                                               \%AUTOMETHODS,
                                               \%TREE_BOTTOM_UP);
    }


    # Implement cumulative methods
    if (%CUMULATIVE || %ANTICUMULATIVE) {
        require Object::InsideOut::Cumulative;

        # Get names for :CUMULATIVE methods
        my (%cum, %cum_loc);
        for my $package (keys(%CUMULATIVE)) {
            for my $info (@{$CUMULATIVE{$package}}) {
                my ($code, $location) = @{$info};
                my $name = sub_name($package, $code, ':CUMULATIVE', $location);
                $cum{$name}{$package} = $code;
                $cum_loc{$name}{$package} = $location;
            }
        }

        # Get names for :CUMULATIVE(BOTTOM UP) methods
        my %anticum;
        for my $package (keys(%ANTICUMULATIVE)) {
            for my $info (@{$ANTICUMULATIVE{$package}}) {
                my ($code, $location) = @{$info};
                my $name = sub_name($package, $code, ':CUMULATIVE(BOTTOM UP)', $location);

                # Check for conflicting definitions of $name
                if ($cum{$name}) {
                    for my $other_package (keys(%{$cum{$name}})) {
                        if ($other_package->isa($package) ||
                            $package->isa($other_package))
                        {
                            my ($pkg,  $file,  $line)  = @{$cum_loc{$name}{$other_package}};
                            my ($pkg2, $file2, $line2) = @{$location};
                            OIO::Attribute->die(
                                'location' => $location,
                                'message'  => "Conflicting definitions for cumulative method '$name'",
                                'Info'     => "Declared as :CUMULATIVE in class '$pkg' (file '$file', line $line), but declared as :CUMULATIVE(BOTTOM UP) in class '$pkg2' (file '$file2' line $line2)");
                        }
                    }
                }

                $anticum{$name}{$package} = $code;
            }
        }
        undef(%CUMULATIVE);      # No longer needed
        undef(%ANTICUMULATIVE);
        undef(%cum_loc);

        # Implement :CUMULATIVE methods
        for my $name (keys(%cum)) {
            for my $package (keys(%{$cum{$name}})) {
                *{$package.'::'.$name} = create_CUMULATIVE(\%TREE_TOP_DOWN,
                                                           $cum{$name});
            }
        }

        # Implement :CUMULATIVE(BOTTOM UP) methods
        for my $name (keys(%anticum)) {
            for my $package (keys(%{$anticum{$name}})) {
                *{$package.'::'.$name} = create_CUMULATIVE(\%TREE_BOTTOM_UP,
                                                           $anticum{$name});
            }
        }
    }


    # Implement overload (-ify) operators
    for my $package (keys(%OVERLOAD)) {
        for my $operation (@{$OVERLOAD{$package}}) {
            my ($attr, $code, $location) = @$operation;
            my $name = sub_name($package, $code, ":$attr", $location);
            {
                my @errs;
                local $SIG{__WARN__} = sub { push(@errs, @_); };

                my $code = sprintf(<<'_CODE_', $package, $OVERLOAD_TYPES{$attr}, $name);
package %s;
    use overload (
    q/%s/ => sub { $_[0]->%s() },
    'fallback' => 1
);
_CODE_
                eval $code;

                if ($@ || @errs) {
                    my ($err) = split(/ at /, $@ || join(" | ", @errs));
                    my ($pkg, $file, $line) = @{$location};
                    OIO::Internal->die(
                        'location' => [ __PACKAGE__, __FILE__, __LINE__ ],
                        'message'  => "Failure overloading :$attr for class '$pkg' (file '$file' line $line)",
                        'Error'    => $err,
                        'Code'     => $code);
                }
            }
        }
    }
    undef(%OVERLOAD);   # No longer needed


    # Implement restricted methods - only callable within hierarchy
    for my $package (keys(%RESTRICTED)) {
        for my $info (@{$RESTRICTED{$package}}) {
            my ($code, $location) = @{$info};
            my $name = sub_name($package, $code, ':RESTRICTED', $location);
            *{$package.'::'.$name} = create_RESTRICTED($package, $name, $code);
        }
    }
    undef(%RESTRICTED);   # No longer needed


    # Implement private methods - only callable from class itself
    for my $package (keys(%PRIVATE)) {
        for my $info (@{$PRIVATE{$package}}) {
            my ($code, $location) = @{$info};
            my $name = sub_name($package, $code, ':PRIVATE', $location);
            *{$package.'::'.$name} = create_PRIVATE($package, $name, $code);
        }
    }
    undef(%PRIVATE);   # No longer needed


    # Implement hidden methods - no longer callable by name
    # Must be done last in this CHECK block
    for my $package (keys(%HIDDEN)) {
        for my $info (@{$HIDDEN{$package}}) {
            my ($code, $location) = @{$info};
            my $name = sub_name($package, $code, ':HIDDEN', $location);
            create_HIDDEN($package, $name);
        }
    }
    undef(%HIDDEN);   # No longer needed

    # These must be done 'by hand' because they're inside the for-loop above
    create_HIDDEN(__PACKAGE__, 'sub_name');
    create_HIDDEN(__PACKAGE__, 'create_HIDDEN');
}


### Thread-Shared Object Support ###

# Contains flags as to whether or not a class is sharing objects between
# threads
#my %IS_SHARING;   # Declared above

sub set_sharing : PRIVATE
{
    my ($class, $sharing, $file, $line) = @_;
    $sharing = ($sharing) ? 1 : 0;

    if (exists($IS_SHARING{$class})) {
        if ($IS_SHARING{$class} != $sharing) {
            my (@loc, $nfile, $nline);
            if ($sharing) {
                @loc  = ($class, $file, $line);
                ($nfile, $nline) = (@{$IS_SHARING{$class}})[1..2];
            } else {
                @loc  = ($class, (@{$IS_SHARING{$class}})[1..2]);
                ($nfile, $nline) = ($file, $line);
            }
            OIO::Code->die(
                'location' => \@loc,
                'message'  => "Can't combine thread-sharing and non-sharing instances of a class in the same application",
                'Info'     => "Class '$class' was declared as non-sharing in '$file' line $line");
        }
    } else {
        $IS_SHARING{$class} = [ $sharing, $file, $line ];
    }
}


# Internal subroutine that determines if a class's objects are shared between
# threads
sub is_sharing : PRIVATE
{
    my $class = $_[0];

    # If not 'use threads::shared;', return false
    if (! $threads::shared::threads_shared) {
        return;
    }

    return ($IS_SHARING{$class}->[0]);
}


### Thread Cloning Support ###

# Thread cloning registry - maintains weak references to non-thread-shared
# objects for thread cloning
my %OBJECTS;

# Thread tracking registry - maintains thread lists for thread-shared objects
# to control object destruction
my %SHARED : shared;

# Thread ID is used to keep CLONE from executing more than once
my $THREAD_ID = 0;


# Called after thread is cloned
sub CLONE
{
    # Don't execute when called for subclasses
    if ($_[0] ne __PACKAGE__) {
        return;
    }

    # Don't execute twice for same thread
    if ($THREAD_ID == threads->tid()) {
        return;
    }

    # Set thread ID for the above
    $THREAD_ID = threads->tid();

    # Process thread-shared objects
    if (%SHARED) {
        lock(%SHARED);

        # Add thread ID to every object in the thread tracking registry
        for my $class (keys(%SHARED)) {
            for my $oid (keys(%{$SHARED{$class}})) {
                push(@{$SHARED{$class}{$oid}}, $THREAD_ID);
            }
        }
    }

    # Process non-thread-shared objects
    for my $class (keys(%OBJECTS)) {
        # Get class tree
        my @tree = @{$TREE_TOP_DOWN{$class}};

        # Get the ID sub for this class, if any
        my $id_sub = $ID_SUB_CACHE{$class};

        # If sharing, then must lock object attribute hashes when updating
        my $lock_field = is_sharing($class);

        # Process each object in the class
        for my $old_id (keys(%{$OBJECTS{$class}})) {
            # Get cloned object associated with old ID
            my $obj = delete($OBJECTS{$class}{$old_id});

            # Replace the old object ID with a new one
            Internals::SvREADONLY($$obj, 0);    # Unlock the object
            if ($id_sub) {
                local $SIG{__DIE__} = 'OIO::trap';
                $$obj = &$id_sub;
            } else {
                $$obj = Scalar::Util::refaddr($obj);
            }
            Internals::SvREADONLY($$obj, 1);    # Lock the object again

            # Update the keys of the attribute hashes with the new object ID
            for my $pkg (@tree) {
                for my $fld (@{$FIELDS{$pkg}}) {
                    lock($fld) if ($lock_field);
                    $fld->{$$obj} = delete($fld->{$old_id});
                }
            }

            # Resave weakened reference to object
            Scalar::Util::weaken($OBJECTS{$class}{$$obj} = $obj);

            # Dispatch any special replication handling
            if (%REPLICATORS) {
                my $pseudo_object = \(my $scalar = $old_id);
                for my $pkg (@tree) {
                    if (my $replicate = $REPLICATORS{$pkg}) {
                        local $SIG{__DIE__} = 'OIO::trap';
                        $replicate->($pseudo_object, $obj);
                    }
                }
            }
        }
    }
}


### Object Methods ###

# For performance considerations, certain methods are exported to all classes
INIT {
    # Default export list
    my @EXPORT = qw(new clone DESTROY _DUMP);

    # Additional exports
    if (Object::InsideOut->can('AUTOLOAD')) {
        push(@EXPORT, 'AUTOLOAD');
    }

    # Export methods to all classes
    no strict 'refs';

    for my $pkg (keys(%TREE_TOP_DOWN)) {
        for my $sym (@EXPORT) {
            my $full_sym = $pkg.'::'.$sym;
            # Only export if method doesn't already exist
            if (! *{$full_sym}{CODE}) {
                *{$full_sym} = \&{$sym};
            }
        }
    }

    # Set the next phase
    $phase = 'RUNNING';
}


# Object Constructor
sub new
{
    my $thing = shift;
    my $class = ref($thing) || $thing;

    # Can't call ->new() on this package
    if ($class eq __PACKAGE__) {
        OIO::Method->die('message' => q/Can't create objects from 'Object::InsideOut' itself/);
    }

    # Gather arguments into a single hash ref
    my $all_args = {};
    while (my $arg = shift) {
        if (ref($arg) eq 'HASH') {
            # Add args from a hash ref
            @{$all_args}{keys(%{$arg})} = values(%{$arg});
        } elsif (ref($arg)) {
            OIO::Args->die(
                'message' => "Bad initializer: @{[ref($arg)]} ref not allowed",
                'Usage'   => q/Args must be 'key=>val' pair(s) and\/or hash ref(s)/);
        } elsif (! @_) {
            OIO::Args->die(
                'message' => "Bad initializer: Missing value for key '$arg'",
                'Usage'   => q/Args must be 'key=>val' pair(s) and\/or hash ref(s)/);
        } else {
            # Add 'key => value' pair
            $all_args->{$arg} = shift;
        }
    }

    # Get thread-sharing flag
    my $am_sharing = is_sharing($class);

    # Create a new 'bare' object
    my $self = Object::InsideOut::Util::create_object($class,
                                                      $ID_SUB_CACHE{$class});
    if ($am_sharing) {
        threads::shared::share($self);
    }

    # Initialize object
    for my $pkg (@{$TREE_TOP_DOWN{$class}}) {
        my $spec = $INIT_ARGS{$pkg};
        my $init = $INITORS{$pkg};

        # Nothing to initialize for this class
        next if (!$spec && !$init);

        # If have InitArgs, then process args with it.  Otherwise, all the
        # args will be sent to the Init subroutine.
        my $args = ($spec) ? Object::InsideOut::Util::process_args($pkg,
                                                                   $self,
                                                                   $spec,
                                                                   $all_args)
                           : $all_args;

        if ($init) {
            # Send remaining args, if any, to Init subroutine
            local $SIG{__DIE__} = 'OIO::trap';
            $init->($self, $args);

        } elsif (%$args) {
            # It's an error if no Init subroutine, and there are unhandled
            # args
            OIO::Args->die(
                'message' => "Unhandled arguments for class '$class': " . join(', ', keys(%$args)),
                'Usage'   => q/Add appropriate 'Field =>' designators to the :InitArgs hash/);
        }
    }

    # Thread support
    if ($am_sharing) {
        # Add thread tracking list for this thread-shared object
        lock(%SHARED);
        if (! exists($SHARED{$class})) {
            $SHARED{$class} = &threads::shared::share({});
        }
        $SHARED{$class}{$$self} = &threads::shared::share([]);
        push(@{$SHARED{$class}{$$self}}, $THREAD_ID);

    } elsif ($threads::threads) {
        # Add non-thread-shared object to thread cloning list
        Scalar::Util::weaken($OBJECTS{$class}{$$self} = $self);
    }

    # Done - return object
    return ($self);
}


# Creates a copy of an object
sub clone
{
    my $parent = $_[0];
    my $class  = ref($parent);

    # Must call ->clone() as an object method
    if (! $class) {
        OIO::Method->die('message' => q/Can't call ->clone() as a class method/);
    }

    # Get thread-sharing flag
    my $am_sharing = is_sharing($class);

    # Create a new 'bare' object
    my $clone = Object::InsideOut::Util::create_object($class,
                                                       $ID_SUB_CACHE{$class});
    if ($am_sharing) {
        threads::shared::share($clone);
    }

    # Clone the object
    for my $pkg (@{$TREE_TOP_DOWN{$class}}) {
        # Clone attributes from the parent
        for my $fld (@{$FIELDS{$pkg}}) {
            lock($fld) if ($am_sharing);
            $fld->{$$clone} = $fld->{$$parent};
        }

        # Dispatch any special replication handling
        if (my $replicate = $REPLICATORS{$pkg}) {
            local $SIG{__DIE__} = 'OIO::trap';
            $replicate->($parent, $clone);
        }
    }

    # Thread support
    if ($am_sharing) {
        # Add thread tracking list for this thread-shared object
        lock(%SHARED);
        if (! exists($SHARED{$class})) {
            $SHARED{$class} = &threads::shared::share({});
        }
        $SHARED{$class}{$$clone} = &threads::shared::share([]);
        push(@{$SHARED{$class}{$$clone}}, $THREAD_ID);

    } elsif ($threads::threads) {
        # Add non-thread-shared object to thread cloning list
        Scalar::Util::weaken($OBJECTS{$class}{$$clone} = $clone);
    }

    # Done - return clone
    return ($clone);
}


# Object Destructor
sub DESTROY
{
    my $self  = $_[0];
    my $class = ref($self);

    if ($$self) {
        if (is_sharing($class)) {
            # Thread-shared object

            # Remove thread ID for this object's thread tracking list
            lock(%SHARED);
            my $tid = pop(@{$SHARED{$class}{$$self}});
            while ($tid != $THREAD_ID) {
                unshift(@{$SHARED{$class}{$$self}}, $tid);
                $tid = pop(@{$SHARED{$class}{$$self}});
            }

            # If object is still active in other threads, then just return
            if (@{$SHARED{$class}{$$self}}) {
                return;
            }

            # Delete the object from the thread tracking registry
            delete($SHARED{$class}{$$self});

        } else {
            # Delete this non-thread-shared object from the thread cloning
            # registry
            delete($OBJECTS{$class}{$$self});
        }


        # If sharing, then must lock object attribute hashes when updating
        my $lock_field = is_sharing($class);

        # Destroy object
        for my $pkg (@{$TREE_BOTTOM_UP{$class}}) {
            # Dispatch any special destruction handling
            if (my $destroy = $DESTROYERS{$pkg}) {
                local $SIG{__DIE__} = 'OIO::trap';
                $destroy->($self);
            }

            # Delete object attributes
            for my $fld (@{$FIELDS{$pkg}}) {
                lock($fld) if ($lock_field);
                delete($fld->{$$self});
            }
        }

        # Erase the object ID - just in case
        Internals::SvREADONLY($$self, 0);       # Unlock the object
        $$self = undef;
    }
}


# String version of the object that contains public data
sub _DUMP
{
    my $self = shift;

    # Gather the data from all the fields in the object's class tree
    my %dump;
    for my $pkg (@{$TREE_TOP_DOWN{ref($self)}}) {
        my @fields = @{$FIELDS{$pkg}};

        # Fields for which we have names
        while (my ($name, $field) = each(%{$DUMP_FIELDS{$pkg}})) {
            if (exists($field->{$$self})) {
                $dump{$pkg}{$name} = $field->{$$self};
            }
            @fields = grep { $_ != $field } @fields;
        }

        # Fields for which names are not known
        for my $field (@fields) {
            if (exists($field->{$$self})) {
                $dump{$pkg}{$field} = $field->{$$self};
            }
        }
    }

    # Create a string version of dumped data if arg is true
    if ($_[0]) {
        require Data::Dumper;
        my $dump = Data::Dumper::Dumper(\%dump);
        chomp($dump);
        $dump =~ s/^.{8}//gm;   # Remove initial 8 chars from each line
        $dump =~ s/ {8}/ /gm;   # Reduce indentation to 1 space per level
        $dump =~ s/;$//s;       # Remove trailing semi-colon
        return $dump;
    }

    # Send back a hash ref to the dumped data
    return (\%dump);
}


### Code Generators ###

# Dynamically creates an AUTOLOAD subroutine
# Called from CHECK block only if some Automethods are defined
sub create_AUTOLOAD : HIDDEN
{
    # $AUTOMETHODS    - ref to %AUTOMETHODS
    # $TREE_BOTTOM_UP - ref to %TREE_BOTTOM_UP
    my ($AUTOMETHODS, $TREE_BOTTOM_UP) = @_;

    return sub {
        my $thing = $_[0];
        my $class = ref($thing) || $thing;

        # Extract the base method name from the fully-qualified name
        my ($method) = our $AUTOLOAD =~ /.*::(.*)/;

        # Find an Automethod
        for my $package (@{$TREE_BOTTOM_UP->{$class}}) {
            if (my $automethod = $AUTOMETHODS->{$package}) {
                # Call the Automethod to get a code ref
                local $CALLER::_ = $_;
                local $_ = $method;
                local $SIG{__DIE__} = 'OIO::trap';
                if (my $code = $automethod->(@_)) {
                    # Go to the code ref returned by the Automethod
                    goto &{$code};
                }
            }
        }

        # Failed to AUTOLOAD
        my $type = ref($thing) ? 'object' : 'class';
        OIO::Method->die('message' => "Can't locate $type method '$method' for package '$class'");
    };
}


# Returns a closure back to the CHECK block that is used to redefine
# UNIVERSAL::can()
sub create_UNIVERSAL_can : HIDDEN
{
    # $univ_can       - ref to the orginal UNIVERSAL::can()
    # $AUTOMETHODS    - ref to %AUTOMETHODS
    # $TREE_BOTTOM_UP - ref to %TREE_BOTTOM_UP
    my ($univ_can, $AUTOMETHODS, $TREE_BOTTOM_UP) = @_;

    return sub {
        # First, try the original UNIVERSAL::can()
        if (my $code = $univ_can->(@_)) {
            return $code;
        }

        # Next, check with the Automethods
        my $thing = $_[0];
        for my $package (@{$TREE_BOTTOM_UP->{ref($thing) || $thing}}) {
            if (my $automethod = $AUTOMETHODS->{$package}) {
                # Call the Automethod to get a code ref
                local $CALLER::_ = $_;
                local $_ = $_[1];    # Method name
                local $SIG{__DIE__} = 'OIO::trap';
                if (my $code = $automethod->(@_)) {
                    # Use the code ref returned by the Automethod
                    my $method_name = $_[1];
                    return sub {
                        my $self = shift;
                        no strict 'refs';
                        $self->$method_name(@_);
                    };
                }
            }
        }

        return;   # Nothing found
    };
}


# Creates object data accessors for classes
sub create_accessors : PRIVATE
{
    my ($package, $hash_ref, $decl) = @_;

    # Parse the accessor declaration
    my $acc_spec;
    {
        my @errs;
        local $SIG{__WARN__} = sub { push(@errs, @_); };

        if ($decl =~ /{/) {
            eval "\$acc_spec = $decl;";
        } else {
            eval "\$acc_spec = { $decl }";
        }

        if ($@ || @errs) {
            my ($err) = split(/ at /, $@ || join(" | ", @errs));
            OIO::Attribute->die(
                'caller_level' => 2,
                'message'      => 'Malformed attribute',
                'Error'        => $err,
                'Attribute'    => "Field( $decl )");
        }
    }

    # Get info for accessors
    my $get  = Object::InsideOut::Util::hash_re($acc_spec, qr/(^acc|get)/i);
    my $set  = Object::InsideOut::Util::hash_re($acc_spec, qr/(^acc|set)/i);
    my $type = Object::InsideOut::Util::hash_re($acc_spec, qr/^type/i);

    # Add field info for '_DUMP', if not already in added from :INIT_ARGS
    my $add_dump = 1;
    while (my ($name, $field) = each(%{$DUMP_FIELDS{$package}})) {
        if ($field == $hash_ref) {
            $add_dump = 0;
            last;
        }
    }
    if ($add_dump) {
        $DUMP_FIELDS{$package}{($get) ? $get : $set} = $hash_ref;
    }

    # Check on accessor names
    my $have_one = 0;
    for my $name ($get, $set) {
        if (defined($name)) {
            if ($name) {
                no strict 'refs';
                # Do not overwrite existing methods
                if (*{$package.'::'.$name}{CODE}) {
                    OIO::Attribute->die(
                        'caller_level' => 2,
                        'message'      => q/Can't create accessor method/,
                        'Info'         => "Method '$name' already exists in class '$package'");
                }
            } else {
                OIO::Attribute->die(
                    'caller_level' => 2,
                    'message'      => q/Can't create accessor method/,
                    'Info'         => q/Accessor name missing in :Field attribute/,
                    'Attribute'    => "Field( $decl )");
            }
            $have_one++;
        }
    }
    if (! $have_one) {
        OIO::Attribute->die(
            'caller_level' => 2,
            'message'      => q/Accessor name missing in :Field attribute/,
            'Info'         => q/Need 'GET', 'SET' or 'ACCESSOR' designator/,
            'Attribute'    => "Field( $decl )");
    }

    # Check type and set default
    if ($type) {
        if ($type =~ /^num(ber|eric)?/i) {
            $type = 'NUMERIC';
        } elsif (uc($type) eq 'LIST' || uc($type) eq 'ARRAY') {
            $type = 'ARRAY';
        } elsif (uc($type) eq 'HASH') {
            $type = 'HASH';
        }
    } else {
        $type = 'NONE';
    }

    # Code to be eval'ed into subroutines
    my $code = '';

    # Create 'set' or combination accessor
    if (defined($set)) {
        # Begin with subroutine declaration in the appropriate package
        $code .= "*${package}::$set = sub {\n";

        # Lock the hash if sharing
        if (is_sharing($package)) {
            $code .= "    lock(\$hash);\n"
        }

        # Add GET portion for combination accessor
        if (defined($get) && $get eq $set) {
            $code .= <<"_COMBINATION_";
    if (\@_ == 1) {
        return (\$hash->\{\${\$_[0]}});
    }
_COMBINATION_
            undef($get);  # That it for 'GET'
        }

        # Else check that set was called with at least one arg
        else {
            $code .= <<"_CHECK_ARGS_";
    if (\@_ < 2) {
        OIO::Args->die('message' => q/Missing arg(s) to '$package->$set'/);
    }
_CHECK_ARGS_
        }

        # Add data type checking
        if ($type eq 'NONE') {
            # No data type check required
            $code .= "    my \$arg = \$_[1];\n";

        } elsif ($type eq 'NUMERIC') {
            # One numeric argument
            $code .= <<"_NUMERIC_";
    my \$arg;
    if (! Scalar::Util::looks_like_number(\$arg = \$_[1])) {
        OIO::Args->die(
            'message' => "Bad argument: \$arg",
            'Usage'   => q/Argument to '$package->$set' must be numeric/);
    }
_NUMERIC_

        } elsif ($type eq 'ARRAY') {
            # List/array - 1+ args or array ref
            $code .= <<'_ARRAY_';
    my $arg;
    if (@_ == 2 && ref($_[1]) eq 'ARRAY') {
        $arg = $_[1];
    } else {
        my @args = @_;
        shift(@args);
        $arg = \@args;
    }
_ARRAY_

        } elsif ($type eq 'HASH') {
            # Hash - pairs of args or hash ref
            $code .= <<"_HASH_";
    my \$arg;
    if (\@_ == 2 && ref(\$_[1]) eq 'HASH') {
        \$arg = \$_[1];
    } elsif (\@_ % 2 == 0) {
        OIO::Args->die(
            'message' => q/Odd number of arguments: Can't create hash ref/,
            'Usage'   => q/'$package->$set' requires a hash ref or an even number of args (to make a hash ref)/);
    } else {
        my \@args = \@_;
        shift(\@args);
        my \%args = \@args;
        \$arg = \\\%args;
    }
_HASH_

        } else {
            # One object or ref arg - exact spelling and case required
            $code .= <<"_OTHER_TYPE_";
    my \$arg;
    if (! Object::InsideOut::Util::is_it(\$arg = \$_[1], '$type')) {
        OIO::Args->die(
            'message' => q/Bad argument: Wrong type/,
            'Usage'   => q/Argument to '$package->$set' must be an object or ref of type '$type'/);
    }
_OTHER_TYPE_
        }

        # Add actual 'set' code
        $code .= (is_sharing($package))
                  ? "    \$hash->\{\${\$_[0]}} = Object::InsideOut::Util::make_shared(\$arg);\n};\n"
                  : "    \$hash->\{\${\$_[0]}} = \$arg;\n};\n";
    }

    # Create 'get' accessor
    if (defined($get)) {
        # Set up locking code
        my $lock = (is_sharing($package)) ? "    lock(\$hash);\n" : '';

        # Build subroutine text
        $code .= <<"_GET_";
*${package}::$get = sub {
$lock    \$hash->{\${\$_[0]}};
};
_GET_
    }

    # Compile the subroutine(s) in the smallest possible lexical scope
    my @errs;
    local $SIG{__WARN__} = sub { push(@errs, @_); };
    {
        my $hash = $hash_ref;
        eval $code;
    }
    if ($@ || @errs) {
        my ($err) = split(/ at /, $@ || join(" | ", @errs));
        OIO::Internal->die(
            'location'    => [ __PACKAGE__, __FILE__, __LINE__ ],
            'message'     => "Failure creating accessor for class '$package'",
            'Error'       => $err,
            'Declaration' => $decl,
            'Code'        => $code);
    }
}


# Returns a closure back to the CHECK block that is used to setup CUMULATIVE
# and CUMULATIVE(BOTTOM UP) methods for a particular method name.
sub create_CUMULATIVE : HIDDEN
{
    # $tree      - ref to either %TREE_TOP_DOWN or %TREE_BOTTOM_UP
    # $code_refs - hash ref by package of code refs for a particular method name
    my ($tree, $code_refs) = @_;

    return sub {
        my @args  = @_;
        my $class = ref($_[0]) || $_[0];
        my $list_context = wantarray;
        my (@results, @classes);

        # Accumulate results
        for my $pkg (@{$tree->{$class}}) {
            if (my $code = $code_refs->{$pkg}) {
                local $SIG{__DIE__} = 'OIO::trap';
                if (defined($list_context)) {
                    push(@classes, $pkg);
                    if ($list_context) {
                        # List context
                        push(@results, $code->(@args));
                    } else {
                        # Scalar context
                        push(@results, scalar($code->(@args)));
                    }
                } else {
                    # void context
                    $code->(@args);
                }
            }
        }

        # Return results
        if (defined($list_context)) {
            if ($list_context) {
                # List context
                return (@results);
            }
            # Scalar context - returns object
            return (Object::InsideOut::Cumulative->new('VALUES'  => \@results,
                                                       'CLASSES' => \@classes));
        }
    };
}


# Returns a 'wrapper' closure back to the CHECK block that restricts a method
# to being only callable from within its class hierarchy
sub create_RESTRICTED : HIDDEN
{
    my ($package, $method, $code) = @_;
    return sub {
        my $caller = caller();
        if ($caller->isa($package) || $package->isa($caller)) {
            goto &{$code}
        }
        OIO::Method->die('message' => "Can't call restricted method '$package->$method' from class '$caller'");
    };
}


# Returns a 'wrapper' closure back to the CHECK block that makes a method
# private (i.e., only callable from within its own class).
sub create_PRIVATE : HIDDEN
{
    my ($package, $method, $code) = @_;
    return sub {
        my $caller = caller();
        if ($caller eq $package) {
            goto &{$code}
        }
        OIO::Method->die('message' => "Can't call private method '$package->$method' from class '$caller'");
    };
}


# Redefines a subroutine in this package to make it uncallable from the outside
# world.
sub create_HIDDEN #: HIDDEN attribute set 'by hand' at end of CHECK block
{
    my ($package, $method) = @_;

    my $code = <<"_CODE_";
sub ${package}::$method {
    OIO::Method->die('message' => q/Can't call hidden method '$package->$method'/);
}
_CODE_

    my @errs;
    local $SIG{__WARN__} = sub { push(@errs, @_); };
    no warnings 'redefine';

    eval $code;

    if ($@ || @errs) {
        my ($err) = split(/ at /, $@ || join(" | ", @errs));
        OIO::Internal->die(
            'location' => [ __PACKAGE__, __FILE__, __LINE__ ],
            'message'  => "Failure hiding '$package->$method'",
            'Error'    => $err,
            'Code'     => $code);
    }
}

}  # End of package's lexical scope

1;

__END__

=head1 NAME

Object::InsideOut - Comprehensive inside-out object support module

=head1 VERSION

This document describes Object::InsideOut version 0.02.00

=head1 SYNOPSIS

 package My::Class; {
     use Object::InsideOut;

     # Generic field with combined accessor
     my %data :Field('Accessor' => 'data');

     # No paramters to ->new()
 }

 package My::Class::Sub; {
     use Object::InsideOut qw(My::Class);

     # List field with separate 'get' and 'set' accessors
     my %info :Field('Get' => 'get_info', 'Set' => 'set_info', 'Type' => 'LIST');

     # Takes 'INFO' as an optional parameter to ->new()
     my %init_args :InitArgs = (
         'INFO' => {
             'Default' => 'none',
             'Type'    => 'LIST',
         },
     );
 }

 package main;

 my $obj = My::Class::Sub->new();
 my $info = $obj->get_info();                    # [ 'none' ]
 my $data = $obj->data();                        # undef
 $obj->data(42);
 $data = $obj->data();                           # 42

 $obj = My::Class::Sub->new('info' => 'help');
 $info = $obj->get_info();                       # [ 'help' ]
 $obj->set_info(qw(foo bar baz));
 $info = $obj->get_info();                       # [ 'foo', 'bar', 'baz' ]

=head1 DESCRIPTION

This module provides comprehensive support for implementing classes using the
inside-out object model.

This module implements inside-out objects as anonymous scalar references that
have been blessed into a class with the scalar containing the ID for the
object (usually its L<refaddr|Scalar::Util/"refaddr EXPR">).  Object data
(i.e., fields) are stored in hashes within the class's package and are keyed
to the object's ID.

The advantages of the inside-out object model over the I<blessed hash> object
model have been extolled elsewhere.  See the informational links under L</"SEE
ALSO">.

This module offers all the capabilities of L<Class::Std> with the following
additional key advantages:

=over

=item Speed

As fast as I<blessed hash> objects for fetching and setting data, and 2.5-4.5
times faster than Class::Std.

=item Threads

Class::Std is not thread safe.  Object::InsideOUt is thread safe, and
thoroughly supports sharing objects between threads using L<threads::shared>.

=item Flexibility

Allows control over object ID specification, accessor naming, parameter name
matching, and more.

=back

=head2 Class Declarations

To use this module, your classes will start with:

    use Object::InsideOut;

Sub-classes inherit from base classes with:

    use Object::InsideOut 'My::Parent';

Multiple inheritance is supported:

    use Object::InsideOut qw(My::Parent Another::Parent);

There is no need for 'use base ...', or to set up @ISA arrays:
Object::InsideOut loads the parent module(s), calls their C<import> functions
and sets up the sub-class's @ISA array.

If a parent module takes parameters, enclose them in an array ref (mandatory)
following the name of the parent class:

    use Object::InsideOut 'My::Parent'      => [ 'param1', 'param2' ],
                          'Another::Parent' => [ 'param' ];

=head2 Field Declarations

Object data fields consist of hashes within a class's package into which data
are stored using the object's ID as the key.  A hash is declared as begin an
object field by following its declaration with the C<:Field> attribute:

    my %info :Field;

(The case of the word I<Field> does not matter, but by convention should not
be all lowercase.)

=head2 Object Creation

Objects are created using the C<new> method which is exported by
Object::InsideOut to each class:

    my $obj = My::Class->new();

Classes do not implement their own C<new> method.  Class-specific object
initialization actions may be handled by C<:Init> methods (see L</"Object
Initialization">).

Parameters are passed in as combinations of C<key =E<gt> value> pairs and/or
hash refs:

    my $obj = My::Class->new('param1' => 'value1');
        # or
    my $obj = My::Class->new({'param1' => 'value1'});
        # or even
    my $obj = My::Class->new(
        'param_X' => 'value_X',
        'param_Y' => 'value_Y',
        {
            'param_A' => 'value_A',
            'param_B' => 'value_B',
        },
        {
            'param_Q' => 'value_Q',
        },
    );

Additionally, parameters can be segregated in hash refs for specific classes:

    my $obj = My::Class->new(
        'foo' => 'bar',
        'My::Class'      => { 'param' => 'value' },
        'Parent::Class'  => { 'data'  => 'info'  },
    );

The initialization methods for both classes in the above will get C<'foo'
=E<gt> 'bar'>, C<My::Class> will also get C<'param' =E<gt> 'value'>, and
C<Parent::Class> will also get C<'data' =E<gt> 'info'>.  In this scheme,
class-specific parameters will override general parameters specified at a
higher level:

    my $obj = My::Class->new(
        'default' => 'bar',
        'Parent::Class'  => { 'default' => 'baz' },
    );

C<My::Class> will get C<'default' =E<gt> 'bar'>, and C<Parent::Class> will get
C<'default' =E<gt> 'baz'>.

Calling C<new> on an object works, too, and operates the same as calling
C<new> for the class of the object (i.e., C<$obj-E<gt>new()> is the same as
C<ref($obj)-E<gt>new()>).

NOTE: You cannot create objects from Object::InsideOut itself:

    # This is an error
    # my $obj = Object::InsideOut->new();

In this way, Object::InsideOut is not an object class, but functions more like
a pragma.

=head2 Object Cloning

Copies of objects can be created using the C<clone> method which is exported
by Object::InsideOut to each class:

    my $obj2 = $obj->clone();

=head2 Object Initialization

Object initialization is accomplished through a combination of an
C<:InitArgs> labelled hash (explained in detail in the L<next
section|/"Object Initialization Argument Specifications">), and an C<:Init>
labelled subroutine.

The C<:InitArgs> labelled hash specifies the parameters to be extracted from
the argument list supplied to the C<new> method.  These parameters are then
sent to the C<:Init> labelled subroutine for processing:

    package My::Class; {
        my %my_field :Field;

        my %init_args :InitArgs = (
            'MY_PARAM' => qr/MY_PARAM/i,
        );

        sub _init :Init
        {
            my ($self, $args) = @_;

            if (exists($args->{'MY_PARAM'})) {
                $my_field($$self) = $args->{'MY_PARAM'};
            }
        }
    }

    package main;

    my $obj = My::Class->new('my_param' => 'data');

(The case of the words I<InitArgs> and I<Init> does not matter, but by
convention should not be all lowercase.)

This C<:Init> labelled subroutine will receive two arguments:  The newly
created object requiring further initialization (i.e., C<$self>); and a hash
ref of supplied arguments that matched C<:InitArgs> specifications.  Data
processed by the subroutine can be placed into the class's field hashes using
the object's ID (i.e., C<$$self>).

=head2 Object Initialization Argument Specifications

The parameters to be handled by the C<new> method are specified in a hash that
is labelled with the C<:InitArgs> attribute.

The simplest parameter specification is just a tag:

    my %init_args :InitArgs = (
        'DATA' => '',
    );

In this case, if a C<key =E<gt> value> pair with an exact match of C<DATA> for
the key is found in the arguments sent to the C<new> method, then C<'DATA'
=E<gt> value> will be included in the argument hash ref sent to the C<:Init>
labelled subroutine.

Rather than counting on exact matches, regular expressions can be used to
specify the parameter:

    my %init_args :InitArgs = (
        'Param' => qr/^PARA?M$/i,
    );

In this case, the argument key could be any of the following: PARAM, PARM,
Param, Parm, param, parm, and so on.  If a match is found, then C<'Param'
=E<gt> value> is sent to the C<:Init> subroutine.  Note that the C<:InitArgs>
hash key is substituted for the original argument key.  This eliminates the
need for any parameter key pattern matching within the C<:Init> subroutine.

With more complex parameter specifications, the syntax changes.  Mandatory
parameters are declared as follows:

    my %init_args :InitArgs = (
        # Mandatory parameter requiring exact matching
        'INFO' => {
            'Mandatory' => 1,
        },
        # Mandatory parameter with pattern matching
        'input' => {
            'Regex'     => qr/^in(?:put)?$/i,
            'Mandatory' => 1,
        },
    );

If a mandatory parameter is missing from the argument list to C<new>, an error
is generated.

For optional parameters, defaults can be specified:

    my %init_args :InitArgs = (
        'LEVEL' => {
            'Regex'   => qr/^lev(?:el)?|lvl$/i,
            'Default' => 3,
        },
    );

The parameter's type can also be specified:

    my %init_args :InitArgs = (
        'LEVEL' => {
            'Regex'   => qr/^lev(?:el)?|lvl$/i,
            'Default' => 3,
            'Type'    => 'Numeric',
        },
    );

Available types are C<Numeric> (or C<Num> or C<Number> - case-insensitive),
C<List> (case-insensitive), a class (e.g., C<My::Class>), or a reference type
(e.g., 'HASH', 'ARRAY', etc.).  The C<List> type allows a single value (that
is then placed in an array ref) or an array ref.  For class and ref types,
exact case and spelling are required.

You can specify automatic processing for a parameter's value such that it is
placed directly info a field hash and not sent to the C<:Init> subroutine:

    my %hosts :Field;

    my %init_args :InitArgs = (
        'HOSTS' => {
            # Allow 'host' or 'hosts' - case-insensitive
            'Regex'     => qr/^hosts?$/i,
            # Mandatory parameter
            'Mandatory' => 1,
            # Allow single value or array ref
            'Type'      => 'List',
            # Automatically put the parameter in %hosts
            'Field'     => \%hosts,
        },
    );

In this case, when a host parameter is found, it is automatically put into the
C<%hosts> hash, and a C<'HOSTS' =E<gt> value> pair is B<not> sent to the
C<:Init> subroutine.  In fact, if you specify fields for all your parameters,
then you don't even need to have an C<:Init> subroutine!  All the work will be
taken care of for you.

(In the above, I<Regex> may be I<Regexp> or just I<Re>, and I<Default> may be
I<Defaults> or I<Def>.  They and the other specifier keys are
case-insensitive, as well.)

=head2 Automatic Accessor Generation

As part of the L</"Field Declarations">, you can optionally specify the
automatic generation of accessor methods:

    my %comment :Field('Accessor' => 'comment');

The above results in Object::InsideOut automatically generating a combined
I<get/set> accessor method that is equivalent to:

    sub comment
    {
        my ($self, $cmt) = @_;

        if (defined($cmt)) {
            return ($comment{$$self} = $cmt);
        }

        return ($comment{$$self});
    }

Or you can specify separate I<get> and/or I<set> accessors:

    my %name :Field('Get' => 'name', 'Set' => 'change_name');
        # or
    my %name :Field('Get' => 'get_name', 'Set' => 'set_name');

Note that the I<set> method (and the combined accessor method when used to set
data) has return value which is the value being set.

Type-checking for the I<set> operation can be specified as well:

    my %level :Field('Accessor' => 'level', 'Type' => 'Numeric');

Available types are C<Numeric> (or C<Num> or C<Number> - case-insensitive),
C<List> or C<Array> (case-insensitive), C<Hash> (case-insensitive), a class
(e.g., C<My::Class>), or a reference type (e.g., 'CODE').  For class and ref
types, exact case and spelling are required.

The C<List/Array> type permits the accessor to accept multiple value (that are
then placed in an array ref) or a single array ref.  The C<Hash> type allows
multiple C<key =E<gt> value> pairs (that are then placed in a hash ref) or a
single hash ref.

Due to limitations in the Perl parser, you cannot use line wrapping with the
C<:Field> attribute:

    # This doesn't work
    # my %level :Field('Get'  => 'level',
    #                  'Set'  => 'set_level',
    #                  'Type' => 'Num');

    # Must be all on one line
    my %level :Field('Get' =>'level', 'Set' => 'set_level', 'Type' => 'Num');

(The word I<Accessor> can be shortened to I<Acc>.  It and the other accessor
specifier keys are case-insensitive.)

=head2 Object ID

By default, the ID of an object is just its L<refaddr|Scalar::Util/"refaddr
EXPR">.  This should suffice for nearly all cases of class development.  If
there is a special need for the module code to control the object ID (see
L<Math::Random::MT::Auto> as an example), then an C<:ID> labelled subroutine
can be specified:

    sub _id :ID
    {
        # Determine a unique object ID
        ...

        return ($id);
    }

For example, a simple sequential numbering scheme (not recommended for real
code):

    my $id_seq = 1;

    sub _id :ID
    {
        return ($id++);
    }

Within any class hierachy only one class may specify an C<:ID> subroutine.

=head2 Object Replication

Object replication occurs explicitly when the C<clone> method is called on an
object, and implicitly when threads are created in a threaded application.  In
nearly all cases, Object::InsideOut will take care of all the details for you.

In rare cases, a class may require special handling for object replication.
It must then provide a subroutine labelled with the C<:Replicate> attribute.
This subroutine will be sent two objects:  The parent and the clone:

    sub _replicate : Replicate
    {
        my ($parent, $clone) = @_;

        # Special object replication processing
    }

In the case of thread cloning, the C<$parent> object is just an blessed
anonymous scalar reference that contains the ID for the object in the parent
thread.

The C<:Replicate> subroutine only needs to deal with the special replication
processing:  Object::InsideOut will handle all the other details.

=head2 Object Destruction

Object::InsideOut exports a C<DESTROY> method to each class that deletes an
object's data from the object field hashes.  If a class requires additional
destruction processing (e.g., closing filehandles), then it must provide a
subroutine labelled with the C<:Destroy> attribute.  This subroutine will be
sent the object that is being destroyed:

    sub _destroy : Destroy
    {
        my $obj = $_[0];

        # Special object destruction processing
    }

The C<:Destroy> subroutine only needs to deal with the special destruction
processing:  The C<DESTROY> method will handle all the other details of object
destruction.

=head2 Automethods

There are significant issues related to Perl's C<AUTOLOAD> mechanism.  Read
Damian Conway's description in L<Class::Std/"C<AUTOMETHOD()>"> for more
details.  Object::InsideOut handles these issues in the same manner.

Classes requiring C<AUTOLOAD> capabilities must provided a subroutine labelled
with the C<:Automethod> attribute.  The C<:Automethod> subroutine will be
called with the object and the arguments in the original method call (the same
as for C<AUTOLOAD>).  The C<:Automethod> subroutine should return either a
subroutine reference that implements the requested method functionality, or
else C<undef> to indicate that it doesn't know how to handle the request.

The name of the method being called is passed as C<$_> instead of
C<$AUTOLOAD>, and does I<not> have the class name prepended to it.  If the
C<:Automethod> subroutine also needs to access the C<$_> from the caller's
scope, it is available as C<$CALLER::_>.

    sub _automethod :Automethod
    {
        my $self = shift;
        my @args = @_;

        my $method_name = $_;

        # If method can be handled by this class
        if (...) {
            my $handler = sub { .... };

            return ($handler);
        }

        # This class cannot handle the method request
        return;
    }

=head2 Cumulative Methods

As with C<Class::Std>, Object::InsideOut provides a mechanism for creating
methods whose effects accumulate through the class hierarchy.  See
L<Class::Std/"C<:CUMULATIVE()>"> for details.  Such methods are tagged with the C<:Cumulative> attribute, and propogate from the I<top down>
through the class hierarchy.  If tagged with C<:Cumulative(bottom up)>, they
will propogated from the object's class upward.

=head2 Restricted and Private Methods

Access to certain methods can be narrowed by use of the C<:Restricted> and
C<:Private> attributes.  C<:Restricted> methods can only be called from within
the class's hierarchy.  C<:Private> methods can only be called from within the
method's class.

Without the above attributes, most methods have I<public> access.  If desired,
you may explicitly label them with the C<:Public> attribute.

=head2 Hidden Methods

For subroutines marked with the following attributes:

=over

=item :ID

=item :Init

=item :Replicate

=item :Destroy

=item :Automethod

=back

Object::InsideOut normally renders them uncallable (hidden) to class and
application code (as they should normally only be needed by Object::InsideOut
itself).  If needed, this behavior can be overridden adding the C<PUBLIC>,
C<RESTRICTED> or C<PRIVATE> keywords following the attribute:

    sub _init :Init(private)    # Callable from within this class
    {
        my ($self, $args) = @_;

        ...
    }

NOTE:  That the above cannot be accomplished by using the corresponding
attributes.  For example:

    # sub _init :Init :Private    # Wrong syntax - doesn't work

=head2 Object Coercion

As with C<Class::Std>, Object::InsideOut provides support for various forms of
object coercion through the C<overload> mechanism.  See
L<Class::Std/"C<:STRINGIFY>"> for details.  The following attributes are
supported:

=over

=item :Stringify

=item :Numerify

=item :Boolify

=item :Arrayify

=item :Hashify

=item :Globify

=item :Codify

=back

Coercing an object to a scalar (C<:Scalarify>) is not supported as C<$$obj> is
the ID of the object and cannot be overridden.

=head2 The C<_DUMP()> Method

Object::InsideOut exports a method called C<_DUMP> to each class that returns
either a hash or string representation of the object that invokes the method.

The hash representation is returned when C<_DUMP> is called without arguments.
The hash ref that is returned has keys for each of the classes that make up
the object's hierarchy.  The values for those keys are hash refs containing
C<key =E<gt> value> pairs for the object's fields.  The name for a field will
be either the tag from the C<:InitArgs> array that is associated with the
field, its I<get> method name, its I<set> method name, or, failing all that, a
string of the form C<HASH(0x....)>.

When called with a I<true> argument, C<_DUMP> returns a string version of the
hash representation using L<Data::Dumper>.

=head1 THREAD SUPPORT

This module fully supports threads (i.e., is thread safe), and for Perl 5.8.0
and beyond also supports the sharing of Object::InsideOut objects between
threads using L<threads::shared>.  To use Object::InsideOut in a threaded
application, you must put C<use threads;> at the beginning of the application.
(The use of C<require threads;> after the program is running is not
supported.)  If object sharing it to be utilized, then C<use threads::shared;>
should follow.

For Perl 5.6.0 to 5.7.1, you can C<use threads;>, but you must call
C<Object::InsideOut->CLONE()> as the first line of the subroutine argument to
C<threads->create()>:

    use threads;

    package My::Class; {
        use Object::InsideOut;
        ...
    }

    package main;

    my $obj = My::Class->new();

    my $thr = threads->create(sub {
            Object::InsideOut->CLONE();

            ...
            $obj->method();
            ...
        });

For Perl 5.7.2 and 5.7.3, you can C<use threads;> with no restrictions.

For Perl 5.8.0 onwards, if you just C<use threads;>, then objects from one
thread will be copied and made available in a child thread.

The addition of <use threads::shared;> in and of itself does not alter the
behavior of Object::InsideOut objects.  The default behavior is to I<not>
share objects between threads (i.e., they act the same as with C<use
threads;> alone).

To enable the sharing of objects between threads, you must specify which
classes will be involved with thread object sharing.  There are two methods
for doing this.  The first involves setting a C<::shared> variable for the
class prior to its use:

    use threads;
    use threads::shared;

    $My::Class::shared = 1;
    use My::Class;

The other method is for a class to add a C<:SHARED> flag to its C<use
Object::InsideOut ...> declaration:

    package My::Class; {
        use Object::InsideOut ':SHARED';
        ...
    }

When either sharing flag is set for one class in an object hierarchy, then all
the classes in the hierarchy are affected.

If a class cannot support thread object sharing (e.g., one of the object
fields contains code refs [which Perl cannot share between threads]), it
should specifically declare this fact:

    package My::Class; {
        use Object::InsideOut ':NOT_SHARED';
        ...
    }

However, you cannot mixed thread object sharing classes with non-sharing
classes in the same class hierarchy:

    use threads;
    use threads::shared;

    package My::Class; {
        use Object::InsideOut ':SHARED';
        ...
    }

    package Other::Class; {
        use Object::InsideOut ':NOT_SHARED';
        ...
    }

    package My::Derived; {
        use Object::InsideOut qw(My::Class Other::Class);   # ERROR!
        ...
    }

Here is a complete example with thread object sharing enabled:

    use threads;
    use threads::shared;

    package My::Class; {
        use Object::InsideOut ':SHARED';

        # One list-type field
        my %data : Field('Accessor' => 'data', 'Type' => 'List');
    }

    package main;

    # New object
    my $obj = My::Class->new();

    # Set the object's 'data' field
    $obj->data(qw(foo bar baz));

    # Print out the object's data
    print(join(', ', @{$obj->data()}), "\n");              # "foo, bar, baz"

    # Create a thread and manipulate the object's data
    my $rc = threads->create(
            sub {
                # Read the object's data
                my $data = $obj->data();
                # Print out the object's data
                print(join(', ', @{$data}), "\n");         # "foo, bar, baz"
                # Change the object's data
                $obj->data(@$data[1..2], 'zooks');
                # Print out the object's modified data
                print(join(', ', @{$obj->data()}), "\n");  # "bar, baz, zooks"
                return (1);
            }
        )->join();

    # Show that the changes in the object are visible in the parent thread
    # I.e., this shows that the object was indeed shared between threads
    print(join(', ', @{$obj->data()}), "\n");              # "bar, baz, zooks"

=head1 DIAGNOSTICS

This module uses C<Exception::Class> for reporting errors.  The base error
class for this module is C<OIO>.

    my $obj;
    eval { $obj = My::Class->new(); };
    if (my $e = OIO->caught()) {
        print(STDERR "Failure creating object: $e\n");
        exit(1);
    }

=over

=item *Invalid HASH attribute

Forgot to 'use Object::InsideOut qw(Parent::Class ...);'

=back

=head1 BUGS AND LIMITATIONS

Cannot use C<require Object::InsideOut;>, nor can you require a class that
uses Object::InsideOut.

Cannot overload an object to a scalar context (i.e., can't C<:SCALARIFY>).

You cannot use two instances of the same class with mixed thread object
sharing in same application.

Cannot use attributes on 'subroutine stubs' (i.e., forward declaration without
later definition) with C:<:Automethod>:

    package My::Class; {
        sub method : Private;   # Will not work

        sub _automethod : Automethod
        {
            # Code to handle call to 'method' stub
        }
    }


Due to limitations in the Perl parser, you cannot use line wrapping with the
C<:Field> attribute.

If 'set' accessor type is 'SCALAR', then can store any inside-out object type
in it.  If 'HASH', then can store any 'ordinary' object type.

If you save an object inside another object when thread-sharing, you must
rebless it when you get it out:

    my $bb = BB->new();
    my $aa = AA->new();
    $aa->save($bb);
    my $cc = $aa->get();
    bless($cc, 'BB');

If your version of Perl does not support the C<weaken> function:
If you C<use threads;>, then you need to manually destroy objects using
C<$obj->DESTROY()>.  If C<use threads;> and C<use threads::shared>, then you
need to manually destroy non-shared objects using C<$obj->DESTROY()>.

There are no known bugs in this module.

Please submit any bugs, problems, suggestions, patches, etc. to:
L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=Object-InsideOut>

=head1 REQUIREMENTS

L<Exception::Class> v1.22 or higher

L<Scalar::Util> v1.17 or higher recommended

=head1 TO DO

Improve these docs.

Improve test suite.

=head1 SEE ALSO

Inside-out Object Model:
L<http://www.perlmonks.org/index.pl?node_id=219378>,
L<http://www.perlmonks.org/index.pl?node_id=483162>,
Chapters 15 and 16 of I<Perl Best Practices> by Damian Conway

=head1 AUTHOR

Jerry D. Hedden, S<E<lt>jdhedden AT 1979 DOT usna DOT comE<gt>>

=head1 COPYRIGHT AND LICENSE

Copyright 2005 Jerry D. Hedden. All rights reserved.

This program is free software; you can redistribute it and/or modify it under
the same terms as Perl itself.

=cut
