#!/usr/bin/perl

# Copyright (C) 2012, Benjamin Drung <bdrung@debian.org>
# Copyright (C) 2016, Jonas Smedegaard <dr@jones.dk>
#
# Permission to use, copy, modify, and/or distribute this software for any
# purpose with or without fee is hereby granted, provided that the above
# copyright notice and this permission notice appear in all copies.
#
# THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
# WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
# ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
# WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
# ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
# OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

use strict;
use warnings;

use Path::Tiny;

use Test::More tests => 23;
use Test::Command;

my @COMMAND
	= ( defined( $ARGV[0] ) and $ARGV[0] eq '--installed' )
	? ('licensecheck')
	: ( 'perl', 'bin/licensecheck' );

sub license
{
	my ( $file, $stdout ) = @_;

	my $command = Test::Command->new(
		cmd => [ @COMMAND, "t/devscripts/$file" ],
	);

	$command->stdout_is_eq("t/devscripts/$file: $stdout\n");
	$command->stderr_is_eq('');
	$command->exit_is_num(0);
}

sub license2
{
	my ( $args, $file, $stdout ) = @_;

	my @args = split( ' ', $args );
	my $command = Test::Command->new(
		cmd => [ @COMMAND, @args, "t/devscripts/$file" ],
	);

	$command->stdout_is_eq("t/devscripts/$file	$stdout\n");
	$command->stderr_is_eq('');
	$command->exit_is_num(0);
}

subtest 'Beerware' => sub {
	license 'beerware.cpp', 'Beerware';
};

subtest 'Boost' => sub {
	license 'boost.h', 'BSL';
};

subtest 'Artistic' => sub {
	license2 '-m --copyright', 'artistic-2-0-modules.pm',
		'Artistic (v2.0)	2009 Moritz Lenz and the SVG::Plot contributors (see file';
	license2 '-m --copyright', 'artistic-2-0.txt',
		'Artistic (v2.0)	2000-2006, The Perl Foundation / 2000 Testophilus Testownik <tester@testity.org>';
	license2 '-m --copyright --deb-fmt', 'artistic-2-0-modules.pm',
		'Artistic-2.0	2009 Moritz Lenz and the SVG::Plot contributors (see file';
	license2 '-m --copyright --deb-fmt', 'artistic-2-0.txt',
		'Artistic-2.0	2000-2006, The Perl Foundation / 2000 Testophilus Testownik <tester@testity.org>';
};

subtest 'Academic' => sub {
	license 'academic.h', 'AFL-3.0';
};

subtest 'Eclipse' => sub {
	license2 '-m --copyright', 'epl.h',
		'EPL-1.0	2004, 2006 IBM Corporation and others';
	license2 '-m --copyright --deb-fmt', 'epl.h',
		'EPL-1.0	2004, 2006 IBM Corporation and others';
};

subtest 'BSD' => sub {
	license 'bsd-1-clause-1.c',          'BSD';
	license 'bsd-3-clause.cpp',          'BSD (3 clause)';
	license 'bsd-3-clause-authorsany.c', 'BSD (3 clause)';
	license2 '-m --copyright',           'mame-style.c',
		'BSD (3 clause)	Miodrag Milanovic';
	license2 '-m --copyright --deb-fmt', 'mame-style.c',
		'BSD-3-Clause	Miodrag Milanovic';
};

subtest 'MultiLine declaration' => sub {

	# test copyright declared on 2 lines
	license2 '-m --copyright', 'bsd-regents.c',
		'BSD (3 clause)	1987, 1993, 1994 The Regents of the University of California. All rights reserved.';

	# or 3 lines
	license2 '-m --copyright', 'texinfo.tex',
		'GPL (v3 or later)	1985, 1986, 1988, 1990, 1991, 1992, 1993, 1994, 1995 1996, 1997, 1998, 1999, 2000, 2001, 2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012 Free Software Foundation, Inc.';

	# BTS #519080
	license2 '-m --copyright', 'multi-line-copyright.c',
		'Public domain GPL (v3)	2008 Aaron Plattner, NVIDIA Corporation / 2005 Lars Knoll & Zack Rusin, Trolltech / 2000 Keith Packard, member of The XFree86 Project, Inc';
};

subtest 'Duplicated copyright' => sub {
	license2 '-m --copyright', 'duplicated-copyright.rb',
		'Apache (v2.0)	2002-2015, OpenNebula Project (OpenNebula.org), C12G Labs';
};

subtest 'LGPL' => sub {
	license 'lgpl-2.1.c',        'LGPL (v2.1)';
	license 'lgpl-2.1.h',        'LGPL (v2.1)';
	license 'lgpl-2.s',          'LGPL (v2)';
	license 'lgpl-2-3.c',        'LGPL (v2 or v3)';
	license 'lgpl-3+.h',         'LGPL (v3 or later)';
	license 'lgpl-variation.c',  'LGPL (v3 or later)';
	license 'lgpl-no-version.h', 'LGPL (unversioned/unknown version)';
	license2 '-m --copyright',   'lgpl-big-blue.h',
		'LGPL (v2.1)	2000, 2010 IBM Corporation and others. All rights reserved';
	license2 '-m --copyright', 'lgpl-digia.c',
		'LGPL (v2.1 or v3)	2014 Digia Plc and/or its subsidiary(-ies)';
	license2 '-m --copyright', 'lgpl-by-dist-zilla.pl',
		'LGPL (v2.1)	2016 Dominique Dumont';
	license2 '-m --copyright --deb-fmt', 'lgpl-big-blue.h',
		'LGPL-2.1	2000, 2010 IBM Corporation and others. All rights reserved';
	license2 '-m --copyright --deb-fmt', 'lgpl-digia.c',
		'LGPL-2.1 or LGPL-3	2014 Digia Plc and/or its subsidiary(-ies)';
	license2 '-m --copyright --deb-fmt', 'lgpl-by-dist-zilla.pl',
		'LGPL-2.1	2016 Dominique Dumont';
};

# Lisp Lesser General Public License (BTS #806424)
# see http://opensource.franz.com/preamble.html
subtest 'LLGPL' => sub {
	license2 '-m --copyright', 'llgpl.lisp',
		'LLGPL	2002, 2004 Sven Van Caekenberghe, Beta Nine BVBA';
};

subtest 'GPL' => sub {
	license 'gpl-1',          'GPL (v1 or later)';
	license 'gpl-2',          'GPL (v2)';
	license 'bug-559429',     'GPL (v2)';
	license 'gpl-2+',         'GPL (v2 or later)';
	license 'gpl-2+.scm',     'GPL (v2 or later)';
	license 'gpl-2-comma.sh', 'GPL (v2)';
	license 'gpl-2-incorrect-address',
		'GPL (v2) (with incorrect FSF address)';
	license 'gpl-3.sh',                    'GPL (v3)';
	license 'gpl-3+',                      'GPL (v3 or later)';
	license 'gpl-3+-with-rem-comment.xml', 'GPL (v3 or later)';
	license 'gpl-3-only.c',                'GPL (v3)';
	license 'gpl-variation.c',             'GPL (v3 or later)';
	license 'gpl-3+.el',                   'GPL (v3 or later)';
	license 'gpl-no-version.h', 'GPL (unversioned/unknown version)';
};

subtest 'Dual' => sub {
	license 'dual.c', 'Public domain GPL (v3)';
};

subtest 'MPL' => sub {
	license 'mpl-1.1.sh',       'MPL (v1.1)';
	license 'mpl-2.0.sh',       'MPL (v2.0)';
	license 'mpl-2.0-comma.sh', 'MPL (v2.0)';
};

subtest 'FreeType' => sub {
	license2 '-m --copyright', 'freetype.c',
		'FreeType	2001, 2012 David Turner, Robert Wilhelm, and Werner Lemberg.';
	license2 '-m --copyright --deb-fmt', 'freetype.c',
		'FTL	2001, 2012 David Turner, Robert Wilhelm, and Werner Lemberg.';
};

subtest 'CDDL' => sub {
	license2 '-m --copyright', 'cddl.h',
		'CDDL	2006 Sun Microsystems, Inc. All rights reserved';
};

subtest 'ISC' => sub {
	license2 '-m --copyright', 'libuv-isc.am',
		'ISC	2013, Ben Noordhuis <info@bnoordhuis.nl>';
};

subtest 'Machine' => sub {
	license2 '-m',        'beerware.cpp', 'Beerware';
	license2 '--machine', 'lgpl-2.1.h',   'LGPL (v2.1)';
	license2 '-m --copyright', 'gpl-2', 'GPL (v2)	2012 Devscripts developers';
};

subtest 'Fortran comments' => sub {
	license 'bsd.f', 'BSD (2 clause)';
};

subtest 'Comments detection' => sub {
	license 'comments-detection.h',   'GPL (v3 or later)';
	license 'comments-detection.txt', '*No copyright* LGPL (v2.1 or later)';
};

subtest 'False positives' => sub {
	license2 '-m --copyright', 'false-positives',
		'Public domain	2013 Devscripts developers';
};

subtest 'Regexp killer' => sub {
	license 'regexp-killer.c', 'UNKNOWN';
};

subtest 'Encoding' => sub {
	license2 '-m --copyright --encoding iso-8859-1', 'copr-iso8859.h',
		'GPL (v2) (with incorrect FSF address)	2011 Heinrich Müller <henmull@src.gnome.org>';
	license2 '-m --copyright --encoding utf8', 'copr-utf8.h',
		"GPL (v2 or later)	2004-2015 Oliva 'f00' Oberto / 2001-2010 Paul 'bar' Stevénsön";

	# test wrong user choice and fallback
	license2 '-m --copyright --encoding utf8', 'copr-iso8859.h',
		'GPL (v2) (with incorrect FSF address)	2011 Heinrich Müller <henmull@src.gnome.org>';
};

subtest 'Info at end' => sub {
	license2 '-m --copyright', 'info-at-eof.h',
		'MIT/X11 (BSD like)	1994-2012 Lua.org, PUC-Rio. All rights reserved';
};
