#!/usr/bin/perl

use strict;
use warnings;

use Software::License;
use Software::LicenseUtils;
use Path::Tiny;

use Test::More;
use Test::Script;

my %LICENSES = (
	Apache_2_0   => 'UNKNOWN',
	FreeBSD      => 'BSD (2 clause)',
	GPL_1        => 'UNKNOWN',
	GPL_2        => 'UNKNOWN',
	GPL_3        => 'UNKNOWN',
	LGPL_2       => 'LGPL (v2)',
	LGPL_2_1     => 'LGPL (v2.1)',
	LGPL_3_0     => 'LGPL (v3)',
	MIT          => 'MIT/X11 (BSD like)',
	Mozilla_2_0  => 'MPL (v2.0)',
	QPL_1_0      => 'QPL',
	Zlib         => 'zlib/libpng',
	CC0_1_0      => 'UNKNOWN',
	GFDL_1_3     => 'UNKNOWN',
	Artistic_1_0 => 'Artistic (v1.0)',
	Artistic_2_0 => 'Artistic (v2.0)',
	Mozilla_1_0  => 'MPL (v1.0)',
	None         => 'UNKNOWN',
	PostgreSQL   => 'UNKNOWN',
	AGPL_3       => 'UNKNOWN',
	SSLeay       => 'BSD (2 clause)',
	Apache_1_1   => 'UNKNOWN',
	Mozilla_1_1  => 'MPL (v1.1)',
	GFDL_1_2     => 'UNKNOWN',
	Sun          => 'UNKNOWN',
	BSD          => 'BSD (3 clause)',
	OpenSSL      => 'UNKNOWN',
	Perl_5       => 'UNKNOWN',
);
plan tests => scalar( 1 + keys %LICENSES );

my $workdir = Path::Tiny->tempdir( CLEANUP => ( not $ENV{PRESERVE} ) );
diag("Detect PRESERVE in environment, so will keep workdir: $workdir")
	if $ENV{PRESERVE};
foreach ( keys %LICENSES ) {
	my $license = Software::LicenseUtils->new_from_short_name(
		{   short_name => $_,
			holder     => 'Testophilus Testownik <tester@testity.org>',
			year       => 2000,
		}
	);
	$workdir->child($_)->spew( $license->notice, $license->license );
}
my $corpus = $workdir;
script_runs(
	[ 'bin/licensecheck', qw(--recursive -c .+), "$corpus" ],
	{ stdout => \my $stdout },
);
foreach ( split /\v+/, $stdout ) {
	if (m{^$workdir/(\w+): (.+)$}) {
		my $file    = $1;
		my $result  = $2;
		my $success = is( $result, $LICENSES{$file}, $file );
		if ( $LICENSES{$file} eq 'UNKNOWN' and $success ) {
			diag("licensecheck failed to parse $file as expected");
		}
	}
	else {
		die "Unexpected output: $_";
	}
}
