=head1 Name

SPVM::Document::Language::Class - Classes in the SPVM Language

=head1 Description

This document describes classes.

=head1 Class

The SPVM language is an object-oriented programing language and has class syntax.

This section describes class syntax.

See L<SPVM::Document::Language::SyntaxParsing> about the grammer of the SPVM language.

=head2 Class Definition

The C<class> keyword defines a L<class type|SPVM::Document::Language::Types/"Class Types">.
  
  class CLASS_NAME {
  
  }
  
  class CLASS_NAME;
  
  class CLASS_NAME : ATTRIBUTES {
  
  }
  
  class CLASS_NAME : ATTRIBUTES;

I<CLASS_NAME> is a L<class name|SPVM::Document::Language::Tokenization/"Class Name">.

I<ATTRIBUTES> is one of
  
  ATTRIBUTES ATTRIBUTE
  ATTRIBUTE

I<ATTRIBUTE> is a L<class attributre|"Class Attributes">.

A class type is also simply called a class.

The L<new operator|/"new Operator"> creates an object from a class.

Compilation Errors:

I<CLASS_NAME> must be a L<class name|SPVM::Document::Language::Tokenization/"Class Name">. Otherwise, a compilation error occurs.

If more than one class is defined in a L<class file|/"Class File">, a compilation error occurs.

Examples:

  # Examepls of class definitions
  class Point {
  
  }

=head2 Class Attributes

The List of Class Attributes:

=begin html

<table>
  <tr>
    <th>
      Class Attributes
   </th>
    <th>
      Descriptions
   </th>
  </tr>
  <tr>
    <td>
      <b>public</b>
    </td>
    <td>
      This class is public. All classes are able to create an object of this class using the <a href="https://metacpan.org/pod/SPVM::Document::Language::Operators#new-Operator">new operator</a>.
    </td>
  </tr>
  <tr>
    <td>
      <b>private</b>
    </td>
    <td>
      This class is private. Classes other than this class are not able to create an object of this class using the <a href="https://metacpan.org/pod/SPVM::Document::Language::Operators#new-Operator">new operator</a>. This is default.
    </td>
  </tr>
  <tr>
    <td>
      <b>protected</b>
    </td>
    <td>
      This class is protected. Only this class and its child classes are able to create an object of this class using the <a href="https://metacpan.org/pod/SPVM::Document::Language::Operators#new-Operator">new operator</a>.
    </td>
  </tr>
  <tr>
    <td>
      <b>interface_t</b>
    </td>
    <td>
      This class is an <a href="https://metacpan.org/pod/SPVM::Document::Language::Types#Interface-Types">interface type</a>. The class definition with this attribute is interpreted as an <a href="#Interface-Definition">interface definiton</a>.
    </td>
  </tr>
  <tr>
    <td>
      <b>mulnum_t</b>
    </td>
    <td>
      This class is a <a href="https://metacpan.org/pod/SPVM::Document::Language::Types#Multi-Numeric-Types">multi-numeric type</a>. The class definition with this attribute is interpreted as an <a href="#Multi-Numeric-Type-Definition">multi-numeric type definiton</a>.
    </td>
  </tr>
  <tr>
    <td>
      <b>pointer</b>
    </td>
    <td>
      The class is a <a href="#Pointer-Class">pointer class</a>.
    </td>
  </tr>
  <tr>
    <td>
      <b>precompile</b>
    </td>
    <td>
      The <code>precompile</code> method attribute of all methods other than a <code>INIT</code> block, methods generated by enumuration values, and methods that does not have their method block is specified.
    </td>
  </tr>
</table>

=end html

Compilation Errors:

Only one of C<private>, C<protected> or C<public> must be specified. Otherwise, a compilation error occurs.

If more than one of C<interface_t>, C<mulnum_t>, and C<pointer> are specified, a compilation error occurs.

Examples:

  # Examples of class attributes
  class Point : public {
  
  }
  
  class Point : public pointer {
  
  }

=head3 Pointer Class

The pointer class is a class with the C<pointer> L<class attribute|/"Class Attributes"> .

  class CLASS_NAME : pointer {
  
  }

An object of a pointer class has the pointer to a native address that is normally a pointer to a C language struct, or a C++ language struct or class.

Examples:

  # Examples of the pointer attribute
  class Foo : pointer {
  
  }

=head2 Class File

A class file is a file where a class is defined.

The name of a class file is the same as a class name, but all C<::> are replaced to C</> and C<.spvm> is added at the end of it.
  
  # Foo
  Foo.spvm
  
  # Foo::Bar
  Foo/Bar.spvm
  
  # Foo::Bar::Baz
  Foo/Bar/Baz.spvm

Compilation Errors:

A class definition must be written in its corresponding class file. Otherwise, a compilation error occurs.

=head2 Version Statement

The C<version> statement declares the version of a class.

  version VERSION_STRING;

This statement declares the version of a class given the L<version string|/"Version String"> I<VERSION_STRING>.

Compilation Errors:

If the version has already been declared, a compilation error occurs.

I<VERSION_STRING> must be a L<version string|/"Version String">. Otherwise, a compilation error occurs.

Examples:
  
  class MyClass {
    version "1.001";
  }
  
  class MyClass {
    version "10.001003";
  }

=head3 Version String

The version string is the string that represents the L<version|/"Version Statement"> of a class.

A version string must be written by the following rules.

The type of a version string is the string type.

It consists of C<0-9>, C<.>. 

The count of C<.> is less than or equal to 1.

It begins with C<0-9>.

It ends with C<0-9>.

The count of C<0-9> after C<.> is divisible by 3.

It is able to be parsed by the C<strtod> function in the C language.

Complication Errors:

If a version string is not valid, a compilation error occurs.

Examples:
  
  # Examples of version strings
  "1.001"
  "10.001003"
  "1"
  "10000"

=head3 Version Number

A version number is a floating point number created from a L<version string|/"Version String"> using the C<strtod> function in the C language.

Examples:

  # Examples of version numbers
  
  # "1.001"
  1.001
  
  # "10.001003"
  10.001003

=head2 use Statement

The C<use> statemenet loads a class.

  use BASIC_TYPE;
  use BASIC_TYPE as CLASS_NAME;

This statement searches for the type I<BASIC_TYPE> in L<class search directories|/"Class Search Directories"> from the beginning, and if found, loads the type.

I<BASIC_TYPE> is a L<class type|SPVM::Document::Language::Types/"Class Types">, an L<interface type|SPVM::Document::Language::Types/"Interface Types">, or a L<multi-numeric type|SPVM::Document::Language::Types/"Multi-Numeric Types">.

See the L<require statement|SPVM::Document::Language::Statements/"require Statement"> about how to load a type without causing a compile error when loading fails,

The follwoing C<use> statement with C<as>

  use BASIC_TYPE as CLASS_NAME;

is expanded to the following code using the L<alias|/"alias Statement"> statement.

  use BASIC_TYPE;
  alias BASIC_TYPE as CLASS_NAME

Compilation Errors:

I<BASIC_TYPE> must be a L<class name|SPVM::Document::Language::Tokenization/"Class Name">. Otherwise, a compilation error occurs.

I<CLASS_NAME> must be a L<class name|SPVM::Document::Language::Tokenization/"Class Name">. Otherwise, a compilation error occurs.

If I<BASIC_TYPE> does not found, a compilation error occurs.

Examples:

  # Examples of the use statement
  class MyClass {
    use Foo;
  }
  
  class MyClass {
    use Sys::Socket::Constant as SOCKET;
  }

=head2 Class Search Directories

Class search directories are directories in which classes are searched for.

These are set outside the program.

Directories set by the C<-I> option of the L<spvm> command and the L<spvmcc> command are added to class search directories.

And directories with C</SPVM> added to the end of each value of Perl's L<@INC|https://perldoc.perl.org/perlvar#@INC> are added to the end of the class search directories.

=head3 Default Loaded Classes

The following classes are loaded by default. 

=over 2

=item * L<Byte|SPVM::Byte>

=item * L<Short|SPVM::Short>

=item * L<Int|SPVM::Int>

=item * L<Long|SPVM::Long>

=item * L<Float|SPVM::Float>

=item * L<Double|SPVM::Double>

=item * L<Bool|SPVM::Bool>

=item * L<Error|SPVM::Error>

=item * L<Error::System|SPVM::Error::System>

=item * L<Error::NotSupported|SPVM::Error::NotSupported>

=item * L<Error::Compile|SPVM::Error::Compile>

=item * L<CommandInfo|SPVM::CommandInfo>

=item * L<Address|SPVM::Address>

=back

=head2 alias Statement

The C<alias> statemenet creates an alias name for a type.
  
  alias BASIC_TYPE as CLASS_NAME;

This statemenet creates an alias name I<CLASS_NAME> for a type I<BASIC_TYPE>.

I<BASIC_TYPE> is a L<class type|SPVM::Document::Language::Types/"Class Types">, an L<interface type|SPVM::Document::Language::Types/"Interface Types">, or a L<multi-numeric type|SPVM::Document::Language::Types/"Multi-Numeric Types">.

Compilation Errors:

I<BASIC_TYPE> must be a L<class name|SPVM::Document::Language::Tokenization/"Class Name">. Otherwise, a compilation error occurs.

I<CLASS_NAME> must be a L<class name|SPVM::Document::Language::Tokenization/"Class Name">. Otherwise, a compilation error occurs.

Examples:

  class MyClass {
    use Sys::Socket::Constant;
    alias Sys::Socket::Constant as SOCKET;
    
    SOCKET->AF_INET;
  }

=head2 allow Statement

The C<allow> statemenet allows a type private accesses to the current class.

  allow BASIC_TYPE;

By default, objects of private classes cannot be created by the the L<new operator|/"new Operator"> from other classes.

And private methods, private fields, and private class variables cannot be accessed from other classes.

This statement allows the type I<BASIC_TYPE> private accesses to the current class.

I<BASIC_TYPE> is loaded by the L<use statement|/"use Statement">.

Examples:

  # Allowing private access
  class MyClass {
    allow SomeClass;
  }

=head2 Inheritance

A class inherits a class using the C<extends> keyword.

  class CLASS_NAME extends PARENT_CLASS_NAME {
    
  }

The class I<CLASS_NAME> that inherits the parent class I<PARENT_CLASS_NAME> is defined.

The field definitions of the parent class I<PARENT_CLASS_NAME> are added to the end of the field definitions of the class I<CLASS_NAME> to the end of its fields.

The L<interface statements|/"interface Statement"> of the parent class I<PARENT_CLASS_NAME> are added to the end of the L<interface statements|/"interface Statement"> of the class I<CLASS_NAME>.

The instance of the class I<CLASS_NAME> can calls instance methods of the parent class I<PARENT_CLASS_NAME> and its super classes.

Compilation Errors:

The parent class I<PARENT_CLASS_NAME> must be a L<class type|SPVM::Document::Language::Types/"Class Types">. Otherwise, a compilation error occurs.

The name of the parant class must be different from the name of the class. Otherwise, a compilation error occurs.

The all super classes must be different from its own class. Otherwise, a compilation error occurs.

The field that name is the same as the field of the super class cannnot be defined. Otherwise, a compilation error occurs.

The class I<CLASS_NAME> interpreted by an interface must satisfy the L<interface requirement|SPVM::Document::Language::Types/"Interface Requirement"> to the parent class I<PARENT_CLASS_NAME>. Otherwise, a compilation error occurs.

Examples:

  class Point3D extends Point {
    
    has z : rw protected int;
    
    static method new : Point3D ($x : int = 0, $y : int = 0, $z : int = 0) {
      my $self = new Point3D;
      
      $self->{x} = $x;
      $self->{y} = $y;
      $self->{z} = $z;
      
      return $self;
    }
    
    method clear : void () {
      $self->SUPER::clear;
      $self->{z} = 0;
    }
    
    method to_string : string () {
      my $x = $self->x;
      my $y = $self->y;
      my $z = $self->z;
      
      my $string = "($x,$y,$z)";
      
      return $string;
    }
    
    method clone : object () {
      my $self_clone = Point3D->new($self->x, $self->y, $self->z);
      
      return $self_clone;
    }
  }

=head2 Interface

This section describes interfaces.

=head3 Interface Definition

A L<class definition|/"Class Definition"> with the C<interface_t> L<class attribute|/"Class Attributes"> defines an L<interface type|SPVM::Document::Language::Types/"Interface Types">.

  class CLASS_NAME : interface_t {
    
  }

An interface type is also simply called an interface.

Objects cannot be created from interface types.

Normally, an interface has L<interface methods|/"Interface Method">.

Compilation Errors:

An interface cannnot have L<field definitions|/"Field Definition">. If so, a compilation error occurs.

An interface cannnot have L<class variable definitions|/"Class Variable Definition">. If so, a compilation error occurs.

Examples:

  # Examples of interface definitions
  class TestCase::Pointable : interface_t {
    interface Stringable;
    
    method x : int ();
    method y : int();
    method to_string : string ();
  }
  
  class Stringable: interface_t {
    method to_string : string ();
    method call_to_string : string () {
      return "foo " . $self->to_string;
    }
  }

  class Stringable: interface_t {
    required method to_string : string ();
    method foo : int ($num : long);
  }

=head3 interface Statement

The C<interface> statement checks if the current class satisfies the L<interface requirement|SPVM::Document::Language::Types/"Interface Requirement"> to an interface.

  interface BASIC_TYPE;

I<BASIC_TYPE> is loaded by the L<use statement|/"use Statement">.

Compilation Errors:

The interface type I<BASIC_TYPE> must be an L<interface type|SPVM::Document::Language::Types/"Interface Types">, ohterwise a compilation error occurs.

The current class must satisfy the L<interface requirement|SPVM::Document::Language::Types/"Interface Requirement"> to the interface type I<BASIC_TYPE>, ohterwise a compilation error occurs.

Examples:
  
  # Examples of the interface statement
  class Point {
    interface Stringable;
    
    method to_string : string () {
      my $x = $self->x;
      my $y = $self->y;
      
      my $string = "($x,$y)";
      
      return $string;
    }
  }

=head2 Anon Class

An anon class is a class without specifying its class name.

=head3 Anon Class Definition

An anon class is defined by the following syntax.

  class {
  
  }

An anon class cannot be defined in a L<class file|Class File>. It is able to be defined in a source code compiled by the L<compile_anon_class|SPVM::Document::NativeAPI::Compiler/"compile_anon_class"> compiler native API.

An anon class has its L<class name|SPVM::Document::Language::Tokenization/"Class Name">, such as C<eval::anon_class::0>, C<eval::anon_class::1>, C<eval::anon_class::2>.

  eval::anon_class::123

L<Examples:>
  
  # Examples of the anon class definition
  class {
    static method sum : int ($num1 : int, $num2 : int) {
      return $num1 + $num2;
    }
  }

=head2 Anon Method Class

An anon method class is a class defined by an L<anon method operator|SPVM::Document::Language::Operators/"Anon Method Operator">.

=head3 Anon Method Class Definition

The anon method class definition has the following syntax.

  ANON_METHOD_CLASS_FIELD_DEFINITION METHOD_DEFINITION

I<ANON_METHOD_CLASS_FIELD_DEFINITION> is an L<anon method class field definition|/"Anon Method Class Field Definition">.

I<METHOD_DEFINITION> is a L<method definition|/"Method Definition">.

The anon method class definition defines a class to which this method belongs.

The name of this class is a string that joins the L<outmost class|/"Outmost Class">, a string C<"anon_method">, the line number and the position of columns where an anon method class definition is written with C<::>.

  MyClass::anon_method::3::23

An anon method class has the same access control as its outmost class.

An anon method class has the same alias names as its outmost class.

Compilation Errors:

The method name of I<METHOD_DEFINITION> must be an empty string. Otherwise, a compilation error occurs.

The method must be an instance method. Otherwise, a compilation error occurs.

Examples:
  
  # Examples of the anon method definition
  class Foo::Bar {
    method my_method : void () {
      my $comparator = (Comparator)method : int ($x1 : object, $x2 : object) {
        my $point1 = (Point)$x1;
        my $point2 = (Point)$x2;
        
        return $point1->x <=> $point2->x;
      };
    }
  }

=head4 Anon Method Class Field Definition

An anon method class field definition defines fields of an L<anon class|/"Anon Class">, which is a part of the L<anon method class definition|Anon Method Class Definition>.
  
  [ANON_METHOD_CLASS_FIELD_DEFINITION_ITEM1, ANON_METHOD_CLASS_FIELD_DEFINITION_ITEM, ANON_METHOD_CLASS_FIELD_DEFINITION_ITEMn]

I<ANON_METHOD_CLASS_FIELD_DEFINITION_ITEM> are one of
  
  has FIELD_NAME : TYPE
  has FIELD_NAME : TYPE = OPERAND
  VAR : TYPE
  
  has FIELD_NAME : ATTRIBUTE1 ATTRIBUTE2 ATTRIBUTEn TYPE
  has FIELD_NAME : ATTRIBUTE1 ATTRIBUTE2 ATTRIBUTEn TYPE = OPERAND
  VAR : ATTRIBUTE1 ATTRIBUTE2 ATTRIBUTEn TYPE

I<FIELD_NAME> is a L<field name|SPVM::Document::Language::Tokenization/"Field Name">.

I<TYPE> is a L<type|SPVM::Document::Language::Types/"Types">.

I<OPERAND> is an L<operator|SPVM::Document::Language::Operators/"Operators">.

C<VAR : TYPE> is expaneded to C<has FIELD_NAME : TYPE = OPERAND>. I<FIELD_NAME> is the same as the name of I<VAR>, but C<$> is removed. I<OPERAND> is I<VAR>.

I<ATTRIBUTE> is a L<field attribute|/"Field Attributes">.

Compilation Errors:

Compilation errors caused by L<Field Definition|/"Field Definition"> could occur.

Examples:
  
  # Examples of the anon method class field definition
  class Foo::Bar {
    method my_method : void () {
      my $foo = 1;
      my $bar = 5L;
      
      my $comparator = (Comparator)[has foo : int = $foo, has bar : long = $bar] method : int ($x1 : object, $x2 : object) {
        my $foo = $self->{foo};
        my $bar = $self->{bar};
        
        say "$foo";
        say "$bar";
      };
    }
  }
  
  # More simple
  class Foo::Bar {
    method my_method : void () {
      my $foo = 1;
      my $bar = 5L;
      
      my $comparator = (Comparator)[$foo : int, $bar : long] method : int ($x1 : object, $x2 : object) {
        say "$foo";
        say "$bar";
      };
    }
  }

=head2 Outmost Class

An outmost class is the outmost defined class. This is the same as the class defined by the L<class definition|/"Class Definition">.

  class OutmostClass {
    
  }

Examples:
  
  # Examples of the outmost class
  class MyClass {
    static method my_method : void () {
      
      # The __PACKAGE__ operator returns the outmost class "MyClass".
      my $outmost_class_name = __PACKAGE__;
      
      # Anon method
      my $cb = method : void () {
        
        # The __PACKAGE__ operator returns the outmost class "MyClass".
        my $outmost_class_name = __PACKAGE__;
      };
    }
  }

=head2 Multi-Numeric Type Definition

A L<multi-numeric type|SPVM::Document::Language::Types/"Multi-Numeric Types"> is defined by the L<class definition|/"Class Definition"> with the C<mulnum_t> L<class attribute|/"Class Attributes">.

  class CLASS_NAME : mulnum_t {
    
  }

Compilation Errors:

The type of all fields must be a L<numeric type|SPVM::Document::Language::Types/"Numeric Types">. Otherwise, a compilation error occurs.

The types of all fields must be a same type. Otherwise, a compilation error occurs.

The length of fields must be less than or equal to 255. Otherwise, a compilation error occurs.

I<CLASS_NAME> must ends with a L<multi-numeric type suffix|/"Multi-Numeric Type Suffix"> corresponding to the type of fields. Otherwise, a compilation error occurs.

I<FIELD_LENGTH> of the L<multi-numeric type suffix|/"Multi-Numeric Type Suffix"> must be the same as the length of fields. Otherwise, a compilation error occurs.

I<TYPE_SUFFIX> of the L<multi-numeric type suffix|/"Multi-Numeric Type Suffix"> must correspond to the type of fields.

Examples:
  
  # Examples of the multi-numeric type definition
  class Complex_2f : mulnum_t {
    re : float;
    im : float;
  }
  
  class Complex_2d : mulnum_t {
    re : double;
    im : double;
  }
  
  class Quaternion_4f : mulnum_t {
    re : float;
    i : float;
    j : float;
    k : float;
  }
  
  class Quaternion_4d : mulnum_t {
    re : double;
    i : double;
    j : double;
    k : double;
  }

=head3 Multi-Numeric Type Suffix

The multi-numeric type must end with the following suffix.

  _(non-spaces)FIELD_LENGTH(non-spaces)TYPE_SUFFIX

I<FIELD_LENGTH> is the length of fields.

I<TYPE_SUFFIX> is a type suffix.

The List of Type Suffixes:

=begin html

<table>
  <tr>
    <th>
      Numeric Types
   </th>
    <th>
     Type Suffixes
   </th>
  </tr>
  <tr>
    <td>
      <b>byte</b>
    </td>
    <td>
      b
    </td>
  </tr>
  <tr>
    <td>
      <b>short</b>
    </td>
    <td>
      s
    </td>
  </tr>
  <tr>
    <td>
      <b>int</b>
    </td>
    <td>
      i
    </td>
  </tr>
  <tr>
    <td>
      <b>long</b>
    </td>
    <td>
      l
    </td>
  </tr>
  <tr>
    <td>
      <b>float</b>
    </td>
    <td>
      f
    </td>
  </tr>
  <tr>
    <td>
      <b>double</b>
    </td>
    <td>
      d
    </td>
  </tr>
</table>

=end html

Examples:
  
  # Examples of the multi-numeric type suffix
  _2f;
  _2d;
  _4f;
  _4d;

=head1 Class Variable

A class variable is a global variable that belongs to a L<class|/"Class">.

=head2 Class Variable Definition

C<our> keyword defines a class variable.

  our CLASS_VARIABLE_NAME : OPT_ATTRIBUTES TYPE;

I<CLASS_VARIABLE_NAME> is a L<class variable name|SPVM::Document::Language::Tokenization/"Class Variable Name"> that does not contains C<::>.

I<TYPE> is a L<type|SPVM::Document::Language::Types/"Types">.

I<OPT_ATTRIBUTES> is one of

  EMPTY
  ATTRIBUTES

I<EMPTY> means nothing exists.

I<ATTRIBUTES> is one of
  
  ATTRIBUTES ATTRIBUTE
  ATTRIBUTE

I<ATTRIBUTE> is a L<class variable attribute|/"Class Variable Attributes">.

Each class variable is initialized by its L<type initial value|SPVM::Document::Language::Types/"Type Initial Value">.

The value of a class variable is got by the operation of L<getting a class variable|SPVM::Document::Language::Operators/"Getting a Class Variable">.

The value of a class variable is set by the operation of L<setting a class variable|SPVM::Document::Language::Operators/"Setting a Class Variable">.

Compilation Errors:

I<CLASS_VARIABLE_NAME> must a L<class variable name|SPVM::Document::Language::Tokenization/"Class Variable Name"> that does not contains C<::>. Otherwise, a compilation error occurs.

I<TYPE> must be a L<numeric type|SPVM::Document::Language::Types/"Numeric Types"> or an L<object type|SPVM::Document::Language::Types/"Object Types">.

I<ATTRIBUTE> must be a L<class variable attribute|/"Class Variable Attributes">. Otherwise, a compilation error occurs.

If two or more class variables that has a same name are defined, a compilation error occurs.

Examples:

  class MyClass {
    our $NUM_BYTE : byte;
    our $NUM_SHORT : short;
    our $NUM_INT : int;
    our $NUM_LONG : long;
    our $NUM_FLOAT : float;
    our $NUM_DOUBLE : double;
    our $POINT : Point;
    
    our $NUM_PUBLIC : public int;
    our $NUM_RO : ro int;
    our $NUM_RW : rw int;
  }

=head2 Class Variable Attributes

The List of Class Variable Attributes:

=begin html

<table>
  <tr>
    <th>
      Class Variable Attributes
   </th>
    <th>
      Descriptions
   </th>
  </tr>
  <tr>
    <td>
      <b>public</b>
    </td>
    <td>
      This class variable is public. All classes can access this class variable.
    </td>
  </tr>
  <tr>
    <td>
      <b>private</b>
    </td>
    <td>
      This class variable is private. All classes ohter than this class cannnot access this class variable. This is default.
    </td>
  </tr>
  <tr>
    <td>
      <b>protected</b>
    </td>
    <td>
      This class variable is protected. All classes ohter than this class and its child classes cannot access this class variable.
    </td>
  </tr>
  <tr>
    <td>
      <b>ro</b>
    </td>
    <td>
      This class variable defines a <a href="#Class-Variable-Getter-Method">getter method</a>.
    </td>
  </tr>
  <tr>
    <td>
      <b>wo</b>
    </td>
    <td>
      This class variable defineds a <a href="#Class-Variable-Setter-Method">setter method</a>.
    </td>
  </tr>
  <tr>
    <td>
      <b>rw</b>
    </td>
    <td>
      This class variable defines a <a href="#Class-Variable-Getter-Method">getter method</a> and a <a href="#Class-Variable-Setter-Method">setter method</a>.
    </td>
  </tr>
</table>

=end html

Compilation Errors:

One of C<private>, C<protected> and C<public> must be specified. Otherwise, a compilation error occurs.

One of C<ro>, C<wo>, and C<rw> must be specified. Otherwise, a compilation error occurs.

=head3 Class Variable Getter Method

A class variable getter method is a method to perform the operation of L<getting a class variable|SPVM::Document::Language::Operators/"Getting a Class Variable">.

This method is a class method that has no arguments.

If the type of the class variable is the byte type or the short type, the return type is the int type. Otherwise, it is the same type as the class variable.

The method name is the same as the class variable name, but C<$> is removed. For example, if the class variable name is C<$FOO>, the method name is C<FOO>.

Examples:

  # Examples of class variable getter methods
  class MyClass {
    our $NUM : ro int;
    
    static method main : void {
      my $num = Foo->NUM;
    }
  }

=head3 Class Variable Setter Method

A class variable setter method is a method to perform the operation of L<setting a class variable|SPVM::Document::Language::Operators/"Setting a Class Variable">.

This method is a class method that has an argument.

If the type of the class variable is the byte type or the short type, the argument type is the int type. Otherwise, it is the same type as the class variable.

The return type is the void type.

The method name is the same as the class variable name, but C<$> is removed and C<SET_> is added to the beginning of it. For example, if the class variable name is C<$FOO>, the method name is C<SET_FOO>.

Examples:

  # Examples of class variable setter methods
  class MyClass {
    our $NUM : wo int;
    
    static method main : void {
      Foo->SET_NUM(3);
    }
  }

=head1 Field

A field is a data member of a L<class|/"Class">.

=head2 Field Definition

The C<has> keyword defines a field.
  
  has FIELD_NAME : OPT_ATTRIBUTES TYPE;

I<FIELD_NAME> is a L<field name|SPVM::Document::Language::Tokenization/"Field Name">.

I<TYPE> is a L<type|SPVM::Document::Language::Types/"Types">.

I<OPT_ATTRIBUTES> is one of

  EMPTY
  ATTRIBUTES

I<EMPTY> means nothing exists.

I<ATTRIBUTES> is one of
  
  ATTRIBUTES ATTRIBUTE
  ATTRIBUTE

I<ATTRIBUTE> is a L<field attribute|/"Field Attributes">.

Each field of an object is initialized by its L<type initial value|SPVM::Document::Language::Types/"Type Initial Value"> when the object is created.

The value of a field is got by the operation of L<getting a field|SPVM::Document::Language::Operators/"Getting a Field">.

The value of a field is set by the operation of L<setting a field|SPVM::Document::Language::Operators/"Setting a Field">.

Compilation Errors:

I<FIELD_NAME> must a L<field name|SPVM::Document::Language::Tokenization/"Field Name">. Otherwise, a compilation error occurs.

I<TYPE> must be a L<numeric type|SPVM::Document::Language::Types/"Numeric Types"> or an L<object type|SPVM::Document::Language::Types/"Object Types">.

I<ATTRIBUTE > must be a L<field attribute|/"Field Attributes">. Otherwise, a compilation error occurs.

If two or more fields that has a same name are defined, a compilation error occurs.

Examples:

  class MyClass {
    has num_byte : byte;
    has num_short : short;
    has num_int : int;
    has num_long : long;
    has num_float : float;
    has num_double : double;
    
    has num_public : public int;
    has num_ro : ro int;
    has num_rw : rw int;
  }

=head2 Field Attributes

The List of Field Attributes:

=begin html

<table>
  <tr>
    <th>
      Field Attributes
   </th>
    <th>
      Descriptions
   </th>
  </tr>
  <tr>
    <td>
      <b>public</b>
    </td>
    <td>
      This field is public. All classes can access this field.
    </td>
  </tr>
  <tr>
    <td>
      <b>private</b>
    </td>
    <td>
      This field is private. All classes ohter than this class cannnot access this field. This is default.
    </td>
  </tr>
  <tr>
    <td>
      <b>protected</b>
    </td>
    <td>
      This field is protected. All classes ohter than this class and its child classes cannot access this field.
    </td>
  </tr>
  <tr>
    <td>
      <b>ro</b>
    </td>
    <td>
      This field defines a <a href="#Field-Getter-Method">getter method</a>.
    </td>
  </tr>
  <tr>
    <td>
      <b>wo</b>
    </td>
    <td>
      This field defineds a <a href="#Field-Setter-Method">setter method</a>.
    </td>
  </tr>
  <tr>
    <td>
      <b>rw</b>
    </td>
    <td>
      This field defines a <a href="#Field-Getter-Method">getter method</a> and a <a href="#Field-Setter-Method">setter method</a>.
    </td>
  </tr>
</table>

=end html

Compilation Errors:

One of C<private>, C<protected> and C<public> must be specified. Otherwise, a compilation error occurs.

One of C<ro>, C<wo>, and C<rw> must be specified. Otherwise, a compilation error occurs.

=head3 Field Getter Method

A field getter method is a method to perform the operation of L<getting a field|SPVM::Document::Language::Operators/"Getting a Field">.

This method is an instance method that has no arguments.

If the type of the field is the byte type or the short type, the return type is the int type. Otherwise, it is the same type as the field.

The method name is the same as the field name. For example, if the field name is C<foo>, the method name is C<foo>.

Examples:

  # Examples of field getter methods
  class MyClass {
    has num : ro int;
    
    static method new {
      return new Foo;
    }
    
    static method main : void {
      my $foo = Foo->new;
      
      my $num = $foo->num;
    }
  }

=head3 Field Setter Method

A field setter method is a method to perform the operation of L<setting a field|SPVM::Document::Language::Operators/"Setting a Field">.

This method is an instance method that has an argument.

If the type of the field is the byte type or the short type, the argument type is the int type. Otherwise, it is the same type as the field.

The return type is the void type.

The method name is the same as the field name, but C<set_> is added to the beginning of it. For example, if the field name is C<foo>, the method name is C<set_foo>.

Examples:

  # Examples of field setter methods
  class MyClass {
    has num : wo int;
    
    static method new {
      return new Foo;
    }
    
    static method main : void {
      my $foo = Foo->new;
      
      $foo->set_num(3);
    }
  }

=head1 Method

=head2 Method Definition

The C<method> keyword defines a method.

  OPT_ATTRIBUTES method METHOD_NAME : RETURN_TYPE (OPT_ARGS) { }
  OPT_ATTRIBUTES method METHOD_NAME : RETURN_TYPE (OPT_ARGS);

I<OPT_ATTRIBUTES> is one of

  EMPTY
  ATTRIBUTES

I<EMPTY> means nothing exists.

I<ATTRIBUTES> is one of
  
  ATTRIBUTES ATTRIBUTE
  ATTRIBUTE

I<ATTRIBUTE> is a L<method attribute|/"Method Attributes">.

I<METHOD_NAME> is a L<method name|SPVM::Document::Language::Tokenization/"Method Name">.

I<RETURN_TYPE> is a L<type|SPVM::Document::Language::Types/"Types">.

I<ARG_ITEM> is one of
  
  ARG_NAME : TYPE
  ARG_NAME : TYPE = VALUE

I<OPT_ATTRIBUTES> is one of

  EMPTY
  ARGS

I<EMPTY> means nothing exists.

I<ARGS> is one of
  
  ARGS , ARG
  ARG

I<ARG> is one of

  ARG_NAME : ARG_TYPE
  ARG_NAME : ARG_TYPE = VALUE

I<ARG_NAME> is a L<local variable name|SPVM::Document::Language::Tokenization/"Local Variable Name">.

I<ARG_TYPE> is a L<type|SPVM::Document::Language::Types/"Types">.

I<VALUE> is a L<literal|SPVM::Document::Language::Tokenization/"Literals"> or C<undef>. 

A method with the C<static> L<method attribute|/"Method Attributes"> is a L<class method|/"Class Method">.

A method without the C<static> L<method attribute|/"Method Attributes"> is an L<instance method|/"Instance Method">.

An argument with I<VALUE> is an optional arguments. If a method call dose not give a value to an optional argument, I<VALUE> is given to the argument.

A interface method defined in an L<interface|/"Interface"> is an L<interface method|/"Interface Method">.

A method with the C<native> L<method attribute|/"Method Attributes"> is a L<native method|/"Native Method">.

Compilation Errors:

The count of I<ARGS> must be less than or equal to 255. Otherwise, a compilation error occurs.

I<ARG_TYPE> must be a L<numeric type|SPVM::Document::Language::Types/"Numeric Types">, a L<multi-numeric type|SPVM::Document::Language::Types/"Multi-Numeric Types">, an L<object type|SPVM::Document::Language::Types/"Object Types">, or a L<reference type|SPVM::Document::Language::Types/"Reference Types">. Otherwise, a compilation error occurs.

I<RETURN_TYPE> must be the void type, a L<numeric type|SPVM::Document::Language::Types/"Numeric Types">, a L<multi-numeric type|SPVM::Document::Language::Types/"Multi-Numeric Types"> or an L<object type|SPVM::Document::Language::Types/"Object Types">. Otherwise, a compilation error occurs.

Methods other than interface methods and native methods must have a method block. Otherwise, a compilation error occurs.

I<VALUE> must satisfy L<assignment requirement|SPVM::Document::Language::Types/"Assignment Requirement"> to I<ARG_TYPE>. Otherwise, a compilation error occurs.

I<METHOD_NAME> must not be C<INIT>. Otherwise, a compilation error occurs.

A non-optional argument must not be placed after an optional argument. Otherwise, a compilation error occurs.

Examples:
  
  class MyClass {
    # Class method
    static method substr : string ($string : string, $offset : int, $length : int = -1) {
      # ...
    }
    
    # Instance method
    method clear : void () {
      
    }
  }

=head2 Method Attributes

The List of Method Attributes:

=begin html

<table>
  <tr>
    <th>
      Method Attributes
   </th>
    <th>
      Descriptions
   </th>
  </tr>
  <tr>
    <td>
      <b>public</b>
    </td>
    <td>
      This method is public. All classes can call this method. This is default.
    </td>
  </tr>
  <tr>
    <td>
      <b>private</b>
    </td>
    <td>
      This method is private. All classes ohter than this class cannnot call this method.
    </td>
  </tr>
  <tr>
    <td>
      <b>protected</b>
    </td>
    <td>
      This method is protected. All classes ohter than this class and its child classes cannot call this method.
    </td>
  </tr>
  <tr>
    <td>
      <b>precompile</b>
    </td>
    <td>
      This method is a <a href="#Precompilation-Method">precompilation method</a>.
    </td>
  </tr>
  <tr>
    <td>
      <b>native</b>
    </td>
    <td>
      This method is a <a href="#Native-Method">native method</a>.
    </td>
  </tr>
  <tr>
    <td>
      <b>required</b>
    </td>
    <td>
      A method that implements this interface method must be defined.
    </td>
  </tr>
</table>

=end html

Compilation Errors:

One of C<private>, C<protected> and C<public> must be specified. Otherwise, a compilation error occurs.

C<required> must be specified to an interface method. Otherwise, a compilation error occurs.

Examples:
  
  # private method
  private method : int sum ($num1 : int, $num2 : int) {
    return $num1 + $num2;
  }
  
  # precompile method
  precompile method : int sum ($num1 : int, $num2 : int) {
    return $num1 + $num2;
  }
  
  # native method
  native method : int sum ($num1 : int, $num2 : int);

=head2 Class Method

A method defined with the C<static> L<method attribute|/"Method Attributes"> is a class method.

A class method can be called by a L<class method call|SPVM::Document::Language::Operators/"Class Method Call">.

=head2 Instance Method

A method defined without the C<static> L<method attribute|/"Method Attributes"> is an instance method.

An instance method can be called by an L<instance method call|SPVM::Document::Language::Operators/"Instance Method Call">.

=head2 Interface Method

An instance method defined in an L<interface|/"Interface"> is an interface method.

Normally, an interface method does not have a method block.

  method my_method : int ();

But, an interface method can have a method block.

  method my_method : int () {
    
  }

An interface method can have the C<required> L<method attribute|/"Method Attributes"> that indicates classes must define a method with the same name that implements this interface method.

  required method my_method : int ();

=head2 INIT Method

A C<INIT> method is a method that is executed just after a program starts.

=head3 INIT Method Definition

A C<INIT> block defines a C<INIT> method to be executed just after the program starts. A C<INIT> method is called only once.

  INIT {
    
  }

A C<INIT> method has no arguments.

The return type of a C<INIT> method is the void type.

If a C<INIT> method is not defined in a class, a C<INIT> method without statements is defined.

If the current class has a parent class, the C<INIT> method of the parent class is called at the beginning of the INIT method of the current class.

And C<INIT> methods of classes loaded by L<use statements|/"use Statement"> are called in loaded order.

  class MyClass extends ParentClass {
    use Foo;
    use Bar;
    
    INIT {
      # INIT methods of the parent class and classes loaded by use statements are called automatically 
      ParentClass->INIT;
      Foo->INIT;
      Bar->INIT;
      
      # Do something
    }
  }

The execution order of C<INIT> methods is not guaranteed except that C<INIT> methods are called explictly and except for the above explanation.

The C<INIT> methods in the L<default loaded class|/"Default Loaded Classes"> are called before C<INIT> methods of other classes, 

Examples:
  
  # Examples of INIT methods
  class MyClass {
    use Point;
    
    our $NUM : int;
    our $STRING : string;
    our $POINT : Point;
    
    # INIT method
    INIT {
      $NUM = 3;
      $STRING = "abc";
      
      $POINT = Point->new(1, 2);
    }
  }

=head2 Destructor

A destructor is a method called just before an object is destroyed.

  method DESTROY : void () {
  
  }

A destructor is an L<instance method|/"Instance Method">.

The name of a destructor is C<DESTROY>.

A destructor has no arguments.

The retrun type is the void type.

If an L<exception|SPVM::Document::Language::ExceptionHandling/"Exception Handling"> is thrown in a destructor, the exception is not thrown. Instead, a warning message is output to L<SPVM's standard error|SPVM::Document::Language::System/"Standard Streams">.

See also L<Garbage Collection|SPVM::Document::Language::GarbageCollection/"Garbage Collection">.

Compilation Errors:

A destructor must be an L<instance method|/"Instance Method">. Otherwise, a compilation error occurs.

The name of a destructor must be C<DESTROY>. Otherwise, a compilation error occurs.

If a destructor has arguments, a compilation error occurs.

The retrun type must be the void type. Otherwise, a compilation error occurs.

If two or more destructor are defined, a compilation error occurs.

Examples:
  
  # Destructor
  class MyClass {
    method DESTROY : void () {
      say "DESTROY";
    }
  }

=head2 Method Override

An instance method in a parent class can be overridden by an instance method with the same name in a child class.

  class ChildClass extends ParentClass {
    # Method Override
    method clear : void () {
      # ...
    }
  }

  class ParentClass {
    
    method clear : void () {
      # ...
    }
  }

The overridding method in the child class must satisfy the L<interface method requirement|SPVM::Document::Language::Types/"Interface Method Requirement"> to the parent method interpretted as an interface method.

Compilation Errors:

The overridding method in the child class must satisfy the L<interface requirement|SPVM::Document::Language::Types/"Interface Requirement"> to the parent method. Otherwise, a compilation error occurs.

=head2 Native Method

A method with the C<native> L<method attribute|/"Method Attributes"> is a native method.

  native sum : int ($num1 : int, $num2 : int);

A native method has no L<method block|/"Method Block">.

The implementation of a native method is written by native languages such as the C language, C<C++>.

See L<SPVM::Document::NativeClass> about the way to write the implementation of a native method.

=head2 Precompilation Method

A method with the C<precompile> L<method attribute|/"Method Attributes"> is a precompilation method.

  precompile sum : int ($num1 : int, $num2 : int) {
  
  }

The source code of each precompilation method is converted to a C source code and it is compiled to the machine code.

And the machine codes of all precompilation methods in a class is linked to a dynamic link library such as C<MyClass.so> on Linux/Unix, C<MyClass.dll> on Windows, C<MyClass.dylib> on Mac.

The address of machine code of each precompilation method in the dynamic link library is bind to a precompilation method.

Precompilation methods need a L<build directory|SPVM::Document::EnvironmentVariables/"SPVM_BUILD_DIR"> such as C<~/.spvm_build> to compile and link them.

=head2 Bootstrap Method

A bootstrap method is a method where a program start.

  void main : void () {
    
  }

The method name is C<main>.

The return type is the void type.

It has no arguments.

=head2 Method Implementation

Normally a method has a method block. L<Statements|SPVM::Document::Language::Statements/"Statements"> can be written in a method block.
  
  static method foo : int ($num1 : int, $num2 : int) {
    
    my $total = $num1 + $num2;
    
    return $total;
  }

=head1 Local Variable

A local variable is a variable that has a L<scope|SPVM::Document::Language::GarbageCollection/"Scope">.

=head2  Local Variable Declaration

A C<my> keyword declares a local variable.

  my LOCAL_VARIABLE_NAME
  my LOCAL_VARIABLE_NAME : TYPE
  my LOCAL_VARIABLE_NAME = VALUE
  my LOCAL_VARIABLE_NAME : TYPE = VALUE

I<LOCAL_VARIABLE_NAME> is a L<local variable name|SPVM::Document::Language::Tokenization/"Local Variable Name">.

I<TYPE> is a L<type|SPVM::Document::Language::Types/"Types">.

If I<TYPE> is ommited, I<TYPE> is set to the type of I<VALUE>. This is called L<type inference|/"Type Inference">.

I<VALUE> is an L<operator|SPVM::Document::Language::Operators/"Operators">.

A local variable is declared in a L<scope block|/"Scope Block">.

A local variable declaration performs the operation of L<pushing a local variable on the mortal stack|SPVM::Document::Language::GarbageCollection/"Pushing a Local Variable on the Mortal Stack">.

A local variable declaration is a L<local variable access|SPVM::Document::Language::Operators/"Local Variable Access">.

Compilation Errors:

I<LOCAL_VARIABLE_NAME> must be a L<local variable name|SPVM::Document::Language::Tokenization/"Local Variable Name">. Otherwise a compilation error occurs.

I<TYPE> must be a L<numeric type|SPVM::Document::Language::Types/"Numeric Types">, an L<object type|SPVM::Document::Language::Types/"Object Types">, the L<multi-numeric type|SPVM::Document::Language::Types/"Multi-Numeric Types">, or a L<reference type|SPVM::Document::Language::Types/"Reference Types">. Otherwise a compilation error occurs.

If I<TYPE> is not resolved, a compilation error occurs.

Examples:

  # Examples of local variable declarations
  my $var : int;
  my $var : Point;
  my $var : Complex_2d;
  my $var : int*;
  
  my $num = 1;
  
  my $num = 1.0;
  
  my $ppp = my $bar = 4;
  
  if (my $bar = 1) {
  
  }
  
  while (my $bar = 1) {
  
  }

=head3 Type Inference

If the type of the local variable declaration is ommited, the type of the local variable is set to the type of the right operand of the assignment operator. This is called type inference.

  # int
  my $num = 1;
  
  # double
  my $num = 1.0;
  
  # Foo
  my $foo = new Foo;

=head1 Enumeration

Enumeration is a syntax that defines 32-bit integers that belongs to a L<class|/"Class">.

=head2 Enumeration Definition

The C<enum> keyword defines an enumeration.

  enum {
    ITEM1,
    ITEM2,
    ITEMn
  }

C<,> after the last enumeration item is allowed.

  enum {
    ITEM1,
    ITEM2,
    ITEM3,
  }

I<ITEM> is one of

  NAME
  NAME = VALUE

I<NAME> is a L<method name|SPVM::Document::Language::Tokenization/"Method Name">.

I<VALUE> is an L<integer literal|SPVM::Document::Language::Tokenization/"Integer Literals"> within the int type. I<VALUE> is converted the value of the int type.

If I<VALUE> of I<ITEM1> is omitted, it is set to 0.

If I<VALUE> of I<ITEMn> is ommited, it is set to the value of the previous item plus 1.

Each enumeration item is converted to a L<method definition|Method Definition> that defines a class method that returns the value of the enumeration item.

The return type of the method is the int type.

Compilation Errors:

I<NAME> must be a L<method name|SPVM::Document::Language::Tokenization/"Method Name">. Otherwise, a compilation error occurs.

I<VALUE> must be an L<integer literal|/"Integer Literal"> within the int type. Otherwise, a compilation error occurs.

Examples:
  
  # Examples of enumerations
  class MyClass {
    enum {
      FLAG1,
      FLAG2,
      FLAG3,
    }
    
    static method main : void () {
      my $flag1 = MyClass->FLAG1;
    }
  }
  
  class MyClass {
    enum {
      FLAG1,
      FLAG2 = 2,
      FLAG3,
    }
  }

=head2 Enumeration Attributes

The List of Enumeration Attributes:

=begin html

<table>
  <tr>
    <th>
      Attributes
   </th>
    <th>
      Descriptions
   </th>
  </tr>
  <tr>
    <td>
      <b>public</b>
    </td>
    <td>
      This enumeration is public. All classes can access all items of this enumeration. This is default.
    </td>
  </tr>
  <tr>
    <td>
      <b>private</b>
    </td>
    <td>
      This enumeration is private. All classes ohter than this class cannnot access all items of this enumeration.
    </td>
  </tr>
  <tr>
    <td>
      <b>protected</b>
    </td>
    <td>
      This enumeration is protected. All classes ohter than this class and its child classes cannot access all items of this enumeration.
    </td>
  </tr>
</table>

=end html

Compilation Errors:

One of C<private>, C<protected> and C<public> must be specified. Otherwise, a compilation error occurs.

=head1 Block

A block is the part enclosed by C<{> and C<}>.

=head2 Class Block

A class block is a block used in a class definition.
  
  class MyClass {
  
  }

=head2 Enumeration Block

An enumeration block is a block used in an enumeration definition.

  enum {
    ONE,
    TWO,
  }

=head2 Scope Block

A scope block is a block that has its L<scope|SPVM::Document::Language::GarbageCollection/"Scope">.

Zero or more L<statements|SPVM::Document::Language::Statements/"Statements"> are written in a scope block.

=head3 Simple Block

A simple block is a scope block.

  {
    1;
  }

A simple block must have at least one statements. Otherwise, it is intepreted as a L<key-value array initialization|SPVM::Document::Language::Operators/"Key-Value Array Initialization">.

=head3 Method Block

A method block is a scope block.

  static method foo : int () {
  
  }

=head3 INIT block

An C<INIT> block is a scope block.

  INIT {
    
  }

=head3 eval Block

An C<eval> block is a scope block.

  eval {
  
  }

=head3 if Block

An C<if> block is a scope block.

  if (CONDITION) {
  
  }

=head3 elsif Block

An C<elsif> block is a scope block.

  elsif (CONDITION) {
  
  }

=head3 else Block

An C<else> block is a scope block.

  else {
  
  }

=head3 for Block

A C<for> block is a scope block.

  for (my $i = 0; $i < 3; $i++) {
  
  }

=head3 while Block

A C<while> block is a scope block.

  while (CONDITION) {
  
  }

=head3 switch Block

A C<switch> block is a scope block.
  
  switch (CONDITION) {
  
  }

=head3 case Block

A C<case> block is a scope block.
  
  case CASE_VALUE1: {
    # ...
  }

=head3 default Block

A C<default> block is a scope block.
  
  default: {
    # ...
  }

=head1 Type Comment

The type comment is a syntax to write a comment for a type.

  TYPE of TYPE_COMMENT
  TYPE of TYPE_COMMENT1|TYPE_COMMENT2|TYPE_COMMENTn

I<TYPE> is a L<type|SPVM::Document::Language::Types/"Types">.

I<TYPE> is a type used in the L<field definition|/"Field Definition">, the L<class variable definition|/"Class Variable Definition">, the L<local variable declaration|/"Local Variable Declaration">, and the return value and the types of arguments of the L<method definition|/"Method Definition">.

I<TYPE_COMMENT> is a L<type|SPVM::Document::Language::Types/"Types">.

Examples:

  # Examples of type comments
  has points : List of Point;
  
  our $POINTS : List of Point;
  
  my $points : List of Point;
  
  static method foo : List of Point ($arg : List of Point) { ... }
  
  my $replace : object of string|Regex::Replacer;

Compilation Errors:

If the type specified as the type comment is not found, a compilation error occurs.

=head1 See Also

=over 2

=item * L<SPVM::Document::Language::SyntaxParsing>

=item * L<SPVM::Document::Language::Operators>

=item * L<SPVM::Document::Language::Statements>

=item * L<SPVM::Document::Language>

=item * L<SPVM::Document>

=back

=head1 Copyright & License

Copyright (c) 2023 Yuki Kimoto

MIT License
