#! /usr/local/bin/perl -w
# $Id: vizrcs.pl,v 1.14 1997/10/05 19:10:47 vax Exp $
# displays RCS files graphically using graphviz/dot
# copyright (c) 1997 VaX#n8 (vax@linkdead.paranoia.com)
# standard berkeley software copyright terms

use strict;
use FileHandle;
use subs qw( dot_prologue process_rlog );

# symbol to revision mapping
%main::symbol = ();
# revision to time mapping
%main::time = ();
# revision to branch (number) mapping
%main::branch = ();

$main::debug = 0;

# dot prologue
print "digraph G {\n\tnode [color=lightblue,style=filled];\n\tranksep=0.25;\n";

my($display_time, $display_branches, $display_head, $display_default_branch);
use Getopt::Long;
GetOptions('time' => \$display_time, 'branch' => \$display_branches,
	   'head' => \$display_head, 'default' => \$display_default_branch);

if (@ARGV) {
    while ($ARGV = shift) {
	my $fh = new FileHandle;
	$fh->open("rlog $ARGV |") or die "$0: could not open $ARGV: $!\n";
	process_rlog($fh);
	$fh->close;
    }
}
else {
    # read rlog output from stdin
    process_rlog(*STDIN);
}

print "}\n";

exit 0;

use Time::Local;
use Sort::Versions;

sub process_rlog {
    my($fh) = @_;
    my $lastnode = "";
    while ($_ = $fh->getline) {
	# head revision
	$display_head and /^head: ([\d+\.]*)$/ and do {
	    $main::symbol{"HEAD"} = $1;
	};
	$display_default_branch and /^branch: ([\d+\.]*)$/ and do {
	    $main::symbol{"default\\nbranch"} = $1;
	};
	# symbolic name
	/^\s+(\S+): ([\d\.]+)$/ and do {
	    warn "$1 is $2\n" if $main::debug;
	    $main::symbol{$1} = $2;
	};
	# end of symbolic names
	/^keyword/ and do {
	    my $numeric;
	    %main::rsymbol = map { $numeric = $_,
				[ grep { $main::symbol{$_} eq $numeric }
				 keys %main::symbol ] } values %main::symbol;
	    foreach (keys %main::rsymbol) {
		warn "$_ has symbols " . join(' ', @{$main::rsymbol{$_}}) . "\n"
		     if $main::debug;
	    }
	};
	# node, 1st line
	/^revision ([\d\.]+)/ and do {
	    my $nodename = $1;
	    warn "node $nodename\n" if $main::debug;
	    print "\"$nodename\" ";
	    die "could not parse $nodename\n"
		unless ($nodename =~ /^((\d+\.)*)\d+$/);
	    $main::branch{$nodename} = defined($1) ? $1 : "";
	    $main::branch{$nodename} =~ s/\.$//;
	    warn "is on branch $main::branch{$nodename}...\n" if $main::debug;
	    $lastnode = $nodename;
	};
	# node, 2nd line
	m!^date:\s+(\d+)/(\d+)/(\d+)\s+(\d+):(\d+):(\d+);
	\s+author:\s+([^;]+);
	\s+state:\s+([^;]+);
	(\s+lines:\s+([\d+-]+)\s+([\d+-]+)|)!x and do {
	    my $time = timegm($6, $5, $4, $3, $2 - 1, $1);
	    my ($author, $state, $l1, $l2) = ($7, $8);
	    $author =~ s/\"/\\\"/g;
	    $state =~ s/\"/\\\"/g;
	    my $labels;
	    if (defined($main::rsymbol{$lastnode})) {
		$labels = join("\\n", $lastnode, @{$main::rsymbol{$lastnode}});
	    }
	    else {
		$labels = $lastnode;
	    }
	    $labels = "label=\"$labels\\n$author\\n$state\"" .
		(($state eq "dead") ? ",color=palevioletred" : "");
	    print "[$labels];\n";
	    $main::time{$lastnode} = $time;
	};
    }
    # set up timeline, if necessary
    if ($display_time) {
	warn "time\n" if $main::debug;
	my $numeric;
	# maps numeric time to nodelist
	%main::rtime = map { $numeric = $_,
			     [ grep { $main::time{$_} eq $numeric }
				 keys %main::time ] } values %main::time;
	foreach (keys %main::rtime) {
	    warn "nodes at time $_; " . join(' ', @{$main::rtime{$_}}) . "\n"
		if $main::debug;
	}
	# maps integers (seconds since epoch) to human-readable versions
	my %time_nodes = map { $_, '"' . gmtime($_) . '"' } keys %main::rtime;
	# print edges
	print("{ node [shape=plaintext]; ",
	      join(" -> ", map { $time_nodes{$_} }
		   sort { $a <=> $b } keys %time_nodes), "; }; \n");
	# constrain ranks
	foreach (keys %time_nodes) {
	    print "{ rank=same; $time_nodes{$_}; " .
		join('; ', map { "\"$_\"" } @{$main::rtime{$_}}), ";}\n";
	}
    }
    my %dups = ();
    # set up edges
    foreach my $branch (grep { !$dups{$_}++ } values %main::branch) {
	# examine one branch point at a time
	warn "branch $branch\n" if $main::debug;
	# narrow down our focus to the revisions from this branch point
	my @revs_on_branch =
	    grep { $main::branch{$_} eq $branch } keys %main::branch;
	# sort the revisions on this branch in order
	@revs_on_branch = sort { versioncmp $a, $b } @revs_on_branch;
	my $parent = $branch;
	$parent =~ s/\.\d+$//;
	my $revision;
	foreach $revision (@revs_on_branch) {
	    print "\"$parent\" -> \"$revision\";\n"
		if $parent ne "1";
	    $parent = $revision;
	}
	# if there is a symbolic name for this branch, place it at
	# the end of the branch, treating it almost as a revision
	if ($display_branches and exists $main::rsymbol{$branch}) {
	    foreach my $branchsymbol (@{$main::rsymbol{$branch}}) {
		print "\"$parent\" -> \"$branchsymbol\" [dir=back,color=red,style=dashed];\n";
		print "\"$branchsymbol\" [shape=plaintext];\n";
	    }
	}
    }
}
