package Unicode::Collate::CJK::Korean;

use 5.006;
use strict;
use warnings;

our $VERSION = '1.16';

my %jamo2prim = (
    '1100', 0x376F,   '1101', 0x3770,   '1102', 0x3771,   '1103', 0x3772,
    '1105', 0x3774,   '1106', 0x3775,   '1107', 0x3776,   '1109', 0x3778,
    '110A', 0x3779,   '110B', 0x377A,   '110C', 0x377B,   '110E', 0x377D,
    '110F', 0x377E,   '1110', 0x377F,   '1111', 0x3780,   '1112', 0x3781,
    '1161', 0x37ED,   '1162', 0x37EE,   '1163', 0x37EF,   '1165', 0x37F1,
    '1166', 0x37F2,   '1167', 0x37F3,   '1168', 0x37F4,   '1169', 0x37F5,
    '116A', 0x37F6,   '116B', 0x37F7,   '116C', 0x37F8,   '116D', 0x37F9,
    '116E', 0x37FA,   '116F', 0x37FB,   '1170', 0x37FC,   '1171', 0x37FD,
    '1172', 0x37FE,   '1173', 0x37FF,   '1174', 0x3800,   '1175', 0x3801,
    '11A8', 0x384B,   '11AB', 0x384E,   '11AF', 0x3852,   '11B7', 0x385A,
    '11B8', 0x385B,   '11BC', 0x385F,   '11BD', 0x3860,
); # for DUCET v7.0.0

my(%u2e, $prim, $wt);
while (<DATA>) {
    last if /^__END__/;
    my @c = split;
    if (@c == 1 && $c[0] =~ /^[A-D]/) { # hangul
	$c[0] =~ s/^.*://;
	$prim = [ map $jamo2prim{$_}, split /-/, $c[0] ];
	$wt = 0x20;
    } else { # ideographs
	for my $c (@c) {
	    next if !$c;
	    $wt++;
	    $u2e{hex($c)} = [ $wt, @$prim ];
	}
    }
}

sub weightKorean {
    my $u = shift;
    return undef if !exists $u2e{$u};
    my @a = @{ $u2e{$u} };
    my $s = shift @a;
    my $p = shift @a;
    return([ $p, $s, 0x2, $u ], @a);
}

1;

# DATA format
# hangul syllable:jamo-jamo(-jamo)
# equiv. ideographs

__DATA__
AC00:1100-1161
4F3D 4F73 5047 50F9 52A0 53EF 5475 54E5 5609 5AC1
5BB6 6687 67B6 67B7 67EF 6B4C 73C2 75C2 7A3C 82DB
8304 8857 8888 8A36 8CC8 8DCF 8EFB 8FE6 99D5 4EEE
50A2 5496 54FF 5777 5B8A 659D 698E 6A9F 73C8 7B33
801E 8238 846D 8B0C
AC01:1100-1161-11A8
523B 5374 5404 606A 6164 6BBC 73CF 811A 89BA 89D2
95A3 537B 54AF 57C6 6409 64F1 6877
AC04:1100-1161-11AB
4F83 520A 58BE 5978 59E6 5E72 5E79 61C7 63C0 6746
67EC 687F 6F97 764E 770B 78F5 7A08 7AFF 7C21 809D
826E 8271 8AEB 9593 5058 6173 681E 69A6 7395 79C6
831B 884E 8D76 8FC0 9F66
AC08:1100-1161-11AF
4E6B 559D 66F7 6E34 78A3 7AED 845B 8910 874E 97A8
5676 696C 79F8 7FAF 880D 9DA1
AC10:1100-1161-11B7
52D8 574E 582A 5D4C 611F 61BE 6221 6562 67D1 6A44
6E1B 7518 75B3 76E3 77B0 7D3A 90AF 9451 9452 9F95
5769 57F3 5D41 5F07 61A8 64BC 6B3F 6B5B 6CD4 6DE6
6F89 77D9 8F57 9163 9E7B
AC11:1100-1161-11B8
5323 5CAC 7532 80DB 9240 9598 97D0
AC15:1100-1161-11BC
525B 5808 59DC 5CA1 5D17 5EB7 5F3A 5F4A 6177 6C5F
757A 7586 7CE0 7D73 7DB1 7F8C 8154 8221 8591 8941
8B1B 92FC 964D 9C47 508B 50F5 58C3 5FFC 625B 6760
6A7F 6BAD 77FC 7A45 7E48 7F61 7F97 7FAB 8333 8C47
97C1
AC1C:1100-1162
4ECB 4EF7 500B 51F1 584F 6137 613E 6168 6539 69EA
6F11 75A5 7686 76D6 7B87 82A5 84CB 93A7 958B 5274
5303 63E9 69E9 73A0 78D5 95D3
AC1D:1100-1162-11A8
5580 5BA2
AC31:1100-1162-11BC
5751 7CB3 7FB9 785C 8CE1 93D7
AC39:1100-1163-11A8
91B5
AC70:1100-1165
5028 53BB 5C45 5DE8 62D2 636E 64DA 64E7 6E20 70AC
795B 8DDD 8E1E 907D 9245 92F8 547F 661B 79EC 7B65
7C67 80E0 8152 82E3 8392 8556 8627 88AA 88FE 99CF
AC74:1100-1165-11AB
4E7E 4EF6 5065 5DFE 5EFA 6106 6957 8171 8654 8E47
9375 9A2B 63F5 728D 7777 8930 8B07 97AC
AC78:1100-1165-11AF
4E5E 5091 6770 6840 4E6C 6705 69A4
AC80:1100-1165-11B7
5109 528D 5292 6AA2 77BC 9210 9ED4 64BF 82A1
AC81:1100-1165-11B8
52AB 602F 8FF2 5226 5227
AC8C:1100-1166
5048 61A9 63ED
ACA9:1100-1167-11A8
64CA 683C 6A84 6FC0 8188 89A1 9694 630C 6BC4 95C3
9ABC 9B32 9D03
ACAC:1100-1167-11AB
5805 727D 72AC 7504 7D79 7E6D 80A9 898B 8B74 9063
9D51 6A2B 72F7 754E 7B67 7E33 7E7E 7F82 8832 9C39
ACB0:1100-1167-11AF
6289 6C7A 6F54 7D50 7F3A 8A23 73A6 89D6 95CB
ACB8:1100-1167-11B7
517C 614A 7B9D 8B19 9257 938C 5094 55DB 5C92 62D1
6B49 7E11 84B9 9EDA 9F38
ACBD:1100-1167-11BC
4EAC 4FD3 501E 50BE 5106 52C1 52CD 537F 5770 5883
5E9A 5F91 6176 61AC 64CE 656C 666F 66BB 66F4 6897
6D87 7085 70F1 749F 74A5 74CA 75D9 786C 78EC 7ADF
7AF6 7D45 7D93 8015 803F 811B 8396 8B66 8F15 9015
93E1 9803 9838 9A5A 9BE8 518F 5244 54FD 60F8 61BC
6243 6AA0 7162 712D 71B2 754A 7AF8 7D86 9848 7F44
8927 8B26 99C9 9BC1 9EE5
ACC4:1100-1168
4FC2 5553 583A 5951 5B63 5C46 60B8 6212 6842 68B0
68E8 6EAA 754C 7678 78CE 7A3D 7CFB 7E6B 7E7C 8A08
8AA1 8C3F 968E 9DC4 5826 70D3 7608 798A 7B53 7DAE
7E18 7F7D 846A 858A 96DE 9AFB
ACE0:1100-1169
53E4 53E9 544A 5471 56FA 59D1 5B64 5C3B 5EAB 62F7
6537 6545 6572 66A0 67AF 69C1 6CBD 75FC 7690 777E
7A3F 7F94 8003 80A1 818F 82E6 82FD 83F0 85C1 8831
88B4 8AA5 8F9C 932E 96C7 9867 9AD8 9F13 4F30 51C5
5233 5859 6772 6832 69C0 69F9 6ADC 726F 768B 76EC
77BD 7A01 7B8D 7BD9 7CD5 7F5F 7F96 7FFA 80EF 89DA
8A41 90DC 9164 9237 9760 9D23 9DF1
ACE1:1100-1169-11A8
54ED 659B 66F2 688F 7A40 8C37 9D60 56B3 69F2 7E20
89F3 8F42
ACE4:1100-1169-11AB
56F0 5764 5D11 6606 68B1 68CD 6EFE 7428 889E 9BE4
5803 5D10 6083 6346 7DC4 886E 88CD 890C 9315 95AB
9AE1 9D7E 9DA4 9F6B
ACE8:1100-1169-11AF
6C68 9AA8 6430 69BE 77FB 9DBB
ACF5:1100-1169-11BC
4F9B 516C 5171 529F 5B54 5DE5 6050 606D 62F1 63A7
653B 73D9 7A7A 86A3 8CA2 978F 5025 5D06 60BE 6831
69D3 7B9C 86E9 86EC 8D1B 8DEB 91ED 9F94
ACF6:1100-1169-11BD
4E32
ACFC:1100-116A
5BE1 6208 679C 74DC 79D1 83D3 8A87 8AB2 8DE8 904E
934B 9846 4F89 581D 5925 5938 64BE 7313 7A1E 7AA0
874C 88F9 8E1D 9299 9A0D
ACFD:1100-116A-11A8
5ED3 69E8 85FF 90ED 6901 7668 8EA9 970D 97B9
AD00:1100-116A-11AB
51A0 5B98 5BEC 6163 68FA 6B3E 704C 742F 74D8 7BA1
7F50 83C5 89C0 8CAB 95DC 9928 4E31 6DAB 721F 76E5
797C 7ABE 7B66 7DB0 8F28 9327 9475 96DA 9874 9AD6
9E1B
AD04:1100-116A-11AF
522E 605D 62EC 9002 4F78 681D 7B48 8052 9AFA 9D30
AD11:1100-116A-11BC
4F8A 5149 5321 58D9 5EE3 66E0 6D38 709A 72C2 73D6
7B50 80F1 945B 6047 6844 6846 720C 7377 78FA 7D56
7E8A 832A 8A86 8A91
AD18:1100-116B
5366 639B 7F6B 54BC 6302 7F63 8A7F
AD34:1100-116C
4E56 5080 584A 58DE 602A 6127 62D0 69D0 9B41 5ABF
5EE5 6060 7470 749D 84AF 8958
AD35:1100-116C-11A8
9998
AD49:1100-116C-11BC
5B8F 7D18 80B1 8F5F 6D64 89E5 8A07 958E
AD50:1100-116D
4EA4 50D1 54AC 55AC 5B0C 5DA0 5DE7 652A 654E 6821
6A4B 72E1 768E 77EF 7D5E 7FF9 81A0 854E 86DF 8F03
8F4E 90CA 9903 9A55 9BAB 4F7C 5604 5610 566D 5699
59E3 618D 649F 6648 669E 69B7 78FD 7A96 8DAB 8E7B
9278 9AB9 9D41 9F69
AD6C:1100-116E
4E18 4E45 4E5D 4EC7 4FF1 5177 52FE 5340 53E3 53E5
548E 5614 5775 57A2 5BC7 5D87 5ED0 61FC 62D8 6551
67B8 67E9 69CB 6B50 6BC6 6BEC 6C42 6E9D 7078 72D7
7396 7403 77BF 77E9 7A76 7D7F 8009 81FC 8205 820A
82DF 8862 8B33 8CFC 8EC0 9011 90B1 9264 92B6 99D2
9A45 9CE9 9DD7 9F9C 4F49 4F5D 4FC5 50B4 5193 52AC
5336 53B9 53F4 5778 59E4 5ABE 5AD7 5C68 5CA3 5F40
6235 6263 6344 6406 6473 662B 6998 6F1A 7486 750C
759A 75C0 766F 7A9B 7AB6 7BDD 7CD7 80CA 849F 86AF
88D8 89AF 8A6C 9058 91E6 97DD 97ED 97EE 98B6 99C8
9B2E 9DC7 9E1C
AD6D:1100-116E-11A8
570B 5C40 83CA 97A0 97AB 9EB4 530A 63AC 8DFC 9EAF
AD70:1100-116E-11AB
541B 7A98 7FA4 88D9 8ECD 90E1 6343 687E 76B8
AD74:1100-116E-11AF
5800 5C48 6398 7A9F 5014 5D1B 6DC8 8A58
AD81:1100-116E-11BC
5BAE 5F13 7A79 7AAE 828E 8EAC 8EB3
AD8C:1100-116F-11AB
5026 5238 52F8 5377 5708 62F3 6372 6B0A 6DC3 7737
52CC 60D3 68EC 7760 7DA3 8737
AD90:1100-116F-11AF
53A5 7357 8568 8E76 95D5
ADA4:1100-1170
673A 6AC3 6F70 8A6D 8ECC 994B 4F79 51E0 5282 5331
6192 6485 6A3B 6C3F 7C0B 7E62 8DEA 95E0 993D 9E82
ADC0:1100-1171
6677 6B78 8CB4 9B3C
ADDC:1100-1172
53EB 572D 594E 63C6 69FB 73EA 7845 7ABA 7AC5 7CFE
8475 898F 8D73 9035 95A8 5232 5AE2 5B00 5DCB 668C
694F 6A1B 6F59 777D 7CFA 866C 866F 8DEC 90BD 95DA
980D 9997
ADE0:1100-1172-11AB
52FB 5747 7547 7B60 83CC 921E 56F7 9E8F
ADE4:1100-1172-11AF
6A58
ADF9:1100-1173-11A8
514B 524B 5287 621F 68D8 6975 9699 4E9F 5C05 5C50
90C4
ADFC:1100-1173-11AB
50C5 52A4 52E4 61C3 65A4 6839 69FF 747E 7B4B 82B9
83EB 89B2 8B39 8FD1 9949 537A 53AA 5890 5DF9 5ED1
6F0C 89D4 8DDF 91FF 9773
AE08:1100-1173-11B7
4ECA 5997 64D2 6611 6A8E 7434 7981 79BD 82A9 887E
887F 895F 9326 552B 5664 5D94 7B12 9EC5
AE09:1100-1173-11B8
4F0B 53CA 6025 6271 6C72 7D1A 7D66 573E 5C8C 7680
790F 7B08 82A8
AE0D:1100-1173-11BC
4E98 5162 77DC 80AF 4E99 6B91
AE30:1100-1175
4F01 4F0E 5176 5180 55DC 5668 573B 57FA 57FC 5914
5947 5993 5BC4 5C90 5D0E 5DF1 5E7E 5FCC 6280 65D7
65E3 671E 671F 675E 68CB 68C4 6A5F 6B3A 6C23 6C7D
6C82 6DC7 7398 7426 742A 7482 74A3 7578 757F 7881
78EF 7941 7947 7948 797A 7B95 7D00 7DBA 7F88 8006
802D 808C 8A18 8B4F 8C48 8D77 9321 9324 98E2 9951
9A0E 9A0F 9A65 9E92 50DB 525E 588D 5C7A 5E8B 5F03
5FEE 612D 638E 6532 65C2 66A3 66C1 68CA 6B67 7081
7309 79A8 7DA5 7DA6 7F87 80B5 82AA 82B0 8604 8641
871D 87E3 8989 89AC 8DC2 9691 980E 9B10 9C2D 9ED6
AE34:1100-1175-11AB
7DCA
AE38:1100-1175-11AF
4F76 5409 62EE 6854 59DE 86E3
AE40:1100-1175-11B7
91D1
B07D:1101-1175-11A8
55AB
B098:1102-1161
513A 5A1C 61E6 62CF 62FF 90A3 6310 632A 689B 7CE5
7CEF
B099:1102-1161-11A8
8AFE
B09C:1102-1161-11AB
6696 7156 96E3 5044 7157 8D67 992A
B0A0:1102-1161-11AF
634F 637A
B0A8:1102-1161-11B7
5357 678F 6960 6E73 7537 5583 67DF
B0A9:1102-1161-11B8
7D0D 8872
B0AD:1102-1161-11BC
56CA 5A18 66E9
B0B4:1102-1162
4E43 5167 5948 67F0 8010 5302 5976 5B2D 8FFA 9F10
B140:1102-1167
5973
B141:1102-1167-11A8
60C4
B144:1102-1167-11AB
5E74 649A 79CA 78BE
B150:1102-1167-11B7
5FF5 606C 62C8 637B
B155:1102-1167-11BC
5BE7 5BD7 4F5E 511C 5680 6FD8
B178:1102-1169
52AA 5974 5F29 6012 7459 99D1 5476 5B65 5CF1 7331
7B2F 81D1
B18D:1102-1169-11BC
6FC3 81BF 8FB2 5102 5665 7A60 91B2
B1CC:1102-116C
60F1 8166 9912
B1E8:1102-116D
5C3F 5ACB 5B32 6DD6 78E0 88CA 9403
B204:1102-116E
5542 8028
B208:1102-116E-11AB
5AE9
B20C:1102-116E-11AF
8A25 5436 80AD
B274:1102-1172
677B 7D10 5FF8 9775
B275:1102-1172-11A8
8844
B2A5:1102-1173-11BC
80FD
B2C8:1102-1175
5C3C 6CE5 5462 6029 67C5 7962 79B0 81A9
B2C9:1102-1175-11A8
533F 6EBA 6635 66B1
B2E4:1103-1161
591A 8336 7239
B2E8:1103-1161-11AB
4E39 4EB6 4F46 55AE 5718 58C7 5F56 65B7 65E6 6A80
6BB5 6E4D 77ED 7AEF 7C1E 7DDE 86CB 8892 9132 935B
6171 62C5 6934 6F19 7649 8011 80C6 8176 8711
B2EC:1103-1161-11AF
64BB 6FBE 737A 75B8 9054 59B2 601B 95E5 977C 97C3
B2F4:1103-1161-11B7
5556 574D 61BA 64D4 66C7 6DE1 6E5B 6F6D 6FB9 75F0
8043 81BD 8541 8983 8AC7 8B5A 931F 510B 5557 5649
58B0 58DC 6BEF 79AB 7F4E 859D 90EF 9EEE 9EF5
B2F5:1103-1161-11B8
6C93 7553 7B54 8E0F 905D
B2F9:1103-1161-11BC
5510 5802 5858 5E62 6207 649E 68E0 7576 7CD6 87B3
9EE8 5018 513B 515A 642A 6A94 6E8F 746D 74AB 77A0
7911 87F7 8960 8B9C 93DC 943A 9933 9939
B300:1103-1162
4EE3 5788 576E 5927 5C0D 5CB1 5E36 5F85 6234 64E1
73B3 81FA 888B 8CB8 968A 9EDB 5113 61DF 65F2 6C4F
7893 9413
B301:1103-1162-11A8
5B85
B355:1103-1165-11A8
5FB7 60B3
B3C4:1103-1169
5012 5200 5230 5716 5835 5857 5C0E 5C60 5CF6 5D8B
5EA6 5F92 60BC 6311 6389 6417 6843 68F9 6AC2 6DD8
6E21 6ED4 6FE4 71FE 76DC 7779 79B1 7A3B 8404 89A9
8CED 8DF3 8E48 9003 9014 9053 90FD 934D 9676 97DC
53E8 58D4 5F22 5FC9 6146 638F 642F 64E3 6AAE 6D2E
6D82 7A0C 83DF 9174 95CD 9780 97B1 9955 9F17
B3C5:1103-1169-11A8
6BD2 7006 7258 72A2 7368 7763 79BF 7BE4 7E9B 8B80
6ADD 9EF7
B3C8:1103-1169-11AB
58A9 60C7 6566 65FD 66BE 6C8C 711E 71C9 8C5A 9813
5F34 6F61 8E89
B3CC:1103-1169-11AF
4E6D 7A81 5484 5817
B3D9:1103-1169-11BC
4EDD 51AC 51CD 52D5 540C 61A7 6771 6850 68DF 6D1E
6F7C 75BC 77B3 7AE5 80F4 8463 9285 4F97 50EE 54C3
578C 5CD2 5F64 6723 6A66 6DB7 825F 82F3 833C 856B
8740 932C 9B97
B450:1103-116E
515C 6597 675C 6793 75D8 7AC7 8373 8C46 9017 982D
6296 6581 809A 8130 86AA 8839 9661
B454:1103-116E-11AB
5C6F 81C0 829A 9041 906F 920D 7A80 8FCD
B458:1103-116E-11AF
4E67
B4DD:1103-1173-11A8
5F97
B4F1:1103-1173-11BC
5D9D 6A59 71C8 767B 7B49 85E4 8B04 9127 9A30 51F3
58B1 6ED5 78F4 7C50 7E22 87A3 9419
B77C:1105-1161
5587 61F6 7669 7F85 863F 87BA 88F8 908F 502E 56C9
66EA 7630 7822 81DD 947C 9A3E 9A58
B77D:1105-1161-11A8
6D1B 70D9 73DE 7D61 843D 916A 99F1 55E0 7296
B780:1105-1161-11AB
4E82 5375 6B04 6B12 703E 721B 862D 9E1E 5B3E 5E71
6514 7053 8974 947E 95CC
B784:1105-1161-11AF
524C 8FA3 57D2 8FA2
B78C:1105-1161-11B7
5D50 64E5 652C 6B16 6FEB 7C43 7E9C 85CD 8964 89BD
5A6A 60CF
B78D:1105-1161-11B8
62C9 81D8 881F 945E
B791:1105-1161-11BC
5ECA 6717 6D6A 72FC 7405 746F 8782 90DE 6994 7860
7A02 83A8 870B 95AC
B798:1105-1162
4F86 5D0D 5FA0 840A 6DF6 9A0B
B7AD:1105-1162-11BC
51B7
B7B5:1105-1163-11A8
63A0 7565 7567
B7C9:1105-1163-11BC
4EAE 5006 5169 51C9 6881 6A11 7CAE 7CB1 7CE7 826F
8AD2 8F1B 91CF 55A8 60A2 690B 6DBC 8E09 9B4E
B824:1105-1167
4FB6 5137 52F5 5442 5EEC 616E 623E 65C5 6ADA 6FFE
792A 85DC 8823 95AD 9A62 9A6A 9E97 9ECE 5122 53B2
5533 68A0 7658 7CF2 8182 81DA 8821 908C 9462
B825:1105-1167-11A8
529B 66C6 6B77 701D 792B 8F62 9742 650A 6ADF 6AEA
7667 8F63 9148
B828:1105-1167-11AB
6190 6200 6523 6F23 7149 7489 7DF4 806F 84EE 8F26
9023 934A 5B4C 695D 6E45 81E0 93C8 9C0A 9C31
B82C:1105-1167-11AF
51BD 5217 52A3 6D0C 70C8 88C2 6312 6369 98B2
B834:1105-1167-11B7
5EC9 6582 6BAE 6FC2 7C3E 5969 7032 78CF
B835:1105-1167-11B8
7375 8E90 9B23
B839:1105-1167-11BC
4EE4 4F36 56F9 5CBA 5DBA 601C 73B2 7B2D 7F9A 7FCE
8046 901E 9234 96F6 9748 9818 9F61 53E6 5464 59C8
5CAD 6624 6B1E 6CE0 79E2 82D3 86C9 8EE8 9D12 9E77
B840:1105-1168
4F8B 6FA7 79AE 91B4 96B7 96B8 9C67
B85C:1105-1169
52DE 6488 64C4 6AD3 6F5E 7018 7210 76E7 8001 8606
865C 8DEF 8F05 9732 9B6F 9DFA 9E75 58DA 6EF7 7388
7646 7A82 826A 826B 8F64 942A 946A 9871 9AD7 9C78
9E15
B85D:1105-1169-11A8
788C 797F 7DA0 83C9 9304 9E7F 9E93 5725 5F54 6DE5
6F09 7C0F 8F46 9A04
B860:1105-1169-11AB
8AD6
B871:1105-1169-11BC
58DF 5F04 6727 7027 74CF 7C60 807E 5131 650F 66E8
7931 8622 96B4 9F8E
B8B0:1105-116C
5121 7028 7262 78CA 8CC2 8CDA 8CF4 96F7 6502 790C
7927 7C5F 7E87 7F4D 8012 857E 8A84 9179 9842
B8CC:1105-116D
4E86 50DA 5BEE 5ED6 6599 71CE 7642 77AD 804A 84FC
907C 9B27 5639 5AFD 64A9 66B8 6F66 7360 7E5A 818B
91AA 9410 98C2 98C9
B8E1:1105-116D-11BC
9F8D 9F92
B8E8:1105-116E
58D8 5A41 5C62 6A13 6DDA 6F0F 763B 7D2F 7E37 851E
8938 93E4 964B 50C2 560D 5D81 617A 802C 87BB 9ACF
B958:1105-1172
5289 65D2 67F3 69B4 6D41 6E9C 700F 7409 7460 7559
7624 786B 8B2C 985E 6A4A 7E32 7E8D 905B 9DB9
B959:1105-1172-11A8
516D 622E 9678 52E0
B95C:1105-1172-11AB
4F96 502B 5D19 6DEA 7DB8 8F2A 6384
B960:1105-1172-11AF
5F8B 6144 6817 5D42 6EA7
B96D:1105-1172-11BC
9686 7643 7ABF
B975:1105-1173-11A8
52D2 808B 6CD0
B984:1105-1173-11B7
51DC 51DB 5EE9 6F9F
B989:1105-1173-11BC
51CC 695E 7A1C 7DBE 83F1 9675 5030 8506
B9AC:1105-1175
4FDA 5229 5398 540F 550E 5C65 60A7 674E 68A8 6D6C
7281 72F8 7406 7483 75E2 7C6C 7F79 7FB8 8389 88CF
88E1 91CC 91D0 96E2 9BC9 4FD0 527A 54E9 5AE0 6D96
6F13 79BB 8385 870A 87AD 8C8D 9090 9B51 9ED0
B9B0:1105-1175-11AB
541D 6F7E 71D0 7498 85FA 8EAA 96A3 9C57 9E9F 5D99
608B 735C 78F7 7CA6 7CBC 7E57 8E99 8F54 9130 93FB
9A4E
B9BC:1105-1175-11B7
6797 6DCB 7433 81E8 9716 75F3
B9BD:1105-1175-11B8
782C 7ACB 7B20 7C92 5CA6
B9C8:1106-1161
6469 746A 75F2 78BC 78E8 99AC 9B54 9EBB 5298 5ABD
879E 87C7 9EBD 9EBF
B9C9:1106-1161-11A8
5BDE 5E55 6F20 819C 83AB 9088 7799 93CC
B9CC:1106-1161-11AB
4E07 534D 5A29 5DD2 5F4E 6162 633D 6669 66FC 6EFF
6F2B 7063 779E 842C 8513 883B 8F13 9945 9C3B 5881
5ADA 5E54 7E35 8B3E 8E63 93CB 93DD 9B18
B9D0:1106-1161-11AF
551C 62B9 672B 6CAB 8309 896A 977A 5E15 79E3
B9DD:1106-1161-11BC
4EA1 5984 5FD8 5FD9 671B 7DB2 7F54 8292 832B 83BD
8F1E 9099 60D8 6C52 6F2D 83BE 87D2 9B4D
B9E4:1106-1162
57CB 59B9 5A92 5BD0 6627 679A 6885 6BCF 7164 7F75
8CB7 8CE3 9081 9B45 5446 6973 6CAC 73AB 771B 82FA
8393 9176 9709
B9E5:1106-1162-11A8
8108 8C8A 964C 9A40 9EA5 8109 8C83 8C98
B9F9:1106-1162-11BC
5B5F 6C13 731B 76F2 76DF 840C 511A 750D 753F 867B
BA71:1106-1167-11A8
51AA 8993 5E4E 7CF8
BA74:1106-1167-11AB
514D 5195 52C9 68C9 6C94 7704 7720 7DBF 7DEC 9762
9EB5 4FDB 6E4E 7CC6 7DDC 9EAA
BA78:1106-1167-11AF
6EC5 8511 7BFE 884A
BA85:1106-1167-11BC
51A5 540D 547D 660E 669D 6927 6E9F 76BF 7791 8317
84C2 879F 9169 9298 9CF4 6D3A
BA8C:1106-1168
8882
BAA8:1106-1169
4FAE 5192 52DF 59C6 5E3D 6155 6478 6479 66AE 67D0
6A21 6BCD 6BDB 725F 7261 7441 7738 77DB 8017 82BC
8305 8B00 8B28 8C8C 4F94 59E5 5AA2 5AEB 6048 65C4
7683 770A 7C8D 7CE2 8004 8765 87CA 927E 9AE6
BAA9:1106-1169-11A8
6728 6C90 7267 76EE 7766 7A46 9DA9 51E9 82DC
BAB0:1106-1169-11AF
6B7F 6C92
BABD:1106-1169-11BC
5922 6726 8499 5E6A 61DE 66DA 6E95 6FDB 77A2 77C7
8268 96FA 9E0F
BB18:1106-116D
536F 5893 5999 5EDF 63CF 6634 6773 6E3A 732B 7AD7
82D7 9328 6DFC 7707 85D0 8C93
BB34:1106-116E
52D9 5DEB 61AE 61CB 620A 62C7 64AB 65E0 6959 6B66
6BCB 7121 73F7 755D 7E46 821E 8302 856A 8AA3 8CBF
9727 9D61 511B 5638 5EE1 81B4 9A16
BB35:1106-116E-11A8
58A8 9ED8 563F
BB38:1106-116E-11AB
5011 520E 543B 554F 6587 6C76 7D0A 7D0B 805E 868A
9580 96EF 5301 6097 61E3 6286 636B 7086 748A
BB3C:1106-116E-11AF
52FF 6C95 7269
BBF8:1106-1175
5473 5A9A 5C3E 5D4B 5F4C 5FAE 672A 68B6 6963 6E3C
6E44 7709 7C73 7F8E 8587 8B0E 8FF7 9761 9EF4 4EB9
5A13 5A84 5ABA 5F25 5F2D 6549 7030 737C 7CDC 7E3B
82FF 863C 9E8B
BBFC:1106-1175-11AB
5CB7 60B6 610D 61AB 654F 65FB 65FC 6C11 6CEF 739F
73C9 7DE1 9594 5FDE 5FDF 668B 6E63 7DCD 7F60 82E0
95A9 9C35 9EFD
BC00:1106-1175-11AF
5BC6 871C 8B10 6A12 6EF5
BC15:1107-1161-11A8
525D 535A 62CD 640F 64B2 6734 6A38 6CCA 73C0 749E
7B94 7C95 7E1B 818A 8236 8584 8FEB 96F9 99C1 4EB3
6B02 7254 939B 99EE 9AC6
BC18:1107-1161-11AB
4F34 534A 53CD 53DB 62CC 642C 6500 6591 69C3 6CEE
6F58 73ED 7554 7622 76E4 76FC 78D0 78FB 792C 7D46
822C 87E0 8FD4 9812 98EF 5ABB 6273 642B 653D 670C
80D6 878C 9816
BC1C:1107-1161-11AF
52C3 62D4 64A5 6E24 6F51 767C 8DCB 91B1 9262 9AEE
9B43 54F1 6D61 8116 9238 9D53
BC29:1107-1161-11BC
5023 508D 574A 59A8 5C28 5E47 5F77 623F 653E 65B9
65C1 6609 678B 699C 6EC2 78C5 7D21 80AA 8180 822B
82B3 84A1 868C 8A2A 8B17 90A6 9632 9F90 4EFF 5396
5E6B 5FAC 6412 65CA 6886 7253 823D 8783 938A 9AE3
9B74
BC30:1107-1162
500D 4FF3 57F9 5F98 62DC 6392 676F 6E43 7119 76C3
80CC 80DA 88F4 88F5 8919 8CE0 8F29 914D 966A 574F
576F 6252 7432 84D3
BC31:1107-1162-11A8
4F2F 4F70 5E1B 67CF 6822 767D 767E 9B44 7CA8
BC88:1107-1165-11AB
5E61 6A0A 7169 71D4 756A 7E41 8543 85E9 98DC 7E59
7FFB 81B0 8629 88A2
BC8C:1107-1165-11AF
4F10 7B4F 7F70 95A5 6A43 7F78
BC94:1107-1165-11B7
51E1 5E06 68B5 6C3E 6C4E 6CDB 72AF 7BC4 8303 7B35
8A09 98BF
BC95:1107-1165-11B8
6CD5 743A
BCBD:1107-1167-11A8
50FB 5288 58C1 64D8 6A97 74A7 7656 78A7 8617 95E2
9739 64D7 7513 7588 895E 9DFF 9F0A
BCC0:1107-1167-11AB
535E 5F01 8B8A 8FA8 8FAF 908A 5FED 6283 7C69 8FAE
8141 8CC6 99E2 9ABF 9D18
BCC4:1107-1167-11AF
5225 77A5 9C49 9F08 5F46 9DE9
BCD1:1107-1167-11BC
4E19 5002 5175 5C5B 5E77 661E 663A 67C4 68C5 70B3
7501 75C5 79C9 7ADD 8F27 9920 9A08 4E26 5840 7D63
7F3E 8FF8 9235 92F2 927C
BCF4:1107-1169
4FDD 5821 5831 5BF6 666E 6B65 6D11 6E7A 6F7D 73E4
752B 83E9 88DC 8913 8B5C 8F14 4FCC 76D9 7C20 8446
974C 9D07 9EFC
BCF5:1107-1169-11A8
4F0F 50D5 5310 535C 5B93 5FA9 670D 798F 8179 832F
8514 8907 8986 8F39 8F3B 99A5 9C12 58A3 5E5E 6251
6FEE 7B99 83D4 8760 876E 9D69
BCF8:1107-1169-11AB
672C
BCFC:1107-1169-11AF
4E76
BD09:1107-1169-11BC
4FF8 5949 5C01 5CEF 5CF0 6367 68D2 70FD 71A2 742B
7E2B 84EC 8702 9022 92D2 9CF3 4E30 5906 7BF7 7D98
83F6 9D0C
BD80:1107-116E
4E0D 4ED8 4FEF 5085 5256 526F 5426 5490 57E0 592B
5A66 5B5A 5B75 5BCC 5E9C 6276 6577 65A7 6D6E 6EA5
7236 7B26 7C3F 7F36 8150 8151 819A 8240 8299 83A9
8A03 8CA0 8CE6 8CFB 8D74 8DBA 90E8 91DC 961C 9644
99D9 9CE7 4EC6 4FD8 5A8D 6294 62CA 638A 6874 6991
6DAA 739E 7954 7B5F 7F58 7F66 80D5 82A3 82FB 8500
86A8 8709 889D 88D2 8DD7 9207 982B 9B92 9EA9
BD81:1107-116E-11A8
5317
BD84:1107-116E-11AB
5206 5429 5674 58B3 5954 596E 5FFF 61A4 626E 6610
6C7E 711A 76C6 7C89 7CDE 7D1B 82AC 8CC1 96F0 4F53
574C 5E09 678C 68FB 68FC 6C1B 6E53 6FC6 7287 755A
780F 7B28 80A6 81B9 8561 8F52 9EFA 9F22
BD88:1107-116E-11AF
4F5B 5F17 5F7F 62C2 5CAA 7953 7D31 8274 8300 97CD
9AF4 9EFB
BD95:1107-116E-11BC
5D29 670B 68DA 787C 7E43 9D6C 580B 6F30 9B05
BE44:1107-1175
4E15 5099 5315 532A 5351 5983 5A62 5E87 60B2 618A
6249 6279 6590 6787 69A7 6BD4 6BD6 6BD7 6BD8 6CB8
7435 75FA 7812 7891 79D5 79D8 7C83 7DCB 7FE1 80A5
813E 81C2 83F2 871A 88E8 8AB9 8B6C 8CBB 9119 975E
98DB 9F3B 4EF3 4FFE 5255 572E 57E4 59A3 5C41 5EB3
60B1 68D0 6911 6C98 6DDD 6DE0 6FDE 72C9 72D2 75DE
75F9 7765 7955 7BE6 7D15 7F86 8153 8298 82BE 8406
84D6 868D 8C94 8D14 8F61 90B3 90EB 959F 9674 970F
97B4 9A11 9A1B 9AC0 9F19
BE48:1107-1175-11AB
56AC 5B2A 5F6C 658C 6AB3 6BAF 6D5C 6FF1 7015 725D
73AD 8CA7 8CD3 983B 5110 64EF 77C9 7E7D 81CF 860B
8C73 90A0 944C 9726 9870 9B02 9B22
BE59:1107-1175-11BC
6191 6C37 8058 9A01 51B0 51ED 51F4 5A09
C0AC:1109-1161
4E4D 4E8B 4E9B 4ED5 4F3A 4F3C 4F7F 4FDF 50FF 53F2
53F8 5506 55E3 56DB 58EB 5962 5A11 5BEB 5BFA 5C04
5DF3 5E2B 5F99 601D 6368 659C 65AF 67F6 67FB 68AD
6B7B 6C99 6CD7 6E23 7009 7345 7802 793E 7940 7960
79C1 7BE9 7D17 7D72 8086 820D 838E 84D1 86C7 88DF
8A50 8A5E 8B1D 8CDC 8D66 8FAD 90AA 98FC 99DF 9E9D
509E 525A 5378 548B 59D2 6942 69AD 6C5C 75E7 76B6
7AE2 7B25 7F37 8721 8997 99DB 9B66 9BCA 9C24
C0AD:1109-1161-11A8
524A 6714 69CA 720D 84B4 9460
C0B0:1109-1161-11AB
5098 522A 5C71 6563 6C55 73CA 7523 759D 7B97 849C
9178 9730 5277 59CD 5B7F 6A75 6F78 6F98 72FB 7E56
8A15 93DF 958A 6BFF
C0B4:1109-1161-11AF
4E77 6492 6BBA 715E 85A9
C0BC:1109-1161-11B7
4E09 6749 68EE 6E17 829F 8518 886B 7CDD 91E4 9B16
C0BD:1109-1161-11B8
63F7 6F81 9212 98AF 5345 553C 6B43 7FE3 9364 9705
970E
C0C1:1109-1161-11BC
4E0A 50B7 50CF 511F 5546 55AA 5617 5B40 5C19 5CE0
5E38 5E8A 5EA0 5EC2 60F3 6851 6A61 6E58 723D 7240
72C0 76F8 7965 7BB1 7FD4 88F3 89F4 8A73 8C61 8CDE
971C 587D 5F9C 664C 6BA4 751E 7DD7 939F 9859 9B3A
C0C8:1109-1162
585E 74BD 8CFD 9C13
C0C9:1109-1162-11A8
55C7 7A61 7D22 8272 69ED 6FC7 7012
C0DD:1109-1162-11BC
7272 751F 7525 7B19 771A 924E
C11C:1109-1165
5885 58FB 5DBC 5E8F 5EB6 5F90 6055 6292 637F 654D
6691 66D9 66F8 6816 68F2 7280 745E 7B6E 7D6E 7DD6
7F72 80E5 8212 85AF 897F 8A93 901D 92E4 9ECD 9F20
566C 5A7F 63DF 6495 6E51 6FA8 7D13 8021 82A7 924F
C11D:1109-1165-11A8
5915 596D 5E2D 60DC 6614 6673 6790 6C50 6DC5 6F5F
77F3 78A9 84C6 91CB 932B 6670 77FD 814A 8203 8725
9250 9F2B
C120:1109-1165-11AB
4ED9 50CA 5148 5584 5B0B 5BA3 6247 657E 65CB 6E32
717D 7401 7444 7487 74BF 766C 79AA 7DDA 7E55 7FA8
817A 81B3 8239 861A 87EC 8A75 8DE3 9078 9291 9425
994D 9BAE 58A1 5AD9 5C1F 5C20 5C73 6103 6B5A 71AF
7B45 7DAB 8B54 8B71 93C7 9A38 9C53 9C7B
C124:1109-1165-11AF
5368 5C51 6954 6CC4 6D29 6E2B 820C 859B 893B 8A2D
8AAA 96EA 9F67 5070 5A9F 63F2 66AC 7207 789F 7A27
7D32
C12C:1109-1165-11B7
5261 66B9 6BB2 7E96 87FE 8D0D 9583 965D 5B45 61B8
647B 7752 8B6B 929B 97F1
C12D:1109-1165-11B8
651D 6D89 71EE 56C1 61FE 7044 8076 8EA1 9477 9873
C131:1109-1165-11BC
57CE 59D3 5BAC 6027 60FA 6210 661F 665F 7329 73F9
76DB 7701 7B6C 8056 8072 8165 8AA0 9192 7446 9A02
C138:1109-1166
4E16 52E2 6B72 6D17 7A05 7B39 7D30 8CB0 5E28 6D12
7E50 86FB
C18C:1109-1169
53EC 562F 5851 5BB5 5C0F 5C11 5DE2 6240 6383 6414
662D 68B3 6CBC 6D88 6EAF 701F 70A4 71D2 7526 758F
758E 7619 7B11 7BE0 7C2B 7D20 7D39 852C 856D 8607
8A34 900D 9061 90B5 92B7 97F6 9A37 4F4B 4FCF 5372
55C9 57FD 5850 612C 634E 6A14 6CDD 7B71 7BBE 7E45
7FDB 8186 8258 86F8 8E08 9165 9704 9B48 9BB9 9C3A
C18D:1109-1169-11A8
4FD7 5C6C 675F 6D91 7C9F 7E8C 8B16 8D16 901F 6D2C
906C
C190:1109-1169-11AB
5B6B 5DFD 640D 84C0 905C 98E1 98E7 98F1
C194:1109-1169-11AF
7387 7AA3 87C0
C1A1:1109-1169-11BC
5B8B 609A 677E 6DDE 8A1F 8AA6 9001 980C 67D7 7AE6
9B06
C1C4:1109-116B
5237 7051 788E 9396 60E2 66EC 7463
C1E0:1109-116C
8870 91D7
C218:1109-116E
4FEE 53D7 55FD 56DA 5782 58FD 5AC2 5B88 5CAB 5CC0
5E25 6101 620D 624B 6388 641C 6536 6578 6A39 6B8A
6C34 6D19 6F31 71E7 72E9 7378 7407 74B2 7626 7761
79C0 7A57 7AEA 7CB9 7D8F 7DAC 7E61 7F9E 8129 8331
8490 84DA 85EA 8896 8AB0 8B90 8F38 9042 9083 916C
9296 92B9 968B 96A7 96A8 96D6 9700 9808 9996 9AD3
9B1A 53DF 552E 5ECB 666C 6BB3 6CC5 6EB2 6FC9 775F
7762 778D 795F 7C54 813A 8184 81B8 8B8E 8C4E 9672
98BC 9948
C219:1109-116E-11A8
53D4 587E 5919 5B70 5BBF 6DD1 6F5A 719F 7421 74B9
8085 83FD 4FF6 500F 5135 5A4C 6A5A 9A4C 9DEB
C21C:1109-116E-11AB
5DE1 5F87 5FAA 6042 65EC 6812 696F 6A53 6B89 6D35
6DF3 73E3 76FE 77AC 7B4D 7D14 8123 821C 8340 84F4
8563 8A62 8AC4 9187 931E 9806 99B4 4F9A 72E5 76F9
7734 7D03 80AB 99E8 9B0A 9D89
C220:1109-116E-11AF
620C 8853 8FF0 9265 7D49
C22D:1109-116E-11BC
5D07 5D27 5D69 83D8
C26C:1109-1171
5005 6DEC 7120
C2AC:1109-1173-11AF
745F 819D 8768 8671
C2B5:1109-1173-11B8
6FD5 62FE 7FD2 8936 8972 6174 71A0 96B0
C2B9:1109-1173-11BC
4E1E 4E58 50E7 52DD 5347 627F 6607 7E69 8805 965E
584D 9B19
C2DC:1109-1175
4F8D 5319 5636 59CB 5AA4 5C38 5C4E 5C4D 5E02 5F11
6043 65BD 662F 6642 67BE 67F4 731C 77E2 793A 7FC5
8494 84CD 8996 8A66 8A69 8AE1 8C55 8C7A 5072 5155
53AE 557B 5852 5EDD 67B2 67F9 6F8C 7DE6 7FE4 8ADF
8AF0 8C49 91C3 9349 984B
C2DD:1109-1175-11A8
57F4 5BD4 5F0F 606F 62ED 690D 6B96 6E5C 7184 7BD2
8755 8B58 8EFE 98DF 98FE 55B0 5AB3 683B
C2E0:1109-1175-11AB
4F38 4F81 4FE1 547B 5A20 5BB8 613C 65B0 6668 71FC
7533 795E 7D33 814E 81E3 8398 85AA 85CE 8703 8A0A
8EAB 8F9B 8FC5 54C2 567A 56DF 59FA 6C5B 77E7 8124
8D10 9823 99EA
C2E4:1109-1175-11AF
5931 5BA4 5BE6 6089 87CB 98CB
C2EC:1109-1175-11B7
5BE9 5C0B 5FC3 6C81 6DF1 700B 751A 82AF 8AF6 68A3
6F6F 71D6 845A 9414 9C4F
C2ED:1109-1175-11B8
4EC0 5341 8FBB
C30D:110A-1161-11BC
96D9
C528:110A-1175
6C0F
C544:110B-1161
4E9E 4FC4 5152 555E 5A25 5CE8 6211 7259 82BD 83AA
86FE 8859 8A1D 963F 96C5 9913 9D09 9D5D 4E2B 54E6
5A3F 5A40 5CE9 75B4 7811 7B0C 8FD3 930F 9D5E
C545:110B-1161-11A8
580A 5CB3 5DBD 5E44 60E1 6115 63E1 6A02 6E25 9102
9354 984E 9C10 9F77 5053 537E 54A2 5594 5669 816D
843C 89A8 8AE4 9D9A 9F76
C548:110B-1161-11AB
5B89 5CB8 6309 664F 6848 773C 96C1 978D 9854 9B9F
6849 72B4 8D0B 9D08
C54C:110B-1161-11AF
65A1 8B01 8ECB 95BC 560E 621E 63E0 7A75 8A10 904F
981E 9D36
C554:110B-1161-11B7
5535 5CA9 5DD6 5EB5 6697 764C 83F4 95C7 557D 5A95
5D53 667B 8164 844A 84ED 8AF3 9837 99A3 9EEF
C555:110B-1161-11B8
58D3 62BC 72CE 9D28
C559:110B-1161-11BC
4EF0 592E 600F 663B 6B83 79E7 9D26 536C 5771 6CF1
76CE 9785
C560:110B-1162
5393 54C0 57C3 5D16 611B 66D6 6DAF 788D 827E 9698
9744 50FE 5509 5540 566F 5A2D 5D15 6328 6371 6B38
6F04 7343 769A 775A 77B9 78D1 7919 8586 85F9 9749
9A03
C561:110B-1162-11A8
5384 627C 6396 6DB2 7E0A 814B 984D 545D 6239 6424
9628
C575:110B-1162-11BC
6AFB 7F4C 9DAF 9E1A 56B6 5AC8 7F43 9DEA
C57C:110B-1163
4E5F 503B 51B6 591C 60F9 63F6 6930 723A 8036 91CE
57DC
C57D:110B-1163-11A8
5F31 7D04 82E5 846F 84BB 85E5 8E8D 721A 79B4 7BDB
7C65 9470 9C2F 9DB8 9FA0
C591:110B-1163-11BC
4F6F 58E4 5B43 6059 63DA 6518 656D 6698 694A 6A23
6D0B 7001 716C 75D2 760D 79B3 7A70 7F8A 8944 8B93
91C0 967D 990A 5F89 6F3E 703C 70CA 7662 773B 8618
8F30 9472 98BA 9A64
C5B4:110B-1165
5704 5FA1 65BC 6F01 7600 79A6 8A9E 99AD 9B5A 9F6C
5709 6554 6DE4 98EB
C5B5:110B-1165-11A8
5104 61B6 6291 6A8D 81C6 7E76
C5B8:110B-1165-11AB
5043 5830 5F66 7109 8A00 8AFA 50BF 533D 5AE3 8B9E
9122 9F34 9F39
C5BC:110B-1165-11AF
5B7C 8616 81EC
C5C4:110B-1165-11B7
4FFA 513C 56B4 5944 63A9 6DF9 5D26 5E7F 66EE 7F68
9183 95B9
C5C5:110B-1165-11B8
5DAA 696D 5DAB 9134
C5D0:110B-1166
605A 66C0
C5D4:110B-1166-11AB
5186
C5EC:110B-1167
4E88 4F59 5982 6B5F 6C5D 74B5 7916 8207 8245 8339
8F3F 8F5D 9918 8201
C5ED:110B-1167-11A8
4EA6 57DF 5F79 6613 75AB 7E79 8B6F 9006 9A5B 5DA7
61CC 6DE2 95BE
C5F0:110B-1167-11AB
56A5 5827 59F8 5A1F 5BB4 5EF6 6350 633B 693D 6C87
6CBF 6D8E 6D93 6DF5 6F14 70DF 7136 7159 71C3 71D5
784F 786F 7B75 7DE3 7E2F 884D 8EDF 925B 9CF6 5157
56E6 57CF 5B3F 6081 63BE 66E3 6ADE 6E37 81D9 839A
8735 8815 8B8C 9DF0
C5F4:110B-1167-11AF
6085 6D85 71B1 95B1 564E
C5FC:110B-1167-11B7
53AD 67D3 708E 7130 7430 8276 82D2 95BB 9AE5 9E7D
5189 5869 61D5 624A 6ABF 6AB6 704E 7069 91C5 995C
9B58 9EF6
C5FD:110B-1167-11B8
66C4 71C1 8449 66C5 7180 7217 9768
C601:110B-1167-11BC
584B 5DB8 5F71 6620 668E 6979 69AE 6C38 6CF3 6E36
6F41 6FDA 701B 702F 7150 71DF 7370 745B 74D4 76C8
7A4E 7E93 82F1 8A60 8FCE 9348 9719 548F 5B34 5B30
6D67 6FF4 766D 78A4 7E08 8811 8D0F 90E2 97FA
C608:110B-1168
4E42 502A 5208 53E1 66F3 6C6D 6FCA 730A 777F 7A62
82AE 85DD 8602 88D4 8A63 8B7D 8C6B 92B3 9713 9810
56C8 5ADB 62FD 639C 6798 7369 7768 7796 7E44 7FF3
82C5 854A 854B 8589 868B 873A 9BE2 9DD6 9E91
C624:110B-1169
4E94 4F0D 4FC9 50B2 5348 543E 5433 55DA 5862 58BA
5967 5A1B 5BE4 609F 61CA 6556 65FF 6664 68A7 6C5A
6FB3 70CF 71AC 7352 7B7D 8708 8AA4 9C32 9F07 4EF5
4FE3 5514 55F7 5641 572C 5AAA 5AEF 5FE4 6160 6342
6C59 7AB9 8071 8323 8956 8B37 8FC3 8FD5 9068 93CA
93D6 96A9 9A41 9F2F
C625:110B-1169-11A8
5C4B 6C83 7344 7389 923A
C628:110B-1169-11AB
6EAB 7465 761F 7A69 7E15 860A 5ABC 614D 6637 6C33
7185 8580 8F40 919E 97DE 9942 9C2E
C62C:110B-1169-11AF
5140 55E2 8183
C639:110B-1169-11BC
58C5 64C1 74EE 7515 7670 7FC1 9095 96CD 9954 5581
5EF1 6EC3 7655 79BA 7F4B 84CA 96DD 9852
C640:110B-116A
6E26 74E6 7AA9 7AAA 81E5 86D9 8778 8A1B 54C7 56EE
5A50 6799 6D3C 7327 7A8A 8435 8B4C
C644:110B-116A-11AB
5A49 5B8C 5B9B 68A1 6900 6D63 73A9 7413 742C 7897
7DE9 7FEB 8118 8155 839E 8C4C 962E 9811 5213 57B8
59A7 5C8F 5FE8 60CB 6DB4 76CC
C648:110B-116A-11AF
66F0
C655:110B-116A-11BC
5F80 65FA 6789 6C6A 738B 5C2B 7007 8FEC
C65C:110B-116B
502D 5A03 6B6A 77EE 5AA7
C678:110B-116C
5916 5D6C 5DCD 7325 754F 504E 5D34 5D54 6E28 7168
78A8 78C8 8075 9697
C694:110B-116D
50E5 51F9 582F 592D 5996 59DA 5BE5 5DA2 62D7 6416
6493 64FE 66DC 6A48 71FF 7464 7A88 7AAF 7E47 7E5E
8000 8170 87EF 8981 8B20 9059 9080 9952 5060 5593
5773 589D 5B08 5E7A 5FAD 5FBC 6B80 6F86 7945 7A7E
7A85 8558 9076 9DC2
C695:110B-116D-11A8
617E 6B32 6D74 7E1F 8925 8FB1 6EBD 84D0
C6A9:110B-116D-11BC
4FD1 50AD 5197 52C7 57C7 5889 5BB9 5EB8 6142 6995
6D8C 6E67 6EB6 7194 7462 7528 752C 8073 8338 84C9
8E0A 9394 93DE 509B 5B82 5D71 6175 6183 69E6 7867
8202 86F9 8E34
C6B0:110B-116E
4E8E 4F51 5076 512A 53C8 53CB 53F3 5B87 5BD3 5C24
611A 6182 65F4 725B 7397 7440 76C2 7950 7991 79B9
7D06 7FBD 828B 85D5 865E 8FC2 9047 90F5 91EA 9685
96E8 96E9 4E8F 4EB4 4FC1 504A 5401 5823 5D4E 5EBD
6745 75A3 76F1 7AFD 8026 8030 8B23 8E3D 935D 9E80
9E8C 9F72
C6B1:110B-116E-11A8
52D6 5F67 65ED 6631 682F 715C 7A36 90C1 980A 71E0
C6B4:110B-116E-11AB
4E91 6A52 6B9E 6F90 7189 8018 82B8 8553 904B 9695
96F2 97FB 60F2 6C84 7BD4 7D1C 9723 97F5
C6B8:110B-116E-11AF
851A 9B31 4E90
C6C5:110B-116E-11BC
718A 96C4
C6D0:110B-116F-11AB
5143 539F 54E1 5713 5712 57A3 5A9B 5AC4 5BC3 6028
613F 63F4 6C85 6D39 6E72 6E90 7230 733F 7457 82D1
8881 8F45 9060 9662 9858 9D1B 51A4 571C 676C 6965
7328 7DA9 82AB 8597 873F 8B1C 92FA 9A35 9D77 9EFF
C6D4:110B-116F-11AF
6708 8D8A 925E 5216 7CA4
C704:110B-1171
4F4D 5049 50DE 5371 570D 59D4 5A01 5C09 6170 6690
6E2D 7232 744B 7DEF 80C3 840E 8466 853F 875F 885B
8918 8B02 9055 97CB 9B4F 559F 5E43 7152 71A8 75FF
8473 885E 8AC9 9036 95C8 97D9 97E1 9927 9AAA
C720:110B-1172
4E73 4F91 5112 516A 552F 55A9 5B7A 5BA5 5E7C 5E7D
5EBE 60A0 60DF 6108 6109 63C4 6538 6709 67D4 67DA
6961 6962 6CB9 6D27 6E38 6FE1 7336 7337 745C 7531
7652 7DAD 81FE 8438 88D5 8A98 8ADB 8AED 8E30 8E42
904A 903E 907A 9149 91C9 936E 5198 5466 56FF 58DD
5E37 63C9 65BF 6CD1 7256 7609 7610 7AAC 7AB3 7C72
7CC5 7DCC 8174 83A0 8555 8564 86B0 86B4 8764 8915
8B89 900C 97A3 9BAA 9EDD 9F2C 9FA5
C721:110B-1172-11A8
5809 6BD3 8089 80B2 5125
C724:110B-1172-11AB
5141 596B 5C39 6F64 73A7 80E4 8D07 9217 958F 6600
92C6
C728:110B-1172-11AF
807F 6F4F 77DE
C735:110B-1172-11BC
620E 701C 7D68 878D 72E8
C740:110B-1173-11AB
57A0 6069 6147 6BB7 8ABE 9280 96B1 542C 569A 5701
57BD 6196 6ABC 6EB5 72FA 73E2 766E 8A14 911E 9F57
C744:110B-1173-11AF
4E59 9CE6
C74C:110B-1173-11B7
541F 6DEB 852D 9670 97F3 98EE 5591 5D1F 5ED5 6114
972A
C74D:110B-1173-11B8
63D6 6CE3 9091 6092 6339 6D65
C751:110B-1173-11BC
51DD 61C9 81BA 9DF9
C758:110B-1174
4F9D 501A 5100 5B9C 610F 61FF 64EC 6905 6BC5 7591
77E3 7FA9 8264 858F 87FB 8863 8ABC 8B70 91AB 5117
51D2 5293 5DB7 6B39 6F2A 7317 7912 8798 9950
C774:110B-1175
4E8C 4EE5 4F0A 5937 59E8 5DF2 5F1B 5F5B 6021 723E
73E5 7570 75CD 79FB 800C 8033 8084 82E1 8351 8CBD
8CB3 9087 98F4 990C 54BF 5768 5C14 5F5D 682E 6D1F
73C6 8A11 8A51 8FE4 96B6
C775:110B-1175-11A8
7037 76CA 7FCA 7FCC 7FFC 8B1A 5F0B 71A4 9DC1
C778:110B-1175-11AB
4EBA 4EC1 5203 5370 54BD 56E0 59FB 5BC5 5F15 5FCD
6E6E 7D6A 8335 8693 8A8D 976D 9777 4EDE 5819 5924
5A63 6268 6C24 6D07 798B 7C7E 82A2 88C0
C77C:110B-1175-11AF
4E00 4F5A 4F7E 58F9 65E5 6EA2 9038 93B0 99B9 6CC6
8EFC
C784:110B-1175-11B7
4EFB 58EC 598A 59D9 6041 7A14 834F 8CC3 7D4D 887D
928B 98EA
C785:110B-1175-11B8
5165 5344 5EFF
C789:110B-1175-11BC
4ECD 5269 5B55 82BF 5AB5
C790:110C-1161
4ED4 523A 54A8 59C9 59FF 5B50 5B57 5B5C 6063 6148
6ECB 7099 716E 7386 74F7 75B5 78C1 7D2B 8005 81EA
8328 8517 85C9 8AEE 8CC7 96CC 5470 5B28 5B56 5B76
67D8 6CDA 7278 7725 7726 7CA2 8014 80FE 8308 8332
83BF 8678 89DC 8A3E 8CB2 8D6D 93A1 983F 9AED 9B93
9DBF 9DD3
C791:110C-1161-11A8
4F5C 52FA 56BC 65AB 6628 707C 70B8 7235 7DBD 828D
914C 96C0 9D72 5C9D 600D 65B1 67DE 6C4B 712F 72B3
788F
C794:110C-1161-11AB
5B71 68E7 6B98 6F7A 76DE 5257 6214 9A4F
C7A0:110C-1161-11B7
5C91 66AB 6F5B 7BB4 7C2A 8836 6D94 6F5C 6FF3
C7A1:110C-1161-11B8
96DC 5361 56C3 7728 78FC 894D
C7A5:110C-1161-11BC
4E08 4ED7 5320 5834 58BB 58EF 596C 5C07 5E33 5E84
5F35 638C 66B2 6756 6A1F 6AA3 6B0C 6F3F 7246 7350
748B 7AE0 7CA7 8178 81DF 81E7 838A 846C 8523 8594
85CF 88DD 8D13 91AC 9577 969C 50BD 5958 599D 5B19
5D82 5EE7 6215 6F33 7242 7634 7CDA 7F98 8407 88C5
8CEC 9123 93D8 9926 9E9E
C7AC:110C-1162
518D 54C9 5728 5BB0 624D 6750 683D 6893 6E3D 6ED3
707D 7E21 88C1 8CA1 8F09 9F4B 9F4E 5908 5D3D 6257
699F 707E 7E94
C7C1:110C-1162-11BC
722D 7B8F 8ACD 931A 5D22 7319 7424 9397
C800:110C-1165
4F47 4F4E 5132 5480 59D0 5E95 62B5 6775 696E 6A17
6CAE 6E1A 72D9 732A 75BD 7BB8 7D35 82E7 83F9 8457
85F7 8A5B 8CAF 8E87 9019 90B8 96CE 9F5F 5B81 5CA8
677C 67E2 6C10 6F74 7026 7274 7F5D 7F9D 82F4 86C6
889B 891A 89DD 8A46 8C6C 967C
C801:110C-1165-11A8
52E3 540A 5AE1 5BC2 6458 6575 6EF4 72C4 7684 7A4D
7B1B 7C4D 7E3E 7FDF 837B 8B2B 8CCA 8D64 8DE1 8E5F
8FEA 8FF9 9069 93D1 6A00 78E7 7CF4 83C2 89BF 9016
99B0
C804:110C-1165-11AB
4F43 4F7A 50B3 5168 5178 524D 526A 5861 587C 5960
5C08 5C55 5EDB 609B 6230 6813 6BBF 6C08 6FB1 714E
7420 7530 7538 7551 7672 7B4C 7B8B 7BAD 7BC6 7E8F
8A6E 8F3E 8F49 923F 9293 9322 942B 96FB 985A 986B
991E 542E 56C0 5AE5 5C47 5DD3 6229 63C3 65C3 6834
69C7 6E54 6FB6 724B 7471 750E 754B 7560 75CA 765C
78DA 7C5B 7FB6 7FE6 8146 819E 8343 8E94 8F07 9085
913D 92D1 932A 975B 9766 9853 98E6 9930 9B0B 9C63
9E07
C808:110C-1165-11AF
5207 622A 6298 6D59 7664 7ACA 7BC0 7D76 5C8A 6662
7A83
C810:110C-1165-11B7
5360 5CBE 5E97 6F38 70B9 7C98 9711 9B8E 9EDE 4F54
588A 73B7 7B18 7C1F 82EB 852A 86C5 8998 98AD 9ECF
C811:110C-1165-11B8
63A5 647A 8776 6904 696A 8728 8DD5 8E40 9C08
C815:110C-1165-11BC
4E01 4E95 4EAD 505C 5075 5448 59C3 5B9A 5E40 5EAD
5EF7 5F81 60C5 633A 653F 6574 65CC 6676 6678 67FE
6968 6A89 6B63 6C40 6DC0 6DE8 6E1F 6E5E 701E 70A1
738E 73FD 753A 775B 7887 798E 7A0B 7A7D 7CBE 7D8E
8247 8A02 8AEA 8C9E 912D 914A 91D8 9266 92CC 9320
9706 9756 975C 9802 9F0E 4F42 53EE 5A67 5A77 6014
639F 686F 6883 68D6 706F 73F5 7594 7B73 839B 8A3C
9049 9172 92E5 975A
C81C:110C-1166
5236 5291 557C 5824 5E1D 5F1F 608C 63D0 68AF 6FDF
796D 7B2C 81CD 85BA 88FD 8AF8 8E44 918D 9664 969B
973D 984C 9F4A 5115 5A23 64E0 7318 7445 7747 7994
7A0A 7DF9 8E36 8E4F 8E8B 9357 9684 97F2 9BA7 9BF7
C870:110C-1169
4FCE 5146 51CB 52A9 5632 5F14 5F6B 63AA 64CD 65E9
6641 66FA 66F9 671D 689D 68D7 69FD 6F15 6F6E 7167
71E5 722A 74AA 773A 7956 795A 79DF 7A20 7A95 7C97
7CDF 7D44 7E70 8087 85FB 86A4 8A54 8ABF 8D99 8E81
9020 906D 91E3 963B 96D5 9CE5 4F7B 50AE 5201 539D
5608 566A 5B25 5F82 61C6 627E 6B82 6FA1 7431 7681
7967 7AC8 7B0A 7CD9 7CF6 7D69 7D5B 80D9 81CA 825A
8526 8729 8A82 8B5F 921F 929A 92FD 9BDB 9D70 9F02
C871:110C-1169-11A8
65CF 7C07 8DB3 93C3 762F
C874:110C-1169-11AB
5B58 5C0A 62F5
C878:110C-1169-11AF
5352 62D9 731D
C885:110C-1169-11BC
5027 5B97 5F9E 60B0 616B 68D5 6DD9 742E 7A2E 7D42
7D9C 7E31 816B 8E2A 8E35 937E 9418 4F00 6152 67CA
6936 6A05 747D 7607 7CBD 87BD 8E64
C88C:110C-116A
4F50 5750 5DE6 5EA7 632B 5249 75E4 839D 9AFD
C8C4:110C-116C
7F6A
C8FC:110C-116E
4E3B 4F4F 4F8F 505A 59DD 80C4 546A 5468 55FE 594F
5B99 5DDE 5EDA 665D 6731 67F1 682A 6CE8 6D32 6E4A
6F8D 70B7 73E0 7587 7C4C 7D02 7D2C 7DA2 821F 86DB
8A3B 8A85 8D70 8E8A 8F33 9031 914E 9152 9444 99D0
4E1F 4F9C 5114 5C0C 5E6C 62C4 7843 7C52 8098 8160
851F 86C0 88EF 8A4B 8CD9 8D8E 8F08 9052 9252 970C
9714 9F04
C8FD:110C-116E-11A8
7AF9 7CA5
C900:110C-116E-11AB
4FCA 5101 51C6 57C8 5BEF 5CFB 6659 6A3D 6D5A 6E96
6FEC 710C 756F 7AE3 8822 9021 9075 96CB 99FF 5642
57FB 58AB 60F7 6499 76B4 7DA7 7F47 8E06 8E72 940F
96BC 9915 9C52 9D54
C904:110C-116E-11AF
8301 4E7C
C911:110C-116E-11BC
4E2D 4EF2 8846 91CD 773E
C989:110C-1173-11A8
537D 5373 559E
C990:110C-1173-11AF
6ADB 9A2D
C999:110C-1173-11B8
696B 6C41 847A 6A9D 857A
C99D:110C-1173-11BC
589E 618E 66FE 62EF 70DD 7511 75C7 7E52 84B8 8B49
8D08 5D92 77F0 7F7E
C9C0:110C-1175
4E4B 53EA 54AB 5730 5740 5FD7 6301 6307 646F 652F
65E8 667A 679D 67B3 6B62 6C60 6C9A 6F2C 77E5 7825
7949 7957 7D19 80A2 8102 81F3 829D 82B7 8718 8A8C
8D04 8DBE 9072 577B 5880 627A 69B0 6CDC 75E3 79EA
7BEA 8210 8E1F 8E93 8EF9 962F 9BA8 9DD9
C9C1:110C-1175-11A8
76F4 7A19 7A37 7E54 8077 799D
C9C4:110C-1175-11AB
5507 55D4 5875 632F 6422 6649 664B 686D 699B 6B84
6D25 6EB1 73CD 7468 74A1 755B 75B9 76E1 771E 778B
79E6 7E09 7E1D 81FB 852F 8897 8A3A 8CD1 8EEB 8FB0
9032 93AD 9663 9673 9707 4FB2 5118 73D2 7A39 84C1
87B4 8D81 9241 9B12
C9C8:110C-1175-11AF
4F84 53F1 59EA 5AC9 5E19 684E 74C6 75BE 79E9 7A92
81A3 86ED 8CEA 8DCC 8FED 57A4 7D70 84BA 90C5 9455
C9D0:110C-1175-11B7
659F 6715 9D06
C9D1:110C-1175-11B8
57F7 6F57 7DDD 8F2F 93F6 96C6 54A0 6222
C9D5:110C-1175-11BC
5FB5 61F2 6F84 6F82 7013 7665 77AA
CC28:110E-1161
4E14 4F98 501F 53C9 55DF 5D6F 5DEE 6B21 6B64 78CB
7B9A 8E49 8ECA 906E 4F7D 5056 5953 5C94 5FA3 69CE
7473 7868
CC29:110E-1161-11A8
6349 643E 7740 7A84 932F 947F 9F6A 6233 64C9 65B2
CC2C:110E-1161-11AB
64B0 6FAF 71E6 74A8 74DA 7AC4 7C12 7E82 7CB2 7E98
8B9A 8D0A 947D 9910 994C 5127 5139 5297 5DD1 6522
6B11 7228 8DB2
CC30:110E-1161-11AF
5239 5BDF 64E6 672D 7D2E 624E 62F6
CC38:110E-1161-11B7
50ED 53C3 5879 6158 6159 61FA 65AC 7AD9 8B92 8B96
5133 53C5 5D84 5DC9 615A 61AF 6519 69E7 6B03 6BDA
8B56 93E8 9471 995E 9A42 9EF2
CC3D:110E-1161-11BC
5009 5021 5275 5531 5A3C 5EE0 5F70 6134 655E 660C
6636 66A2 69CD 6EC4 6F32 7316 7621 7A93 8139 8259
83D6 84BC 5000 5096 51D4 5231 60B5 60DD 6227 6436
6919 6C05 7472 7A97 7ABB 8E4C 92F9 9306 95B6 9B2F
9DAC
CC44:110E-1162
50B5 57F0 5BC0 5BE8 5F69 63A1 7826 7DB5 83DC 8521
91C7 91F5 68CC 831D
CC45:110E-1162-11A8
518A 67F5 7B56 8CAC 5616 5E58 78D4 7B27 7C00 86B1
CC98:110E-1165
51C4 59BB 60BD 8655 6DD2 840B 8904 89B7 90EA
CC99:110E-1165-11A8
501C 5254 5C3A 617D 621A 62D3 64F2 65A5 6ECC 7620
810A 8E60 965F 96BB 544E 5767 5849 60D5 6357 646D
8734 8DD6 8E91
CC9C:110E-1165-11AB
4EDF 5343 5598 5929 5DDD 64C5 6CC9 6DFA 7394 7A7F
821B 85A6 8CE4 8E10 9077 91E7 95E1 9621 97C6 4FF4
5029 50E2 5103 6D0A 6FFA 74E9 7946 7C81 81F6 828A
831C 8350 84A8 8546 8695 8FBF 975D
CCA0:110E-1165-11AF
51F8 54F2 5586 5FB9 64A4 6F88 7DB4 8F1F 8F4D 9435
525F 555C 57D1 60D9 6387 6B60 9295 9323 98FB 992E
CCA8:110E-1165-11B7
50C9 5C16 6CBE 6DFB 751B 77BB 7C3D 7C64 8A79 8AC2
5E68 5FDD 60C9 6A90 6AFC 7038 7C37 895C
CCA9:110E-1165-11B8
581E 59BE 5E16 6377 7252 758A 776B 8ADC 8CBC 8F12
5022 546B 558B 6017 893A
CCAD:110E-1165-11BC
5EF3 6674 6DF8 807D 83C1 8ACB 9751 9BD6 51CA 570A
873B 9D84
CCB4:110E-1166
5243 66FF 6D95 6EEF 7DE0 8AE6 902E 905E 9AD4 568F
5F58 68E3 6BA2 780C 8482 8515 855E 8EC6 9746 9AF0
CD08:110E-1169
521D 527F 54E8 6194 6284 62DB 68A2 6912 695A 6A35
7092 7126 785D 7901 790E 79D2 7A0D 8096 8278 82D5
8349 8549 8C82 8D85 9162 918B 91AE 5062 50EC 52AD
52E6 564D 5AF6 5CA7 5CED 5D95 600A 6084 6100 676A
71CB 7D83 8016 8A9A 8B59 8DA0 8EFA 8FE2 9214 936B
936C 9798 9866 9AEB 9DE6 9F60
CD09:110E-1169-11A8
4FC3 56D1 71ED 77D7 8700 89F8 66EF 7225 77DA 85A5
8E85 9AD1
CD0C:110E-1169-11AB
5BF8 5FD6 6751 90A8 540B
CD1D:110E-1169-11BC
53E2 585A 5BF5 60A4 6181 6460 7E3D 8070 8525 9283
8471 84EF 93E6 9A18 9A44
CD2C:110E-116A-11AF
64AE
CD5C:110E-116C
50AC 5D14 6700 562C 6467 69B1 6F3C 7480 78EA 7E17
8127
CD94:110E-116E
589C 62BD 63A8 690E 6978 6A1E 6E6B 76BA 79CB 82BB
8429 8ACF 8DA8 8FFD 9112 914B 919C 9310 9318 939A
96DB 9A36 9C0D 50E6 557E 5A35 5E1A 60C6 6376 63EB
6425 7503 7633 9F9D 7B92 7BA0 7C09 7E0B 7E10 84AD
966C 96B9 97A6 9A05 9B4B 9C0C 9D7B 9D96 9DB5 9E84
9EA4
CD95:110E-116E-11A8
4E11 755C 795D 7AFA 7B51 7BC9 7E2E 84C4 8E59 8E74
8EF8 9010 59AF 8233 8C56 8E5C 9F00
CD98:110E-116E-11AB
6625 693F 7443
CD9C:110E-116E-11AF
51FA 672E 9EDC 79EB
CDA9:110E-116E-11BC
5145 5FE0 6C96 87F2 885D 8877 51B2 5FE1 73EB
CDCC:110E-1170
60B4 81B5 8403 8D05 60F4 63E3 75A9 7601 9847
CDE8:110E-1171
53D6 5439 5634 5A36 5C31 708A 7FE0 805A 8106 81ED
8DA3 9189 9A5F 9DF2 51A3 6A47 6BF3
CE21:110E-1173-11A8
5074 4EC4 53A0 60FB 6E2C 5EC1 6603
CE35:110E-1173-11BC
5C64
CE58:110E-1175
4F88 5024 55E4 5CD9 5E5F 6065 6894 6CBB 6DC4 71BE
75D4 75F4 7661 7A1A 7A49 7DC7 7DFB 7F6E 81F4 86A9
8F1C 96C9 99B3 9F52 536E 54C6 5BD8 5DF5 7564 75D3
7D7A 83D1 8599 892B 8C78 8DF1 9319 9624 9BD4 9D19
9D1F 9D44
CE59:110E-1175-11A8
5247 52C5 98ED 6555
CE5C:110E-1175-11AB
89AA 6AEC 85FD 896F 9F54
CE60:110E-1175-11AF
4E03 67D2 6F06
CE68:110E-1175-11B7
4FB5 5BE2 6795 6C88 6D78 741B 7827 91DD 937C 5BD6
5FF1 6939 6C89 90F4 92DF 99F8
CE69:110E-1175-11B8
87C4
CE6D:110E-1175-11BC
79E4 7A31
CF8C:110F-116B
5FEB 5672 592C
D0C0:1110-1161
4ED6 54A4 553E 58AE 59A5 60F0 6253 62D6 6736 6955
8235 9640 99B1 99DD 4F57 579E 62D5 67C1 6A62 6CB1
8A6B 8DCE 8EB1 99DE 9B80 9D15 9F09
D0C1:1110-1161-11A8
502C 5353 5544 577C 6258 64E2 666B 67DD 6FC1 6FEF
7422 7438 8A17 9438 62C6 6A50 6CB0 6DBF 77FA 7C5C
8600 8E14 9034
D0C4:1110-1161-11AB
5451 5606 5766 5F48 619A 6B4E 7058 70AD 7DBB 8A95
61BB 6524 6BAB 7671 9A52
D0C8:1110-1161-11AF
596A 812B 4FBB
D0D0:1110-1161-11B7
63A2 7708 803D 8CAA 55FF 5FD0 9156
D0D1:1110-1161-11B8
5854 642D 69BB 509D 584C 6428
D0D5:1110-1161-11BC
5B95 5E11 6E6F 8569 71D9 76EA 78AD 862F
D0DC:1110-1162
514C 53F0 592A 6020 614B 6B86 6C70 6CF0 7B1E 80CE
82D4 8DC6 90B0 98B1 57ED 5A27 5B61 62AC 8FE8 99C4
99D8
D0DD:1110-1162-11A8
64C7 6FA4
D0F1:1110-1162-11BC
6491 6490 725A
D130:1110-1165
6504
D1A0:1110-1169
514E 5410 571F 8A0E
D1A4:1110-1169-11AB
564B 5678 74F2
D1B5:1110-1169-11BC
615F 6876 75DB 7B52 7D71 901A 606B 6A0B 7B69
D1F4:1110-116C
5806 69CC 817F 892A 9000 9839 96A4
D22C:1110-116E
5078 5957 59AC 6295 900F 9B2A 5992 6E1D 9AB0
D241:1110-116E-11BC
4F5F
D2B9:1110-1173-11A8
615D 7279 5FD2
D2C8:1110-1173-11B7
95D6
D30C:1111-1161
5761 5A46 5DF4 628A 64AD 64FA 6777 6CE2 6D3E 722C
7436 7834 7F77 82AD 8DDB 9817 53F5 5991 5CA5 6015
705E 7238 73BB 76A4 7B06 7C38 8019 83E0 8469 9131
D310:1111-1161-11AB
5224 5742 677F 7248 74E3 8CA9 8FA6 9211 962A 6C74
D314:1111-1161-11AF
516B 53ED 634C 6733 6C43
D328:1111-1162
4F69 5504 6096 6557 6C9B 6D7F 724C 72FD 7A17 8987
8C9D 5B5B 65C6 73EE 9708 9738
D33D:1111-1162-11BC
5F6D 6F8E 70F9 81A8 7830 794A 87DA 87DB
D345:1111-1163-11A8
610E
D3B8:1111-1167-11AB
4FBF 504F 6241 7247 7BC7 7DE8 7FE9 904D 97AD 9A19
533E 5FA7 60FC 7DF6 8251 8439 8759 890A 8ADE
D3C4:1111-1167-11B7
8CB6 782D 7A86
D3C9:1111-1167-11BC
576A 5E73 67B0 840D 8A55 6026 62A8 6CD9 82F9 84F1
9B83
D3D0:1111-1168
5420 5B16 5E63 5EE2 5F0A 6583 80BA 853D 9589 965B
655D 72F4 7358 7648
D3EC:1111-1169
4F48 5305 530D 530F 5486 54FA 5703 5E03 6016 629B
62B1 6355 6CE1 6D66 75B1 7832 80DE 812F 82DE 8461
84B2 888D 8912 900B 92EA 98FD 9B91 5124 5E96 6661
66D3 70AE 70B0 8216 8AA7 924B 9784 9914 9BC6
D3ED:1111-1169-11A8
5E45 66B4 66DD 7011 7206
D45C:1111-116D
4FF5 527D 5F6A 6153 6753 6A19 6F02 74E2 7968 8868
8C79 98C7 98C4 9A43 50C4 52E1 560C 5AD6 647D 6B8D
719B 7E39 88F1 93E2 9463 9ADF 9C3E
D488:1111-116E-11B7
54C1 7A1F 7980
D48D:1111-116E-11BC
6953 8AF7 8C4A 98A8 99AE 760B 8451
D53C:1111-1175
5F7C 62AB 75B2 76AE 88AB 907F 9642 8A56 8F9F 9781
9AF2
D53D:1111-1175-11A8
8177
D544:1111-1175-11AF
5339 5F3C 5FC5 6CCC 73CC 7562 758B 7B46 82FE 999D
4F56 5487 6EED 7BF3 7F7C 84FD 89F1 8E55 97B8 97E0
99DC 9D6F
D54D:1111-1175-11B8
4E4F 903C 506A
D558:1112-1161
4E0B 4F55 53A6 590F 5EC8 6630 6CB3 7455 8377 8766
8CC0 9050 971E 9C15 5440 5687 5C88 61D7 7146 7615
7F45 935C
D559:1112-1161-11A8
58D1 5B78 8650 8B14 9DB4 72E2 7627 76AC 786E 90DD
9DFD
D55C:1112-1161-11AB
5BD2 6068 608D 65F1 6C57 6F22 6FA3 701A 7F55 7FF0
9591 9592 9650 97D3 50E9 5AFA 5AFB 634D 66B5 9588
99FB 9DF3 9F3E
D560:1112-1161-11AF
5272 8F44 778E
D568:1112-1161-11B7
51FD 542B 54B8 5563 558A 6ABB 6DB5 7DD8 8266 929C
9677 9E79 839F 83E1 8AF4 8F5E 95DE
D569:1112-1161-11B8
5408 54C8 76D2 86E4 95A4 95D4 965C 530C 55D1 67D9
69BC 6E98 76CD 90C3
D56D:1112-1161-11BC
4EA2 4F09 59EE 5AE6 5DF7 6052 6297 676D 6841 6C86
6E2F 7F38 809B 822A 9805 592F 6046 7095 7F3F 980F
D574:1112-1162
4EA5 5055 54B3 5793 595A 5B69 5BB3 61C8 6977 6D77
7023 87F9 89E3 8A72 8AE7 9082 99ED 9AB8 548D 5DB0
5EE8 6B2C 736C 744E 75CE 85A4 91A2 9826 9BAD
D575:1112-1162-11A8
52BE 6838 7FEE 8988
D589:1112-1162-11BC
5016 5E78 674F 8347 884C 60BB
D5A5:1112-1163-11BC
4EAB 5411 56AE 73E6 9115 97FF 9909 9957 9999 858C
D5C8:1112-1165
5653 589F 865B 8A31 6B54
D5CC:1112-1165-11AB
61B2 6AF6 737B 8ED2 5DDA 5E70 6507
D5D0:1112-1165-11AF
6B47
D5D8:1112-1165-11B7
96AA 9A57 5DAE 736B 7381
D601:1112-1167-11A8
5955 7200 8D6B 9769 5F08 6D2B 7131 9B29
D604:1112-1167-11AB
4FD4 5CF4 5F26 61F8 665B 6CEB 70AB 7384 73B9 73FE
7729 774D 7D43 7D62 7E23 8237 8852 8CE2 9249 986F
5107 5B1B 6621 7404 75C3 770C 7E6F 7FFE 8706 8AA2
92D7 99FD
D608:1112-1167-11AF
5B51 7A74 8840 9801 7D5C 8D90
D610:1112-1167-11B7
5ACC
D611:1112-1167-11B8
4FE0 5354 593E 5CFD 633E 6D79 72F9 8105 8107 83A2
92CF 9830 5327 53F6 57C9 604A 608F 611C 7BCB
D615:1112-1167-11BC
4EA8 5144 5211 578B 5F62 6CC2 6ECE 7005 7050 70AF
7192 73E9 7469 834A 87A2 8861 9008 90A2 93A3 99A8
5910 5A19 8A57 8FE5 9658
D61C:1112-1168
516E 5F57 60E0 6167 66B3 8559 8E4A 91AF 978B 5092
5612 5BED 5FAF 69E5 76FB 8B11 8B7F
D638:1112-1169
4E4E 4E92 547C 58D5 58FA 597D 5CB5 5F27 6236 6248
660A 6667 6BEB 6D69 6DCF 6E56 6EF8 6F94 6FE0 6FE9
705D 72D0 7425 745A 74E0 7693 795C 7CCA 7E1E 80E1
82A6 846B 84BF 864E 865F 8774 8B77 8C6A 93AC 9800
9865 512B 51B1 5637 5AED 5AEE 6019 6C8D 6EC8 6EEC
7292 7322 769C 769E 7B8E 8055 9190 992C 9B0D
D639:1112-1169-11A8
60D1 6216 9177
D63C:1112-1169-11AB
5A5A 660F 6DF7 6E3E 743F 9B42 5702 60DB 6EB7 711C
95BD
D640:1112-1169-11AF
5FFD 60DA 7B0F 56EB
D64D:1112-1169-11BC
54C4 5F18 6C5E 6CD3 6D2A 70D8 7D05 8679 8A0C 9D3B
664E 6F92 7BCA 9277 9B28
D654:1112-116A
5316 548C 5B05 6A3A 706B 7575 798D 79BE 82B1 83EF
8A71 8B41 8CA8 9774 4FF0 5629 5A72 64ED 756B 9A4A
9FA2
D655:1112-116A-11A8
64F4 652B 78BA 78BB 7A6B 77CD 77E1 792D 944A
D658:1112-116A-11AB
4E38 559A 5950 5BA6 5E7B 60A3 63DB 6B61 6665 6853
6E19 7165 74B0 7D08 9084 9A69 9C25 5BF0 61FD 64D0
74DB 7696 7746 7D59 8C62 8F58 9370 9436 9B1F
D65C:1112-116A-11AF
6D3B 6ED1 733E 8C41 95CA 86DE
D669:1112-116A-11BC
51F0 5E4C 5FA8 604D 60F6 6130 614C 6643 6644 69A5
6CC1 6E5F 6EC9 6F62 714C 749C 7687 7BC1 7C27 8352
8757 9051 968D 9EC3 55A4 5A93 6033 745D 8093 8CBA
93A4
D68C:1112-116C
532F 56DE 5EFB 5F8A 6062 6094 61F7 6666 6703 6A9C
6DEE 6FAE 7070 736A 7E6A 81BE 8334 86D4 8AA8 8CC4
4F6A 6803 6D04 6ED9 76D4 8A7C 8FF4 982E 9C60
D68D:1112-116C-11A8
5283 7372 5684
D6A1:1112-116C-11BC
5B96 6A6B 9404 6F8B 921C 9ECC
D6A8:1112-116D
54EE 5686 5B5D 6548 6585 66C9 689F 6D8D 6DC6 723B
80B4 9175 9A4D 509A 56C2 5D24 6BBD 7187 769B 8653
991A
D6C4:1112-116E
4FAF 5019 539A 540E 543C 5589 55C5 5E3F 5F8C 673D
7166 73DD 9005 543D 55A3 5795 5820 6DB8 7334 7BCC
8A61 8B43 9157 9931
D6C8:1112-116E-11AB
52DB 52F3 5864 58CE 7104 718F 71FB 85B0 8A13 6688
66DB 720B 736F 7E81 8477 9442
D6CC:1112-116E-11AF
6B3B
D6D9:1112-116E-11BC
85A8
D6E4:1112-116F-11AB
55A7 6684 714A 8431 70DC 8AE0 8AFC
D6FC:1112-1170
5349 5599 6BC1 71EC 8294 866B 867A
D718:1112-1171
5F59 5FBD 63EE 6689 7147 8AF1 8F1D 9EBE 649D 7FEC
D734:1112-1172
4F11 643A 70CB 7566 8667 54BB 64D5 96B3 9AF9 9D42
D73C:1112-1172-11AF
6064 8B4E 9DF8 5379
D749:1112-1172-11BC
5147 51F6 5308 6D36 80F8 605F 80F7
D751:1112-1173-11A8
9ED1
D754:1112-1173-11AB
6615 6B23 7098 75D5 5F88 5FFB 6380 712E 8A22 91C1
D758:1112-1173-11AF
5403 5C79 7D07 8A16 4EE1 6C54 7599 8FC4 9F55
D760:1112-1173-11B7
6B20 6B3D 6B46 5EDE
D761:1112-1173-11B8
5438 6070 6D3D 7FD5 564F 6B59 6F5D 7FD6
D765:1112-1173-11BC
8208
D76C:1112-1174
50D6 51DE 559C 566B 56CD 59EC 5B09 5E0C 6199 6198
6231 665E 66E6 7199 71B9 71BA 72A7 79A7 7A00 7FB2
54A5 550F 563B 6095 6232 66BF 6B37 71F9 7214 8C68
993C
D790:1112-1175-11AF
8A70 72B5 7E88 896D 9821 9EE0
__END__

=head1 NAME

Unicode::Collate::CJK::Korean - weighting CJK Unified Ideographs
for Unicode::Collate

=head1 SYNOPSIS

    use Unicode::Collate;
    use Unicode::Collate::CJK::Korean;

    my $collator = Unicode::Collate->new(
        overrideCJK => \&Unicode::Collate::CJK::Korean::weightKorean
    );

=head1 DESCRIPTION

C<Unicode::Collate::CJK::Korean> provides C<weightKorean()>,
that is adequate for C<overrideCJK> of C<Unicode::Collate>
and makes tailoring of CJK Unified Ideographs in the order
of CLDR's Korean ordering.

=head1 SEE ALSO

=over 4

=item CLDR - Unicode Common Locale Data Repository

L<http://cldr.unicode.org/>

=item Unicode Locale Data Markup Language (LDML) - UTS #35

L<http://www.unicode.org/reports/tr35/>

=item L<Unicode::Collate>

=item L<Unicode::Collate::Locale>

=back

=cut
