/***********************************************************************/
/* COMMSET2.C - SET commands O-Z                                       */
/***********************************************************************/
/*
 * THE - The Hessling Editor. A text editor similar to VM/CMS xedit.
 * Copyright (C) 1991-1993 Mark Hessling
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to me.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                     email: M.Hessling@gu.edu.au
 * 36 David Road                     Phone: +61 7 849 7731
 * Holland Park                      Fax:   +61 7 875 5314
 * QLD 4121
 * Australia
 */

/*
$Header: C:\THE\RCS\commset2.c 1.4 1993/09/01 16:25:53 MH Interim MH $
*/

#include <stdio.h>

#include "the.h"
#include "proto.h"

/*#define DEBUG 1*/

/*--------------------------- global data -----------------------------*/
bool rexx_output=FALSE;

/*-------------------------- external data ----------------------------*/
extern char in_profile;    /* indicates if processing profile */
extern VIEW_DETAILS *vd_current;
extern char current_screen;
extern SCREEN_DETAILS screen[MAX_SCREENS];        /* screen structures */
/*man-start*********************************************************************
COMMAND
     pending - set status of pending prefix commands

SYNTAX
     [SET] PENDing ON string
     [SET] PENDing OFF
     [SET] PENDing BLOCK string

DESCRIPTION
     The PENDING command allows the user to insert or remove commands
     from the pending prefix list.

     [SET] PENDing ON 'string', simulates the user typing 'string'
     in the prefix area of the focus line.

     [SET] PENDing OFF, removes any pending prefix command from the
     focus line.

     [SET] PENDing BLOCK 'string', simulates the user typing 'string'
     in the prefix area of the focus line and identifies the prefix 
     command to be a BLOCK command.

COMPATIBILITY
     XEDIT: Does not support ERROR option.
     KEDIT: N/A

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Pending(char *params)
#else
int Pending(params)
char *params;
#endif
/***********************************************************************/
{
#define PEND_ON    1
#define PEND_OFF   2
#define PEND_BLOCK 3
/*------------------------- external data -----------------------------*/
extern char *pre_rec;
extern unsigned short pre_rec_len;
/*--------------------------- local data ------------------------------*/
#define PEN_PARAMS  3
 char *word[PEN_PARAMS+1];
 unsigned short num_params;
 int rc=RC_OK;
 LINE *curr;
 long true_line;
 int command;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Pending");
#endif
/*---------------------------------------------------------------------*/
/* Validate parameters.                                                */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,PEN_PARAMS,WORD_DELIMS,TEMP_PARAM);
/*---------------------------------------------------------------------*/
/* If no arguments, error.                                             */
/*---------------------------------------------------------------------*/
 if (num_params == 0)
   {
    display_error(3,(char *)""); 
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* If more than 2 arguments, error.                                    */
/*---------------------------------------------------------------------*/
 if (num_params > 2)
   {
    display_error(2,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Validate first parameter...                                         */
/*---------------------------------------------------------------------*/
 if (equal((char *)"off",word[0],3))
    command = PEND_OFF;
 else
    if (equal((char *)"on",word[0],2))
       command = PEND_ON;
    else
       if (equal((char *)"block",word[0],5))
          command = PEND_BLOCK;
       else
         {
          display_error(1,word[0]);
#ifdef TRACE
          trace_return();
#endif
          return(RC_INVALID_OPERAND);
         }
 true_line = get_true_line();
 post_process_line(CURRENT_VIEW->focus_line);
 switch(command)
   {
/*---------------------------------------------------------------------*/
/* PENDING ON and PENDING BLOCK...                                     */
/*---------------------------------------------------------------------*/
    case PEND_ON:
    case PEND_BLOCK:
/*---------------------------------------------------------------------*/
/* The second argument must be present and <= PREFIX_WIDTH.            */
/*---------------------------------------------------------------------*/
         if (num_params != 2)
           {
            display_error(3,(char *)"");
            rc = RC_INVALID_OPERAND;
            break;
           }
         if (strlen(word[1]) > PREFIX_WIDTH)
           {
            display_error(1,word[1]); /* different error ?? */
            rc = RC_INVALID_OPERAND;
            break;
           }
/*---------------------------------------------------------------------*/
/* Copy the string into pre_rec and set its length.                    */
/*---------------------------------------------------------------------*/
         memset(pre_rec,' ',PREFIX_WIDTH);
         strcpy(pre_rec,word[1]);
         pre_rec_len = strlen(word[1]);
         pre_rec[pre_rec_len] = ' ';
         pre_rec[PREFIX_WIDTH] = '\0';
         curr = lll_find(CURRENT_FILE->first_line,true_line);
         if (command == PEND_BLOCK)
            add_prefix_command(curr,true_line,TRUE);
         else
            add_prefix_command(curr,true_line,FALSE);
         break;
/*---------------------------------------------------------------------*/
/* PENDING OFF...                                                      */
/*---------------------------------------------------------------------*/
    case PEND_OFF:
         if (num_params != 1)
           {
            display_error(2,(char *)"");
            rc = RC_INVALID_OPERAND;
            break;
           }
         curr = lll_find(CURRENT_FILE->first_line,true_line);
         clear_pending_prefix_command(curr->pre,curr);
         memset(pre_rec,' ',PREFIX_WIDTH);
         pre_rec_len = 0;
         break;
   }
 if (rc == RC_OK)
   {
    if (!in_profile)
       show_page();
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     point - give a name to the current line

SYNTAX
     [SET] Point .name [OFF]

DESCRIPTION
     The POINT command assignes the specified name to the current
     line, or removes the name from the line with the specified name.
     A valid line name must start with a '.' followed by an alphabetic
     character. eg. .a .fred and .GG are valid names; fred and .3 are
     invalid line names.

     When a line is moved within the same file, its line name stays
     with the line.

COMPATIBILITY
     XEDIT: Compatible. See below.
     KEDIT: Compatible. See below.
     Does not allow for multiple names for the same line.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Point(char *params)
#else
int Point(params)
char *params;
#endif
/***********************************************************************/
{
/*------------------------- external date -----------------------------*/
/*--------------------------- local data ------------------------------*/
#define POI_PARAMS  2
 char *word[POI_PARAMS+1];
 unsigned short num_params;
 int rc;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Point");
#endif
/*---------------------------------------------------------------------*/
/* Validate parameters.                                                */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,POI_PARAMS,WORD_DELIMS,TEMP_PARAM);
 if (num_params < 1)
   {
    display_error(3,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Turning on line name...                                             */
/*---------------------------------------------------------------------*/
 if (num_params == 1)
   {
    if (word[0][0] != '.'
    ||  !isalpha(word[0][1]))
      {
       display_error(18,word[0]);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    if ((rc = execute_set_point(word[0],get_true_line(),TRUE)) != RC_OK)
      {
#ifdef TRACE
       trace_return();
#endif
       return(rc);
      }
   }
/*---------------------------------------------------------------------*/
/* Turning off line name...                                            */
/*---------------------------------------------------------------------*/
 else
   {
    if (!equal((char *)"off",word[1],3))
      {
       display_error(1,word[1]);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    if ((rc = execute_set_point(word[0],get_true_line(),FALSE)) != RC_OK)
      {
#ifdef TRACE
       trace_return();
#endif
       return(rc);
      }
   }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     prefix - set prefix area attributes

SYNTAX
     [SET] PREfix ON [Left|Right]
     [SET] PREfix OFF
     [SET] PREfix Synonym newname oldname

DESCRIPTION
     The first form of the PREFIX allows the user to display the
     prefix area and optionally to select the position the prefix
     should be displayed at.
     The second form, turns the display of the prefix area off.
     Executed from within the profile, the only effect is that the
     defaults for all files is changed.
     Executed from the command line, the PREFIX command changes the
     current window displays to reflect the required options.

     The third form of the PREFIX command allows the users to specify
     a synonym for a prefix command or REXX prefix macro. The 'newname' 
     is the command entered in the prefix area and 'oldname' corresponds 
     to an existing prefix command or a REXX macro file in the MACROPATH 
     ending in .the or whatever the value of MACROEXT is at the time the 
     prefix command is executed. The 'oldname' can also be the fully 
     qualified filename of a REXX macro.

COMPATIBILITY
     XEDIT: Compatible. See below.
     KEDIT: Compatible. See below.
     Does not support PREFIX NULLS.

DEFAULT
     PREFIX ON LEFT

STATUS
     Strange display behaviour occurs when you have multiple files with
     mixed prefix settings.
**man-end**********************************************************************/
#ifdef PROTO
int Prefix(char *params)
#else
int Prefix(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- extern data -----------------------------*/
extern char PREFIXx;
/*--------------------------- local data ------------------------------*/
#define PRE_PARAMS  4
 char *word[PRE_PARAMS+1];
 char parm=(char)UNDEFINED_OPERAND;
 register int i;
 unsigned short num_params;
 unsigned short x,y;
 char pre_col;
 short rc;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Prefix");
#endif
 num_params = param_split(params,word,PRE_PARAMS,WORD_DELIMS,TEMP_PARAM);
 if (equal((char *)"on",word[0],2))
   {
    if (num_params > 2)
      {
       display_error(2,(char *)"");
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    if (equal((char *)"left",word[1],1))
       parm = PREFIX_LEFT;
    else
       if (equal((char *)"right",word[1],1))
          parm = PREFIX_RIGHT;
       else
          if (num_params == 1)       /* no left/right, default to left */
             parm = PREFIX_LEFT;
          else
            {
             display_error(1,word[1]);
#ifdef TRACE
             trace_return();
#endif
             return(RC_INVALID_OPERAND);
            }
   }
 else
   {
    if (equal((char *)"off",word[0],3))
      {
       if (num_params > 1)
         {
          display_error(2,(char *)"");
#ifdef TRACE
          trace_return();
#endif
          return(RC_INVALID_OPERAND);
         }
       if (strcmp(word[1],"") == 0)
          parm = PREFIX_OFF;
       else
         {
          display_error(2,word[1]);
#ifdef TRACE
          trace_return();
#endif
          return(RC_INVALID_OPERAND);
         }
      }
    else
       if (equal((char *)"synonym",word[0],1))
         {
          if (num_params < 3)
            {
             display_error(3,(char *)"");
#ifdef TRACE
             trace_return();
#endif
             return(RC_INVALID_OPERAND);
            }
          if (num_params > 3)
            {
             display_error(2,(char *)"");
#ifdef TRACE
             trace_return();
#endif
             return(RC_INVALID_OPERAND);
            }
          if (strlen(word[1]) > PREFIX_WIDTH)
            {
             display_error(37,word[1]);
#ifdef TRACE
             trace_return();
#endif
             return(RC_INVALID_OPERAND);
            }
          rc = add_prefix_synonym(word[1],word[2]);
#ifdef TRACE
          trace_return();
#endif
          return(rc);
         }
       else
         {
          display_error(1,word[0]);
#ifdef TRACE
          trace_return();
#endif
          return(RC_INVALID_OPERAND);
         }
   }

 PREFIXx = parm;
/*---------------------------------------------------------------------*/
/* Now for the real work. Depending on the current state of the prefix */
/* and the intentions of the user, turn on or off the prefix area and  */
/* rearrange the MAIN window.                                          */
/*---------------------------------------------------------------------*/
 switch(CURRENT_VIEW->prefix)
   {
    case PREFIX_LEFT:
    case PREFIX_RIGHT:
/*---------------------------------------------------------------------*/
/* If the prefix is already on and in the same place as requested,     */
/* ignore the command.                                                 */
/*---------------------------------------------------------------------*/
         if (CURRENT_VIEW->prefix == parm)
            break;
         switch(parm)
           {
            case PREFIX_LEFT:
            case PREFIX_RIGHT:
/*---------------------------------------------------------------------*/
/* Now we have to swap the position of the MAIN and PREFIX windows.    */
/* The simplest way is to move the windows to their new positions.     */
/*---------------------------------------------------------------------*/
                 if (CURRENT_VIEW->prefix == PREFIX_LEFT)
                   {
                    CURRENT_SCREEN.top_col -= PREFIX_WIDTH;
                    pre_col = CURRENT_SCREEN.origin_x+CURRENT_SCREEN.screen_cols-PREFIX_WIDTH;
                   }
                 else
                   {
                    CURRENT_SCREEN.top_col += PREFIX_WIDTH;
                    pre_col = CURRENT_SCREEN.origin_x;
                   }
                 if (!in_profile)
                   {
                    rc = mvwin(CURRENT_WINDOW_MAIN,CURRENT_SCREEN.top_row,
                                              CURRENT_SCREEN.top_col);
                    rc = mvwin(CURRENT_WINDOW_PREFIX,CURRENT_SCREEN.top_row,
                                              pre_col);
                   }
                 CURRENT_VIEW->prefix = parm;
                 break;
            case PREFIX_OFF:                /* prefix to be turned off */
                 if (!in_profile)
                   {
                    delwin(CURRENT_WINDOW_PREFIX);
                    getyx(CURRENT_WINDOW_MAIN,y,x);
                   }
                 CURRENT_SCREEN.cols += PREFIX_WIDTH;
                 CURRENT_SCREEN.top_col = CURRENT_SCREEN.origin_x;
                 CURRENT_VIEW->prefix = PREFIX_OFF;
                 if (!in_profile)
                   {
                    CURRENT_WINDOW_MAIN = adjust_window(CURRENT_WINDOW_MAIN,CURRENT_SCREEN.top_row,
                                                   CURRENT_SCREEN.top_col,
                                                   CURRENT_SCREEN.rows,
                                                   CURRENT_SCREEN.cols);
                    wattrset(CURRENT_WINDOW_MAIN,colour[ATTR_FILEAREA]);
#if !defined(NO_KEYPAD)
                    keypad(CURRENT_WINDOW_MAIN,TRUE);
#endif
                   }
                 if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
                    CURRENT_VIEW->previous_window = WINDOW_MAIN;
                 else
                   {
                    CURRENT_VIEW->current_window = WINDOW_MAIN;
                    CURRENT_VIEW->previous_window = WINDOW_COMMAND;
                   }
                 if (!in_profile)
                   {
                    show_page();
                    wmove(CURRENT_WINDOW_MAIN,y,x);
                   }
                 break;
           }
         break;
    case PREFIX_OFF:                        /* prefix is currently off */
         switch(parm)
           {
            case PREFIX_LEFT:
            case PREFIX_RIGHT:
/*---------------------------------------------------------------------*/
/* Determine where the prefix is to go.                                */
/*---------------------------------------------------------------------*/
                 if (parm == PREFIX_LEFT)      /* prefix to go on left */
                   {
                    pre_col = CURRENT_SCREEN.origin_x;
                    CURRENT_SCREEN.top_col = CURRENT_SCREEN.origin_x+PREFIX_WIDTH;
                   }
                 else
                    pre_col = CURRENT_SCREEN.origin_x+CURRENT_SCREEN.screen_cols-PREFIX_WIDTH;
                 CURRENT_SCREEN.cols -= PREFIX_WIDTH;

                 if (!in_profile)
                    getyx(CURRENT_WINDOW_MAIN,y,x);

                 CURRENT_VIEW->prefix = parm;

                 if (!in_profile)
                   {
                    CURRENT_WINDOW_MAIN = adjust_window(CURRENT_WINDOW_MAIN,CURRENT_SCREEN.top_row,
                                                   CURRENT_SCREEN.top_col,
                                                   CURRENT_SCREEN.rows,
                                                   CURRENT_SCREEN.cols);
                    wattrset(CURRENT_WINDOW_MAIN,colour[ATTR_FILEAREA]);
#if !defined(NO_KEYPAD)
                    keypad(CURRENT_WINDOW_MAIN,TRUE);
#endif
                    CURRENT_WINDOW_PREFIX = newwin(CURRENT_SCREEN.rows,PREFIX_WIDTH,
                                         CURRENT_SCREEN.top_row,pre_col);
                    wattrset(CURRENT_WINDOW_PREFIX,colour[ATTR_PENDING]);
#if !defined(NO_KEYPAD)
                    keypad(CURRENT_WINDOW_PREFIX,TRUE);
#endif
                    show_page();
/*---------------------------------------------------------------------*/
/* If the previous column position is now no longer valid, set the     */
/* column position to 0.                                               */
/*---------------------------------------------------------------------*/
                    if (x > CURRENT_SCREEN.cols)
                       x = 0;
                    wmove(CURRENT_WINDOW_MAIN,y,x);
                   }
                 break;
            case PREFIX_OFF:
/*---------------------------------------------------------------------*/
/* If the prefix is already off, ignore request to turn it off.        */
/*---------------------------------------------------------------------*/
                 break;
           }
         break;
   }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     printer - define printer spooler name

SYNTAX
     [SET] PRINTER spooler

DESCRIPTION
     The PRINTER command sets up the print spooler name to determine
     where output from the PRINT command goes.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

SEE ALSO
     print

STATUS
     Incomplete. This command does nothing in the DOS version.
**man-end**********************************************************************/
#ifdef PROTO
int Printer(char *params)
#else
int Printer(params)
char *params;
#endif
/***********************************************************************/
{
/*------------------------- external date -----------------------------*/
#if defined(UNIX) || defined(OS2)
 extern char *spooler_name;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Printer");
#endif
 if ((spooler_name = (char *)realloc(spooler_name,strlen(params)+1)) == NULL)
   {
    display_error(30,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_OUT_OF_MEMORY);
   }
 strcpy(spooler_name,params);
#endif
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     reprofile - indicate if profile file to be executed for all files

SYNTAX
     [SET] REPROFile ON|OFF

DESCRIPTION
     The REPROFILE command allows the user to determine if the profile 
     file is to reexecuted for files subsequenlty edited.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

DEFAULT
     OFF

SEE ALSO
     xedit

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
int Reprofile(char *params)
#else
int Reprofile(params)
char *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool REPROFILEx;
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Reprofile");
#endif
 if (equal((char *)"off",params,3))
    REPROFILEx = FALSE;
 else
    if (equal((char *)"on",params,2))
       REPROFILEx = TRUE;
    else
      {
       display_error(1,(char *)params);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     reserved - reserve a line

SYNTAX
     [SET] RESERved +|-n text|OFF

DESCRIPTION
     The RESERVED command reseves a line for the display of arbitrary
     text by the user. The position is determined by +|-n.
     This number, if positive, specifies the line relative from the
     top of the display. A negative number is relative from the
     bottom of the display.

     By specifying a line, say +3, then THREE lines from the top will
     be reserved, with the supplied text being displayed in line 3.

     The idline of a file will always be displayed after any reserved
     lines.

     The status area will always be displayed on the very last line
     of the screen and any reserved lines will display above that.
     The status area then is effectively line -0;

COMPATIBILITY
     XEDIT: Compatible. See below.
     KEDIT: Compatible. See below.
     Does not allow a colour to be specified. Use SET COLOUR instead.

STATUS
     Not implemented.
**man-end**********************************************************************/
#ifdef PROTO
int Reserved(char *params)
#else
int Reserved(params)
char *params;
#endif
/***********************************************************************/
{
extern int reserved_top,reserved_bottom;
/*--------------------------- local data ------------------------------*/
#define RSR_PARAMS  2
 char *word[RSR_PARAMS+1];
 char parm[RSR_PARAMS];
 unsigned short num_params;
 int line;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Reserved");
#endif
 display_error(0,(char *)"This function has not yet been implemented");
 return(RC_OK);
#ifdef TO_BE_IMPLEMENTED
 num_params = param_split(params,word,RSR_PARAMS,WORD_DELIMS,TEMP_PARAM);
 if (num_params < 2)
    {
     display_error(3,(char *)"");
#ifdef TRACE
     trace_return();
#endif
     return(RC_INVALID_OPERAND);
    }
 if (*word[0] != '+'
 &&  *word[0] != '-')
   {
     display_error(1,word[0]);
#ifdef TRACE
     trace_return();
#endif
     return(RC_INVALID_OPERAND);
    }
 if (valid_integer(word[0]) == NO)
   {
    display_error(4,word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 line = atoi(word[0]);
 if ((line > 0 && line > 10)
 ||  (line < 0 && line*(-1) > 10))
   {
    display_error(6,word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (line > 0)
    reserved_top = max(reserved_top,line);
 if (line < 0)
    reserved_bottom = min(reserved_bottom,line);
/*---------------------------------------------------------------------*/
/* This is where we have to resize the display (if necessary) and      */
/* add the string to stdscr.                                           */
/*---------------------------------------------------------------------*/
 attrset(colour[ATTR_RESERVED]);
 if (line > 0)
    mvaddstr(line-1,0,word[1]);
 else
    mvaddstr(LINES-1+line,0,word[1]);
 clrtoeol();
 wnoutrefresh(stdscr);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
#endif
}
#if !defined(NOREXX)
/*man-start*********************************************************************
COMMAND
     rexxoutput - indicate where REXX output is to go

SYNTAX
     [SET] REXXOUTput File|Display n

DESCRIPTION
     The REXXOUTPUT command indicates where output from the REXX
     interpreter is to go; either captured to a file in the ring
     or displayed in a scrolling fashion on the screen.

     Also specified is the maximum number of lines from the REXX
     interpreter that are to be displayed or captured. This is
     particularly useful when a REXX macro gets into an infinite
     loop.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

DEFAULT
     Display 1000

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Rexxoutput(char *params)
#else
int Rexxoutput(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- extern data -----------------------------*/
extern long CAPREXXMAXx;
extern bool CAPREXXOUTx;
extern bool rexx_output;/*indicates if we are currently trapping output*/
/*--------------------------- local data ------------------------------*/
#define REX_PARAMS  2
 char *word[REX_PARAMS+1];
 register int i;
 unsigned short num_params;
 long num_pages,num_lines;
 unsigned short x,y;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Rexxoutput");
#endif
 if (rexx_output)
   {
    display_error(0,(char *)"Error: Unable to alter REXXOUTPUT settings");
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 num_params = param_split(params,word,REX_PARAMS,WORD_DELIMS,TEMP_PARAM);
 if (num_params < 2)
   {
    display_error(3,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (equal((char *)"file",word[0],1))
    CAPREXXOUTx = TRUE;
 else
    if (equal((char *)"display",word[0],1))
       CAPREXXOUTx = FALSE;
    else
      {
       display_error(1,(char *)word[0]);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }

 if (!valid_positive_integer(word[1]))
   {
    display_error(4,(char *)word[1]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 CAPREXXMAXx =  atol(word[1]);
 return(RC_OK);
}
#endif
/*man-start*********************************************************************
COMMAND
     scope - sets which lines are to be excluded from commands

SYNTAX
     [SET] SCOPE All|Display

DESCRIPTION
     The SCOPE command indicates whether lines not displayed as
     the result of a SET DISPLAY or ALL command are included in
     the scope of lines to be acted upon by other THE commands.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

DEFAULT
     Display

SEE ALSO
     SET DISPLAY, SET SELECT, ALL

STATUS
     Not started.
**man-end**********************************************************************/
#ifdef PROTO
int Scope(char *params)
#else
int Scope(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 int rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Scope");
#endif
 display_error(0,(char *)"This function has not yet been implemented");
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     screen - specifiy number of screens displayed

SYNTAX
     [SET] SCReen n [Horizontal|Vertical]

DESCRIPTION
     The SCREEN command specifies the number of views of file(s) to
     display on screen at once. If the number of views specified is 2
     and only one file is currently in the ring, two views of the
     same file are displayed.

COMPATIBILITY
     XEDIT: Does not support Size,Width or Define options.
     KEDIT: Does not support Size option.
     Only 2 views are supported.

STATUS
     Not complete. (Unavailable)
**man-end**********************************************************************/
#ifdef PROTO
int Screen(char *params)
#else
int Screen(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
#define SCR_PARAMS  2
 char *word[SCR_PARAMS+1];
 register int i;
 unsigned short num_params,num_views;
 char save_display_screens;
 bool save_horizontal;
 unsigned short x,y;
 char horiz=(-1);
 VIEW_DETAILS *save_current_view;
 int rc;

#ifndef VMS
 SCREEN_DETAILS *screen0 = &screen[0];
 SCREEN_DETAILS *screen1 = &screen[1];
#endif
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Screen");
#endif
 display_error(0,(char *)"This function has not yet been implemented");

#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
#ifdef TO_BE_IMPLEMENTED
 num_params = param_split(params,word,SCR_PARAMS,WORD_DELIMS,TEMP_PARAM);
 if (!valid_positive_integer(word[0]))
   {
    display_error(1,word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if ((num_views = atoi(word[0])) > MAX_SCREENS)
   {
    display_error(6,word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (num_views == 1)
    horiz = TRUE;
 else
   {
    if (equal((char *)"horizontal",word[1],1)
    || strcmp(word[1],"") == 0)
       horiz = TRUE;
    if (equal((char *)"vertical",word[1],1))
       horiz = FALSE;
    if (horiz == (-1))
      {
       display_error(1,word[1]);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
   }
/*---------------------------------------------------------------------*/
/* Set the global variable display_screens to indicate the number of   */
/* screens currently displayed and the orientation of those screens    */
/* Save the old values first so we know how the screens were oriented. */
/*---------------------------------------------------------------------*/
 save_display_screens = display_screens;
 save_horizontal = horizontal;

 display_screens = (char)num_views;
 horizontal=(bool)horiz;

 if (in_profile)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* If there is no change to the screens, exit.                         */
/*---------------------------------------------------------------------*/
 if (display_screens == save_display_screens
 &&  horizontal == save_horizontal)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* CURRENTLY ONLY FOR MAKING 2 VIEWS. NOT FOR GOING BACK TO 1 VIEW.....*/
/*---------------------------------------------------------------------*/
 post_process_line(CURRENT_VIEW->focus_line);

 save_current_view = CURRENT_VIEW;

 set_screen_defaults();

 if (!horizontal && display_screens > 1)
    wnoutrefresh(divider);

 if (display_screens == 1)
   {
    if (CURRENT_SCREEN.screen_view->file_for_view == OTHER_SCREEN.screen_view->file_for_view)
      {
       CURRENT_VIEW = OTHER_SCREEN.screen_view;
       free_view_memory();
       CURRENT_VIEW = save_current_view;
      }
    else
      {
       CURRENT_VIEW = OTHER_SCREEN.screen_view;
       for (i=0;i<VIEW_WINDOWS;i++)
           getyx(CURRENT_VIEW->win[i],save_coord_y[i],save_coord_x[i]);
       getyx(CURRENT_WINDOW,y,x);
       delete_windows();
       if ((rc = set_up_windows()) != RC_OK)
         {
#ifdef TRACE
          trace_return();
#endif
          return(rc);
         }
       for (i=0;i<VIEW_WINDOWS;i++)
           wmove(CURRENT_VIEW->win[i],save_coord_y[i],save_coord_x[i]);
       y = get_row_for_focus_line(CURRENT_VIEW->current_row,
                            CURRENT_VIEW->focus_line,
                            CURRENT_VIEW->current_line);
       if (x > CURRENT_SCREEN.cols)
          x = 0;
       wmove(CURRENT_WINDOW,y,x);
       CURRENT_VIEW = save_current_view;
      }
    current_screen = 0;
    CURRENT_SCREEN.screen_view = CURRENT_VIEW;
   }

 getyx(CURRENT_WINDOW,y,x);
 delete_windows();
 if ((rc = set_up_windows()) != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 repaint_screen();

 if (display_screens == 1)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/* now create the new view, if this is the only file */

 if (number_of_views == 1)
   {
    if ((rc = defaults_for_other_files()) != RC_OK)
      {
#ifdef TRACE
       trace_return();
#endif
       return(rc);
      }
    CURRENT_FILE = save_current_view->file_for_view;
    CURRENT_FILE->file_views++;
    x = y = 0;
   }
 else
   {
    if (NEXT_VIEW == (VIEW_DETAILS *)NULL)
       CURRENT_VIEW = vd_first;
    else
       CURRENT_VIEW = NEXT_VIEW;
    getyx(CURRENT_WINDOW,y,x);
    delete_windows();
   }
 current_screen = 1;
 if ((rc = set_up_windows()) != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 CURRENT_SCREEN.screen_view = CURRENT_VIEW;
 repaint_screen();
 if (!horizontal)
   {
    CURRENT_VIEW = save_current_view;
    current_screen = 0;
    pre_process_line(CURRENT_VIEW->focus_line);
   }

#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
#endif
}
/*man-start*********************************************************************
COMMAND
     select - sets the selection level for the specified lines

SYNTAX
     [SET] SELect [+|-]n target

DESCRIPTION
     The SELECT command sets the selection level for the indicated
     lines equal to n (if no signs are specified) or adds or subtracts
     n from the selection level currently set for the lines in the
     target.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     SET SCOPE, SET SELECT, ALL

STATUS
     Not started.
**man-end**********************************************************************/
#ifdef PROTO
int Select(char *params)
#else
int Select(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Select");
#endif
 display_error(0,(char *)"This function has not yet been implemented");

#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     stay - set condition of cursor position after CHANGE/LOCATE commands

SYNTAX
     [SET] STAY ON|OFF

DESCRIPTION
     The STAY set command determines what line is displayed as the current
     line after an unsuccessful LOCATE command or after a CHANGE command.
     With STAY ON, the current line remains where it currently is. With
     STAY OFF, after an unsuccessful LOCATE, the current line is "Bottom
     of File". After a successful CHANGE, the current line is the last
     line affected by the CHANGE command.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

DEFAULT
     STAY ON

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
int Stay(char *params)
#else
int Stay(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- extern data -----------------------------*/
extern char STAYx;
/*--------------------------- local data ------------------------------*/
#define STA_PARAMS  1
 char parm[STA_PARAMS];
 char *word[STA_PARAMS+1];
 register int i;
 unsigned short num_params;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Stay");
#endif
 num_params = param_split(params,word,STA_PARAMS,WORD_DELIMS,TEMP_PARAM);

 parm[0] = (char)UNDEFINED_OPERAND;
 if (equal((char *)"on",word[0],2))
    parm[0] = ON;
 if (equal((char *)"off",word[0],3))
    parm[0] = OFF;
 if (parm[0] == (char)UNDEFINED_OPERAND)
   {
    display_error(1,word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 STAYx = CURRENT_VIEW->stay = parm[0];
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     tabkey - set characteristics of the tab key

SYNTAX
     [SET] TABKey Tab|Character Tab|Character

DESCRIPTION
     The TABKEY sets the action to be taken when the tab key is hit.
     Depending on the insert mode, the tab key will either display
     a raw tab character or will move to the next tab column.
     The first operand refers to the behaviour of the TAB key when
     INSERTMODE is OFF; the second operand specifies the behaviour
     when the TAB key is hit when INSERTMODE is ON.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

DEFAULT
     TAB CHARACTER

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
int Tabkey(char *params)
#else
int Tabkey(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- extern data -----------------------------*/
extern char tabkey_insert,tabkey_overwrite;
/*--------------------------- local data ------------------------------*/
#define TKY_PARAMS  3
 char *word[TKY_PARAMS+1];
 unsigned short num_params;
 char tabo=tabkey_overwrite;
 char tabi=tabkey_insert;
 int rc=RC_INVALID_OPERAND;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Tabkey");
#endif
/*---------------------------------------------------------------------*/
/* Validate the parameters that have been supplied.                    */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,TKY_PARAMS,WORD_DELIMS,TEMP_PARAM);
 switch(num_params)
   {
/*---------------------------------------------------------------------*/
/* Too few parameters, error.                                          */
/*---------------------------------------------------------------------*/
    case 0: 
    case 1:
         display_error(3,(char *)""); 
         break;
/*---------------------------------------------------------------------*/
/* 2 parameters, validate them...                                      */
/*---------------------------------------------------------------------*/
    case 2: 
/*---------------------------------------------------------------------*/
/* Validate first parameter; overwrite mode setting...                 */
/*---------------------------------------------------------------------*/
         if (equal((char *)"character",word[0],1))
           {
            tabo = 'C';
            rc = RC_OK;
           }
         if (equal((char *)"tab",word[0],1))
           {
            tabo = 'T';
            rc = RC_OK;
           }
/*---------------------------------------------------------------------*/
/* If not a valid first parameter, display an error and exit.          */
/*---------------------------------------------------------------------*/
         if (rc != RC_OK)
           {
            display_error(1,word[0]);
            break;
           }
         rc = RC_INVALID_OPERAND;
/*---------------------------------------------------------------------*/
/* Validate second parameter; insert mode setting...                   */
/*---------------------------------------------------------------------*/
         if (equal((char *)"character",word[1],1))
           {
            tabi = 'C';
            rc = RC_OK;
           }
/*---------------------------------------------------------------------*/
/* Validate second parameter; insert mode setting...                   */
/*---------------------------------------------------------------------*/
         if (equal((char *)"tab",word[1],1))
           {
            tabi = 'T';
            rc = RC_OK;
           }
/*---------------------------------------------------------------------*/
/* If not a valid second parameter, display an error and exit.         */
/*---------------------------------------------------------------------*/
         if (rc != RC_OK)
           {
            display_error(1,word[1]);
            break;
           }
         rc = RC_OK;
         break;
/*---------------------------------------------------------------------*/
/* Too many parameters...                                              */
/*---------------------------------------------------------------------*/
    default:
         display_error(2,(char *)""); 
         break;
   }
/*---------------------------------------------------------------------*/
/* If valid parameters, change the settings...                         */
/*---------------------------------------------------------------------*/
 if (rc == RC_OK)
   {
    tabkey_insert = tabi;
    tabkey_overwrite = tabo;
   }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     tabs - set tab length for TAB key

SYNTAX
     [SET] TABS INCR n

DESCRIPTION
     The TABS command determines the position of tab columns when the
     TAB key is pressed.

COMPATIBILITY
     XEDIT: Does not support specification of actual tab columns.
     KEDIT: Does not support specification of actual tab columns.

SEE ALSO
     SOS TABF

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Tabs(char *params)
#else
int Tabs(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- extern data -----------------------------*/
extern char TABSx;
/*--------------------------- local data ------------------------------*/
#define TABS_PARAMS  2
 char *word[TABS_PARAMS+1];
 char parm[TABS_PARAMS];
 register int i;
 unsigned short num_params;
 unsigned short x,y;
 short rc;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Tabs");
#endif
 num_params = param_split(params,word,TABS_PARAMS,WORD_DELIMS,TEMP_PARAM);
 for (i=0;i<TABS_PARAMS;i++)
     parm[0] = (char)UNDEFINED_OPERAND;
 if (equal((char *)"incr",word[0],2))
    parm[0] = TRUE;
 if (parm[0] == (char)UNDEFINED_OPERAND)
   {
    display_error(1,word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (!valid_positive_integer(word[1]))
   {
    display_error(4,word[1]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 parm[1] = (char)atoi(word[1]);
 if (parm[1] < 1)
   {
    display_error(5,word[1]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (parm[1] > 32)
   {
    display_error(6,word[1]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 TABSx = CURRENT_VIEW->tabs = parm[1];
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     tabsin - set tab processing on file input

SYNTAX
     [SET] TABSIn ON|OFF [n]

DESCRIPTION
     The TABSIN command determines if tabs read from a file are to be
     expanded to spaces and if so how many spaces.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Does not support TABQUOTE option.

SEE ALSO
     tabsout

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Tabsin(char *params)
#else
int Tabsin(params)
char *params;
#endif
/***********************************************************************/
{
/*---------------------------------------------------------------------*/
/* The settings for TABSIN is a global value, despite it supposedly    */
/* being a file level value.                                           */
/*---------------------------------------------------------------------*/
/*--------------------------- extern data -----------------------------*/
extern char TABI_ONx;
extern char TABI_Nx;
/*--------------------------- local data ------------------------------*/
#define TABI_PARAMS  3
 char *word[TABI_PARAMS+1];
 unsigned short num_params;
 int rc=RC_INVALID_OPERAND;
 char tabsts=TABI_ONx;
 char tabn  =TABI_Nx;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Tabsin");
#endif
/*---------------------------------------------------------------------*/
/* Validate the parameters that have been supplied.                    */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,TABI_PARAMS,WORD_DELIMS,TEMP_PARAM);
 switch(num_params)
   {
/*---------------------------------------------------------------------*/
/* Too few parameters, error.                                          */
/*---------------------------------------------------------------------*/
    case 0: 
         display_error(3,(char *)""); 
         break;
/*---------------------------------------------------------------------*/
/* 1 or 2 parameters, validate them...                                 */
/*---------------------------------------------------------------------*/
    case 1:
    case 2: 
/*---------------------------------------------------------------------*/
/* Validate first parameter; on or off...                              */
/*---------------------------------------------------------------------*/
         if (equal((char *)"on",word[0],2))
           {
            tabsts = TRUE;
            rc = RC_OK;
           }
         if (equal((char *)"off",word[0],3))
           {
            tabsts = FALSE;
            rc = RC_OK;
           }
/*---------------------------------------------------------------------*/
/* If not a valid first parameter, display an error and exit.          */
/*---------------------------------------------------------------------*/
         if (rc != RC_OK)
           {
            display_error(1,word[0]);
            break;
           }
/*---------------------------------------------------------------------*/
/* For 1 parameter, don't check any more.                              */
/*---------------------------------------------------------------------*/
         if (num_params == 1)
            break;
/*---------------------------------------------------------------------*/
/* Validate second parameter; number of spaces for a TAB...            */
/*---------------------------------------------------------------------*/
         if (!valid_positive_integer(word[1]))
           {
            display_error(4,word[1]);
            break;
           }
         tabn = (char)atoi(word[1]);
/*---------------------------------------------------------------------*/
/* tabn must be between 1 and 32...                                    */
/*---------------------------------------------------------------------*/
         if (tabn < 1)
           {
            display_error(5,word[1]);
            break;
           }
         if (tabn > 32)
           {
            display_error(6,word[1]);
            break;
           }
         rc = RC_OK;
         break;
/*---------------------------------------------------------------------*/
/* Too many parameters...                                              */
/*---------------------------------------------------------------------*/
    default:
         display_error(2,(char *)""); 
         break;
   }
/*---------------------------------------------------------------------*/
/* If valid parameters, change the settings...                         */
/*---------------------------------------------------------------------*/
 if (rc == RC_OK)
   {
    TABI_ONx = tabsts;
    TABI_Nx = tabn;
/*---------------------------------------------------------------------*/
/* If this command is issued from the profile file, we need to run     */
/* EXPAND ALL on it, as we have already read in the file.              */
/*---------------------------------------------------------------------*/
    if (in_profile)
       rc = Expand("ALL");
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     tabsout - set tab processing on file output

SYNTAX
     [SET] TABSOut ON n
     [SET] TABSOut OFF

DESCRIPTION
     The TABSOUT command determines if spaces written to a file are to be
     compressed to tabs and if so how many spaces.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

SEE ALSO
     tabsin

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Tabsout(char *params)
#else
int Tabsout(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- extern data -----------------------------*/
extern char TABO_ONx;
extern char TABO_Nx;
/*--------------------------- local data ------------------------------*/
#define TABO_PARAMS  3
 char *word[TABO_PARAMS+1];
 unsigned short num_params;
 int rc=RC_INVALID_OPERAND;
 char tabsts=CURRENT_FILE->tabsout_on;
 char tabn  =CURRENT_FILE->tabsout_num;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Tabsout");
#endif
/*---------------------------------------------------------------------*/
/* Validate the parameters that have been supplied.                    */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,TABO_PARAMS,WORD_DELIMS,TEMP_PARAM);
 switch(num_params)
   {
/*---------------------------------------------------------------------*/
/* Too few parameters, error.                                          */
/*---------------------------------------------------------------------*/
    case 0: 
         display_error(3,(char *)""); 
         break;
/*---------------------------------------------------------------------*/
/* 1 or 2 parameters, validate them...                                 */
/*---------------------------------------------------------------------*/
    case 1:
    case 2: 
/*---------------------------------------------------------------------*/
/* Validate first parameter; on or off...                              */
/*---------------------------------------------------------------------*/
         if (equal((char *)"on",word[0],2))
           {
            tabsts = TRUE;
            rc = RC_OK;
           }
         if (equal((char *)"off",word[0],3))
           {
            tabsts = FALSE;
            rc = RC_OK;
           }
/*---------------------------------------------------------------------*/
/* If not a valid first parameter, display an error and exit.          */
/*---------------------------------------------------------------------*/
         if (rc != RC_OK)
           {
            display_error(1,word[0]);
            break;
           }
/*---------------------------------------------------------------------*/
/* For 1 parameter, don't check any more.                              */
/*---------------------------------------------------------------------*/
         if (num_params == 1)
            break;
/*---------------------------------------------------------------------*/
/* Validate second parameter; number of spaces for a TAB...            */
/*---------------------------------------------------------------------*/
         if (!valid_positive_integer(word[1]))
           {
            display_error(4,word[1]);
            break;
           }
         tabn = (char)atoi(word[1]);
/*---------------------------------------------------------------------*/
/* tabn must be between 1 and 32...                                    */
/*---------------------------------------------------------------------*/
         if (tabn < 1)
           {
            display_error(5,word[1]);
            break;
           }
         if (tabn > 32)
           {
            display_error(6,word[1]);
            break;
           }
         rc = RC_OK;
         break;
/*---------------------------------------------------------------------*/
/* Too many parameters...                                              */
/*---------------------------------------------------------------------*/
    default:
         display_error(2,(char *)""); 
         break;
   }
/*---------------------------------------------------------------------*/
/* If valid parameters, change the settings...                         */
/*---------------------------------------------------------------------*/
 if (rc == RC_OK)
   {
    TABO_ONx = CURRENT_FILE->tabsout_on = tabsts;
    TABO_Nx = CURRENT_FILE->tabsout_num = tabn;
   }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     verify - set column display limits

SYNTAX
     [SET] Verify first [last]

DESCRIPTION
     The VERIFY command sets the column limits for the display of the
     current file. 'first' specifies the first column to be displayed
     and 'last' specifies the last column to be displayed.

     If no 'last' option is specified '*' is assumed.

COMPATIBILITY
     XEDIT: Does not implement HEX display nor multiple column pairs.
     KEDIT: Does not implement HEX display nor multiple column pairs.

SEE ALSO
     Zone

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Verify(char *params)
#else
int Verify(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- extern data -----------------------------*/
 extern unsigned short VER_STAx;
 extern unsigned short VER_COLx;
 extern unsigned short VER_ENDx;
/*--------------------------- local data ------------------------------*/
#define VER_PARAMS  2
 char *word[VER_PARAMS+1];
 unsigned short num_params;
 long col1,col2;
 unsigned short x,y;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Verify");
#endif
/*---------------------------------------------------------------------*/
/* Validate the parameters that have been supplied. One or two         */
/* parameters can be supplied. The first parameter MUST be a positive  */
/* integer. The second can be a positive integer or '*'. If no second  */
/* parameter is supplied, '*' is assumed. The second parameter MUST be */
/* >= first parameter. '*' is regarded as the biggest number and is    */
/* literally max_line_length.                                          */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,VER_PARAMS,WORD_DELIMS,TEMP_PARAM);
 if (num_params == 0)
   {
    display_error(3,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (num_params > 2)
   {
    display_error(2,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (!valid_positive_integer(word[0]))
   {
    display_error(4,word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 col1 = atol(word[0]);
 if (num_params == 1)
     col2 = max_line_length;
 else
     if (strcmp(word[1],"*") == 0)
        col2 = max_line_length;
     else
        if (!valid_positive_integer(word[1]))
          {
           display_error(4,word[1]);
#ifdef TRACE
           trace_return();
#endif
           return(RC_INVALID_OPERAND);
          }
        else
           col2 = atol(word[1]);

 if (col2 > max_line_length)
    col2 = max_line_length;
 if (col1 > col2)
   {
    display_error(6,word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 CURRENT_VIEW->verify_start = col1;
 CURRENT_VIEW->verify_col = col1;
 CURRENT_VIEW->verify_end   = col2;
 if (in_profile)
   {
    VER_STAx = col1;
    VER_COLx = col1;
    VER_ENDx = col2;
   }
 else
   {
    getyx(CURRENT_WINDOW,y,x);
    show_page();
    if (CURRENT_VIEW->current_window != WINDOW_COMMAND)
       wmove(CURRENT_WINDOW,y,x);
   }

#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     wordwrap - set wordwrap feature on or off

SYNTAX
     [SET] WORDWrap ON|OFF

DESCRIPTION
     The WORDWRAP set command determines whether wordwrap occurs when
     the cursor moves past the right margin (as set my [SET] MARGINS 
     command).
     With WORDWRAP ON, the line, from the beginning of the word that
     exceeds the right margin, is wrapped onto the next line. The cursor
     position stays in the same position relative to the current word.
     WIth WORDWRAP OFF, no word wrap occurs.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

DEFAULT
     WORDWRAP OFF

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Wordwrap(char *params)
#else
int Wordwrap(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- extern data -----------------------------*/
extern char WORx;
/*--------------------------- local data ------------------------------*/
#define WOR_PARAMS  1
 char parm[WOR_PARAMS];
 char *word[WOR_PARAMS+1];
 register int i;
 unsigned short num_params;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Wordwrap");
#endif
 num_params = param_split(params,word,WOR_PARAMS,WORD_DELIMS,TEMP_PARAM);

 parm[0] = (char)UNDEFINED_OPERAND;
 if (equal((char *)"on",word[0],2))
    parm[0] = ON;
 if (equal((char *)"off",word[0],3))
    parm[0] = OFF;
 if (parm[0] == (char)UNDEFINED_OPERAND)
   {
    display_error(1,word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 CURRENT_VIEW->wordwrap = parm[0];
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     zone - set column limits for editing

SYNTAX
     [SET] Zone first [last]

DESCRIPTION
     The ZONE command sets the column limits for various other editor
     commands, such as 'locate' and 'change'. It effectively restricts
     to the specified columns those parts of the file which can be
     acted upon.

     If no 'last' option is specified '*' is assumed.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     Verify

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Zone(char *params)
#else
int Zone(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- extern data -----------------------------*/
 extern unsigned short ZON_STAx;
 extern unsigned short ZON_ENDx;
 extern char file_read; /* indicates if we have read the file */
/*--------------------------- local data ------------------------------*/
#define ZON_PARAMS  2
 char *word[ZON_PARAMS+1];
 unsigned short num_params;
 long col1,col2;
 unsigned short x,y;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset2.c:Zone");
#endif
/*---------------------------------------------------------------------*/
/* Validate the parameters that have been supplied. One only           */
/* parameter MUST be supplied. The first parameter MUST be a positive  */
/* integer. The second can be a positive integer or '*'. If no second  */
/* parameter is supplied, ERROR.          The second parameter MUST be */
/* >= first parameter. '*' is regarded as the biggest number and is    */
/* literally max_line_length.                                          */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,ZON_PARAMS,WORD_DELIMS,TEMP_PARAM);
 if (num_params < 2)
   {
    display_error(3,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (num_params > 2)
   {
    display_error(2,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (!valid_positive_integer(word[0]))
   {
    display_error(4,word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 col1 = atol(word[0]);
 if (strcmp(word[1],"*") == 0)
    col2 = max_line_length;
 else
    if (!valid_positive_integer(word[1]))
      {
       display_error(4,word[1]);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    else
       col2 = atol(word[1]);

 if (col2 > max_line_length)
    col2 = max_line_length;
 if (col1 > col2)
   {
    display_error(6,word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 CURRENT_VIEW->zone_start = col1;
 CURRENT_VIEW->zone_end   = col2;
 if (in_profile)
   {
    ZON_STAx = col1;
    ZON_ENDx = col2;
   }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
