;;; This module implements a simple Scheme evaluator.
;;;
;;; Initialization of this module will assure that all modules in the
;;; "standard" library are initialized.

;*              Copyright 1989 Digital Equipment Corporation
;*                         All Rights Reserved
;*
;* Permission to use, copy, and modify this software and its documentation is
;* hereby granted only under the following terms and conditions.  Both the
;* above copyright notice and this permission notice must appear in all copies
;* of the software, derivative works or modified versions, and any portions
;* thereof, and both notices must appear in supporting documentation.
;*
;* Users of this software agree to the terms and conditions set forth herein,
;* and hereby grant back to Digital a non-exclusive, unrestricted, royalty-free
;* right and license under any changes, enhancements or extensions made to the
;* core functions of the software, including but not limited to those affording
;* compatibility with other hardware or software environments, but excluding
;* applications which incorporate this software.  Users further agree to use
;* their best efforts to return to Digital any such changes, enhancements or
;* extensions that they make and inform Digital of noteworthy uses of this
;* software.  Correspondence should be provided to Digital at:
;* 
;*                       Director of Licensing
;*                       Western Research Laboratory
;*                       Digital Equipment Corporation
;*                       100 Hamilton Avenue
;*                       Palo Alto, California  94301  
;* 
;* This software may be distributed (but not offered for sale or transferred
;* for compensation) to third parties, provided such third parties agree to
;* abide by the terms and conditions of this notice.  
;* 
;* THE SOFTWARE IS PROVIDED "AS IS" AND DIGITAL EQUIPMENT CORP. DISCLAIMS ALL
;* WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
;* MERCHANTABILITY AND FITNESS.   IN NO EVENT SHALL DIGITAL EQUIPMENT
;* CORPORATION BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
;* DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
;* PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
;* ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
;* SOFTWARE.

(module sceval
    (top-level EVAL)
    (with scrt1 scrt2 scrt3 scrt4 scrt5 scrt6 scrt7))

(include "repdef.sc")

(define-c-external (PROCEDURE->ADDRESS tscp) unsigned "sc_procedureaddress")

;;; The Scheme interpreter starts here.

(define (EVAL form . env)
    (exec (expand form) (if env (car env) '())))

(define (NEW-ENV idl vals env)
    (cond ((null? idl)
	   (if vals (error 'NEW-ENV "Too many arguments to function"))
	   env)
	  ((symbol? idl)
	   (cons (cons idl vals) env))
	  ((null? vals)
	   (error 'NEW-ENV "Too few arguments to function"))
	  (else
	   (cons (cons (car idl) (car vals))
		 (new-env (cdr idl) (cdr vals) env)))))

(define (LOOKUP id env)
    (let ((pair (assq id env)))
	 (if pair
	     (cdr pair)
	     (let ((val (top-level-value id)))
		  (if (eq? val $_undefined)
		      (error id "Top-level symbol is undefined")
		      val)))))

(define (ASSIGN id val env)
        (let ((pair (assq id env)))
         (if pair (set-cdr! pair val) (set-top-level-value! id val))))

(define (EXEC exp env)
    ((lap (x y) (LOOPSTACKTRACE x y)) exp env)
    (cond ((symbol? exp)
	   (lookup exp env))
	  ((pair? exp)
	   (case (car exp)
		 ((quote)
		  (cadr exp))
		 ((lambda)
		  (let ((env  env)
			(vars (cadr exp))
			(body (cons 'begin (cddr exp))))
		       (lambda vals
			       (if interpreted-tail-call
				   (list body (new-env vars vals env))
				   (exec body (new-env vars vals env))))))
		 ((if)
		  (if (exec (cadr exp) env)
		      (exec (caddr exp) env)
		      (exec (cadddr exp) env)))
		 ((set!)
		  (assign (cadr exp) (exec (caddr exp) env) env))
		 ((begin)
		  (do ((exps (cdr exp) (cdr exps)))
		      ((null? (cdr exps)) (exec (car exps) env))
		      (exec (car exps) env)))
		 (else
		  (let ((function (exec (car exp) env))
			(args (let loop ((x (cdr exp)) (env env))
				   (if (null? x)
				       x
				       (cons (exec (car x) env)
					     (loop (cdr x) env)))))
			(exp-env '()))
		       (cond ((not (procedure? function))
			      (error 'exec
				     "Argument value is not a function: ~s"
				     (car exp)))
			     ((eqv? (procedure->address function)
				    (procedure->address interpreted-proc))
			      (set! interpreted-tail-call #t)
			      (set! exp-env (apply function args))
			      (set! interpreted-tail-call #f)
			      (exec (car exp-env) (cadr exp-env)))
			     (else (apply function args)))))))
	  ((number? exp) exp)
	  ((string? exp) exp)
	  ((char? exp) exp)
	  ((boolean? exp) exp)
	  (else (error 'exec "Argument is not self-evaluating: ~s" exp))))

(define INTERPRETED-TAIL-CALL '())

(define INTERPRETED-PROC (exec '(lambda (x) x) '()))
