;;; SCHEME->C Runtime Library

;*              Copyright 1989 Digital Equipment Corporation
;*                         All Rights Reserved
;*
;* Permission to use, copy, and modify this software and its documentation is
;* hereby granted only under the following terms and conditions.  Both the
;* above copyright notice and this permission notice must appear in all copies
;* of the software, derivative works or modified versions, and any portions
;* thereof, and both notices must appear in supporting documentation.
;*
;* Users of this software agree to the terms and conditions set forth herein,
;* and hereby grant back to Digital a non-exclusive, unrestricted, royalty-free
;* right and license under any changes, enhancements or extensions made to the
;* core functions of the software, including but not limited to those affording
;* compatibility with other hardware or software environments, but excluding
;* applications which incorporate this software.  Users further agree to use
;* their best efforts to return to Digital any such changes, enhancements or
;* extensions that they make and inform Digital of noteworthy uses of this
;* software.  Correspondence should be provided to Digital at:
;* 
;*                       Director of Licensing
;*                       Western Research Laboratory
;*                       Digital Equipment Corporation
;*                       100 Hamilton Avenue
;*                       Palo Alto, California  94301  
;* 
;* This software may be distributed (but not offered for sale or transferred
;* for compensation) to third parties, provided such third parties agree to
;* abide by the terms and conditions of this notice.  
;* 
;* THE SOFTWARE IS PROVIDED "AS IS" AND DIGITAL EQUIPMENT CORP. DISCLAIMS ALL
;* WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
;* MERCHANTABILITY AND FITNESS.   IN NO EVENT SHALL DIGITAL EQUIPMENT
;* CORPORATION BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
;* DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
;* PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
;* ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
;* SOFTWARE.

(module scrt4
    (top-level
	VECTOR? VECTOR VECTOR-LENGTH VECTOR-REF VECTOR-SET!
	VECTOR->LIST LIST->VECTOR VECTOR-FILL!
	PROCEDURE? APPLY MAP FOR-EACH FORCE MAKE-PROMISE
	C-STRING->STRING
	C-BYTE-REF C-SHORTINT-REF C-SHORTUNSIGNED-REF C-INT-REF
	C-UNSIGNED-REF C-TSCP-REF C-FLOAT-REF C-DOUBLE-REF
	C-BYTE-SET! C-SHORTINT-SET! C-SHORTUNSIGNED-SET! C-INT-SET!
        C-UNSIGNED-SET! C-TSCP-SET! C-FLOAT-SET! C-DOUBLE-SET!
	SCHEME-BYTE-REF SCHEME-INT-REF SCHEME-TSCP-REF
	SCHEME-BYTE-SET! SCHEME-INT-SET! SCHEME-TSCP-SET!
	BIT-AND BIT-OR BIT-NOT BIT-XOR BIT-LSH BIT-RSH
	WHEN-UNREFERENCED SIGNAL))

;;; 6.8  Vectors.

(define (VECTOR? x) (vector? x))

(define (VECTOR . x) (list->vector x))

(define (VECTOR-LENGTH x) (vector-length x))

(define (VECTOR-REF x y) (vector-ref x y))

(define (VECTOR-SET! x y z) (vector-set! x y z))

(define (VECTOR->LIST x)
    (do ((i (- (vector-length x) 1) (- i 1))
	 (l '()))
	((= i -1) l)
	(set! l (cons (vector-ref x i) l))))

(define (LIST->VECTOR x)
    (do ((v (make-vector (length x)))
	 (x x (cdr x))
	 (i 0 (+ i 1)))
	((null? x) v)
	(vector-set! v i (car x))))

(define (VECTOR-FILL! v x)
    (do ((i (- (vector-length v) 1) (- i 1)))
	((= i -1) v)
	(vector-set! v i x)))

;;; 6.9  Control features.

(define (PROCEDURE? x) (procedure? x))

(define (APPLY proc args . opt)
    (if opt
	(apply-two proc (cons args (let loop ((opt opt))
					(if (cdr opt)
					    (cons (car opt) (loop (cdr opt)))
					    (car opt)))))
	(apply-two proc args)))

(define (MAP proc args . opt)
    (let loop ((args (cons args opt)) (head '()) (tail '()))
	 (if (not (null? (car args)))
	     (let ((val (cons (apply proc (map car args)) '())))
		  (if (null? head)
		      (loop (map cdr args) val val)
		      (loop (map cdr args) head (set-cdr! tail val))))
	     head)))

(define (FOR-EACH proc args . opt)
    (do ((args (cons args opt) (map cdr args)))
	((null? (car args)))
	(apply proc (map car args))))

(define (FORCE object) (object))

(define (MAKE-PROMISE proc)
    (let ((already-run? #f)
	  (result #f))
	 (lambda ()
		 (unless already-run?
			 (set! result (proc))
			 (set! already-run? #t))
		 result)))

;;; *.*  Functions to access C structures.  Use at your own risk!

(define-c-external (SC_CSTRINGTOSTRING pointer) tscp "sc_cstringtostring")

(define (C-STRING->STRING s) (sc_cstringtostring s))

(define (C-BYTE-REF struct x)
    ((lap (struct x)
	  (INT_TSCP (MBYTE (TSCP_POINTER struct) (TSCP_INT x))))
     struct x))

(define (C-SHORTINT-REF struct x)
    ((lap (struct x)
	  (INT_TSCP (MSINT (TSCP_POINTER struct) (TSCP_INT x))))
     struct x))

(define (C-SHORTUNSIGNED-REF struct x)
    ((lap (struct x)
	  (UNSIGNED_TSCP (MSUNSIGNED (TSCP_POINTER struct) (TSCP_INT x))))
     struct x))

(define (C-INT-REF struct x)
    ((lap (struct x)
	  (INT_TSCP (MINT (TSCP_POINTER struct) (TSCP_INT x))))
     struct x))

(define (C-UNSIGNED-REF struct x)
    ((lap (struct x)
	  (UNSIGNED_TSCP (MUNSIGNED (TSCP_POINTER struct) (TSCP_INT x))))
     struct x))

(define (C-TSCP-REF struct x)
    ((lap (struct x)
	  (MTSCP (TSCP_POINTER struct) (TSCP_INT x)))
     struct x))

(define (C-FLOAT-REF struct x)
    ((lap (struct x)
	  (DOUBLE_TSCP (CDOUBLE (MFLOAT (TSCP_POINTER struct) (TSCP_INT x)))))
     struct x))

(define (C-DOUBLE-REF struct x)
    ((lap (struct x)
	  (DOUBLE_TSCP (MDOUBLE (TSCP_POINTER struct) (TSCP_INT x))))
     struct x))

(define (C-BYTE-SET! struct x v)
    ((lap (struct x v)
	  (SET (MBYTE (TSCP_POINTER struct) (TSCP_INT x)) (TSCP_INT v)))
     struct x v)
    v)

(define (C-SHORTINT-SET! struct x v)
    ((lap (struct x v)
	  (SET (MSINT (TSCP_POINTER struct) (TSCP_INT x)) (TSCP_INT v)))
     struct x v)
    v)

(define (C-SHORTUNSIGNED-SET! struct x v)
    ((lap (struct x v)
	  (SET (MSUNSIGNED (TSCP_POINTER struct) (TSCP_INT x))
	       (TSCP_UNSIGNED v)))
     struct x v)
    v)

(define (C-INT-SET! struct x v)
    ((lap (struct x v)
	  (SET (MINT (TSCP_POINTER struct) (TSCP_INT x)) (TSCP_INT v)))
     struct x v)
    v)

(define (C-UNSIGNED-SET! struct x v)
    ((lap (struct x v)
	  (SET (MUNSIGNED (TSCP_POINTER struct) (TSCP_INT x))
	       (TSCP_UNSIGNED v)))
     struct x v)
    v)

(define (C-TSCP-SET! struct x v)
    ((lap (struct x v)
	  (SET (MTSCP (TSCP_POINTER struct) (TSCP_INT x))
	       v))
     struct x v)
    v)

(define (C-FLOAT-SET! struct x v)
    ((lap (struct x v)
	  (SET (MFLOAT (TSCP_POINTER struct) (TSCP_INT x)) (TSCP_DOUBLE v)))
     struct x v)
    v)

(define (C-DOUBLE-SET! struct x v)
    ((lap (struct x v)
	  (SET (MDOUBLE (TSCP_POINTER struct) (TSCP_INT x)) (TSCP_DOUBLE v)))
     struct x v)
    v)

;;; *.*  Functions to access Scheme structures.  Use at your own risk!	   

(define-in-line (SCHEME-PTR? x)
    ((lap (x) (BOOLEAN (BITAND (TSCPTAG x) 1))) x))

(define (SCHEME-BYTE-REF struct x)
    (if (not (scheme-ptr? struct))
	(error 'SCHEME-BYTE-REF "Structure is not a SCHEME pointer: ~s"
	       struct))
    ((lap (struct x) (INT_TSCP (MBYTE (T_U struct) (TSCP_INT x))))
     struct x))

(define (SCHEME-INT-REF struct x)
    (if (not (scheme-ptr? struct))
	(error 'SCHEME-INT-REF "Structure is not a SCHEME pointer: ~s"
	       struct))
    ((lap (struct x) (INT_TSCP (MINT (T_U struct) (TSCP_INT x))))
     struct x))

(define (SCHEME-TSCP-REF struct x)
    (if (not (scheme-ptr? struct))
	(error 'SCHEME-TSCP-REF "Structure is not a SCHEME pointer: ~s"
	       struct))
    ((lap (struct x) (_TSCP (MINT (T_U struct) (TSCP_INT x))))
     struct x))

(define (SCHEME-BYTE-SET! struct x v)
    (if (not (scheme-ptr? struct))
	(error 'SCHEME-BYTE-SET! "Structure is not a SCHEME pointer: ~s"
	       struct))
    ((lap (struct x v) (SET (MBYTE (T_U struct) (TSCP_INT x)) (TSCP_INT v)))
     struct x v)
    v)

(define (SCHEME-INT-SET! struct x v)
    (if (not (scheme-ptr? struct))
	(error 'SCHEME-INT-SET! "Structure is not a SCHEME pointer: ~s"
	       struct))
    ((lap (struct x v) (SET (MINT (T_U struct) (TSCP_INT x)) (TSCP_INT v)))
     struct x v)
    v)

(define (SCHEME-TSCP-SET! struct x v)
    (if (not (scheme-ptr? struct))
	(error 'SCHEME-TSCP-SET! "Structure is not a SCHEME pointer: ~s"
	       struct))
    ((lap (struct x v) (SETGENTL (MINT (T_U struct) (TSCP_INT x)) (INT v)))
     struct x v)
    v)

;;; *.*  Bit operations.

(define (BIT-AND x . y)
    (let loop ((result x) (y y))
	 (if (null? y)
	     result
	     (loop ((lap (x y) (unsigned_tscp (bitand (tscp_unsigned x)
						      (tscp_unsigned y))))
		    (car y) result)
		   (cdr y)))))

(define (BIT-OR x . y)
    (let loop ((result x) (y y))
	 (if (null? y)
	     result
	     (loop ((lap (x y) (unsigned_tscp (bitor (tscp_unsigned x)
						     (tscp_unsigned y))))
		    (car y) result)
		   (cdr y)))))

(define (BIT-NOT x) (bit-xor x -1))

(define (BIT-XOR x . y)
    (let loop ((result x) (y y))
	 (if (null? y)
	     result
	     (loop ((lap (x y) (unsigned_tscp (bitxor (tscp_unsigned x)
						      (tscp_unsigned y))))
		    (car y) result)
		   (cdr y)))))

(define (BIT-LSH x y)
    ((lap (x y) (unsigned_tscp (bitlsh (tscp_unsigned x) (tscp_unsigned y))))
     x y))

(define (BIT-RSH x y)
    ((lap (x y) (unsigned_tscp (bitrsh (tscp_unsigned x) (tscp_unsigned y))))
     x y))

;;; *.* Garbage collection finalization for unreferenced objects.

(define-external WHENFREED "sc" "whenfreed")

(define (WHEN-UNREFERENCED obj proc)
    (let ((result #f))
	 (if (and proc (not (procedure? proc)))
	     (error 'WHEN-FREED "Argument is not a PROCEDURE: ~s" proc))
	 (set! whenfreed
	       (let loop ((l whenfreed))
		    (if l
			(if (eq? (caar l) obj)
			    (begin (set! result (cdar l))
				   (if proc
				       (cons (cons obj proc) (cdr l))
				       (cdr l)))
			    (cons (car l) (loop (cdr l))))
			(if proc
			    (list (cons obj proc))
			    '()))))
	 result))

(define (SC_WHENFREED) whenfreed)

;;; *.* Signal handling

(define SIGNALS (make-vector 32 #f))

(define-c-external (UNIX-SIGNAL int pointer) pointer "signal")

(define-c-external (SC_ONSIGNAL1 int) void "sc_onsignal1")

(define (SIGNAL sig handler)
    (if (or (negative? sig) (>= sig (vector-length signals)))
	(error 'SIGNAL "Argument is not a valid SIGNAL: ~s" sig))
    (if (and (not (procedure? handler)) (not (number? handler)))
	(error 'SIGNAL "Argument is not a valid SIGNAL HANDLER: ~s" handler))
    (let ((prev-scheme (vector-ref signals sig))
	  (prev-unix (if (procedure? handler)
			 (unix-signal sig
			     (lambda (sig)
				     (let ((sig ((lap (x) (C_FIXED (INT x)))
						 sig)))
					  (sc_onsignal1 sig) #t)))
			 (unix-signal sig handler))))
	 (vector-set! signals sig handler)
	 (if (procedure? prev-scheme) prev-scheme prev-unix)))

;;; sc_onsignal1 calls the following procedure with the signal number to
;;; transfer control to the Scheme trap handler.

(define (ONSIGNAL2 sig) ((vector-ref signals sig) sig))
