(herald (assembler dkas t 10))

;;; Interface to the assembler for the David Kranz Orbit code generator

(lset *assembler-retains-pointers?* '#f)

(lset *current-assembly-section* 0)
(lset *current-ib* 0)
(lset *current-template-emitter* 0)

(define (assemble-init-1 c machine template-emitter)
  (let* ((s (cons-assembly-section machine))
         (ib (as-tag s (generate-symbol 'lap-entry))))
    (bind ((*current-template-emitter* template-emitter))
      (cond (*assembler-retains-pointers?*
             (set *current-assembly-section* s)
             (set *current-ib* ib)
             (c))
            (else
             (bind ((*current-assembly-section* s)
                    (*current-ib* ib))
               (c)))))))

(define (as-debug)
  (set *assembly-comments?* '#t)
  (set *assembler-retains-pointers?* '#t))

(define (as-undebug)
  (set *assembly-comments?* '#f)
  (set *current-assembly-section* 0)
  (set *current-ib* 0)
  (set *assembler-retains-pointers?* '#f))

(define (old-assemble)
  (new-assemble *current-assembly-section*)
  (print-section-statistics *current-assembly-section* *noise+terminal*)
  (bits-bv (assembly-section-bits *current-assembly-section*)))

(define assemble old-assemble)

(define (%emit opcode-fg . operands)
  (as-emit *current-ib* (apply opcode-fg operands))
  (flush-delayed-comments))

(define (%%emit fg)
  (as-emit *current-ib* fg)
  (flush-delayed-comments))

;;; ---------------- instruction emission

(define (emit-jump symbolic-jop 1tag 0tag)
  (let ((jop (xcond ((fixnum? symbolic-jop) symbolic-jop)
                    ((eq? symbolic-jop 'jneq) jump-op/jn=)
                    ((eq? symbolic-jop 'jgeq) jump-op/j>=)
                    ((eq? symbolic-jop 'jgtr) jump-op/j>)
                    ((eq? symbolic-jop 'jeql) jump-op/j=)
                    ((eq? symbolic-jop 'jmp)  jump-op/jabs)
                    )))
     (as-emit-jump *current-assembly-section* *current-ib* jop 1tag 0tag)
     (flush-delayed-comments)))

;;; Try to force the 1tag block to follow the current block

(define (emit-avoid-jump symbolic-jop 1tag 0tag)
  (emit-jump symbolic-jop 1tag 0tag)
  (maybe-set-ib-follower *current-ib*
                         (as-tag *current-assembly-section* 1tag))
  )

(define (emit-template code-node handler-node)
  (let ((cib (as-data-tag *current-assembly-section* code-node))
        (hib (as-data-tag *current-assembly-section* handler-node)))
    (let ((tib (as-data-tag *current-assembly-section* cib)))
      (*current-template-emitter* code-node cib hib tib)
      (set *current-ib* cib)
      (flush-delayed-comments)
      )))

(define (emit-tag code-node)
  (let ((ib (as-tag *current-assembly-section* code-node)))
    (set *current-ib* ib)
    (cond ((node? code-node)
           (set (ib-name ib)
                (xcond ((lambda-node? code-node)
                        (lambda-name code-node))
                       ((leaf-node? code-node)
                        (variable-unique-name code-node))))))))

(define (code-vector-offset label)
  (fixnum-ashr (label-offset *current-assembly-section* label) 3))

;;; ---------------- Lap

(define (lap-transduce items)
  (process-lap-list items
                    *current-assembly-section*
                    (as-tag *current-assembly-section*
                            (generate-symbol 'lap-entry))))

(define (no-lap-value exp)
  (*value orbit-env exp))

;;; ---------------- comment emission

(lset *delayed-comments* '())

;;; Comment for next thing emitted
(define (emit-comment the-comment)
   (push *delayed-comments* the-comment))

;;; Comment is tacked on to previously emitted instruction.
(define (comment-now the-comment)
   (as-comment *current-ib* the-comment))

(define-integrable (flush-delayed-comments)
  (cond ((not (null? *delayed-comments*))
         (as-comments *current-ib* *delayed-comments*)
         (set *delayed-comments* '()))))

;;; ---------------- Listing
;;; listings with hex codes

(define (quicklist)
  (as-list *current-assembly-section*))

;;; listings without hex codes

;(define (quicklist)
;  (print-listing (terminal-output) 
;                 (assembly-section-ib-vector *current-assembly-section*)
;                 0 
;                 '#f))

(define (qlist)
  (print-listing (terminal-output) 
                 (assembly-section-ib-vector *current-assembly-section*)
                 0 
                 '#f))

(define (flist filespec)
  (listing-to-file filespec *current-assembly-section*))

