/* 
 * sxSelect.c --
 *
 *	This file implements a standard way of dealing with the
 *	current window selection.  It has two major features:
 *	1. Selections come in different formats:  a selection consists
 *	   of a format (which is a character string) and an arbitrary
 *	   bunch of bytes.
 *	2. The selection is accessed by callback procedures.  Rather
 *	   than storing the selection in a common place, the module
 *	   that "owns" the selection provides a procedure to call
 *	   whenever the contents of the selection are needed.
 *
 * Copyright (C) 1986, 1988 Regents of the University of California.
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The University of California
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#ifndef lint
static char rcsid[] = "$Header: /sprite/src/lib/sx/RCS/sxSelect.c,v 1.6 92/06/04 17:00:57 jhh Exp $ SPRITE (Berkeley)";
#endif not lint

#include <X11/Xlib.h>
#include <strings.h>
#include <sys/time.h>
#include "sx.h"
#include "sxInt.h"
#include "tk.h"
#include "tkInt.h"
#include <X11/Xatom.h>

static int (*selFetchProc)() = NULL;	/* Procedure to call to fetch
					 * selection, or NULL. */
static void (*selChangeProc)();		/* Procedure to call when selection
					 * changes. */
static ClientData selClientData;	/* Argument to pass to procedures. */


static int initialized = 0;

/*
 * Internal procedures referenced before they're defined:
 */

static void	SelInit();
static int 	SelGetProc();
static int 	SelFetchProc();
static void	SelChangeProc();

TkWindow		tkWindow;
static Tk_Window	*tkwin = (Tk_Window *) &tkWindow;
static TkDisplay	tkDisplay;
static TkDisplay	*tkdisp = &tkDisplay;
TkDisplay		*tkDisplayList;
int	debug = 0;

typedef struct {
    char *string;		/* Contents of selection are
				 * here.  This space is malloc-ed. */
    int bytesAvl;		/* Total number of bytes available
				 * at string. */
    int bytesUsed;		/* Bytes currently in use in string,
				 * not including the terminating
				 * NULL. */
} GetInfo;

extern void		SxTk_CreateSelHandler _ANSI_ARGS_((Tk_Window tkwin,
			    Atom target, Tk_SelectionProc *proc,
			    ClientData clientData, Atom format));
extern void		SxTk_OwnSelection _ANSI_ARGS_((Tk_Window tkwin,
			    Tk_LostSelProc *proc, ClientData clientData));
extern int		SxTk_GetSelection _ANSI_ARGS_((Tk_Window tkwin, 
			    Atom target, Tk_GetSelProc *proc,
			    ClientData clientData));

extern 

/*
 *----------------------------------------------------------------------
 *
 * Sx_SelectionSet --
 *
 *	Change the selection.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The selection is changed.  All that happens is to save the
 *	name of the procedure to call when the selection is needed.
 *	FetchProc must be prepared to return part or all of the
 *	selection anytime it is called (up until the next time changeProc
 *	is called).  FetchProc has the following structure:
 *
 *	int
 *	fetchProc(clientData, desiredFormat, firstByte, numBytes,
 *	        valuePtr, formatPtr)
 *	    ClientData clientData;
 *	    char *desiredFormat;
 *	    int firstByte;
 *	    int numBytes;
 *	    char *valuePtr;
 *	    char *formatPtr;
 *	{
 *	}
 *
 *	In the calls to fetchProc, clientData will be a copy of
 *	the clientData parameter passed in here.  The other parameters,
 *	plus the return value, all have the same meanings as they do
 *	for Sx_SelectionGet.
 *
 *	ChangeProc is invoked in the following way:
 *
 *	void
 *	changeProc(clientData)
 *	    ClientData clientData;
 *	{
 *	}
 *
 *	The clientData parameter is identical to the clientData parameter
 *	passed in here.  ChangeProc will be invoked exactly once, the
 *	next time Sx_SelectionSet or Sx_SelectionClear is invoked.
 *
 *----------------------------------------------------------------------
 */

void
Sx_SelectionSet(display, fetchProc, changeProc, clientData)
    Display *display;		/* Connection to server. */
    int (*fetchProc)();		/* Procedure to invoke whenever the
				 * contents of the selection are needed. */
    void (*changeProc)();	/* Procedure to invoke the next time the
				 * selection is changed.  This procedure
				 * will be invoked exactly once. */
    ClientData clientData;	/* Arbitrary value to pass to above
				 * procedures. */
{
    if (!initialized) {
	SelInit(display);
    }
    if (debug) {
	printf("In Sx_SelectionSet\n");
    }
    if (selFetchProc != NULL) {
	(*selChangeProc)(selClientData);
    }
    SxTk_CreateSelHandler(tkwin, XA_STRING, (Tk_SelectionProc *) SelFetchProc, 
	clientData, XA_STRING);

    SxTk_OwnSelection(tkwin, (Tk_LostSelProc *) SelChangeProc, clientData);
    selFetchProc = fetchProc;
    selChangeProc = changeProc;
    selClientData = clientData;
}


/*
 *----------------------------------------------------------------------
 *
 * Sx_SelectionClear --
 *
 *	Cancel any current selection.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	From now until the next call to Sx_SelectionSet, there will
 *	be no selection.
 *
 *----------------------------------------------------------------------
 */

/*ARGSUSED*/
void
Sx_SelectionClear(display)
    Display *display;		/* Connection to server. */
{
    if (debug) {
	printf("Selection cleared\n");
    }
    if (selFetchProc != NULL) {
	(*selChangeProc)(selClientData);
    }
    selFetchProc = NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * Sx_SelectionGet --
 *
 *	Return part or all of the current selection.
 *
 * Results:
 *	If there is no selection at present, -1 is returned.  Otherwise,
 *	the return value indicates the number of bytes of selection
 *	information stored at *valuePtr (if it is less than numBytes,
 *	then the end of the selection was reached).  *ValuePtr gets
 *	filled in with bytes of the selection,  starting with the
 *	firstByte'th byte in the selection and containing either all
 *	the remaining bytes in the selection or numBytes bytes,
 *	whichever is less.  The selection will be returned in the
 *	closest possible form to what was requested with desiredFormat.
 *	The actual format in which the selection is returned will be
 *	placed at *formatPtr as an ASCII string containing not more
 *	than SX_FORMAT_SIZE characters (including the terminating 0).
 *
 * Side effects:
 *	None.
 *
 * Note:
 *	The formats of selections are completely up to the choice of the
 *	clients that manage them and will probably evolve over time.
 *	The one predefined format is "text":  in this case, the selection
 *	bytes consist of ASCII characters (not null-terminated).
 *
 *----------------------------------------------------------------------
 */

int
Sx_SelectionGet(display, desiredFormat, firstByte, numBytes, valuePtr,
    formatPtr)
    Display *display;		/* Connection to X server. */
    char *desiredFormat;	/* Form in which the selection should
				 * be returned, if possible. */
    int firstByte;		/* Index of the first byte of the selection
				 * that caller is interested in. */
    int numBytes;		/* Maximum number of bytes that may be
				 * stored at *valuePtr. */
    char *valuePtr;		/* Where to store selection information.
				 * Note: the information stored here
				 * will not necessarily be null-terminated.
				 * Depends on format of selection. */
    char *formatPtr;		/* Actual format of returned selection gets
				 * stored here, null-terminated.  Must be
				 * at least SX_FORMAT_SIZE bytes here. */
{
    GetInfo 	getInfo;
    int		result;

    if (debug) {
	printf("In Sx_SelectionGet\n");
    }
    if (!initialized) {
	SelInit(display);
    }
    if ((!strcmp(desiredFormat, "string")) && 
	(!strcmp(desiredFormat, "text"))) {
	    return -1;
    }
    getInfo.string = (char *) ckalloc(100);
    getInfo.bytesAvl = 100;
    getInfo.bytesUsed = 0;
    result = SxTk_GetSelection(tkwin, XA_STRING, SelGetProc, 
	(ClientData) &getInfo);
    if (result != TCL_OK) {
	numBytes = -1;
	goto done;
    }
    if (firstByte + numBytes > getInfo.bytesUsed) {
	numBytes = getInfo.bytesUsed - firstByte;
    }
    bcopy(&getInfo.string[firstByte], valuePtr, numBytes);
    strcpy(formatPtr, "text");
done:
    ckfree(getInfo.string);
    return numBytes;
}

/*
 *--------------------------------------------------------------
 *
 * SelGetProc --
 *
 *	This procedure is invoked to process pieces of the
 *	selection as they arrive during Tk_GetSelection.
 *
 * Results:
 *	Always returns TCL_OK.
 *
 * Side effects:
 *	Bytes are stored in getInfoPtr->string, which is 
 *	expanded if necessary.
 *
 *--------------------------------------------------------------
 */

	/* ARGSUSED */
static int
SelGetProc(clientData, portion)
    ClientData clientData;	/* Information about partially-
				 * assembled result. */
    char *portion;		/* New information to be appended. */
{
    register GetInfo *getInfoPtr = (GetInfo *) clientData;
    int newLength;

    if (debug) {
	printf("In SelGetProc\n");
    }
    newLength = strlen(portion) + getInfoPtr->bytesUsed;

    /*
     * Grow the result area if we've run out of space.
     */

    if (newLength >= getInfoPtr->bytesAvl) {
	char *newString;

	getInfoPtr->bytesAvl *= 2;
	if (getInfoPtr->bytesAvl <= newLength) {
	    getInfoPtr->bytesAvl = newLength + 1;
	}
	newString = (char *) ckalloc((unsigned) getInfoPtr->bytesAvl);
	memcpy((char *) newString, (char *) getInfoPtr->string,
		getInfoPtr->bytesUsed);
	ckfree(getInfoPtr->string);
	getInfoPtr->string = newString;
    }

    /*
     * Append the new data to what was already there.
     */

    strcpy(getInfoPtr->string + getInfoPtr->bytesUsed, portion);
    getInfoPtr->bytesUsed = newLength;
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * SelFetchProc --
 *
 *	This procedure is invoked when the selection is fetched
 *	and we are the owner. Converts between the fetching as
 *	done by Tk and Sx.
 *
 * Results:
 *	
 *
 * Side effects:
 *
 *--------------------------------------------------------------
 */

/* ARGSUSED */
static int
SelFetchProc(clientData, offset, buffer, maxBytes)
    ClientData 	clientData;
    int 	offset;
    char 	*buffer;
    int 	maxBytes;
{
    int		bytes;
    char	format[SX_FORMAT_SIZE];

    if (debug) {
	printf("In SelFetchProc\n");
    }
    if (selFetchProc == NULL) {
	if (debug) {
	    printf("Selection cleared\n");
	}
	return -1;
    }
    bytes = (*selFetchProc)(clientData, "text", offset, maxBytes, buffer,
		format);
	if (debug) {
	    printf("bytes = %d\n", bytes);
	}
    if (bytes < 0) {
	return bytes;
    }
    if (debug) {
	printf("format = %s, buffer = %s\n", format, buffer);
    }
    if (strcmp(format, "text")) {
	return -1;
    }
    return bytes;
}

/*
 *--------------------------------------------------------------
 *
 * SelChangeProc --
 *
 *	This procedure is invoked when the selection is changed.
 *	Converts between the Tk and Sx interfaces.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

/* ARGSUSED */
static void
SelChangeProc(clientData)
    ClientData 	clientData;
{
    if (debug) {
	printf("In SelChangeProc\n");
    }
    if (selFetchProc != NULL) {
	(*selChangeProc)(clientData);
    }
    selFetchProc = NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * SelInit --
 *
 *	Initialize information needed for selection.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Atoms and various other things get set up.
 *
 *----------------------------------------------------------------------
 */

static void
SelInit(display)
    Display *display;		/* Connection to server. */
{
    Window			window;

    initialized = 1;
    if (debug) {
	printf("In SelInit\n");
    }
    /* 
     * Create a window that will be the owner any time the selection is
     * held by this process.
     */
    window = XCreateSimpleWindow(display,
	    RootWindow(display, DefaultScreen(display)), 0, 0, 1, 1,
	    0, 0, 0);
    if (window == 0) {
	Sx_Panic(display,
		"Sx selection module couldn't create selection window.");
    }
    if (debug) {
	printf("Selection window is 0x%x\n", window);
    }
    /* 
     * Set up the tkwin and tkdisp data structures. This is a total hack.
     */
    bzero((char *) &tkWindow, sizeof(tkWindow));
    bzero((char *) &tkDisplay, sizeof(tkDisplay));
    tkWindow.display = display;
    tkWindow.dispPtr = tkdisp;
    tkWindow.dispPtr->display = display;
    tkWindow.window = window;
    tkdisp->nextPtr = NULL;
    tkDisplayList = tkdisp;
#ifdef NOTDEF
    XSynchronize(display, True);
#endif
}
