
/*
 * bltGrAxis.c --
 *
 *	This module implements coordinate axes for the BLT graph widget.
 *
 *	Copyright 1993-2004 George A Howlett.
 *
 *	Permission is hereby granted, free of charge, to any person
 *	obtaining a copy of this software and associated documentation
 *	files (the "Software"), to deal in the Software without
 *	restriction, including without limitation the rights to use,
 *	copy, modify, merge, publish, distribute, sublicense, and/or
 *	sell copies of the Software, and to permit persons to whom the
 *	Software is furnished to do so, subject to the following
 *	conditions:
 *
 *	The above copyright notice and this permission notice shall be
 *	included in all copies or substantial portions of the
 *	Software.
 *
 *	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
 *	KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 *	WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *	PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
 *	OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 *	OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 *	OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 *	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "bltGraph.h"
#include "bltGrElem.h"
#include <X11/Xutil.h>

#define DEF_NUM_TICKS	4	/* Each minor tick is 20% */


#define MAXTICKS	10001

#define FCLAMP(x)	((((x) < 0.0) ? 0.0 : ((x) > 1.0) ? 1.0 : (x)))

/*
 * Round x in terms of units
 */
#define UROUND(x,u)		(Round((x)/(u))*(u))
#define UCEIL(x,u)		(ceil((x)/(u))*(u))
#define UFLOOR(x,u)		(floor((x)/(u))*(u))

#define NUMDIGITS		15	/* Specifies the number of
					 * digits of accuracy used
					 * when outputting axis tick
					 * labels. */
enum TickRange {
    AXIS_TIGHT, AXIS_LOOSE, AXIS_ALWAYS_LOOSE
};

#define AXIS_PAD_TITLE		2	/* Padding for axis title. */

/* Axis flags: */

#define AXIS_ACTIVE       (1<<0)
#define AXIS_CONFIG_MAJOR (1<<4) /* User specified major tick intervals. */
#define AXIS_CONFIG_MINOR (1<<5) /* User specified minor tick intervals. */
#define AXIS_ONSCREEN	  (1<<6) /* Axis is displayed on the screen via
				  * the "use" operation */
#define AXIS_DIRTY	  (1<<7)
#define AXIS_DELETE_PENDING (1<<8) /* Indicates that the axis was
				    * scheduled for deletion. The
				    * actual deletion may be deferred
				    * until the axis is no longer in
				    * use.  */

#define HORIZMARGIN(m)	(!((m)->site & 0x1))	/* Even sites are horizontal */

typedef struct {
    int axis;		/* Length of the axis.  */
    int t1;		/* Length of a major tick (in pixels). */
    int t2;		/* Length of a minor tick (in pixels). */
    int label;		/* Distance from axis to tick label.  */
} AxisInfo;

static Blt_OptionParseProc ObjToLimit;
static Blt_OptionPrintProc LimitToObj;
static Blt_CustomOption limitOption = {
    ObjToLimit, LimitToObj, NULL, (ClientData)0
};

static Blt_OptionFreeProc  FreeTicks;
static Blt_OptionParseProc ObjToTicks;
static Blt_OptionPrintProc TicksToObj;
static Blt_CustomOption majorTicksOption = {
    ObjToTicks, TicksToObj, FreeTicks, (ClientData)AXIS_CONFIG_MAJOR,
};
static Blt_CustomOption minorTicksOption = {
    ObjToTicks, TicksToObj, FreeTicks, (ClientData)AXIS_CONFIG_MINOR,
};
static Blt_OptionFreeProc  FreeAxis;
static Blt_OptionPrintProc AxisToObj;
static Blt_OptionParseProc ObjToAxis;
Blt_CustomOption bltXAxisOption = {
    ObjToAxis, AxisToObj, FreeAxis, (ClientData)OBJECT_CLASS_X_AXIS
};
Blt_CustomOption bltYAxisOption = {
    ObjToAxis, AxisToObj, FreeAxis, (ClientData)OBJECT_CLASS_Y_AXIS
};

static Blt_OptionFreeProc  FreeFormat;
static Blt_OptionParseProc ObjToFormat;
static Blt_OptionPrintProc FormatToObj;
static Blt_CustomOption formatOption = {
    ObjToFormat, FormatToObj, FreeFormat, (ClientData)0,
};
static Blt_OptionParseProc ObjToLoose;
static Blt_OptionPrintProc LooseToObj;
static Blt_CustomOption looseOption = {
    ObjToLoose, LooseToObj, NULL, (ClientData)0,
};

#define DEF_AXIS_COMMAND		(char *)NULL
#define DEF_AXIS_DESCENDING		"no"
#define DEF_AXIS_FOREGROUND		RGB_BLACK
#define DEF_AXIS_HIDE			"no"
#define DEF_AXIS_JUSTIFY		"center"
#define DEF_AXIS_LIMITS_FORMAT	        (char *)NULL
#define DEF_AXIS_LINE_WIDTH		"1"
#define DEF_AXIS_LOGSCALE		"no"
#define DEF_AXIS_LOOSE			"no"
#define DEF_AXIS_RANGE			"0.0"
#define DEF_AXIS_ANGLE			"0.0"
#define DEF_AXIS_SCROLL_INCREMENT 	"10"
#define DEF_AXIS_SHIFTBY		"0.0"
#define DEF_AXIS_SHOWTICKS		"yes"
#define DEF_AXIS_STEP			"0.0"
#define DEF_AXIS_STEP			"0.0"
#define DEF_AXIS_SUBDIVISIONS		"2"
#define DEF_AXIS_TAGS			"all"
#define DEF_AXIS_TICKS			"0"
#ifdef WIN32
#define DEF_AXIS_TICK_FONT		"{Arial Narrow} 8"
#else
#define DEF_AXIS_TICK_FONT		"Courier 10"
#endif
#define DEF_AXIS_TICK_LENGTH		"8"
#define DEF_AXIS_TITLE_ALTERNATE	"0"
#define DEF_AXIS_TITLE_FG		RGB_BLACK
#define DEF_AXIS_TITLE_FONT		"Arial 11"
#define DEF_AXIS_X_STEP_BARCHART	"1.0"
#define DEF_AXIS_X_SUBDIVISIONS_BARCHART "0"
#define DEF_AXIS_BACKGROUND		(char *)NULL
#define DEF_AXIS_BORDERWIDTH		"0"
#define DEF_AXIS_RELIEF			"flat"
#define DEF_AXIS_ACTIVE_RELIEF		"flat"
#define DEF_AXIS_ACTIVE_BACKGROUND	STD_ACTIVE_BACKGROUND
#define DEF_AXIS_ACTIVE_FOREGROUND	STD_ACTIVE_FOREGROUND
#define DEF_AXIS_GRID_DASHES		"dot"
#define DEF_AXIS_GRID_FOREGROUND	RGB_GREY64
#define DEF_AXIS_GRID_LINE_WIDTH	"0"
#define DEF_AXIS_GRID_MINOR		"yes"
#define DEF_AXIS_GRIDLINES_GRAPH	"no"
#define DEF_AXIS_GRIDLINES_BARCHART	"yes"

static Blt_ConfigSpec configSpecs[] =
{
    {BLT_CONFIG_BACKGROUND, "-activebackground", "activeBackground", 
	"ActiveBackground", DEF_AXIS_ACTIVE_BACKGROUND, 
	Blt_Offset(Axis, activeBg), ALL_GRAPHS | BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_COLOR, "-activeforeground", "activeForeground",
	"ActiveForeground", DEF_AXIS_ACTIVE_FOREGROUND,
	Blt_Offset(Axis, activeFgColor), ALL_GRAPHS}, 
    {BLT_CONFIG_RELIEF, "-activerelief", "activeRelief", "Relief",
	DEF_AXIS_ACTIVE_RELIEF, Blt_Offset(Axis, activeRelief),
	ALL_GRAPHS | BLT_CONFIG_DONT_SET_DEFAULT}, 
    {BLT_CONFIG_DOUBLE, "-autorange", "autoRange", "AutoRange",
	DEF_AXIS_RANGE, Blt_Offset(Axis, windowSize),
        ALL_GRAPHS | BLT_CONFIG_DONT_SET_DEFAULT}, 
    {BLT_CONFIG_BACKGROUND, "-background", "background", "Background",
	DEF_AXIS_BACKGROUND, Blt_Offset(Axis, normalBg),
	ALL_GRAPHS | BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_SYNONYM, "-bg", "background", (char *)NULL, (char *)NULL, 0, 0},
    {BLT_CONFIG_LIST, "-bindtags", "bindTags", "BindTags",
	DEF_AXIS_TAGS, Blt_Offset(Axis, object.tags), 
	ALL_GRAPHS | BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_SYNONYM, "-bd", "borderWidth", (char *)NULL, 
        (char *)NULL, 0, ALL_GRAPHS},
    {BLT_CONFIG_PIXELS, "-borderwidth", "borderWidth", "BorderWidth",
	DEF_AXIS_BORDERWIDTH, Blt_Offset(Axis, borderWidth),
	ALL_GRAPHS | BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_COLOR, "-color", "color", "Color",
	DEF_AXIS_FOREGROUND, Blt_Offset(Axis, tickColor), ALL_GRAPHS},
    {BLT_CONFIG_STRING, "-command", "command", "Command",
	DEF_AXIS_COMMAND, Blt_Offset(Axis, formatCmd),
	BLT_CONFIG_NULL_OK | ALL_GRAPHS},
    {BLT_CONFIG_BOOLEAN, "-descending", "descending", "Descending",
	DEF_AXIS_DESCENDING, Blt_Offset(Axis, descending),
	ALL_GRAPHS | BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_SYNONYM, "-fg", "color", (char *)NULL, 
        (char *)NULL, 0, ALL_GRAPHS},
    {BLT_CONFIG_SYNONYM, "-foreground", "color", (char *)NULL, 
        (char *)NULL, 0, ALL_GRAPHS},
    {BLT_CONFIG_COLOR, "-gridcolor", "gridColor", "GridColor", 
	DEF_AXIS_GRID_FOREGROUND, Blt_Offset(Axis, gridColor), ALL_GRAPHS},
    {BLT_CONFIG_DASHES, "-griddashes", "gridDashes", "GridDashes", 
	DEF_AXIS_GRID_DASHES, Blt_Offset(Axis, gridDashes), 
	BLT_CONFIG_NULL_OK | ALL_GRAPHS},
    {BLT_CONFIG_BOOLEAN, "-gridlines", "girdLines", "GridLines",
	DEF_AXIS_GRIDLINES_BARCHART, Blt_Offset(Axis, showGrid), BARCHART},
    {BLT_CONFIG_BOOLEAN, "-gridlines", "girdLines", "GridLines",
	DEF_AXIS_GRIDLINES_GRAPH, Blt_Offset(Axis, showGrid), 
	GRAPH | STRIPCHART},
    {BLT_CONFIG_PIXELS, "-gridlinewidth", "gridLineWidth", "GridLineWidth",
	DEF_AXIS_GRID_LINE_WIDTH, Blt_Offset(Axis, gridLineWidth),
	BLT_CONFIG_DONT_SET_DEFAULT | ALL_GRAPHS},
    {BLT_CONFIG_BOOLEAN, "-gridminor", "gridMinor", "GridMinor", 
	DEF_AXIS_GRID_MINOR, Blt_Offset(Axis, gridMinor), 
	BLT_CONFIG_DONT_SET_DEFAULT | ALL_GRAPHS},
    {BLT_CONFIG_BOOLEAN, "-hide", "hide", "Hide",
	DEF_AXIS_HIDE, Blt_Offset(Axis, object.hidden),
	ALL_GRAPHS | BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_JUSTIFY, "-justify", "justify", "Justify",
	DEF_AXIS_JUSTIFY, Blt_Offset(Axis, titleJustify),
	ALL_GRAPHS | BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_BOOLEAN, "-labeloffset", "labelOffset", "LabelOffset",
        (char *)NULL, Blt_Offset(Axis, labelOffset), ALL_GRAPHS}, 
    {BLT_CONFIG_COLOR, "-limitscolor", "limitsColor", "Color",
	DEF_AXIS_FOREGROUND, Blt_Offset(Axis, limitsTextStyle.color), 
	ALL_GRAPHS},
    {BLT_CONFIG_FONT, "-limitsfont", "limitsFont", "Font",
	DEF_AXIS_TICK_FONT, Blt_Offset(Axis, limitsTextStyle.font), ALL_GRAPHS},
    {BLT_CONFIG_CUSTOM, "-limitsformat", "limitsFormat", "LimitsFormat",
        (char *)NULL, Blt_Offset(Axis, limitsFormats),
	BLT_CONFIG_NULL_OK | ALL_GRAPHS, &formatOption},
    {BLT_CONFIG_PIXELS, "-linewidth", "lineWidth", "LineWidth",
	DEF_AXIS_LINE_WIDTH, Blt_Offset(Axis, lineWidth),
	ALL_GRAPHS | BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_BOOLEAN, "-logscale", "logScale", "LogScale",
	DEF_AXIS_LOGSCALE, Blt_Offset(Axis, logScale),
	ALL_GRAPHS | BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_CUSTOM, "-loose", "loose", "Loose", DEF_AXIS_LOOSE, 0, 
	ALL_GRAPHS | BLT_CONFIG_DONT_SET_DEFAULT, &looseOption},
    {BLT_CONFIG_CUSTOM, "-majorticks", "majorTicks", "MajorTicks",
	(char *)NULL, Blt_Offset(Axis, t1Ptr),
	BLT_CONFIG_NULL_OK | ALL_GRAPHS, &majorTicksOption},
    {BLT_CONFIG_CUSTOM, "-max", "max", "Max", (char *)NULL, 
	Blt_Offset(Axis, reqMax), ALL_GRAPHS, &limitOption},
    {BLT_CONFIG_CUSTOM, "-min", "min", "Min", (char *)NULL, 
	Blt_Offset(Axis, reqMin), ALL_GRAPHS, &limitOption},
    {BLT_CONFIG_CUSTOM, "-minorticks", "minorTicks", "MinorTicks",
	(char *)NULL, Blt_Offset(Axis, t2Ptr), 
	BLT_CONFIG_NULL_OK | ALL_GRAPHS, &minorTicksOption},
    {BLT_CONFIG_RELIEF, "-relief", "relief", "Relief",
	DEF_AXIS_RELIEF, Blt_Offset(Axis, relief), 
        ALL_GRAPHS | BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_DOUBLE, "-rotate", "rotate", "Rotate",
	DEF_AXIS_ANGLE, Blt_Offset(Axis, tickAngle),
	ALL_GRAPHS | BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_STRING, "-scrollcommand", "scrollCommand", "ScrollCommand",
	(char *)NULL, Blt_Offset(Axis, scrollCmdPrefix),
	ALL_GRAPHS | BLT_CONFIG_NULL_OK},
    {BLT_CONFIG_PIXELS_POSITIVE, "-scrollincrement", "scrollIncrement", 
	"ScrollIncrement", DEF_AXIS_SCROLL_INCREMENT, 
	Blt_Offset(Axis, scrollUnits), ALL_GRAPHS|BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_CUSTOM, "-scrollmax", "scrollMax", "ScrollMax", (char *)NULL, 
	Blt_Offset(Axis, scrollMax),  ALL_GRAPHS, &limitOption},
    {BLT_CONFIG_CUSTOM, "-scrollmin", "scrollMin", "ScrollMin",
	(char *)NULL, Blt_Offset(Axis, scrollMin), ALL_GRAPHS, &limitOption},
    {BLT_CONFIG_DOUBLE, "-shiftby", "shiftBy", "ShiftBy",
	DEF_AXIS_SHIFTBY, Blt_Offset(Axis, shiftBy),
	ALL_GRAPHS | BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_BOOLEAN, "-showticks", "showTicks", "ShowTicks",
	DEF_AXIS_SHOWTICKS, Blt_Offset(Axis, showTicks),
	ALL_GRAPHS | BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_DOUBLE, "-stepsize", "stepSize", "StepSize",
	DEF_AXIS_STEP, Blt_Offset(Axis, reqStep),
	ALL_GRAPHS | BLT_CONFIG_DONT_SET_DEFAULT},
    {BLT_CONFIG_INT, "-subdivisions", "subdivisions", "Subdivisions",
	DEF_AXIS_SUBDIVISIONS, Blt_Offset(Axis, reqNumMinorTicks), ALL_GRAPHS},
    {BLT_CONFIG_FONT, "-tickfont", "tickFont", "Font",
	DEF_AXIS_TICK_FONT, Blt_Offset(Axis, tickFont), ALL_GRAPHS},
    {BLT_CONFIG_PIXELS_ANY, "-ticklength", "tickLength", "TickLength",
	DEF_AXIS_TICK_LENGTH, Blt_Offset(Axis, tickLength), ALL_GRAPHS},
    {BLT_CONFIG_STRING, "-title", "title", "Title",
	(char *)NULL, Blt_Offset(Axis, title),
	BLT_CONFIG_DONT_SET_DEFAULT | BLT_CONFIG_NULL_OK | ALL_GRAPHS},
    {BLT_CONFIG_BOOLEAN, "-titlealternate", "titleAlternate", "TitleAlternate",
	DEF_AXIS_TITLE_ALTERNATE, Blt_Offset(Axis, titleAlternate),
	BLT_CONFIG_DONT_SET_DEFAULT | ALL_GRAPHS},
    {BLT_CONFIG_COLOR, "-titlecolor", "titleColor", "Color", 
	DEF_AXIS_FOREGROUND, Blt_Offset(Axis, titleColor), 	
	ALL_GRAPHS},
    {BLT_CONFIG_FONT, "-titlefont", "titleFont", "Font", DEF_AXIS_TITLE_FONT, 
	Blt_Offset(Axis, titleFont), ALL_GRAPHS},
    {BLT_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};

/* Forward declarations */
static void DestroyAxis(Axis *axisPtr);
static int GetAxisByClass(Tcl_Interp *interp, Graph *graphPtr, Tcl_Obj *objPtr,
	int classId, Axis **axisPtrPtr);

static int lastMargin;

INLINE static int
Round(double x)
{
    return (int) (x + ((x < 0.0) ? -0.5 : 0.5));
}

static void
SetAxisRange(AxisRange *rangePtr, double min, double max)
{
    rangePtr->min = min;
    rangePtr->max = max;
    rangePtr->range = max - min;
    if (FABS(rangePtr->range) < DBL_EPSILON) {
	rangePtr->range = 1.0;
    }
    rangePtr->scale = 1.0 / rangePtr->range;
}

/*
 * ----------------------------------------------------------------------
 *
 * InRange --
 *
 *	Determines if a value lies within a given range.
 *
 *	The value is normalized and compared against the interval
 *	[0..1], where 0.0 is the minimum and 1.0 is the maximum.
 *	DBL_EPSILON is the smallest number that can be represented
 *	on the host machine, such that (1.0 + epsilon) != 1.0.
 *
 *	Please note, *max* can't equal *min*.
 *
 * Results:
 *	If the value is within the interval [min..max], 1 is 
 *	returned; 0 otherwise.
 *
 * ----------------------------------------------------------------------
 */
INLINE static int
InRange(double x, AxisRange *rangePtr)
{
    if (rangePtr->range < DBL_EPSILON) {
	return (FABS(rangePtr->max - x) >= DBL_EPSILON);
    } else {
	double norm;

	norm = (x - rangePtr->min) * rangePtr->scale;
	return ((norm >= -DBL_EPSILON) && ((norm - 1.0) < DBL_EPSILON));
    }
}

INLINE static int
AxisIsHorizontal(Graph *graphPtr, Axis *axisPtr)
{
    return ((axisPtr->object.classId == OBJECT_CLASS_Y_AXIS) == 
	    graphPtr->inverted);
}


static void
ReleaseAxis(Axis *axisPtr)
{
    if (axisPtr != NULL) {
	axisPtr->refCount--;
	assert(axisPtr->refCount >= 0);
	if (axisPtr->refCount == 0) {
	    DestroyAxis(axisPtr);
	}
    }
}

/* ----------------------------------------------------------------------
 * Custom option parse and print procedures
 * ----------------------------------------------------------------------
 */

/*ARGSUSED*/
static void
FreeAxis(
    ClientData clientData,
    Display *display,		/* Not used. */
    char *widgRec,
    int offset)
{
    Axis **axisPtrPtr = (Axis **)(widgRec + offset);

    if (*axisPtrPtr != NULL) {
	ReleaseAxis(*axisPtrPtr);
	*axisPtrPtr = NULL;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ObjToAxis --
 *
 *	Converts the name of an axis to a pointer to its axis structure.
 *
 * Results:
 *	The return value is a standard Tcl result.  The axis flags are
 *	written into the widget record.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ObjToAxis(
    ClientData clientData,	/* Class identifier of the type of 
				 * axis we are looking for. */
    Tcl_Interp *interp,		/* Interpreter to send results back to. */
    Tk_Window tkwin,		/* Used to look up pointer to graph. */
    Tcl_Obj *objPtr,		/* String representing new value. */
    char *widgRec,		/* Pointer to structure record. */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    int classId = (int)clientData;
    Axis **axisPtrPtr = (Axis **)(widgRec + offset);
    Axis *axisPtr;
    Graph *graphPtr;

    if (flags & BLT_CONFIG_NULL_OK) {
	char *string;

	string  = Tcl_GetString(objPtr);
	if (string[0] == '\0') {
	    ReleaseAxis(*axisPtrPtr);
	    *axisPtrPtr = NULL;
	    return TCL_OK;
	}
    }
    graphPtr = Blt_GetGraphFromWindowData(tkwin);
    assert(graphPtr);
    if (GetAxisByClass(interp, graphPtr, objPtr, classId, &axisPtr) 
	!= TCL_OK) {
	return TCL_ERROR;
    }
    ReleaseAxis(*axisPtrPtr);
    *axisPtrPtr = axisPtr;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * AxisToObj --
 *
 *	Convert the window coordinates into a string.
 *
 * Results:
 *	The string representing the coordinate position is returned.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static Tcl_Obj *
AxisToObj(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,
    Tk_Window tkwin,		/* Not used. */
    char *widgRec,		/* Pointer to structure record .*/
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Axis *axisPtr = *(Axis **)(widgRec + offset);

    return Tcl_NewStringObj((axisPtr == NULL) ? "" : axisPtr->object.name, -1);
}

/*ARGSUSED*/
static void
FreeFormat(
    ClientData clientData,
    Display *display,		/* Not used. */
    char *widgRec,
    int offset)
{
    Axis *axisPtr = (Axis *)(widgRec);

    if (axisPtr->limitsFormats != NULL) {
	Blt_Free(axisPtr->limitsFormats);
	axisPtr->limitsFormats = NULL;
    }
    axisPtr->nFormats = 0;
}

/*
 *----------------------------------------------------------------------
 *
 * ObjToFormat --
 *
 *	Convert the name of virtual axis to an pointer.
 *
 * Results:
 *	The return value is a standard Tcl result.  The axis flags are
 *	written into the widget record.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ObjToFormat(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to. */
    Tk_Window tkwin,		/* Used to look up pointer to graph */
    Tcl_Obj *objPtr,		/* String representing new value. */
    char *widgRec,		/* Pointer to structure record. */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Axis *axisPtr = (Axis *)(widgRec);
    char **argv;
    int argc;

    if (Tcl_SplitList(interp, Tcl_GetString(objPtr), &argc, &argv) != TCL_OK) {
	return TCL_ERROR;
    }
    if (argc > 2) {
	Tcl_AppendResult(interp, "too many elements in limits format list \"",
		Tcl_GetString(objPtr), "\"", (char *)NULL);
	Blt_Free(argv);
	return TCL_ERROR;
    }
    if (axisPtr->limitsFormats != NULL) {
	Blt_Free(axisPtr->limitsFormats);
    }
    axisPtr->limitsFormats = argv;
    axisPtr->nFormats = argc;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * FormatToObj --
 *
 *	Convert the window coordinates into a string.
 *
 * Results:
 *	The string representing the coordinate position is returned.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static Tcl_Obj *
FormatToObj(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,
    Tk_Window tkwin,		/* Not used. */
    char *widgRec,		/* Widget record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Axis *axisPtr = (Axis *)(widgRec);
    Tcl_Obj *objPtr;

    if (axisPtr->nFormats == 0) {
	objPtr = Tcl_NewStringObj("", -1);
    } else {
	char *string;

	string = Tcl_Merge(axisPtr->nFormats, axisPtr->limitsFormats); 
	objPtr = Tcl_NewStringObj(string, -1);
	Blt_Free(string);
    }
    return objPtr;
}

/*
 * ----------------------------------------------------------------------
 *
 * ObjToLimit --
 *
 *	Convert the string representation of an axis limit into its numeric
 *	form.
 *
 * Results:
 *	The return value is a standard Tcl result.  The symbol type is
 *	written into the widget record.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ObjToLimit(
    ClientData clientData,	/* Either AXIS_CONFIG_MIN or AXIS_CONFIG_MAX.
				 * Indicates which axis limit to set. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    Tk_Window tkwin,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representing new value. */
    char *widgRec,		/* Pointer to structure record. */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    double *limitPtr = (double *)(widgRec + offset);
    char *string;

    string = Tcl_GetString(objPtr);
    if (string[0] == '\0') {
	*limitPtr = VALUE_UNDEFINED;
    } else if (Blt_ExprDoubleFromObj(interp, objPtr, limitPtr) != TCL_OK) {
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * LimitToObj --
 *
 *	Convert the floating point axis limits into a string.
 *
 * Results:
 *	The string representation of the limits is returned.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static Tcl_Obj *
LimitToObj(
    ClientData clientData,	/* Either LMIN or LMAX */
    Tcl_Interp *interp,
    Tk_Window tkwin,		/* Not used. */
    char *widgRec,		/* */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    double limit = *(double *)(widgRec + offset);
    Tcl_Obj *objPtr;

    if (DEFINED(limit)) {
	objPtr = Tcl_NewDoubleObj(limit);
    } else {
	objPtr = Tcl_NewStringObj("", -1);
    }
    return objPtr;
}

/*ARGSUSED*/
static void
FreeTicks(
    ClientData clientData,
    Display *display,		/* Not used. */
    char *widgRec,
    int offset)
{
    Axis *axisPtr = (Axis *)widgRec;
    Ticks **ticksPtrPtr = (Ticks **) (widgRec + offset);
    unsigned long mask = (unsigned long)clientData;

    axisPtr->flags &= ~mask;
    if (*ticksPtrPtr != NULL) {
	Blt_Free(*ticksPtrPtr);
    }
    *ticksPtrPtr = NULL;
}

/*
 * ----------------------------------------------------------------------
 *
 * ObjToTicks --
 *
 *
 * Results:
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ObjToTicks(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    Tk_Window tkwin,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representing new value. */
    char *widgRec,		/* Pointer to structure record. */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Axis *axisPtr = (Axis *)widgRec;
    Tcl_Obj **objv;
    Ticks **ticksPtrPtr = (Ticks **) (widgRec + offset);
    Ticks *ticksPtr;
    int objc;
    unsigned long mask = (unsigned long)clientData;

    if (Tcl_ListObjGetElements(interp, objPtr, &objc, &objv) != TCL_OK) {
	return TCL_ERROR;
    }
    axisPtr->flags &= ~mask;
    ticksPtr = NULL;
    if (objc > 0) {
	int i;

	ticksPtr = Blt_Malloc(sizeof(Ticks) + (objc * sizeof(double)));
	assert(ticksPtr);
	for (i = 0; i < objc; i++) {
	    double value;

	    if (Blt_ExprDoubleFromObj(interp, objv[i], &value) != TCL_OK) {
		Blt_Free(ticksPtr);
		return TCL_ERROR;
	    }
	    ticksPtr->values[i] = value;
	}
    }
    ticksPtr->nTicks = objc;
    FreeTicks(clientData, Tk_Display(tkwin), widgRec, offset);
    axisPtr->flags |= mask;
    *ticksPtrPtr = ticksPtr;
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * TicksToObj --
 *
 *	Convert array of tick coordinates to a list.
 *
 * Results:
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static Tcl_Obj *
TicksToObj(
    ClientData clientData,	/* Holds a pointer to the graph. */
    Tcl_Interp *interp,
    Tk_Window tkwin,		/* Not used. */
    char *widgRec,		/* */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Ticks *ticksPtr = *(Ticks **) (widgRec + offset);
    Tcl_Obj *listObjPtr;

    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **)NULL);
    if (ticksPtr != NULL) {
	int i;

	for (i = 0; i < ticksPtr->nTicks; i++) {
	    Tcl_Obj *objPtr;

	    objPtr = Tcl_NewDoubleObj(ticksPtr->values[i]);
	    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
	}
    }
    return listObjPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * ObjToLoose --
 *
 *	Convert a string to one of three values.
 *		0 - false, no, off
 *		1 - true, yes, on
 *		2 - always
 * Results:
 *	If the string is successfully converted, TCL_OK is returned.
 *	Otherwise, TCL_ERROR is returned and an error message is left in
 *	interpreter's result field.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ObjToLoose(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,		/* Interpreter to send results back to */
    Tk_Window tkwin,		/* Not used. */
    Tcl_Obj *objPtr,		/* String representing new value. */
    char *widgRec,		/* Pointer to structure record. */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Axis *axisPtr = (Axis *)(widgRec);
    Tcl_Obj **objv;
    int i;
    int objc;
    int values[2];

    if (Tcl_ListObjGetElements(interp, objPtr, &objc, &objv) != TCL_OK) {
	return TCL_ERROR;
    }
    if ((objc < 1) || (objc > 2)) {
	Tcl_AppendResult(interp, "wrong # elements in loose value \"",
	    Tcl_GetString(objPtr), "\"", (char *)NULL);
	return TCL_ERROR;
    }
    for (i = 0; i < objc; i++) {
	char *string;

	string = Tcl_GetString(objv[i]);
	if ((string[0] == 'a') && (strcmp(string, "always") == 0)) {
	    values[i] = AXIS_LOOSE;
	} else {
	    int bool;

	    if (Tcl_GetBooleanFromObj(interp, objv[i], &bool) != TCL_OK) {
		return TCL_ERROR;
	    }
	    values[i] = bool;
	}
    }
    axisPtr->looseMin = axisPtr->looseMax = values[0];
    if (objc > 1) {
	axisPtr->looseMax = values[1];
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * LooseToObj --
 *
 * Results:
 *	The string representation of the auto boolean is returned.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static Tcl_Obj *
LooseToObj(
    ClientData clientData,	/* Not used. */
    Tcl_Interp *interp,
    Tk_Window tkwin,		/* Not used. */
    char *widgRec,		/* Widget record */
    int offset,			/* Offset to field in structure */
    int flags)	
{
    Axis *axisPtr = (Axis *)widgRec;
    Tcl_Obj *listObjPtr;

    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **)NULL);
    if (axisPtr->looseMin == AXIS_TIGHT) {
	Tcl_ListObjAppendElement(interp, listObjPtr, Tcl_NewBooleanObj(FALSE));
    } else if (axisPtr->looseMin == AXIS_LOOSE) {
	Tcl_ListObjAppendElement(interp, listObjPtr, Tcl_NewBooleanObj(TRUE));
    } else if (axisPtr->looseMin == AXIS_ALWAYS_LOOSE) {
	Tcl_ListObjAppendElement(interp, listObjPtr, 
		Tcl_NewStringObj("always", 6));
    }
    if (axisPtr->looseMin != axisPtr->looseMax) {
	if (axisPtr->looseMax == AXIS_TIGHT) {
	    Tcl_ListObjAppendElement(interp, listObjPtr, 
		Tcl_NewBooleanObj(FALSE));
	} else if (axisPtr->looseMax == AXIS_LOOSE) {
	    Tcl_ListObjAppendElement(interp, listObjPtr, 
		Tcl_NewBooleanObj(TRUE));
	} else if (axisPtr->looseMax == AXIS_ALWAYS_LOOSE) {
	    Tcl_ListObjAppendElement(interp, listObjPtr, 
		Tcl_NewStringObj("always", 6));
	}
    }
    return listObjPtr;
}

static void
FreeLabels(Blt_Chain *chainPtr)
{
    Blt_ChainLink *lp;

    for (lp = Blt_ChainFirstLink(chainPtr); lp != NULL; 
	 lp = Blt_ChainNextLink(lp)) {
	TickLabel *labelPtr;

	labelPtr = Blt_ChainGetValue(lp);
	Blt_Free(labelPtr);
    }
    Blt_ChainReset(chainPtr);
}

/*
 * ----------------------------------------------------------------------
 *
 * MakeLabel --
 *
 *	Converts a floating point tick value to a string to be used as its
 *	label.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Returns a new label in the string character buffer.  The formatted
 *	tick label will be displayed on the graph.
 *
 * ---------------------------------------------------------------------- 
 */
static TickLabel *
MakeLabel(
    Graph *graphPtr,
    Axis *axisPtr,		/* Axis structure */
    double value)		/* Value to be convert to a decimal string */
{
#define TICK_LABEL_SIZE		200
    char string[TICK_LABEL_SIZE + 1];
    TickLabel *labelPtr;

    /* Generate a default tick label based upon the tick value.  */
    if (axisPtr->logScale) {
	sprintf(string, "1E%d", ROUND(value));
    } else {
	sprintf(string, "%.*g", NUMDIGITS, value);
    }

    if (axisPtr->formatCmd != NULL) {
	Tcl_Interp *interp = graphPtr->interp;
	Tk_Window tkwin = graphPtr->tkwin;

	/*
	 * A Tcl proc was designated to format tick labels. Append the path
	 * name of the widget and the default tick label as arguments when
	 * invoking it. Copy and save the new label from interp->result.
	 */
	Tcl_ResetResult(interp);
	if (Tcl_VarEval(interp, axisPtr->formatCmd, " ", Tk_PathName(tkwin),
		" ", string, (char *)NULL) != TCL_OK) {
	    Tcl_BackgroundError(interp);
	} else {
	    /* 
	     * The proc could return a string of any length, so arbitrarily 
	     * limit it to what will fit in the return string. 
	     */
	    strncpy(string, Tcl_GetStringResult(interp), TICK_LABEL_SIZE);
	    string[TICK_LABEL_SIZE] = '\0';
	    
	    Tcl_ResetResult(interp); /* Clear the interpreter's result. */
	}
    }
    labelPtr = Blt_Malloc(sizeof(TickLabel) + strlen(string));
    assert(labelPtr);
    strcpy(labelPtr->string, string);
    labelPtr->anchorPos.x = labelPtr->anchorPos.y = DBL_MAX;
    return labelPtr;
}

/*
 * ----------------------------------------------------------------------
 *
 * Blt_InvHMap --
 *
 *	Maps the given screen coordinate back to a graph coordinate.
 *	Called by the graph locater routine.
 *
 * Results:
 *	Returns the graph coordinate value at the given window
 *	y-coordinate.
 *
 * ----------------------------------------------------------------------
 */
double
Blt_InvHMap(Axis *axisPtr, double x)
{
    double value;

    x = (double)(x - axisPtr->screenMin) * axisPtr->screenScale;
    if (axisPtr->descending) {
	x = 1.0 - x;
    }
    value = (x * axisPtr->axisRange.range) + axisPtr->axisRange.min;
    if (axisPtr->logScale) {
	value = EXP10(value);
    }
    return value;
}

/*
 * ----------------------------------------------------------------------
 *
 * Blt_InvVMap --
 *
 *	Maps the given screen y-coordinate back to the graph
 *	coordinate value. Called by the graph locater routine.
 *
 * Results:
 *	Returns the graph coordinate value for the given screen
 *	coordinate.
 *
 * ----------------------------------------------------------------------
 */
double
Blt_InvVMap(Axis *axisPtr, double y) /* Screen coordinate */
{
    double value;

    y = (double)(y - axisPtr->screenMin) * axisPtr->screenScale;
    if (axisPtr->descending) {
	y = 1.0 - y;
    }
    value = ((1.0 - y) * axisPtr->axisRange.range) + axisPtr->axisRange.min;
    if (axisPtr->logScale) {
	value = EXP10(value);
    }
    return value;
}

/*
 * ----------------------------------------------------------------------
 *
 * Blt_HMap --
 *
 *	Map the given graph coordinate value to its axis, returning a window
 *	position.
 *
 * Results:
 *	Returns a double precision number representing the window coordinate
 *	position on the given axis.
 *
 * ----------------------------------------------------------------------
 */
double
Blt_HMap(Axis *axisPtr, double x)
{
    if ((axisPtr->logScale) && (x != 0.0)) {
	x = log10(FABS(x));
    }
    /* Map graph coordinate to normalized coordinates [0..1] */
    x = (x - axisPtr->axisRange.min) * axisPtr->axisRange.scale;
    if (axisPtr->descending) {
	x = 1.0 - x;
    }
    return (x * axisPtr->screenRange + axisPtr->screenMin);
}

/*
 * ----------------------------------------------------------------------
 *
 * Blt_VMap --
 *
 *	Map the given graph coordinate value to its axis, returning a window
 *	position.
 *
 * Results:
 *	Returns a double precision number representing the window coordinate
 *	position on the given axis.
 *
 * ----------------------------------------------------------------------
 */
double
Blt_VMap(Axis *axisPtr, double y)
{
    if ((axisPtr->logScale) && (y != 0.0)) {
	y = log10(FABS(y));
    }
    /* Map graph coordinate to normalized coordinates [0..1] */
    y = (y - axisPtr->axisRange.min) * axisPtr->axisRange.scale;
    if (axisPtr->descending) {
	y = 1.0 - y;
    }
    return ((1.0 - y) * axisPtr->screenRange + axisPtr->screenMin);
}

/*
 * ----------------------------------------------------------------------
 *
 * Blt_Map2D --
 *
 *	Maps the given graph x,y coordinate values to a window position.
 *
 * Results:
 *	Returns a XPoint structure containing the window coordinates of
 *	the given graph x,y coordinate.
 *
 * ----------------------------------------------------------------------
 */
Point2D
Blt_Map2D(
    Graph *graphPtr,
    double x, double y,		/* Graph x and y coordinates */
    Axis2D *axesPtr)		/* Specifies which axes to use */
{
    Point2D point;

    if (graphPtr->inverted) {
	point.x = Blt_HMap(axesPtr->y, y);
	point.y = Blt_VMap(axesPtr->x, x);
    } else {
	point.x = Blt_HMap(axesPtr->x, x);
	point.y = Blt_VMap(axesPtr->y, y);
    }
    return point;
}

/*
 * ----------------------------------------------------------------------
 *
 * Blt_InvMap2D --
 *
 *	Maps the given window x,y coordinates to graph values.
 *
 * Results:
 *	Returns a structure containing the graph coordinates of
 *	the given window x,y coordinate.
 *
 * ----------------------------------------------------------------------
 */
Point2D
Blt_InvMap2D(
    Graph *graphPtr,
    double x, double y,		/* Window x and y coordinates */
    Axis2D *axesPtr)		/* Specifies which axes to use */
{
    Point2D point;

    if (graphPtr->inverted) {
	point.x = Blt_InvVMap(axesPtr->x, y);
	point.y = Blt_InvHMap(axesPtr->y, x);
    } else {
	point.x = Blt_InvHMap(axesPtr->x, x);
	point.y = Blt_InvVMap(axesPtr->y, y);
    }
    return point;
}


static void
GetDataLimits(Axis *axisPtr, double min, double max)
{
    if (axisPtr->valueRange.min > min) {
	axisPtr->valueRange.min = min;
    }
    if (axisPtr->valueRange.max < max) {
	axisPtr->valueRange.max = max;
    }
}

static void
FixAxisRange(Axis *axisPtr)
{
    double min, max;
    /*
     * When auto-scaling, the axis limits are the bounds of the element
     * data.  If no data exists, set arbitrary limits (wrt to log/linear
     * scale).
     */
    min = axisPtr->valueRange.min;
    max = axisPtr->valueRange.max;

    if (min == DBL_MAX) {
	if (DEFINED(axisPtr->reqMin)) {
	    min = axisPtr->reqMin;
	} else {
	    min = (axisPtr->logScale) ? 0.001 : 0.0;
	}
    }
    if (max == -DBL_MAX) {
	if (DEFINED(axisPtr->reqMax)) {
	    max = axisPtr->reqMax;
	} else {
	    max = 1.0;
	}
    }
    if (min >= max) {
	double value;

	/*
	 * There is no range of data (i.e. min is not less than max), 
	 * so manufacture one.
	 */
	value = min;
	if (value == 0.0) {
	    min = -0.1, max = 0.1;
	} else {
	    double x;

	    x = FABS(value) * 0.1;
	    min = value - x, max = value + x;
	}
    }
    SetAxisRange(&axisPtr->valueRange, min, max);

    /*   
     * The axis limits are either the current data range or overridden
     * by the values selected by the user with the -min or -max
     * options.
     */
    axisPtr->min = min;
    axisPtr->max = max;
    if (DEFINED(axisPtr->reqMin)) {
	axisPtr->min = axisPtr->reqMin;
    }
    if (DEFINED(axisPtr->reqMax)) { 
	axisPtr->max = axisPtr->reqMax;
    }
    if (axisPtr->max < axisPtr->min) {

	/*   
	 * If the limits still don't make sense, it's because one
	 * limit configuration option (-min or -max) was set and the
	 * other default (based upon the data) is too small or large.
	 * Remedy this by making up a new min or max from the
	 * user-defined limit.
	 */

	if (!DEFINED(axisPtr->reqMin)) {
	    axisPtr->min = axisPtr->max - (FABS(axisPtr->max) * 0.1);
	}
	if (!DEFINED(axisPtr->reqMax)) {
	    axisPtr->max = axisPtr->min + (FABS(axisPtr->max) * 0.1);
	}
    }
    /* 
     * If a window size is defined, handle auto ranging by shifting
     * the axis limits. 
     */
    if ((axisPtr->windowSize > 0.0) && 
	(!DEFINED(axisPtr->reqMin)) && (!DEFINED(axisPtr->reqMax))) {
	if (axisPtr->shiftBy < 0.0) {
	    axisPtr->shiftBy = 0.0;
	}
	max = axisPtr->min + axisPtr->windowSize;
	if (axisPtr->max >= max) {
	    if (axisPtr->shiftBy > 0.0) {
		max = UCEIL(axisPtr->max, axisPtr->shiftBy);
	    }
	    axisPtr->min = max - axisPtr->windowSize;
	}
	axisPtr->max = max;
    }
    if ((axisPtr->max != axisPtr->prevMax) || 
	(axisPtr->min != axisPtr->prevMin)) {
	/* Indicate if the axis limits have changed */
	axisPtr->flags |= AXIS_DIRTY;
	/* and save the previous minimum and maximum values */
	axisPtr->prevMin = axisPtr->min;
	axisPtr->prevMax = axisPtr->max;
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * NiceNum --
 *
 *	Reference: Paul Heckbert, "Nice Numbers for Graph Labels",
 *		   Graphics Gems, pp 61-63.  
 *
 *	Finds a "nice" number approximately equal to x.
 *
 * ----------------------------------------------------------------------
 */
static double
NiceNum(
    double x,
    int round)			/* If non-zero, round. Otherwise take ceiling
				 * of value. */
{
    double expt;		/* Exponent of x */
    double frac;		/* Fractional part of x */
    double nice;		/* Nice, rounded fraction */

    expt = floor(log10(x));
    frac = x / EXP10(expt);	/* between 1 and 10 */
    if (round) {
	if (frac < 1.5) {
	    nice = 1.0;
	} else if (frac < 3.0) {
	    nice = 2.0;
	} else if (frac < 7.0) {
	    nice = 5.0;
	} else {
	    nice = 10.0;
	}
    } else {
	if (frac <= 1.0) {
	    nice = 1.0;
	} else if (frac <= 2.0) {
	    nice = 2.0;
	} else if (frac <= 5.0) {
	    nice = 5.0;
	} else {
	    nice = 10.0;
	}
    }
    return nice * EXP10(expt);
}

static Ticks *
GenerateTicks(TickSweep *sweepPtr)
{
    Ticks *ticksPtr;

    ticksPtr = Blt_Malloc(sizeof(Ticks) + (sweepPtr->nSteps * sizeof(double)));
    assert(ticksPtr);
    ticksPtr->nTicks = 0;

    if (sweepPtr->step == 0.0) { 
	int i;
	/* Precomputed log10 values [1..10] */
	static double logTable[] = {
	    0.0, 
	    0.301029995663981, 
	    0.477121254719662, 
	    0.602059991327962, 
	    0.698970004336019, 
	    0.778151250383644, 
	    0.845098040014257,
	    0.903089986991944, 
	    0.954242509439325, 
	    1.0
	};
	/* Hack: A zero step indicates to use log values. */
	for (i = 0; i < sweepPtr->nSteps; i++) {
	    ticksPtr->values[i] = logTable[i];
	}
    } else {
	double value;
	int i;
    
	value = sweepPtr->initial; /* Start from smallest axis tick */
	for (i = 0; i < sweepPtr->nSteps; i++) {
	    value = UROUND(value, sweepPtr->step);
	    ticksPtr->values[i] = value;
	    value += sweepPtr->step;
	}
    }
    ticksPtr->nTicks = sweepPtr->nSteps;
    return ticksPtr;
}

/*
 * ----------------------------------------------------------------------
 *
 * LogScaleAxis --
 *
 * 	Determine the range and units of a log scaled axis.
 *
 * 	Unless the axis limits are specified, the axis is scaled
 * 	automatically, where the smallest and largest major ticks encompass
 * 	the range of actual data values.  When an axis limit is specified,
 * 	that value represents the smallest(min)/largest(max) value in the
 * 	displayed range of values.
 *
 * 	Both manual and automatic scaling are affected by the step used.  By
 * 	default, the step is the largest power of ten to divide the range in
 * 	more than one piece.
 *
 *	Automatic scaling:
 *	Find the smallest number of units which contain the range of values.
 *	The minimum and maximum major tick values will be represent the
 *	range of values for the axis. This greatest number of major ticks
 *	possible is 10.
 *
 * 	Manual scaling:
 *   	Make the minimum and maximum data values the represent the range of
 *   	the values for the axis.  The minimum and maximum major ticks will be
 *   	inclusive of this range.  This provides the largest area for plotting
 *   	and the expected results when the axis min and max values have be set
 *   	by the user (.e.g zooming).  The maximum number of major ticks is 20.
 *
 *   	For log scale, there's the possibility that the minimum and
 *   	maximum data values are the same magnitude.  To represent the
 *   	points properly, at least one full decade should be shown.
 *   	However, if you zoom a log scale plot, the results should be
 *   	predictable. Therefore, in that case, show only minor ticks.
 *   	Lastly, there should be an appropriate way to handle numbers
 *   	<=0.
 *
 *          maxY
 *            |    units = magnitude (of least significant digit)
 *            |    high  = largest unit tick < max axis value
 *      high _|    low   = smallest unit tick > min axis value
 *            |
 *            |    range = high - low
 *            |    # ticks = greatest factor of range/units
 *           _|
 *        U   |
 *        n   |
 *        i   |
 *        t  _|
 *            |
 *            |
 *            |
 *       low _|
 *            |
 *            |_minX________________maxX__
 *            |   |       |      |       |
 *     minY  low                        high
 *           minY
 *
 *
 * 	numTicks = Number of ticks
 * 	min = Minimum value of axis
 * 	max = Maximum value of axis
 * 	range    = Range of values (max - min)
 *
 * 	If the number of decades is greater than ten, it is assumed
 *	that the full set of log-style ticks can't be drawn properly.
 *
 * Results:
 *	None
 *
 * ---------------------------------------------------------------------- */
static void
LogScaleAxis(Axis *axisPtr, double min, double max)
{
    double range;
    double tickMin, tickMax;
    double majorStep, minorStep;
    int nMajor, nMinor;

    nMajor = nMinor = 0;
    /* Suppress compiler warnings. */
    majorStep = minorStep = 0.0;
    tickMin = tickMax = VALUE_UNDEFINED;
    if (min < max) {
	min = (min != 0.0) ? log10(FABS(min)) : 0.0;
	max = (max != 0.0) ? log10(FABS(max)) : 1.0;

	tickMin = floor(min);
	tickMax = ceil(max);
	range = tickMax - tickMin;
	
	if (range > 10) {
	    /* There are too many decades to display a major tick at every
	     * decade.  Instead, treat the axis as a linear scale.  */
	    range = NiceNum(range, 0);
	    majorStep = NiceNum(range / DEF_NUM_TICKS, 1);
	    tickMin = UFLOOR(tickMin, majorStep);
	    tickMax = UCEIL(tickMax, majorStep);
	    nMajor = (int)((tickMax - tickMin) / majorStep) + 1;
	    minorStep = EXP10(floor(log10(majorStep)));
	    if (minorStep == majorStep) {
		nMinor = 4, minorStep = 0.2;
	    } else {
		nMinor = Round(majorStep / minorStep) - 1;
	    }
	} else {
	    if (tickMin == tickMax) {
		tickMax++;
	    }
	    majorStep = 1.0;
	    nMajor = (int)(tickMax - tickMin + 1); /* FIXME: Check this. */
	    
	    minorStep = 0.0;	/* This is a special hack to pass
				 * information to the GenerateTicks
				 * routine. An interval of 0.0 tells
				 *	1) this is a minor sweep and 
				 *	2) the axis is log scale.  
				 */
	    nMinor = 10;
	}
	if ((axisPtr->looseMin == AXIS_TIGHT) || 
	    ((axisPtr->looseMin == AXIS_LOOSE) && 
	     (DEFINED(axisPtr->reqMin)))) {
	    tickMin = min;
	    nMajor++;
	}
	if ((axisPtr->looseMax == AXIS_TIGHT) || 
	    ((axisPtr->looseMax == AXIS_LOOSE) &&
	     (DEFINED(axisPtr->reqMax)))) {
	    tickMax = max;
	}
    }
    axisPtr->majorSweep.step = majorStep;
    axisPtr->majorSweep.initial = floor(tickMin);
    axisPtr->majorSweep.nSteps = nMajor;
    axisPtr->minorSweep.initial = axisPtr->minorSweep.step = minorStep;
    axisPtr->minorSweep.nSteps = nMinor;

    SetAxisRange(&axisPtr->axisRange, tickMin, tickMax);
}

/*
 * ----------------------------------------------------------------------
 *
 * LinearScaleAxis --
 *
 * 	Determine the units of a linear scaled axis.
 *
 *	The axis limits are either the range of the data values mapped
 *	to the axis (autoscaled), or the values specified by the -min
 *	and -max options (manual).
 *
 *	If autoscaled, the smallest and largest major ticks will
 *	encompass the range of data values.  If the -loose option is
 *	selected, the next outer ticks are choosen.  If tight, the
 *	ticks are at or inside of the data limits are used.
 *
 * 	If manually set, the ticks are at or inside the data limits
 * 	are used.  This makes sense for zooming.  You want the
 * 	selected range to represent the next limit, not something a
 * 	bit bigger.
 *
 *	Note: I added an "always" value to the -loose option to force
 *	      the manually selected axes to be loose. It's probably
 *	      not a good idea.
 *
 *          maxY
 *            |    units = magnitude (of least significant digit)
 *            |    high  = largest unit tick < max axis value
 *      high _|    low   = smallest unit tick > min axis value
 *            |
 *            |    range = high - low
 *            |    # ticks = greatest factor of range/units
 *           _|
 *        U   |
 *        n   |
 *        i   |
 *        t  _|
 *            |
 *            |
 *            |
 *       low _|
 *            |
 *            |_minX________________maxX__
 *            |   |       |      |       |
 *     minY  low                        high
 *           minY
 *
 * 	numTicks = Number of ticks
 * 	min = Minimum value of axis
 * 	max = Maximum value of axis
 * 	range    = Range of values (max - min)
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	The axis tick information is set.  The actual tick values will
 *	be generated later.
 *
 * ----------------------------------------------------------------------
 */
static void
LinearScaleAxis(Axis *axisPtr, double min, double max)
{
    double step;
    double tickMin, tickMax;
    double axisMin, axisMax;
    int nTicks;

    nTicks = 0;
    step = 1.0;
    /* Suppress compiler warning. */
    axisMin = axisMax = tickMin = tickMax = VALUE_UNDEFINED;
    if (min < max) {
	double range;

	range = max - min;
	/* Calculate the major tick stepping. */
	if (axisPtr->reqStep > 0.0) {
	    /* An interval was designated by the user.  Keep scaling
	     * it until it fits comfortably within the current range
	     * of the axis.  */
	    step = axisPtr->reqStep;
	    while ((2 * step) >= range) {
		step *= 0.5;
	    }
	} else {
	    range = NiceNum(range, 0);
	    step = NiceNum(range / DEF_NUM_TICKS, 1);
	}
	
	/* Find the outer tick values. Add 0.0 to prevent getting -0.0. */
	axisMin = tickMin = floor(min / step) * step + 0.0;
	axisMax = tickMax = ceil(max / step) * step + 0.0;
	
	nTicks = Round((tickMax - tickMin) / step) + 1;
    } 
    axisPtr->majorSweep.step = step;
    axisPtr->majorSweep.initial = tickMin;
    axisPtr->majorSweep.nSteps = nTicks;

    /*
     * The limits of the axis are either the range of the data
     * ("tight") or at the next outer tick interval ("loose").  The
     * looseness or tightness has to do with how the axis fits the
     * range of data values.  This option is overridden when the user
     * sets an axis limit (by either -min or -max option).  The axis
     * limit is always at the selected limit (otherwise we assume that
     * user would have picked a different number).
     */
    if ((axisPtr->looseMin == AXIS_TIGHT) || 
	((axisPtr->looseMin == AXIS_LOOSE) &&
	 (DEFINED(axisPtr->reqMin)))) {
	axisMin = min;
    }
    if ((axisPtr->looseMax == AXIS_TIGHT) || 
	((axisPtr->looseMax == AXIS_LOOSE) &&
	 (DEFINED(axisPtr->reqMax)))) {
	axisMax = max;
    }
    SetAxisRange(&axisPtr->axisRange, axisMin, axisMax);

    /* Now calculate the minor tick step and number. */

    if ((axisPtr->reqNumMinorTicks > 0) && 
	((axisPtr->flags & AXIS_CONFIG_MAJOR) == 0)) {
	nTicks = axisPtr->reqNumMinorTicks - 1;
	step = 1.0 / (nTicks + 1);
    } else {
	nTicks = 0;		/* No minor ticks. */
	step = 0.5;		/* Don't set the minor tick interval
				 * to 0.0. It makes the GenerateTicks
				 * routine create minor log-scale tick
				 * marks.  */
    }
    axisPtr->minorSweep.initial = axisPtr->minorSweep.step = step;
    axisPtr->minorSweep.nSteps = nTicks;
}


static void
SweepTicks(Axis *axisPtr)
{
    if ((axisPtr->flags & AXIS_CONFIG_MAJOR) == 0) {
	if (axisPtr->t1Ptr != NULL) {
	    Blt_Free(axisPtr->t1Ptr);
	}
	axisPtr->t1Ptr = GenerateTicks(&axisPtr->majorSweep);
    }
    if ((axisPtr->flags & AXIS_CONFIG_MINOR) == 0) {
	if (axisPtr->t2Ptr != NULL) {
	    Blt_Free(axisPtr->t2Ptr);
	}
	axisPtr->t2Ptr = GenerateTicks(&axisPtr->minorSweep);
    }
}

/*
 * ----------------------------------------------------------------------
 *
 * Blt_ResetAxes --
 *
 * Results:
 *	None.
 *
 * ----------------------------------------------------------------------
 */
void
Blt_ResetAxes(Graph *graphPtr)
{
    Blt_ChainLink *lp;
    Blt_HashEntry *hp;
    Blt_HashSearch cursor;

    /* FIXME: This should be called whenever the display list of
     *	      elements change. Maybe yet another flag INIT_STACKS to
     *	      indicate that the element display list has changed.
     *	      Needs to be done before the axis limits are set.
     */
    Blt_InitFreqTable(graphPtr);
    if ((graphPtr->mode == MODE_STACKED) && (graphPtr->nStacks > 0)) {
	Blt_ComputeStacks(graphPtr);
    }
    /*
     * Step 1:  Reset all axes. Initialize the data limits of the axis to
     *		impossible values.
     */
    for (hp = Blt_FirstHashEntry(&graphPtr->axes.table, &cursor);
	hp != NULL; hp = Blt_NextHashEntry(&cursor)) {
	Axis *axisPtr;

	axisPtr = (Axis *)Blt_GetHashValue(hp);
	axisPtr->min = axisPtr->valueRange.min = DBL_MAX;
	axisPtr->max = axisPtr->valueRange.max = -DBL_MAX;
    }

    /*
     * Step 2:  For each element that's to be displayed, get the smallest
     *		and largest data values mapped to each X and Y-axis.  This
     *		will be the axis limits if the user doesn't override them 
     *		with -min and -max options.
     */
    for (lp = Blt_ChainFirstLink(graphPtr->elements.displayList);
	 lp != NULL; lp = Blt_ChainNextLink(lp)) {
	Element *elemPtr;
	Region2D exts;

	elemPtr = Blt_ChainGetValue(lp);
	(*elemPtr->procsPtr->extentsProc) (elemPtr, &exts);
	GetDataLimits(elemPtr->axes.x, exts.left, exts.right);
	GetDataLimits(elemPtr->axes.y, exts.top, exts.bottom);
    }
    /*
     * Step 3:  Now that we know the range of data values for each axis,
     *		set axis limits and compute a sweep to generate tick values.
     */
    for (hp = Blt_FirstHashEntry(&graphPtr->axes.table, &cursor);
	 hp != NULL; hp = Blt_NextHashEntry(&cursor)) {
	Axis *axisPtr;
	double min, max;

	axisPtr = (Axis *)Blt_GetHashValue(hp);
	FixAxisRange(axisPtr);

	/* Calculate min/max tick (major/minor) layouts */
	min = axisPtr->min;
	max = axisPtr->max;
	if ((DEFINED(axisPtr->scrollMin)) && (min < axisPtr->scrollMin)) {
	    min = axisPtr->scrollMin;
	}
	if ((DEFINED(axisPtr->scrollMax)) && (max > axisPtr->scrollMax)) {
	    max = axisPtr->scrollMax;
	}
	if (axisPtr->logScale) {
	    LogScaleAxis(axisPtr, min, max);
	} else {
	    LinearScaleAxis(axisPtr, min, max);
	}

	if ((axisPtr->flags & (AXIS_DIRTY | AXIS_ONSCREEN)) ==
	    (AXIS_DIRTY | AXIS_ONSCREEN)) {
	    graphPtr->flags |= REDRAW_BACKING_STORE;
	}
    }

    graphPtr->flags &= ~RESET_AXES;

    /*
     * When any axis changes, we need to layout the entire graph.
     */
    graphPtr->flags |= (GET_AXIS_GEOMETRY | LAYOUT_NEEDED | MAP_ALL | 
			REDRAW_WORLD);
}

/*
 * ----------------------------------------------------------------------
 *
 * ResetTextStyles --
 *
 *	Configures axis attributes (font, line width, label, etc) and
 *	allocates a new (possibly shared) graphics context.  Line cap
 *	style is projecting.  This is for the problem of when a tick
 *	sits directly at the end point of the axis.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 * Side Effects:
 *	Axis resources are allocated (GC, font). Axis layout is
 *	deferred until the height and width of the window are known.
 *
 * ----------------------------------------------------------------------
 */
static void
ResetTextStyles(Graph *graphPtr, Axis *axisPtr)
{
    GC newGC;
    XGCValues gcValues;
    unsigned long gcMask;

    Blt_TextResetStyle(graphPtr->tkwin, &axisPtr->limitsTextStyle);

    gcMask = (GCForeground | GCLineWidth | GCCapStyle);
    gcValues.foreground = axisPtr->tickColor->pixel;
    gcValues.font = Blt_FontId(axisPtr->tickFont);
    gcValues.line_width = LineWidth(axisPtr->lineWidth);
    gcValues.cap_style = CapProjecting;

    newGC = Tk_GetGC(graphPtr->tkwin, gcMask, &gcValues);
    if (axisPtr->tickGC != NULL) {
	Tk_FreeGC(graphPtr->display, axisPtr->tickGC);
    }
    axisPtr->tickGC = newGC;

    /* Assuming settings from above GC */
    gcValues.foreground = axisPtr->activeFgColor->pixel;
    newGC = Tk_GetGC(graphPtr->tkwin, gcMask, &gcValues);
    if (axisPtr->activeTickGC != NULL) {
	Tk_FreeGC(graphPtr->display, axisPtr->activeTickGC);
    }
    axisPtr->activeTickGC = newGC;

    gcValues.background = gcValues.foreground = axisPtr->gridColor->pixel;
    gcValues.line_width = LineWidth(axisPtr->gridLineWidth);
    gcMask = (GCForeground | GCBackground | GCLineWidth);
    if (LineIsDashed(axisPtr->gridDashes)) {
	gcValues.line_style = LineOnOffDash;
	gcMask |= GCLineStyle;
    }
    newGC = Blt_GetPrivateGC(graphPtr->tkwin, gcMask, &gcValues);
    if (LineIsDashed(axisPtr->gridDashes)) {
	Blt_SetDashes(graphPtr->display, newGC, &axisPtr->gridDashes);
    }
    if (axisPtr->gridGC != NULL) {
	Blt_FreePrivateGC(graphPtr->display, axisPtr->gridGC);
    }
    axisPtr->gridGC = newGC;
}

/*
 * ----------------------------------------------------------------------
 *
 * DestroyAxis --
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Resources (font, color, gc, labels, etc.) associated with the
 *	axis are deallocated.
 *
 * ----------------------------------------------------------------------
 */
static void
DestroyAxis(Axis *axisPtr)
{
    int flags;
    Graph *graphPtr = axisPtr->object.graphPtr;

    flags = Blt_GraphType(graphPtr);
    Blt_FreeOptions(configSpecs, (char *)axisPtr, graphPtr->display, flags);
    if (graphPtr->bindTable != NULL) {
	Blt_DeleteBindings(graphPtr->bindTable, axisPtr);
    }
    if (axisPtr->linkPtr != NULL) {
	Blt_ChainDeleteLink(axisPtr->chainPtr, axisPtr->linkPtr);
    }
    if (axisPtr->object.name != NULL) {
	Blt_Free(axisPtr->object.name);
    }
    if (axisPtr->hashPtr != NULL) {
	Blt_DeleteHashEntry(&graphPtr->axes.table, axisPtr->hashPtr);
    }
    Blt_TextFreeStyle(graphPtr->display, &axisPtr->limitsTextStyle);

    if (axisPtr->tickGC != NULL) {
	Tk_FreeGC(graphPtr->display, axisPtr->tickGC);
    }
    if (axisPtr->activeTickGC != NULL) {
	Tk_FreeGC(graphPtr->display, axisPtr->activeTickGC);
    }
#ifdef notdef
    if (axisPtr->t1Ptr != NULL) {
	Blt_Free(axisPtr->t1Ptr);
    }
    if (axisPtr->t2Ptr != NULL) {
	Blt_Free(axisPtr->t2Ptr);
    }
    if (axisPtr->limitsFormats != NULL) {
	Blt_Free(axisPtr->limitsFormats);
    }
    if (axisPtr->object.tags != NULL) {
	Blt_Free(axisPtr->object.tags);
    }
#endif
    FreeLabels(axisPtr->tickLabels);
    Blt_ChainDestroy(axisPtr->tickLabels);
    if (axisPtr->segments != NULL) {
	Blt_Free(axisPtr->segments);
    }
    Blt_Free(axisPtr);
}

static double titleAngle[4] =	/* Rotation for each axis title */
{
    0.0, 90.0, 0.0, 270.0
};

/*
 * ----------------------------------------------------------------------
 *
 * AxisOffsets --
 *
 *	Determines the sites of the axis, major and minor ticks,
 *	and title of the axis.
 *
 * Results:
 *	None.
 *
 * ----------------------------------------------------------------------
 */
static void
AxisOffsets(
    Graph *graphPtr,
    Axis *axisPtr,
    int margin,
    int offset,
    AxisInfo *infoPtr)
{
    Margin *marginPtr;
    int pad;			/* Offset of axis from interior region. This
				 * includes a possible border and the axis
				 * line width. */
    int axisline;
    int t1, t2, labelOffset;
    int inset, mark;
    int x, y;

    axisPtr->titleAngle = titleAngle[margin];
    marginPtr = graphPtr->margins + margin;

    axisline = t1 = t2 = 0;
    labelOffset = AXIS_PAD_TITLE;
    if (axisPtr->lineWidth > 0) {
	if (axisPtr->showTicks) {
	    t1 = ABS(axisPtr->tickLength);
	    t2 = (t1 * 10) / 15;
	}
	labelOffset = t1 + AXIS_PAD_TITLE + axisPtr->lineWidth;
    }
    /* Adjust offset for the interior border width and the line width */

    pad = 1;
    if (graphPtr->plotBorderWidth > 0) {
	pad += graphPtr->plotBorderWidth + 1;
    }

    /*
     * Pre-calculate the x-coordinate positions of the axis, tick
     * labels, and the individual major and minor ticks.
     */
    inset = pad + axisPtr->lineWidth / 2;
    switch (margin) {
    case MARGIN_TOP:
	mark = graphPtr->top - offset - pad;
	axisline = mark - axisPtr->lineWidth / 2 - 1;
	axisPtr->tickAnchor = TK_ANCHOR_S;
	axisPtr->left = axisPtr->screenMin - inset - 2;
	axisPtr->right = axisPtr->screenMin + axisPtr->screenRange + inset - 1;
	if (graphPtr->stackAxes) {
	    axisPtr->top = mark - marginPtr->axesOffset;
	} else {
	    axisPtr->top = mark - axisPtr->height;
	}
	axisPtr->bottom = mark;
	if (axisPtr->titleAlternate) {
	    x = graphPtr->right + AXIS_PAD_TITLE;
	    y = mark - (axisPtr->height  / 2);
	    axisPtr->titleAnchor = TK_ANCHOR_W;
	} else {
	    x = (axisPtr->right + axisPtr->left) / 2;
	    if (graphPtr->stackAxes) {
		y = mark - marginPtr->axesOffset + AXIS_PAD_TITLE;
	    } else {
		y = mark - axisPtr->height + AXIS_PAD_TITLE;
	    }
	    axisPtr->titleAnchor = TK_ANCHOR_N;
	}
	axisPtr->titlePos.x = x;
	axisPtr->titlePos.y = y;
	break;

    case MARGIN_BOTTOM:
	/*
	 *   ---------------------------------- bottom + plot borderwidth
	 *      mark --------------------
	 *          ===================== axisline (linewidth)
	 *                   tick
	 *		    title
	 */
	mark = graphPtr->bottom + pad + offset;
	axisline = mark + axisPtr->lineWidth / 2 + 1;
	axisPtr->tickAnchor = TK_ANCHOR_N;
	axisPtr->left = axisPtr->screenMin - inset - 2;
	axisPtr->right = axisPtr->screenMin + axisPtr->screenRange + inset - 1;
	axisPtr->top = mark;
	if (graphPtr->stackAxes) {
	    axisPtr->bottom = mark + marginPtr->axesOffset - 1;
	} else {
	    axisPtr->bottom = mark + axisPtr->height - 1;
	}
	if (axisPtr->titleAlternate) {
	    x = graphPtr->right + AXIS_PAD_TITLE;
	    y = mark + (axisPtr->height / 2);
	    axisPtr->titleAnchor = TK_ANCHOR_W; 
	} else {
	    x = (axisPtr->right + axisPtr->left) / 2;
	    if (graphPtr->stackAxes) {
		y = mark + marginPtr->axesOffset - AXIS_PAD_TITLE;
	    } else {
		y = mark + axisPtr->height - AXIS_PAD_TITLE;
	    }
	    axisPtr->titleAnchor = TK_ANCHOR_S; 
	}
	axisPtr->titlePos.x = x;
	axisPtr->titlePos.y = y;
	break;

    case MARGIN_LEFT:
	/*
	 *                    mark
	 *                  |  : 
	 *                  |  :      
	 *                  |  : 
	 *                  |  :
	 *                  |  : 
	 *     axisline
	 */
	mark = graphPtr->left - pad - offset;
	axisline = mark - axisPtr->lineWidth - 1;
	axisPtr->tickAnchor = TK_ANCHOR_E;
	if (graphPtr->stackAxes) {
	    axisPtr->left = mark - marginPtr->axesOffset;
	} else {
	    axisPtr->left = mark - axisPtr->width;
	}
	axisPtr->right = mark;
	axisPtr->top = axisPtr->screenMin - inset - 2;
	axisPtr->bottom = axisPtr->screenMin + axisPtr->screenRange + inset - 1;
	if (axisPtr->titleAlternate) {
	    x = mark - (axisPtr->width / 2);
	    y = graphPtr->top - AXIS_PAD_TITLE;
	    axisPtr->titleAnchor = TK_ANCHOR_SW; 
	} else {
	    if (graphPtr->stackAxes) {
		x = mark - marginPtr->axesOffset;
	    } else {
		x = mark - axisPtr->width + AXIS_PAD_TITLE;
	    }
	    y = (axisPtr->bottom + axisPtr->top) / 2;
	    axisPtr->titleAnchor = TK_ANCHOR_W; 
	} 
	axisPtr->titlePos.x = x;
	axisPtr->titlePos.y = y;
	break;

    case MARGIN_RIGHT:
	mark = graphPtr->right + offset + pad;
	axisline = mark + axisPtr->lineWidth / 2 + 1;
	axisPtr->tickAnchor = TK_ANCHOR_W;
	axisPtr->left = mark;
	if (graphPtr->stackAxes) {
	    axisPtr->right = mark + marginPtr->axesOffset - 1;
	} else {
	    axisPtr->right = mark + axisPtr->width - 1;
	}
	axisPtr->top = axisPtr->screenMin - inset - 2;
	axisPtr->bottom = axisPtr->screenMin + axisPtr->screenRange + inset -1;
	if (axisPtr->titleAlternate) {
	    x = mark + (axisPtr->width / 2);
	    y = graphPtr->top - AXIS_PAD_TITLE;
	    axisPtr->titleAnchor = TK_ANCHOR_SE; 
	} else {
	    if (graphPtr->stackAxes) {
		x = mark + marginPtr->axesOffset - AXIS_PAD_TITLE;
	    } else {
		x = mark + axisPtr->width - AXIS_PAD_TITLE;
	    }
	    y = (axisPtr->bottom + axisPtr->top) / 2;
	    axisPtr->titleAnchor = TK_ANCHOR_E;
	}
	axisPtr->titlePos.x = x;
	axisPtr->titlePos.y = y;
	break;

    case MARGIN_NONE:
	axisline = 0;
	break;
    }
    if ((margin == MARGIN_LEFT) || (margin == MARGIN_TOP)) {
	t1 = -t1, t2 = -t2;
	labelOffset = -labelOffset;
    }
    infoPtr->axis = axisline;
    infoPtr->t1 = axisline + t1;
    infoPtr->t2 = axisline + t2;
    infoPtr->label = axisline + labelOffset;

    if (axisPtr->tickLength < 0) {
	int hold;
	
	/* Ticks to run in the opposite direction. Swap the axisline
	 * and t1 values */
	hold = infoPtr->t1;
	infoPtr->t1 = infoPtr->axis;
	infoPtr->axis = hold;
    }
}

static void
MakeAxisLine(
    Graph *graphPtr,
    Axis *axisPtr,		/* Axis information */
    int line,
    Segment2D *sp)
{
    double min, max;

    min = axisPtr->axisRange.min;
    max = axisPtr->axisRange.max;
    if (axisPtr->logScale) {
	min = EXP10(min);
	max = EXP10(max);
    }
    if (AxisIsHorizontal(graphPtr, axisPtr)) {
	sp->p.x = Blt_HMap(axisPtr, min);
	sp->q.x = Blt_HMap(axisPtr, max);
	sp->p.y = sp->q.y = line;
    } else {
	sp->q.x = sp->p.x = line;
	sp->p.y = Blt_VMap(axisPtr, min);
	sp->q.y = Blt_VMap(axisPtr, max);
    }
}


static void
MakeTick(
    Graph *graphPtr,
    Axis *axisPtr,
    double value,
    int tick, int line,		/* Lengths of tick and axis line. */
    Segment2D *sp)
{
    if (axisPtr->logScale) {
	value = EXP10(value);
    }
    if (AxisIsHorizontal(graphPtr, axisPtr)) {
	sp->p.x = sp->q.x = Blt_HMap(axisPtr, value);
	sp->p.y = line;
	sp->q.y = tick;
    } else {
	sp->p.x = line;
	sp->p.y = sp->q.y = Blt_VMap(axisPtr, value);
	sp->q.x = tick;
    }
}

static void
MakeSegments(Graph *graphPtr, Axis *axisPtr, AxisInfo *infoPtr)
{
    int arraySize;
    int nMajorTicks, nMinorTicks;
    Segment2D *segments;
    Segment2D *sp;

    if (axisPtr->segments != NULL) {
	Blt_Free(axisPtr->segments);
    }
    nMajorTicks = nMinorTicks = 0;
    if (axisPtr->t1Ptr != NULL) {
	nMajorTicks = axisPtr->t1Ptr->nTicks;
    }
    if (axisPtr->t2Ptr != NULL) {
	nMinorTicks = axisPtr->t2Ptr->nTicks;
    }
    arraySize = 1 + (nMajorTicks * (nMinorTicks + 1));
    segments = Blt_Malloc(arraySize * sizeof(Segment2D));
    assert(segments);

    sp = segments;
    if (axisPtr->lineWidth > 0) {
	/* Axis baseline */
	MakeAxisLine(graphPtr, axisPtr, infoPtr->axis, sp);
	sp++;
    }
    if (axisPtr->showTicks) {
	Blt_ChainLink *lp;
	double labelPos;
	int i;
	int isHoriz;

	isHoriz = AxisIsHorizontal(graphPtr, axisPtr);
	for (i = 0; i < axisPtr->t1Ptr->nTicks; i++) {
	    double t1, t2;
	    int j;

	    t1 = axisPtr->t1Ptr->values[i];
	    /* Minor ticks */
	    for (j = 0; j < axisPtr->t2Ptr->nTicks; j++) {
		t2 = t1 + (axisPtr->majorSweep.step * 
			   axisPtr->t2Ptr->values[j]);
		if (InRange(t2, &axisPtr->axisRange)) {
		    MakeTick(graphPtr, axisPtr, t2, infoPtr->t2, infoPtr->axis,
			     sp);
		    sp++;
		}
	    }
	    if (!InRange(t1, &axisPtr->axisRange)) {
		continue;
	    }
	    /* Major tick */
	    MakeTick(graphPtr, axisPtr, t1, infoPtr->t1, infoPtr->axis, sp);
	    sp++;
	}

	lp = Blt_ChainFirstLink(axisPtr->tickLabels);
	labelPos = (double)infoPtr->label;

	for (i = 0; i < axisPtr->t1Ptr->nTicks; i++) {
	    double t1;
	    TickLabel *labelPtr;
	    Segment2D seg;

	    t1 = axisPtr->t1Ptr->values[i];
	    if (axisPtr->labelOffset) {
		t1 += axisPtr->majorSweep.step * 0.5;
	    }
	    if (!InRange(t1, &axisPtr->axisRange)) {
		continue;
	    }
	    labelPtr = Blt_ChainGetValue(lp);
	    lp = Blt_ChainNextLink(lp);
	    MakeTick(graphPtr, axisPtr, t1, infoPtr->t1, infoPtr->axis, &seg);
	    /* Save tick label X-Y position. */
	    if (isHoriz) {
		labelPtr->anchorPos.x = seg.p.x;
		labelPtr->anchorPos.y = labelPos;
	    } else {
		labelPtr->anchorPos.x = labelPos;
		labelPtr->anchorPos.y = seg.p.y;
	    }
	}
    }
    axisPtr->segments = segments;
    axisPtr->nSegments = sp - segments;
    assert(axisPtr->nSegments <= arraySize);
}

/*
 * -----------------------------------------------------------------
 *
 * MapAxis --
 *
 *	Pre-calculates positions of the axis, ticks, and labels (to be
 *	used later when displaying the axis).  Calculates the values
 *	for each major and minor tick and checks to see if they are in
 *	range (the outer ticks may be outside of the range of plotted
 *	values).
 *
 *	Line segments for the minor and major ticks are saved into one
 *	XSegment array so that they can be drawn by a single
 *	XDrawSegments call. The positions of the tick labels are also
 *	computed and saved.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Line segments and tick labels are saved and used later to draw
 *	the axis.
 *
 * -----------------------------------------------------------------
 */
static void
MapAxis(
    Graph *graphPtr, 		/* Graph containing the axis. */
    Axis *axisPtr, 		/* Axis to be mapped.  */
    int offset, 		/* Offset of axis from the edge of the
				 * plotting area. */
    int margin)			/* Margin of the graph that holds this
				 * axis. */
{
    AxisInfo info;

    if (AxisIsHorizontal(graphPtr, axisPtr)) {
	axisPtr->screenMin = graphPtr->hOffset;
	axisPtr->width = graphPtr->right - graphPtr->left;
	axisPtr->screenRange = graphPtr->hRange;
    } else {
	axisPtr->screenMin = graphPtr->vOffset;
	axisPtr->height = graphPtr->bottom - graphPtr->top;
	axisPtr->screenRange = graphPtr->vRange;
    }
    axisPtr->screenScale = 1.0 / axisPtr->screenRange;
    AxisOffsets(graphPtr, axisPtr, margin, offset, &info);
    MakeSegments(graphPtr, axisPtr, &info);
}

/*
 * -----------------------------------------------------------------
 *
 * MapStackedAxis --
 *
 *	Pre-calculates positions of the axis, ticks, and labels (to be
 *	used later when displaying the axis).  Calculates the values
 *	for each major and minor tick and checks to see if they are in
 *	range (the outer ticks may be outside of the range of plotted
 *	values).
 *
 *	Line segments for the minor and major ticks are saved into one
 *	XSegment array so that they can be drawn by a single
 *	XDrawSegments call. The positions of the tick labels are also
 *	computed and saved.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Line segments and tick labels are saved and used later to draw
 *	the axis.
 *
 * -----------------------------------------------------------------
 */
static void
MapStackedAxis(
    Graph *graphPtr, 		/* Graph containing the axis. */
    Axis *axisPtr, 		/* Axis to be mapped.  */
    int count,			/* Order of axis among axes to be drawn in
				 * this margin. */
    int margin)			/* Margin of the graph that holds this
				 * axis. */
{
    AxisInfo info;
    int slice, width, height;

    if (AxisIsHorizontal(graphPtr, axisPtr)) {
	slice = graphPtr->hRange / graphPtr->margins[margin].axes->nLinks;
	axisPtr->screenMin = graphPtr->hOffset;
	axisPtr->width = slice;
    } else {
	slice = graphPtr->vRange / graphPtr->margins[margin].axes->nLinks;
	axisPtr->screenMin = graphPtr->vOffset;
	axisPtr->height = slice;
    }
#define AXIS_PAD 2
    Blt_GetTextExtents(axisPtr->tickFont, 0, "0", &width, &height);
    axisPtr->screenMin += (slice * count) + AXIS_PAD + height / 2;
    axisPtr->screenRange = slice - 2 * AXIS_PAD - height;
    axisPtr->screenScale = 1.0f / axisPtr->screenRange;
    AxisOffsets(graphPtr, axisPtr, margin, 0, &info);
    MakeSegments(graphPtr, axisPtr, &info);
    
}

/*
 *----------------------------------------------------------------------
 *
 * AdjustViewport --
 *
 *	Adjusts the offsets of the viewport according to the scroll mode.
 *	This is to accommodate both "listbox" and "canvas" style scrolling.
 *
 *	"canvas"	The viewport scrolls within the range of world
 *			coordinates.  This way the viewport always displays
 *			a full page of the world.  If the world is smaller
 *			than the viewport, then (bizarrely) the world and
 *			viewport are inverted so that the world moves up
 *			and down within the viewport.
 *
 *	"listbox"	The viewport can scroll beyond the range of world
 *			coordinates.  Every entry can be displayed at the
 *			top of the viewport.  This also means that the
 *			scrollbar thumb weirdly shrinks as the last entry
 *			is scrolled upward.
 *
 * Results:
 *	The corrected offset is returned.
 *
 *----------------------------------------------------------------------
 */
static double
AdjustViewport(double offset, double windowSize)
{
    /*
     * Canvas-style scrolling allows the world to be scrolled
     * within the window.
     */
    if (windowSize > 1.0) {
	if (windowSize < (1.0 - offset)) {
	    offset = 1.0 - windowSize;
	}
	if (offset > 0.0) {
	    offset = 0.0;
	}
    } else {
	if ((offset + windowSize) > 1.0) {
	    offset = 1.0 - windowSize;
	}
	if (offset < 0.0) {
	    offset = 0.0;
	}
    }
    return offset;
}

static int
GetAxisScrollInfo(
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv,
    double *offsetPtr,
    double windowSize,
    double scrollUnits)
{
    char *string;
    char c;
    double offset;
    size_t length;

    offset = *offsetPtr;
    string = Tcl_GetString(objv[0]);
    c = string[0];
    length = strlen(string);
    if ((c == 's') && (strncmp(string, "scroll", length) == 0)) {
	int count;
	double fract;

	assert(objc == 3);
	/* scroll number unit/page */
	if (Tcl_GetIntFromObj(interp, objv[1], &count) != TCL_OK) {
	    return TCL_ERROR;
	}
	string = Tcl_GetString(objv[2]);
	c = string[0];
	length = strlen(string);
	if ((c == 'u') && (strncmp(string, "units", length) == 0)) {
	    fract = (double)count * scrollUnits;
	} else if ((c == 'p') && (strncmp(string, "pages", length) == 0)) {
	    /* A page is 90% of the view-able window. */
	    fract = (double)count * windowSize * 0.9;
	} else {
	    Tcl_AppendResult(interp, "unknown \"scroll\" units \"", string,
		"\"", (char *)NULL);
	    return TCL_ERROR;
	}
	offset += fract;
    } else if ((c == 'm') && (strncmp(string, "moveto", length) == 0)) {
	double fract;

	assert(objc == 2);
	/* moveto fraction */
	if (Tcl_GetDoubleFromObj(interp, objv[1], &fract) != TCL_OK) {
	    return TCL_ERROR;
	}
	offset = fract;
    } else {
	int count;
	double fract;

	/* Treat like "scroll units" */
	if (Tcl_GetIntFromObj(interp, objv[0], &count) != TCL_OK) {
	    return TCL_ERROR;
	}
	fract = (double)count * scrollUnits;
	offset += fract;
	/* CHECK THIS: return TCL_OK; */
    }
    *offsetPtr = AdjustViewport(offset, windowSize);
    return TCL_OK;
}

/*
 * -----------------------------------------------------------------
 *
 * DrawAxis --
 *
 *	Draws the axis, ticks, and labels onto the canvas.
 *
 *	Initializes and passes text attribute information through
 *	TextStyle structure.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Axis gets drawn on window.
 *
 * -----------------------------------------------------------------
 */
static void
DrawAxis(Graph *graphPtr, Drawable drawable, Axis *axisPtr)
{
    Blt_Background bg;
    int relief;

    if (axisPtr->flags & AXIS_ACTIVE) {
	bg = axisPtr->activeBg;
	relief = axisPtr->activeRelief;
    } else {
	bg = axisPtr->normalBg;
	relief = axisPtr->relief;
    }
    if (bg != NULL) {
	Blt_FillBackgroundRectangle(graphPtr->tkwin, drawable, bg, 
		axisPtr->left, axisPtr->top, axisPtr->right - axisPtr->left, 
		axisPtr->bottom - axisPtr->top, axisPtr->borderWidth, relief);
    }
    if (axisPtr->title != NULL) {
	TextStyle ts;

	Blt_TextInitStyle(ts);
	Blt_TextSetAngle(ts, axisPtr->titleAngle);
	Blt_TextSetFont(ts, axisPtr->titleFont);
	Blt_TextSetPadding(ts, 1, 2, 0, 0);
	Blt_TextSetAnchor(ts, axisPtr->titleAnchor);
	Blt_TextSetJustify(ts, axisPtr->titleJustify);
	if (axisPtr->flags & AXIS_ACTIVE) {
	    Blt_TextSetForeground(ts, axisPtr->activeFgColor);
	} else {
	    Blt_TextSetForeground(ts, axisPtr->titleColor);
	}
	Blt_DrawText(graphPtr->tkwin, drawable, axisPtr->title, &ts, 
		(int)axisPtr->titlePos.x, (int)axisPtr->titlePos.y);
    }
    if (axisPtr->scrollCmdPrefix != NULL) {
	double viewWidth, viewMin, viewMax;
	double worldWidth, worldMin, worldMax;
	double fract;
	int isHoriz;

	worldMin = axisPtr->valueRange.min;
	worldMax = axisPtr->valueRange.max;
	if (DEFINED(axisPtr->scrollMin)) {
	    worldMin = axisPtr->scrollMin;
	}
	if (DEFINED(axisPtr->scrollMax)) {
	    worldMax = axisPtr->scrollMax;
	}
	viewMin = axisPtr->min;
	viewMax = axisPtr->max;
	if (viewMin < worldMin) {
	    viewMin = worldMin;
	}
	if (viewMax > worldMax) {
	    viewMax = worldMax;
	}
	if (axisPtr->logScale) {
	    worldMin = log10(worldMin);
	    worldMax = log10(worldMax);
	    viewMin = log10(viewMin);
	    viewMax = log10(viewMax);
	}
	worldWidth = worldMax - worldMin;	
	viewWidth = viewMax - viewMin;
	isHoriz = AxisIsHorizontal(graphPtr, axisPtr);

	if (isHoriz != axisPtr->descending) {
	    fract = (viewMin - worldMin) / worldWidth;
	} else {
	    fract = (worldMax - viewMax) / worldWidth;
	}
	fract = AdjustViewport(fract, viewWidth / worldWidth);

	if (isHoriz != axisPtr->descending) {
	    viewMin = (fract * worldWidth);
	    axisPtr->min = viewMin + worldMin;
	    axisPtr->max = axisPtr->min + viewWidth;
	    viewMax = viewMin + viewWidth;
	    if (axisPtr->logScale) {
		axisPtr->min = EXP10(axisPtr->min);
		axisPtr->max = EXP10(axisPtr->max);
	    }
	    Blt_UpdateScrollbar(graphPtr->interp, axisPtr->scrollCmdPrefix,
		(viewMin / worldWidth), (viewMax / worldWidth));
	} else {
	    viewMax = (fract * worldWidth);
	    axisPtr->max = worldMax - viewMax;
	    axisPtr->min = axisPtr->max - viewWidth;
	    viewMin = viewMax + viewWidth;
	    if (axisPtr->logScale) {
		axisPtr->min = EXP10(axisPtr->min);
		axisPtr->max = EXP10(axisPtr->max);
	    }
	    Blt_UpdateScrollbar(graphPtr->interp, axisPtr->scrollCmdPrefix,
		(viewMax / worldWidth), (viewMin / worldWidth));
	}
    }
    if (axisPtr->showTicks) {
	Blt_ChainLink *lp;
	TextStyle ts;

	Blt_TextInitStyle(ts);
	Blt_TextSetAngle(ts, axisPtr->tickAngle);
	Blt_TextSetFont(ts, axisPtr->tickFont);
	Blt_TextSetPadding(ts, 2, 2, 0, 0);
	Blt_TextSetAnchor(ts, axisPtr->tickAnchor);
	if (axisPtr->flags & AXIS_ACTIVE) {
	    Blt_TextSetForeground(ts, axisPtr->activeFgColor);
	} else {
	    Blt_TextSetForeground(ts, axisPtr->tickColor);
	}
	for (lp = Blt_ChainFirstLink(axisPtr->tickLabels); lp != NULL;
	    lp = Blt_ChainNextLink(lp)) {	
	    TickLabel *labelPtr;

	    labelPtr = Blt_ChainGetValue(lp);

	    /* Draw major tick labels */
	    Blt_DrawText(graphPtr->tkwin, drawable, labelPtr->string, &ts, 
		(int)labelPtr->anchorPos.x, (int)labelPtr->anchorPos.y);
	}
    }
    if ((axisPtr->nSegments > 0) && (axisPtr->lineWidth > 0)) {	
	GC gc;

	if (axisPtr->flags & AXIS_ACTIVE) {
	    gc = axisPtr->activeTickGC;
	} else {
	    gc = axisPtr->tickGC;
	}
	/* Draw the tick marks and axis line. */
	Blt_Draw2DSegments(graphPtr->display, drawable, gc, axisPtr->segments, 
		axisPtr->nSegments);
    }
}

/*
 * -----------------------------------------------------------------
 *
 * AxisToPostScript --
 *
 *	Generates PostScript output to draw the axis, ticks, and
 *	labels.
 *
 *	Initializes and passes text attribute information through
 *	TextStyle structure.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	PostScript output is left in graphPtr->interp->result;
 *
 * -----------------------------------------------------------------
 */
/* ARGSUSED */
static void
AxisToPostScript(Blt_PostScript ps, Axis *axisPtr)
{
    Blt_Background bg;
    int relief;

    if (axisPtr->flags & AXIS_ACTIVE) {
	bg = axisPtr->activeBg;
	relief = axisPtr->activeRelief;
    } else {
	bg = axisPtr->normalBg;
	relief = axisPtr->relief;
    }
    if (bg != NULL) {
	Tk_3DBorder border;

	border = Blt_BackgroundBorder(bg);
	Blt_Fill3DRectangleToPostScript(ps, border, (double)axisPtr->left, 
		(double)axisPtr->top, (double)(axisPtr->right - axisPtr->left),
		(double)(axisPtr->bottom - axisPtr->top), axisPtr->borderWidth,
		relief);
    }
    if (axisPtr->title != NULL) {
	TextStyle ts;

	Blt_TextInitStyle(ts);
	Blt_TextSetAngle(ts, axisPtr->titleAngle);
	Blt_TextSetFont(ts, axisPtr->titleFont);
	Blt_TextSetPadding(ts, 1, 2, 0, 0);
	Blt_TextSetAnchor(ts, axisPtr->titleAnchor);
	Blt_TextSetJustify(ts, axisPtr->titleJustify);
	if (axisPtr->flags & AXIS_ACTIVE) {
	    Blt_TextSetForeground(ts, axisPtr->activeFgColor);
	} else {
	    Blt_TextSetForeground(ts, axisPtr->titleColor);
	}
	Blt_TextToPostScript(ps, axisPtr->title, &ts, axisPtr->titlePos.x, 
		axisPtr->titlePos.y);
    }
    if (axisPtr->showTicks) {
	Blt_ChainLink *lp;
	TextStyle ts;

	Blt_TextInitStyle(ts);
	Blt_TextSetAngle(ts, axisPtr->tickAngle);
	Blt_TextSetFont(ts, axisPtr->tickFont);
	Blt_TextSetPadding(ts, 2, 2, 0, 0);
	Blt_TextSetAnchor(ts, axisPtr->tickAnchor);
	Blt_TextSetForeground(ts, axisPtr->tickColor);

	for (lp = Blt_ChainFirstLink(axisPtr->tickLabels); lp != NULL; 
	     lp = Blt_ChainNextLink(lp)) {
	    TickLabel *labelPtr;

	    labelPtr = Blt_ChainGetValue(lp);
	    Blt_TextToPostScript(ps, labelPtr->string, &ts, 
		labelPtr->anchorPos.x, labelPtr->anchorPos.y);
	}
    }
    if ((axisPtr->nSegments > 0) && (axisPtr->lineWidth > 0)) {
	Blt_LineAttributesToPostScript(ps, axisPtr->tickColor,
		axisPtr->lineWidth, (Blt_Dashes *)NULL, CapButt, JoinMiter);
	Blt_2DSegmentsToPostScript(ps, axisPtr->segments, axisPtr->nSegments);
    }
}

static void
MakeGridLine(
    Graph *graphPtr,
    Axis *axisPtr,
    double value,
    Segment2D *sp)
{
    if (axisPtr->logScale) {
	value = EXP10(value);
    }
    /* Grid lines run orthogonally to the axis */
    if (AxisIsHorizontal(graphPtr, axisPtr)) {
	sp->p.y = graphPtr->top;
	sp->q.y = graphPtr->bottom;
	sp->p.x = sp->q.x = Blt_HMap(axisPtr, value);
    } else {
	sp->p.x = graphPtr->left;
	sp->q.x = graphPtr->right;
	sp->p.y = sp->q.y = Blt_VMap(axisPtr, value);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * MapGridlines --
 *
 *	Assembles the grid lines associated with an axis. Generates
 *	tick positions if necessary (this happens when the axis is
 *	not a logical axis too).
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */
static void
MapGridlines(Graph *graphPtr, Axis *axisPtr)
{
    Segment2D *segments, *sp;
    Ticks *t1Ptr, *t2Ptr;
    int needed;
    int i;

    if (axisPtr == NULL) {
	return;
    }
    t1Ptr = axisPtr->t1Ptr;
    if (t1Ptr == NULL) {
	t1Ptr = GenerateTicks(&axisPtr->majorSweep);
    }
    t2Ptr = axisPtr->t2Ptr;
    if (t2Ptr == NULL) {
	t2Ptr = GenerateTicks(&axisPtr->minorSweep);
    }
    needed = t1Ptr->nTicks;
    if (axisPtr->gridMinor) {
	needed += (t1Ptr->nTicks * t2Ptr->nTicks);
    }
    if (needed == 0) {
	return;			
    }
    if (needed == axisPtr->nAllocated) {
	segments = axisPtr->gridSegments;
    } else {
	segments = Blt_Malloc(sizeof(Segment2D) * needed);
	if (segments == NULL) {
	    return;		/* Can't allocate memory for grid segments. */
	}
    }
    sp = segments;
    for (i = 0; i < t1Ptr->nTicks; i++) {
	double value;

	value = t1Ptr->values[i];
	if (axisPtr->gridMinor) {
	    int j;

	    for (j = 0; j < t2Ptr->nTicks; j++) {
		double subValue;

		subValue = value + (axisPtr->majorSweep.step * 
				    t2Ptr->values[j]);
		if (InRange(subValue, &axisPtr->axisRange)) {
		    MakeGridLine(graphPtr, axisPtr, subValue, sp);
		    sp++;
		}
	    }
	}
	if (InRange(value, &axisPtr->axisRange)) {
	    MakeGridLine(graphPtr, axisPtr, value, sp);
	    sp++;
	}
    }

    if (t1Ptr != axisPtr->t1Ptr) {
	Blt_Free(t1Ptr);	/* Free generated ticks. */
    }
    if (t2Ptr != axisPtr->t2Ptr) {
	Blt_Free(t2Ptr);	/* Free generated ticks. */
    }
    if ((axisPtr->gridSegments != segments) && 
	(axisPtr->gridSegments != NULL)) {
	Blt_Free(axisPtr->gridSegments);
    }
    axisPtr->gridSegments = segments;
    axisPtr->nAllocated = needed;
    axisPtr->nUsed = sp - segments;
}

/*
 *----------------------------------------------------------------------
 *
 * GetAxisGeometry --
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */
static void
GetAxisGeometry(Graph *graphPtr, Axis *axisPtr)
{
    int height;

    FreeLabels(axisPtr->tickLabels);
    height = 0;
    if (axisPtr->lineWidth > 0) {
	/* Leave room for axis baseline (and pad) */
	height += axisPtr->lineWidth + 2;
    }
    if (axisPtr->showTicks) {
	int pad;
	int i, nLabels;
	int maxWidth, maxHeight;

	SweepTicks(axisPtr);
	
	if (axisPtr->t1Ptr->nTicks < 0) {
	    fprintf(stderr, "%s major ticks can't be %d\n", 
		axisPtr->object.name, axisPtr->t1Ptr->nTicks);
	    abort();
	}
	if (axisPtr->t1Ptr->nTicks > MAXTICKS) {
	    fprintf(stderr, "too big, %s major ticks can't be %d\n",
		    axisPtr->object.name, axisPtr->t1Ptr->nTicks);
	    abort();
	}
	
	maxHeight = maxWidth = 0;
	nLabels = 0;
	for (i = 0; i < axisPtr->t1Ptr->nTicks; i++) {
	    TickLabel *labelPtr;
	    double x, x2;
	    int lw, lh;

	    x2 = x = axisPtr->t1Ptr->values[i];
	    if (axisPtr->labelOffset) {
		x2 += axisPtr->majorSweep.step * 0.5;
	    }
	    if (!InRange(x2, &axisPtr->axisRange)) {
		continue;
	    }
	    labelPtr = MakeLabel(graphPtr, axisPtr, x);
	    Blt_ChainAppend(axisPtr->tickLabels, labelPtr);
	    nLabels++;
	    /* 
	     * Get the dimensions of each tick label.  
	     * Remember tick labels can be multi-lined and/or rotated. 
	     */
	    Blt_GetTextExtents(axisPtr->tickFont, 0, labelPtr->string, &lw,&lh);
	    labelPtr->width = lw;
	    labelPtr->height = lh;

	    if (axisPtr->tickAngle != 0.0) {
		double rWidth, rHeight;

		Blt_GetBoundingBox(lw, lh, axisPtr->tickAngle, &rWidth, 
			&rHeight, (Point2D *)NULL);
		lw = ROUND(rWidth);
		lh = ROUND(rHeight);
	    }
	    if (maxWidth < lw) {
		maxWidth = lw;
	    }
	    if (maxHeight < lh) {
		maxHeight = lh;
	    }
	}
	assert(nLabels <= axisPtr->t1Ptr->nTicks);
	
	/* Because the axis cap style is "CapProjecting", we need to
	 * account for an extra 1.5 linewidth at the end of each
	 * line.  */

	pad = ((axisPtr->lineWidth * 15) / 10);
	
	if (AxisIsHorizontal(graphPtr, axisPtr)) {
	    height += maxHeight + pad;
	} else {
	    height += maxWidth + pad;
	}
	height += 2 * AXIS_PAD_TITLE;
	if (axisPtr->lineWidth > 0) {
	    /* Distance from axis line to tick label. */
	    height += ABS(axisPtr->tickLength);
	}
    }

    if (axisPtr->title != NULL) {
	if (axisPtr->titleAlternate) {
	    if (height < axisPtr->titleHeight) {
		height = axisPtr->titleHeight;
	    }
	} else {
	    height += axisPtr->titleHeight + AXIS_PAD_TITLE;
	}
    }

    /* Correct for orientation of the axis. */
    if (AxisIsHorizontal(graphPtr, axisPtr)) {
	axisPtr->height = height;
    } else {
	axisPtr->width = height;
    }
}

/*
 *----------------------------------------------------------------------
 *
 * GetMarginGeometry --
 *
 *	Examines all the axes in the given margin and determines the 
 *	area required to display them.  
 *
 *	Note: For multiple axes, the titles are displayed in another
 *	      margin. So we must keep track of the widest title.
 *	
 * Results:
 *	Returns the width or height of the margin, depending if it
 *	runs horizontally along the graph or vertically.
 *
 * Side Effects:
 *	The area width and height set in the margin.  Note again that
 *	this may be corrected later (mulitple axes) to adjust for
 *	the longest title in another margin.
 *
 *----------------------------------------------------------------------
 */
static int
GetMarginGeometry(Graph *graphPtr, Margin *marginPtr)
{
    Blt_ChainLink *lp;
    int width, height;
    int isHoriz;
    int length, count;

    isHoriz = HORIZMARGIN(marginPtr);

    /* Count the number of visible axes. */
    count = 0;
    length = width = height = 0;
    if (graphPtr->stackAxes) {
	for (lp = Blt_ChainFirstLink(marginPtr->axes); lp != NULL;
	     lp = Blt_ChainNextLink(lp)) {
	    Axis *axisPtr;
	    
	    axisPtr = Blt_ChainGetValue(lp);
	    if ((!axisPtr->object.hidden) && (axisPtr->flags & AXIS_ONSCREEN)) {
		count++;
		if (graphPtr->flags & GET_AXIS_GEOMETRY) {
		    GetAxisGeometry(graphPtr, axisPtr);
		}
		if (isHoriz) {
		    if (height < axisPtr->height) {
			height = axisPtr->height;
		    }
		} else {
		    if (width < axisPtr->width) {
			width = axisPtr->width;
		    }
		}
	    }
	}
    } else {
	for (lp = Blt_ChainFirstLink(marginPtr->axes); lp != NULL;
	     lp = Blt_ChainNextLink(lp)) {
	    Axis *axisPtr;
	    
	    axisPtr = Blt_ChainGetValue(lp);
	    if ((!axisPtr->object.hidden) && (axisPtr->flags & AXIS_ONSCREEN)) {
		count++;
		if (graphPtr->flags & GET_AXIS_GEOMETRY) {
		    GetAxisGeometry(graphPtr, axisPtr);
		}
		if ((axisPtr->titleAlternate) && 
		    (length < axisPtr->titleWidth)) {
		    length = axisPtr->titleWidth;
		}
		if (isHoriz) {
		    height += axisPtr->height;
		} else {
		    width += axisPtr->width;
		}
	    }
	}
    }
    /* Enforce a minimum size for margins. */
    if (width < 3) {
	width = 3;
    }
    if (height < 3) {
	height = 3;
    }
    marginPtr->nAxes = count;
    marginPtr->axesTitleLength = length;
    marginPtr->width = width;
    marginPtr->height = height;
    marginPtr->axesOffset = (isHoriz) ? height : width;
    return marginPtr->axesOffset;
}

/*
 *----------------------------------------------------------------------
 *
 * ComputeMargins --
 *
 *	Computes the size of the margins and the plotting area.  We
 *	first compute the space needed for the axes in each margin.
 *	Then how much space the legend will occupy.  Finally, if the
 *	user has requested a margin size, we override the computed
 *	value.
 *
 * Results:
 *
 *---------------------------------------------------------------------- */
static void
ComputeMargins(Graph *graphPtr)
{
    int left, right, top, bottom;
    int width, height;
    int insets;

    /* 
     * Step 1:  Compute the amount of space needed to display the axes
     *		(there many be zero or more) associated with the margin.
     */
    top =    GetMarginGeometry(graphPtr, &graphPtr->topMargin);
    bottom = GetMarginGeometry(graphPtr, &graphPtr->bottomMargin);
    left =   GetMarginGeometry(graphPtr, &graphPtr->leftMargin);
    right =  GetMarginGeometry(graphPtr, &graphPtr->rightMargin);

    /* 
     * Step 2:  Add the graph title height to the top margin. 
     */
    if (graphPtr->title != NULL) {
	top += graphPtr->titleHeight;
    }
    insets = 2 * (graphPtr->inset + graphPtr->plotBorderWidth);

    /* 
     * Step 3:  Use the current estimate of the plot area to compute 
     *		the legend size.  Add it to the proper margin.
     */
    width = graphPtr->width - (insets + left + right);
    height = graphPtr->height - (insets + top + bottom);
    Blt_MapLegend(graphPtr->legend, width, height);
    if (!Blt_LegendIsHidden(graphPtr->legend)) {
	switch (Blt_LegendSite(graphPtr->legend)) {
	case LEGEND_RIGHT:
	    right += Blt_LegendWidth(graphPtr->legend) + 2;
	    break;
	case LEGEND_LEFT:
	    left += Blt_LegendWidth(graphPtr->legend) + 2;
	    break;
	case LEGEND_TOP:
	    top += Blt_LegendHeight(graphPtr->legend) + 2;
	    break;
	case LEGEND_BOTTOM:
	    bottom += Blt_LegendHeight(graphPtr->legend) + 2;
	    break;
	case LEGEND_XY:
	case LEGEND_PLOT:
	case LEGEND_WINDOW:
	    /* Do nothing. */
	    break;
	}
    }

    /* 
     * Recompute the plotarea, now accounting for the legend. 
     */
    width = graphPtr->width - (insets + left + right);
    height = graphPtr->height - (insets + top + bottom);

    /*
     * Step 5:	If necessary, correct for the requested plot area 
     *		aspect ratio.
     */
    if (graphPtr->aspect > 0.0) {
	double ratio;

	/* 
	 * Shrink one dimension of the plotarea to fit the requested
	 * width/height aspect ratio.  
	 */
	ratio = (double)width / (double)height;
	if (ratio > graphPtr->aspect) {
	    int scaledWidth;

	    /* Shrink the width. */
	    scaledWidth = (int)(height * graphPtr->aspect);
	    if (scaledWidth < 1) {
		scaledWidth = 1;
	    }
	    right += (width - scaledWidth); /* Add the difference to
					     * the right margin. */
	    /* CHECK THIS: width = scaledWidth; */
	} else {
	    int scaledHeight;

	    /* Shrink the height. */
	    scaledHeight = (int)(width / graphPtr->aspect);
	    if (scaledHeight < 1) {
		scaledHeight = 1;
	    }
	    top += (height - scaledHeight); /* Add the difference to
					     * the top margin. */
	    /* CHECK THIS: height = scaledHeight; */
	}
    }

    /* 
     * Step 6:	If there's multiple axes in a margin, the axis 
     *		titles will be displayed in the adjoining margins.
     *		Make sure there's room for the longest axis titles. 
     */

    if (top < graphPtr->leftMargin.axesTitleLength) {
	top = graphPtr->leftMargin.axesTitleLength;
    }
    if (right < graphPtr->bottomMargin.axesTitleLength) {
	right = graphPtr->bottomMargin.axesTitleLength;
    }
    if (top < graphPtr->rightMargin.axesTitleLength) {
	top = graphPtr->rightMargin.axesTitleLength;
    }
    if (right < graphPtr->topMargin.axesTitleLength) {
	right = graphPtr->topMargin.axesTitleLength;
    }

    /* 
     * Step 7:  Override calculated values with requested margin 
     *		sizes. 
     */

    graphPtr->leftMargin.width = left;
    graphPtr->rightMargin.width = right;
    graphPtr->topMargin.height =  top;
    graphPtr->bottomMargin.height = bottom;
	    
    if (graphPtr->leftMargin.reqSize > 0) {
	graphPtr->leftMargin.width = graphPtr->leftMargin.reqSize;
    }
    if (graphPtr->rightMargin.reqSize > 0) {
	graphPtr->rightMargin.width = graphPtr->rightMargin.reqSize;
    }
    if (graphPtr->topMargin.reqSize > 0) {
	graphPtr->topMargin.height = graphPtr->topMargin.reqSize;
    }
    if (graphPtr->bottomMargin.reqSize > 0) {
	graphPtr->bottomMargin.height = graphPtr->bottomMargin.reqSize;
    }
}

/*
 * -----------------------------------------------------------------
 *
 * Blt_LayoutMargins --
 *
 * 	Calculate the layout of the graph.  Based upon the data,
 *	axis limits, X and Y titles, and title height, determine
 *	the cavity left which is the plotting surface.  The first
 *	step get the data and axis limits for calculating the space
 *	needed for the top, bottom, left, and right margins.
 *
 * 	1) The LEFT margin is the area from the left border to the
 *	   Y axis (not including ticks). It composes the border
 *	   width, the width an optional Y axis label and its padding,
 *	   and the tick numeric labels. The Y axis label is rotated
 *	   90 degrees so that the width is the font height.
 *
 * 	2) The RIGHT margin is the area from the end of the graph
 *	   to the right window border. It composes the border width,
 *	   some padding, the font height (this may be dubious. It
 *	   appears to provide a more even border), the max of the
 *	   legend width and 1/2 max X tick number. This last part is
 *	   so that the last tick label is not clipped.
 *
 *           Window Width
 *      ___________________________________________________________
 *      |          |                               |               |
 *      |          |   TOP  height of title        |               |
 *      |          |                               |               |
 *      |          |           x2 title            |               |
 *      |          |                               |               |
 *      |          |        height of x2-axis      |               |
 *      |__________|_______________________________|_______________|  W
 *      |          | -plotpady                     |               |  i
 *      |__________|_______________________________|_______________|  n
 *      |          | top                   right   |               |  d
 *      |          |                               |               |  o
 *      |   LEFT   |                               |     RIGHT     |  w
 *      |          |                               |               |
 *      | y        |     Free area = 104%          |      y2       |  H
 *      |          |     Plotting surface = 100%   |               |  e
 *      | t        |     Tick length = 2 + 2%      |      t        |  i
 *      | i        |                               |      i        |  g
 *      | t        |                               |      t  legend|  h
 *      | l        |                               |      l   width|  t
 *      | e        |                               |      e        |
 *      |    height|                               |height         |
 *      |       of |                               | of            |
 *      |    y-axis|                               |y2-axis        |
 *      |          |                               |               |
 *      |          |origin 0,0                     |               |
 *      |__________|_left_________________bottom___|_______________|
 *      |          |-plotpady                      |               |
 *      |__________|_______________________________|_______________|
 *      |          | (xoffset, yoffset)            |               |
 *      |          |                               |               |
 *      |          |       height of x-axis        |               |
 *      |          |                               |               |
 *      |          |   BOTTOM   x title            |               |
 *      |__________|_______________________________|_______________|
 *
 * 3) The TOP margin is the area from the top window border to the top
 *    of the graph. It composes the border width, twice the height of
 *    the title font (if one is given) and some padding between the
 *    title.
 *
 * 4) The BOTTOM margin is area from the bottom window border to the
 *    X axis (not including ticks). It composes the border width, the height
 *    an optional X axis label and its padding, the height of the font
 *    of the tick labels.
 *
 * The plotting area is between the margins which includes the X and Y axes
 * including the ticks but not the tick numeric labels. The length of
 * the ticks and its padding is 5% of the entire plotting area.  Hence the
 * entire plotting area is scaled as 105% of the width and height of the
 * area.
 *
 * The axis labels, ticks labels, title, and legend may or may not be
 * displayed which must be taken into account.
 *
 *
 * -----------------------------------------------------------------
 */
void
Blt_LayoutMargins(Graph *graphPtr)
{
    int width, height;
    int titleY;
    int left, right, top, bottom;

    ComputeMargins(graphPtr);
    left = graphPtr->leftMargin.width + graphPtr->inset + 
	graphPtr->plotBorderWidth;
    right = graphPtr->rightMargin.width + graphPtr->inset + 
	graphPtr->plotBorderWidth;
    top = graphPtr->topMargin.height + graphPtr->inset + 
	graphPtr->plotBorderWidth;
    bottom = graphPtr->bottomMargin.height + graphPtr->inset + 
	graphPtr->plotBorderWidth;

    /* Based upon the margins, calculate the space left for the graph. */
    width = graphPtr->width - (left + right);
    height = graphPtr->height - (top + bottom);
    if (width < 1) {
	width = 1;
    }
    if (height < 1) {
	height = 1;
    }
    graphPtr->left = left;
    graphPtr->right = left + width;
    graphPtr->bottom = top + height;
    graphPtr->top = top;

    graphPtr->vOffset = top + graphPtr->padTop;
    graphPtr->vRange = height - PADDING(graphPtr->padY);
    graphPtr->hOffset = left + graphPtr->padLeft;
    graphPtr->hRange = width - PADDING(graphPtr->padX);

    if (graphPtr->vRange < 1) {
	graphPtr->vRange = 1;
    }
    if (graphPtr->hRange < 1) {
	graphPtr->hRange = 1;
    }
    graphPtr->hScale = 1.0 / (double)graphPtr->hRange;
    graphPtr->vScale = 1.0 / (double)graphPtr->vRange;

    /*
     * Calculate the placement of the graph title so it is centered within the
     * space provided for it in the top margin
     */
    titleY = graphPtr->titleHeight;
    graphPtr->titleY = (titleY / 2) + graphPtr->inset;
    graphPtr->titleX = (graphPtr->right + graphPtr->left) / 2;

}

/*
 * ----------------------------------------------------------------------
 *
 * ConfigureAxis --
 *
 *	Configures axis attributes (font, line width, label, etc).
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 * Side Effects:
 *	Axis layout is deferred until the height and width of the
 *	window are known.
 *
 * ----------------------------------------------------------------------
 */

static int
ConfigureAxis(Graph *graphPtr, Axis *axisPtr)
{
    char errMsg[200];
    double angle;

    /* Check the requested axis limits. Can't allow -min to be greater
     * than -max, or have undefined log scale limits.  */
    if (((DEFINED(axisPtr->reqMin)) && (DEFINED(axisPtr->reqMax))) &&
	(axisPtr->reqMin >= axisPtr->reqMax)) {
	sprintf(errMsg, "impossible limits (min %g >= max %g) for axis \"%s\"",
	    axisPtr->reqMin, axisPtr->reqMax, axisPtr->object.name);
	Tcl_AppendResult(graphPtr->interp, errMsg, (char *)NULL);
	/* Bad values, turn on axis auto-scaling */
	axisPtr->reqMin = axisPtr->reqMax = VALUE_UNDEFINED;
	return TCL_ERROR;
    }
    if ((axisPtr->logScale) && (DEFINED(axisPtr->reqMin)) &&
	(axisPtr->reqMin <= 0.0)) {
	sprintf(errMsg, "bad logscale limits (min=%g,max=%g) for axis \"%s\"",
	    axisPtr->reqMin, axisPtr->reqMax, axisPtr->object.name);
	Tcl_AppendResult(graphPtr->interp, errMsg, (char *)NULL);
	/* Bad minimum value, turn on auto-scaling */
	axisPtr->reqMin = VALUE_UNDEFINED;
	return TCL_ERROR;
    }
    angle = FMOD(axisPtr->tickAngle, 360.0);
    if (angle < 0.0) {
	angle += 360.0;
    }
    if (axisPtr->normalBg != NULL) {
	Blt_SetBackgroundChangedProc(axisPtr->normalBg, Blt_UpdateGraph, 
		graphPtr);
    }
    if (axisPtr->activeBg != NULL) {
	Blt_SetBackgroundChangedProc(axisPtr->activeBg, Blt_UpdateGraph, 
		graphPtr);
    }
    axisPtr->tickAngle = angle;
    ResetTextStyles(graphPtr, axisPtr);

    axisPtr->titleWidth = axisPtr->titleHeight = 0;
    if (axisPtr->title != NULL) {
	int w, h;

	Blt_GetTextExtents(axisPtr->titleFont, 0, axisPtr->title, &w, &h);
	axisPtr->titleWidth = (short int)w;
	axisPtr->titleHeight = (short int)h;
    }

    /* 
     * Don't bother to check what configuration options have changed.
     * Almost every option changes the size of the plotting area
     * (except for -color and -titlecolor), requiring the graph and
     * its contents to be completely redrawn.
     *
     * Recompute the scale and offset of the axis in case -min, -max
     * options have changed.  
     */
    graphPtr->flags |= REDRAW_WORLD;
    if (!Blt_ConfigModified(configSpecs, "-*color", "-background", "-bg",
		    (char *)NULL)) {
	graphPtr->flags |= (MAP_WORLD | RESET_AXES);
	axisPtr->flags |= AXIS_DIRTY;
    }
    Blt_EventuallyRedrawGraph(graphPtr);

    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * CreateAxis --
 *
 *	Create and initialize a structure containing information to
 * 	display a graph axis.
 *
 * Results:
 *	The return value is a standard Tcl result.
 *
 * ----------------------------------------------------------------------
 */
static Axis *
CreateAxis( 
    Graph *graphPtr,
    char *name,			/* Identifier for axis. */
    int margin)
{
    Axis *axisPtr;
    Blt_HashEntry *hp;
    int isNew;

    if (name[0] == '-') {
	Tcl_AppendResult(graphPtr->interp, "name of axis \"", name, 
			 "\" can't start with a '-'", (char *)NULL);
	return NULL;
    }
    hp = Blt_CreateHashEntry(&graphPtr->axes.table, name, &isNew);
    if (!isNew) {
	axisPtr = (Axis *)Blt_GetHashValue(hp);
	if ((axisPtr->flags & AXIS_DELETE_PENDING) == 0) {
	    Tcl_AppendResult(graphPtr->interp, "axis \"", name,
		"\" already exists in \"", Tk_PathName(graphPtr->tkwin), "\"",
		(char *)NULL);
	    return NULL;
	}
	axisPtr->flags &= ~AXIS_DELETE_PENDING;
    } else {
	axisPtr = Blt_Calloc(1, sizeof(Axis));
	assert(axisPtr);

	axisPtr->object.name = Blt_Strdup(name);
	axisPtr->hashPtr = hp;
	Blt_GraphSetObjectClass(&axisPtr->object, OBJECT_CLASS_NONE);
	axisPtr->object.graphPtr = graphPtr;
	axisPtr->looseMin = axisPtr->looseMax = AXIS_TIGHT;
	axisPtr->reqNumMinorTicks = 2;
	axisPtr->scrollUnits = 10;
	axisPtr->showTicks = TRUE;
	axisPtr->reqMin = axisPtr->reqMax = VALUE_UNDEFINED;
	axisPtr->scrollMin = axisPtr->scrollMax = VALUE_UNDEFINED;
	axisPtr->gridMinor = TRUE;

	if ((graphPtr->classId == OBJECT_CLASS_BAR_ELEMENT) && 
	    ((margin == MARGIN_TOP) || (margin == MARGIN_BOTTOM))) {
	    axisPtr->reqStep = 1.0;
	    axisPtr->reqNumMinorTicks = 0;
	} 
	if ((margin == MARGIN_RIGHT) || (margin == MARGIN_TOP)) {
	    axisPtr->object.hidden = TRUE;
	}
	Blt_TextInitStyle(axisPtr->limitsTextStyle);
	axisPtr->tickLabels = Blt_ChainCreate();
	axisPtr->lineWidth = 1;
	Blt_SetHashValue(hp, axisPtr);
    }
    return axisPtr;
}

static int
GetAxisFromObj(
    Tcl_Interp *interp,
    Graph *graphPtr,		/* Graph widget record. */
    Tcl_Obj *objPtr,		/* Name of the axis to be searched for. */
    Axis **axisPtrPtr)		/* (out) Pointer to found axis structure. */
{
    Blt_HashEntry *hp;
    char *name;

    *axisPtrPtr = NULL;
    name = Tcl_GetString(objPtr);
    hp = Blt_FindHashEntry(&graphPtr->axes.table, name);
    if (hp != NULL) {
	Axis *axisPtr;

	axisPtr = (Axis *)Blt_GetHashValue(hp);
	if ((axisPtr->flags & AXIS_DELETE_PENDING) == 0) {
	    *axisPtrPtr = axisPtr;
	    return TCL_OK;
	}
    }
    if (interp != NULL) {
	Tcl_AppendResult(interp, "can't find axis \"", name, "\" in \"", 
		Tk_PathName(graphPtr->tkwin), "\"", (char *)NULL);
    }
    return TCL_ERROR;
}

static int
GetAxisByClass(
    Tcl_Interp *interp,
    Graph *graphPtr,
    Tcl_Obj *objPtr,
    int classId,
    Axis **axisPtrPtr)
{
    Axis *axisPtr;

    if (GetAxisFromObj(interp, graphPtr, objPtr, &axisPtr) != TCL_OK) {
	return TCL_ERROR;
    }
    if (classId != OBJECT_CLASS_NONE) {
	if ((axisPtr->refCount == 0) || 
	    (axisPtr->object.classId == OBJECT_CLASS_NONE)) {
	    /* Set the axis type on the first use of it. */
	    Blt_GraphSetObjectClass(&axisPtr->object, classId);
	} else if (axisPtr->object.classId != classId) {
	    if (interp != NULL) {
  	        Tcl_AppendResult(interp, "axis \"", Tcl_GetString(objPtr),
		    "\" is already in use on an opposite ", 
			axisPtr->object.className, "-axis", 
			(char *)NULL);
	    }
	    return TCL_ERROR;
	}
	axisPtr->refCount++;
    }
    *axisPtrPtr = axisPtr;
    return TCL_OK;
}

void
Blt_DestroyAxes(Graph *graphPtr)
{
    {
	Blt_HashEntry *hp;
	Blt_HashSearch cursor;

	for (hp = Blt_FirstHashEntry(&graphPtr->axes.table, &cursor);
	     hp != NULL; hp = Blt_NextHashEntry(&cursor)) {
	    Axis *axisPtr;
	    
	    axisPtr = (Axis *)Blt_GetHashValue(hp);
	    axisPtr->hashPtr = NULL;
	    DestroyAxis(axisPtr);
	}
    }
    Blt_DeleteHashTable(&graphPtr->axes.table);
    {
	int i;
	
	for (i = 0; i < 4; i++) {
	    Blt_ChainDestroy(graphPtr->axisChain[i]);
	}
    }
    Blt_DeleteHashTable(&graphPtr->axes.tagTable);
    Blt_ChainDestroy(graphPtr->axes.displayList);
}

int
Blt_DefaultAxes(Graph *graphPtr)
{
    int i;
    static char *axisNames[4] = { "x", "y", "x2", "y2" } ;
    int flags;

    flags = Blt_GraphType(graphPtr);
    for (i = 0; i < 4; i++) {
	Blt_Chain *chainPtr;
	Axis *axisPtr;
	int classId;

	chainPtr = Blt_ChainCreate();
	graphPtr->axisChain[i] = chainPtr;

	/* Create a default axis for each chain. */
	axisPtr = CreateAxis(graphPtr, axisNames[i], i);
	if (axisPtr == NULL) {
	    return TCL_ERROR;
	}
	axisPtr->refCount = 1;	/* Default axes are assumed in use. */
	classId = (i & 1) ? OBJECT_CLASS_Y_AXIS : OBJECT_CLASS_X_AXIS;
	Blt_GraphSetObjectClass(&axisPtr->object, classId);
	axisPtr->flags |= AXIS_ONSCREEN;

	/*
	 * Blt_ConfigureComponentFromObj creates a temporary child window 
	 * by the name of the axis.  It's used so that the Tk routines
	 * that access the X resource database can describe a single 
	 * component and not the entire graph.
	 */
 	if (Blt_ConfigureComponentFromObj(graphPtr->interp, graphPtr->tkwin,
		axisPtr->object.name, "Axis", configSpecs, 0, (Tcl_Obj **)NULL,
		(char *)axisPtr, flags) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (ConfigureAxis(graphPtr, axisPtr) != TCL_OK) {
	    return TCL_ERROR;
	}
	axisPtr->linkPtr = Blt_ChainAppend(chainPtr, axisPtr);
	axisPtr->chainPtr = chainPtr;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ActivateOp --
 *
 * 	Activates the axis, drawing the axis with its -activeforeground,
 *	-activebackgound, -activerelief attributes.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side Effects:
 *	Graph will be redrawn to reflect the new axis attributes.
 *
 *----------------------------------------------------------------------
 */
static int
ActivateOp(
    Tcl_Interp *interp,
    Graph *graphPtr,
    Axis *axisPtr,
    int objc,
    Tcl_Obj *CONST *objv)
{
    char *string;

    string = Tcl_GetString(objv[2]);
    axisPtr->flags |= AXIS_ACTIVE;
    if (string[0] == 'a') {
	axisPtr->flags |= AXIS_ACTIVE;
    } else {
	axisPtr->flags &= ~AXIS_ACTIVE;
    }
    if ((!axisPtr->object.hidden) && (axisPtr->flags & AXIS_ONSCREEN)) {
	graphPtr->flags |= DRAW_MARGINS;
	Blt_EventuallyRedrawGraph(graphPtr);
    }
    return TCL_OK;
}

/*----------------------------------------------------------------------
 *
 * BindOp --
 *
 *    .g axis bind axisName sequence command
 *
 *----------------------------------------------------------------------
 */
static int
BindOp(
    Tcl_Interp *interp,
    Graph *graphPtr,
    Axis *axisPtr,
    int objc,		
    Tcl_Obj *CONST *objv)
{
    return Blt_ConfigureBindingsFromObj(interp, graphPtr->bindTable,
          Blt_MakeAxisTag(graphPtr, axisPtr->object.name), objc, objv);
}
          
/*
 * ----------------------------------------------------------------------
 *
 * CgetOp --
 *
 *	Queries axis attributes (font, line width, label, etc).
 *
 * Results:
 *	Return value is a standard Tcl result.  If querying configuration
 *	values, interp->result will contain the results.
 *
 * ----------------------------------------------------------------------
 */
/* ARGSUSED */
static int
CgetOp(
    Tcl_Interp *interp,
    Graph *graphPtr,
    Axis *axisPtr,
    int objc,		
    Tcl_Obj *CONST *objv)
{
    return Blt_ConfigureValueFromObj(interp, graphPtr->tkwin, configSpecs,
	(char *)axisPtr, objv[0], Blt_GraphType(graphPtr));
}

/*
 * ----------------------------------------------------------------------
 *
 * ConfigureOp --
 *
 *	Queries or resets axis attributes (font, line width, label, etc).
 *
 * Results:
 *	Return value is a standard Tcl result.  If querying configuration
 *	values, interp->result will contain the results.
 *
 * Side Effects:
 *	Axis resources are possibly allocated (GC, font). Axis layout is
 *	deferred until the height and width of the window are known.
 *
 * ----------------------------------------------------------------------
 */
static int
ConfigureOp(
    Tcl_Interp *interp,
    Graph *graphPtr,
    Axis *axisPtr,
    int objc,		
    Tcl_Obj *CONST *objv)
{
    int flags;

    flags = BLT_CONFIG_OBJV_ONLY | Blt_GraphType(graphPtr);
    if (objc == 0) {
	return Blt_ConfigureInfoFromObj(interp, graphPtr->tkwin, configSpecs,
	    (char *)axisPtr, (Tcl_Obj *)NULL, flags);
    } else if (objc == 1) {
	return Blt_ConfigureInfoFromObj(interp, graphPtr->tkwin, configSpecs,
	    (char *)axisPtr, objv[0], flags);
    }
    if (Blt_ConfigureWidgetFromObj(interp, graphPtr->tkwin, configSpecs, 
	objc, objv, (char *)axisPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }
    if (ConfigureAxis(graphPtr, axisPtr) != TCL_OK) {
	return TCL_ERROR;
    }
    if (axisPtr->flags & AXIS_ONSCREEN) {
	if (!Blt_ConfigModified(configSpecs, "-*color", "-background", "-bg",
				(char *)NULL)) {
	    graphPtr->flags |= REDRAW_BACKING_STORE;
	}
	graphPtr->flags |= DRAW_MARGINS;
	Blt_EventuallyRedrawGraph(graphPtr);
    }
    return TCL_OK;
}



/*
 *--------------------------------------------------------------
 *
 * LimitsOp --
 *
 *	This procedure returns a string representing the axis limits
 *	of the graph.  The format of the string is { left top right bottom}.
 *
 * Results:
 *	Always returns TCL_OK.  The interp->result field is
 *	a list of the graph axis limits.
 *
 *--------------------------------------------------------------
 */
/*ARGSUSED*/
static int
LimitsOp(
    Tcl_Interp *interp,
    Graph *graphPtr,
    Axis *axisPtr,
    int objc,		
    Tcl_Obj *CONST *objv)
{
    Tcl_Obj *listObjPtr;
    double min, max;

    if (graphPtr->flags & RESET_AXES) {
	Blt_ResetAxes(graphPtr);
    }
    if (axisPtr->logScale) {
	min = EXP10(axisPtr->axisRange.min);
	max = EXP10(axisPtr->axisRange.max);
    } else {
	min = axisPtr->axisRange.min;
	max = axisPtr->axisRange.max;
    }
    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **)NULL);
    Tcl_ListObjAppendElement(interp, listObjPtr, Tcl_NewDoubleObj(min));
    Tcl_ListObjAppendElement(interp, listObjPtr, Tcl_NewDoubleObj(max));
    Tcl_SetObjResult(interp, listObjPtr);
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * InvTransformOp --
 *
 *	Maps the given window coordinate into an axis-value.
 *
 * Results:
 *	Returns a standard Tcl result.  interp->result contains
 *	the axis value. If an error occurred, TCL_ERROR is returned
 *	and interp->result will contain an error message.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
InvTransformOp(
    Tcl_Interp *interp,
    Graph *graphPtr,
    Axis *axisPtr,
    int objc,		
    Tcl_Obj *CONST *objv)
{
    double y;			/* Real graph coordinate */
    int sy;			/* Integer window coordinate*/

    if (graphPtr->flags & RESET_AXES) {
	Blt_ResetAxes(graphPtr);
    }
    if (Tcl_GetIntFromObj(interp, objv[0], &sy) != TCL_OK) {
	return TCL_ERROR;
    }
    /*
     * Is the axis vertical or horizontal?
     *
     * Check the site where the axis was positioned.  If the axis is
     * virtual, all we have to go on is how it was mapped to an
     * element (using either -mapx or -mapy options).  
     */
    if (AxisIsHorizontal(graphPtr, axisPtr)) {
	y = Blt_InvHMap(axisPtr, (double)sy);
    } else {
	y = Blt_InvVMap(axisPtr, (double)sy);
    }
    Tcl_SetDoubleObj(Tcl_GetObjResult(interp), y);
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * TransformOp --
 *
 *	Maps the given axis-value to a window coordinate.
 *
 * Results:
 *	Returns a standard Tcl result.  interp->result contains
 *	the window coordinate. If an error occurred, TCL_ERROR
 *	is returned and interp->result will contain an error
 *	message.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
TransformOp(
    Tcl_Interp *interp,
    Graph *graphPtr,
    Axis *axisPtr,		/* Axis */
    int objc,			/* Not used. */
    Tcl_Obj *CONST *objv)
{
    double x;

    if (graphPtr->flags & RESET_AXES) {
	Blt_ResetAxes(graphPtr);
    }
    if (Blt_ExprDoubleFromObj(interp, objv[0], &x) != TCL_OK) {
	return TCL_ERROR;
    }
    if (AxisIsHorizontal(graphPtr, axisPtr)) {
	x = Blt_HMap(axisPtr, x);
    } else {
	x = Blt_VMap(axisPtr, x);
    }
    Tcl_SetIntObj(Tcl_GetObjResult(interp), (int)x);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * UseOp --
 *
 *	Changes the virtual axis used by the logical axis.
 *
 * Results:
 *	A standard Tcl result.  If the named axis doesn't exist
 *	an error message is put in interp->result.
 *
 * .g xaxis use "abc def gah"
 * .g xaxis use [lappend abc [.g axis use]]
 *
 *--------------------------------------------------------------
 */
/*ARGSUSED*/
static int
UseOp(
    Tcl_Interp *interp,
    Graph *graphPtr,
    Axis *axisPtr,		/* Not used. */
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_Chain *chainPtr;
    Blt_ChainLink *lp;
    Tcl_Obj **axisObjv;
    int classId;
    int axisObjc;
    int i;

    chainPtr = graphPtr->margins[lastMargin].axes;
    if (objc == 0) {
	Tcl_Obj *listObjPtr;

	listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **)NULL);
	for (lp = Blt_ChainFirstLink(chainPtr); lp != NULL;
	     lp = Blt_ChainNextLink(lp)) {
	    axisPtr = Blt_ChainGetValue(lp);
	    Tcl_ListObjAppendElement(interp, listObjPtr,
		Tcl_NewStringObj(axisPtr->object.name, -1));
	}
	Tcl_SetObjResult(interp, listObjPtr);
	return TCL_OK;
    }
    if ((lastMargin == MARGIN_BOTTOM) || (lastMargin == MARGIN_TOP)) {
	classId = (graphPtr->inverted) 
	    ? OBJECT_CLASS_Y_AXIS : OBJECT_CLASS_X_AXIS;
    } else {
	classId = (graphPtr->inverted) 
	    ? OBJECT_CLASS_X_AXIS : OBJECT_CLASS_Y_AXIS;
    }
    if (Tcl_ListObjGetElements(interp, objv[0], &axisObjc, &axisObjv) 
	!= TCL_OK) {
	return TCL_ERROR;
    }
    for (lp = Blt_ChainFirstLink(chainPtr); lp!= NULL; 
	 lp = Blt_ChainNextLink(lp)) {
	axisPtr = Blt_ChainGetValue(lp);
	axisPtr->linkPtr = NULL;
	axisPtr->flags &= ~AXIS_ONSCREEN;
	/* Clear the axis type if it's not currently used.*/
	if (axisPtr->refCount == 0) {
	    Blt_GraphSetObjectClass(&axisPtr->object, OBJECT_CLASS_NONE);
	}
    }
    Blt_ChainReset(chainPtr);
    for (i = 0; i < axisObjc; i++) {
	if (GetAxisFromObj(interp, graphPtr, axisObjv[i], &axisPtr) != TCL_OK){
	    return TCL_ERROR;
	}
	if (axisPtr->object.classId == OBJECT_CLASS_NONE) {
	    Blt_GraphSetObjectClass(&axisPtr->object, classId);
	} else if (axisPtr->object.classId != classId) {
	    Tcl_AppendResult(interp, "wrong type axis \"", 
		axisPtr->object.name, "\": can't use ", 
		axisPtr->object.className, " type axis.", (char *)NULL); 
	    return TCL_ERROR;
	}
	if (axisPtr->linkPtr != NULL) {
	    /* Move the axis from the old margin's "use" list to the new. */
	    Blt_ChainUnlinkLink(axisPtr->chainPtr, axisPtr->linkPtr);
	    Blt_ChainAppendLink(chainPtr, axisPtr->linkPtr);
	} else {
	    axisPtr->linkPtr = Blt_ChainAppend(chainPtr, axisPtr);
	}
	axisPtr->chainPtr = chainPtr;
	axisPtr->flags |= AXIS_ONSCREEN;
    }
    graphPtr->flags |= (GET_AXIS_GEOMETRY | LAYOUT_NEEDED | RESET_AXES);
    /* When any axis changes, we need to layout the entire graph.  */
    graphPtr->flags |= (MAP_WORLD | REDRAW_WORLD);
    Blt_EventuallyRedrawGraph(graphPtr);
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * CreateVirtualOp --
 *
 *	Creates a new axis.
 *
 * Results:
 *	Returns a standard Tcl result.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
CreateVirtualOp(
    Tcl_Interp *interp,
    Graph *graphPtr,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Axis *axisPtr;
    int flags;

    axisPtr = CreateAxis(graphPtr, Tcl_GetString(objv[3]), MARGIN_NONE);
    if (axisPtr == NULL) {
	return TCL_ERROR;
    }
    flags = Blt_GraphType(graphPtr);
    if ((Blt_ConfigureComponentFromObj(interp, graphPtr->tkwin, 
	axisPtr->object.name, "Axis", configSpecs, objc - 4, objv + 4, 
	(char *)axisPtr, flags) != TCL_OK) || 
	(ConfigureAxis(graphPtr, axisPtr) != TCL_OK)) {
	DestroyAxis(axisPtr);
	return TCL_ERROR;
    }
    Tcl_SetStringObj(Tcl_GetObjResult(interp), axisPtr->object.name, -1);
    return TCL_OK;
}
/*
 *----------------------------------------------------------------------
 *
 * ActivateVirtualOp --
 *
 * 	Activates the axis, drawing the axis with its -activeforeground,
 *	-activebackgound, -activerelief attributes.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side Effects:
 *	Graph will be redrawn to reflect the new axis attributes.
 *
 *----------------------------------------------------------------------
 */
static int
ActivateVirtualOp(
    Tcl_Interp *interp,
    Graph *graphPtr,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Axis *axisPtr;

    if (GetAxisFromObj(interp, graphPtr, objv[3], &axisPtr) != TCL_OK) {
	return TCL_ERROR;
    }
    return ActivateOp(interp, graphPtr, axisPtr, objc, objv);
}


/*----------------------------------------------------------------------
 *
 * BindVirtualOp --
 *
 *    .g axis bind axisName sequence command
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
BindVirtualOp(
    Tcl_Interp *interp,
    Graph *graphPtr,
    int objc,
    Tcl_Obj *CONST *objv)
{
    if (objc == 3) {
	Blt_HashEntry *hp;
	Blt_HashSearch cursor;
	Tcl_Obj *listObjPtr;

	listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **)NULL);
	for (hp = Blt_FirstHashEntry(&graphPtr->axes.tagTable, &cursor);
	     hp != NULL; hp = Blt_NextHashEntry(&cursor)) {
	    char *tagName;
	    Tcl_Obj *objPtr;

	    tagName = Blt_GetHashKey(&graphPtr->axes.tagTable, hp);
	    objPtr = Tcl_NewStringObj(tagName, -1);
	    Tcl_ListObjAppendElement(interp, listObjPtr, objPtr);
	}
	Tcl_SetObjResult(interp, listObjPtr);
	return TCL_OK;
    }
    return Blt_ConfigureBindingsFromObj(interp, graphPtr->bindTable, 
	Blt_MakeAxisTag(graphPtr, Tcl_GetString(objv[3])), objc - 4, objv + 4);
}


/*
 * ----------------------------------------------------------------------
 *
 * CgetVirtualOp --
 *
 *	Queries axis attributes (font, line width, label, etc).
 *
 * Results:
 *	Return value is a standard Tcl result.  If querying configuration
 *	values, interp->result will contain the results.
 *
 * ----------------------------------------------------------------------
 */
/* ARGSUSED */
static int
CgetVirtualOp(
    Tcl_Interp *interp,
    Graph *graphPtr,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Axis *axisPtr;

    if (GetAxisFromObj(interp, graphPtr, objv[3], &axisPtr) != TCL_OK) {
	return TCL_ERROR;
    }
    return CgetOp(interp, graphPtr, axisPtr, objc - 4, objv + 4);
}

/*
 * ----------------------------------------------------------------------
 *
 * ConfigureVirtualOp --
 *
 *	Queries or resets axis attributes (font, line width, label, etc).
 *
 * Results:
 *	Return value is a standard Tcl result.  If querying configuration
 *	values, interp->result will contain the results.
 *
 * Side Effects:
 *	Axis resources are possibly allocated (GC, font). Axis layout is
 *	deferred until the height and width of the window are known.
 *
 * ----------------------------------------------------------------------
 */
static int
ConfigureVirtualOp(
    Tcl_Interp *interp,
    Graph *graphPtr,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Tcl_Obj *CONST *options;
    int i;
    int nNames, nOpts;

    /* Figure out where the option value pairs begin */
    objc -= 3;
    objv += 3;
    for (i = 0; i < objc; i++) {
	Axis *axisPtr;
	char *string;

	string = Tcl_GetString(objv[i]);
	if (string[0] == '-') {
	    break;
	}
	if (GetAxisFromObj(interp, graphPtr, objv[i], &axisPtr) != TCL_OK) {
	    return TCL_ERROR;
	}
    }
    nNames = i;			/* Number of pen names specified */
    nOpts = objc - i;		/* Number of options specified */
    options = objv + i;		/* Start of options in objv  */

    for (i = 0; i < nNames; i++) {
	Axis *axisPtr;

	if (GetAxisFromObj(interp, graphPtr, objv[i], &axisPtr) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (ConfigureOp(interp, graphPtr, axisPtr, nOpts, options) != TCL_OK) {
	    break;
	}
    }
    if (i < nNames) {
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * DeleteVirtualOp --
 *
 *	Deletes one or more axes.  The actual removal may be deferred
 *	until the axis is no longer used by any element. The axis
 *	can't be referenced by its name any longer and it may be
 *	recreated.
 *
 * Results:
 *	Returns a standard Tcl result.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
DeleteVirtualOp(
    Tcl_Interp *interp,
    Graph *graphPtr,
    int objc,
    Tcl_Obj *CONST *objv)
{
    int i;

    for (i = 3; i < objc; i++) {
	Axis *axisPtr;

	if (GetAxisFromObj(interp, graphPtr, objv[i], &axisPtr) != TCL_OK) {
	    return TCL_ERROR;
	}
	axisPtr->flags |= AXIS_DELETE_PENDING;
	if (axisPtr->refCount == 0) {
	    DestroyAxis(axisPtr);
	}
    }
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * GetVirtualOp --
 *
 *    Returns the name of the picked axis (using the axis
 *    bind operation).  Right now, the only name accepted is
 *    "current".
 *
 * Results:
 *    A standard Tcl result.  The interpreter result will contain
 *    the name of the axis.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
GetVirtualOp(
    Tcl_Interp *interp,
    Graph *graphPtr,
    int objc,		
    Tcl_Obj *CONST *objv)
{
    Axis *axisPtr;

    axisPtr = Blt_GetCurrentItem(graphPtr->bindTable);
    /* Report only on axes. */
    if ((axisPtr != NULL) && 
	((axisPtr->object.classId & OBJECT_CLASS_AXIS) || 
	 (axisPtr->object.classId == OBJECT_CLASS_NONE))) {
	char c;
	char  *string;

	string = Tcl_GetString(objv[3]);
	c = string[0];
	if ((c == 'c') && (strcmp(string, "current") == 0)) {
	    Tcl_SetStringObj(Tcl_GetObjResult(interp), axisPtr->object.name,-1);
	} else if ((c == 'd') && (strcmp(string, "detail") == 0)) {
	    Tcl_SetStringObj(Tcl_GetObjResult(interp), axisPtr->detail, -1);
	}
    }
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * InvTransformVirtualOp --
 *
 *	Maps the given window coordinate into an axis-value.
 *
 * Results:
 *	Returns a standard Tcl result.  interp->result contains
 *	the axis value. If an error occurred, TCL_ERROR is returned
 *	and interp->result will contain an error message.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
InvTransformVirtualOp(
    Tcl_Interp *interp,
    Graph *graphPtr,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Axis *axisPtr;

    if (GetAxisFromObj(interp, graphPtr, objv[3], &axisPtr) != TCL_OK) {
	return TCL_ERROR;
    }
    return InvTransformOp(interp, graphPtr, axisPtr, objc - 4, objv + 4);
}

/*
 *--------------------------------------------------------------
 *
 * LimitsVirtualOp --
 *
 *	This procedure returns a string representing the axis limits
 *	of the graph.  The format of the string is { left top right bottom}.
 *
 * Results:
 *	Always returns TCL_OK.  The interp->result field is
 *	a list of the graph axis limits.
 *
 *--------------------------------------------------------------
 */
/*ARGSUSED*/
static int
LimitsVirtualOp(
    Tcl_Interp *interp,
    Graph *graphPtr,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Axis *axisPtr;

    if (GetAxisFromObj(interp, graphPtr, objv[3], &axisPtr) != TCL_OK) {
	return TCL_ERROR;
    }
    return LimitsOp(interp, graphPtr, axisPtr, objc - 4, objv + 4);
}

/*
 * ----------------------------------------------------------------------
 *
 * NamesVirtualOp --
 *
 *	Return a list of the names of all the axes.
 *
 * Results:
 *	Returns a standard Tcl result.
 *
 * ----------------------------------------------------------------------
 */

/*ARGSUSED*/
static int
NamesVirtualOp(
    Tcl_Interp *interp,
    Graph *graphPtr,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Tcl_Obj *listObjPtr;

    listObjPtr = Tcl_NewListObj(0, (Tcl_Obj **)NULL);
    if (objc == 3) {
	Blt_HashEntry *hp;
	Blt_HashSearch cursor;

	for (hp = Blt_FirstHashEntry(&graphPtr->axes.table, &cursor);
	     hp != NULL; hp = Blt_NextHashEntry(&cursor)) {
	    Axis *axisPtr;

	    axisPtr = (Axis *)Blt_GetHashValue(hp);
	    if (axisPtr->flags & AXIS_DELETE_PENDING) {
		continue;
	    }
	    Tcl_ListObjAppendElement(interp, listObjPtr, 
		     Tcl_NewStringObj(axisPtr->object.name, -1));
	}
    } else {
	Blt_HashEntry *hp;
	Blt_HashSearch cursor;

	for (hp = Blt_FirstHashEntry(&graphPtr->axes.table, &cursor);
	     hp != NULL; hp = Blt_NextHashEntry(&cursor)) {
	    Axis *axisPtr;
	    int i;

	    axisPtr = (Axis *)Blt_GetHashValue(hp);
	    for (i = 3; i < objc; i++) {
		char *pattern;

		pattern = Tcl_GetString(objv[i]);
		if (Tcl_StringMatch(axisPtr->object.name, pattern)) {
		    Tcl_ListObjAppendElement(interp, listObjPtr, 
			Tcl_NewStringObj(axisPtr->object.name, -1));
		    break;
		}
	    }
	}
    }
    Tcl_SetObjResult(interp, listObjPtr);
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * TransformVirtualOp --
 *
 *	Maps the given axis-value to a window coordinate.
 *
 * Results:
 *	Returns a standard Tcl result.  interp->result contains
 *	the window coordinate. If an error occurred, TCL_ERROR
 *	is returned and interp->result will contain an error
 *	message.
 *
 * ----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
TransformVirtualOp(
    Tcl_Interp *interp,
    Graph *graphPtr,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Axis *axisPtr;

    if (GetAxisFromObj(interp, graphPtr, objv[3], &axisPtr) != TCL_OK) {
	return TCL_ERROR;
    }
    return TransformOp(interp, graphPtr, axisPtr, objc - 4, objv + 4);
}

static int
ViewVirtualOp(
    Tcl_Interp *interp,
    Graph *graphPtr,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Axis *axisPtr;
    double axisOffset, scrollUnits;
    double fract;
    double viewMin, viewMax, worldMin, worldMax;
    double viewWidth, worldWidth;

    if (GetAxisFromObj(interp, graphPtr, objv[3], &axisPtr) != TCL_OK) {
	return TCL_ERROR;
    }
    worldMin = axisPtr->valueRange.min;
    worldMax = axisPtr->valueRange.max;
    /* Override data dimensions with user-selected limits. */
    if (DEFINED(axisPtr->scrollMin)) {
	worldMin = axisPtr->scrollMin;
    }
    if (DEFINED(axisPtr->scrollMax)) {
	worldMax = axisPtr->scrollMax;
    }
    viewMin = axisPtr->min;
    viewMax = axisPtr->max;
    /* Bound the view within scroll region. */ 
    if (viewMin < worldMin) {
	viewMin = worldMin;
    } 
    if (viewMax > worldMax) {
	viewMax = worldMax;
    }
    if (axisPtr->logScale) {
	worldMin = log10(worldMin);
	worldMax = log10(worldMax);
	viewMin = log10(viewMin);
	viewMax = log10(viewMax);
    }
    worldWidth = worldMax - worldMin;
    viewWidth = viewMax - viewMin;

    /* Unlike horizontal axes, vertical axis values run opposite of
     * the scrollbar first/last values.  So instead of pushing the
     * axis minimum around, we move the maximum instead. */

    if (AxisIsHorizontal(graphPtr, axisPtr) != axisPtr->descending) {
	axisOffset = viewMin - worldMin;
	scrollUnits = (double)axisPtr->scrollUnits * graphPtr->hScale;
    } else {
	axisOffset = worldMax - viewMax;
	scrollUnits = (double)axisPtr->scrollUnits * graphPtr->vScale;
    }
    if (objc == 4) {
	/* Note: Bound the fractions between 0.0 and 1.0 to support
	 * "canvas"-style scrolling. */
	fract = axisOffset / worldWidth;
	Tcl_AppendElement(interp, Blt_Dtoa(interp, FCLAMP(fract)));
	fract = (axisOffset + viewWidth) / worldWidth;
	Tcl_AppendElement(interp, Blt_Dtoa(interp, FCLAMP(fract)));
	return TCL_OK;
    }
    fract = axisOffset / worldWidth;
    if (GetAxisScrollInfo(interp, objc - 4, objv + 4, &fract, 
		viewWidth / worldWidth, scrollUnits) != TCL_OK) {
	return TCL_ERROR;
    }
    if (AxisIsHorizontal(graphPtr, axisPtr) != axisPtr->descending) {
	axisPtr->reqMin = (fract * worldWidth) + worldMin;
	axisPtr->reqMax = axisPtr->reqMin + viewWidth;
    } else {
	axisPtr->reqMax = worldMax - (fract * worldWidth);
	axisPtr->reqMin = axisPtr->reqMax - viewWidth;
    }
    if (axisPtr->logScale) {
	axisPtr->reqMin = EXP10(axisPtr->reqMin);
	axisPtr->reqMax = EXP10(axisPtr->reqMax);
    }
    graphPtr->flags |= (GET_AXIS_GEOMETRY | LAYOUT_NEEDED | RESET_AXES);
    Blt_EventuallyRedrawGraph(graphPtr);
    return TCL_OK;
}

int
Blt_VirtualAxisOp(
    Graph *graphPtr,
    Tcl_Interp *interp,
    int objc,
    Tcl_Obj *CONST *objv)
{
    Blt_Op proc;
    int result;
    static Blt_OpSpec axisOps[] =
    {
	{"activate", 1, (Blt_Op)ActivateVirtualOp, 4, 4, "axisName",},
	{"bind", 1, (Blt_Op)BindVirtualOp, 3, 6, "axisName sequence command",},
	{"cget", 2, (Blt_Op)CgetVirtualOp, 5, 5, "axisName option",},
	{"configure", 2, (Blt_Op)ConfigureVirtualOp, 4, 0,
	    "axisName ?axisName?... ?option value?...",},
	{"create", 2, (Blt_Op)CreateVirtualOp, 4, 0,
	    "axisName ?option value?...",},
	{"deactivate", 3, (Blt_Op)ActivateVirtualOp, 4, 4, "axisName",},
	{"delete", 3, (Blt_Op)DeleteVirtualOp, 3, 0, "?axisName?...",},
	{"get", 1, (Blt_Op)GetVirtualOp, 4, 4, "name",},
	{"invtransform", 1, (Blt_Op)InvTransformVirtualOp, 5, 5,
	    "axisName value",},
	{"limits", 1, (Blt_Op)LimitsVirtualOp, 4, 4, "axisName",},
	{"names", 1, (Blt_Op)NamesVirtualOp, 3, 0, "?pattern?...",},
	{"transform", 1, (Blt_Op)TransformVirtualOp, 5, 5, "axisName value",},
	{"view", 1, (Blt_Op)ViewVirtualOp, 4, 7,
	    "axisName ?moveto fract? ?scroll number what?",},
    };
    static int nAxisOps = sizeof(axisOps) / sizeof(Blt_OpSpec);

    proc = Blt_GetOpFromObj(interp, nAxisOps, axisOps, BLT_OP_ARG2, 
	objc, objv, 0);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    result = (*proc) (interp, graphPtr, objc, objv);
    return result;
}

int
Blt_AxisOp(
    Tcl_Interp *interp,
    Graph *graphPtr,
    int margin,
    int objc,
    Tcl_Obj *CONST *objv)
{
    int result;
    Blt_Op proc;
    Axis *axisPtr;
    static Blt_OpSpec axisOps[] =
    {
	{"activate", 1, (Blt_Op)ActivateOp, 3, 3, "",},
	{"bind", 1, (Blt_Op)BindOp, 2, 5, "sequence command",},
	{"cget", 2, (Blt_Op)CgetOp, 4, 4, "option",},
	{"configure", 2, (Blt_Op)ConfigureOp, 3, 0, "?option value?...",},
	{"deactivate", 1, (Blt_Op)ActivateOp, 3, 3, "",},
	{"invtransform", 1, (Blt_Op)InvTransformOp, 4, 4, "value",},
	{"limits", 1, (Blt_Op)LimitsOp, 3, 3, "",},
	{"transform", 1, (Blt_Op)TransformOp, 4, 4, "value",},
	{"use", 1, (Blt_Op)UseOp, 3, 4, "?axisName?",},
    };
    static int nAxisOps = sizeof(axisOps) / sizeof(Blt_OpSpec);

    proc = Blt_GetOpFromObj(interp, nAxisOps, axisOps, BLT_OP_ARG2, objc, objv,
	0);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    lastMargin = margin;	/* Set global variable to the margin
				 * in the argument list. Needed only
				 * for UseOp. */
    axisPtr = Blt_GetFirstAxis(graphPtr->margins[margin].axes);
    if (axisPtr == NULL) {
	Tcl_AppendResult(interp, "no axes used on ", Tcl_GetString(objv[1]),
			 (char *)NULL);
	return TCL_ERROR;
    }
    result = (*proc)(interp, graphPtr, axisPtr, objc - 3, objv + 3);
    return result;
}

void
Blt_MapAxes(Graph *graphPtr)
{
    int margin;
    
    for (margin = 0; margin < 4; margin++) {
	Blt_Chain *chainPtr;
	Blt_ChainLink *lp;
	int count, offset;

	chainPtr = graphPtr->margins[margin].axes;
	count = offset = 0;
	for (lp = Blt_ChainFirstLink(chainPtr); lp != NULL; 
	     lp = Blt_ChainNextLink(lp)) {
	    Axis *axisPtr;

	    axisPtr = Blt_ChainGetValue(lp);
	    if ((!axisPtr->object.hidden) && 
		(axisPtr->flags & AXIS_ONSCREEN)) {
		if (graphPtr->stackAxes) {
		    MapStackedAxis(graphPtr, axisPtr, count, margin);
		} else {
		    MapAxis(graphPtr, axisPtr, offset, margin);
		}
		if (axisPtr->showGrid) {
		    MapGridlines(graphPtr, axisPtr);
		}
		offset += (AxisIsHorizontal(graphPtr, axisPtr)) 
		    ? axisPtr->height : axisPtr->width;
		count++;
	    }
	}
    }
}

void
Blt_DrawAxes(Graph *graphPtr, Drawable drawable)
{
    int i;

    for (i = 0; i < 4; i++) {
	Blt_ChainLink *lp;

	for (lp = Blt_ChainLastLink(graphPtr->margins[i].axes); 
	     lp != NULL; lp = Blt_ChainPrevLink(lp)) {
	    Axis *axisPtr;

	    axisPtr = Blt_ChainGetValue(lp);
	    if ((!axisPtr->object.hidden) && 
		(axisPtr->flags & AXIS_ONSCREEN)) {
		DrawAxis(graphPtr, drawable, axisPtr);
	    }
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DrawGrids --
 *
 *	Draws the grid lines associated with each axis.
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */
void
Blt_DrawGrids(Graph *graphPtr, Drawable drawable) 
{
    int i;

    for (i = 0; i < 4; i++) {
	Blt_ChainLink *lp;

	for (lp = Blt_ChainFirstLink(graphPtr->margins[i].axes); lp != NULL; 
	     lp = Blt_ChainNextLink(lp)) {
	    Axis *axisPtr;

	    axisPtr = Blt_ChainGetValue(lp);
	    if ((!axisPtr->object.hidden) && (axisPtr->flags & AXIS_ONSCREEN) &&
		(axisPtr->showGrid)) {
		Blt_Draw2DSegments(graphPtr->display, drawable, axisPtr->gridGC,
			axisPtr->gridSegments, axisPtr->nUsed);
	    }
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_DrawGrids --
 *
 *	Draws the grid lines associated with each axis.
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */
void
Blt_GridsToPostScript(Graph *graphPtr, Blt_PostScript ps) 
{
    int i;

    for (i = 0; i < 4; i++) {
	Blt_ChainLink *lp;


	for (lp = Blt_ChainFirstLink(graphPtr->margins[i].axes); lp != NULL; 
	     lp = Blt_ChainNextLink(lp)) {
	    Axis *axisPtr;

	    axisPtr = Blt_ChainGetValue(lp);
	    if ((!axisPtr->object.hidden) && (axisPtr->flags & AXIS_ONSCREEN) &&
		(axisPtr->showGrid)) {
		Blt_FormatToPostScript(ps, "%% Axis %s: grid line attributes\n",
				       axisPtr->object.name);
		Blt_LineAttributesToPostScript(ps, axisPtr->gridColor, 
			axisPtr->gridLineWidth, &axisPtr->gridDashes, CapButt, 
			JoinMiter);
		Blt_FormatToPostScript(ps, "%% Axis %s: grid line segments\n",
			axisPtr->object.name);
		Blt_2DSegmentsToPostScript(ps, axisPtr->gridSegments, 
			axisPtr->nUsed);
	    }
	}
    }
}

void
Blt_AxesToPostScript(Graph *graphPtr, Blt_PostScript ps) 
{
    int i;

    for (i = 0; i < 4; i++) {
	Blt_ChainLink *lp;

	for (lp = Blt_ChainFirstLink(graphPtr->margins[i].axes); 
	     lp != NULL; lp = Blt_ChainNextLink(lp)) {
	    Axis *axisPtr;

	    axisPtr = Blt_ChainGetValue(lp);
	    if ((!axisPtr->object.hidden) && 
		(axisPtr->flags & AXIS_ONSCREEN)) {
		AxisToPostScript(ps, axisPtr);
	    }
	}
    }
}


/*
 * ----------------------------------------------------------------------
 *
 * Blt_DrawAxisLimits --
 *
 *	Draws the min/max values of the axis in the plotting area. 
 *	The text strings are formatted according to the "sprintf"
 *	format descriptors in the limitsFormats array.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Draws the numeric values of the axis limits into the outer
 *	regions of the plotting area.
 *
 * ----------------------------------------------------------------------
 */
void
Blt_DrawAxisLimits(Graph *graphPtr, Drawable drawable)
{
    Blt_HashEntry *hp;
    Blt_HashSearch cursor;
    char minString[200], maxString[200];
    int vMin, hMin, vMax, hMax;

#define SPACING 8
    vMin = vMax = graphPtr->left + graphPtr->padLeft + 2;
    hMin = hMax = graphPtr->bottom - graphPtr->padBottom - 2;	/* Offsets */

    for (hp = Blt_FirstHashEntry(&graphPtr->axes.table, &cursor);
	hp != NULL; hp = Blt_NextHashEntry(&cursor)) {
	Axis *axisPtr;
	Dim2D textDim;
	char *minFmt, *maxFmt;
	char *minPtr, *maxPtr;
	int isHoriz;

	axisPtr = (Axis *)Blt_GetHashValue(hp);

	if (axisPtr->nFormats == 0) {
	    continue;
	}
	isHoriz = AxisIsHorizontal(graphPtr, axisPtr);
	minPtr = maxPtr = NULL;
	minFmt = maxFmt = axisPtr->limitsFormats[0];
	if (axisPtr->nFormats > 1) {
	    maxFmt = axisPtr->limitsFormats[1];
	}
	if (minFmt[0] != '\0') {
	    minPtr = minString;
	    sprintf(minString, minFmt, axisPtr->axisRange.min);
	}
	if (maxFmt[0] != '\0') {
	    maxPtr = maxString;
	    sprintf(maxString, maxFmt, axisPtr->axisRange.max);
	}
	if (axisPtr->descending) {
	    char *tmp;

	    tmp = minPtr, minPtr = maxPtr, maxPtr = tmp;
	}
	if (maxPtr != NULL) {
	    if (isHoriz) {
		Blt_TextSetAngle(axisPtr->limitsTextStyle, 90.0);
		Blt_TextSetAnchor(axisPtr->limitsTextStyle, TK_ANCHOR_SE);
		Blt_DrawText2(graphPtr->tkwin, drawable, maxPtr,
		    &axisPtr->limitsTextStyle, graphPtr->right, hMax, &textDim);
		hMax -= (textDim.height + SPACING);
	    } else {
		Blt_TextSetAngle(axisPtr->limitsTextStyle, 0.0);
		Blt_TextSetAnchor(axisPtr->limitsTextStyle, TK_ANCHOR_NW);
		Blt_DrawText2(graphPtr->tkwin, drawable, maxPtr,
		    &axisPtr->limitsTextStyle, vMax, graphPtr->top, &textDim);
		vMax += (textDim.width + SPACING);
	    }
	}
	if (minPtr != NULL) {
	    Blt_TextSetAnchor(axisPtr->limitsTextStyle, TK_ANCHOR_SW);
	    if (isHoriz) {
		Blt_TextSetAngle(axisPtr->limitsTextStyle, 90.0);
		Blt_DrawText2(graphPtr->tkwin, drawable, minPtr,
		    &axisPtr->limitsTextStyle, graphPtr->left, hMin, &textDim);
		hMin -= (textDim.height + SPACING);
	    } else {
		Blt_TextSetAngle(axisPtr->limitsTextStyle, 0.0);
		Blt_DrawText2(graphPtr->tkwin, drawable, minPtr,
		    &axisPtr->limitsTextStyle, vMin, graphPtr->bottom, &textDim);
		vMin += (textDim.width + SPACING);
	    }
	}
    }				/* Loop on axes */
}

void
Blt_AxisLimitsToPostScript(Graph *graphPtr, Blt_PostScript ps)
{
    Blt_HashEntry *hp;
    Blt_HashSearch cursor;
    double vMin, hMin, vMax, hMax;
    char string[200];

#define SPACING 8
    vMin = vMax = graphPtr->left + graphPtr->padLeft + 2;
    hMin = hMax = graphPtr->bottom - graphPtr->padBottom - 2;	/* Offsets */
    for (hp = Blt_FirstHashEntry(&graphPtr->axes.table, &cursor);
	 hp != NULL; hp = Blt_NextHashEntry(&cursor)) {
	Axis *axisPtr;
	char *minFmt, *maxFmt;
	int textWidth, textHeight;

	axisPtr = (Axis *)Blt_GetHashValue(hp);

	if (axisPtr->nFormats == 0) {
	    continue;
	}
	minFmt = maxFmt = axisPtr->limitsFormats[0];
	if (axisPtr->nFormats > 1) {
	    maxFmt = axisPtr->limitsFormats[1];
	}
	if (*maxFmt != '\0') {
	    sprintf(string, maxFmt, axisPtr->axisRange.max);
	    Blt_GetTextExtents(axisPtr->tickFont, 0, string, &textWidth,
		&textHeight);
	    if ((textWidth > 0) && (textHeight > 0)) {
		if (axisPtr->object.classId == OBJECT_CLASS_X_AXIS) {
		    Blt_TextSetAngle(axisPtr->limitsTextStyle, 90.0);
		    Blt_TextSetAnchor(axisPtr->limitsTextStyle, TK_ANCHOR_SE);
		    Blt_TextToPostScript(ps, string, &axisPtr->limitsTextStyle, 
			(double)graphPtr->right, hMax);
		    hMax -= (textWidth + SPACING);
		} else {
		    Blt_TextSetAngle(axisPtr->limitsTextStyle, 0.0);
		    Blt_TextSetAnchor(axisPtr->limitsTextStyle, TK_ANCHOR_NW);
		    Blt_TextToPostScript(ps, string, &axisPtr->limitsTextStyle,
			vMax, (double)graphPtr->top);
		    vMax += (textWidth + SPACING);
		}
	    }
	}
	if (*minFmt != '\0') {
	    sprintf(string, minFmt, axisPtr->axisRange.min);
	    Blt_GetTextExtents(axisPtr->tickFont, 0, string, &textWidth,
		&textHeight);
	    if ((textWidth > 0) && (textHeight > 0)) {
		Blt_TextSetAnchor(axisPtr->limitsTextStyle, TK_ANCHOR_SW);
		if (axisPtr->object.classId == OBJECT_CLASS_X_AXIS) {
		    Blt_TextSetAngle(axisPtr->limitsTextStyle, 90.0);
		    Blt_TextToPostScript(ps, string, &axisPtr->limitsTextStyle, 
			(double)graphPtr->left, hMin);
		    hMin -= (textWidth + SPACING);
		} else {
		    Blt_TextSetAngle(axisPtr->limitsTextStyle, 0.0);
		    Blt_TextToPostScript(ps, string, &axisPtr->limitsTextStyle, 
			vMin, (double)graphPtr->bottom);
		    vMin += (textWidth + SPACING);
		}
	    }
	}
    }
}

Axis *
Blt_GetFirstAxis(Blt_Chain *chainPtr)
{
    Blt_ChainLink *lp;

    lp = Blt_ChainFirstLink(chainPtr);
    if (lp == NULL) {
	return NULL;
    }
    return Blt_ChainGetValue(lp);
}

Axis *
Blt_NearestAxis(graphPtr, x, y)
    Graph *graphPtr;
    int x, y;                 /* Point to be tested */
{
    Blt_HashEntry *hp;
    Blt_HashSearch cursor;
    
    for (hp = Blt_FirstHashEntry(&graphPtr->axes.table, &cursor); hp != NULL; 
	 hp = Blt_NextHashEntry(&cursor)) {
	Axis *axisPtr;

	axisPtr = (Axis *)Blt_GetHashValue(hp);
	if ((axisPtr->object.hidden) || 
	    (!(axisPtr->flags & AXIS_ONSCREEN))) {
	    continue;		/* Don't check hidden axes or axes
				 * that are virtual. */
	}
	if (axisPtr->showTicks) {
	    Blt_ChainLink *lp;

	    for (lp = Blt_ChainFirstLink(axisPtr->tickLabels); lp != NULL; 
		 lp = Blt_ChainNextLink(lp)) {	
		TickLabel *labelPtr;
		Point2D t;
		double rWidth, rHeight;
		Point2D bbox[5];

		labelPtr = Blt_ChainGetValue(lp);
		Blt_GetBoundingBox(labelPtr->width, labelPtr->height, 
			axisPtr->tickAngle, &rWidth, &rHeight, bbox);
		t = Blt_AnchorPoint(labelPtr->anchorPos.x, 
			labelPtr->anchorPos.y, rWidth, rHeight, 
			axisPtr->tickAnchor);
		t.x = x - t.x - (rWidth * 0.5);
		t.y = y - t.y - (rHeight * 0.5);

		bbox[4] = bbox[0];
		if (Blt_PointInPolygon(&t, bbox, 5)) {
		    axisPtr->detail = "label";
		    return axisPtr;
		}
	    }
	}
	if (axisPtr->title != NULL) { /* and then the title string. */
	    Point2D bbox[5];
	    Point2D t;
	    double rWidth, rHeight;
	    int width, height;

	    Blt_GetTextExtents(axisPtr->titleFont, 0, axisPtr->title,
		&width, &height);
	    Blt_GetBoundingBox(width, height, axisPtr->titleAngle, &rWidth, 
		&rHeight, bbox);
	    t = Blt_AnchorPoint(axisPtr->titlePos.x, axisPtr->titlePos.y, 
		rWidth, rHeight, axisPtr->titleAnchor);
	    /* Translate the point so that the 0,0 is the upper left 
	     * corner of the bounding box.  */
	    t.x = x - t.x - (rWidth * 0.5);
	    t.y = y - t.y - (rHeight * 0.5);
	    
	    bbox[4] = bbox[0];
	    if (Blt_PointInPolygon(&t, bbox, 5)) {
		axisPtr->detail = "title";
		return axisPtr;
	    }
	}
	if (axisPtr->lineWidth > 0) { /* Check for the axis region */
	    if ((x <= axisPtr->right) && (x >= axisPtr->left) && 
		(y <= axisPtr->bottom) && (y >= axisPtr->top)) {
		axisPtr->detail = "line";
		return axisPtr;
	    }
	}
    }
    return NULL;
}
 
ClientData
Blt_MakeAxisTag(Graph *graphPtr, CONST char *tagName)
{
    Blt_HashEntry *hp;
    int isNew;

    hp = Blt_CreateHashEntry(&graphPtr->axes.tagTable, tagName, &isNew);
    assert(hp);
    return Blt_GetHashKey(&graphPtr->axes.tagTable, hp);
}

