
/*
 * bltPicturePng.c --
 *
 * This module implements PNG file format conversion routines for
 * the picture image type in the BLT toolkit.
 *
 *	Copyright 2003-2005 George A Howlett.
 *
 *	Permission is hereby granted, free of charge, to any person
 *	obtaining a copy of this software and associated documentation
 *	files (the "Software"), to deal in the Software without
 *	restriction, including without limitation the rights to use,
 *	copy, modify, merge, publish, distribute, sublicense, and/or
 *	sell copies of the Software, and to permit persons to whom the
 *	Software is furnished to do so, subject to the following
 *	conditions:
 *
 *	The above copyright notice and this permission notice shall be
 *	included in all copies or substantial portions of the
 *	Software.
 *
 *	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
 *	KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 *	WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *	PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
 *	OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 *	OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 *	OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 *	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */

#include "bltInt.h"

#ifdef HAVE_LIBPNG

#include <bltSink.h>
#include <bltHash.h>
#include "bltPicture.h"
#include "bltPictureFormats.h"

#include <X11/Xutil.h>

typedef struct Blt_PictureStruct Picture;

#include <png.h>

typedef struct {
    Tcl_DString errors;
    Tcl_DString warnings;
    int nErrors, nWarnings;
} PngMessage;

static void
PngError(
    png_struct *pngPtr, 
    const char *mesg)
{
    PngMessage *mesgPtr;

    mesgPtr = png_get_error_ptr(pngPtr);
    Tcl_DStringAppend(&mesgPtr->errors, mesg, -1);
    mesgPtr->nErrors++;
}

static void
PngWarning(
    png_structp png, 
    const char *mesg)
{
    PngMessage *mesgPtr;

    mesgPtr = png_get_error_ptr(png);
    Tcl_DStringAppend(&mesgPtr->warnings, mesg, -1);
    mesgPtr->nWarnings++;
}

static void
PngReadFromSink(
    png_structp png, 
    png_byte *out, 
    png_size_t nWanted)
{
    Blt_DataSink data;
    
    data = png_get_io_ptr(png);
    if (Blt_SinkBytesLeft(data) < nWanted) {
	nWanted = Blt_SinkBytesLeft(data);
    }
    if (nWanted > 0) {
	memcpy(out, Blt_SinkPointer(data), nWanted);
	Blt_SinkIncrCursor(data, nWanted);
    }
}

static void
PngWriteToSink(
    png_struct *pngPtr, 
    png_byte *out, 
    png_size_t nBytes)
{
    Blt_DataSink data;
    int need;

    data = png_get_io_ptr(pngPtr);
    need = Blt_SinkCursor(data) + nBytes;
    if (need > Blt_SinkLength(data)) {
	if (!Blt_SinkSetLength(data, need)) {
	    return;
	}
    }
    memcpy(Blt_SinkPointer(data), out, nBytes);
    Blt_SinkIncrCursor(data, nBytes);
    if (Blt_SinkCursor(data) > Blt_SinkMark(data)) {
	Blt_SinkSetMark(data, Blt_SinkCursor(data));
    }
}

static void
PngFlush(png_struct *pngPtr)
{
    /* Do nothing */
}


/*
 *----------------------------------------------------------------------
 *
 * Blt_IsPng --
 *
 *      Attempts to parse a PNG file header.
 *
 * Results:
 *      Returns 1 is the header is PNG and 0 otherwise.  Note that
 *      only the signature is compared and the validity of the image 
 *      contents is not checked here.  That's done in
 *      Blt_PngToPicture.
 *
 *----------------------------------------------------------------------
 */
int 
Blt_IsPng(Blt_DataSink data)
{
#define PNG_BYTES_TO_CHECK 4
    Blt_SinkResetCursor(data);
    if (Blt_SinkMark(data) > PNG_BYTES_TO_CHECK) {
	unsigned char *bp;
	int bool;

	bp = Blt_SinkPointer(data);
	bool = (png_sig_cmp(bp, (png_size_t)0, PNG_BYTES_TO_CHECK) == 0);
	return bool;
    }
    return FALSE;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_PngToPicture --
 *
 *      Reads a PNG file and converts it into a picture.
 *
 * Results:
 *      The picture is returned.  If an error occured, such
 *	as the designated file could not be opened, NULL is returned.
 *
 *----------------------------------------------------------------------
 */
Blt_Picture
Blt_PngToPicture(
    Tcl_Interp *interp, 
    char *fileName,
    Blt_DataSink data)
{
    Picture *destPtr;
    PngMessage message;
    int transform;
    png_infop info;
    png_structp png;
    unsigned int bitDepth, colorType;
    unsigned int interlace, filter, nChannels;
    unsigned int width, height;
    
    Tcl_SetErrorCode(interp, "", (char *)NULL);

    Tcl_DStringInit(&message.errors);
    Tcl_DStringInit(&message.warnings);
    message.nErrors = message.nWarnings = 0;

    Tcl_DStringAppend(&message.errors, "error reading \"", -1);
    Tcl_DStringAppend(&message.errors, fileName, -1);
    Tcl_DStringAppend(&message.errors, "\": ", -1);
    png = png_create_read_struct(PNG_LIBPNG_VER_STRING, &message, PngError, 
	PngWarning);
    if (png == NULL) {
	return NULL;
    }

    destPtr = NULL;		/* Mark to indicate failure. */

    info = png_create_info_struct(png);
    if (info == NULL) {
	goto bad;
    }
    if (setjmp(png->jmpbuf)) {
	goto bad;
    }
    png_set_read_fn(png, data, PngReadFromSink);

    transform = (PNG_TRANSFORM_EXPAND | /* Expand 1,2 and 4 bit to 8 bits. */
		 PNG_TRANSFORM_STRIP_16); /* Convert 16-bit components to 8. */

    png_read_png(png, info, transform, (void *)NULL);

    bitDepth =  png_get_bit_depth(png, info);
    colorType = png_get_color_type(png, info);
    filter =    png_get_filter_type(png, info);
    height =    png_get_image_height(png, info);
    interlace = png_get_interlace_type(png, info);
    nChannels = png_get_channels(png, info);
    width =     png_get_image_width(png, info);
    
#ifdef notdef
    fprintf(stderr, "%s: %dx%d, bit_depth=%d, channels=%d, interlace=%d\n",
	    fileName, width, height, bitDepth, nChannels, interlace);
    fprintf(stderr, "colortype= %s %s %s \n",
	    (colorType & 1) ? "palette" : "",
	    (colorType & 2) ? "color"   : "",
	    (colorType & 4) ? "alpha"   : "");
#endif

    if (colorType & PNG_COLOR_MASK_ALPHA) {
	assert((nChannels == 4) || (nChannels == 2));
    }
    destPtr = Blt_CreatePicture(width, height);
    if (destPtr == NULL) {
	return NULL;
    }
    {
	int y;
	Pix32 *destRowPtr;
	png_byte **row_pointers;

	destRowPtr = destPtr->bits;
	row_pointers = png_get_rows(png, info);
	switch (nChannels) {
	case 4:			/* 1 red, 1 green, 1 blue, 1 alpha */
	    for (y = 0; y < height; y++) {
		Pix32 *dp;
		png_byte *sp;
		int x;
		
		dp = destRowPtr;
		sp = row_pointers[y];
		for (x = 0; x < width; x++) {
		    dp->Red = sp[0];
		    dp->Green = sp[1];
		    dp->Blue = sp[2];
		    dp->Alpha = sp[3];
		    dp++, sp += 4;
		}
		destRowPtr += destPtr->pixelsPerRow;
	    }
	    destPtr->flags |= (BLT_PICTURE_BLEND | BLT_PICTURE_COLOR);
	    break;
	case 3:			/* 1 red, 1 green, 1 blue */
	    for (y = 0; y < height; y++) {
		Pix32 *dp;
		unsigned char *sp;
		int x;
		
		dp = destRowPtr;
		sp = row_pointers[y];
		for (x = 0; x < width; x++) {
		    dp->Red = sp[0];
		    dp->Green = sp[1];
		    dp->Blue = sp[2];
		    dp->Alpha = ALPHA_OPAQUE;
		    dp++, sp += 3;
		}
		destRowPtr += destPtr->pixelsPerRow;
	    }
	    destPtr->flags |= BLT_PICTURE_COLOR;
	    break;
	case 2:			/* 1 greyscale, 1 alpha */
	    for (y = 0; y < height; y++) {
		Pix32 *dp;
		unsigned char *sp;
		int x;
		
		dp = destRowPtr;
		sp = row_pointers[y];
		for (x = 0; x < width; x++) {
		    dp->Red = dp->Green = dp->Blue = sp[0];
		    dp->Alpha = sp[1];
		    dp++, sp += 2;
		}
		destRowPtr += destPtr->pixelsPerRow;
	    }
	    destPtr->flags |= BLT_PICTURE_BLEND;
	    break;
	case 1:			/* 1 greyscale */
	    for (y = 0; y < height; y++) {
		Pix32 *dp;
		unsigned char *sp;
		int x;
		
		dp = destRowPtr;
		sp = row_pointers[y];
		for (x = 0; x < width; x++) {
		    dp->Red = dp->Green = dp->Blue = *sp++;
		    dp->Alpha = ALPHA_OPAQUE;
		    dp++;
		}
		destRowPtr += destPtr->pixelsPerRow;
	    }
	    break;
	}
    }
 bad:
    png_destroy_read_struct(&png, &info, (png_infop *)NULL);
    if (message.nWarnings > 0) {
	Tcl_SetErrorCode(interp, "PICTURE", "PNG_READ_WARNINGS", 
		Tcl_DStringValue(&message.errors), (char *)NULL);
    }
    Tcl_DStringFree(&message.warnings);
    if (message.nErrors > 0) {
	Tcl_DStringResult(interp, &message.errors);
    }
    Tcl_DStringFree(&message.errors);
    return destPtr;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_PictureToPng --
 *
 *      Writes a PNG format image to the provided data buffer.
 *
 * Results:
 *      A standard Tcl result.  If an error occured, TCL_ERROR is
 *	returned and an error message will be place in the interpreter
 *	result. Otherwise, the data sink will contain the binary
 *	output of the image.
 *
 * Side Effects:
 *	Memory is allocated for the data sink.
 *
 *----------------------------------------------------------------------
 */
int
Blt_PictureToPng(
    Tcl_Interp *interp, 
    Blt_Picture picture,
    Blt_DataSink data)
{
    PngMessage message;
    int nColors;
    png_infop info;
    png_structp png;
    unsigned int colorType, nChannels, bitsPerPixel;

    Tcl_SetErrorCode(interp, "", (char *)NULL);

    Tcl_DStringInit(&message.errors);
    Tcl_DStringInit(&message.warnings);
    message.nErrors = message.nWarnings = 0;

    Tcl_DStringAppend(&message.errors, "error writing PNG output: ", -1);
    png = png_create_write_struct(PNG_LIBPNG_VER_STRING, &message, PngError, 
	PngWarning);
    if (png == NULL) {
	return TCL_ERROR;
    }
    info = png_create_info_struct(png);
    if (info == NULL) {
	png_destroy_write_struct(&png, (png_infop *)NULL);
	return TCL_ERROR;
    }
    if (setjmp(png->jmpbuf)) {
	goto bad;
    }
    png_set_write_fn(png, (void *)data, PngWriteToSink, PngFlush);

    png_set_compression_level(png, Z_BEST_COMPRESSION);
    nColors = Blt_QueryColors(picture, (Blt_HashTable *)NULL);
    if (Blt_PictureIsColor(picture)) {
	nChannels = 3;
	colorType = PNG_COLOR_TYPE_RGB;
    } else {
	nChannels = 1;
	colorType = PNG_COLOR_TYPE_GRAY;
    }
    if (!Blt_PictureIsOpaque(picture)) {
	nChannels++;
	colorType |= PNG_COLOR_MASK_ALPHA;
    }
    bitsPerPixel = 8;

    png_set_IHDR(png, info, 
	Blt_PictureWidth(picture), /* Width */
	Blt_PictureHeight(picture), /* Height */
	bitsPerPixel, 		/* Bits per pixel. */
	colorType, 		/* Color type: RGB or GRAY */
	PNG_INTERLACE_NONE, 	/* Interlace */
	PNG_COMPRESSION_TYPE_DEFAULT, /* Compression */
	PNG_FILTER_TYPE_DEFAULT); /* Filter */

    png_write_info(png, info);
    png_set_packing(png);

    {
	Picture *srcPtr;
	Pix32 *srcRowPtr;
	int bytesPerRow;
	int y;
	unsigned char **rowArray;
	unsigned char *destBuffer, *destRowPtr;
	
	srcPtr = picture;
	bytesPerRow = nChannels * srcPtr->width;
	destBuffer = Blt_Malloc(bytesPerRow * srcPtr->height);
	if (destBuffer == NULL) {
	    goto bad;
	}
	rowArray = Blt_Malloc(sizeof(unsigned char *) * srcPtr->height);
	if (rowArray == NULL) {
	    Blt_Free(destBuffer);
	    goto bad;
	}
	destRowPtr = destBuffer;
	srcRowPtr = srcPtr->bits;
	switch (nChannels) {
	case 4:			/* RGBA */
	    for (y = 0; y < srcPtr->height; y++) {
		Pix32 *sp;
		unsigned char *dp;
		int x;
		
		sp = srcRowPtr;
		dp = destRowPtr;
		rowArray[y] = destRowPtr;
		for (x = 0; x < srcPtr->width; x++) {
		    dp[0] = sp->Red;
		    dp[1] = sp->Green;
		    dp[2] = sp->Blue;
		    dp[3] = sp->Alpha;
		    sp++, dp += 4;
		}
		destRowPtr += bytesPerRow;
		srcRowPtr += srcPtr->pixelsPerRow;
	    }
	    break;
	    
	case 3:			/* RGB, 100% opaque image.  */
	    for (y = 0; y < srcPtr->height; y++) {
		Pix32 *sp;
		unsigned char *dp;
		int x;
		
		sp = srcRowPtr;
		dp = destRowPtr;
		rowArray[y] = destRowPtr;
		for (x = 0; x < srcPtr->width; x++) {
		    dp[0] = sp->Red;
		    dp[1] = sp->Green;
		    dp[2] = sp->Blue;
		    sp++, dp += 3;
		}
		destRowPtr += bytesPerRow;
		srcRowPtr += srcPtr->pixelsPerRow;
	    }
	    break;
	    
	case 2:			/* Greyscale with alpha component.  */
	    for (y = 0; y < srcPtr->height; y++) {
		Pix32 *sp, *send;
		unsigned char *dp;
		
		sp = srcRowPtr, dp = destRowPtr;
		for (send = sp + srcPtr->width; sp < send; sp++) {
		    dp[0] = sp->Red;
		    dp[1] = sp->Alpha;
		    dp += 2;
		}
		rowArray[y] = destRowPtr;
		destRowPtr += bytesPerRow;
		srcRowPtr += srcPtr->pixelsPerRow;
	    }
	    break;
	    
	case 1:			/* Greyscale, 100% opaque image.  */
	    for (y = 0; y < srcPtr->height; y++) {
		Pix32 *sp;
		unsigned char *dp;
		int x;
		
		sp = srcRowPtr;
		dp = destRowPtr;
		rowArray[y] = destRowPtr;
		for (x = 0; x < srcPtr->width; x++) {
		    *dp++ = sp->Red;
		    sp++;
		}
		destRowPtr += bytesPerRow;
		srcRowPtr += srcPtr->pixelsPerRow;
	    }
	    break;
	}
	png_write_image(png, rowArray);
	png_write_end(png, NULL);

	Blt_Free(destBuffer);
	Blt_Free(rowArray);
    }

 bad:
    png_destroy_write_struct(&png, &info);
    if (message.nWarnings > 0) {
	Tcl_SetErrorCode(interp, "PICTURE", "PNG_WRITE_WARNINGS", 
		Tcl_DStringValue(&message.errors), (char *)NULL);
    }
    Tcl_DStringFree(&message.warnings);
    if (message.nErrors > 0) {
	Tcl_DStringResult(interp, &message.errors);
	return TCL_ERROR;
    } 
    Tcl_DStringFree(&message.errors);
    return TCL_OK;
}

#endif /* HAVE_LIBPNG */

