
/*
 * bltSwitch.c --
 *
 * This module implements command/argument switch parsing procedures
 * for the BLT toolkit.
 *
 *	Copyright 1991-2004 George A Howlett.
 *
 *	Permission is hereby granted, free of charge, to any person
 *	obtaining a copy of this software and associated documentation
 *	files (the "Software"), to deal in the Software without
 *	restriction, including without limitation the rights to use,
 *	copy, modify, merge, publish, distribute, sublicense, and/or
 *	sell copies of the Software, and to permit persons to whom the
 *	Software is furnished to do so, subject to the following
 *	conditions:
 *
 *	The above copyright notice and this permission notice shall be
 *	included in all copies or substantial portions of the
 *	Software.
 *
 *	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
 *	KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 *	WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *	PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
 *	OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 *	OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 *	OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 *	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "bltInt.h"
#include <stdarg.h>

#include "bltSwitch.h"

/*
 *--------------------------------------------------------------
 *
 * FindSwitchSpec --
 *
 *	Search through a table of configuration specs, looking for
 *	one that matches a given argvName.
 *
 * Results:
 *	The return value is a pointer to the matching entry, or NULL
 *	if nothing matched.  In that case an error message is left
 *	in the interp's result.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static Blt_SwitchSpec *
FindSwitchSpec(
    Tcl_Interp *interp,		/* Used for reporting errors. */
    Blt_SwitchSpec *specs,	/* Pointer to table of configuration
				 * specifications for a widget. */
    char *name,			/* Name identifying a particular switch. */
    int needFlags,		/* Flags that must be present in matching
				 * entry. */
    int hateFlags)		/* Flags that must NOT be present in
				 * matching entry. */
{
    Blt_SwitchSpec *sp;
    char c;			/* First character of current argument. */
    Blt_SwitchSpec *matchPtr;	/* Matching spec, or NULL. */
    size_t length;

    c = name[1];
    length = strlen(name);
    matchPtr = NULL;
    
    for (sp = specs; sp->type != BLT_SWITCH_END; sp++) {
	if (sp->switchName == NULL) {
	    continue;
	}
	if (((sp->flags & needFlags) != needFlags) || (sp->flags & hateFlags)) {
	    continue;
	}
	if ((sp->switchName[1] != c) || 
	    (strncmp(sp->switchName, name, length) != 0)) {
	    continue;
	}
	if (sp->switchName[length] == '\0') {
	    return sp;		/* Stop on a perfect match. */
	}
	if (matchPtr != NULL) {
	    Tcl_AppendResult(interp, "ambiguous switch \"", name, "\"", 
		(char *) NULL);
	    return (Blt_SwitchSpec *) NULL;
	}
	matchPtr = sp;
    }
    if (matchPtr == NULL) {
	Tcl_AppendResult(interp, "unknown switch \"", name, "\"", (char *)NULL);
	return (Blt_SwitchSpec *) NULL;
    }
    return matchPtr;
}

/*
 *--------------------------------------------------------------
 *
 * DoSwitch --
 *
 *	This procedure applies a single configuration switch
 *	to a widget record.
 *
 * Results:
 *	A standard Tcl return value.
 *
 * Side effects:
 *	WidgRec is modified as indicated by specPtr and value.
 *	The old value is recycled, if that is appropriate for
 *	the value type.
 *
 *--------------------------------------------------------------
 */
static int
DoSwitch(
    Tcl_Interp *interp,		/* Interpreter for error reporting. */
    Blt_SwitchSpec *sp,		/* Specifier to apply. */
    Tcl_Obj *objPtr,		/* Value to use to fill in widgRec. */
    void *record)		/* Record whose fields are to be
				 * modified.  Values must be properly
				 * initialized. */
{
    do {
	char *ptr;

	ptr = (char *)record + sp->offset;
	switch (sp->type) {
	case BLT_SWITCH_BOOLEAN:
	    if (Tcl_GetBooleanFromObj(interp, objPtr, (int *)ptr) != TCL_OK) {
		return TCL_ERROR;
	    }
	    break;

	case BLT_SWITCH_DOUBLE:
	    if (Tcl_GetDoubleFromObj(interp, objPtr, (double *)ptr) != TCL_OK) {
		return TCL_ERROR;
	    }
	    break;

	case BLT_SWITCH_OBJ:
	    *(Tcl_Obj **)ptr = objPtr;
	    break;

	case BLT_SWITCH_FLOAT:
	    {
		double value;

		if (Tcl_GetDoubleFromObj(interp, objPtr, &value) != TCL_OK) {
		    return TCL_ERROR;
		}
		*(float *)ptr = (float)value;
	    }
	    break;

	case BLT_SWITCH_INT:
	    if (Tcl_GetIntFromObj(interp, objPtr, (int *)ptr) != TCL_OK) {
		return TCL_ERROR;
	    }
	    break;

	case BLT_SWITCH_INT_NONNEGATIVE:
	    {
		long value;
		
		if (Blt_GetCountFromObj(interp, objPtr, COUNT_NONNEGATIVE, 
			&value) != TCL_OK) {
		    return TCL_ERROR;
		}
		*(int *)ptr = (int)value;
	    }
	    break;

	case BLT_SWITCH_INT_POSITIVE:
	    {
		long value;
		
		if (Blt_GetCountFromObj(interp, objPtr, COUNT_POSITIVE, 
			&value) != TCL_OK) {
		    return TCL_ERROR;
		}
		*(int *)ptr = (int)value;
	    }
	    break;

	case BLT_SWITCH_LIST:
	    {
		int argc;

		if (Tcl_SplitList(interp, Tcl_GetString(objPtr), &argc, 
				  (char ***)ptr) != TCL_OK) {
		    return TCL_ERROR;
		}
	    }
	    break;

	case BLT_SWITCH_LONG:
	    if (Tcl_GetLongFromObj(interp, objPtr, (long *)ptr) != TCL_OK) {
		return TCL_ERROR;
	    }
	    break;

	case BLT_SWITCH_LONG_NONNEGATIVE:
	    {
		long value;
		
		if (Blt_GetCountFromObj(interp, objPtr, COUNT_NONNEGATIVE, 
			&value) != TCL_OK) {
		    return TCL_ERROR;
		}
		*(long *)ptr = value;
	    }
	    break;

	case BLT_SWITCH_LONG_POSITIVE:
	    {
		long value;
		
		if (Blt_GetCountFromObj(interp, objPtr, COUNT_POSITIVE, &value)
			!= TCL_OK) {
		    return TCL_ERROR;
		}
		*(long *)ptr = value;
	    }
	    break;

	case BLT_SWITCH_STRING: 
	    {
		char *old, *new, **strPtr;
		char *string;

		string = Tcl_GetString(objPtr);
		strPtr = (char **)ptr;
		new = ((*string == '\0') && (sp->flags & BLT_SWITCH_NULL_OK))
		    ? NULL : Blt_Strdup(string);
		old = *strPtr;
		if (old != NULL) {
		    Blt_Free(old);
		}
		*strPtr = new;
	    }
	    break;

	case BLT_SWITCH_CUSTOM:
	    if ((*sp->customPtr->parseProc)(sp->customPtr->clientData, interp,
		sp->switchName, objPtr, (char *)record, sp->offset, sp->flags) 
		!= TCL_OK) {
		return TCL_ERROR;
	    }
	    break;

	default: 
	    Tcl_AppendResult(interp, "bad switch table: unknown type \"",
		 Blt_Itoa(sp->type), "\"", (char *)NULL);
	    return TCL_ERROR;
	}
	sp++;
    } while ((sp->switchName == NULL) && (sp->type != BLT_SWITCH_END));
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * Blt_ParseSwitches --
 *
 *	Process command-line switches to fill in fields of a record
 *	with resources and other parameters.
 *
 * Results:
 *	Returns the number of arguments comsumed by parsing the
 *	command line.  If an error occurred, -1 will be returned
 *	and an error messages can be found as the interpreter
 *	result.
 *
 * Side effects:
 *	The fields of widgRec get filled in with information from
 *	argc/argv.  Old information in widgRec's fields gets recycled.
 *
 *--------------------------------------------------------------
 */
int
Blt_ParseSwitches(
    Tcl_Interp *interp,		/* Interpreter for error reporting. */
    Blt_SwitchSpec *specs,	/* Describes legal switches. */
    int objc,			/* Number of elements in argv. */
    Tcl_Obj *CONST *objv,	/* Command-line switches. */
    void *record,		/* Record whose fields are to be
				 * modified.  Values must be properly
				 * initialized. */
    int flags)			/* Used to specify additional flags
				 * that must be present in switch specs
				 * for them to be considered.  */
{
    Blt_SwitchSpec *sp;
    int count;
    int needFlags;		/* Specs must contain this set of flags
				 * or else they are not considered. */
    int hateFlags;		/* If a spec contains any bits here, it's
				 * not considered. */

    needFlags = flags & ~(BLT_SWITCH_USER_BIT - 1);
    hateFlags = 0;

    /*
     * Pass 1:  Clear the change flags on all the specs so that we 
     *          can check it later.
     */
    for (sp = specs; sp->type != BLT_SWITCH_END; sp++) {
	sp->flags &= ~BLT_SWITCH_SPECIFIED;
    }
    /*
     * Pass 2:  Process the arguments that match entries in the specs.
     *		It's an error if the argument doesn't match anything.
     */
    for (count = 0; count < objc; count++) {
	char *arg;

	arg = Tcl_GetString(objv[count]);
	if (flags & BLT_SWITCH_OBJV_PARTIAL) {
	    /* 
	     * If the argument doesn't start with a '-' (not a switch)
	     * or is '--', stop processing and return the number of
	     * arguments comsumed. 
	     */
	    if (arg[0] != '-') {
		return count;
	    }
	    if ((arg[1] == '-') && (arg[2] == '\0')) {
		return count + 1; /* include the "--" in the count. */
	    }
	}
	sp = FindSwitchSpec(interp, specs, arg, needFlags, hateFlags);
	if (sp == NULL) {
	    return -1;
	}
	if (sp->type == BLT_SWITCH_FLAG) {
	    char *ptr;
	    
	    ptr = (char *)record + sp->offset;
	    *((int *)ptr) |= sp->value;
	} else if (sp->type == BLT_SWITCH_VALUE) {
	    char *ptr;
	    
	    ptr = (char *)record + sp->offset;
	    *((int *)ptr) = sp->value;
	} else {
	    count++;
	    if (count == objc) {
		Tcl_AppendResult(interp, "value for \"", arg, "\" missing", 
				 (char *) NULL);
		return -1;
	    }
	    if (DoSwitch(interp, sp, objv[count], record) != TCL_OK) {
		char msg[100];

		sprintf(msg, "\n    (processing \"%.40s\" switch)", 
			sp->switchName);
		Tcl_AddErrorInfo(interp, msg);
		return -1;
	    }
	}
	sp->flags |= BLT_SWITCH_SPECIFIED;
    }
    return count;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_FreeSwitches --
 *
 *	Free up all resources associated with switches.
 *
 * Results:
 *	None.
 *
 *----------------------------------------------------------------------
 */
/* ARGSUSED */
void
Blt_FreeSwitches(
    Blt_SwitchSpec *specs,	/* Describes legal switches. */
    void *record,		/* Record whose fields contain current
				 * values for switches. */
    int needFlags)		/* Used to specify additional flags
				 * that must be present in config specs
				 * for them to be considered. */
{
    Blt_SwitchSpec *sp;

    for (sp = specs; sp->type != BLT_SWITCH_END; sp++) {
	if ((sp->flags & needFlags) == needFlags) {
	    char *ptr;

	    ptr = (char *)record + sp->offset;
	    switch (sp->type) {
	    case BLT_SWITCH_STRING:
	    case BLT_SWITCH_LIST:
		if (*((char **) ptr) != NULL) {
		    Blt_Free(*((char **) ptr));
		    *((char **) ptr) = NULL;
		}
		break;

	    case BLT_SWITCH_CUSTOM:
		if ((*(char **)ptr != NULL) && 
		    (sp->customPtr->freeProc != NULL)) {
		    (*sp->customPtr->freeProc)((char *)record, sp->offset, 
			sp->flags);
		}
		break;

	    default:
		break;
	    }
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_SwitchModified --
 *
 *      Given the configuration specifications and one or more switch
 *      patterns (terminated by a NULL), indicate if any of the
 *      matching switches has been reset.
 *
 * Results:
 *      Returns 1 if one of the switches have changed, 0 otherwise.
 *
 *----------------------------------------------------------------------
 */
int 
Blt_SwitchChanged TCL_VARARGS_DEF(Blt_SwitchSpec *, arg1)
{
    va_list argList;
    Blt_SwitchSpec *specs;
    Blt_SwitchSpec *sp;
    char *switchName;

    specs = TCL_VARARGS_START(Blt_SwitchSpec *, arg1, argList);
    while ((switchName = va_arg(argList, char *)) != NULL) {
	for (sp = specs; sp->type != BLT_SWITCH_END; sp++) {
	    if ((Tcl_StringMatch(sp->switchName, switchName)) &&
		(sp->flags & BLT_SWITCH_SPECIFIED)) {
		va_end(argList);
		return 1;
	    }
	}
    }
    va_end(argList);
    return 0;
}

int 
Blt_ExprDoubleFromObj(Tcl_Interp *interp, Tcl_Obj *objPtr, double *valuePtr)
{
    /* First try to extract the value as a double precision number. */
    if (Tcl_GetDoubleFromObj((Tcl_Interp *)NULL, objPtr, valuePtr) == TCL_OK) {
	return TCL_OK;
    }
    /* Then try to parse it as an expression. */
    if (Tcl_ExprDouble(interp, Tcl_GetString(objPtr), valuePtr) == TCL_OK) {
	return TCL_OK;
    }
    return TCL_ERROR;
}

int 
Blt_ExprIntFromObj(
    Tcl_Interp *interp, 
    Tcl_Obj *objPtr, 
    int *valuePtr)
{
    long lvalue;

    /* First try to extract the value as a simple integer. */
    if (Tcl_GetIntFromObj((Tcl_Interp *)NULL, objPtr, valuePtr) == TCL_OK) {
	return TCL_OK;
    }
    /* Otherwise try to parse it as an expression. */
    if (Tcl_ExprLong(interp, Tcl_GetString(objPtr), &lvalue) == TCL_OK) {
	*valuePtr = lvalue;
	return TCL_OK;
    }
    return TCL_ERROR;
}

int
Blt_ObjIsEmpty(Tcl_Obj *objPtr)
{
    int length;

    if (objPtr == NULL) {
	return TRUE;
    } 
    if (objPtr->bytes != NULL) {
	length = objPtr->length;
    } else {
	Tcl_GetStringFromObj(objPtr, &length);
    }
    return (length == 0);
}
