
/*
 * bltText.h --
 *
 *
 *	Copyright 1993-2004 George A Howlett.
 *
 *	Permission is hereby granted, free of charge, to any person
 *	obtaining a copy of this software and associated documentation
 *	files (the "Software"), to deal in the Software without
 *	restriction, including without limitation the rights to use,
 *	copy, modify, merge, publish, distribute, sublicense, and/or
 *	sell copies of the Software, and to permit persons to whom the
 *	Software is furnished to do so, subject to the following
 *	conditions:
 *
 *	The above copyright notice and this permission notice shall be
 *	included in all copies or substantial portions of the
 *	Software.
 *
 *	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY
 *	KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 *	WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *	PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS
 *	OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
 *	OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 *	OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 *	SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#ifndef _BLT_TEXT_H
#define _BLT_TEXT_H

#define DEF_TEXT_FLAGS (TK_PARTIAL_OK | TK_IGNORE_NEWLINES)
#define UPDATE_GC	1

/*
 * ----------------------------------------------------------------------
 *
 * TextFragment --
 *
 * ----------------------------------------------------------------------
 */
typedef struct {
    char *text;			/* Text to be displayed */

    short int x, y;		/* X-Y offset of the baseline from the
				 * upper-left corner of the bbox. */

    short int sx, sy;		/* See bltWinUtil.c */

    unsigned short int count;	/* Number of bytes in text. The actual
				 * character count may differ because of
				 * multi-byte UTF encodings. */

    unsigned short int width;	/* Width of segment in pixels. This
				 * information is used to draw
				 * PostScript strings the same width
				 * as X. */
} TextFragment;

/*
 * ----------------------------------------------------------------------
 *
 * TextLayout --
 *
 * ----------------------------------------------------------------------
 */
typedef struct {
    int nFrags;			/* # fragments of text */
    short int width, height;	/* Dimensions of text bounding box */
    TextFragment fragments[1];	/* Information about each fragment of text */
} TextLayout;

/*
 * ----------------------------------------------------------------------
 *
 * TextStyle --
 *
 * 	A somewhat convenient structure to hold text attributes that
 * 	determine how a text string is to be displayed on the screen
 * 	or drawn with PostScript commands.  The alternative is to pass
 * 	lots of parameters to the drawing and printing routines. This
 * 	seems like a more efficient and less cumbersome way of
 * 	passing parameters.
 *
 * ----------------------------------------------------------------------
 */
typedef struct {
    unsigned int state;		/* If non-zero, indicates to draw text
				 * in the active color */
    XColor *color;		/* Color to draw the text. */
    Blt_Font font;		/* Font to use to draw text */
    Tk_3DBorder border;		/* Background color of text.  This is also
				 * used for drawing disabled text. */
    double angle;		/* Rotation of text in degrees. */
    Tk_Justify justify;		/* Justification of the text string. This
				 * only matters if the text is composed
				 * of multiple lines. */
    Tk_Anchor anchor;		/* Indicates how the text is anchored around
				 * its x,y coordinates. */
    Blt_Pad padX, padY;		/* # pixels padding of around text region. */
    unsigned short int leader;	/* # pixels spacing between lines of text. */

    /* Private fields. */
    unsigned short int flags;
    GC gc;			/* GC used to draw the text */
} TextStyle;


extern TextLayout *Blt_TextCreateLayout(char *string, TextStyle *tsPtr);

extern void Blt_TextDrawLayout(Tk_Window tkwin, Drawable drawable, 
	TextLayout *textPtr, TextStyle *tsPtr, int x, int y, int xMax);

extern void Blt_TextGetExtents(TextStyle *tsPtr, char *text, int *widthPtr, 
	int *heightPtr);

extern void Blt_TextResetStyle(Tk_Window tkwin, TextStyle *tsPtr);

extern void Blt_TextFreeStyle(Display *display, TextStyle *tsPtr);

extern void Blt_TextSetDrawStyle (TextStyle *tsPtr, Blt_Font font, GC gc, 
	XColor *fgColor, double angle, Tk_Anchor anchor, Tk_Justify justify, 
	int leader);

extern void Blt_TextSetPrintStyle(TextStyle *tsPtr, Blt_Font font, 
	XColor *fgColor, XColor *bgColor, double angle, Tk_Anchor anchor, 
	Tk_Justify justify, int leader);

extern void Blt_DrawText(Tk_Window tkwin, Drawable drawable, char *string, 
	TextStyle *tsPtr, int x, int y);

extern void Blt_DrawText2(Tk_Window tkwin, Drawable drawable, char *string, 
	TextStyle *tsPtr, int x, int y, Dim2D * dimPtr);

extern Pixmap Blt_TextToBitmap(Tk_Window tkwin, TextLayout *textPtr, 
	TextStyle *tsPtr, int *widthPtr, int *heightPtr, int xMax);

extern int Blt_DrawTextWithRotatedFont(Tk_Window tkwin, Drawable drawable, 
	double angle, TextStyle *tsPtr, TextLayout *textPtr, int x, int y,
	int xMax);

extern void Blt_GetTextExtents(Blt_Font font, int leader, CONST char *string, 
	int *widthPtr, int *heightPtr);

#define Blt_TextGetAnchor(ts)		((ts).anchor)
#define Blt_TextGetAngle(ts)		((ts).angle)
#define Blt_TextGetBorder(ts)		((ts).border)
#define Blt_TextGetFont(ts)		((ts).font)
#define Blt_TextGetForeground(ts)	((ts).color)
#define Blt_TextGetJustify(ts)		((ts).justify)
#define Blt_TextGetLeader(ts)		((ts).leader)

#define Blt_TextSetAnchor(ts, a)	((ts).anchor = (a))
#define Blt_TextSetAngle(ts, r)		((ts).angle = (float)(r))
#define Blt_TextSetBorder(ts, b)	((ts).border = (b))
#define Blt_TextSetFont(ts, f)		\
	(((ts).font != (f)) ? ((ts).font = (f), (ts).flags |= UPDATE_GC) : 0)
#define Blt_TextSetForeground(ts, c)    \
	(((ts).color != (c)) ? ((ts).color = (c), (ts).flags |= UPDATE_GC) : 0)
#define Blt_TextSetJustify(ts, j)	((ts).justify = (j))
#define Blt_TextSetLeader(ts, l)	((ts).leader = (l))
#define Blt_TextSetPadding(ts, l, r, t, b)    \
	((ts).padX.side1 = (l), \
	(ts).padX.side2 = (r),  \
	(ts).padY.side1 = (t),  \
	(ts).padY.side2 = (b))
#define Blt_TextSetState(ts, s)		((ts).state = (s))

#define Blt_TextInitStyle(ts)		\
	((ts).anchor = TK_ANCHOR_NW, \
	(ts).color = (XColor *)NULL, \
	(ts).font = NULL, \
	(ts).justify = TK_JUSTIFY_LEFT, \
	(ts).leader = 0, \
	(ts).padX.side1 = (ts).padX.side2 = 0, \
	(ts).padY.side1 = (ts).padY.side2 = 0, \
	(ts).state = 0, \
	(ts).flags = 0, \
        (ts).gc = NULL, \
	(ts).angle = 0.0)

#endif /* _BLT_TEXT_H */
