/*
 * t4init.cpp --
 *
 *	Implementation of the T4Graph initialization procedures.
 *
 *	Authors: Jacob Levy and Jean-Claude Wippler.
 *		 jyl@best.com	jcw@equi4.com
 *
 * Copyright (c) 2000-2003, JYL Software Inc.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE, EVEN IF
 * JYL SOFTWARE INC. IS MADE AWARE OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "t4graphrep.h"

/*
 * The following code was contributed by David Gravereux. It deals
 * with making t4graph stub compliant on Win32 systems automatically.
 */

#ifdef _MSC_VER
	/*
	 * Only do this when MSVC++ is compiling us.
	 */
#   ifdef USE_TCL_STUBS
	/*
	 * Mark this .obj as needing tcl's Stubs library.
	 */
#	pragma comment(lib, "tclstub" \
		STRINGIFY(JOIN(TCL_MAJOR_VERSION,TCL_MINOR_VERSION)) ".lib")
#	if !defined(_MT) || !defined(_DLL) || defined(_DEBUG)
	/*
	 * This fixes a bug with how the Stubs library was compiled.
	 * The requirement for msvcrt.lib from tclstubXX.lib should
	 * be removed.
	 */
#	    pragma comment(linker, "-nodefaultlib:msvcrt.lib")
#	endif
#   else
	/*
	 * Mark this .obj needing the import library
	 */
#	pragma comment(lib, "tcl" \
		STRINGIFY(JOIN(TCL_MAJOR_VERSION,TCL_MINOR_VERSION)) ".lib")
#   endif
#endif

/*
 * Tgraph_RealInit --
 *
 *	This procedure initializes the T4Graph package in an interpreter.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	Creates a "t4graph" namespace and several commands within the
 *	namespace in the given interpreter.
 */

extern "C" int
Tgraph_RealInit(Tcl_Interp *interp)
{
#ifdef USE_TCL_STUBS
    if (Tcl_InitStubs(interp, "8.1", 0) == NULL) {
	return TCL_ERROR;
    }
#endif

    /*
     * Initialize the GenObj package.
     */

    Genobj_Init(interp);

    /*
     * Set up exit handler so that all storages get closed on exit.
     * This can be called multiple times, it does nothing the second
     * and subsequent times it is called.
     */

    T4Graph_SetupExitHandler();

    /*
     * Ensure the interpreter is set up to close all storages that
     * were registered in it, when the interpreter is destroyed.
     */

    T4Graph_RegisterInterp(interp);

    /*
     * If the interpreter is not safe, create the "::tgraph::open" command:
     */

    if (!Tcl_IsSafe(interp)) {
	(void) Tcl_CreateObjCommand(interp,
				    (char *) "::tgraph::open",
				    T4Graph_OpenStorageProc,
				    (ClientData) NULL,
				    (Tcl_CmdDeleteProc *) NULL);
    }

    /*
     * Create the "::tgraph::event" command:
     */

    (void) Tcl_CreateObjCommand(interp,
				(char *) "::tgraph::event",
				T4Graph_EventProc,
				(ClientData) NULL,
				(Tcl_CmdDeleteProc *) NULL);

    /*
     * Create the "::tgraph::version" command:
     */

    (void) Tcl_CreateObjCommand(interp,
				(char *) "::tgraph::version",
				T4Graph_VersionProc,
				(ClientData) NULL,
				(Tcl_CmdDeleteProc *) NULL);

    /*
     * All went well, so declare that we provide the T4Graph package.
     */

    return Tcl_PkgProvide(interp, (char *) "tgraph", (char *) "1.0.0");
}
