#
# artclsCommon.t
#

# server related data
set Server(port) 	119; # as per RFC 977
set Server(in) 		_bogusFile
set Server(out)		_bogusFile
set Server(postingallowed) 0

###
### ---- nntp client functions
###

# disconnect from the nntp server
proc nntpDisconnect {} {
    global Server
    case $Server(in) "" {
	# no connection
    } default {
	flush $Server(in)
	nntpPuts "QUIT"
	catch {close $Server(in)}
    }
}

# connect to the server
# Returns:
#
# if successful		return the servers greeting message (see RFC 977)
#
# if unsuccessful	return a message of form
#			"588 <free form error text here>"
proc nntpConnect {} {
    global Server Profile
    if {[file executable /usr/etc/mconnect]} {
	set finout \
	    [open "|/usr/etc/mconnect -p $Server(port) $Profile(Server)" w+]
    } else {
	# oh well, use telnet
	set finout \
	    [open "|telnet $Profile(Server) $Server(port)" w+]
    }
    if {[eof $finout]} {
	set Server(in) ""
	set Server(out) ""
	return "588 cannot open pipe to NNTPserver"
    }
    set Server(in) $finout
    set Server(out) $finout

    # trash any startup messages, when we get the NNTP server greeting
    # return it
    while {[gets $Server(in) l]>0} {
	case $l in {[0-9]*NNTP*} {
	    statusMsg $l
	    return $l
	} default {
	    statusMsg $l
	}
    }
    set Server(in) ""
    set Server(out) ""
    return "588 NNTP connect to server '$Profile(Server)' failed"
}

proc nntpRestart {} {
    global Server
    nntpDisconnect
    set rval [nntpClientInit]
    case $rval {200 201} {
	return 0
    }
    return -1
}

# write string to server 'ala' puts
proc nntpPuts {s} {
    global Server
    if {[catch { puts $Server(out) $s; flush $Server(out) } err]} {
	statusMsg "error writing to server: '$err'"
	return -1;
    }
    return 0;
}

# read string from server ala 'gets'
proc nntpGets {s} {
    global Server
    uplevel 1 [format "gets %s %s" $Server(in) [list $s]]
}

# decode a NNTP server response message
# messages are of the form
# DDD <textstring>\r\n
# 	D = decimal digit
#
# DDD 		is placed into variable  named $code in callers namespace
#<textstring> 	is placed into variable  named $msg in callers namespace
proc nntpResponse {resp code msg} {
    uplevel 1 regexp {{([0-9]+)[ 	]+(.*)}} $resp _junk $code $msg
}

proc statusMsg {s} {
    puts stderr [format "arTCLs: %s" $s]
    flush stderr
}

# nntpGetGroups --
#
# get a listing of all the news groups from the server and 
# place them into the global asscoiative array Groups
# the key for the array is the group name. the values are lists
# of the form
#
#	{ FirstArticle# LastArticle# Flagcharacter }
#
# Flagcharacter values
#	y = posting allowed n = no posting m = moderated
#
# Note: some servers may not produce 'm' flags

proc nntpGetGroups {} {
    global Server Groups; unset Groups
    set Groups(.) { _First _Last _Flag }

    # make request
    if {[nntpPuts "LIST"]<0} {
	return -1
    }

    # get response
    if {[gets $Server(in) resp]<0} {
	return -1
    }

    # evaluate response message
    # 215		- list of newsgroups follow
    # anything else	- error is an error really

    regexp {([0-9]+)[ 	]+(.*)} $resp junk code msg
    case $code in {215} {
	# statusMsg $msg
    } {[0-9]*} {
	error "NNTP server Error $code: $msg"
    } default {
	error "NNTP server Confused!"
    }

    # get groups from server

    while {1} {
	if {[gets $Server(in) resp]<0} {
	    return -1
	}
	case $resp in {.*} {
	    break
	} default {
	    regexp  \
		{([^ 	]+)[ 	]+([0-9]+)[ 	]+([0-9]+)[ 	]+(.).*} \
		$resp junk gname last first opt

	    # strip obnoxious leading zeroes
	    set first [string trimleft $first 0]
	    if {[string length $first]==0} { set first 0 }

	    set last [string trimleft $last 0]
	    if {[string length $last]==0} { set last 0 }

	    lappend Groups($gname) $first $last $opt
	}
    }
    return 0
}

proc nntpGroup {statusvar groupname} {
    global Server
    nntpPuts "GROUP $groupname"
    gets $Server(in) resp
    # evaluate RFC 977 GROUP response message
    # 411	- no such newsgroup
    # 211	- message contains secondary info of form
    #		  n f l s
    #			n = estim. # of articles
    #			f = first article
    #			l = last article
    #			s = name of group
    regexp {([0-9]+)[ 	]+(.*)} $resp junk code msg
    case $code in {211} {
	if {[scan $msg "%d %d %d %s" n f l s]!=4} {
	    set msg "500 malformed GROUP response <$msg> from server"
	    set code 500
	} else {
	    set msg $resp
	}
    } {411} {
    } default {
	set msg "$code - NNTP server Error - $msg"
    }
    uplevel 1 "set $statusvar $code"
    return $msg
}

# return a list of articles for 'groupname' via 'key'
# using 'sel' as regexp for selection criteria
# a single : or ! are special in that they select all 
# unread or read articles articles respectively
# . is also special it will get all articles available
# read or unread.
#
# status is the name of a variable which this func will set to non-zero
# there was an error in getting the article. it will be set to zero if
# no errors occured
#
proc nntpGetArticles {statusvar groupname {key subject} {sel :}} {
    global Server
    set artlist {}
    uplevel 1 "set $statusvar 0"

    # set the current group to $groupname
    set msg [nntpGroup s $groupname]
    if {$s!=211} {
	uplevel 1 "set $statusvar -1"
	statusMsg $msg
	return  $artlist
    }

    #	n = estim. # of articles
    #	f = first article
    #	l = last article
    #	s = name of group
    scan $msg "%d %d %d %d %s" code n f l s

    # cut down on what headers we ask for if we are 
    # just interested in unread stuff. we ask for the first unread
    # article to the last available article. 
    #
    # XXX
    #
    # we should build an exact list of unread articles, but then we have 
    # to worry about staying within the line length for a legal nntp 
    # command etc.
    if {$sel==":"} {
	while {$f<=$l} {
	    if {[rcNewArticle $groupname $f]} {
		break
	    } else {
		incr f
	    }
	}
    }

    # request
    if {[nntpPuts "XHDR $key $f-$l"]<0} {
	uplevel 1 "set $statusvar -1"
	return $artlist
    }

    # response. should be of form
    # ddd <text>
    # where text is something like "subject fields to follow"
    # ddd is the number of header lines to follow
    gets $Server(in) resp
    if {[scan $resp "%d %s" code msg]!=2} {
	uplevel 1 "set $statusvar -1"
	statusMsg "Error: in XHDR response <$resp>"
	return $artlist
    }

    while {[gets $Server(in) xline]>=0} {
	case $xline {.\r .} {
	    return $artlist
	} default {
	    if {![regexp {([0-9]+)[ 	]+(.*)} \
		    [string trimright $xline "\r\n"] junk art val]} {
		statusMsg "bad XHDR line <$xline>"
	    } else {
		case $sel : {
		    if {[rcNewArticle $groupname $art]} { 
			lappend artlist [list . $art $val]
		    }
		} ! {
		    if {![rcNewArticle $groupname $art]} {
			lappend artlist [list ! $art $val]
		    }
		} "." {
			lappend artlist [list . $art $val]
		} default {
		    # to be done ....
		}
	    }
	}
    }
    statusMsg "Error: in XHDR retrieval"
    uplevel 1 "set $statusvar -1"
    return {}
}

# nntpArticlePipe {msgid flags pcmd}
#
# msgid:
#    The <msg-id> to use as the source of the pipeline	
#
# flags:
#    b == pipe just the article body
#    t == just print oput the pipeline cmd on stderr do not actually
#         run the cmd
# pcmd:
#     the pipeline to run the artcile thru
#     Note: do not put a leading | on the pcmd argument
#     nntpArticlePipe will automatically do this

proc nntpArticlePipe {flags pcmd {msgid "current"}} {
    global Server ArticleHeader

    if {$msgid=="current"} {
	if {![info exists ArticleHeader(Message-ID)]} {
	    statusMsg "no current article to pipe to \n\t '$pcmd'"
	    return -1
	}
	set msgid $ArticleHeader(Message-ID)
    }

    case $flags {*t*} {
	# for hackers..
	# just print what the assembled command looks like
	puts stderr  "for: $msgid"
	puts stderr "|sed -e s/\\r$// -e s/^\\.\\./\\./ | $pcmd"
	return 0
    }

    # collapse leading double dots and get rid of the <cr>'s
    set p [open "|sed -e s/\\r$// -e s/^\\.\\./\\./ | $pcmd" "w"]

    case $flags {*b*} {
	nntpPuts "BODY $msgid"
    } default {
	nntpPuts "ARTICLE $msgid"
    }
    gets $Server(in) resp
    scan $resp "%d" code
    case $code {22?} {
	while {[gets $Server(in) ln]>=0} {
	    case $ln {.\r .} {
		# done
		close $p
		return 0
	    } default {
		puts $p $ln
	    }
	}
	statusMsg "nntpArticlePipe: read error"
	return -1
    } default {
	statusMsg "nntpArticlePipe: $resp"
    }
    close $p
    return 0
}

# connect to NNTP server and set up main global data structures
proc nntpClientInit {} {
    global Server Groups

    # connect to server
    set resp [nntpConnect]

    # evaluate ready message
    # 200 - server ready , posting is allowed
    # 201 - server ready , no posting allowed
    regexp {([0-9]+)[ 	]+(.*)} $resp junk code msg

    case $code in 588 {
	error "Error $code: $resp"
    } 200 {
	statusMsg "NNTP server ready. Posting allowed"
	set Server(postingallowed) 1
    } 201 {
	set Server(postingallowed) 0
	statusMsg "NNTP server ready. No Posting allowed. Sorry"
    } default {
	error "NNTP server Confused!"
    }
    return code
}

