# mime.tcl
#
# MIME message display.
#
# Copyright (c) 1993 Xerox Corporation.
# Use and copying of this software and preparation of derivative works based
# upon this software are permitted. Any distribution of this software or
# derivative works must comply with all applicable United States export
# control laws. This software is made available AS IS, and Xerox Corporation
# makes no warranty about the software, its performance or its conformity to
# any specification.

proc Mime_Init {} {
    global mime env
    if [info exists mime(init)] {
	return
    }
    # Make sure Metamail is on the path
    set hit 0
    foreach dir [split $env(PATH) :] {
	if {[string compare $dir $mime(dir)] == 0} {
	    set hit 1
	    break
	}
    }
    if {! $hit} {
	set env(PATH) $mime(dir):$env(PATH)
    }

    set mime(init) 1
    set mime(seed) 1
    set mime(junkfiles) {}

    set types [concat [option get . mimeTypes {}] [option get . mimeUTypes {}]]
    Exmh_Debug MimeTypes $types
    set mime(showproc,default) 			Mime_ShowDefault
    if {[llength $types] == 0} {
	set mime(showproc,text/plain) 			Mime_ShowText
	set mime(showproc,text/enriched)		Mime_ShowText
	set mime(showproc,text/richtext)		Mime_ShowText
	set mime(showproc,text/x-pgp)			Mime_ShowText
	set mime(showproc,multipart) 			Mime_ShowMultipart
	set mime(showproc,application/octet-stream) 	Mime_ShowApplicationOctet
	set mime(showproc,application/postscript)   	Mime_ShowDirect
	set mime(showproc,message/external-body)	Mime_ShowMessageExternal
	set mime(showproc,message/rfc822)		Mime_ShowRfc822
	set mime(showproc,image) 			Mime_ShowDirect
	set mime(showproc,audio) 			Mime_ShowAudio
    } else {
	foreach type $types {
	    set func [option get . mime_$type {}]
	    if {$func != {}} {
		set mime(showproc,$type) $func
	    }
	}
    }

    Preferences_Add "MIME" \
"MIME is the Multipurpose Internet Mail Extension that allows a variety of message types to be transfered and displayed for you." {
	{mime(enabled) mimeEnabled	ON {Enable MIME display}
"This controls whether or not MIME-format messages are parsed
and displayed.  If it is disabled, then the messages are
displayed as plain text."}
	{mime(showType) mimeShowType	ON {Show MIME types}
"This controls whether or not the MIME type information for each
part of a multi-part message is displayed.  The type is prefixed
with -- to offset it from the rest of the information."}
	{mime(showPrelude) mimeShowPrelude	OFF {Show MIME prelude}
"This controls whether or not the information between the mail headers
and the official start of a multipart message is displayed.  Sometimes
this has useful information, while other times it has warnings about
the rest of the message being in MIME format."}
	{mime(ftpMethod) ftpMethod	{CHOICE expect ftp {ftp -n} metamail}
	    {FTP access method}
"Sometimes the automatic FTP transfer fails because of
problems logging into the remote host.  This option lets
you try a few different approachs to find the one that
works for you:
expect - use the ftp.expect script to download the file.
ftp - use ftp and feed user and password to it via a pipe.
ftp -n - use the ftp no-auto-login feature.
metamail - pass control to metamail and let it try."}
	{mime(ftpCommand) ftpCommand	ftp {FTP command name}
"You may need to run a different command than \"ftp\" in
order to get out onto the internet from inside your company
network.  The command will be invoked with the site name as
the only argument, possibly with the -n flag, which depends on
your choice of the FTP access method.  If you use the expect
script, you'll have to tweak that by hand."}
	{mime(isoFont) isoFont	-*-lucida-*-*-*-*-*-*-*-*-*-*-iso8859-* {ISO-8859 font}
"This font is used to display MIME text that is in the
ISO-8859-1 character set.  This kicks in regardless of your
font setting for the message display window.  Override this
feature by clearing out the font name."}
    }
}
proc Mime_Enabled {} {
    global mime
    return $mime(enabled)
}
proc Mime_Cleanup {} {
    global mime
    set cmd rm
    foreach f $mime(junkfiles) {
	if [file exists $f] {
	    lappend cmd $f
	}
    }
    if {$cmd != "rm"} {
	eval exec $cmd
    }
    set mime(junkfiles) {}
}

proc Mime_ShowBody { tkw fileName contentType encoding} {
    global mime mimeHdr
    catch {unset mimeHdr}
    Exmh_Status "Formatting MIME message ..." red
    set mimeHdr(0,0,file) $fileName
    MimeHeader 0 0 $contentType $encoding
    MimeShow $tkw 0 0
}
proc MimeShow {tkw depth part} {
    global mime mimeHdr
    set fileName $mimeHdr($depth,$part,file)
    set type $mimeHdr($depth,$part,type)
    foreach t [list $type [file dirname $type] default] {
	if [info exists mime(showproc,$t)] {
	    if [$mime(showproc,$t) $tkw $depth $part] {
		exec rm $fileName
	    }
	    return
	}
    }
}
proc MimeHeader { depth part contentType encoding } {
    global mimeHdr
    # mimeHdr contains state about nested body parts:
    # mimeHdr($depth,$part,type)	Content-Type
    # mimeHdr($depth,$part,encoding)	Content-Transfer-Encoding
    # mimeHdr($depth,$part,file)	Tmp file containing body
    # mimeHdr($depth,$part,params)	Parameter names (e.g., boundary)
    # mimeHdr($depth,$part,param,$key)	Parameter value
    # mimeHdr($depth,$part,hdrs)	List of subpart mail headers
    # mimeHdr($depth,$part,hdr,$key)	Subpart mail header value

    set params [split $contentType \;]
    set type [string tolower [string trim [lindex $params 0]]]
    if {$type == "text"} {
	set type text/plain
    }
    set mimeHdr($depth,$part,hdr,content-type) $contentType
    set mimeHdr($depth,$part,type) $type
    set mimeHdr($depth,$part,encoding) $encoding
    set mimeHdr($depth,$part,params) {}
    foreach sub [lrange $params 1 end] {
	set sub [string trim $sub]
	if [regexp {([^=]+)=(.+)} $sub match key val] {
	    set key [string trim [string tolower $key]]
	    set val [string trim $val \ \"]
	    lappend mimeHdr($depth,$part,params) $key
	    set mimeHdr($depth,$part,param,$key) $val
	}
    }
    return $type
}

proc Mime_ShowDefault { tkw depth part args} {
    global mimeHdr

    MimeInsertInfo $tkw $depth $part
    TextButton $tkw "Pass content to metamail..." \
	[concat [list MimeMetaMail $mimeHdr($depth,$part,hdr,content-type) \
			   $mimeHdr($depth,$part,encoding) \
			   $mimeHdr($depth,$part,file)] \
			   $args]
    return 0
}
proc Mime_ShowDirect { tkw depth part } {
    return [Mime_ShowDefault $tkw $depth $part -e -d]
}
proc MimeInsertInfo { tkw depth part } {
    global mimeHdr
    if [info exists mimeHdr($depth,$part,hdr,content-description)] {
	    $tkw insert insert \
		"Description: $mimeHdr($depth,$part,hdr,content-description) "
    }
    $tkw insert insert "($mimeHdr($depth,$part,type))\n"
    foreach key $mimeHdr($depth,$part,params) {
	$tkw insert insert "\t$key = $mimeHdr($depth,$part,param,$key)\n"
    }
}

proc MimeMetaMail { contentType encoding fileName args } {
    global mimeHdr msgHdr
    if [catch {
	Exmh_Status "metamail $fileName"
	set mcmd [list metamail -b \
		    -c $contentType \
		    -E $encoding \
		    -f [MsgParseFrom $msgHdr(from)] \
		    -m exmh]
	# recall that eval concats its arguments, thus exploding things for us
	Exmh_Debug [concat exec $mcmd $args $fileName]
	eval exec $mcmd $args $fileName > /dev/null &
	Exmh_Status ok
    } err] {
	 Exmh_Status "$err"
    }
}

proc Mime_ShowText { tkw depth part } {
    global mimeHdr mime
    set charset us-ascii
    if [info exists mimeHdr($depth,$part,param,charset)] {
	set charset $mimeHdr($depth,$part,param,charset)
	switch -regexp $charset {
	    (us-ascii|US-ASCII) { set charset us-ascii }
	    (iso|ISO)-8859-1 { set charset iso-8859-1 }
	    default { $tkw insert insert "(Warning: unknown charset <$charset>)\n\n" }
	}
    }
    set subtype [file tail $mimeHdr($depth,$part,type)]
    set encoding $mimeHdr($depth,$part,encoding)
    set fileName $mimeHdr($depth,$part,file)
    if ![regexp {[78]bit} $encoding] {
	set newFile [Mime_TempFile /tmp/exmh.decode]
	if ![MimeDecode $mimeHdr($depth,$part,file) $newFile $encoding] {
	    $tkw insert insert "(Decode failed - raw text follows)\n\n"
	    set subtype plain
	} else {
	    set fileName $newFile
	}
    }
    Exmh_Status text/$subtype
    if [catch {open $fileName r} fileIO] {
	Exmh_Status "Cannot open body $fileName: $fileIO"
	return 1
    }
    case $subtype {
	default {
	    set start [$tkw index insert]
	    $tkw insert insert [read $fileIO]
	    if {$charset == "iso-8859-1"} {
		$tkw tag add ISO $start [$tkw index insert]
		if [catch {$tkw tag configure ISO -font $mime(isoFont)} err] {
		    Exmh_Debug Mime_ShowText $err
		}
	    }
	}
	{enriched richtext} { Rich_Display $tkw $fileIO $subtype}
    }
    close $fileIO
    if [info exists newFile] {
	exec rm $newFile
    }
    return 1
}
proc Mime_ShowRfc822 { tkw depth part } {
    global mimeHdr
    Exmh_Debug Mime_ShowRfc822 $depth $part
    set fileName $mimeHdr($depth,$part,file)
    if [catch {open $fileName r} fileIO] {
	Exmh_Status "Cannot open body $fileName: $fileIO"
	return 1
    }

    # "Chop" up the file, which will only be a single message,
    # but this fills out the mimeHdr data structure and our
    # MimeShowPart below will handle
    # the possible recursion of the message is itself a multi-part.

    set np [MimeMultiChop $tkw $depth $part $fileIO {} text/plain]
    close $fileIO
    Exmh_Debug RFS822 MimeMultiChop got $np parts
    for {set p 1} {$p <= $np} {incr p} {
	if [info exists mimeHdr($depth-$part,$p,hdrs)] {
	    foreach hdr $mimeHdr($depth-$part,$p,hdrs) {
		$tkw insert insert "$hdr: $mimeHdr($depth-$part,$p,hdr,$hdr)\n"
	    }
	}
	$tkw insert insert \n
	MimeShowPart $tkw $depth-$part $p
    }
    return 1
}
proc MimeDecode { fileName name encoding } {
    set ok 1
    if [catch {
	case $encoding {
	    default {
		Exmh_Status "cat > $name"
		exec cat < $fileName > $name
	    }
	    {8bit 7bit} {
		Exmh_Status "cat > $name"
		exec cat < $fileName > $name
	    }
	    base64 {
		Exmh_Status "mimencode -u -b > $name"
		exec mimencode < $fileName -u -b > $name
	    }
	    quoted-printable {
		Exmh_Status "mimencode -u -q > $name"
		exec mimencode < $fileName -u -q > $name
	    }
	}
    } err] {
	Exmh_Status "Decode failed: $err"
	set ok 0
    }
    return $ok
}

proc Mime_ShowApplicationOctet { tkw depth part } {
    global mimeHdr

    $tkw insert insert "You have received an encoded file.\n"
    MimeInsertInfo $tkw $depth $part
    if [info exists mimeHdr($depth,$part,param,name)] {
	set name [string trim $mimeHdr($depth,$part,param,name) \ \" ]
    } else {
	set name {}
    }
    TextButton $tkw "Open file transfer dialog..." \
	[list MimeFileTransfer $mimeHdr($depth,$part,file) $name \
				$mimeHdr($depth,$part,encoding)]
    return 0
}

proc MimeFileTransfer { fileName name encoding } {
    set name [FSBox "Select the name of the downloaded file:" $name ]
    if {$name != {}} {
	if [MimeDecode $fileName $name $encoding] {
	    Exmh_Status "Conversion ok"
	}
    }
}
proc Mime_ShowAudio { tkw depth part } {
    global mimeHdr

    $tkw insert insert "You have received an audio file\n"
    MimeInsertInfo $tkw $depth $part
    TextButton $tkw "Play audio..." \
	[list MimeDoAudio $mimeHdr($depth,$part,file) \
			  $mimeHdr($depth,$part,encoding) $depth $part]
    return 0
}
proc MimeDoAudio { fileName encoding depth part } {
    global mimeHdr
    if [info exists mimeHdr($depth,$part,audio)] {
	Exmh_Status "Playing audio..."
	exec showaudio < $mimeHdr($depth,$part,audio)
	Exmh_Status ""
    } else {
	set name [Mime_TempFile /tmp/exmh.audio]
	if [MimeDecode $fileName $name $encoding] {
	    Exmh_Status "Playing audio..."
	    exec showaudio < $name
	    set mimeHdr($depth,$part,audio) $name
	    Exmh_Status ""
	}
    }
}
proc Mime_ShowMessageExternal { tkw depth part } {
    global mimeHdr mime

    if [catch {set mimeHdr($depth,$part,param,access-type)} atype] {
	return [Mime_ShowDefault $tkw $depth $part]
    }
    if {$atype != "anon-ftp"} {
	return [Mime_ShowDefault $tkw $depth $part]
    }
    MimeInsertInfo $tkw $depth $part
    TextButton $tkw "Open FTP transfer dialog..." \
			    [list MimeFTPTransfer $tkw $depth $part ]
    return 0
}
proc MimeFTPTransfer { tkw depth part } {
    global mime mimeHdr
    set site $mimeHdr($depth,$part,param,site)
    set directory $mimeHdr($depth,$part,param,directory)
    set theirname $mimeHdr($depth,$part,param,name)
    if {$mime(ftpMethod)  == "metamail"} {
	set myname foo
    } else {
	set myname [FSBox "Select the name of the downloaded file:" $theirname ]
    }
    if {$myname != {}} {
	if [catch {
	    case $mime(ftpMethod) {
		expect {
		    Exmh_Status "ftp.expect $site ..."
		    busy exec ftp.expect $site $directory $theirname $myname
		}
		ftp* {
		    Exmh_Status "$mime(ftpCommand) -n $site ..."
		    busy MimeFTPInner $site $directory $theirname $myname
		}
		metamail {
		    MimeMetaMail $mimeHdr($depth,$part,hdr,content-type) \
			   $mimeHdr($depth,$part,encoding) \
			   $mimeHdr($depth,$part,file) \
			   -d
		}
		default {
		    error "Unknown ftpMethod $mime(ftpMethod)"
		}
	    }
	} err] {
	    if [Exwin_Toplevel .ftpmsg "FTP error"] {
		Widget_Message .ftpmsg msg -aspect 1500 -relief raised
	    }
	    .ftpmsg msg configure -text \
"Messages generated during FTP transfer:

$err
"
	} else {
	    Exmh_Status "FTP transfer complete"
	}
    }
}
proc MimeFTPInner {site directory theirname myname} {
    global env mime
    if {$mime(ftpMethod) == "ftp -n"} {
	set pipe [open "|$mime(ftpCommand) -n $site " w]
	puts $pipe "user anonymous $env(USER)@"
    } else {
	set pipe [open "|$mime(ftpCommand) $site" w]
	puts $pipe anonymous
	puts $pipe $env(USER)@
    }
    puts $pipe "cd $directory"
    puts $pipe "get $theirname $myname"
    puts $pipe "quit"
    close $pipe
}

proc Mime_ShowMultipart { tkw depth part } {
    global mimeHdr

    if ![info exists mimeHdr($depth,$part,param,boundary)] {
	$tkw insert insert "No <boundary> parameter for multipart message\n"
	$tkw insert insert "Raw content follows...\n\n"
	return [Mime_ShowText $tkw $depth $part]
    }
    set fileName $mimeHdr($depth,$part,file)
    set boundary $mimeHdr($depth,$part,param,boundary)
    set type $mimeHdr($depth,$part,type)

    if [catch {open $fileName r} fileIO] {
	$tkw insert insert "Mime_ShowMultipart $fileName: $fileIO\n"
	return 0
    }
    Exmh_Status "Mime_ShowMultipart $depth $part $type"
    set mimeHdr($depth,$part,numParts) \
	[MimeMultiChop $tkw $depth $part $fileIO $boundary \
	    [expr {($type == "multipart/digest") ? \
		"message/rfc822" : "text/plain"}]]
    close $fileIO

    # Display it

    MimeShowPart $tkw $depth $part

    return 1
}
proc MimeMultiChop {tkw depth part fileIO boundary {defType text/plain} } {
    global mimeHdr mime

    set depth $depth-$part
    set part 0
    if {$boundary == {}} {	;# Hack for message/RFC822
	set state rfc822
    } else {
	set state prolog	;# prolog > header > body [ > header > body ]
    }

    # Read and parse multipart, diverting subparts to files

    while {1} {
	set numBytes [gets $fileIO line]
	if {$numBytes < 0} {
	    catch {close $tmpFile}
	    return $part
	}
	if {($numBytes == 0) && ($state == "header")} {
	    set state body
	    if [catch {set mimeHdr($depth,$part,hdr,content-type)} contentType] {
		set contentType $defType
	    }
	    if [catch {set mimeHdr($depth,$part,hdr,content-transfer-encoding)} encoding] {
		set encoding 7bit
	    }
	    set encoding [string trim [string tolower $encoding] \ \" ]
	    set type [MimeHeader $depth $part $contentType $encoding]
	    Exmh_Status "Saving $type ..."
	    if {[file dirname $type] == "multipart"} {
		if ![catch {set mimeHdr($depth,$part,param,boundary)} bndy] {
		    set mimeHdr($depth,$part,numParts) \
			[MimeMultiChop $tkw $depth $part $fileIO $bndy]
		}
	    }
	    continue
	}
	if [regexp ^-- $line] {
	    #Could be a boundary line
	    set trailer [string range $line 2 end]
	    if {[string compare $boundary $trailer] == 0} {
		if {$state != "prolog"} {
		    close $tmpFile
		}
		incr part
		set state header
		set mimeHdr($depth,$part,file) \
		    [Mime_TempFile /tmp/exmh.$depth.$part]
		set tmpFile [open $mimeHdr($depth,$part,file) w]
		Exmh_Status "MimeMultiChop $depth $part $mimeHdr($depth,$part,file)"
		continue
	    } else {
		if {[string compare $boundary-- $trailer] == 0} {
		    if {$state != "prolog"} {
			close $tmpFile
		    }
		    return $part
		}
	    }
	}
	if {($state == "rfc822") && ($numBytes > 0)} {
	    set state header
	    incr part
	    set mimeHdr($depth,$part,file) \
		    [Mime_TempFile /tmp/exmh.$depth.$part]
	    set tmpFile [open $mimeHdr($depth,$part,file) w]
	    Exmh_Status "MimeMultiChop $depth $part $mimeHdr($depth,$part,file)"
	}
	case $state {
	    header {
		set continue [regexp {^[ 	]} $line]
		if {! $continue} {
		    if [regexp -indices {^([^:]+):} $line match hdr] {
			set cur [string tolower \
			    [eval {string range $line} $hdr]]
			set mimeHdr($depth,$part,hdr,$cur) \
			    [string trim \
				[string range $line \
				    [expr [lindex $match 1]+1] end]]
			lappend mimeHdr($depth,$part,hdrs) $cur
		    }
		} else {
		    if [regexp -indices {^[ 	]+} $line match] {
			append mimeHdr($depth,$part,hdr,$cur) \
			    [string range $line \
				    [expr [lindex $match 1]+1] end]
		    }
		}
	    }
	    body {
		# Divert sub-body to a temp file.
		puts $tmpFile $line
	    }
	    prolog {
		if {$mime(showPrelude)} {
		    $tkw insert insert $line\n
		}
	    }
	}
    }
}
proc MimeShowPart { tkw depth part {defType text/plain} } {
    global mimeHdr mime
    if [catch {set mimeHdr($depth,$part,type)} type] {
	set type $defType
    }
    if {$mime(showType)} {
	$tkw insert insert --$type\n
    }
    if [info exists mimeHdr($depth,$part,hdr,content-description)] {
	$tkw insert insert $mimeHdr($depth,$part,hdr,content-description)\n\n
    }
    if ![regexp multipart $type] {
	MimeShow $tkw $depth $part
	return
    }
    set numParts $mimeHdr($depth,$part,numParts)
    set origDepth $depth
    set origPart $part
    set depth $depth-$part
    case [file tail $type] {
	{mixed default} {
	    for {set p 1} {$p <= $numParts} {incr p} {
		MimeShowPart $tkw $depth $p
	    }
	}
	digest {
	    Exmh_Debug DIGEST with $numParts parts
	    for {set p 1} {$p <= $numParts} {incr p} {
		Exmh_Debug digest $depth $p 
		MimeShowPart $tkw $depth $p message/rfc822
	    }
	}
	alternative {
	    global mime
	    set chosenPart {}
	    for {set p 1} {$p <= $numParts} {incr p} {
		set type $mimeHdr($depth,$p,type)
		foreach t [list $type [file dirname $type]] {
		    if [info exists mime(showproc,$t)] {
			set chosenPart $p
			break
		    }
		}
	    }
	    if {$chosenPart == {}} {
		# Can't display anything.  Unroll the whole thing
		# so the user can choose what to send to metamail.
		# This is mainly to work around the fact that the
		# sub-parts are not collected into a single file for me.
		$tkw insert insert "You have to pick an alternative...\n"

		for {set p 1} {$p <= $numParts} {incr p} {
		    MimeShowPart $tkw $depth $p
		}
	    } else {
		MimeShowPart $tkw $depth $chosenPart
	    }
	}
    }
}
proc Mime_TempFile { basename } {
    global mime
    set uid 0
    while {[file exists $basename.$uid]} {
	incr uid
    }
    lappend mime(junkfiles) $basename.$uid
    return $basename.$uid
}

proc Mime_Debug { args } {
    puts stderr $args
}
