/* 
 * mxWidget.c -- 
 *
 *	This module implements an editable text widget based on the Mx editor.
 *	This is the top level file for the widget; the guts of the
 *	edit and display engine are elsewhere.
 *
 * Copyright 1990 Regents of the University of California.
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The University of California
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 *
 * Widgetized by Brent Welch while at Xerox-PARC.
 * Copyright (c) 1992 Xerox Corporation.
 * Use and copying of this software and preparation of derivative works based
 * upon this software are permitted. Any distribution of this software or
 * derivative works must comply with all applicable United States export
 * control laws. This software is made available AS IS, and Xerox Corporation
 * makes no warranty about the software, its performance or its conformity to
 * any specification.
 */

#ifndef lint
static char rcsid[] = "$Header: /project/tcl/src/mxedit/RCS/mxWidget.c,v 2.3 1993/06/29 23:18:08 welch Exp $ SPRITE (Berkeley)";
#endif

#include "default.h"
#include "tkConfig.h"
#include "tkInt.h"

#include "errno.h"

#include "mxWidget.h"
#include "tclHash.h"

#include <string.h>
#include <stdio.h>
#include <unistd.h>

/*
 * Default cursor to use in Mx windows (change the names of these
 * things when running lint to avoid stupid complaints):
 */

#ifdef lint
#define cursor_bits mx_bits
#define cursMask_bits mxMask_bits
#endif

#include "cursor.bits"
#include "cursMask.bits"
static Cursor cursor;

/*
 * Information used for parsing configuration specs:
 */

static Tk_ConfigSpec configSpecs[] = {
    /*
     * Editor-related options
     */
    {TK_CONFIG_STRING, "-file", "file", "File",
	"/tmp/scratch", Tk_Offset(MxWidget, fileName), 0, NULL},

    {TK_CONFIG_FONT, "-font", "font", "Font",
	DEF_ENTRY_FONT, Tk_Offset(MxWidget, fontPtr),
	0, NULL},
    /*
     * Name of adjunct scrollbar.  Use a la listbox -scroll
     */
    {TK_CONFIG_STRING, "-scroll", "scroll", "Scroll",
	".mx0.scroll", Tk_Offset(MxWidget, scrollWidget),
	0, NULL},
    /*
     * Name of feedback command.  It will get invoked as
     * feedbackCmd "msg"
     * An appropriate command is created by mxedit.tk
     */
    {TK_CONFIG_STRING, "-feedback", "feedback", "Feedback",
	"feedback.mx0", Tk_Offset(MxWidget, feedbackCmd),
	0, NULL},
    /*
     * Colors
     */
    {TK_CONFIG_SYNONYM, "-bg", "background", (char *) NULL,
	(char *) NULL, 0, 0, NULL},
    {TK_CONFIG_COLOR, "-background", "background", "Background",
	DEF_BUTTON_BG_COLOR, Tk_Offset(MxWidget, background),
	TK_CONFIG_COLOR_ONLY, NULL},
    {TK_CONFIG_COLOR, "-background", "background", "Background",
	DEF_BUTTON_BG_MONO, Tk_Offset(MxWidget, background),
	TK_CONFIG_MONO_ONLY, NULL},
    {TK_CONFIG_SYNONYM, "-fg", "foreground", (char *) NULL,
	(char *) NULL, 0, 0, NULL},
    {TK_CONFIG_COLOR, "-foreground", "foreground", "Foreground",
	DEF_BUTTON_FG, Tk_Offset(MxWidget, foreground), 0},
    {TK_CONFIG_COLOR, "-selector", "selector", "Foreground",
	DEF_BUTTON_SELECTOR_COLOR, Tk_Offset(MxWidget, selectorFg),
	0, NULL},
    /*
     * Geometry is in character units.
     */
    {TK_CONFIG_STRING, "-geometry", "geometry", "Geometry",
	DEF_FRAME_GEOMETRY, Tk_Offset(MxWidget, geometry), 0, NULL},

    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0, NULL}
};

/*
 * Forward declarations for procedures defined later in this file:
 */

static int		MxWidgetCmd _ANSI_ARGS_((ClientData clientData,
			    Tcl_Interp *interp, int argc, char **argv));
int			MxConfigureWidget _ANSI_ARGS_((Tcl_Interp *interp,
			    MxWidget *mxwPtr, int argc, char **argv,
			    int flags));
static MxWidget *	CreateMxWidget _ANSI_ARGS_((Tcl_Interp *interp,
			    Tk_Window tkwin, char *pathName));
static void		DestroyMxWidget _ANSI_ARGS_((ClientData clientData));
int			MxWidgetInstanceCmd _ANSI_ARGS_((ClientData clientData,
			    Tcl_Interp *interp, int argc, char **argv));

/*
 *--------------------------------------------------------------
 *
 * MxWidgetInstall --
 *
 *	Initialize this widget.
 *
 * Results:
 *	None
 *
 * Side effects:
 *	Registers the TCL command that creates MX widgets.
 *
 *--------------------------------------------------------------
 */

int
MxWidgetInstall(w, interp)
    Tk_Window w;		/* Main window */
    Tcl_Interp *interp;		/* Current interpreter. */
{
    Tcl_CreateCommand(interp, MX_TCL_CMD, MxWidgetCmd, (ClientData) w,
	    (void (*)()) NULL);

    Tcl_SetVar(interp, "mxeditVersion", MX_VERSION, TCL_GLOBAL_ONLY);
    Tcl_SetVar(interp, "mxVersion", MX_VERSION, TCL_GLOBAL_ONLY);
    {
	int Mx_FileActiveCmd(), Mx_FileIdentCmd();
	int Tcl_GetclockCmd(), Tcl_FmtclockCmd();
	int Tk_CutBufferCmd();
	/*
	 * Register a command that checks if we are editing a file already,
	 * and another that returns a unique ident string for a file.
	 */
	Tcl_CreateCommand (interp, "mxFileActive", Mx_FileActiveCmd,
			   (ClientData)NULL, (void (*)())NULL);
	Tcl_CreateCommand (interp, "mxFileIdent", Mx_FileIdentCmd,
			   (ClientData)NULL, (void (*)())NULL);
	/*
	 * Register a few commands taken from Extended Tcl
	 */
	Tcl_CreateCommand (interp, "getclock", Tcl_GetclockCmd, 
			  (ClientData)NULL, (void (*)())NULL);
	Tcl_CreateCommand (interp, "fmtclock", Tcl_FmtclockCmd, 
			  (ClientData)NULL, (void (*)())NULL);
        /*
         * And a cutbuffer extension.
	 */
	Tcl_CreateCommand (interp, "cutbuffer", Tk_CutBufferCmd,
			    (ClientData)w, (void (*)())NULL);

    }

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * MxWidgetCmd --
 *
 *	This procedure is invoked to process the "mxedit" Tcl command.
 *	This creates the widget and defines a new Tcl command under
 *	the pathname of the created widget.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	This is a composition of the Create and Configure procedures.
 *
 *--------------------------------------------------------------
 */

static int
MxWidgetCmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Main window associated with
				 * interpreter. */
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{
    register MxWidget *mxwPtr;
    Tk_Window tkwin = (Tk_Window) clientData;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args:  should be \"",
		argv[0], " pathName [options]\"", (char *) NULL);
	return TCL_ERROR;
    }
#ifdef notdef
    {
	int i;
	fprintf(stderr, "mxedit: ");
	for (i=1 ; i<argc ; i++) {
	    fprintf(stderr, "\"%s\" ", argv[i]);
	}
	fprintf(stderr, "\n");
    }
#endif
    mxwPtr = CreateMxWidget(interp, tkwin, argv[1]);
    if (mxwPtr == NULL) {
	return TCL_ERROR;
    }
    if (MxConfigureWidget(interp, mxwPtr, argc-2, argv+2, 0) != TCL_OK) {
	Tk_DestroyWindow(mxwPtr->tkwin);
	return TCL_ERROR;
    }

    interp->result = Tk_PathName(mxwPtr->tkwin);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * CreateMxWidget --
 *
 *	This procedure creates and initializes a new widget.
 *
 * Results:
 *	The return value is a pointer to a structure describing
 *	the new widget.  If an error occurred, then the return
 *	value is NULL and an error message is left in interp->result.
 *
 * Side effects:
 *	Memory is allocated, a Tk_Window is created, etc.
 *
 *----------------------------------------------------------------------
 */

static MxWidget *
CreateMxWidget(interp, tkwin, pathName)
    Tcl_Interp *interp;		/* Used for error reporting. */
    Tk_Window tkwin;		/* Window to use for resolving pathName. */
    char *pathName;		/* Name for new window. */
{
    register MxWidget *mxwPtr;

    /*
     * Create the tkwin window, reusing the tkwin variable.
     */

    tkwin = Tk_CreateWindowFromPath(interp, tkwin, pathName, (char *) NULL);
    if (tkwin == NULL) {
	return (MxWidget *) NULL;
    }
    Tk_SetClass(tkwin, MX_TCL_CMD);

    /*
     * Initialize the data structure for the mxedit widget.
     */

    mxwPtr = (MxWidget *) ckalloc(sizeof(MxWidget));
    bzero((char *)mxwPtr, sizeof(MxWidget));

    mxwPtr->tkwin = tkwin;
    mxwPtr->interp = interp;
    mxwPtr->pathname = (char *)malloc(1+strlen(pathName));
    strcpy(mxwPtr->pathname, pathName);
    mxwPtr->type = 0;
    mxwPtr->textGC = None;
    mxwPtr->grayGC = None;
    mxwPtr->reverseGC = None;
    mxwPtr->flags = 0;

    Tk_CreateEventHandler(mxwPtr->tkwin, ExposureMask|StructureNotifyMask,
	    MxWidgetEventProc, (ClientData) mxwPtr);
    Tcl_CreateCommand(interp, Tk_PathName(mxwPtr->tkwin),
	    MxWidgetInstanceCmd, (ClientData) mxwPtr, (void (*)()) NULL);
    Tk_CreateFocusHandler(mxwPtr->tkwin,
	    MxFocusProc, (ClientData) mxwPtr);
#ifdef notdef
    (void) Tk_CreateEventHandler(mxwPtr->tkwin, ButtonPressMask
	    |ButtonReleaseMask|Button1MotionMask|Button3MotionMask
	    /* |LeaveWindowMask|EnterWindowMask */,
	    MxMouseProc, (ClientData) mxwPtr);
#endif

    MxCmdInit(mxwPtr);

    return mxwPtr;
}
/*
 *----------------------------------------------------------------------
 *
 * DestroyMxWidget --
 *
 *	This procedure is invoked by Tk_EventuallyFree or Tk_Release
 *	to clean up the internal structure of the widget at a safe time
 *	(when no-one is using it anymore).  Tk_EventuallyFree is called
 *	in response to a destroy event, ultimately generated by
 *	Tk_DestroyWindow.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Everything associated with the widget is freed up.
 *
 *----------------------------------------------------------------------
 */

void
MxDestroyWidget(clientData)
    ClientData clientData;		/* MxWidget pointer */
{
    register MxWidget *mxwPtr = (MxWidget *) clientData;
    MxFileInfo *fileInfoPtr;

    MxCleanupRedisplay(mxwPtr);
    MxHistoryCleanup(mxwPtr);
    MxManagerRecordClose(mxwPtr->interp);
    fileInfoPtr = mxwPtr->fileInfoPtr;
    if (fileInfoPtr != NULL) {
	if (fileInfoPtr->caretWindow == mxwPtr) {
	    fileInfoPtr->caretWindow = NULL;
	}
	MxDetachFileInfo(mxwPtr);
	MxRecycleFileInfo(fileInfoPtr);
    }
    Tk_CancelIdleCall(MxUpdateWidget, (ClientData)mxwPtr);
    Tk_CreateFocusHandler(mxwPtr->tkwin, NULL, NULL);

    if (mxwPtr->fontPtr != NULL) {
	Tk_FreeFontStruct(mxwPtr->fontPtr);
    }
    if (mxwPtr->foreground != NULL) {
	Tk_FreeColor(mxwPtr->foreground);
    }
    if (mxwPtr->background != NULL) {
	Tk_FreeColor(mxwPtr->background);
    }
    if (mxwPtr->activeFg != NULL) {
	Tk_FreeColor(mxwPtr->activeFg);
    }
    if (mxwPtr->textGC != None) {
	Tk_FreeGC(Tk_Display(mxwPtr->tkwin), mxwPtr->textGC);
    }
    if (mxwPtr->selectorFg != NULL) {
	Tk_FreeColor(mxwPtr->selectorFg);
    }
    if (mxwPtr->pathname != NULL) {
	ckfree(mxwPtr->pathname);
    }
    if (mxwPtr->fileName != NULL) {
	ckfree(mxwPtr->fileName);
    }
    if (mxwPtr->scrollWidget != NULL) {
	ckfree(mxwPtr->scrollWidget);
    }
    if (mxwPtr->feedbackCmd != NULL) {
	ckfree(mxwPtr->feedbackCmd);
    }
    if (mxwPtr->geometry != NULL) {
	ckfree(mxwPtr->geometry);
    }
    ckfree((char *) mxwPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * MxConfigureWidget --
 *
 *	This procedure is called to process an argv/argc list, plus
 *	the Tk option database, in order to configure (or
 *	reconfigure) an mxedit widget.
 *
 * Results:
 *	The return value is a standard Tcl result.  If TCL_ERROR is
 *	returned, then interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as text string, colors, font,
 *	etc. get set for mxwPtr;  old resources get freed, if there
 *	were any.  The mxedit widget is redisplayed.
 *
 *----------------------------------------------------------------------
 */

int
MxConfigureWidget(interp, mxwPtr, argc, argv, flags)
    Tcl_Interp *interp;		/* Used for error reporting. */
    MxWidget *mxwPtr;		/* Information about widget;  may or may
				 * not already have values for some fields. */
    int argc;			/* Number of valid entries in argv. */
    char **argv;		/* Arguments. */
    int flags;			/* Flags to pass to Tk_ConfigureWidget. */
{
    XGCValues gcValues;
    GC newGC;
    int dummy;
    Mx_Position eof;
    int charsHigh, charsWide;
    char *statusString;
    char msg[300];


    if (Tk_ConfigureWidget(interp, mxwPtr->tkwin, configSpecs,
	    argc, argv, (char *) mxwPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }
#ifdef notdef
    /* See what we really got for the font */
    Tk_ConfigureInfo(interp, mxwPtr->tkwin, configSpecs, (char *)mxwPtr, "-font", 0);
	fprintf(stderr, "Tk_ConfigureInfo: %s\n", interp->result);
#endif
    /*
     * Only to file-related configuration once.  After that you have
     * to use the "switch" widget instance command to change files.
     */
    if (mxwPtr->fileInfoPtr == NULL) {
	/*
	 * Check for non-blank file name.  Tk_ConfigureWidget gives
	 * us a blank string for free.
	 * No file name results in a scratch window.
	 */
	if (strlen(mxwPtr->fileName) == 0) {
	    ckfree(mxwPtr->fileName);
	    mxwPtr->fileName = NULL;
	}
	dummy = MxSetupFileInfo(mxwPtr, mxwPtr->fileName,
		0, MX_UNDO|MX_DELETE, mxwPtr->interp);
	if (dummy != TCL_OK) {
	    return dummy;
	}
	Tcl_SetVar(mxwPtr->interp, "mxFile", mxwPtr->fileInfoPtr->name,
		    TCL_GLOBAL_ONLY);
    
	eof = Mx_EndOfFile(mxwPtr->fileInfoPtr->file);
	statusString = "";
	if (access(mxwPtr->fileInfoPtr->name, W_OK) != 0) {
	    if (errno == EACCES) {
		statusString = " (read-only)";
	    } else if (errno == ENOENT) {
		statusString = " (new file)";
	    }
	}
    
	sprintf(msg, "%d", eof.lineIndex+1);
	Tcl_SetVar(mxwPtr->interp, "mxLines", msg, TCL_GLOBAL_ONLY);

	/*
	 * The cursor stuff also doesn't change during a reconfiguration.
	 */

	/*
	 * Define a cursor (mouse pointer).
	 */
	mxwPtr->cursor = Tk_GetCursorFromData(mxwPtr->interp, mxwPtr->tkwin,
		cursor_bits, cursMask_bits, cursor_width, cursor_height,
		cursor_x_hot, cursor_y_hot,
		Tk_GetUid("black"), Tk_GetUid("white"));
	if (mxwPtr->cursor == NULL) {
	    fprintf(stderr, "Tk_GetCursorFromData failed: %s\n", interp->result);
	} else if (Tk_IsMapped(mxwPtr->tkwin)) {
	    XDefineCursor(Tk_Display(mxwPtr->tkwin), Tk_WindowId(mxwPtr->tkwin),
		mxwPtr->cursor);
	}
	/*
	 * Set style of insertion marker.
	 */
	mxwPtr->cursorMode = CARET;
    }

    Tk_SetWindowBackground(mxwPtr->tkwin, mxwPtr->background->pixel);

    gcValues.font = mxwPtr->fontPtr->fid;
    gcValues.foreground = mxwPtr->foreground->pixel;
    gcValues.background = mxwPtr->background->pixel;
    newGC = Tk_GetGC(mxwPtr->tkwin, GCForeground|GCBackground|GCFont, &gcValues);
    if (mxwPtr->textGC != None) {
	Tk_FreeGC(Tk_Display(mxwPtr->tkwin), mxwPtr->textGC);
    }
    mxwPtr->textGC = newGC;
    if (mxwPtr->reverseGC != None) {
	Tk_FreeGC(Tk_Display(mxwPtr->tkwin), mxwPtr->reverseGC);
	mxwPtr->reverseGC = None;
    }

    /*
     * Recompute the geometry for the editting window.
     */
    if (mxwPtr->geometry != NULL) {
	if (sscanf(mxwPtr->geometry, "%dx%d", &charsWide, &charsHigh) != 2) {
	    char string[100];
	    sprintf(string,
		"Ignoring bad geometry \"%s\", expected widthxheight, in chars",
		    mxwPtr->geometry);
	    MxOutputMsg(mxwPtr, string);
	} else {
	    mxwPtr->widthChars = charsWide;
	    mxwPtr->heightLines = charsHigh;
	}
    }
    MxSetupRedisplay(mxwPtr, mxwPtr->widthChars, mxwPtr->heightLines);
    Tk_GeometryRequest(mxwPtr->tkwin, mxwPtr->width, mxwPtr->height);


    /*
     * Lastly, arrange for the widget to be redisplayed.
     */

    mxwPtr->flags |= NEEDS_UPDATE;
    if (Tk_IsMapped(mxwPtr->tkwin) && !(mxwPtr->flags & REDRAW_PENDING)) {
	Tk_DoWhenIdle(MxUpdateWidget, (ClientData) mxwPtr);
	mxwPtr->flags |= REDRAW_PENDING;
    }

    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * MxWidgetInstanceCmd --
 *
 *	This procedure is invoked to process the Tcl command
 *	that corresponds to a widget managed by this module.
 *	Keystrokes are bound to calls on this command, for example.
 *
 *	Because of the large number of commands on an mxedit, a hash
 *	table from argv[1]'s to C-procedures is used to vector to
 *	the right handler.  The handlers are implemented in
 *	mxCmdAM.c mxCmdNR.c and mxCmdSZ.c
 *
 *	Note also that argv[1] can be "!", in which case the operation is
 *	batched together with previous operations for the purpose of
 *	loging operations in the undo/redo log.  Only when an op comes
 *	along without the "!" syntax is an entry made in the undo/redo log.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */

int
MxWidgetInstanceCmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Information about button widget. */
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{
    register MxWidget *mxwPtr = (MxWidget *) clientData;
    int result = TCL_OK;
    int undo = 1;		/* By default, all actions are undoable */
    Tcl_HashEntry *entryPtr;	/* For commands hash table */
    int (*handler)();		/* Procedure to implement the command */

    if (argc < 2) {
	sprintf(interp->result,
		"wrong # args: should be \"%.50s [!] operation [arg arg ...]\"",
		argv[0]);
	return TCL_ERROR;
    }
    if (argv[1][0] == '!' && argv[1][1] == '\0') {
	/*
	 * A '!' before an action batches up undo for the action.  This is
	 * used to optimize the undo log so there isn't an entry for each char.
	 */
	undo = 0;
	argv++, argc--;
	if (argc < 2) {
	    sprintf(interp->result,
		"wrong # args: should be \"%.50s [!] operation [arg arg ...]\"",
		argv[0]);
	    return TCL_ERROR;
	}
    }
    handler = MxCmdFind(argv[1]);
    if (handler == NULL) {
	sprintf(interp->result,
	    "MxWidgetInstanceCmd: no operation \"%s\"", argv[1]);
	return TCL_ERROR;
    }
    if (undo) {
	Undo_Mark(mxwPtr->fileInfoPtr->log);
    }
    /*
     * Shift off the widget name, leaving the mx-original TCL command.
     */
    argv++, argc--;

    Tk_Preserve((ClientData) mxwPtr);
    result = (*handler)(mxwPtr, mxwPtr->interp, argc, argv);

    if (Tk_IsMapped(mxwPtr->tkwin) && !(mxwPtr->flags & REDRAW_PENDING)) {
	Tk_DoWhenIdle(MxUpdateWidget, (ClientData) mxwPtr);
	mxwPtr->flags |= REDRAW_PENDING;
    }
    if (undo) {
	Undo_Mark(mxwPtr->fileInfoPtr->log);
    }
    Tk_Release((ClientData) mxwPtr);
   return result;
}

/*
 *----------------------------------------------------------------------
 *
 * MxWidgetStateChangeCallback --
 *
 *	This procedure makes a callback to the application script to
 *	inform it that the dirty state of the file has changed.
 *
 * Results:
 *	None
 *
 * Side effects:
 *	Calls out to the application script via Tcl_Eval.
 *
 *----------------------------------------------------------------------
 */

void
MxWidgetStateChangeCallback(mxwPtr)
    register MxWidget *mxwPtr;
{
    static char buf[] = "mxHistory ignore mxStateChangeCallback";
    (void) Tcl_Eval(mxwPtr->interp, buf, 0, (char **) NULL);
}

/*
 *----------------------------------------------------------------------
 *
 * MxWidgetSizeChangeCallback --
 *
 *	This procedure makes a callback to the application script to
 *	inform it that the geometry of the window has changed.
 *
 * Results:
 *	None
 *
 * Side effects:
 *	Calls out to the application script via Tcl_Eval.
 *
 *----------------------------------------------------------------------
 */

void
MxWidgetSizeChangeCallback(mxwPtr)
    register MxWidget *mxwPtr;
{
    static char buf[] = "mxHistory ignore mxSizeChangeCallback";
    (void) Tcl_Eval(mxwPtr->interp, buf, 0, (char **) NULL);
}
