/*
 * mxWidget.h --
 *
 *	This contains definitions of the per-file and per-widget data
 *	structures.
 *
 * Copyright (c) 1992 Xerox Corporation.
 * Use and copying of this software and preparation of derivative works based
 * upon this software are permitted. Any distribution of this software or
 * derivative works must comply with all applicable United States export
 * control laws. This software is made available AS IS, and Xerox Corporation
 * makes no warranty about the software, its performance or its conformity to
 * any specification.
 */

#ifndef _MXWIDGET
#define _MXWIDGET

#include "sprite.h"
#include "tk.h"

#define bzero(s,n)	memset((s),0,(n))

/*
 * The tcl command that creates an mxWidget
 */
#define MX_TCL_CMD	"mxedit"

/*
 * The definitions below are dummy ones, so that clients don't have
 * to see the real contents of data structures.  Instead, clients just
 * pass tokens to the procedures that manipulate the structures.
 */

typedef struct Mx_File		*Mx_File;
typedef struct Mx_Floater	*Mx_Floater;
typedef struct Mx_Spy		*Mx_Spy;
typedef struct Undo_Log		*Undo_Log;

/*
 * The structure below is used as an indicator of a location within
 * a file.  Below it are macros for comparing two positions.
 */

typedef struct {
    int lineIndex;	/* Index of line (0 means first line in file). */
    int charIndex;	/* Index of character within a line. */
} Mx_Position;

#define MX_POS_EQUAL(a, b) 	(((a).lineIndex == (b).lineIndex) \
				&& ((a).charIndex == (b).charIndex))

#define MX_POS_LESS(a, b) 	(((a).lineIndex < (b).lineIndex) \
				|| (((a).lineIndex == (b).lineIndex) \
				&& ((a).charIndex < (b).charIndex)))

#define MX_POS_LEQ(a, b) 	(((a).lineIndex < (b).lineIndex) \
				|| (((a).lineIndex == (b).lineIndex) \
				&& ((a).charIndex <= (b).charIndex)))

/*
 * For each file that's open, regardless of how many windows in
 * which it's visible, there's one structure of the following
 * type with overall information about the file.
 */

typedef struct MxFileInfo {
    Mx_File file;			/* File storage. */
    char *name;				/* Name of file (NULL for typescripts
					 * or empty if this is a scratch
					 * window). */
    time_t lastMod;			/* Time file was last modified. */
    Undo_Log log;			/* Token for undo/recovery info, or
					 * NULL if no undo-ing. */
    struct MxHighlight *hlPtr;		/* First in list of highlights
					 * associated with this file.  Each
					 * highlight gets displayed in all
					 * the windows that view the file. */
    struct MxWidget *mxwPtr;		/* First in list of windows that view
					 * this file. */
    struct MxFileInfo *nextPtr;		/* Next in list of MxFileInfo structs */
    int flags;				/* Miscellaneous flags (see below). */

    /*
     * Information about the caret:
     */

    Mx_Position caretFirst;		/* Location of character that caret's
					 * just to left of. */
    Mx_Position caretLast;		/* Identical to caretFirst (needed for
					 * caretFloater, below. */
    struct MxWidget *caretWindow;	/* Window in which caret is being
					 * displayed.  If caret isn't
					 * currently displayed in a window
					 * in this file, then this is NULL. */
    /*
     * Info about detached (from caret) selections
     */
    Mx_Position floatLeft;		/* Left end of start of detached sel */
    Mx_Position floatRight;		/* Right end of start of detached sel */

    /*
     * Information used only by Mx:
     */
     
    struct MxHighlight *openParenPtr;	/* Open parenthesis being highlighted
					 * in this file (NULL if none). */
    struct MxHighlight *closeParenPtr;	/* Close parenthesis being highlighted
					 * in this file (NULL if none). */
#ifdef notdef
    /*
     * Information used only by Tx.
     */

    void (*inputProc)();	/* Procedure to invoke to process input from
				 * keyboard. */
    void (*sizeChangeProc)();	/* Procedure to invoke when size of Tx window
				 * changes (to reset termcap, for example). */
    void (*closeProc)();	/* Procedure to invoke when last window on
				 * this typescript is closed. */
    ClientData clientData;	/* Additional stuff to pass to inputProc
				 * and closeProc. */
    int viLines;		/* If not in vi mode, this is < 0.  Otherwise,
    				 * it indicates the number of lines being used
				 * at the top of the typescript for vi-style
				 * terminal emulation. */
#endif
} MxFileInfo;

/* Flag values for MxFileInfo structures:
 * 
 * DELETE:		1 means delete file on last window close.
 * MODIFIED:		1 means this file had been modified as of the last
 *			it was checked.  This is used only to tell when the
 *			file becomes modified or unmodified again.
 * NO_ESCAPES:		(Tx only) 1 means ignore escape sequences in output
 *			destined for typescript;  0 means interpret them as
 *			Tx commands.
 */

#define DELETE		1
#define MODIFIED	4
#define NO_ESCAPES	8

/* Related flags used by the File manager routines.
 * MX_UNDO:		1 means create undo log and use it to track changes
 *			in the file (name must be non-NULL).  0 means no
 *			undoing.
 * MX_DELETE:		1 means close the Mx_File when the last window on
 *			the window is deleted.  0 means don't ever close
 *			the file:  the caller will handle it.
 */
#define MX_UNDO		1
#define MX_DELETE	2

/*
 * The data structure below keeps track of one line on the display,
 * which corresponds to a line or a piece of a line from a file.
 * When the end of the file is above the bottom of the window, the
 * lines that are past the last one in the file have positions that
 * continue to be numbered up sequentially, as if the lines existed
 * after all and contained a single newline character (this is needed
 * to allow the screen to be scrolled).
 */

typedef struct {
    Mx_Position position;	/* Position within the file of the first
				 * character to be displayed on this line
				 * of the display.  The character index is
				 * 0 except for the trailing parts of lines
				 * that wrap around. */
    int length;			/* How many characters to display on this
				 * line.  Includes the newline character,
				 * if this display line is the last for
				 * the file line. */
    int redisplayFirst;		/* Index (within file line, not display line)
				 * of first character that needs to be
				 * redisplayed, if any.  -1 means line is
				 * up-to-date:  nothing to redisplay. */
    int redisplayLast;		/* Index within file of last char that needs
				 * to be redisplayed, if any. */
} MxDisplayLine;
    

/*
 * A data structure of the following type is kept for each
 * widget managed by this file:
 */

typedef struct MxWidget {
    Tk_Window tkwin;		/* Window that displays the text.  NULL
				 * means that the window has been destroyed. */
    Tcl_Interp *interp;		/* Interpreter associated with view. */
    Tcl_Interp *gblInterp;	/* Process-global interpreter context */
    int type;			/* Type of widget:  restricts
				 * operations that may be performed
				 * on widget.  See below for
				 * possible values. */
    /*
     * Editor information.
     */
    char *fileName;		/* Name of file */
    struct MxWidget *nextPtr;	/* Next widget displaying same file */
    MxFileInfo *fileInfoPtr;	/* Information about file being displayed
				 * in window. */
    /*
     * Colors and GCs
     */
    XColor *foreground;		/* Foreground color in normal mode. */
    XColor *activeFg;		/* Foreground color in active mode.  NULL
				 * means use normalFg instead. */
    XColor *background;		/* Background color in normal mode. */
    XColor *activeBg;		/* Background color in active mode.  NULL
				 * means use normalBg instead. */
    XColor *selectorFg;		/* Color for selector. */
    Cursor cursor;		/* Cursor for use in widget. */
    GC textGC;			/* Graphics context for displaying text,
				 * underlines, boxes, and the caret. */
    GC grayGC;			/* Used for drawing MX_GRAY highlights.  Starts
				 * off as None, gets created when first
				 * needed. */
    GC reverseGC;		/* Used for drawing MX_REVERSE highlights.
				 * Starts off as None, gets created when first
				 * needed. */
    int cursorMode;		/* Type of cursor to use. */

    /*
     * Redisplay information:  managed by mxDisplay.c.  The charWidths
     * information is provided to augment what's in *fontPtr:  it is
     * complete (i.e. there's a value for every single ASCII character),
     * and it's also got "correct" values for control characters (i.e.
     * the width of the two-character sequence "^X").  Tabs and newlines
     * are still special;  they have charWidths < 0.
     */

    XFontStruct *fontPtr;	/* Information about text font, or NULL. */
    int fontHeight;		/* Height of each line, in pixels. */
    unsigned char *charWidths;	/* Width of each character, in pixels.
				 * See note above. */
    char *geometry;		/* Gemetry specification */
    int width, height;		/* Dimensions of widget, in pixels. */
    int heightLines;		/* Height of widget in lines of text. */
    int widthChars;		/* Width of widget in characters. */
    MxDisplayLine *linePtr;	/* Pointer to array of line descriptors. */
    int lineArraySize;		/* Size of linePtr array (>= heightLines). */
    int tabWidth;		/* Width of a tab in this font. */
    Mx_Position keyPosition;	/* When the view changes size, try to
				 * keep this position centered in window. */
    Mx_Spy spy;			/* Used to monitor changes in the file. */
    Pixmap eofPixmap;		/* Pattern to use for areas off the end of
    				 * the file. */
    int shiftFirst;		/* First display line to be shifted during
				 * next display update. */
    int shiftNumLines;		/* How many lines to shift during update.
				 * 0 means there's no shift pending. */
    int shiftDistance;		/* Amount by which to shift: may be positive
				 * or negative. */
    GC eofGc;			/* For the stippled end-of-file area. */
    int eofYOrigin;		/* Current origin for drawing eof stipple;
				 * must be adjusted as window scrolls to keep
				 * alignment. */
    /*
     * Stuff for TX, which is not yet supported.
     */
    int viLines;

    /*
     * Information for history recording.  Managed by mxHistory.c.
     */

    int histEnabled;		/* > 0 means history recording is on;  < 0
				 * means it's disabled. */
    char *histBuffer;		/* String containing current history info
				 * (dynamically allocated). */
    int histSpace;		/* Total size of histBuffer, in bytes. */
    int histCurSize;		/* Current length of string in histBuffer,
				 * not including terminating null. */
    Tcl_Trace histTrace;	/* Token for history tracer. */

    /*
     * Scrolling state.
     */

    int lastY;			/* Last location when button was pressed */

    /*
     * Widget names
     */
    char *pathname;		/* TK pathname of mxedit widget */
    char *scrollWidget;		/* TK pathname of scrollbar widget */
    char *feedbackCmd;		/* Name produced by feedbackSetup */

    /*
     * Miscellaneous information:
     */
    int quoteFlag;		/* Use to interpret keystokes */
    int flags;			/* Various flags;  see below for definitions. */
} MxWidget;



/*
 * The flag bits for MxWidgets are:
 *
 * FOCUS_WINDOW:	1 means this window has the focus (i.e., caret
 *			should be displayed, except when it's temporarily
 *			turned off to process commands).
 * NEEDS_UPDATE:	1 means that changes have been made to the window's
 *			contents but the screen hasn't been redisplayed to
 *			reflect all the changes.
 * DELAY_REDISPLAY:	1 means a ConfigureNotify event has been received
 *			but the corresponding Expose event has not been
 *			received.  In this case, redisplay is delayed until
 *			the Expose event arrives (in order to avoid double-
 *			redisplay).
 * REDRAW_SLIVER:	1 means there's a small strip at the bottom of the
 *			window that's too small to hold a whole line but
 *			needs to be redisplayed in the appropriate
 *			background.  This bit gets set in MxRedisplayRegion.
 * SLIVER_EOF:		1 means that the last time the sliver was displayed,
 *			the end-of-file stipple was used.  0 means the
 *			last display was in the background color.
 * DESTROYED:		1 means that the window has been destroyed, but hasn't
 *			been cleaned up yet.  This flag is needed to avoid
 *			accidentally touching the window and thereby causing
 *			an X error.
 * MAPPED:		1 means the window is mapped;  0 means it hasn't
 *			yet been mapped.
 * CARET_VISIBLE:	1 means the caret was visible in this window before
 *			characters were output to the typescript, so the window
 *			should be scrolled after the output in order to keep
 *			the caret visible.  This flag bit is only used by Tx.
 * TX_MAIN_WINDOW:	1 means this is the "main" window on a typescript: the
 *			one whose size is reflected in termcap and winsize
 *			information.  This flag bit is only used by Tx.
 * REDRAW_PENDING:	Non-zero means a DoWhenIdle handler
 *			has already been queued to redraw
 *			this window.
 */
#define TX_WIDGET	1
#define FOCUS_WINDOW	4
#define NEEDS_UPDATE	0x10
#define DELAY_REDISPLAY	0x20
#define REDRAW_SLIVER	0x40
#define SLIVER_EOF	0x80
#define DESTROYED	0x100
#define MAPPED		0x200
#define CARET_VISIBLE	0x400
#define TX_MAIN_WINDOW	0x800
#define REDRAW_PENDING	0x1000

/*
 * Modes for the cursor.
 */
#define OFF 0
#define CARET 1
#define BLOCK 2
#define VIBLOCK 3

/*
 * For each highlighted area of a file, there's a structure of the
 * following shape that describes the highlight.
 */

typedef struct MxHighlight {
    MxFileInfo *fileInfoPtr;	/* Information about file containing
				 * highlight. */
    Mx_Position first;		/* First byte of highlighted information. */
    Mx_Position last;		/* Last byte of highlighted information. */
    Mx_Floater floater;		/* Keeps first and last up-to-date when
				 * file is modified. */
    int style;			/* If < 0, gives style in which to highlight.
				 * If >= 0, gives background color; text is
				 * simply redrawn in different colors. */
    unsigned long foreground;	/* Foreground color for highlighting. */
    struct MxHighlight *nextPtr;/* Next in list of highlights associated with
				 * this file. */
} MxHighlight;

/*
 * The flag bits below are used as parameters to Mx_CreateSpy to
 * indicate when a spy procedure should be called.
 */

#define MX_BEFORE		1
#define MX_AFTER		2

/*
 * The values below may be passed to Mx_HighlightCreate to specify
 * how a highlight is to be displayed:
 */

#define MX_REVERSE		1
#define MX_GRAY			2
#define MX_UNDERLINE		3
#define MX_BOX			4

/*
 * Constants for undo module:
 */

#define UNDO_ID_LENGTH 60
#define UNDO_NAME_LENGTH 600

/*
 * Exported procedures:
 */

extern void		Mx_MarkClean();
extern Mx_Position	Mx_EndOfFile();
extern Mx_File		Mx_FileLoad();
extern void		Mx_FileClose();
extern int		Mx_FileWrite();
extern Mx_Floater	Mx_FloaterCreate();
extern void		Mx_FloaterDelete();
extern char *		Mx_GetLine();
extern Mx_Position	Mx_Offset();
extern void		Mx_ReplaceBytes();
extern Mx_Spy		Mx_SpyCreate();
extern void		Mx_SpyDelete();
extern Mx_Position	Mx_ZeroPosition;

extern int		Mx_SearchMask();
extern int		Mx_SearchParen();
extern int		Mx_SearchPattern();
extern int		Mx_SearchRegExp();
extern void		Mx_ReplaceRegExp();
extern int		Mx_SearchWord();
extern char *		Mx_CompileRegExp();
extern int		Mx_GetTag();

extern Undo_Log		Undo_LogCreate();
extern void		Undo_LogDelete();
extern int		Undo_FindLog();
extern void		Undo_Mark();
extern void		Undo_SetVersion();
extern int		Undo_Last();
extern int		Undo_More();
extern int		Undo_Recover();

extern void		Mx_Cleanup();
extern int		mx_FileCount;

/*
 * Top-level TCL procedures supplied by the mxWidget implementation.
 */
extern int Mx_BindCmd();
extern int Mx_CaretCmd();
extern int Mx_CleanCmd();
extern int Mx_ColumnCmd();
extern int Mx_ConfigureCmd();
extern int Mx_ControlCmd();
extern int Mx_DeleteCmd();
extern int Mx_ExtractCmd();
extern int Mx_GridsizeCmd();
extern int Mx_HistoryCmd();
extern int Mx_IdentCmd();
extern int Mx_IndentCmd();
extern int Mx_IndexCmd();
extern int Mx_InsertCmd();
extern int Mx_InsertSelCmd();
extern int Mx_MarkCmd();
extern int Mx_MarkParenCmd();
extern int Mx_MessageCmd();
extern int Mx_NewlineCmd();
extern int Mx_QuitCmd();
extern int Mx_QuoteCmd();
extern int Mx_ReadCmd();
extern int Mx_SelectionCmd();
extern int Mx_ScanCmd();
extern int Mx_SeeCmd();
extern int Mx_SwitchCmd();
extern int Mx_TaginfoCmd();
extern int Mx_UndoCmd();
extern int Mx_ViewCmd();
extern int Mx_WriteCmd();
extern int Mx_WrittenCmd();
extern int Mx_ReplaceCmd();
extern int Mx_ResetCmd();
extern int Mx_SearchCmd();

/*
 * Procedures exported by mxWidget.c for use by the rest of Mx:
 */
extern void		MxDestroyWidget();
extern void		MxWidgetSizeChangeCallback();
extern void		MxWidgetStateChangeCallback();

/*
 * Procedures exported by mxFileMgr.c for use by the rest of Mx:
 */
extern void		Mx_Cleanup();
extern void		Mx_CheckFileState();
extern int		MxSwitchFile();
extern int		MxSetupFileInfo();
extern void		MxDetachFileInfo();
extern void		MxRecycleFileInfo();
extern int		MxResetFile();

/*
 * Procedures and constants exported by mxDisplay.c for use by the rest of Mx:
 */

extern void		MxWidgetEventProc();
extern int		MxCharToPoint();
extern void		MxCleanupRedisplay();
extern void		MxGetInWindow();
extern int		MxFindPosition();
extern int		MxMeasureChars();
extern void		MxRedisplayRange();
extern void		MxRedisplayRegion();
extern void		MxSetupRedisplay();
extern void		MxUpdateWidget();
extern void		MxEventuallyRedraw();

/*
 * Procedures exported by mxHighlight.c for use by the rest of Mx:
 */

extern void		MxCaretRedisplayText();
extern void		MxCaretSetPosition();
extern void		MxDisplayCaret();
extern void		MxDisplayHighlights();
extern MxHighlight *	MxHighlightCreate();
extern void		MxHighlightDelete();
extern void		MxHighlightGetRange();
extern void		MxHighlightSetRange();

/*
 * Procedures exported by mxSelect.c for use by the rest of Mx:
 */

extern MxHighlight *	MxSelection;
extern MxFileInfo *	MxSelectedFile;
extern void		MxSelectionClear();
extern void		MxSelectionSet();
extern int		MxGetSelRange();

/*
 * Procedures exported mxIndent.c for use by the rest of Mx:
 */
extern void MxCleanIndent();
extern int MxGetIndent();
extern void MxSetIndent();

/*
 * Procedures exported mxHistory.c for use by the rest of Mx:
 */
void MxHistoryOn();
void MxHistoryOff();
void MxHistoryNext();
void MxHistoryCleanup();

/*
 * Procedures exported CmdUtils.c for use by the rest of Mx:
 */
extern void		Mx_Panic();
extern void		MxCmdInit();
extern int	      (*MxCmdFind())();

extern char *		(mxOpenParens[]);
extern char *		(mxCloseParens[]);
extern char		mxWordMask[16];
extern int		MxCheckWritten();
extern void		MxCvtToPrintable();	/* unused */
extern int		MxDoCmd();		/* unused */
extern void		MxFocusProc();
extern int		MxGetMark();
extern time_t		MxLastMod();
extern void		MxMarkParens();
extern void		MxMouseProc();
extern void		MxOutputMsg();

#endif /* not _MXWIDGET */

