/* 
 * mxCmdNR.c --
 *
 *	This file contains the top-level command procedures for
 *	all Mx commands whose names begin with the letters n-r.
 *	Some of these commands are also used by Tx.
 *
 * Copyright (C) 1987, 1988 Regents of the University of California
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any purpose and without
 * fee is hereby granted, provided that the above copyright
 * notice appear in all copies.  The University of California
 * makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 *
 * Copyright (c) 1992 Xerox Corporation.
 * Use and copying of this software and preparation of derivative works based
 * upon this software are permitted. Any distribution of this software or
 * derivative works must comply with all applicable United States export
 * control laws. This software is made available AS IS, and Xerox Corporation
 * makes no warranty about the software, its performance or its conformity to
 * any specification.
 */

#ifndef lint
static char rcsid[] = "$Header: /import/tcl7/src/mxedit/RCS/mxCmdNR.c,v 2.2 1994/02/01 18:15:43 welch Exp $ SPRITE (Berkeley)";
#endif not lint

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <sys/file.h>
#include <stdlib.h>

#include "mxWidget.h"
#include "regex.h"

extern char *regOldLine;

/*
 *----------------------------------------------------------------------
 *
 * Mx_NewlineCmd --
 *
 *	This is the top-level procedure that implements the
 *	"newline" command that inserts a newline character.
 *	This automatically provides indentation.
 *
 * Results:
 *	See the man page.
 *
 * Side effects:
 *	See the man page.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
int
Mx_NewlineCmd(mxwPtr, interp, argc, argv)
    register MxWidget *mxwPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    register MxFileInfo *fileInfoPtr = mxwPtr->fileInfoPtr;
    Mx_Position insertPoint;
    int indent;

    if (argc != 1) {
	sprintf(interp->result, "too many args: should be just \"%.50s\"",
		argv[0]);
	return TCL_ERROR;
    }
    insertPoint = fileInfoPtr->caretFirst;

    /*
     * Duplicate the old line's indentation on the new line.
     */

    indent = MxGetIndent(fileInfoPtr->file, insertPoint.lineIndex);
    Mx_ReplaceBytes(fileInfoPtr->file, insertPoint, insertPoint, "\n");
    (void) MxSetIndent(fileInfoPtr->file,
	    fileInfoPtr->caretFirst.lineIndex, indent);
    MxGetInWindow(mxwPtr, fileInfoPtr->caretFirst, mxwPtr->heightLines-1,
	    0);
    MxCleanIndent(fileInfoPtr->file, insertPoint.lineIndex);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Mx_QuitCmd --
 *
 *	This is the top-level procedure that implements the
 *	"quit" command.  This prepares the widget for destruction.
 *
 * Results:
 *	See the man page.
 *
 * Side effects:
 *	See the man page.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
int
Mx_QuitCmd(mxwPtr, interp, argc, argv)
    register MxWidget *mxwPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    if (argc > 1) {
	sprintf(interp->result, "too many args: should be \"%.50s\"",
		argv[0]);
	return TCL_ERROR;
    }
    mxwPtr->flags |= DESTROYED;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Mx_QuoteCmd --
 *
 *	This is the top-level procedure that implements the
 *	"quote" command.  This quotes the next input character,
 *	dodging the keybinding associated with it.
 *
 * Results:
 *	See the man page.
 *
 * Side effects:
 *	See the man page.
 *
 *----------------------------------------------------------------------
 */

	/* ARGSUSED */
int
Mx_QuoteCmd(mxwPtr, interp, argc, argv)
    register MxWidget *mxwPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    if (argc > 1) {
	sprintf(interp->result, "too many args: should be \"%.50s\"",
		argv[0]);
	return TCL_ERROR;
    }
    mxwPtr->quoteFlag = 1;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Mx_ReadCmd --
 *
 *	This is the top-level procedure that implements the
 *	"read" command.  This inserts the contents of a file
 *	at the insert point.
 *
 * Results:
 *	See the man page.
 *
 * Side effects:
 *	See the man page.
 *
 *----------------------------------------------------------------------
 */

int
Mx_ReadCmd(mxwPtr, interp, argc, argv)
    register MxWidget *mxwPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    int streamID;
#define BUFFER_SIZE 4096
    char buffer[BUFFER_SIZE + 1];
    int result = TCL_OK;
    char *fileName;
    Tcl_DString string;

    if (argc != 2) {
	sprintf(interp->result, "wrong # args: should be \"%.50s file\"",
		argv[0]);
	return TCL_ERROR;
    }

    fileName = Tcl_TildeSubst(interp, argv[1], &string);
    if (fileName == NULL) {
	return TCL_ERROR;
    }
    streamID = open(fileName, O_RDONLY, 0);
    Tcl_DStringFree(&string);
    if (streamID < 0) {
	sprintf(interp->result, "couldn't read file \"%.50s\": %.50s",
		argv[1], strerror(errno));
	return TCL_ERROR;
    }
    while (1) {
	int count;

	count = read(streamID, buffer, BUFFER_SIZE);
	if (count < 0) {
	    sprintf(interp->result,
		    "error while reading \"%.50s\": %.50s", argv[1],
		    strerror(errno));
	    result = TCL_ERROR;
	    goto done;
	}
	if (count == 0) {
	    goto done;
	}
	buffer[count] = 0;
	Mx_ReplaceBytes(mxwPtr->fileInfoPtr->file,
		mxwPtr->fileInfoPtr->caretFirst,
		mxwPtr->fileInfoPtr->caretFirst, buffer);
    }

    done:
    close(streamID);
    MxGetInWindow(mxwPtr, mxwPtr->fileInfoPtr->caretFirst,
	    mxwPtr->heightLines-1, 0);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * Mx_ReplaceCmd --
 *
 *	This is the top-level procedure that implements the
 *	"replace" command.  See the man page for details on what it
 *	does and what it returns.
 *
 * Results:
 *	See the man page.
 *
 * Side effects:
 *	See the man page.
 *
 *----------------------------------------------------------------------
 */

int
Mx_ReplaceCmd(mxwPtr, interp, argc, argv)
    register MxWidget *mxwPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Mx_Position start, stop;		/* Range in which to replace all
					 * occurrences. */
    Mx_Position first, last;		/* Range of thing to be replaced. */
    char *pattern=0, *replace=0;	/* Search, replacement strings. */
    int rangeReplace;			/* 1 means all instances of
					 * pattern will be replaced in the
					 * range between start and stop.  0
					 * means just replace the selection. */
    int getFromWindow;			/* 1 means get pattern and replace
					 * from search window rather than
					 * from command line. */
    Mx_Floater floater;			/* In range replacement, keeps track of
					 * the space left to check (the range
					 * may grow or shrink as things are
					 * replaced). */
    int numReplaced;			/* Number of replacements made. */
    int result;
    int regExp=1;			/* 1 if using reg. exp. matching */
    char *errStr;			/* Error string */
    char *value;

    /*
     * Parse arguments.
     */

    if (argc == 1) {
	getFromWindow = 1;
	rangeReplace = 0;
    } else {
	int length;

	length = strlen(argv[1]);
	if (strncmp(argv[1], "range", length) == 0) {
	    rangeReplace = 1;
	    if (argc == 4) {
		getFromWindow = 1;
	    } else if (argc == 6) {
		getFromWindow = 0;
		pattern = argv[4];
		replace = argv[5];
	    } else {
		sprintf(interp->result, "wrong # args: should be \"%.50s range first last [pattern string]\"",
			argv[0]);
		return TCL_ERROR;
	    }
	    result = MxGetMark(mxwPtr, argv[2], &start);
	    if (result != TCL_OK) {
		return result;
	    }
	    result = MxGetMark(mxwPtr, argv[3], &stop);
	    if (result != TCL_OK) {
		return result;
	    }
	} else if (strncmp(argv[1], "selection", length) == 0) {
	    rangeReplace = 0;
	    if (argc != 3) {
		sprintf(interp->result,
			"wrong # args: should be \"%.50s selection string\"",
			argv[0]);
		return TCL_ERROR;
	    }
	    replace = argv[2];
	    getFromWindow = 0;
	} else {
	    sprintf(interp->result,
		    "bad option for \"%.50s\": should be range or selection\"",
		    argv[0]);
	    return TCL_ERROR;
	}
    }

    /*
     * Pick up the search and replacement strings from the search
     * window, if desired.
     */
    if (getFromWindow) {
	pattern = Tcl_GetVar(mxwPtr->interp, "mxSearchString", TCL_GLOBAL_ONLY);
	if ((pattern == NULL) || (pattern[0] == '\0')) {
	    sprintf(interp->result, "no search string");
	    return TCL_ERROR;
	}
	replace = Tcl_GetVar(mxwPtr->interp, "mxReplaceString", TCL_GLOBAL_ONLY);
	if ((replace == NULL) || (replace[0] == '\0')) {
	    sprintf(interp->result, "no replace string");
	    return TCL_ERROR;
	}
    }
    value = Tcl_GetVar(mxwPtr->interp, "mxNoRegExps", 1);
    if ((value != NULL) && (*value == '1')) {
        regExp = 0;
    }

    /*
     * For non-range replacements, just replace the selection (there
     * better be one!) with the replacement string.
     */
    
    if (!rangeReplace) {
	result = MxGetSelRange(mxwPtr, &first, &last);
	if (result != TCL_OK) {
	    return result;
	}
	if (regExp && regOldLine != NULL) {
	    Mx_ReplaceRegExp(mxwPtr, first, last, replace, &start);
	} else {
	    Mx_ReplaceBytes(mxwPtr->fileInfoPtr->file, first,
		    Mx_Offset(mxwPtr->fileInfoPtr->file, last, 1), replace);
	}
	MxGetInWindow(mxwPtr, first, mxwPtr->heightLines/2, 0);
	return TCL_OK;
    }

    /*
     * Range replacement:  search the entire range that's selected,
     * and replace each instance of the pattern.
     */

    if (regExp) {
	if (pattern[0] == '\0') {
	    sprintf(interp->result, "0 replacements made");
	    return TCL_OK;
	}
	errStr = Mx_CompileRegExp(pattern);
	if (errStr != NULL) {
	    if (errStr != NULL) {
		sprintf(interp->result, "bad \"%.50s\" pattern: %.50s",
		        argv[0], errStr);
		return TCL_ERROR;
	    }
	}
    }
    numReplaced = 0;
    floater = Mx_FloaterCreate(mxwPtr->fileInfoPtr->file, &start, &stop);
    while (MX_POS_LEQ(start, stop)) {
	if (regExp) {
	    if (!Mx_SearchRegExp(mxwPtr->fileInfoPtr->file, start,
		    stop, &first, &last) || MX_POS_LESS(stop, last)) {
		break;
	    }
	    Mx_ReplaceRegExp(mxwPtr, first, last,
		    replace, &start);
	} else {
	    if (!Mx_SearchPattern(mxwPtr->fileInfoPtr->file, start,
		    stop, pattern, &first, &last) || MX_POS_LESS(stop, last)) {
		break;
	    }
	    Mx_ReplaceBytes(mxwPtr->fileInfoPtr->file, first,
		    Mx_Offset(mxwPtr->fileInfoPtr->file, last, 1),
		    replace);
	    start = Mx_Offset(mxwPtr->fileInfoPtr->file, first,
		    strlen(replace));
	}
	numReplaced++;
	MxGetInWindow(mxwPtr, first, mxwPtr->heightLines/2, 0);
    }
    Mx_FloaterDelete(floater);
    sprintf(interp->result, "%d replacements made", numReplaced);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Mx_ResetCmd --
 *
 *	This is the top-level procedure that implements the
 *	"reset" command.  This resets the file from the copy on disk.
 *
 * Results:
 *	See the man page.
 *
 * Side effects:
 *	See the man page.
 *
 *----------------------------------------------------------------------
 */

int
Mx_ResetCmd(mxwPtr, interp, argc, argv)
    register MxWidget *mxwPtr;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    int result;
    int force = 0;

    if (argc > 2) {
	sprintf(interp->result, "wrong # args: should be \"%.50s [force]\"",
		argv[0]);
	return TCL_ERROR;
    }
    if (argc == 2) {
	if (strcmp(argv[1], "force") != 0) {
	    sprintf(interp->result, "bad argument: should be \"%.50s [force]\"",
		argv[0]);
	    return TCL_ERROR;
	} else {
	    force = 1;
	}
    }
    if (!force) {
	result = MxCheckWritten(interp, mxwPtr, 1);
	if (result != TCL_OK) {
	    return result;
	}
    }
    return MxResetFile(mxwPtr);
}
